/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* GNOME Volume Applet
 * Copyright (C) 2004 Ronald Bultje <rbultje@ronald.bitfreak.net>
 *
 * applet.c: the main applet
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/* this is for lrint */
#define _ISOC99_SOURCE
#include <math.h>
#include <string.h>

#include <glib-object.h>
#include <gdk/gdkkeysyms.h>

#include <gtk/gtkaboutdialog.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkicontheme.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkmessagedialog.h>
#include <gtk/gtkrange.h>
#include <gtk/gtkwidget.h>

#include <gconf/gconf-client.h>

#include "applet.h"

static void	gnome_volume_applet_class_init	(GnomeVolumeAppletClass *klass);
static void	gnome_volume_applet_init	(GnomeVolumeApplet *applet);
static void	gnome_volume_applet_dispose	(GObject   *object);

static void     gnome_volume_applet_size_allocate (GtkWidget     *widget, 
						   GtkAllocation *allocation);

static void	gnome_volume_applet_popup_dock	(GnomeVolumeApplet *applet);
static void	gnome_volume_applet_popdown_dock (GnomeVolumeApplet *applet);

static gboolean	gnome_volume_applet_scroll	(GtkWidget *widget,
						 GdkEventScroll *event);
static gboolean	gnome_volume_applet_button	(GtkWidget *widget,
						 GdkEventButton *event);
static gboolean	gnome_volume_applet_key		(GtkWidget *widget,
						 GdkEventKey *event);
static void	gnome_volume_applet_refresh	(GnomeVolumeApplet *applet,
						 gboolean           force_refresh);
static void	cb_volume			(GtkAdjustment *adj,
						 gpointer   data);
static void	cb_theme_change                (GtkIconTheme *icon_theme,
						gpointer      data);
static void	cb_stop_scroll_events		(GtkWidget *widget,
						 GdkEvent  *event);

static GtkEventBoxClass *parent_class = NULL;

GType
gnome_volume_applet_get_type (void)
{
  static GType gnome_volume_applet_type = 0;

  if (!gnome_volume_applet_type) {
    static const GTypeInfo gnome_volume_applet_info = {
      sizeof (GnomeVolumeAppletClass),
      NULL,
      NULL,
      (GClassInitFunc) gnome_volume_applet_class_init,
      NULL,
      NULL,
      sizeof (GnomeVolumeApplet),
      0,
      (GInstanceInitFunc) gnome_volume_applet_init,
      NULL
    };

    gnome_volume_applet_type =
	g_type_register_static (GTK_TYPE_EVENT_BOX,
				"GnomeVolumeApplet",
				&gnome_volume_applet_info, 0);
  }

  return gnome_volume_applet_type;
}

int
set_volume(GnomeVolumeApplet *applet, int value)
{
  int vol;

  if (value > VOLUME_MAX) value = VOLUME_MAX;
  if (value < VOLUME_MIN) value = VOLUME_MIN;

  vol = value | (value << 8);
  if (ioctl(applet->mixerfd, SOUND_MIXER_WRITE_VOLUME, &vol)==-1)
  {
    g_error("Can't open mixer device for writing\n");
    return -1;
  }

  return value;
}

int
get_volume(GnomeVolumeApplet *applet)
{
  int vol;

  if (ioctl(applet->mixerfd, SOUND_MIXER_READ_VOLUME, &vol)==-1)
  {
    g_error("Can't open mixer device for reading\n");
    return -1;
  }

  return ( (vol & 0xff) + ((vol >> 8) & 0xff) ) / 2;
}

static void
init_pixbufs (GnomeVolumeApplet *applet)
{
  static const gchar *pix_filenames[] = {
    "audio-volume-muted",
    "audio-volume-low",
    "audio-volume-medium",
    "audio-volume-high",
    NULL
  };
  gint n;
  
  for (n = 0; pix_filenames[n] != NULL; n++) {
    if (applet->pix[n])
      g_object_unref (applet->pix[n]);
    
    applet->pix[n] = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (),
					       pix_filenames[n],
					       applet->panel_size - 4,
					       0,
					       NULL);
    if (gtk_widget_get_default_direction () == GTK_TEXT_DIR_RTL) {
      GdkPixbuf *temp;

      temp = gdk_pixbuf_flip (applet->pix[n], TRUE);
      g_object_unref (G_OBJECT (applet->pix[n]));
      applet->pix[n] = temp;
    }
  }
}

static void
gnome_volume_applet_class_init (GnomeVolumeAppletClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  GtkWidgetClass *gtkwidget_class = GTK_WIDGET_CLASS (klass);

  parent_class = g_type_class_ref (GTK_TYPE_EVENT_BOX);

  gobject_class->dispose = gnome_volume_applet_dispose;
  gtkwidget_class->key_press_event = gnome_volume_applet_key;
  gtkwidget_class->button_press_event = gnome_volume_applet_button;
  gtkwidget_class->scroll_event = gnome_volume_applet_scroll;
  gtkwidget_class->size_allocate = gnome_volume_applet_size_allocate;

  /* FIXME:
   * - style-set.
   */
}

static void
gnome_volume_applet_init (GnomeVolumeApplet *applet)
{
  GtkWidget *dock;
  GtkWidget *image;
  AtkObject *ao;

  applet->timeout = 0;
  applet->elements = NULL;
  applet->client = gconf_client_get_default ();
  applet->lock = FALSE;
  applet->prefs = NULL;
  applet->dock = NULL;
  applet->panel_size = 24;

  /* init pixbufs */
  init_pixbufs (applet);

  /* icon (our main UI) */
  image = gtk_image_new ();
  applet->image = GTK_IMAGE (image);
  gtk_container_add (GTK_CONTAINER (applet), image);
  gtk_widget_show (image);
  gtk_window_set_default_icon_name ("multimedia-volume-control");

  /* dock window (expanded UI) */
  applet->pop = FALSE;

  /* tooltip over applet */
  gtk_widget_set_tooltip_text (GTK_WIDGET (applet), _("Volume Control"));

  /* prevent scroll events from reaching the tooltip */
  g_signal_connect (G_OBJECT (applet),
		    "event-after", G_CALLBACK (cb_stop_scroll_events),
		    NULL);

  /* handle icon theme changes */
  g_signal_connect (gtk_icon_theme_get_default (),
		    "changed", G_CALLBACK (cb_theme_change),
		    applet);

  /* i18n */
  ao = gtk_widget_get_accessible (GTK_WIDGET (applet));
  atk_object_set_name (ao, _("Volume Control"));

  dock = gnome_volume_applet_dock_new (GTK_ORIENTATION_VERTICAL);
  /* parent, for signal forwarding */
  gtk_widget_set_parent (dock, GTK_WIDGET (applet));
  applet->dock = GNOME_VOLUME_APPLET_DOCK (dock);
}

gboolean
moblin_volume_applet_setup (GnomeVolumeApplet *applet)
{
  GtkObject *adj;
  int devmask;

  if ((applet->mixerfd = open(MIXER_DEVICE, O_RDWR)) < 0) {
    g_error("Can't open mixer device: %s\n", MIXER_DEVICE);
    return FALSE;
  }

  if (ioctl(applet->mixerfd, SOUND_MIXER_READ_DEVMASK, &devmask) == -1) {
    g_error("Can't open mixer device for reading\n");
    return FALSE;
  }

  adj = gtk_adjustment_new (50, 0, 100, 4, 10, 0);
  gtk_adjustment_set_value (GTK_ADJUSTMENT (adj), get_volume (applet));
  gnome_volume_applet_dock_change (applet->dock, GTK_ADJUSTMENT (adj));
  g_signal_connect (adj, "value-changed", G_CALLBACK (cb_volume), applet);
  gnome_volume_applet_refresh (applet, TRUE);

  gtk_widget_show (GTK_WIDGET (applet));

  return TRUE;
}

static void
gnome_volume_applet_dispose (GObject *object)
{
  GnomeVolumeApplet *applet = GNOME_VOLUME_APPLET (object);
  gint n;

  gnome_volume_applet_popdown_dock (applet);

  if (applet->timeout) {
    g_source_remove (applet->timeout);
    applet->timeout = 0;
  }

  for (n = 0; n < 5; n++) {
    if (applet->pix[n] != NULL) {
      g_object_unref (G_OBJECT (applet->pix[n]));
      applet->pix[n] = NULL;
    }
  }

  G_OBJECT_CLASS (parent_class)->dispose (object);
}

/*
 * popup (show) or popdown (hide) the dock.
 */

static void
gnome_volume_applet_popup_dock (GnomeVolumeApplet *applet)
{
  GtkWidget *widget = GTK_WIDGET (applet);
  gint x, y;

  gdk_window_get_origin (widget->window, &x, &y);
  y += widget->allocation.height + widget->allocation.y;
  gtk_window_move (GTK_WINDOW (applet->dock), x, y);
  gtk_widget_show_all (GTK_WIDGET (applet->dock));

  /* grab input */
  gtk_widget_grab_focus (GTK_WIDGET (applet->dock->scale));
  gtk_grab_add (widget);
  gdk_pointer_grab (widget->window, TRUE,
		    GDK_BUTTON_PRESS_MASK |
		    GDK_BUTTON_RELEASE_MASK |
		    GDK_POINTER_MOTION_MASK,
		    NULL, NULL, GDK_CURRENT_TIME);
  gdk_keyboard_grab (widget->window, TRUE, GDK_CURRENT_TIME);

  /* set menu item as active */
  gtk_widget_set_state (GTK_WIDGET (applet), GTK_STATE_SELECTED);

  /* keep state */
  applet->pop = TRUE;
}

static void
gnome_volume_applet_popdown_dock (GnomeVolumeApplet *applet)
{
  GtkWidget *widget = GTK_WIDGET (applet);

  if (!applet->pop)
    return;

  /* release input */
  gdk_keyboard_ungrab (GDK_CURRENT_TIME);
  gdk_pointer_ungrab (GDK_CURRENT_TIME);
  gtk_grab_remove (widget);

  /* hide */
  gtk_widget_hide_all (GTK_WIDGET (applet->dock));

  /* set menu item as active */
  gtk_widget_set_state (GTK_WIDGET (applet), GTK_STATE_NORMAL);

  /* keep state */
  applet->pop = FALSE;
}

static void
gnome_volume_applet_pop_dock (GnomeVolumeApplet *applet)
{
  if (applet->pop) {
    gnome_volume_applet_popdown_dock (applet);
  } else {
    gnome_volume_applet_popup_dock (applet);
  }
}

#if 0
static void
gnome_volume_applet_run_mixer (GnomeVolumeApplet *applet)
{
  GnomeDesktopItem *ditem;
  GError *error = NULL;

  if ((ditem = gnome_desktop_item_new_from_basename ("gnome-volume-control.desktop", 0, NULL))) {
    gnome_desktop_item_set_launch_time (ditem, gtk_get_current_event_time ());
    gnome_desktop_item_launch_on_screen (ditem, NULL,
	                                 GNOME_DESKTOP_ITEM_LAUNCH_ONLY_ONE,
					 gtk_widget_get_screen (GTK_WIDGET (applet)),
					 -1, &error);
    gnome_desktop_item_unref (ditem);
  }
  else {
    gdk_spawn_command_line_on_screen (
	      gtk_widget_get_screen (GTK_WIDGET (applet)),
	      "gnome-volume-control", &error);
  }

  if (error) {
    GtkWidget *dialog;

    dialog = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_ERROR,
				     GTK_BUTTONS_CLOSE,
				     _("Failed to start Volume Control: %s"),
				     error->message);
    g_signal_connect (dialog, "response",
		      G_CALLBACK (gtk_widget_destroy), NULL);
    gtk_widget_show (dialog);
    g_error_free (error);
  }
}
#endif

/*
 * Control events, change volume and so on.
 */

static gboolean
gnome_volume_applet_scroll (GtkWidget      *widget,
			    GdkEventScroll *event)
{
  GnomeVolumeApplet *applet = GNOME_VOLUME_APPLET (widget);

  if (event->type == GDK_SCROLL) {
    switch (event->direction) {
      case GDK_SCROLL_UP:
      case GDK_SCROLL_DOWN: {
        GtkAdjustment *adj = gtk_range_get_adjustment (applet->dock->scale);
        gdouble volume = adj->value;

        if (event->direction == GDK_SCROLL_UP) {
          volume += adj->step_increment;
          if (volume > adj->upper)
            volume = adj->upper;
        } else {
          volume -= adj->step_increment;
          if (volume < adj->lower)
            volume = adj->lower;
        }

        gtk_range_set_value (applet->dock->scale, volume);
        return TRUE;
      }
      default:
        break;
    }
  }

  if (GTK_WIDGET_CLASS (parent_class)->scroll_event)
    return GTK_WIDGET_CLASS (parent_class)->scroll_event (widget, event);
  else
    return FALSE;
}

static gboolean
gnome_volume_applet_button (GtkWidget      *widget,
			    GdkEventButton *event)
{
  GnomeVolumeApplet *applet = GNOME_VOLUME_APPLET (widget);

  if (event->window != GTK_WIDGET (applet)->window &&
      event->type == GDK_BUTTON_PRESS) {
    gnome_volume_applet_popdown_dock (applet);
    return TRUE;
  } else if (event->window == GTK_WIDGET (applet)->window) {
    switch (event->button) {
      case 1:
        switch (event->type) {
          case GDK_BUTTON_PRESS:
            gnome_volume_applet_pop_dock (applet);
            return TRUE;
          case GDK_2BUTTON_PRESS:
            gnome_volume_applet_popdown_dock (applet);
#if 0
            gnome_volume_applet_run_mixer (applet);
#endif
            return TRUE;
          default:
            break;
        }
        break;
      case 2: /* move */
      case 3: /* menu */
        if (applet->pop) {
          gnome_volume_applet_popdown_dock (applet);
          return TRUE;
        }
        break;
      default:
        break;
    }
  }

  if (GTK_WIDGET_CLASS (parent_class)->button_press_event)
    return GTK_WIDGET_CLASS (parent_class)->button_press_event (widget, event);

  return FALSE;
}

static gboolean
gnome_volume_applet_key (GtkWidget   *widget,
			 GdkEventKey *event)
{
  GnomeVolumeApplet *applet = GNOME_VOLUME_APPLET (widget);

  switch (event->keyval) {
    case GDK_KP_Enter:
    case GDK_ISO_Enter:
    case GDK_3270_Enter:
    case GDK_Return:
    case GDK_space:
    case GDK_KP_Space:
      gnome_volume_applet_pop_dock (applet);
      return TRUE;
#if 0
    case GDK_m:
      if (event->state == GDK_CONTROL_MASK) {
        gnome_volume_applet_toggle_mute (applet);
        return TRUE;
      }
      break;
    case GDK_o:
      if (event->state == GDK_CONTROL_MASK) {
        gnome_volume_applet_run_mixer (applet);
        return TRUE;
      }
      break;
#endif
    case GDK_Escape:
      gnome_volume_applet_popdown_dock (applet);
      return TRUE;
    case GDK_Page_Up:
    case GDK_Page_Down:
    case GDK_Left:
    case GDK_Right:
    case GDK_Up:
    case GDK_Down: {
      GtkAdjustment *adj = gtk_range_get_adjustment (applet->dock->scale);
      gdouble volume = adj->value, increment;

      if (event->state != 0)
        break;

      if (event->keyval == GDK_Up || event->keyval == GDK_Down 
         ||event->keyval == GDK_Left)
        increment = adj->step_increment;
      else
        increment = adj->page_increment;

      if (event->keyval == GDK_Page_Up || event->keyval == GDK_Up
         ||event->keyval == GDK_Right) {
        volume += increment;
        if (volume > adj->upper)
          volume = adj->upper;
      } else {
        volume -= increment;
        if (volume < adj->lower)
          volume = adj->lower;
      }

      gtk_range_set_value (applet->dock->scale, volume);
      return TRUE;
    }
    default:
      break;
  }

  return GTK_WIDGET_CLASS (parent_class)->key_press_event (widget, event);
}

void gnome_volume_applet_size_allocate (GtkWidget     *widget, 
					GtkAllocation *allocation)
{
  GnomeVolumeApplet *applet = GNOME_VOLUME_APPLET (widget);

  if (GTK_WIDGET_CLASS (parent_class)->size_allocate)
    GTK_WIDGET_CLASS (parent_class)->size_allocate (widget, allocation);

  if (applet->panel_size == allocation->height)
    return;
  applet->panel_size = allocation->height;

  init_pixbufs (applet);
  gnome_volume_applet_refresh (applet, TRUE);
}

/*
 * Volume changed.
 */

static void
cb_volume (GtkAdjustment *adj,
	   gpointer data)
{
  GnomeVolumeApplet *applet = data;
  gdouble v;

  if (applet->lock)
    return;
  applet->lock = TRUE;

  v = gtk_adjustment_get_value (adj);

  set_volume(applet, (int)v);
  applet->lock = FALSE;
  applet->force_next_update = TRUE;

  gnome_volume_applet_refresh (GNOME_VOLUME_APPLET (data), FALSE);
}

/*
 * Automatic timer. Check for changes.
 */

#define STATE(vol,m) (((gint) vol << 1) | (m ? 1 : 0))

static void
gnome_volume_applet_refresh (GnomeVolumeApplet *applet,
			     gboolean           force_refresh)
{
  GdkPixbuf *pixbuf;
  gint n, volume;
  char tooltip_str[100];
  gboolean mute = FALSE, did_change;

  volume = get_volume (applet);
  if (volume <= 0) {
    mute = TRUE;
    n = 0;
  } else {
    /* select image */
    n = 3 * volume / VOLUME_MAX + 1;
    if (n < 1)
      n = 1;
    if (n > 3)
      n = 3;
  }

  did_change = (force_refresh || applet->force_next_update);
  applet->force_next_update = FALSE;

  if (did_change) {
    if (mute) {
      pixbuf = applet->pix[0];
    } else {
      pixbuf = applet->pix[n];
    }
    gtk_image_set_from_pixbuf (applet->image, pixbuf);
  } else {
	return;
  }

  if (mute) {
    sprintf(tooltip_str, "Volume: muted");
  } else {
    sprintf(tooltip_str, "Volume: %d%%", volume);
  }

  gtk_widget_set_tooltip_text (GTK_WIDGET (applet), tooltip_str);

  applet->lock = TRUE;
  gtk_range_set_value (applet->dock->scale, volume);
  applet->lock = FALSE;
}

static void
cb_theme_change (GtkIconTheme *icon_theme,
		 gpointer data)
{
  GnomeVolumeApplet *applet = GNOME_VOLUME_APPLET (data);

  init_pixbufs (applet);
  gnome_volume_applet_refresh (applet, TRUE);
}

/*
 * Block the tooltips event-after handler on scroll events.
 */

static void
cb_stop_scroll_events (GtkWidget *widget,
		       GdkEvent  *event)
{
  if (event->type == GDK_SCROLL)
    g_signal_stop_emission_by_name (widget, "event-after");
}

MoblinVolumeApplet* 
moblin_volume_applet_new()
{
MoblinVolumeApplet *obj;

  if(!gnome_volume_applet_get_type())
  {
    g_error("Failed to register type GnomeVolumeApplet\n");
    return NULL;
  }
  obj = g_new0(MoblinVolumeApplet, 1);
  obj->applet = (GnomeVolumeApplet*)g_object_new(gnome_volume_applet_get_type(), NULL);

  return obj;
}
