/*
 *  Moblin Background Management Application
 *  Source: capplets/background/wp-capplet.c
 *  Author: Todd Brandt <todd.e.brandt@intel.com>
 *  Copyright 2007-2008 Intel Corp (moblin.org)
 *
 *  Based on capplets/background/gnome-wp-capplet.c from
 *  gnome-control-center-2.18.1: Rodney Dawes <dobey@ximian.com>
 *  Copyright 2003-2006 Novell, Inc. (www.novell.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2 of the GNU General Public License
 *  as published by the Free Software Foundation
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 *
 */

#include "wp-capplet.h"

/* global data */
gchar *original_filename = NULL;
gchar *master_color_value = NULL;
gchar *original_color_value = NULL;
GdkColor *master_color = NULL;

enum {
        PRELOAD_PHASE=0,
	LOAD_PHASE1,
	LOAD_PHASE2,
	LOAD_PHASE3,
	LOAD_PHASE4,
	LOAD_PHASE_DONE
};

static int loadphase = PRELOAD_PHASE;
static void wp_props_load_wallpaper (gchar * key, 
	MoblinWPItem * item, MoblinWPCapplet * capplet);

static void moblin_wp_capplet_scroll_to_item (MoblinWPCapplet * capplet,
					     MoblinWPItem * item) {
  GtkTreePath * path;

  g_return_if_fail (capplet != NULL);

  if (item == NULL)
    return;

  path = gtk_tree_row_reference_get_path (item->rowref);
  gtk_tree_view_set_cursor (GTK_TREE_VIEW (capplet->treeview), path,
			    NULL, FALSE);
  gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (capplet->treeview),
				path, NULL, TRUE, 0.5, 0.0);
  gtk_tree_path_free (path);
}

static MoblinWPItem * moblin_wp_add_image (MoblinWPCapplet * capplet,
					 const gchar * filename) {
  MoblinWPItem * item;

  item = g_hash_table_lookup (capplet->wphash, filename);
  if (item == NULL) {
    item = moblin_wp_item_new (filename, capplet->wphash, capplet->thumbs);
    if (item != NULL) {
      wp_props_load_wallpaper (item->filename, item, capplet);
    }
  }

  return item;
}

static void 
moblin_wp_props_wp_selected (GtkTreeSelection *selection,
	MoblinWPCapplet *capplet)
{
  GtkTreeIter iter;
  GtkTreeModel * model;
  MoblinWPItem * item;
  GConfChangeSet * cs;
  gchar * wpfile;

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    gtk_tree_model_get (model, &iter, 2, &wpfile, -1);

    item = g_hash_table_lookup (capplet->wphash, wpfile);

    cs = gconf_change_set_new ();

    if (!strcmp (item->filename, "(none)")) {
      gconf_change_set_set_string (cs, BGKEY(WP_FILE), "");
      gconf_change_set_set_string (cs, GNOMEBGKEY(WP_FILE), "");
    } else {
      gchar * uri;

      if (g_utf8_validate (item->filename, -1, NULL))
	uri = g_strdup (item->filename);
      else
	uri = g_filename_to_utf8 (item->filename, -1, NULL, NULL, NULL);

      gconf_change_set_set_string (cs, BGKEY(WP_FILE), uri);
      gconf_change_set_set_string (cs, GNOMEBGKEY(WP_FILE), uri);
      g_free (uri);
    }

    gconf_change_set_set_string (cs, BGKEY(BG_PCOLOR), master_color_value);
    gconf_change_set_set_string (cs, GNOMEBGKEY(BG_PCOLOR), master_color_value);

    gconf_client_commit_change_set (capplet->client, cs, TRUE, NULL);

    gconf_change_set_unref (cs);

    g_free (wpfile);
  }
}

static void moblin_wp_props_wp_revert (MoblinWPCapplet * capplet) {
  GConfChangeSet * cs;

    cs = gconf_change_set_new ();

    gconf_change_set_set_string (cs, BGKEY(WP_FILE), original_filename);
    gconf_change_set_set_string (cs, GNOMEBGKEY(WP_FILE), original_filename);
    gconf_change_set_set_string (cs, BGKEY(BG_PCOLOR), original_color_value);
    gconf_change_set_set_string (cs, GNOMEBGKEY(BG_PCOLOR), original_color_value);

    gconf_client_commit_change_set (capplet->client, cs, TRUE, NULL);

    gconf_change_set_unref (cs);
}

static void 
moblin_wp_remove_wp (gchar *key, 
	MoblinWPItem *item, MoblinWPCapplet *capplet)
{
  GtkTreePath * path;
  GtkTreeIter iter;

    if (item->rowref != NULL) 
    {
	path = gtk_tree_row_reference_get_path(item->rowref);
	if (path != NULL) {
	    gtk_tree_model_get_iter (capplet->model, 
		&iter, path);
	    gtk_tree_path_free (path);
	    gtk_list_store_remove(
		GTK_LIST_STORE(capplet->model), &iter);
	}
    }
}

void 
moblin_wp_main_quit (MoblinWPCapplet *capplet)
{
  int i;

    for(i = 0; i < NUM_GCONF_CALLBACKS; i++)
	gconf_client_notify_remove(capplet->client,
	capplet->gconf_cb_ids[i]);

    g_hash_table_foreach (capplet->wphash, 
	(GHFunc) moblin_wp_remove_wp, capplet);
    g_object_unref (capplet->thumbs);
    g_object_unref (capplet->client);
    g_free (capplet);

    if (original_filename != NULL)
	g_free(original_filename);
    if (master_color_value != NULL)
	g_free(master_color_value);
    if (original_color_value != NULL)
	g_free(original_color_value);

    original_filename=NULL;
    master_color_value=NULL; 
    original_color_value=NULL; 
}

static void 
wallpaper_properties_clicked (GtkWidget * dialog,
	gint response_id, MoblinWPCapplet * capplet)
{
    switch (response_id) {
    case GTK_RESPONSE_CANCEL:
	moblin_wp_props_wp_revert(capplet);
    case GTK_RESPONSE_CLOSE:
    case GTK_RESPONSE_DELETE_EVENT:
    case GTK_RESPONSE_OK:
	moblin_wp_main_quit (capplet);
	gtk_widget_destroy (dialog);
	gtk_main_quit();
	break;
    }
}

static void 
update_thumbnail(MoblinWPItem *item,
	GtkTreeIter *iter, MoblinWPCapplet *capplet)
{
  GdkPixbuf * pixbuf;

    pixbuf = moblin_wp_item_get_thumbnail(item, 
	capplet->thumbs);
    gtk_list_store_set(GTK_LIST_STORE(capplet->model),
	iter, 0, pixbuf, -1);
    g_object_unref (pixbuf);
}

static void 
moblin_wp_color_changed (MoblinWPCapplet * capplet,
	gboolean update_gconf) 
{
  MoblinWPItem *item = NULL;
  GtkTreeIter iter;
  gchar * wpfile;

    g_free (master_color_value);

    gtk_color_button_get_color(
	GTK_COLOR_BUTTON(capplet->pc_picker), master_color);

    master_color_value = g_strdup_printf ("#%02X%02X%02X",
				     master_color->red >> 8,
				     master_color->green >> 8,
				     master_color->blue >> 8);

    if (update_gconf)
    {
	gconf_client_set_string (capplet->client,
	    BGKEY(BG_PCOLOR), master_color_value, NULL);
	gconf_client_set_string (capplet->client, 
	    GNOMEBGKEY(BG_PCOLOR), master_color_value, NULL);
    }

    if(loadphase > LOAD_PHASE2)
    {
	/* The treeview is up and populated */
	if(gtk_tree_model_get_iter_first(
           GTK_TREE_MODEL(capplet->model), &iter)) 
	{
	    gtk_tree_model_get (GTK_TREE_MODEL(capplet->model), 
		&iter, 2, &wpfile, -1);
	    if((item = g_hash_table_lookup(capplet->wphash, 
		wpfile)) != NULL)
	    {
		/* update "Solid Color Background" thumbnail */
		update_thumbnail(item, &iter, capplet);
		if(update_gconf) 
		{
		    /* select "Solid Color Background" only */
		    /* if we're called by the color picker */
		    moblin_wp_capplet_scroll_to_item(capplet, 
			item);
		}
	    }
	    g_free (wpfile);
	}
    }
    else
    {
	/* no treeview yet, set the gconf keys directly */
	gconf_client_set_string (capplet->client,
	    BGKEY(WP_FILE), "", NULL);
	gconf_client_set_string (capplet->client, 
	    GNOMEBGKEY(WP_FILE), "", NULL);
    }
}

static void
moblin_wp_scolor_changed (GtkWidget * widget,
			MoblinWPCapplet * capplet)
{
    moblin_wp_color_changed (capplet, TRUE);
}

static void 
jump_to_first_item (GtkWidget * widget,
		MoblinWPCapplet * capplet)
{
GtkTreePath *first=NULL;
GtkTreeIter iter;

    gtk_range_set_value(GTK_RANGE(capplet->jumpslider), 0);
    if(gtk_tree_model_get_iter_first(GTK_TREE_MODEL(capplet->model),
	&iter))
    {
	first = gtk_tree_model_get_path (capplet->model, &iter);
	if(first) 
	{
	    gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW (capplet->treeview),
		first, NULL, FALSE, 0, 0);
	    gtk_tree_path_free (first);
	}
    }
}

static void jump_to_last_item (GtkWidget * widget,
				     MoblinWPCapplet * capplet)
{
GtkTreePath *last=NULL;
GtkTreeIter iter;
gint num;

    num = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(capplet->model),
	NULL);
    if((num > 0)&&
	gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(capplet->model),
	&iter, NULL, num-1))
    {
	gtk_range_set_value(GTK_RANGE(capplet->jumpslider), num-1);
	last = gtk_tree_model_get_path (capplet->model, &iter);
	if(last) 
	{
	    gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW (capplet->treeview),
		last, NULL, FALSE, 0, 0);
	    gtk_tree_path_free (last);
	}
    }
}

static void jump_to_nth_item (GtkWidget * widget,
				     MoblinWPCapplet * capplet)
{
GtkTreePath *item=NULL;
GtkTreeIter iter;
gint n, num;

    n = (int)gtk_range_get_value(GTK_RANGE(widget));
    if((num = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(capplet->model),
	NULL)) <= 0) return;
    if((n >= 0)&&(n < num)&&
	gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(capplet->model),
	&iter, NULL, n))
    {
	item = gtk_tree_model_get_path (capplet->model, &iter);
	if(item) 
	{
	    gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW (capplet->treeview),
		item, NULL, FALSE, 0, 0);
	    gtk_tree_path_free (item);
	}
    }
}

static void scroll_treeview (GtkWidget * widget,
				     MoblinWPCapplet * capplet)
{
GtkTreeIter iter;
GtkTreeModel * model;
GtkTreeSelection * selection;
gchar *rowstr;
gint row;

    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (capplet->treeview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
	rowstr = gtk_tree_model_get_string_from_iter(model, &iter);
	if(rowstr&&(sscanf(rowstr, "%d", &row) == 1)) {
	    gtk_range_set_value(GTK_RANGE(capplet->jumpslider), row);
	}
	g_free(rowstr);
    }
}

static void 
wp_props_load_wallpaper (gchar * key,
	MoblinWPItem * item, MoblinWPCapplet * capplet)
{
  GtkTreeIter iter;
  GtkTreePath * path;

    gtk_list_store_append(GTK_LIST_STORE (capplet->model), &iter);
    moblin_wp_item_update_description(item);
    gtk_list_store_set (GTK_LIST_STORE(capplet->model), &iter,
			1, item->description,
			2, item->filename,
			-1);
    path = gtk_tree_model_get_path (capplet->model, &iter);
    item->rowref = gtk_tree_row_reference_new (capplet->model, path);
    gtk_tree_path_free (path);
}

/* the function that gets looped to display a */
/* thumbnail in an image's item in the treeview */
static void 
wp_props_display_pixbuf (gchar *key,
	MoblinWPCapplet * capplet)
{
  MoblinWPItem *item;
  GtkTreeIter iter;
  GtkTreePath * path;

    item = g_hash_table_lookup (capplet->wphash, key);
    if (item->pixbuf != NULL) {
	moblin_wp_item_update_description (item);
	path = gtk_tree_row_reference_get_path(item->rowref);
	gtk_tree_model_get_iter (capplet->model, &iter, path);
	gtk_list_store_set (GTK_LIST_STORE (capplet->model), &iter,
			0, item->pixbuf,
			1, item->description,
			2, item->filename,
			-1);
	gtk_tree_path_free (path);
	g_object_unref(item->pixbuf);
	item->pixbuf = NULL;
    }
}

static gint
sfunc(gconstpointer a, gconstpointer b)
{
	return g_strcasecmp((gchar *)a, (gchar *)b);
}

/* LOAD PHASE 4 (final) */
/* display pixbufs (thumbnails) is done in an idle */
/* loop since it needs to be in sync with the GUI. The */
/* thumbnails are already loaded into a hash table so it's all */
/* RAM from here, no disk accesses, so it should be */
/* fast */
static gboolean
moblin_wp_display_pixbufs (void * data)
{
  MoblinWPCapplet *capplet = (MoblinWPCapplet*)data;
  GList *keylist;

    keylist = g_hash_table_get_keys(capplet->wphash);
    keylist = g_list_sort(keylist, sfunc);
    g_list_foreach (keylist,
	(GFunc) wp_props_display_pixbuf, capplet);
    g_list_free(keylist);

    loadphase = LOAD_PHASE_DONE;
    return FALSE;
}

/* the function that gets looped to create a */
/* thumbnail from an image */
static void 
wp_props_load_pixbuf (gchar *key,
	MoblinWPItem *item,
	MoblinWPCapplet *capplet)
{
    item->pixbuf = moblin_wp_item_get_thumbnail (item, capplet->thumbs);
}

/* LOAD PHASE 3 */
/* load pixbufs thread, this is spawned in the background */
/* so that the user can operate the GUI while the thumbnail */
/* icons are being generated from the image files. The disk */
/* is being accessed hard in this function, so it can take */
/* several seconds if there are a thousand or more photos */
/* The user can switch photos just from the text filenames and */
/* quit if this takes too long */
static gpointer
moblin_wp_load_pixbufs (gpointer data)
{
  MoblinWPCapplet *capplet = (MoblinWPCapplet*)data;

    g_hash_table_foreach (capplet->wphash,
	(GHFunc) wp_props_load_pixbuf, capplet);

    loadphase = LOAD_PHASE4;
    g_idle_add (moblin_wp_display_pixbufs, capplet);
    return NULL;
}

/* the function that gets looped to display an */
/* entry in the tree view */
static void 
moblin_wp_display_image (gchar *key,
	MoblinWPCapplet * capplet)
{
  MoblinWPItem *item;

    item = g_hash_table_lookup (capplet->wphash, key);
    wp_props_load_wallpaper(key, item, capplet);
}

/* LOAD PHASE 2 */
/* display images is done in an idle loop since it */
/* needs to be in sync with the GUI. The images are */
/* already loaded into a hash table so it's all */
/* RAM from here, no disk accesses, so it should be */
/* fast */
static gboolean
moblin_wp_display_images (void * data)
{
  MoblinWPCapplet * capplet = (MoblinWPCapplet *) data;
  gchar *imagepath;
  MoblinWPItem * item;
  gint num;
  GList *keylist;

    keylist = g_hash_table_get_keys(capplet->wphash);
    keylist = g_list_sort(keylist, sfunc);
    g_list_foreach (keylist,
	(GFunc) moblin_wp_display_image, capplet);
    g_list_free(keylist);

    gdk_window_set_cursor (capplet->window->window, NULL);

    /* add in the "Solid Color Background" item */
    item = g_hash_table_lookup (capplet->wphash, "(none)");
    if (item == NULL)
    {
	item = moblin_wp_item_new ("(none)", capplet->wphash, capplet->thumbs);
	if (item) 
	{
	    wp_props_load_wallpaper (item->filename, item, capplet);
	}
    }

    if((original_filename == NULL)||
       (original_filename[0] == '\0'))
    {
	/* handle the Solid Color Background case */
	if(item)
	{
	    moblin_wp_capplet_scroll_to_item (capplet, item);
	}
    }
    else
    {
	/* handle the current wallpaper image case */

	if(g_utf8_validate(original_filename, -1, NULL)&&
	   g_file_test(original_filename, G_FILE_TEST_EXISTS))
	{
	    imagepath = g_strdup(original_filename);
	}
	else
	{
	    imagepath = g_filename_from_utf8(original_filename,
		-1, NULL, NULL, NULL);
	}
	if((item = g_hash_table_lookup(capplet->wphash, 
	    imagepath)) == NULL)
	{
	    item = moblin_wp_add_image (capplet, imagepath);
	}
	moblin_wp_capplet_scroll_to_item (capplet, item);
	g_free (imagepath);
    }

    /* if there are lots of pictures, add in the jumpslider */
    num = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(capplet->model), NULL);
    if(num > 9)
    {
	g_signal_connect (G_OBJECT (capplet->treeview), "cursor-changed",
		    G_CALLBACK (scroll_treeview), capplet);
	gtk_range_set_range(GTK_RANGE(capplet->jumpslider), 0, num+9);
	scroll_treeview (capplet->treeview, capplet);
	gtk_widget_show_all(capplet->vbox_quick);
    }

    loadphase = LOAD_PHASE3;
    g_thread_create(moblin_wp_load_pixbufs, capplet, FALSE, NULL);

    return FALSE;
}

/* LOAD PHASE 1 */
/* load images thread, this is spawned in the background */
/* so that the user can operate the GUI while the images */
/* are being found and cataloged. The disk is being accessed */
/* hard in this function, so it can take several seconds if */
/* there are a thousand or more photos */
static gpointer
moblin_wp_load_images (gpointer data)
{
  MoblinWPCapplet * capplet = (MoblinWPCapplet *) data;

    load_images (capplet);

    /* displaying the entries has to be done in */
    /* the main loop in order to be in sync with */
    /* the dialog GUI */
    loadphase = LOAD_PHASE2;
    g_idle_add (moblin_wp_display_images, capplet);

    return NULL;
}

static gint 
moblin_wp_list_sort (GtkTreeModel * model,
	GtkTreeIter * a, GtkTreeIter * b,
	MoblinWPCapplet * capplet)
{
  gchar * foo, * bar;
  gint retval;

  gtk_tree_model_get (model, a, 2, &foo, -1);
  gtk_tree_model_get (model, b, 2, &bar, -1);

  if (!strcmp (foo, "(none)")) {
    retval = -1;
  } else if (!strcmp (bar, "(none)")) {
    retval =  1;
  } else {
    retval = -1;
  }

  g_free (foo);
  g_free (bar);

  return retval;
}

static void gconf_file_changed (GConfClient * client, guint id,
				   GConfEntry * entry,
				   MoblinWPCapplet * capplet) {
  GtkTreeSelection * selection;
  GtkTreeModel * model;
  GtkTreeIter iter;
  MoblinWPItem * item;
  gchar * wpfile, * selected;
  const gchar * uri;

  uri = gconf_value_get_string (entry->value);
  if (g_utf8_validate (uri, -1, NULL) && g_file_test (uri, G_FILE_TEST_EXISTS))
    wpfile = g_strdup (uri);
  else
    wpfile = g_filename_from_utf8 (uri, -1, NULL, NULL, NULL);

  item = g_hash_table_lookup (capplet->wphash, wpfile);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (capplet->treeview));
  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    gtk_tree_model_get (model, &iter, 2, &selected, -1);
    if (strcmp (selected, wpfile) != 0) {
      if (item != NULL) {
	moblin_wp_capplet_scroll_to_item (capplet, item);
      } else {
	item = moblin_wp_add_image (capplet, wpfile);
	moblin_wp_capplet_scroll_to_item (capplet, item);
      }
    }
    g_free (wpfile);
    g_free (selected);
  }
}

static void 
gconf_color_changed (GConfClient * client, guint id,
	GConfEntry * entry, MoblinWPCapplet * capplet) 
{
  GdkColor color;
  const gchar * colorhex;

    colorhex = gconf_value_get_string (entry->value);
    if(strcmp(colorhex, master_color_value))
    {
	gdk_color_parse (colorhex, &color);
	gtk_color_button_set_color (GTK_COLOR_BUTTON (capplet->pc_picker), &color);
	moblin_wp_color_changed (capplet, FALSE);
    }
}

static GtkWidget *
wallpaper_properties_init (void) 
{
  MoblinWPCapplet * capplet;
  GladeXML * dialog;
  GtkCellRenderer * renderer;
  GtkTreeViewColumn * column;
  GtkTreeSelection * selection;
  GdkCursor * cursor;
  GtkWidget *widget, *box;
  GdkColor color1;

  gnome_vfs_init ();

  gtk_rc_parse_string ("style \"wp-tree-defaults\" {\n"
		       "  GtkTreeView::horizontal-separator = 6\n"
		       "} widget_class \"*TreeView*\""
		       " style \"wp-tree-defaults\"\n"
		       "style \"wp-dialog-defaults\" {\n"
		       "  GtkDialog::content-area-border = 0\n"
		       "  GtkDialog::action-area-border = 12\n"
		       "} widget_class \"*Dialog*\""
		       " style \"wp-dialog-defaults\"\n");

  capplet = g_new0 (MoblinWPCapplet, 1);
  capplet->client = gconf_client_get_default ();

  /* get the gconf values */
  original_filename = gconf_client_get_string (capplet->client,
	BGKEY(WP_FILE), NULL);
  master_color_value = gconf_client_get_string (capplet->client,
	BGKEY(BG_PCOLOR), NULL);
  if(master_color_value == NULL)
	master_color_value = strdup("#000000");
  original_color_value = g_strdup(master_color_value);

  gdk_color_parse (master_color_value, &color1);
  master_color = gdk_color_copy (&color1);

  /* setup the gconf callbacks */
  gconf_client_add_dir (capplet->client, GCONF_BACKGROUND_DIR,
			GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
  capplet->gconf_cb_ids[0] = gconf_client_notify_add (capplet->client,
			   BGKEY(WP_FILE),
			   (GConfClientNotifyFunc) gconf_file_changed,
			   capplet, NULL, NULL);
  capplet->gconf_cb_ids[1] = gconf_client_notify_add (capplet->client,
			   BGKEY(BG_PCOLOR),
			   (GConfClientNotifyFunc) gconf_color_changed,
			   capplet, NULL, NULL);

  capplet->wphash = g_hash_table_new (g_str_hash, g_str_equal);
  capplet->thumbs = gnome_thumbnail_factory_new (GNOME_THUMBNAIL_SIZE_NORMAL);

  /* setup the dialog */
  dialog = glade_xml_new(MA_GLADE_DIR "/background.glade", NULL, NULL);
  capplet->window = glade_xml_get_widget (dialog, "moblin_wp_properties");
  gtk_widget_realize (capplet->window);

  /* reparent the treeview from the scrolled window to a moko */
  capplet->treeview = glade_xml_get_widget (dialog, "wp_tree");
  capplet->mokoscroll = moko_finger_scroll_new();
  gtk_widget_reparent(capplet->treeview, capplet->mokoscroll);

  /* destroy the scrolled window */
  widget = glade_xml_get_widget(dialog, "scrolledwindow1");
  box = glade_xml_get_widget(dialog, "wp_hbox");
  gtk_container_remove(GTK_CONTAINER(box), widget);

  /* add the moko where the scrolled window used to be */
  gtk_box_pack_start(GTK_BOX(box), capplet->mokoscroll, TRUE, TRUE, 0);
  gtk_box_reorder_child(GTK_BOX(box), capplet->mokoscroll, 0);
  gtk_widget_show (capplet->mokoscroll);

  capplet->model = GTK_TREE_MODEL (gtk_list_store_new (3, GDK_TYPE_PIXBUF,
						       G_TYPE_STRING,
						       G_TYPE_STRING));

  gtk_tree_view_set_model (GTK_TREE_VIEW (capplet->treeview), capplet->model);

  column = gtk_tree_view_column_new ();
  renderer = gtk_cell_renderer_pixbuf_new ();
  gtk_tree_view_column_pack_start (column, renderer, FALSE);
  gtk_tree_view_column_set_attributes (column, renderer,
				       "pixbuf", 0,
				       NULL);

  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, FALSE);
  gtk_tree_view_column_set_attributes (column, renderer,
				       "markup", 1,
				       NULL);
  gtk_tree_view_column_set_spacing (column, 6);

  gtk_tree_view_append_column (GTK_TREE_VIEW (capplet->treeview), column);

  gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (capplet->model), 2,
				   (GtkTreeIterCompareFunc) moblin_wp_list_sort,
				   capplet, NULL);

  gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (capplet->model),
					2, GTK_SORT_ASCENDING);

  /* setup all the callbacks for the GUI buttons */
  capplet->pc_picker = glade_xml_get_widget (dialog, "pcpicker");
  g_signal_connect (G_OBJECT (capplet->pc_picker), "color-set",
		    G_CALLBACK (moblin_wp_scolor_changed), capplet);

  capplet->jumpfirst = glade_xml_get_widget (dialog, "firstjump");
  g_signal_connect (G_OBJECT (capplet->jumpfirst), "pressed",
		    G_CALLBACK (jump_to_first_item), capplet);

  capplet->jumplast = glade_xml_get_widget (dialog, "lastjump");
  g_signal_connect (G_OBJECT (capplet->jumplast), "pressed",
		    G_CALLBACK (jump_to_last_item), capplet);

  capplet->jumpslider = glade_xml_get_widget (dialog, "quickjump");
  g_signal_connect (G_OBJECT (capplet->jumpslider), "value-changed",
		    G_CALLBACK (jump_to_nth_item), capplet);

  capplet->vbox_quick = glade_xml_get_widget (dialog, "vbox_quick");
  gtk_widget_hide_all(capplet->vbox_quick);

  g_signal_connect (G_OBJECT (capplet->window), "response",
		    G_CALLBACK (wallpaper_properties_clicked), capplet);

  /* initialize the color picker dialog */
  gtk_color_button_set_color (GTK_COLOR_BUTTON (capplet->pc_picker),
				master_color);

  gtk_widget_show (capplet->window);

  cursor = gdk_cursor_new_for_display (gdk_display_get_default (),
				       GDK_WATCH);
  gdk_window_set_cursor (capplet->window->window, cursor);
  gdk_cursor_unref (cursor);

  loadphase = LOAD_PHASE1;
  g_thread_create(moblin_wp_load_images, capplet, TRUE, NULL);

  /* when the selection changes, set it to store the new settings */
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (capplet->treeview));
  g_signal_connect (G_OBJECT (selection), "changed",
		    G_CALLBACK (moblin_wp_props_wp_selected), capplet);

  return capplet->window;
}

int main(int argc, char *argv[], char *env[])
{
  static GtkWidget * window;

  gtk_init(&argc, &argv);

  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);

  window = wallpaper_properties_init ();

  gtk_main();
  return 0;
}
