//
// =========================================================================
//
//
//
//  22 July 2001
//
//  SGI make:
//  cc -o ncont ncont.cpp cparser.c ccplib.c mmdb.a -lm -lC
//
//
//  Command line:
//
//   ncont  xyzin pdbfile <<eof
//   ? keyword  parameter(s)
//   ? keyword  parameter(s)
//   ? .......  .........
//   ? eof
//
// -------------------------------------------------------------------------
//
//   Input syntax:
//
//   Keyword    Parameter(s)
//  ~~~~~~~~~  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//
//   source     selection of source atom set
//
//   target     selection of target atom set
//
//   mindist    minimal contact distance (in A), default 0.0
//
//   maxdist    maximal contact distance (in A), default 4.0
//
//   seqdist    minimal "sequence distance", or difference in indices
//              (_not_ in sequence numbers) of residues between those
//              containing the contacting atoms. The default value is
//              1, which means that contacting atoms may be found in
//              the neighbouring and farer residues, but not in the
//              same residue).
//
//   sort       [off|source|target|distance]  [inc|dec]
//              ('off' and 'inc' default). This parameter specifies
//              the subject and order for sorting the contacts on
//              output. For example, 'sort source dec' means sorting
//              by decreasing index of the source atoms (index is
//              essentially the atom's serial number in a correct PDB
//              file). Subject 'off' (default) means no sorting.
//
//   cells      [off|0|1|2]
//              ('off' default). If cells is set to 'off', the program
//              seeks contacts only between the atoms found in the
//              coordinate file. Other values induce construction of
//              unit cell(s) and looking for contacts also between source
//              atoms and target atoms generated in the cell(s).
//                 0  generates only one (primary) unit cell, in which
//                    the source and target atoms are originally found
//                 1  generates the primary cell +/- 1 cell in all
//                    directions (27 cells in total)
//                 2  generates the primary cell +/- 2 cells in all
//                    directions (125 cells in total)
//
//   symmetry   input of the space group symmetry name, e.g. 'P 21 21 21'
//              (without quotation marks, spaces _are_ significant).
//              This parameter is mandatory if coordinate file does not
//              specify the space group symmetry.
//
//   geometry   a b c alpha beta gamma
//              input of the unit cell dimensions (space-separated
//              real numbers). This parameter is mandatory if coordinate
//              file does not specify the cell parameters.
//
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//
//   Specification of the selection sets (parameters for keywords
//   'source' and 'target'):
//
//   /mdl/chn/s1.i1-s2.i2/at[el]:aloc | selcode
//   /mdl/chn/*(res).ic/at[el]:aloc | selcode
//
//   where no spaces are allowed within the selection line (until the
//   optional " | selcode". The slashes separate the hierarchical levels
//   of models, chains, residues and atoms.
//
//   Notations:
//
//    mdl   - the model's serial number or 0 or '*' for any model
//            (default).
//    chn   - the chain ID or list of chain IDs like 'A,B,C' or
//            '*' for any chain (default).
//    s1,s2 - the starting and ending residue sequence numbers
//            or '*' for any sequence number (default).
//    i1,i2 - the residues insertion codes or '*' for any
//            insertion code. If the sequence number other than  
//            '*' is specified, then insertion code defaults to ""
//            (no insertion code), otherwise the default is '*'.
//    res   - residue name or list of residue names like 'ALA,SER'
//            or '*' for any residue name (default)
//    at    - atom name or list of atom names like 'CA,N1,O' or
//            '*' for any atom name (default)
//    el    - chemical element name or list of chemical element
//            names like 'C,N,O', or '*' for any chemical element
//            name (default)
//    aloc  - the alternative location indicator or list of
//            alternative locations like 'A,B,C', or '*' for any
//            alternate location. If the atom name and chemical
//            element name is specified (both may be '*'), then
//            the alternative location indicator defaults to ""
//            (no alternate location), otherwise the default is
//             '*'.
//
//   Values for chain IDs, residue names, atom names, chemical element
//   names and alternative location indicators may be negated by
//   prefix '!'. For example, '!A,B,C' for the list of chain names
//   means 'any chain ID but A,B,C'.
//
//    selcode - code for repeating selections:
//           OR   logical "or"  with previous selection (default)
//           AND  logical "and" with previous selection
//           XOR  logical "xor" with previous selection
//
//   selcode is useful only when you cannot achieve the desired
//   selection by single source/target statements.
//
//   Generally, any hierarchical element as well as the selection
//   code may be omitted, in which case it is replaced for
//   default (see above). This makes the following examples valid:
//
//    *                   select all atoms
//    /1                  select all atoms in model 1
//    A,B                 select all atoms in chains A and B in
//                        all models
//    /1//                select all atoms in chain without chainID
//                        in model 1
//    /*/,A,B/            select all atoms in chain without chainID,
//                        chain A and B in all models
//    33-120              select all atoms in residues 33. to 120.
//                        in all chains and models
//    A/33.A-120.B        select all atoms in residues 33.A to
//                        120.B in chain A only, in all models
//    A/33.-120.A/[C]     select all carbons in residues 33. to
//                        120.A in chain A, in all models
//    CA[C]               select all C-alphas in all
//                        models/chains/residues
//    A//[C]              select all carbons in chain A, in all models
//    (!ALA,SER)          select all atoms in any residues but
//                        ALA and SER, in all models/chains
//    /1/A/(GLU)/CA[C]    select all C-alphas in GLU residues of
//                        chain A, model 1
//    /1/A/*(GLU)./CA[C}: same as above
//    [C]:,A              select all carbons without alternative
//                        location indicator and carbons in alternate
//                        location A
//
//    NOTE: if a selection contains comma(s), the selection sentence must
// be embraced by quotation marks, which indicate to the input parser that
// the sentence is a single input parameter rather than a set of comma-
// separated arguments.
//
// -------------------------------------------------------------------------
//
//  Examples:
//
//  1. Find contacts between chain A and chain B:
//
//     ncont  xyzin rnase.pdb <<eof
//     ? source  A
//     ? target  B
//     ? maxdist 4
//     ? eof
//
//     Output:
// 
//  ###############################################################
//  ###############################################################
//  ###############################################################
//  ### CCP PROGRAM SUITE: ncont  ##########
//  ###############################################################
//  User:   Run date:   Run time:
// 
// 
//  Please reference: Collaborative Computational Project, Number 4. 1994.
//  The CCP4 Suite: Programs for Protein Crystallography. Acta Cryst. D50, 760-763.
//  as well as any specific reference in the program write-up.
//
//
//  ---------------------------------------------------------------------------
//
//  PDB file rnase.pdb has been read in.
//
//  ---------------------------------------------------------------------------
//   Input cards
//
// Data line--- source  A
// Data line--- target  B
// Data line--- maxdist 4
// 
//  ---------------------------------------------------------------------------
// 
//  Selected  751  source atoms
//  Selected  751  target atoms
// 
//  ---------------------------------------------------------------------------
// 
//   14 contacts found:
// 
//       SOURCE ATOMS              TARGET ATOMS        DISTANCE
// 
//  /1/A/ 40(ARG). / NH2[ N]:  /1/B/ 61(GLY). / O  [ O]:   3.07
//                             /1/B/ 63(ARG). / N  [ N]:   3.96
//  /1/A/ 40(ARG). / CZ [ C]:  /1/B/ 61(GLY). / O  [ O]:   3.90
//  /1/A/ 40(ARG). / CG [ C]:  /1/B/ 61(GLY). / CA [ C]:   3.87
//  /1/A/ 40(ARG). / C  [ C]:  /1/B/ 61(GLY). / N  [ N]:   3.83
//  /1/A/ 40(ARG). / O  [ O]:  /1/B/ 60(PRO). / C  [ C]:   3.74
//                             /1/B/ 61(GLY). / N  [ N]:   2.71
//                             /1/B/ 61(GLY). / CA [ C]:   3.29
//  /1/A/ 41(GLU). / O  [ O]:  /1/B/ 60(PRO). / CB [ C]:   3.72
//  /1/A/ 42(SER). / OG [ O]:  /1/B/ 58(ILE). / CG2[ C]:   3.69
//  /1/A/ 42(SER). / CB [ C]:  /1/B/ 58(ILE). / CG2[ C]:   3.73
//  /1/A/ 46(THR). / CG2[ C]:  /1/B/ 30(TYR). / OH [ O]:   3.56
//                             /1/B/ 30(TYR). / CZ [ C]:   3.75
//  /1/A/ 46(THR). / OG1[ O]:  /1/B/ 58(ILE). / CD1[ C]:   3.50
//  ncont:  Normal termination
// 
//
//  2. Find contacts between carbons of chain A and whole chains B
//     of all same-cell symmetry mates:
//
//     ncont  xyzin rnase.pdb <<eof
//     ? source  A//[C]
//     ? target  B
//     ? maxdist 4
//     ? cells   0
//     ? symmetry P 21 21 21
//     ? eof 
//
//     Output:
// 
//  ###############################################################
//  ###############################################################
//  ###############################################################
//  ### CCP PROGRAM SUITE: ncont  ##########
//  ###############################################################
//  User:   Run date:   Run time:
// 
// 
//  Please reference: Collaborative Computational Project, Number 4. 1994.
//  The CCP4 Suite: Programs for Protein Crystallography. Acta Cryst. D50, 760-763.
//  as well as any specific reference in the program write-up.
//
//
//  ------------------------------------------------------------------------------
//
//  PDB file rnase.pdb has been read in.
//
//  ------------------------------------------------------------------------------
//   Input cards
//
// Data line--- source A//[C]
// Data line--- target B
// Data line--- cells 0
// Data line--- symm P 21 21 21
// 
//  ---------------------------------------------------------------------------
// 
//  Selected  465  source atoms
//  Selected  751  target atoms
// 
//  Unit cell numbering:   1
// 
//  Symmetry transformation matrices in cell 1:
// 
//   Operation X,Y,Z
//       [    1.0000     0.0000     0.0000]  [    0.0000]
//   T = [    0.0000     1.0000     0.0000]  [    0.0000]
//       [    0.0000     0.0000     1.0000]  [    0.0000]
// 
//   Operation 1/2-X,-Y,1/2+Z
//       [   -1.0000     0.0000     0.0000]  [   32.4500]
//   T = [    0.0000    -1.0000     0.0000]  [    0.0000]
//       [    0.0000     0.0000     1.0000]  [   19.3950]
// 
//   Operation -X,1/2+Y,1/2-Z
//       [   -1.0000     0.0000     0.0000]  [    0.0000]
//   T = [    0.0000     1.0000     0.0000]  [   39.1600]
//       [    0.0000     0.0000    -1.0000]  [   19.3950]
// 
//   Operation 1/2+X,1/2-Y,-Z
//       [    1.0000     0.0000     0.0000]  [   32.4500]
//   T = [    0.0000    -1.0000     0.0000]  [   39.1600]
//       [    0.0000     0.0000    -1.0000]  [    0.0000]
// 
//  ---------------------------------------------------------------------------
//
//   14 contacts found:
//
//       SOURCE ATOMS               TARGET ATOMS         DISTANCE CELL   SYMMETRY
//
//  /1/A/  40(ARG). / CZ [ C]:  /1/B/  61(GLY). / O  [ O]:   3.90 111 X,Y,Z
//  /1/A/  40(ARG). / CG [ C]:  /1/B/  61(GLY). / CA [ C]:   3.87 111 X,Y,Z
//  /1/A/  40(ARG). / C  [ C]:  /1/B/  61(GLY). / N  [ N]:   3.83 111 X,Y,Z
//  /1/A/  42(SER). / CB [ C]:  /1/B/  58(ILE). / CG2[ C]:   3.73 111 X,Y,Z
//  /1/A/  46(THR). / CG2[ C]:  /1/B/  30(TYR). / OH [ O]:   3.56 111 X,Y,Z
//                              /1/B/  30(TYR). / CZ [ C]:   3.75 111 X,Y,Z
//  /1/A/   6(VAL). / CG2[ C]:  /1/B/  95(THR). / O  [ O]:   3.89 111 1/2+X,1/2-Y,-Z
//  /1/A/   6(VAL). / CA [ C]:  /1/B/  95(THR). / CB [ C]:   3.86 111 1/2+X,1/2-Y,-Z
//                              /1/B/  95(THR). / O  [ O]:   3.81 111 1/2+X,1/2-Y,-Z
//  /1/A/   6(VAL). / C  [ C]:  /1/B/  95(THR). / O  [ O]:   3.70 111 1/2+X,1/2-Y,-Z
//  /1/A/   7(CYS). / CB [ C]:  /1/B/  95(THR). / O  [ O]:   3.14 111 1/2+X,1/2-Y,-Z
//                              /1/B/  96(CYS). / CA [ C]:   3.99 111 1/2+X,1/2-Y,-Z
//  /1/A/   7(CYS). / CA [ C]:  /1/B/  95(THR). / O  [ O]:   3.55 111 1/2+X,1/2-Y,-Z
//  /1/A/  10(ALA). / CB [ C]:  /1/B/  95(THR). / O  [ O]:   3.86 111 1/2+X,1/2-Y,-Z
//  ncont:  Normal termination
//
// =========================================================================
//

#ifndef  __STDLIB_H
#include <stdlib.h>
#endif

#ifndef  __STRING_H
#include <string.h>
#endif

#ifndef  __MMDB_Manager__
#include "mmdb_manager.h"
#endif


#define __CPlusPlus

#ifndef  __CCPLib__
#include "ccplib.h"
#endif

#ifndef  __CParser__
#include "cparser.h"
#endif


// -----------  Interpreter-assisting functions  ---------------

int  GetSelectionKey ( int ntok, PARSERTOKEN * token, int & selKey )  {
  selKey = SKEY_OR;
  if (ntok>3)  {
    if (keymatch("|",token[2].word)) {
      if (ntok>4)  {
        if (keymatch("OR" ,token[3].word))  selKey = SKEY_OR;
        else if (keymatch("XOR",token[3].word))  selKey = SKEY_XOR;
        else if (keymatch("AND",token[3].word))  selKey = SKEY_AND;
        else  {
          ccperror ( 1,"wrong selection key" );
          return -98;
        }
      } else  {
        ccperror ( 1,"no selection key" );
        return -98;
      }
    } else  {
      ccperror ( 1,"wrong syntax of the selection statement" );
      return -98;
    }
  }
  return 0;
}


//  -------------------------------------------------------------

long  makeGroupID ( int i, int j, int k, int m, int ncells )  {
  return  (i+ncells) | ((j+ncells)<<8) | ((k+ncells)<<16) | (m<<24);
}

void  parseGroupID ( long group, int & i, int & j, int & k, int & m )  {
  i = (int(group)     & 0x000000FF) + 1;
  j = (int(group>>8)  & 0x000000FF) + 1;
  k = (int(group>>16) & 0x000000FF) + 1;
  m =  int(group>>24) & 0x0000FFFF;
}

int  hesh4 ( int i, int j, int k, int m, int ncells, int nsymops )  {
int  n=2*ncells+1;
  return  (i+ncells)*n*n*nsymops +
          (j+ncells)*n*nsymops +
          (k+ncells)*nsymops +
          m;
}


// -------------------  Main program  --------------------------

int main ( int argc, char ** argv, char ** env )  {
int          RC,lcount,i,j,k,m,ns,nh,ci,id;
char         S[500],S1[500];
CMMDBManager MMDB;
CSymOp       SymOp;
realtype     minDist,maxDist;
int          seqDist;
int          selSrc,selTrg;
int          selKey;
int          ncells,nSymOps;
char         symGroup[500];
PPCAtom      Source ,Target;
int          nSource,nTarget;
mat44        TMatrix;
PSContact    contact;
int          ncontacts,sortmode;
ivector      contstat;
pstr *       symOpTitle;

// input parser parameters
int           ntok=0;
char          line[201],*key;
PARSERTOKEN * token=NULL;
PARSERARRAY * parser;



  //  1.  General CCP4 initializations
  ccp4fyp         ( argc, argv );
  ccp4ProgramName ( argv[0] );
  ccp4rcs         ( argv[0] );


  //  2.  Make routine initializations, which must always be done
  //      before working with MMDB
  InitMatType();


  //  3.  Read coordinate file.
  //    3.1 Set all necessary read flags -- check with the top of
  //        file  mmdb_file.h  as needed

  printf (
    "\n"
    " ----------------------------------------------------"
    "--------------------------\n"
    "\n" );

  MMDB.SetFlag ( MMDBF_PrintCIFWarnings );

  //    3.2 Read coordinate file by its logical name
  RC = MMDB.ReadCoorFile1 ( "XYZIN" );

  //    3.3 Check for possible errors:
  if (RC) {
    //  An error was encountered. MMDB provides an error messenger
    //  function for easy error message printing.
    printf ( " ***** ERROR #%i READ:\n\n %s\n\n",RC,GetErrorDescription(RC) );
    //  Location of the error may be identified as precise as line
    //  number and the line itself (PDB only. Errors in mmCIF are
    //  located by category/item name. Errors of reading BINary files
    //  are not locatable and the files are not editable). This
    //  information is now retrieved from MMDB input buffer:
    MMDB.GetInputBuffer ( S,lcount );
    if (lcount>=0) 
      printf ( "       LINE #%i:\n%s\n\n",lcount,S );
    else if (lcount==-1)
      printf ( "       CIF ITEM: %s\n\n",S );
    //  now quit
    return 1;
  } else  {
    //  MMDB allows to identify the type of file that has been just
    //  read:
    switch (MMDB.GetFileType())  {
      case MMDB_FILE_PDB    : printf ( " PDB"         );  break;
      case MMDB_FILE_CIF    : printf ( " mmCIF"       );  break;
      case MMDB_FILE_Binary : printf ( " MMDB binary" );  break;
      default : printf ( " Unknown (report as a bug!)" );
    }
    printf ( " file %s has been read in.\n",getenv("XYZIN") );
  }




  //  4.  Interprete cards from standard input stream

  printf (
    "\n"
    " ----------------------------------------------------"
    "--------------------------\n"
    "  Input cards\n\n" );

  //  4.1 We will select atoms in the course of reading the input
  //      cards. Each _new_ selection starts with creation of
  //      the selection handle (a handle may be used in several
  //      selections)
  selSrc  = MMDB.NewSelection();
  selTrg  = MMDB.NewSelection();

  //  4.2 Assign default values for minimal and maximal contact
  //      distances and the sequence distance
  minDist  = 0.0;
  maxDist  = 4.0;
  seqDist  = 1;
  sortmode = CNSORT_OFF;
  ncells   = -1;

  /* Initialise a parser array used by cparser
     This is used to return the tokens and associated info
     Set maximum number of tokens per line to 20 */
  parser = (PARSERARRAY *) cparse_start(20);

  if (parser == NULL) ccperror ( 1,"Couldn't create parser array" );

  /* Example of how to set up cparser to use non-default delimiters
     In this example, only allow tabs as delimiters */
  /*
     if (! cparse_delimiters(parser,"\t",""))
        ccperror(1,"Couldn't reset delimiters");
  */
  
  /* Set some convenient pointers to members of the parser array */
  key   = parser->keyword;
  token = parser->token;

  /* Read lines from stdin until END/end keyword is entered or
     EOF is reached */
  RC   = 0;

  while (!RC) {

    /* Always blank the line before calling cparser
       to force reading from stdin */
    line[0] = '\0';

    /* Call cparser to read input line and break into tokens
       Returns the number of tokens, or zero for eof */
    ntok = cparser(line,200,parser,1);

    if (ntok < 1) {

      /* End of file encountered */
      RC = 111;

    } else {      

      /* Keyword interpretation starts here */

      /* Traditionally CCP4 keywords are case-independent and
	 and only use the first four characters.

	 Use keymatch to compare first four characters of
	 arguments (independent of case), i.e.
	 keymatch("RESO",key)
	 keymatch("reso",key)
	 keymatch("RESOLUTION",key)
	 keymatch("resol",key)
	 keymatch("rEsOlUtIoN",key)
	 would all give the same result.
      */

      if (keymatch("SOURCE",key))  {
	if (ntok<2) {
	  ccperror ( 1,"selection statement not found" );
          RC = -99;
        } else  {
          RC = GetSelectionKey ( ntok,token,selKey );
          if (!RC)  RC = MMDB.Select ( selSrc,STYPE_ATOM,
                                       token[1].fullstring,selKey );
        }
      } else if (keymatch("TARGET",key))  {
	if (ntok<2) {
	  ccperror ( 1,"selection statement not found" );
          RC = -99;
        } else  {
          RC = GetSelectionKey ( ntok,token,selKey );
          if (!RC)  RC = MMDB.Select ( selTrg,STYPE_ATOM,
                                       token[1].fullstring,selKey );
        }
      } else if (keymatch("MINDIST",key))  {
	if (ntok != 2) {
	  ccperror ( 1,"MINDIST requires a single numerical argument" );
	  RC = -100;
	} else if (!token[1].isnumber) {
	  ccperror ( 1,"MINDIST argument must be numerical" );
	  RC = -101;
	} else {
	  minDist = token[1].value;
	  if (minDist<0.0) {
	    ccperror ( 1,"MINDIST argument must be non-negative" );
	    RC = -102;
	  }
	}
      } else if (keymatch("MAXDIST",key))  {
	if (ntok != 2) {
	  ccperror ( 1,"MAXDIST requires a single numerical argument" );
	  RC = -103;
	} else if (!token[1].isnumber) {
	  ccperror ( 1,"MAXDIST argument must be numerical" );
	  RC = -104;
	} else {
	  maxDist = token[1].value;
	  if (maxDist<=0.0) {
	    ccperror ( 1,"MAXDIST argument must be positive" );
	    RC = -105;
	  }
	}
      } else if (keymatch("SEQDIST",key))  {
	if (ntok!= 2) {
	  ccperror ( 1,"SEQDIST requires a single numerical argument" );
	  RC = -106;
	} else if (!token[1].isnumber) {
	  ccperror ( 1,"SEQDIST argument must be numerical" );
	  RC = -107;
	} else {
	  seqDist = mround ( token[1].value );
	  if (seqDist<=0.0) {
	    ccperror ( 1,"SEQDIST argument must be positive" );
	    RC = -108;
	  }
	}
      } else if (keymatch("SORT",key))  {
	if (ntok<2) {
	  ccperror ( 1,"SORT requires 2nd argument - "
                       "OFF | TARGET | SOURCE | DISTANCE" );
	  RC = -109;
        } else  {
          if (keymatch("OFF",token[1].word))  sortmode = CNSORT_OFF;
          else  {
            sortmode = CNSORT_1INC;
            if (keymatch("TARGET",token[1].word))
                sortmode = CNSORT_2INC;
            else if (keymatch("DISTANCE",token[1].word))
                sortmode = CNSORT_DINC;
            if (ntok>2)  {
              if (keymatch("DEC" ,token[2].word))  {
                if (sortmode==CNSORT_1INC)       sortmode = CNSORT_1DEC;
                else if (sortmode==CNSORT_2INC)  sortmode = CNSORT_2DEC;
                                           else  sortmode = CNSORT_DDEC;
              }
            }
          }
        }
      } else if (keymatch("CELLS",key))  {
	if (ntok<2) {
	  ccperror ( 1,"CELLS requires 2nd argument - OFF | 0 | 1 | 2" );
	  RC = -110;
        } else if (keymatch("OFF",token[1].word))  ncells = -1;
        else if (keymatch("0",token[1].word))      ncells = 0;
        else if (keymatch("1",token[1].word))      ncells = 1;
        else if (keymatch("2",token[1].word))      ncells = 2;
        else if (keymatch("3",token[1].word))      ncells = 3;
        else  {
	  ccperror ( 1,"Argument of CELLS not understood" );
	  RC = -111;
        }
      } else if (keymatch("SYMMETRY",key)) {
	if (ntok<2) {
          ccperror ( 1,"SYMMETRY requires 2nd argument, e.g. P 21 21 21" );
          RC = -112;
        } else {
          i = token[ntok-1].iend - token[1].ibeg + 1;
          strcpy_ncss ( symGroup,&line[token[1].ibeg],i );
          symGroup[i] = char(0);
          i = MMDB.SetSpaceGroup ( UpperCase(symGroup) );
          switch (i)  {
            case SYMOP_NoLibFile         : 
                    ccperror ( 1,"symop.lib file not found" );
                    RC = -113;
                  break;
            case SYMOP_UnknownSpaceGroup :
                    ccperror ( 1,"Unknown space symmetry group" );
                    RC = -114;
                  break;
            case SYMOP_NoSymOps          :
                    ccperror ( 1,"no symmetry operations found" );
                    RC = -115;
                  break;
            default : ;
          }
        }
      } else if (keymatch("GEOMETRY",key)) {
	if (ntok!= 7) {
	  ccperror ( 1,"GEOMETRY requires 6 parameters:"
                       " a b c alpha beta gamma" );
	  RC = -116;
	} else {
          i = 1;
          while ((i<ntok) && token[i].isnumber && (token[i].value>0.0)) i++;
	  if (i<ntok) {
	    printf ( "%ith GEOMETRY parameter is not numerical "
                     "or positive",i );
	    RC = -117;
	  } else
            MMDB.SetCell ( token[1].value,token[2].value,token[3].value,
                           token[4].value,token[5].value,token[6].value,0 );
	}
      }	else  {
	printf ( "Unrecognised keyword \"%s\"\n",token[0].fullstring );
	RC = -118;
      }
    }
  }

  if (RC==111)  RC = 0;  // normal return from the parser loop

  /* Clean up parser array */
  cparse_end ( parser );


  //  4.4 Check completeness of crystallographic information
  if ((!RC) && (ncells>=0))  {
    if (!MMDB.isCrystInfo())  {
      ccperror ( 1,"Missing cell parameters (a,b,c,alpha,beta,gamma)" );
      RC = -150;
    } else if (!MMDB.isSpaceGroup())  {
      ccperror ( 1,"Missing space group of symmetry" );
      RC = -151;
    } else if (!MMDB.isTransfMatrix())  {
      ccperror ( 1,"Orthogonalization/fractionalization is impossible" );
      RC = -152;
    } else  {
      nSymOps = MMDB.GetNumberOfSymOps();
      if (nSymOps==0)  {
        ccperror ( 1,"No symmetry operations found" );
        RC = -153;
      }
    }
  }

  if (RC)  return 2;

  //  4.5 Selected atoms may be now accessed through the selection
  //      index. Selection index is merely a vector of pointers
  //      on the selected atoms. Check the function and its
  //      parameters in file  mmdb_selmngr.h
  MMDB.GetSelIndex ( selSrc,Source,nSource );
  MMDB.GetSelIndex ( selTrg,Target,nTarget );

  printf (
    "\n"
    " ----------------------------------------------------"
    "--------------------------\n"
    "\n"
    " Selected  %i  source atoms\n"
    " Selected  %i  target atoms\n",
    nSource,nTarget );

  if (nSource<=0)  {
    ccperror ( 1,"NO SOURCE ATOMS SELECTED" );
    return 3;
  }

  if (nTarget<=0)  {
    ccperror ( 1,"NO TARGET ATOMS SELECTED" );
    return 4;
  }

  if (ncells>=0)  {
    printf ( "\n Unit cell numbering:  " );
    for (i=-ncells;i<=ncells;i++)
      printf ( "%2i",i+ncells+1 );
    printf ( "\n\n Symmetry transformation matrices in cell %i:\n",ncells+1 );
    for (m=0;m<nSymOps;m++)
      if (MMDB.GetTMatrix(TMatrix,m,0,0,0)==SYMOP_Ok)
        printf (  "\n"
          "  Operation %s\n"
          "      [%10.4f %10.4f %10.4f]  [%10.4f]\n" 
          "  T = [%10.4f %10.4f %10.4f]  [%10.4f]\n"
          "      [%10.4f %10.4f %10.4f]  [%10.4f]\n",
          MMDB.GetSymOp(m),
          TMatrix[0][0],TMatrix[0][1],TMatrix[0][2],TMatrix[0][3],
          TMatrix[1][0],TMatrix[1][1],TMatrix[1][2],TMatrix[1][3],
	  TMatrix[2][0],TMatrix[2][1],TMatrix[2][2],TMatrix[2][3] );
  }



  //  5.  Find contacts between the selected source and target sets
  //      of  atoms. Here we use dynamical allocation of contacts'
  //      index. See details and the function description in file
  //      mmdb_selmngr.h

  contact   = NULL;  // this is a _must_ for dynamically allocated index
  ncontacts = 0;     // no contacts in the begining

  contstat  = NULL;

  if (ncells<0)  {
    // no unit cell modeling is required; find contacts between
    // atoms given in coordinate file only
    MMDB.SeekContacts ( Source,nSource,Target,nTarget,
                        minDist,maxDist,seqDist,
                        contact,ncontacts,0,NULL );
    symOpTitle = NULL;
  } else  {
    // construct all necessary cells for target atoms and run the
    // contact-looking procedure for each cell. In this implementation,
    // no new chains/residues/atoms are created; the contact-looking
    // procedure merely adjust coordinates by applying a transformation
    // matrix for symmetry mates and neighbouring unit cells.
    ns = hesh4 ( ncells,ncells,ncells,nSymOps,ncells,nSymOps );
    GetVectorMemory ( contstat,ns,0 );
    symOpTitle = new pstr[ns];
    for (i=0;i<ns;i++)
      symOpTitle[i] = NULL;
    ns = 0;
    for (i=-ncells;i<=ncells;i++)
      for (j=-ncells;j<=ncells;j++)
        for (k=-ncells;k<=ncells;k++)
          for (m=0;m<nSymOps;m++)  {
            //   GetMatrix(..) calculates the transformation matrix
            // for mth symmetry operation, which places atoms
            // into unit cell, shifted by i,j,k in a,b,c - directions,
            // respectively (in fractional space), from the principal
            // unit cell of the coordinate file. 
            if (MMDB.GetTMatrix(TMatrix,m,i,j,k)!=SYMOP_Ok)  {
              printf (
                " *** unexpected error at calculating the transformation\n"
                "     matrix for symmetry operation %i in cell %i%i%i.\n",
                m,i+ncells+1,j+ncells+1,k+ncells+1 );
            } else  {
              //   SeekContacts(..) finds all contacts between the
              // source and target atoms, transforming the target
              // according to the (optional) transformation matrix.
              //   makeGroupID(..) encodes the values of i,j,k,m
              // into single long integer to store it with the
              // contact found. These values are then used for
              // printout.
              MMDB.SeekContacts ( Source,nSource,Target,nTarget,
                                  minDist,maxDist,seqDist,
                                  contact,ncontacts,0,&TMatrix,
                                  makeGroupID(i,j,k,m,ncells) );
            }
            // store contact statistics by cell and operation
            contstat[ns++] = ncontacts;
            // compile the symmetry operation title
            nh = hesh4 ( i,j,k,m,ncells,nSymOps );
            if (!symOpTitle[nh])
              symOpTitle[nh] = new char[100];
            SymOp.SetSymOp   ( MMDB.GetSymOp(m) );
            SymOp.GetTMatrix ( TMatrix );
            TMatrix[0][3] += i;
            TMatrix[1][3] += j;
            TMatrix[2][3] += k;
            SymOp.CompileOpTitle ( symOpTitle[nh],TMatrix,False );
          }
  }

  //  5.  Print contacts. Check with description of the structure
  //      SContact in file mmdb_selmngr.h .

  if (ncontacts<=0)  {

    printf ( 
      "\n"
      " ----------------------------------------------------"
      "--------------------------\n"
      "\n"
      "  NO CONTACTS FOUND.\n"
      "\n" );

  } else  {

    //  Sort contacts according to the requested sorting mode.
    SortContacts ( contact,ncontacts,sortmode );

    printf ( 
      "\n"
      " ----------------------------------------------------"
      "--------------------------\n"
      "\n"
      "  %i contacts found:\n"
      "\n"
      "      SOURCE ATOMS               TARGET ATOMS         DISTANCE",
      ncontacts );

    if (ncells>=0)  printf ( " CELL   SYMMETRY" );
    printf ( "\n\n" );

    id = MinInt4;
    for (ci=0;ci<ncontacts;ci++)  {
      Target[contact[ci].id2]->GetAtomIDfmt ( S1 );
      Source[contact[ci].id1]->GetAtomIDfmt ( S  );
      if (id!=contact[ci].id1)  id = contact[ci].id1;
                          else  memset ( S,' ',strlen(S) );
      if (ncells>=0)  {
        parseGroupID ( contact[ci].group,i,j,k,m );
        nh = hesh4 ( i-ncells-1,j-ncells-1,k-ncells-1,m,ncells,nSymOps );
        printf ( " %s %s %5.2f %1i%1i%1i %s\n",S,S1,contact[ci].dist, i,j,k,
                   symOpTitle[nh] );
      } else
        printf ( " %s %s %5.2f\n",S,S1,contact[ci].dist );
    }

  }

  if (ncells>=0)  {
    for (i=ns-1;i>0;i--)
      contstat[i] -= contstat[i-1];
    printf (
 "\n\n"
 "--------------------------------------------------------------------------"
 "\n\n   Contact statistics:"
 "\n\n"
 "  CELL   FRACT COORD-S   SYM NO   SYM OPERATION              NUM OF CONT-S"
 "\n\n"    );
    ns = 0;
    for (i=-ncells;i<=ncells;i++)
      for (j=-ncells;j<=ncells;j++)
        for (k=-ncells;k<=ncells;k++)
          for (m=0;m<nSymOps;m++)  {
            if (contstat[ns]>0)  {
              nh = hesh4 ( i,j,k,m,ncells,nSymOps );
              printf ( "  %1i%1i%1i      %2i %2i %2i    %5i     %-30s  %7i\n",
                       i+ncells+1,j+ncells+1,k+ncells+1, i,j,k,
                       m,symOpTitle[nh],contstat[ns] );
            }
            ns++;
          }
  }

  printf ( "\n\n  Total %i contacts\n\n"
 "--------------------------------------------------------------------------"
 "\n\n",ncontacts );

  //  since the contact index was dynamically allocated,
  //  the application must dispose it when it is no longer needed:
  if (contact)  delete[] contact;

  FreeVectorMemory ( contstat,0 );

  if (symOpTitle)  {
    ns = hesh4 ( ncells,ncells,ncells,nSymOps,ncells,nSymOps );
    for (i=0;i<ns;i++)
      if (symOpTitle[i])  delete[] symOpTitle[i];
    delete[] symOpTitle;
  }

  ccperror ( 0,"Normal termination" );

  return 0;

}

