//
// =================================================================
//
//  MacroMolecular Data Base (MMDB)
//
//  File  sel_exmp3.cpp
//
//  Example:
//     Selection of atoms and looking for contacts between two
//  selections.
//
//  15 May 2001
//
//  SGI make:  f77 -o sel_exmp3 sel_exmp3.cpp mmdb.a -lm -lC
//
// =================================================================
//

#ifndef  __STRING_H
#include <string.h>
#endif

#ifndef  __MMDB_Manager__
#include "mmdb_manager.h"
#endif



void  PrintInstructions()  {
  printf ( 
    "\n"
    "A MacroMolecular Data Base (MMDB) example:\n"
    "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
    "\n"
    "Selecting atoms and looking for contacts between two\n"
    "selections.\n"
    "\n"
    "Command line syntax:\n"
    "~~~~~~~~~~~~~~~~~~~~\n"
    "\n"
    "sel_exmp3 coor_file\n"
    "\n"
    "where 'coor_file' is a coordinate file in PDB, mmCIF or\n"
    "BINary format.\n"
    "\n"
         );
}


int main ( int argc, char ** argv, char ** env )  {
PCMMDBManager  MMDBManager;
int            RC,lcount;
char           S[500],AID[100];
int            selHnd1,selHnd2;
int            nSelAtoms1,nSelAtoms2,i;
PPCAtom        SelAtom1,SelAtom2;
PSContact      contact;
int            ncontacts;

  if (argc<2)  {
    PrintInstructions();
    return 1;
  }

  //  1.  Make routine initializations, which must always be done
  //      before working with MMDB
  InitMatType();

  //  2.  Create an instance of MMDB
  MMDBManager = new CMMDBManager();

  //  3.  Read coordinate file.
  //    3.1 Set all necessary read flags -- check with the top of
  //        file  mmdb_file.h  as needed
  MMDBManager->SetFlag ( MMDBF_PrintCIFWarnings );

  //    3.2 MMDB provides the following function for reading
  //        coordinate files with automatic format recognition
  //        for PDB, mmCIF and MMDB BINary files:
  RC = MMDBManager->ReadCoorFile ( argv[1] );

  //    3.3 Check for possible errors:
  if (RC) {
    //  An error was encountered. MMDB provides an error messenger
    //  function for easy error message printing.
    printf ( " ***** ERROR #%i READ:\n\n %s\n\n",RC,GetErrorDescription(RC) );
    //  Location of the error may be identified as precise as line
    //  number and the line itself (PDB only. Errors in mmCIF are
    //  located by category/item name. Errors of reading BINary files
    //  are not locatable and the files are not editable). This
    //  information is now retrieved from MMDB input buffer:
    MMDBManager->GetInputBuffer ( S,lcount );
    if (lcount>=0) 
      printf ( "       LINE #%i:\n%s\n\n",lcount,S );
    else if (lcount==-1)
      printf ( "       CIF ITEM: %s\n\n",S );
    //  dispose instance of MMDB and quit:
    delete MMDBManager;
    return 2;
  } else  {
    //  MMDB allows to identify the type of file that has been just
    //  read:
    switch (MMDBManager->GetFileType())  {
      case MMDB_FILE_PDB    : printf ( " PDB"         );  break;
      case MMDB_FILE_CIF    : printf ( " mmCIF"       );  break;
      case MMDB_FILE_Binary : printf ( " MMDB binary" );  break;
      default : printf ( " Unknown (report as a bug!)" );
    }
    printf ( " file %s has been read in.\n",argv[1] );
  }


  //  4.  Select atoms.
  //    4.1 Each _new_ selection starts with creation of selection
  //        handle (a handle may be used in several selections)
  selHnd1 = MMDBManager->NewSelection();
  selHnd2 = MMDBManager->NewSelection();

  //    4.2 Select atoms as needed. Check the selection function
  //        and its parameters in file  mmdb_selmngr.h
  //  first selection - C_alphas of chain A
  MMDBManager->SelectAtoms ( selHnd1, 1,"A",ANY_RES,"*",ANY_RES,"*",
                             "*","CA","C","*" );
  //  second selection - sulphurs of all chains
  MMDBManager->SelectAtoms ( selHnd2, 1,"*",ANY_RES,"*",ANY_RES,"*",
                             "*","*","S","*" );
  //  plus we add /1/A/34/CA to 2nd set to see that it will
  //  produce the same selections like in examples 1 and 2
  //  (sel_exmp1.cpp and sel_exmp2.cpp):
  MMDBManager->SelectAtoms ( selHnd2, 1,"A",34,"",34,"","*","CA","*","*" );

  //    4.3 Selected atoms may be accessed through the selection
  //        index. Selection index is merely a vector of pointers
  //        on the selected atoms. Check the function and its
  //        parameters in file  mmdb_selmngr.h
  MMDBManager->GetSelIndex ( selHnd1, SelAtom1,nSelAtoms1 );
  MMDBManager->GetSelIndex ( selHnd2, SelAtom2,nSelAtoms2 );
  printf ( "  Selection #1:  %i atoms.\n",nSelAtoms1 );
  printf ( "  Selection #2:  %i atoms.\n",nSelAtoms2 );

  //    4.4 Find contacts between 2 selected sets. Here we use
  //        dynamical allocation of contacts' index. See details
  //        and the function description in file  mmdb_selmngr.h
  contact   = NULL;  // this is a _must_ for dynamically allocated index

  //  here we put 2nd selection first because this allows us easier
  //  to output all C_alpha contacts for each sulphur atoms plus
  //  /1/A/34/CA . Otherwise we would obtain all sulphur contacts
  //  plus /1/A/34/CA contacts for each /1/*/*/CA, which is the same
  //  information just arranged differently in array 'contact'.
  MMDBManager->SeekContacts ( SelAtom2,nSelAtoms2,
                              SelAtom1,nSelAtoms1,
                              0.0,10.0,0,contact,ncontacts );
  printf ( "  %i contacts found:\n",ncontacts );

  //  print contacts. Check with description of the structure SContact
  //  in file mmdb_selmngr.h .
  for (i=0;i<ncontacts;i++)
    printf ( " %s <-> %s   %10.4f A\n",
             SelAtom2[contact[i].id1]->GetAtomID(S),
             SelAtom1[contact[i].id2]->GetAtomID(AID),
             contact[i].dist );

  //  since the contact index was dynamically allocated,
  //  the application must dispose it when it is no longer needed:
  if (contact)  delete contact;

  //  5. A formally correct style of programming requires you
  //     to explicitely release any allocated memory before
  //     your application quits.
  //     NOTE: we do not dispose the selection indices SelAtom1
  //     and SelAtom2. they will be disposed by their manager
  //     (which is MMDBManager) automatically. An attempt to
  //     dispose them from the application  will cause a crash.
  if (MMDBManager)  delete MMDBManager;

  return 0;

}



