# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Spectrum and peaklist processing functions.

# load libs
import math

# load modules
import scimath
from nucleus import commfce


# ----
def linearEq(parameters, x):
    """ Define function for linear fit. """
    a, b = parameters
    return a*x + b
# ----


# ----
def quadraticEq(parameters, x):
    """ Define function for quadratic fit. """
    a, b, c = parameters
    return a*x*x + b*x + c
# ----


# ----
def calibrateData(data, parameters, equation='linear'):
    """ Recalculate data points by given equation and parameters. """

    calibrated = []
    for x in range(len(data)):
        point = data[x][:]
        if equation == 'quadratic':
            point[0] = point[0] - quadraticEq(parameters, point[0])
        else:
            point[0] = point[0] - linearEq(parameters, point[0])
        calibrated.append(point)

    return calibrated
# ----


# ----
def cropData(data, minX, maxX):
    """ Crop data to selected range. """

    # get indexes
    i1 = 0
    i2 = len(data)
    for x, point in enumerate(data):
        if point[0] < minX:
            i1 = x+1
        if point[0] > maxX:
            i2 = x
            break

    return data[i1:i2]
# ----


# ----
def referenceCalibration(data, equation='linear'):
    """ Do reference calibration. """

    # calculate differences
    delta = []
    for point in data:
        diff = point[0] - point[1]
        delta.append((point[0], diff))

    # quadratic fitting
    if equation == 'quadratic':
        fit = scimath.leastSquaresFit(quadraticEq, (1, 0, 0), delta)

    # linear fitting
    else:
        fit = scimath.leastSquaresFit(linearEq, (0.5, 0), delta)

    return fit
# ----


# ----
def statisticalCalibration(data, index=0, equation='linear'):
    """ Do statistical calibration. (For protein digests only!!!) """

    # calculate differences
    delta = []
    for point in data:
        if type(point) in (tuple, list):
            point = point[index]
        diff = calcDecimalDiff(point, calcDecimal(point))
        delta.append((point, diff))

    # quadratic fitting
    if equation == 'quadratic':
        fit = scimath.leastSquaresFit(quadraticEq, (1, 0, 0), delta)

    # linear fitting
    else:
        fit = scimath.leastSquaresFit(linearEq, (0.5, 0), delta)

    return fit
# ----


# ----
def calcDecimal(mass):
    """ Calculate decimal value for given mass. (For peptides only!)"""

    decimal = math.modf(round(mass)*1.00048)[0]
    return decimal
# ----


# ----
def calcDecimalDiff(mass, decimal):
    """ Calculate decimal difference for given values. """

    diff = math.modf(mass)[0] - decimal
    if diff < -0.5:
        diff = 1 + diff
    elif diff > 0.5:
        diff = 1 - diff

    return diff
# ----


# ----
def makeCalibrationCurve(minX, maxX, parameters):
    """ Count data to simulate calibration curve. """

    points = []
    seqment = (maxX - minX) / 100
    for x in range(100):
        mz = minX + (x * seqment)
        if parameters[2] == 'quadratic':
            diff = quadraticEq(parameters[0], mz)
        else:
            diff = linearEq(parameters[0], mz)
        points.append([mz, diff])

    return points
# ----
