# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: View sorted list of all fragments.

# load libs
import wx
from copy import copy

# load modules
from nucleus import mwx
from dlg_listpanel import listPanel


class mFragListPanel(listPanel):
    """ View list of all fragments. """

    # ----
    def __init__(self, parent, config, data):
        listPanel.__init__(self, parent, "Fragments List")

        self.config = config
        self.data = copy(data)

        # set defaults
        self.listType = 'mFrag'
        self.massCol = 0
        self.sortBy = 0

        # make panel layout
        self.makePanel()
    # ----


    # ----
    def makeDataList(self):
        """ Make list of all fragments. """

        # make list
        self.dataList = mwx.ListCtrl(self.panel, -1, size=(300, 400))

        # make columns
        self.dataList.InsertColumn(0, "Mass/z", wx.LIST_FORMAT_RIGHT)
        self.dataList.InsertColumn(1, "Fragment Type", wx.LIST_FORMAT_LEFT)

        # set event
        self.dataList.Bind(wx.EVT_LIST_COL_CLICK, self.onSortList)
        self.dataList.Bind(wx.EVT_LIST_ITEM_SELECTED, self.onItemSelected)
    # ----


    # ----
    def updateDataList(self, evt=0):
        """ Update list of data. """

        # clear list
        self.dataList.DeleteAllItems()

        # check data
        if not self.data:
            return

        # get view
        view = self.view_combo.GetValue()

        # paste data
        row = 0
        format = '%0.' + `self.config.cfg['common']['digits']` + 'f'
        for i, fragment in enumerate(self.data):

            # show matched fragments only
            if view == 'Matched' and not fragment[2]:
                continue

            # show missed fragments only
            elif view == 'Missed' and fragment[2]:
                continue

            # format mass
            mass = format % fragment[0]

            # paste data to list
            self.dataList.InsertStringItem(row, mass)
            self.dataList.SetStringItem(row, 1, str(fragment[1]))

            # mark matched fragments
            if fragment[2] and view != 'Matched':
                self.dataList.SetItemBackgroundColour(row, self.config.cfg['colours']['matched'])

            # mark filtered fragments
            if fragment[3]:
                self.dataList.SetItemTextColour(row, self.config.cfg['colours']['grayed'])

            # row added
            row += 1

        # set columns width
        self.dataList.SetColumnWidth(0, wx.LIST_AUTOSIZE)
        self.dataList.SetColumnWidth(1, wx.LIST_AUTOSIZE)
        self.dataList.updateLastCol()
    # ----


    # ----
    def formatDataForExport(self):
        """ Format peptides list for export. """

        # get view
        view = self.view_combo.GetValue()

        # set formats
        digitsFormat = '%0.' + `self.config.cfg['common']['digits']` + 'f'
        listFormat = digitsFormat + '\t%s'

        # get data
        buff = ''
        for x in range(len(self.data)):

            # show matched fragments only
            if view == 'Matched' and not self.data[x][2]:
                continue

            # show unmatched fragments only
            elif view == 'Missed' and self.data[x][2]:
                continue

            # paste data
            buff += listFormat % tuple(self.data[x][0:2])
            buff += '\n'

        return buff
    # ----
