# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Base class for list panel.

# load libs
import wx
import os.path
from copy import copy

# load modules
from nucleus import mwx
from nucleus import commfce


class listPanel(wx.MiniFrame):
    """ Base class for list panel. """

    # ----
    def __init__(self, parent, title):
        wx.MiniFrame.__init__(self, parent, -1, title, style=wx.DEFAULT_FRAME_STYLE|wx.TINY_CAPTION_HORIZ)

        self.panel = wx.Panel(self, -1)
        self.parent = parent
        self.listType = ''
        self.data = []

        # defaults
        self.massCol = 3
        self.sortBy = 3
        self.sortDesc = False
    # ----


    # ----
    def makePanel(self):
        """ Make panel layout. """

        # make items
        
        self.makeDataList()
        viewButtons = self.makeViewButtons()
        mainButtons = self.makeMainButtons()

        # paste data to list
        self.updateDataList()

        # pack main frame
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        if wx.Platform == '__WXMAC__':
            mainSizer.Add(self.dataList, 1, wx.EXPAND|wx.ALL, 20)
            mainSizer.Add(viewButtons, 0, wx.BOTTOM|wx.ALIGN_CENTER, 20)
            mainSizer.Add(mainButtons, 0, wx.BOTTOM|wx.ALIGN_CENTER, 20)
        else:
            mainSizer.Add(self.dataList, 1, wx.EXPAND|wx.ALL, 3)
            mainSizer.Add(viewButtons, 0, wx.EXPAND|wx.ALL|wx.ALIGN_CENTER, 3)
            mainSizer.Add(mainButtons, 0, wx.ALL|wx.ALIGN_CENTER, 5)

        # set events
        self.Bind(wx.EVT_CLOSE, self.onClose)

        # fit layout
        self.panel.SetSizer(mainSizer)
        mainSizer.Fit(self)
        self.SetMinSize(self.GetSize())
        self.Centre()
    # ----


    # ----
    def makeViewButtons(self):
        """ Make box with view buttons. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self.panel, -1, "View Options"), wx.HORIZONTAL)

        zoom_choices = ['Current', '0.1', '0.5', '1', '5', '10', '20', '50', '100']
        zoom_label = wx.StaticText(self.panel, -1, "Zoom: ")
        self.zoom_combo = wx.ComboBox(self.panel, -1, '10', size=(65, -1), choices=zoom_choices, style=wx.CB_READONLY)
        zoom_units = wx.StaticText(self.panel, -1, ' %')
        
        filter_choices = ['All', 'Matched', 'Missed']
        filter_label = wx.StaticText(self.panel, -1, "Filter: ")
        self.view_combo = wx.ComboBox(self.panel, -1, 'All', size=(90, -1), choices=filter_choices, style=wx.CB_READONLY)

        # pack items
        mainBox.Add(zoom_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT|wx.BOTTOM, 5)
        mainBox.Add(self.zoom_combo, 0, wx.BOTTOM, 5)
        mainBox.Add(zoom_units, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT|wx.BOTTOM, 5)
        mainBox.Add(filter_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.LEFT|wx.BOTTOM, 5)
        mainBox.Add(self.view_combo, 0, wx.RIGHT|wx.BOTTOM, 5)

        # set events
        self.view_combo.Bind(wx.EVT_COMBOBOX, self.updateDataList)

        return mainBox
    # ----


    # ----
    def makeMainButtons(self):
        """ Make box with main buttons. """

        # make items
        Export_button = wx.Button(self.panel, -1, "Export")
        Close_button = wx.Button(self.panel, -1, "Close")

        # pack items
        mainBox = wx.BoxSizer(wx.HORIZONTAL)
        mainBox.Add(Export_button, 0, wx.LEFT|wx.RIGHT, 5)
        mainBox.Add(Close_button, 0, wx.LEFT|wx.RIGHT, 5)

        # set events
        Export_button.Bind(wx.EVT_BUTTON, self.onExport)
        Close_button.Bind(wx.EVT_BUTTON, self.onClose)

        return mainBox
    # ----


    # ----
    def updatePanel(self, data):
        """ Update list of data. """

        self.data = copy(data)
        self.updateDataList()
    # ----


    # ----
    def onSortList(self, evt):
        """ Sort list by specified column. """

        # get selected column
        selectedCol = evt.m_col

        # specific sorting
        if self.listType == 'mCut' and selectedCol != 0:
            selectedCol += 1

        # get order
        if self.sortBy == selectedCol and not self.sortDesc:
            self.sortDesc = True
        else:
            self.sortDesc = False
            self.sortBy = selectedCol

        # sort data
        self.data = commfce.sortMultiList(self.data, self.sortBy, self.sortDesc)

        # update peptide list
        self.updateDataList()
    # ----


    # ----
    def onItemSelected(self, evt):
        """ Highlight value in the spectrum. """

        # get selected mass
        item = self.dataList.GetItem(evt.m_itemIndex, self.massCol)
        point = float(item.GetText())

        # get zoom
        zoom = self.zoom_combo.GetValue()
        if zoom == 'Current':
            zoom = None
        else:
            zoom = float(zoom)

        # highlight point
        self.parent.onSpectrumHighlightPoint(point, zoom)
    # ----


    # ----
    def onExport(self, evt):
        """ Export data. """
        
        # get path from document
        path = self.parent.docData.getPath()
        path = os.path.dirname(path)

        # make filename
        if self.listType == 'mCut':
            fileName = 'digest_list.txt'
        elif self.listType == 'mFrag':
            fileName = 'fragments_list.txt'

        # raise export dialog
        dlg = wx.FileDialog(self, "Export List", path, fileName, 'ASCII List (*.txt)|*.txt', wx.SAVE|wx.OVERWRITE_PROMPT)
        if dlg.ShowModal() == wx.ID_OK:
            path = dlg.GetPath()
            dlg.Destroy()
        else:
            dlg.Destroy()
            return

        # get formated data
        formatedData = self.formatDataForExport()

        # save file
        try:
            save = file(path, 'w')
            save.write(formatedData.encode("utf-8"))
            save.close()
        except IOError:
            dlg = wx.MessageDialog(self.parent, "File cannot be saved! Check permissions please.", "Save Error", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
    # ----


    # ----
    def onClose(self, evt):
        """ Close Panel. """
        self.parent.destroyListPanel()
    # ----
