/*
 * Written by Bastien Chevreux (BaCh)
 *
 * Copyright (C) 1997-2000 by the German Cancer Research Center (Deutsches
 *   Krebsforschungszentrum, DKFZ Heidelberg) and Bastien Chevreux
 * Copyright (C) 2000 and later by Bastien Chevreux
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 * Free Software Foundation, Inc., 
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 * 
 */

/*
 * functions to output results of assembly
 * sourced out from assembly.C to get smaller file sizes and better modularity
 */


#ifndef lint
static char vcid3[] = "$Id$";
#endif /* lint */


#include "assembly_output.H"

#include <iomanip>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
// for getting username etc
#include <pwd.h>

// getting time in GBF output
#include <time.h>

#include <boost/lexical_cast.hpp>




#ifndef PUBLICQUIET
//#define CEBUGNPQ(bla)  {cout << bla; cout.flush();}
#define CEBUGNPQ(bla)  
#else
#define CEBUGNPQ(bla)
#endif

//#define CEBUGNPQ(bla)  {cout << bla; cout.flush();}
//#define CEBUG(bla)  {cout << bla; cout.flush();}



namespace assout {


/*************************************************************************
 *
 * returns true if file already existed and we can append
 * opens the ofstream
 *
 *
 *************************************************************************/

  bool openFileForAppend(const string & filename, ofstream & fout, bool deleteanyway)
  {
    struct stat st;
    if(deleteanyway || stat(filename.c_str(),&st)) {
      CEBUG("Opening " << filename << " and truncating.\n");
      fout.open(filename.c_str(), ios::out | ios::trunc);
      return false;
    }
    CEBUG("Opening " << filename << " and appending.\n");
    fout.open(filename.c_str(), ios::out | ios::app);
    return true;
  }



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

  void saveContigReadList(list<Contig> & clist, const string & filename, bool deleteoldfile)
  {
    FUNCSTART("void saveContigReadList(list<Contig> & clist, const string & filename, bool deleteoldfile)");

    // The next line is here to fool the compiler to also
    // include the CVS string into the binary
    // line has no effect and will put optimised away anyway
    (void) vcid3;


    // cout << "Saving project contig<->read list to file: " << filename << endl;

    ofstream fout;
    if(!openFileForAppend(filename,fout,deleteoldfile)){
      Contig::dumpContigReadList_Head(fout);
    }

    for(uint32 savewhat=0; savewhat<2; savewhat++){
      list<Contig>::iterator contigI=clist.begin();
      for(;contigI!=clist.end(); contigI++){
	bool saveme=false;
	if(savewhat==0 && contigI->getNumReadsInContig()>1) saveme=true;
	if(savewhat==1 && contigI->getNumReadsInContig()==1) saveme=true;
	if(saveme){
	  try{
	    contigI->dumpContigReadList_Body(fout);
	  }
	  catch (Notify n) {
	    cerr << "Error while dumping " << contigI->getContigName() << ".\n";
	    n.handleError(THISFUNC);
	  }
	}
      }
    }
    fout.close();

    FUNCEND();
  }

  void saveContigReadList(Contig & con, const string & filename, bool deleteoldfile)
  {
    FUNCSTART("void saveContigReadList(Contig & con, const string & filename)");

    ofstream fout;
    if(!openFileForAppend(filename,fout,deleteoldfile)){
      Contig::dumpContigReadList_Head(fout);
    }
    con.dumpContigReadList_Body(fout);
    fout.close();

    FUNCEND();
  }


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

  void saveStatistics(list<Contig> & clist, const string & filename, bool deleteoldfile)
  {
    FUNCSTART("void saveStatistics(list<Contig> & clist, const string & filename, bool deleteoldfile)");

    //cout << "Saving project statistics to file: " << filename << endl;

    ofstream statout;
    if(!openFileForAppend(filename,statout, deleteoldfile)){
      Contig::dumpContigStatistics_Head(statout);
    }

    for(uint32 savewhat=0; savewhat<2; savewhat++){
      list<Contig>::iterator contigI=clist.begin();
      for(;contigI!=clist.end(); contigI++){
	bool saveme=false;
	if(savewhat==0 && contigI->getNumReadsInContig()>1) saveme=true;
	if(savewhat==1 && contigI->getNumReadsInContig()==1) saveme=true;
	if(saveme){
	  try{
	    contigI->dumpContigStatistics_Body(statout);
	  }
	  catch (Notify n) {
	    cerr << "Error while dumping " << contigI->getContigName() << ".\n";
	    n.handleError(THISFUNC);
	  }
	}
      }
    }

    statout.close();

    FUNCEND();
  }
  void saveStatistics(Contig & con, const string & filename, bool deleteoldfile)
  {
    FUNCSTART("void saveStatistics(Contig & con, const string & filename)");

    ofstream statout;
    try{
      if(!openFileForAppend(filename,statout, deleteoldfile)){
	Contig::dumpContigStatistics_Head(statout);
      }
      con.dumpContigStatistics_Body(statout);
    }
    catch (Notify n) {
      cerr << "Error while dumping " << con.getContigName() << ".\n";
      n.handleError(THISFUNC);
    }
    statout.close();

    FUNCEND();
  }

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

  void saveAssemblyInfo(AssemblyInfo & asi, const string & filename, bool deleteoldfile)
  {
    FUNCSTART("void saveAssemblyInfo(AssemblyInfo & asi, const string & filename, bool deleteoldfile)");

    (void) deleteoldfile;

    ofstream infoout;
    openFileForAppend(filename,infoout, true);
    dateStamp(infoout);
    infoout << '\n' << asi;
    infoout.close();

    FUNCEND();
  }


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

  void saveReadTagList(list<Contig> & clist, const string & filename, bool deleteoldfile)
  {
    FUNCSTART("void saveReadTagList(list<Contig> & clist, const string & filename)");

    //cout << "Saving read tag list to file: " << filename << endl;

    ofstream rtout;
    if(!openFileForAppend(filename,rtout, deleteoldfile)){
      Contig::dumpReadTagList_Head(rtout);
    }

    for(uint32 savewhat=0; savewhat<2; savewhat++){
      list<Contig>::iterator contigI=clist.begin();
      for(;contigI!=clist.end(); contigI++){
	bool saveme=false;
	if(savewhat==0 && contigI->getNumReadsInContig()>1) saveme=true;
	if(savewhat==1 && contigI->getNumReadsInContig()==1) saveme=true;
	if(saveme){
	  try{
	    contigI->dumpReadTagList_Body(rtout);
	  }
	  catch (Notify n) {
	    cerr << "Error while dumping " << contigI->getContigName() << ".\n";
	    n.handleError(THISFUNC);
	  }
	}
      }
    }

    rtout.close();

    FUNCEND();
  }
  void saveReadTagList(Contig & con, const string & filename, bool deleteoldfile)
  {
    FUNCSTART("void saveReadTagList(Contig & con, const string & filename)");

    ofstream rtout;
    try{
      if(!openFileForAppend(filename,rtout,deleteoldfile)){
	Contig::dumpReadTagList_Head(rtout);
      }
      con.dumpReadTagList_Body(rtout);
    }
    catch (Notify n) {
      cerr << "Error while dumping " << con.getContigName() << ".\n";
      n.handleError(THISFUNC);
    }
    rtout.close();

    FUNCEND();
  }


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

  void saveConsensusTagList(list<Contig> & clist, const string & filename, bool deleteoldfile)
  {
    FUNCSTART("void saveConsensusTagList(list<Contig> & clist, const string & filename, bool deleteoldfile)");

    //cout << "Saving contig tag list to file: " << filename << endl;

    ofstream ctout;
    if(!openFileForAppend(filename,ctout,deleteoldfile)){
      Contig::dumpConsensusTagList_Head(ctout);
    }

    for(uint32 savewhat=0; savewhat<2; savewhat++){
      list<Contig>::iterator contigI=clist.begin();
      for(;contigI!=clist.end(); contigI++){
	bool saveme=false;
	if(savewhat==0 && contigI->getNumReadsInContig()>1) saveme=true;
	if(savewhat==1 && contigI->getNumReadsInContig()==1) saveme=true;
	if(saveme){
	  try{
	    contigI->dumpConsensusTagList_Body(ctout);
	  }
	  catch (Notify n) {
	    cerr << "Error while dumping " << contigI->getContigName() << ".\n";
	    n.handleError(THISFUNC);
	  }
	}
      }
    }

    ctout.close();

    FUNCEND();
  }

  void saveConsensusTagList(Contig & con, const string & filename, bool deleteoldfile)
  {
    FUNCSTART("void saveConsensusTagList(Con & con, const string & filename)");

    ofstream ctout;
    try{
      if(!openFileForAppend(filename,ctout,deleteoldfile)){
	Contig::dumpConsensusTagList_Head(ctout);
      }
      con.dumpConsensusTagList_Body(ctout);
    }
    catch (Notify n) {
      cerr << "Error while dumping " << con.getContigName() << ".\n";
      n.handleError(THISFUNC);
    }
    ctout.close();

    FUNCEND();
  }




/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

  void getPreviousLocusTag(const list<gbfsummary_t> & gbfs, list<gbfsummary_t>::const_iterator gbfsI, string & locustag, string & gene)
  {
    locustag=gbfsI->locustag;
    gene.clear();
    bool found=false;
    for(;!found;){
      if(gbfsI == gbfs.begin()) break;
      gbfsI--;
      if(gbfsI->locustag != locustag){
	found=true;
	locustag=gbfsI->locustag;
	gene=gbfsI->gene;
	while(gene.empty()){
	  if(gbfsI == gbfs.begin()) break;
	  gbfsI--;
	  if(gbfsI->locustag==locustag) {
	    gene=gbfsI->gene;
	  }else{
	    break;
	  }
	}
      }
    }
    if(!found) locustag.clear();
  }

  void getNextLocusTag(const list<gbfsummary_t> & gbfs, list<gbfsummary_t>::const_iterator gbfsI, string & locustag, string & gene)
  {
    locustag=gbfsI->locustag;
    gene.clear();
    bool found=false;
    for(;!found;){
      gbfsI++;
      if(gbfsI == gbfs.end()) break;
      if(gbfsI->locustag != locustag){
	found=true;
	locustag=gbfsI->locustag;
	gene=gbfsI->gene;
	while(gene.empty()){
	  gbfsI++;
	  if(gbfsI == gbfs.end() || gbfsI->locustag != locustag) break;
	  gene=gbfsI->gene;
	}
      }
    }
    if(!found) locustag.clear();
  }

/*************************************************************************
 *
 * this function relies on the fact that Contig::getGBFSummary()
 *  is used with the option to fake intergenic features
 *
 *************************************************************************/

  void saveSNPList(list<Contig> & clist, const string & filename, bool deleteoldfile)
  {
    FUNCSTART("void saveSNPList(list<Contig> & clist, const string & filename, bool deleteoldfile)");
    
    cout << "Saving SNP list to file: " << filename << endl;

    ofstream saout;
    if(!openFileForAppend(filename,saout,deleteoldfile)){
      saout << "#Note: this file is only useful whan having assembled with data\n"
	"\n#containing GenBank features."
	"\n#A list of all SNP tags (SROc, SIOc, SAOc and eventually MCVc is"
	"\n#in the 'consensustaglist' file.\n"
	"# Contig\t"
	"Contig position\t"
	"Reference\t"
	"Reference position\t"
	"Reference map position\t"
	"SNP type\t"
	"Feature type\t"
	"Hit locus\t"
	"Gene name\t"
	"EC num\t"
	"Product\t"
	"Function\t"
	"Note\n";
    }

    saout << setprecision(1) << fixed;

    list<Contig>::iterator cI=clist.begin();
    for(;cI!=clist.end(); cI++){
      cI->sortConsensusTags();
      vector<Contig::contigread_t>::const_iterator R=cI->getContigReads().begin();
      for(; R != cI->getContigReads().end() ; R++) {
	const_cast<Read &>(R->read).sortTags();
      }
    }

    // we will look at all GBF features (empty allowed)
    vector<multitag_t::mte_id_t> allowedfeatures;
    // but not at Fsrc
    vector<multitag_t::mte_id_t> forbiddenfeatures;
    forbiddenfeatures.push_back(Read::REA_tagentry_idFsrc);
    //forbiddenfeatures.push_back("FCDS");

    string gene;
    string function;
    string ecnumber;
    string product;
    string note;

    try{
      for(cI=clist.begin();cI!=clist.end(); cI++){
	list<gbfsummary_t> allGBfeatures;
	cI->getGBFSummary(allGBfeatures,allowedfeatures,forbiddenfeatures, true);

	vector<Contig::consensustag_t>::const_iterator consT=cI->getConsensusTags().begin();
	for(; consT != cI->getConsensusTags().end() ; consT++) {

	  if(!(consT->identifier==Contig::CON_tagentry_idSAOc
	       || consT->identifier==Contig::CON_tagentry_idSIOc
	       || consT->identifier==Contig::CON_tagentry_idSROc
	       )) continue;
	  bool infeature=false;
	
	  CEBUG("consT ID: " << consT->identifier << '\t' << consT->from << '\t' << consT->to << endl);

	  list<gbfsummary_t>::const_iterator gbfsI=allGBfeatures.begin();

	  for(; gbfsI != allGBfeatures.end(); gbfsI++){
	    if(gbfsI->identifier == multitag_t::getIdentifierStr(Read::REA_tagentry_idFCDS)) continue;
	    CEBUG(' ' << gbfsI->identifier << '\t' << gbfsI->locustag << '\t' << gbfsI->cfrom << '\t' << gbfsI->cto << '\n');

	    if(gbfsI->cfrom <= consT->from && gbfsI->cto >= consT->to){
	      const Read & featureread=cI->getReadAtIndex(gbfsI->readindex);
	      CEBUG("is in read: " << featureread.getName() << '\n');
	      CEBUG(featureread);
	      //CEBUG("Hit!\n");
	      string serialc;
	      infeature=true;

	      CEBUG("#1" << endl);

	      cI->concatAllGBFInfoForLocus(allGBfeatures, gbfsI, "; ", gene, function, ecnumber, product, note);

	      CEBUG("#2" << endl);

	      int32 freadposfrom=cI->getRealReadPos(consT->from,gbfsI->readindex);
	      CEBUG("#3" << endl);
	      int32 adjposfroml=featureread.getLowerNonGapAdjustmentPosOfReadPos(freadposfrom)+1;
	      CEBUG("#4" << endl);
	      int32 adjposfromr=featureread.getUpperNonGapAdjustmentPosOfReadPos(freadposfrom)+1;
	      CEBUG("#5" << endl);

	      //<< '\t' << I->paddedPos2UnpaddedPos(consT->from)
	      saout << cI->getContigName()
		    << '\t' << consT->from
		    << '\t' << featureread.getName()
		    << '\t' << adjposfroml;
	      if(adjposfroml!=adjposfromr) saout << ':' << adjposfromr;
	      saout << '\t' << (1.0/featureread.getLenSeq()*adjposfroml*360)
		    << '\t' << consT->getIdentifierStr()
		    << '\t' << gbfsI->identifier
		    << '\t' << gbfsI->locustag
		    << '\t' << gene
		    << '\t' << ecnumber
		    << '\t' << product
		    << '\t' << function
		    << '\t' << note
		    << endl;
	      //<< '\n';
	    }
	  }

	  if(!infeature) {
	    saout << cI->getContigName()
		  << '\t' << consT->from
		  << "\tn/a"
		  << "\tn/a"
		  << "\tn/a"
		  << '\t' << consT->getIdentifierStr()
		  << '\t' 
		  << '\t'
		  << '\t'
		  << '\t'
		  << '\t'
		  << '\t'
		  << "\tNot covered by any GenBank feature"
		  << endl;
	  }
	}
      }
    }
    catch (Notify n) {
      cerr << "Error while dumping " << cI->getContigName() << ".\n";
      n.handleError(THISFUNC);
    }
    FUNCEND();
  }





  struct shortgbfinfo_t {
    string identifier;
    string locustag;

    string gene;
    string function;
    string ecnumber;
    string product;
    string note;

    friend ostream & operator<<(ostream &ostr, const shortgbfinfo_t & sgi){
      ostr << "identifier: " << sgi.identifier
	   << "\nlocustag  : " << sgi.locustag
	   << "\ngene      : " << sgi.gene
	   << "\nfunction  : " << sgi.function
	   << "\necnumber  : " << sgi.ecnumber
	   << "\nproduct   : " << sgi.product
	   << "\nnote      : " << sgi.note
	   << endl;
      return ostr;
    }
  };

  struct snpenv_t {
    list<Contig>::iterator cI;
    string snpjumpname;

    list<gbfsummary_t> gbfslist;
    list<shortgbfinfo_t> sgilist;
    
    uint32 from;
    uint32 to;
    uint32 numsnps;   

    friend ostream & operator<<(ostream &ostr, const snpenv_t & se){
      ostr << "Contig: " << se.cI->getContigName()
	   << "\njumpname: " << se.snpjumpname
	   << "\nfrom      : " << se.from
	   << "\nto        : " << se.to
	   << "\nnumsnps   : " << se.numsnps
	   << endl;

      ostr << "gbfslist: ";
      if(se.gbfslist.empty()){
	ostr << "empty\n";
      }else{
	ostr << "\n";
	list<gbfsummary_t>::const_iterator gbfsI=se.gbfslist.begin();
	for(; gbfsI!=se.gbfslist.end(); gbfsI++){
	  ostr << *gbfsI;
	}
      }
      ostr << "sgilist: ";
      if(se.sgilist.empty()){
	ostr << "empty\n";
      }else{
	ostr << "\n";
	list<shortgbfinfo_t>::const_iterator sgI=se.sgilist.begin();
	for(; sgI!=se.sgilist.end(); sgI++){
	  ostr << *sgI;
	}
      }

      return ostr;
    }

  };



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

//#define CEBUG(bla) {cout << bla; cout.flush();}

  void saveSNPSurroundingAsHTML(list<Contig> & clist, const string & filename, bool deleteoldfile)
  {
    FUNCSTART("void saveSNPList(list<Contig> & clist, const string & filename, bool deleteoldfile)");
    
    cout << "Saving SNP surroundings as HTML to file: " << filename << endl;

    ofstream htmlout;
    if(!openFileForAppend(filename,htmlout,deleteoldfile)){
      dumpHTMLHeader("SOME PROJECT NAME",htmlout);
    }
    htmlout << "<p>";

    //htmlout << setprecision(1) << fixed;

    list<Contig>::iterator cI=clist.begin();
    for(;cI!=clist.end(); cI++){
      cI->sortConsensusTags();
      vector<Contig::contigread_t>::const_iterator R=cI->getContigReads().begin();
      for(; R != cI->getContigReads().end() ; R++) {
	const_cast<Read &>(R->read).sortTags();
      }
    }
    
   CEBUG("Num contigs: " << clist.size() << '\n');

    uint32 surrounding=39;
    uint32 maxdistformerge=5;


    // we will look at all GBF features (empty allowed)
    vector<multitag_t::mte_id_t> allowedfeatures;
    // but not at Fsrc
    vector<multitag_t::mte_id_t> forbiddenfeatures;
    forbiddenfeatures.push_back(Read::REA_tagentry_idFsrc);

    list<gbfsummary_t> allGBfeatures;

    // two loops: first to collect all names, then to 
    //  print out navigation and snps

    list<snpenv_t> snpenvironments;
    uint32 snpjumpnamenr=0;
    
    snpenv_t dummysnpenv;
    dummysnpenv.from=0;
    dummysnpenv.to=0;
    dummysnpenv.numsnps=1;
    for(cI=clist.begin();cI!=clist.end(); cI++){
      cI->getGBFSummary(allGBfeatures,allowedfeatures,forbiddenfeatures, true);
      vector<Contig::consensustag_t>::const_iterator consT=cI->getConsensusTags().begin();
      for(; consT != cI->getConsensusTags().end() ; consT++) {
	if(!(consT->identifier==Contig::CON_tagentry_idSAOc
	     || consT->identifier==Contig::CON_tagentry_idSIOc
	     || consT->identifier==Contig::CON_tagentry_idSROc
	     || consT->identifier==Contig::CON_tagentry_idMCVc
	     )) continue;
	// 
	CEBUG("Found tag: " << consT->identifier << "\t" << consT->from << '\n');
	if(snpenvironments.empty()){
	  dummysnpenv.cI=cI;
	  dummysnpenv.from=consT->from;
	  dummysnpenv.to=consT->to;
	  snpenvironments.push_back(dummysnpenv);	  
	}else{
	  if(snpenvironments.back().cI == cI 
	     && consT->to <= snpenvironments.back().to+maxdistformerge){
	    snpenvironments.back().to=consT->to;
	    snpenvironments.back().numsnps+=1;
	  }else{
	    dummysnpenv.cI=cI;
	    dummysnpenv.from=consT->from;
	    dummysnpenv.to=consT->to;
	    snpenvironments.push_back(dummysnpenv);	  
	  }
	}
      }
    }

    CEBUG("SNP environments: " << snpenvironments.size() << '\n');

//    list<snpenv_t>::iterator seI=snpenvironments.begin();
//    for(; seI != snpenvironments.end(); seI++){
//      ostringstream snpcoordostr;      
//      
//      bool featurefound=false;
//      list<gbfsummary_t>::const_iterator gbfsI=allGBfeatures.begin();
//      for(; gbfsI != allGBfeatures.end() && !featurefound; gbfsI++){
//	if(gbfsI->cfrom <= seI->from && gbfsI->cto >= seI->to){
//	  CEBUG(' ' << gbfsI->identifier << '\t' << gbfsI->locustag << '\t' << gbfsI->cfrom << '\t' << gbfsI->cto << endl);
//	  CEBUG("Hit!\n");
//	  featurefound=true;
//
//	  seI->identifier=gbfsI->identifier;
//	  seI->locustag=gbfsI->locustag;
//	  cI=seI->cI;
//	  const Read & featureread=cI->getReadAtIndex(gbfsI->readindex);
//	  
//	  cI->concatAllGBFInfoForLocus(allGBfeatures, gbfsI, "; ", seI->gene, seI->function, seI->ecnumber, seI->product, seI->note);
//	  
//	  int32 freadposfrom=cI->getRealReadPos(seI->from,gbfsI->readindex);
//	  CEBUG("#b1" << endl);
//	  int32 adjposfroml=featureread.getLowerNonGapAdjustmentPosOfReadPos(freadposfrom)+1;
//	  CEBUG("#b2" << endl);
//	  int32 adjposfromr=featureread.getUpperNonGapAdjustmentPosOfReadPos(freadposfrom)+1;
//	  
//	  snpcoordostr << gbfsI->locustag << '_' << adjposfroml;
//	  if(adjposfroml!=adjposfromr) snpcoordostr << ':' << adjposfromr;
//	  
//	  seI->snpjumpname=snpcoordostr.str();
//
//	  CEBUG("jumpname: " << seI->snpjumpname << endl);
//	}
//      }
//      if(!featurefound) {
//	CEBUG("WARNING: No feature found???\n" << *seI << endl);
//      }
//    }

    list<snpenv_t>::iterator seI=snpenvironments.begin();
    {
      for(; seI != snpenvironments.end(); seI++){
	CEBUG("Checking seI: " << *seI);

	ostringstream snpcoordostr;
	string firstlocus="";
	string lastlocus="";
	int32 lowestpos=0x7fffffff;
	int32 highestpos=0;

	bool featurefound=false;
	list<gbfsummary_t>::const_iterator gbfsI=allGBfeatures.begin();
	for(; gbfsI != allGBfeatures.end() && !featurefound; gbfsI++){
	  //if(gbfsI->cfrom <= seI->from && gbfsI->cto >= seI->to){
	  if((seI->from >= gbfsI->cfrom && seI->from <= gbfsI->cto)
	     || (seI->to >= gbfsI->cfrom && seI->to <= gbfsI->cto)
	     || (seI->from <= gbfsI->cfrom && seI->to >= gbfsI->cto)){
	    
	    CEBUG(' ' << gbfsI->identifier << '\t' << gbfsI->locustag << '\t' << gbfsI->cfrom << '\t' << gbfsI->cto << endl);
	    CEBUG("Hit!\n");
	    
	    //// we can stop if we find a Fgen
	    //if(gbfsI->identifier == "Fgen"){
	    //  featurefound=true;
	    //}
	    
	    seI->gbfslist.push_back(*gbfsI);
	    
	    shortgbfinfo_t sgitmp;
	    sgitmp.identifier=gbfsI->identifier;
	    sgitmp.locustag=gbfsI->locustag;
	    cI=seI->cI;
	    cI->concatAllGBFInfoForLocus(allGBfeatures, gbfsI, "; ", sgitmp.gene, sgitmp.function, sgitmp.ecnumber, sgitmp.product, sgitmp.note);
	    seI->sgilist.push_back(sgitmp);
	    
	    const Read & featureread=cI->getReadAtIndex(gbfsI->readindex);
	    int32 freadposfrom=cI->getRealReadPos(seI->from,gbfsI->readindex);
	    int32 adjposfroml=featureread.getLowerNonGapAdjustmentPosOfReadPos(freadposfrom)+1;
	    int32 adjposfromr=featureread.getUpperNonGapAdjustmentPosOfReadPos(freadposfrom)+1;
	    
	    if(adjposfroml<lowestpos){
	      lowestpos=adjposfroml;
	    }
	    if(adjposfromr>highestpos){
	      highestpos=adjposfromr;
	    }
	    if(!gbfsI->locustag.empty()){
	      if(firstlocus.empty()) firstlocus=gbfsI->locustag;
	      lastlocus=gbfsI->locustag;
	    }
	  }
	}
	if(firstlocus.empty()){
	  snpcoordostr << "nodesc_" << lowestpos;
	}else{
	  if(firstlocus==lastlocus){
	    snpcoordostr << firstlocus << '_' << lowestpos;
	  }else{
	    snpcoordostr << firstlocus << '-' << lastlocus << '_' << lowestpos;
	  }
	}
	if(lowestpos!=highestpos) snpcoordostr << ':' << highestpos;
	seI->snpjumpname=snpcoordostr.str();
	CEBUG("jumpname: " << seI->snpjumpname << endl);

	if(seI->gbfslist.empty()) {
	  CEBUG("WARNING: No feature found???\n" << *seI << endl);
	}
      }
    }

    // for each snpenvironment: 
    //  if there were gene or CDS: keep just those entries with the same locus tags
    //  else if there were entries with locus tags, just keep those entries
    {
      seI=snpenvironments.begin();
      for(; seI != snpenvironments.end(); seI++){
	list<gbfsummary_t>::iterator gbfslI=seI->gbfslist.begin();
	bool hasGENE=false;
	bool hasCDS=false;
	bool hasLOCUS=false;
	for(; gbfslI != seI->gbfslist.end(); gbfslI++){
	  if(!gbfslI->locustag.empty()){
	    hasLOCUS=true;
	    if(gbfslI->identifier=="Fgen"){
	      hasGENE=true;
	    }else if(gbfslI->identifier=="FCDS") {
	      hasCDS=true;
	    }
	  }
	}

	if(hasGENE || hasCDS || hasLOCUS){
	  CEBUG("SOMETHING TO DELETE? " << hasGENE << hasCDS << hasLOCUS << *seI);
	  gbfslI=seI->gbfslist.begin();
	  list<shortgbfinfo_t>::iterator sgiI=seI->sgilist.begin();
	  for(; gbfslI != seI->gbfslist.end(); gbfslI++, sgiI++){
	    bool deleteit=false;
	    if(hasGENE){
	      if(gbfslI->identifier != "Fgen") deleteit=true;
	    }else if(hasCDS){
	      if(gbfslI->identifier != "FCDS") deleteit=true;
	    }else if(hasLOCUS){
	      if(gbfslI->locustag.empty()) deleteit=true;
	    }
	    if(deleteit){
	      CEBUG("DELETE gbfs: " << *gbfslI);
	      gbfslI=seI->gbfslist.erase(gbfslI);
	      CEBUG("DELETE sgi: " << *sgiI);
	      sgiI=seI->sgilist.erase(sgiI);
	    }
	  }
	  CEBUG("AFTER DELETE: " << *seI);
	}else{
	  CEBUG("NOTHING TO DELETE: " << *seI);
	}
      }
    }


    // perfect, create jumpcoord strings etc.











    {
      string currentcontig;
      for(seI=snpenvironments.begin(); seI != snpenvironments.end(); seI++){
	cI=seI->cI;
	CEBUG("New seI. Contig: " << cI->getContigName() << endl);
	CEBUG(*seI);
	if(currentcontig != cI->getContigName()) {
	  if(!currentcontig.empty()){
	    htmlout << "</table><p>";
	  }
	  currentcontig = cI->getContigName();
	  htmlout << "<H1>Difference regions for " << currentcontig << "</H1>\n";

	  {
	    uint32 numbbstrains=0;
	    vector<bool> seenstrains(cI->getContigReads().size(),false);
	    vector<Contig::contigread_t>::const_iterator crI=cI->getContigReads().begin();
	    vector<Contig::contigread_t>::const_iterator crE=cI->getContigReads().end();
	    for(; crI != crE; crI++){
	      if(crI->orpid>=0 && crI->read.isBackbone()){
		if(!seenstrains[crI->read.getStrainID()]){
		  numbbstrains++;
		  seenstrains[crI->read.getStrainID()]=true;
		}
	      }
	    }
	    if(numbbstrains>0){
	      if(numbbstrains==1) {
		htmlout << "Backbone strain: ";
	      }else{
		htmlout << "Backbone strains: ";
	      }
	      crI=cI->getContigReads().begin();
	      for(; crI != crE; crI++){
		if(crI->orpid>=0){
		  if(seenstrains[crI->read.getStrainID()]){
		    seenstrains[crI->read.getStrainID()]=false;
		    htmlout << crI->read.getStrain() << ' ';
		  }
		}
	      }
	      htmlout << "<p>\n";
	    }

	    uint32 numstrains=0;
	    crI=cI->getContigReads().begin();
	    for(; crI != crE; crI++){
	      if(crI->orpid>=0 && crI->read.isBackbone()==false){
		if(!seenstrains[crI->read.getStrainID()]){
		  numstrains++;
		  seenstrains[crI->read.getStrainID()]=true;
		}
	      }
	    }
	    if(numstrains>0){
	      if(numbbstrains==1) {
		htmlout << "Mapped strain: ";
	      }else{
		htmlout << "Mapped strains: ";
	      }
	      crI=cI->getContigReads().begin();
	      for(; crI != crE; crI++){
		if(crI->orpid>=0){
		  if(seenstrains[crI->read.getStrainID()]){
		    seenstrains[crI->read.getStrainID()]=false;
		    htmlout << crI->read.getStrain() << ' ';
		  }
		}
	      }
	      htmlout << "<p>\n";
	    }
	  }


	  htmlout << "<table BORDER=\"0\" CELLSPACING=\"10\" CELLPADDING=\"0\">\n";
	}
	
	htmlout << "<tr align=\"left\">"
		<< "<td CLASS=\"jtable1\"><a href=\"#" 
		<< seI->snpjumpname << "\">" 
		<< seI->snpjumpname << "</a></td>"
		<< "<td  CLASS=\"jtable2\">";
	uint32 num=0;
	list<shortgbfinfo_t>::const_iterator sgiI=seI->sgilist.begin();
	for(; sgiI!=seI->sgilist.end(); sgiI++, num++){
	  if(!(sgiI->gene).empty()){
	    if(num>0) htmlout << "; ";
	    htmlout << sgiI->gene;
	  }
	}
	htmlout << "</td>"
		<< "<td  CLASS=\"jtable3\">";
	if(seI->sgilist.size()==0) {
	  // valid if left or right of annotated backbone
	  //htmlout << "0 locus hit??? Strange.";
	} else if(seI->sgilist.size()>1) {
	  htmlout << "Multiple locus hit";
	}else{
	  bool mustbr=false;
	  if(!seI->sgilist.begin()->function.empty()){
	    htmlout << seI->sgilist.begin()->function;
	    mustbr=true;
	  }
	  if(!seI->sgilist.begin()->product.empty()){
	    if(mustbr) htmlout << "<br>";
	    htmlout << seI->sgilist.begin()->product;
	    mustbr=true;
	  }
	  if(!seI->sgilist.begin()->note.empty()){
	    if(mustbr) htmlout << "<br>";
	    htmlout << seI->sgilist.begin()->note;
	  }

	  //if(!seI->sgilist.begin()->function.empty()){
	  //  htmlout << seI->sgilist.begin()->function;
	  //}else{
	  //  htmlout << seI->sgilist.begin()->note;
	  //}
	}
	htmlout << "</td></tr>\n";
      }
      if(!currentcontig.empty()){
	htmlout << "</table><p>";
      }
    }

    for(seI=snpenvironments.begin(); seI != snpenvironments.end(); seI++){
      cI=seI->cI;

      htmlout << "<H1>" 
	      << "<a NAME=\"" << seI->snpjumpname << "\"></a>"
	      << seI->snpjumpname << "</H1>\n";
      if(seI != snpenvironments.begin()){
	seI--;
	htmlout << "<a href=\"#" << seI->snpjumpname << "\">";
	seI++;
      }
      htmlout << "previous";
      if(seI != snpenvironments.begin()){
	htmlout << "</a>";
      }
      htmlout << "  ";
      seI++;
      if(seI != snpenvironments.end()){
	htmlout << "<a href=\"#" << seI->snpjumpname << "\">";
      }
      htmlout << "next";
      if(seI != snpenvironments.end()){
	htmlout << "</a>";
      }
      seI--;

      htmlout << "<br/><br/>";

      uint32 num=0;
      list<shortgbfinfo_t>::const_iterator sgiI=seI->sgilist.begin();
      for(; sgiI!=seI->sgilist.end(); sgiI++, num++){
	if(sgiI->identifier != "Figr"){
	  htmlout << "Hitting: " << sgiI->locustag;
	  if(!sgiI->gene.empty()){
	    htmlout << " (" << sgiI->gene << ')';
	  }
	  htmlout << "<br>\n";
	}
	if(!sgiI->product.empty()){
	  htmlout << "Product: " << sgiI->product << "<br>\n";
	}
	if(!sgiI->function.empty()){
	  htmlout << "Function: " << sgiI->function << "<br>\n";
	}
	if(!sgiI->note.empty()){
	  htmlout << "Note: " << sgiI->note << "<br>\n";
	}
      }


      cI->dumpAsHTML(htmlout, seI->from - surrounding, seI->to + surrounding);
    }

    htmlout.close();

    CEBUG("\nDone with this contig" << endl);
  }

#define CEBUG(bla)



/*************************************************************************
 *
 * result may contain '@' for positions not covered by a strain.
 *  if you do not want that, set fillholesinstraingenomes to true
 *
 *************************************************************************/

  void makeAllStrainGenomes(Contig & contigI, base_quality_t minqual, string & consseq, vector<base_quality_t> & consqual, vector<string> & strain_consseq, vector< vector<base_quality_t> > & strain_consqual, strainid2name_t & strainnames_in_contig, bool fillholesinstraingenomes)
  {
    FUNCSTART("void makeAllStrainGenomes(Contig & contigI, string & consseq, vector<base_quality_t> & consqual, vector<string> & strain_consseq, vector< vector<base_quality_t> > & strain_consqual, strainid2name_t & strainnames_in_contig, bool fillholesinstraingenomes)");

    CEBUGNPQ("Making main genome consensus\n");
    
    // make a common consensus for all strains: mincoverage=0, minqual given
    contigI.calcConsensi(0,minqual);

    // fetch the combined consensus 
    contigI.newConsensusGet(consseq, consqual);
    
    CEBUGNPQ("Making strain genome consensi\n");
    
    // and make a specific consensus for every strain present in
    //  this contig
    // We'll do by fetching the strain specific consensus and apply, if wanted,
    //  the function to fill in gaps and Ns from the combined consensus
    strainnames_in_contig.clear();

    // search the strains present in the contig
    // TODO: contig now track strains present ... should use that!
    const vector<Contig::contigread_t> & creads=contigI.getContigReads();
    {
      vector<Contig::contigread_t>::const_iterator R=creads.begin();
      for(; R != creads.end(); R++){
	//strainids_in_contig.insert(R->read.getStrainID());
	if(strainnames_in_contig.find(R->read.getStrainID())==strainnames_in_contig.end()){
	  strainnames_in_contig.insert(s_idname_pair_t(R->read.getStrainID(),R->read.getStrain()));
	  CEBUGNPQ(static_cast<int32>(R->read.getStrainID()) << "\t-- " << R->read.getStrain() << endl);
	}
      }
    }
    
    CEBUGNPQ("Number of strains present: " << strainnames_in_contig.size() << endl);
    strain_consseq.clear();
    strain_consqual.clear();
      
    if(strainnames_in_contig.size()>0) {
      strainid2name_t::const_iterator SI=strainnames_in_contig.end();
      SI--;
      int32 largest=(SI->first)+1;
      strain_consseq.resize(largest);
      strain_consqual.resize(largest);
      
      // fill the consensus of the strains present
      SI=strainnames_in_contig.begin();
      for(;SI != strainnames_in_contig.end(); SI++) {
	contigI.newConsensusGet(strain_consseq[SI->first], 
				strain_consqual[SI->first], 
				SI->first);

	if(fillholesinstraingenomes) {
	  // now merge the strain consensus:
	  //  fill up N or holes in the strain consensus with bases from
	  //  the main consensus
	  BUGIFTHROW(consseq.size() != strain_consseq[SI->first].size(),"consseq.size() != strain_consseq[SI->first].size())?");
	  //CEBUGNPQ("strain_consseq[" << (uint16) *SI << "]: " <<  strain_consseq[*SI] << endl);
	  for(uint32 i=0; i<consseq.size(); i++){
	    if(strain_consseq[SI->first][i]=='@'){
	      strain_consseq[SI->first][i]=static_cast<char>(tolower(consseq[i]));
	      strain_consqual[SI->first][i]=consqual[i];
	    }
	  }
	}
      }
    }
    FUNCEND();
  }


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

  struct proteinchangesummary_t {
    bool isaffected;

    bool firstcodonisstart;

    bool changedstart;
    bool destroyedstart;
    bool changedstop;
    bool destroyedstop;
    bool prematurestop;

    uint32 mutinintergenic;

    uint32 insertioninlocus;
    uint32 deletioninlocus;
    uint32 silentinlocus;
    uint32 aachangeinlocus;

    uint32 insertionuntranslated;    //
    uint32 deletionuntranslated;     //
    uint32 silentuntranslated;       //
    uint32 aachangeuntranslated;     //

    uint32 proteinlenchanged;
    string proteininfeature;

    string coveragestatus;
  };




  void saveFeatureAnalysis(list<Contig> & clist, ReadPool & rpool, const string & faname, const string & fsname, const string & fcname, bool deleteoldfile)
  {
    FUNCSTART("saveFeatureAnalysis(list<Contig> & clist, ReadPool & rpool, const string & faname, const string & faname)");

    cout << "Saving feature analysis to file: " << faname << endl;
    ofstream faout;
    if(!openFileForAppend(faname,faout,deleteoldfile)){
      faout << "# Contig\t"
	    << "Reference strain\t"
	    << "Mutant strain\t"
	    << "Internal contig pos\t"
	    << "Position in reference\t"
	    << "Map pos in reference\t"
	    << "Locus\t"
	    << "Direction\t"
	    << "Gene name\t"
	    << "Distance of SNP\t"
	    << "DNA effect\t"
	    << "Nucleotide change\t"
	    << "Codon in reference\t"
	    << "Codon in mutant\t"
	    << "Quality reference\t"
	    << "Quality mutant\t"
	    << "AA change\t"
	    << "AA position reference\t"
	    << "Codon position reference\t"
	    << "AA position mutant\t"
	    << "Codon position mutant\t"
	    << "Effect of mutation on protein\t"
	    << "Product\t"
	    << "Function\t"
	    << "Note\n";
    }

    faout << setprecision(1) << fixed;

    cout << "Saving impact summary to file: " << fsname << endl;
    ofstream fsout;
    if(!openFileForAppend(fsname,fsout,deleteoldfile)){
      fsout << "# Locus\tGene name\tFeature type\tGenome map pos\t"
	
	    << "Interesting?\t"
	    << "Your own filter\t"

	    << "Coverage status\t"

	    << "First codon is start\t"

	    << "Changed start codon\t"
	    << "Destroyed start codon\t"
	    << "Changed stop codon\t"
	    << "Destroyed stop codon\t"
	    << "Premature stop codon\t"

	    << "Intergenic mutation\t"

	    << "Insertion in locus\t"
	    << "Deletion in locus\t"
	    << "Silent in locus\t"
	    << "AA change in locus\t"
      
	    << "Insertion untranslated\t"
	    << "Deletion untranslated\t"
	    << "Silent untranslated\t"
	    << "AA change untranslated\t"
      
	    << "Product\tFunction\tNote"

	    << endl;
    }

    fsout << setprecision(1) << fixed;

    cout << "Saving feature sequences to file: " << fcname << endl;
    ofstream fcout;
    if(!openFileForAppend(fcname,fcout,deleteoldfile)){
      fcout << "# Contig\tLocus\tGene name\tFeature strain name\tMutant strain name 1 ...\tProtein in feature strain\tProtein in mutant strain 1 ...\tDNA in feature strain\tDNA in mutant strain 1 ...\n";
    }

    // we will look at all GBF features (empty == all allowed)
    vector<multitag_t::mte_id_t> allowedfeatures;
    //allowedfeatures.push_back("Fgen");
    allowedfeatures.push_back(Read::REA_tagentry_idFCDS);
    allowedfeatures.push_back(Read::REA_tagentry_idFexn);
    allowedfeatures.push_back(Read::REA_tagentry_idFint);
    allowedfeatures.push_back(Read::REA_tagentry_idFmRN);
    allowedfeatures.push_back(Read::REA_tagentry_idFm_R);
    allowedfeatures.push_back(Read::REA_tagentry_idFpRN);
    allowedfeatures.push_back(Read::REA_tagentry_idFrRN);
    allowedfeatures.push_back(Read::REA_tagentry_idFscR);
    allowedfeatures.push_back(Read::REA_tagentry_idFsnR);
    allowedfeatures.push_back(Read::REA_tagentry_idFtRN);

    // but not at Fsrc and Fgen
    vector<multitag_t::mte_id_t> forbiddenfeatures;
    //forbiddenfeatures.push_back("Fsrc");
    //forbiddenfeatures.push_back("Fgen");
    //forbiddenfeatures.push_back("Fcon");
    //forbiddenfeatures.push_back("Fmod");
    //forbiddenfeatures.push_back("Fold");
    //forbiddenfeatures.push_back("Frpr");
    //forbiddenfeatures.push_back("Frpu");
    //forbiddenfeatures.push_back("F???");
    //forbiddenfeatures.push_back("Fvar");
    
    // now, go through each contig and analyse the effect of SNPs on feature
    //  sequences

    list<Contig>::iterator contigI=clist.begin();
    for(;contigI!=clist.end(); contigI++){
      contigI->sortConsensusTags();

      // copy first all Genbank features of all reads into one big vector
      //  (makes the search loop afterwards easier)

      list<gbfsummary_t> allGBfeatures;
      contigI->getGBFSummary(allGBfeatures,allowedfeatures,forbiddenfeatures,true);

      // continue with next contig if there are no features in here
      if(allGBfeatures.empty()) continue;

      CEBUGNPQ("Preparing statistics\n");

      vector<proteinchangesummary_t> allproteinchanges;
      allproteinchanges.reserve(allGBfeatures.size());
      proteinchangesummary_t pcs = {
	false,
	false,
	false,
	false,
	false,
	false,
	false,
	0,
	0,0,0,0,
	0,0,0,0,
	0,"",
	""
      };
      {
	list<gbfsummary_t>::const_iterator F=allGBfeatures.begin();
	for(; F!=allGBfeatures.end(); F++){
	  allproteinchanges.push_back(pcs);
	}
      }


      // idee: nicht nur einen Consensus generieren, sondern fr jeden
      //       strain einen eigenen. Dann kann man einfach strainweise
      //       die auswirkungen bei Proteinen abschtzen
      //  contig.C umndern.
      //  dann nicht abgedeckte stellen in einem strain (N's) jeweils durch 
      //    IUPAC consensus anderer strains auffllen
      
      string consseq;
      vector<base_quality_t> consqual;
      vector<string> strain_consseq;
      vector< vector<base_quality_t> > strain_consqual;
      strainid2name_t strainnames_in_contig;
      
      makeAllStrainGenomes(*contigI,
			   0,
			   consseq,
			   consqual,
			   strain_consseq,
			   strain_consqual,
			   strainnames_in_contig,
			   false);

      CEBUGNPQ("Going through SNPs\n");

      // make a quick lookup vector for which position is intergenic
      vector<uint8> intergeniclookup(consseq.size(),0);
      {
	list<gbfsummary_t>::const_iterator F=allGBfeatures.begin();
	for(; F != allGBfeatures.end(); F++){
	  if(F->identifier == "Figr"){
	    for(uint32 iglpos=F->cfrom; iglpos <= F->cto; iglpos++){
	      if(iglpos<intergeniclookup.size()) intergeniclookup[iglpos]=1;
	    }
	  }
	}
      }

      // Now, go through each GB features and analyse the impact 
      //  of SNPs to it
      // Also calculate the "real" dna and protein sequence for every
      //  strain on the fly

      vector< vector<string> > strain_featuredna;
      vector< vector<string> > strain_featureprot;
      strain_featuredna.resize(allGBfeatures.size());
      strain_featureprot.resize(allGBfeatures.size());
      for(uint32 i=0; i<allGBfeatures.size(); i++){
	strain_featuredna[i].resize(strain_consseq.size());
	strain_featureprot[i].resize(strain_consseq.size());
      }

      list<gbfsummary_t>::const_iterator F=allGBfeatures.begin();
      vector<proteinchangesummary_t>::iterator P=allproteinchanges.begin();
      for(uint32 featurecount=0; F!=allGBfeatures.end(); F++, P++, featurecount++){
	if(contigI->getConsensusTags().begin() == contigI->getConsensusTags().end()) continue;

	CEBUGNPQ("We're looking at " << F->locustag << '\t' << F->cfrom << '\t' << F->cto << endl);
	// unsigned, always true: if(F->cfrom<0 || F->cto<0
	if(F->cfrom >= consseq.size() || F->cto >= consseq.size()){
	  CEBUGNPQ("Out of bounds, next." << endl);
	  P->coveragestatus="out of contig";
	  continue;
	}

	const Read & featureread=contigI->getReadAtIndex(F->readindex);

	// featuredir is also the increment
	int8 featuredir=F->direction;;
	CEBUGNPQ("Increment/featuredir is: " << static_cast<int16>(featuredir) << endl);

	P->coveragestatus="ok";
	P->firstcodonisstart=true;

	// collect information on the DNA and protein sequences
	for(uint32 strainid=0; strainid<strain_consseq.size(); strainid++){
	  if(strain_consseq[strainid].size()){

	    // check whether we have non covered bases in this strain
	    bool missingcompletely=false;
	    {
	      uint32 basesnocov=0;
	      
	      const char * dnap=strain_consseq[strainid].c_str();
	      dnap+=F->cfrom;
	      for(uint32 ii=F->cfrom; ii<= F->cto; dnap++, ii++){
		if(*dnap=='@') basesnocov++;
	      }
	      
	      CEBUGNPQ("basesnocov: " << basesnocov << '\n');
	      if(basesnocov >= (F->cto - F->cfrom)){
		P->coveragestatus="completely missing?";
		missingcompletely=true;
	      }else if(basesnocov > 0
		       && P->coveragestatus =="ok"){
		P->coveragestatus="partly missing?";
	      }
	    }

	    // look how feature is translated
	    if(featuredir>0) {
	      uint32 translateto=F->cto;
	      if(F->mustbetranslated) translateto=F->cfrom;
	      dptools::dnaToProtein(strain_consseq[strainid],
				    strain_featureprot[featurecount][strainid],
				    strain_featuredna[featurecount][strainid],
				    F->cfrom,
				    translateto,
				    featuredir,
				    F->translationtable,
				    F->codonstart);
	    }else{
	      uint32 translateto=F->cfrom;
	      if(F->mustbetranslated) translateto=F->cto;
	      dptools::dnaToProtein(strain_consseq[strainid],
				    strain_featureprot[featurecount][strainid],
				    strain_featuredna[featurecount][strainid],
				    F->cto,
				    translateto,
				    featuredir,
				    F->translationtable,
				    F->codonstart);
	    }
	    CEBUGNPQ("Protein for strain " << strainid << ": " << strain_featureprot[featurecount][strainid] << endl);
	    CEBUGNPQ("DNA for strain " << strainid << ": " << strain_featuredna[featurecount][strainid] << endl);

	    // clear "protein" of features that are not to be translated 
	    //  or where the coverage is completely missing
	    if(!F->mustbetranslated
	      || missingcompletely){
	      strain_featureprot[featurecount][strainid].clear();
	      P->firstcodonisstart=false;
	    }else if(strain_featuredna[featurecount][strainid].size() >= 3
	      && P->firstcodonisstart){
	      P->firstcodonisstart=dptools::isCodonStart(F->translationtable,
							 strain_featuredna[featurecount][strainid][0],
							 strain_featuredna[featurecount][strainid][1],
							 strain_featuredna[featurecount][strainid][2]);
	      CEBUGNPQ("firstcodonisstart: " << P->firstcodonisstart << '\n');
	    }
	  }
	}



	// go through each SNP consensustag
	vector<Contig::consensustag_t>::const_iterator CTact;
	vector<Contig::consensustag_t>::const_iterator CTlast;

	CTact=contigI->getConsensusTags().begin();
	CTlast=contigI->getConsensusTags().end();
	CTlast--;
	if(featuredir<0) {
	  std::swap(CTact,CTlast);
	}

	bool lastconstagreached=false;
	for(; !lastconstagreached; CTact+=featuredir){
	  if(CTact==CTlast) lastconstagreached=true;

	  if(CTact->identifier == Contig::CON_tagentry_idSAOc
	     || CTact->identifier == Contig::CON_tagentry_idSIOc
	     || CTact->identifier == Contig::CON_tagentry_idSROc) {

	    bool mustconsider=false;
	    int32 reldistance=0;
	    //int8 updownlocation=0;
	    // look where the SNP is located: in the feature (location =0), 
	    //  upstream (-1) or downstream (1) 

	    CEBUGNPQ("Testing F " << F->cfrom << " " << F->cto << "\t");
	    CEBUGNPQ(CTact->getIdentifierStr() << " " << CTact->from << " " << CTact->to << endl);

	    // In dubio pro reo. To counter 
	    //   - wrong annotation in initial genome or 
	    //   - SNPs in mutant strain
	    // we look at the coordinates as dictated by the protein lengths
	    //  and decide upon these whether a SNP is inlocus, 
	    //  up- or downstream


	    int32 realfcfrom=F->cfrom;
	    int32 realfcto=F->cto;
	    {
	      // first, get max protein length
	      
	      int32 newgenelen=0;
	      {
		size_t maxprotlen=0;
		for(uint32 strainid=0; strainid<strain_consseq.size(); strainid++){
		  maxprotlen=max(maxprotlen,strain_featureprot[featurecount][strainid].size());
		}
		newgenelen=static_cast<int32>(maxprotlen*3);
	      }

	      if(newgenelen > static_cast<int32>(F->cto - F->cfrom)){
		CEBUG("Must adapt gene length to: " << newgenelen << '\n');
		if(featuredir>0){
		  realfcto=F->cfrom+newgenelen;
		}else{
		  realfcfrom=F->cto-newgenelen;
		}
	      }
	    }

	    bool taginintergenic=false;
	    // always >=0: CTact->from >=0
	    if(CTact->from < consseq.size()
	       && intergeniclookup[CTact->from]){
	      taginintergenic=true;
	    }

	    if(static_cast<int32>(CTact->from) >= realfcfrom 
	       && static_cast<int32>(CTact->from) <= realfcto){
	      mustconsider=true;
	      if(featuredir>0){
		reldistance=CTact->from - (realfcfrom);
	      }else{
		reldistance=(realfcto)-(CTact->from);
	      }
	    }

	    if(mustconsider){
	      CEBUGNPQ("Possible candidate for: " << F->locustag << " (" << F->gene << ")\n");
	      CEBUGNPQ("  Location: " << CTact->from << endl);

	      char baseinfeaturestrain=static_cast<char>(toupper(contigI->getBaseInRead(CTact->from,F->readindex)));
	      if(featuredir<0) baseinfeaturestrain=dptools::getComplementIUPACBase(baseinfeaturestrain);
	    
	      CEBUGNPQ("  baseinfeaturestrain: " << baseinfeaturestrain << "\n");
	    
	      // compile some information on the position in the strain
	      //  containing the feature
	      string codoninfeature;
	      vector<char> aainfeature;
	      vector<bool> isstartinfeature;
	      int32 aanumberinfeature=0;
	      int8  posinaainfeature=0;

	      if(F->mustbetranslated){
		if(featuredir>0) {
		  dptools::infoOnAAatDNAPos(strain_consseq[F->strainid],
					    CTact->from,
					    realfcfrom,
					    featuredir,
					    F->translationtable,
					    F->codonstart,
					    codoninfeature,
					    aainfeature,
					    isstartinfeature,
					    aanumberinfeature,
					    posinaainfeature);
		}else{
		  dptools::infoOnAAatDNAPos(strain_consseq[F->strainid],
					    CTact->from,
					    realfcto,
					    featuredir,
					    F->translationtable,
					    F->codonstart,
					    codoninfeature,
					    aainfeature,
					    isstartinfeature,
					    aanumberinfeature,
					    posinaainfeature);
		}

		//BUGIFTHROW(aainfeature.size()==0,"aainfeature.size()==0?");
		//BUGIFTHROW(isstartinfeature.size()==0,"isstartinfeature.size()==0?");
		if(aainfeature.empty()) aainfeature.push_back('?');
		if(isstartinfeature.empty()) isstartinfeature.push_back(false);
	      } else {
		aainfeature.push_back('/');
		isstartinfeature.push_back(false);
	      }


	      CEBUGNPQ("\nLooking for strains ...");

	      strainid2name_t::const_iterator SIC=strainnames_in_contig.begin();
	      for(; SIC!= strainnames_in_contig.end(); SIC++){

		//for(uint32 actstrainid=0; actstrainid<strain_consseq.size();actstrainid++){
		int32 actstrainid=SIC->first;
		CEBUGNPQ("Strainid: " << actstrainid << " ... ");
		// do not look in own strain of feature
		if(actstrainid==F->strainid) continue;
		// do not look in empty strains when inlocus
		if(strain_featuredna[featurecount][actstrainid].size()==0) continue;

		CEBUGNPQ("CHECK!\n");

		char baseinstrain='?';
		if(featuredir>0){
		  baseinstrain=static_cast<char>(toupper(strain_consseq[actstrainid][CTact->from]));
		}else{
		  baseinstrain=static_cast<char>(toupper(dptools::getComplementIUPACBase(strain_consseq[actstrainid][CTact->from])));
		}
		CEBUGNPQ("baseinfeaturestrain : " << baseinfeaturestrain<<endl);
		CEBUGNPQ("baseinstrain " << actstrainid << ": " << baseinstrain<<endl);
		if(baseinstrain==baseinfeaturestrain) continue;

		CEBUGNPQ("Ok, base in strain different from feature.\n");

		P->isaffected=true;

		bool makesframeshift=false;
		string dnaeffect="basechange";
		if(baseinfeaturestrain=='*' || baseinstrain=='*'){
		  makesframeshift=true;
		  if(baseinstrain=='*'){
		    dnaeffect="deletion";
		  }else{
		    dnaeffect="insertion";
		  }
		}

		CEBUGNPQ("Makes frame shift: " << makesframeshift << '\n');
		CEBUGNPQ("Effect on DNA: " << dnaeffect << '\n')
		
		// compile some information on the position in the strain
		//  we are looking at
		string codoninstrain;
		vector<char> aainstrain;
		vector<bool> isstartinstrain;
		int32 aanumberinstrain=0;
		int8  posinaainstrain=0;
		if(F->mustbetranslated){
		  CEBUGNPQ("must be translated.\n");
		  if(featuredir>0) {
		    dptools::infoOnAAatDNAPos(strain_consseq[actstrainid],
					      CTact->from,
					      realfcfrom,
					      featuredir,
					      F->translationtable,
					      F->codonstart,
					      codoninstrain,
					      aainstrain,
					      isstartinstrain,
					      aanumberinstrain,
					      posinaainstrain);
		  }else{
		    dptools::infoOnAAatDNAPos(strain_consseq[actstrainid],
					      CTact->from,
					      realfcto,
					      featuredir,
					      F->translationtable,
					      F->codonstart,
					      codoninstrain,
					      aainstrain,
					      isstartinstrain,
					      aanumberinstrain,
					      posinaainstrain);
		  }
		  //BUGIFTHROW(aainstrain.size()==0,"aainstrain.size()==0?");
		  //BUGIFTHROW(isstartinstrain.size()==0,"isstartinstrain.size()==0?");
		  if(aainstrain.empty()) aainstrain.push_back('?');
		  if(isstartinstrain.empty()) isstartinstrain.push_back(false);
		}else{
		  CEBUGNPQ("not translated.\n");
		  aainstrain.push_back('/');
		  isstartinstrain.push_back(false);
		}

		string effectonprot="none";
		if(F->mustbetranslated){
		  // untranslated SNPs are SNPs that occur within the
		  //  feature, but where the protein still ended earlier
		  bool untranslated=false;
		  if(aanumberinstrain>=static_cast<int32>(strain_featureprot[featurecount][actstrainid].size())) {
		    untranslated=true;
		    
		    CEBUGNPQ("Untranslated!\tAAnumberinstrain: " << aanumberinstrain << "\tAAnumberinfeatureprot: " << strain_featureprot[featurecount][actstrainid].size() << endl);
		    
		  }
		  
		  // frameshift, aa changes and silents
		  if(makesframeshift) {
		    CEBUGNPQ("#1\n");
		    effectonprot="frameshift";
		    // corrective measures for "untranslated"
		    if(untranslated){
		      CEBUGNPQ("#2a\n");
		      if(baseinstrain=='*') {
			CEBUGNPQ("#3a\n");
			P->deletionuntranslated+=1;
		      }else{
			CEBUGNPQ("#4a\n");
			P->insertionuntranslated+=1;
		      }
		    }else{
		      CEBUGNPQ("#2b\n");
		      if(baseinstrain=='*') {
			CEBUGNPQ("#3b\n");
			P->deletioninlocus+=1;
		      }else{
			CEBUGNPQ("#4b\n");
			P->insertioninlocus+=1;
		      }
		    }
		  }else if(aainfeature[0]!=aainstrain[0]){
		    CEBUGNPQ("#5\n");
		    effectonprot="aa change";
		    
		    if(untranslated){
		      CEBUGNPQ("#6\n");
		      P->aachangeuntranslated+=1;
		    }else{
		      CEBUGNPQ("#7\n");
		      P->aachangeinlocus+=1;
		    }
		  }else{
		    CEBUGNPQ("#8\n");
		    effectonprot="silent";
		    if(untranslated){
		      CEBUGNPQ("#9\n");
		      P->silentuntranslated+=1;
		    }else{
		      CEBUGNPQ("#10\n");
		      P->silentinlocus+=1;
		    }
		  }
		  if(aainfeature[0]!='*' && aainstrain[0]=='*'){
		    CEBUGNPQ("#11\n");
		    effectonprot+=" / creates new stop codon";
		    P->prematurestop=true;
		  }
		  
		  // destroy/change Start/Stop
		  if(aanumberinfeature==0 && isstartinfeature[0]) {
		    CEBUGNPQ("#12\n");
		    if(!isstartinstrain[0]){
		      CEBUGNPQ("#13\n");
		      effectonprot+=" / destroys start codon";
		      P->destroyedstart=true;
		    }else{
		      CEBUGNPQ("#14\n");
		      effectonprot+=" / switches to another start codon";
		      P->changedstart=true;
		    }
		  }
		  if(aainfeature[0]=='*'){
		    CEBUGNPQ("#15\n");
		    if(aainstrain[0]!='*'){
		      CEBUGNPQ("#16\n");
		      effectonprot+=" / destroys stop codon";
		      P->destroyedstart=true;
		    }else{
		      CEBUGNPQ("#17\n");
		      effectonprot+=" / switches to another stop codon";
		      P->changedstart=true;
		    }
		  }
		  if(untranslated){
		    CEBUGNPQ("#18\n");
		    effectonprot+=" :: untranslated, protein ended earlier";
		  }
		}else{
		  CEBUGNPQ("#19\n");
		  effectonprot="unknown";

		  if(taginintergenic){
		    CEBUGNPQ("#20\n");
		    P->mutinintergenic+=1;
		  }
		}


		// for the feature analysis, only print out inlocus
		int32 freadposfrom=contigI->getRealReadPos(CTact->from,F->readindex);
		CEBUG("#c1" << endl);
		int32 adjposfroml=featureread.getLowerNonGapAdjustmentPosOfReadPos(freadposfrom)+1;
		CEBUG("#c2" << endl);
		int32 adjposfromr=featureread.getUpperNonGapAdjustmentPosOfReadPos(freadposfrom)+1;
		
		faout << contigI->getContigName() << "\t";
		//faout << contigI->getReadAtIndex(F->readindex).getName() << "\t";
		faout << rpool.getStrainOfStrainID(contigI->getReadAtIndex(F->readindex).getStrainID()) << "\t";
		faout << rpool.getStrainOfStrainID(actstrainid) << "\t";
		//faout << CTact->from << "\t" << contigI->paddedPos2UnpaddedPos(CTact->from) << "\t";
		faout << CTact->from << "\t" << adjposfroml;
		if(adjposfroml!=adjposfromr) faout << ':' << adjposfromr;
		faout << '\t' << (1.0/contigI->getContigLength()*adjposfroml*360);
		faout << '\t' << F->locustag << "\t";
		if(featuredir<0) {
		  faout << "complement\t";
		}else{
		  faout << "forward\t";
		}
		faout << F->gene << "\t";
		faout.flush();
		
		//hitother not used as this is pure biology
		//faout << "hitsother?" << "\t";
		//faout.flush();
		faout << reldistance << "\t";
		faout.flush();
		faout << dnaeffect << "\t";
		faout.flush();
		faout << baseinfeaturestrain << "->" << baseinstrain << "\t";
		faout << codoninfeature << "\t" << codoninstrain << "\t";
		faout.flush();
		
		faout << static_cast<uint16>(contigI->getQualityInRead(CTact->from,F->readindex)) << "\t";
		faout << static_cast<uint16>(consqual[CTact->from]) << "\t";
		
		faout.flush();
		
		faout << aainfeature[0] << "->" << aainstrain[0] << "\t";
		
		faout << aanumberinfeature << "\t";
		faout << static_cast<int16>(posinaainfeature) << "\t";
		faout << aanumberinstrain << "\t";
		faout << static_cast<int16>(posinaainstrain) << "\t";
		
		faout.flush();
		
		faout << effectonprot << "\t";
		faout << F->product << "\t";
		faout << F->function << "\t";
		faout << F->note << "\t";
		
		faout << endl;
	      }
	    }
	  }
	}
      }

      // went through all features of contig
      // now print summary for each interesting feature

      cout << "Summary of changes:\n";

      F=allGBfeatures.begin();
      P=allproteinchanges.begin();
      for(uint32 featurecount=0; F!=allGBfeatures.end(); F++, P++, featurecount++){
	CEBUGNPQ("Summary for " << F->locustag << '\t' << F->cfrom << '\t' << F->cto << endl);

	int32 freadposfrom=contigI->getRealReadPos(F->cfrom,F->readindex);
	CEBUG("#d1" << endl);
	int32 adjposfroml=contigI->getReadAtIndex(F->readindex).getLowerNonGapAdjustmentPosOfReadPos(freadposfrom)+1;
	CEBUG("#d2" << endl);

	fsout << F->locustag << '\t' << F->gene;

	fsout << '\t' << F->identifier;

	fsout << '\t' << (1.0/contigI->getReadAtIndex(F->readindex).getLenSeq()*adjposfroml*360) << '\t';

	// make a quick summary whether one should look at this protein
	{
	  // interestlevel 0 = no, 1 = perhaps, 2 = yes
	  uint32 interestlevel=0;
	  if(
	    P->changedstart
	    || P->changedstop

	    || P->silentinlocus > 0

	    || P->silentuntranslated    > 0

	    ){
	    CEBUGNPQ("Interest 1\n");
	    interestlevel=1;
	  }
	  if(
	    P->destroyedstart
	    || P->destroyedstop
	    || P->prematurestop
	    
	    || P->insertioninlocus > 0
	    || P->deletioninlocus  > 0
	    || P->aachangeinlocus  > 0
	    
	    || P->insertionuntranslated > 0           // in dubio pro reo
	    || P->deletionuntranslated  > 0           // in dubio pro reo
	    || P->aachangeuntranslated  > 0           // in dubio pro reo

	    || (P->coveragestatus!="ok"
		&& P->coveragestatus!="out of contig")
	    ){
	    CEBUGNPQ("Interest 2/std\n");
	    interestlevel=2;
	  }

	  // account for wrong annotation
	  if(!P->firstcodonisstart
	     &&	(P->silentinlocus >0 
		 || P->insertionuntranslated > 0           // in dubio pro reo
		 || P->deletionuntranslated  > 0           // in dubio pro reo
		 || P->aachangeuntranslated  > 0           // in dubio pro reo
	       )){
	    CEBUGNPQ("Interest 2/account wrong anno\n");
	    interestlevel=2;
	  }

	  // intergenics are always interesting
	  if(F->identifier == "Figr"
	     && P->mutinintergenic){
	    interestlevel=2;
	    CEBUGNPQ("Interest 2/intergenic\n");
	  }


	  if(interestlevel==0){
	    fsout << "no\t";
	  }else if(interestlevel==1){
	    fsout << "perhaps\t";
	  }else{
	    fsout << "yes\t";
	  }
	}

	fsout << '\t';       // own filter

	fsout << P->coveragestatus << '\t';

	if(F->identifier == "FCDS"){
	  if(P->firstcodonisstart) {fsout << "yes\t";}else{fsout << "no\t";}
	  if(P->changedstart) {fsout << "yes\t";}else{fsout << "no\t";}
	  if(P->destroyedstart) {fsout << "yes\t";}else{fsout << "no\t";}
	  if(P->changedstop) {fsout << "yes\t";}else{fsout << "no\t";}
	  if(P->destroyedstop) {fsout << "yes\t";}else{fsout << "no\t";}
	  if(P->prematurestop) {fsout << "yes\t";}else{fsout << "no\t";}
	}else{
	  fsout << "n/a\t"
		<< "n/a\t"
		<< "n/a\t"
		<< "n/a\t"
		<< "n/a\t"
		<< "n/a\t";
	}

	fsout << P->mutinintergenic << "\t";

	fsout << P->insertioninlocus << "\t";
	fsout << P->deletioninlocus << "\t";
	fsout << P->silentinlocus << "\t";
	fsout << P->aachangeinlocus << "\t";

	fsout << P->insertionuntranslated << "\t";
	fsout << P->deletionuntranslated << "\t";
	fsout << P->silentuntranslated << "\t";
	fsout << P->aachangeuntranslated << "\t";

	fsout << F->product << "\t";
	fsout << F->function << "\t";
	fsout << F->note;

	fsout << endl;


	fcout << contigI->getContigName() << "\t" <<  F->locustag << "\t" << F->gene;
	fcout << "\t" << rpool.getStrainOfStrainID(F->strainid); 
	// always start with strainid=1 as strainid 0 is "default", which
	//  we do not want
	for(uint32 strainid=1; strainid<strain_consseq.size(); strainid++){
	  if(strainid!=static_cast<uint32>(F->strainid)){
	    fcout << "\t" << rpool.getStrainOfStrainID(strainid);
	  }
	}

	string reference=strain_featureprot[featurecount][F->strainid];
	adjustCaseOfSequences(reference, strain_featureprot[featurecount]);

	fcout << "\t" << reference;

	for(uint32 strainid=1; strainid<strain_consseq.size(); strainid++){
	  if(strainid!=static_cast<uint32>(F->strainid)){
	    fcout << "\t" << strain_featureprot[featurecount][strainid];
	  }
	}

	reference=strain_featuredna[featurecount][F->strainid];
	adjustCaseOfSequences(reference, strain_featuredna[featurecount]);

	fcout << "\t" << reference;

	for(uint32 strainid=1; strainid<strain_consseq.size(); strainid++){
	  if(strainid!=static_cast<uint32>(F->strainid)){
	    fcout << "\t" << strain_featuredna[featurecount][strainid];
	  }
	}
	fcout << endl;
      }
    }

    fcout.close();
    fsout.close();
    faout.close();

    cout << "Done with feature analysis." << endl;

  }



/*************************************************************************
 *
 * Lowercase all strings, then compare each mutant string to reference string
 *
 * If there's a difference, uppercase reference string at that pos
 *  as well as the given mutant string at that pos
 *
 *************************************************************************/

  void adjustCaseOfSequences(string & reference, vector<string> & mutants)
  {
    size_t maxstring=reference.size();
    for(uint32 m=0; m<mutants.size(); m++) maxstring=max(maxstring,mutants[m].size());

    CEBUG("maxstring: " << maxstring << endl);

    for(size_t i=0; i<maxstring; i++){
      if(i<reference.size()) {
	reference[i]=static_cast<char>(tolower(reference[i]));
	bool ismutant=false;
	for(size_t m=0; m<mutants.size(); m++) {
	  if(i<mutants[m].size()){
	    mutants[m][i]=static_cast<char>(tolower(mutants[m][i]));
	    CEBUG("r[i] / m[m][i] 1: " << reference[i] << ' ' << mutants[m][i] << '\n');

	    if(reference[i] != mutants[m][i]){
	      CEBUG("r[i] / m[m][i] 1.6: " << reference[i] << ' ' << mutants[m][i] << '\n');
	      mutants[m][i]=static_cast<char>(toupper(mutants[m][i]));
	    }
	    CEBUG("r[i] / m[m][i] 2: " << reference[i] << ' ' << mutants[m][i] << '\n');
	  }else if(m!=0 || mutants[m].size()!=0){
	    // the above is for catching the "default" strain which might be
	    //  completely empty in assemblies where each read has a strain attached

	    // this mutant has ended, uppercase reference
	    ismutant=true;
	  }
	}
	if(ismutant) reference[i]=static_cast<char>(toupper(reference[i]));
      }else{
	// reference string has already ended, uppercase mutants
	for(size_t m=0; m<mutants.size(); m++) {
	  if(i<mutants[m].size()) mutants[m][i]=static_cast<char>(toupper(mutants[m][i]));
	}
      }
    }

    CEBUG("refstr: " << reference << '\n');
    for(uint32 m=0; m<mutants.size(); m++) {
      CEBUG("mutstr: " << mutants[m] << '\n');
    }

  }


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

  void saveAsFASTA(list<Contig> & clist, const string & filename, const string & paddedfilename, bool deleteoldfile)
  {
    FUNCSTART("void saveAsFASTA(list<Contig> & clist, const string & filename, const string & paddedfilename)");

    ofstream fastaout;
    ofstream fastapaddedout;
    ofstream qualout;
    ofstream qualpaddedout;

    string qualname;
    string paddedqualname;
    if(filename.size()) {
      qualname=filename+".qual";
      paddedqualname=paddedfilename+".qual";
    }

    openFileForAppend(filename.c_str(), fastaout, deleteoldfile);
    openFileForAppend(paddedfilename.c_str(), fastapaddedout, deleteoldfile);
    openFileForAppend(qualname.c_str(), qualout, deleteoldfile);
    openFileForAppend(paddedqualname.c_str(), qualpaddedout, deleteoldfile);

    // Save contigs first, then singlets
    for(uint32 savewhat=0; savewhat<2; savewhat++){
      list<Contig>::iterator contigI=clist.begin();
      for(;contigI!=clist.end();contigI++){
	bool saveme=false;
	if(savewhat==0 && contigI->getNumReadsInContig()>1) saveme=true;
	if(savewhat==1 && contigI->getNumReadsInContig()==1) saveme=true;
	if(saveme){
	  try{
	    Contig::setCoutType(Contig::AS_FASTAPADDED);
	    fastapaddedout << *contigI;
	    Contig::setCoutType(Contig::AS_FASTAPADDEDQUAL);
	    qualpaddedout << *contigI;
	    Contig::setCoutType(Contig::AS_FASTA);
	    fastaout << *contigI;
	    Contig::setCoutType(Contig::AS_FASTAQUAL);
	    qualout << *contigI;
	  }
	  catch (Notify n) {
	    cerr << "Error while dumping " << contigI->getContigName() << ".\n";
	    n.handleError(THISFUNC);
	  }
	}
      }
    }

    fastaout.close();
    qualout.close();

    FUNCEND();
  }

  void saveAsFASTA(Contig & con, const string & filename, const string & paddedfilename, bool deleteoldfile)
  {
    FUNCSTART("void saveAsFASTA(Contig & con, const string & filename, const string & paddedfilename)");

    ofstream fastaout;
    ofstream fastapaddedout;
    ofstream qualout;
    ofstream qualpaddedout;

    string qualname;
    string paddedqualname;
    if(filename.size()) {
      qualname=filename+".qual";
      paddedqualname=paddedfilename+".qual";
    }

    openFileForAppend(filename.c_str(), fastaout, deleteoldfile);
    openFileForAppend(paddedfilename.c_str(), fastapaddedout, deleteoldfile);
    openFileForAppend(qualname.c_str(), qualout, deleteoldfile);
    openFileForAppend(paddedqualname.c_str(), qualpaddedout, deleteoldfile);

    try{
      Contig::setCoutType(Contig::AS_FASTAPADDED);
      fastapaddedout << con;
      Contig::setCoutType(Contig::AS_FASTAPADDEDQUAL);
      qualpaddedout << con;
      Contig::setCoutType(Contig::AS_FASTA);
      fastaout << con;
      Contig::setCoutType(Contig::AS_FASTAQUAL);
      qualout << con;
    }
    catch (Notify n) {
      cerr << "Error while dumping " << con.getContigName() << ".\n";
      n.handleError(THISFUNC);
    }

    fastaout.close();
    fastapaddedout.close();
    qualout.close();
    qualpaddedout.close();
    
    FUNCEND();
  }


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

  void saveStrainsAsFASTAQ(list<Contig> & clist, const ReadPool & rp, const string & paddedfilename, bool asfastq, uint32 mincoverage, base_quality_t minqual, bool deleteoldfile, bool fillholesinstrain)
  {
    FUNCSTART("void saveAsFASTA(list<Contig> & clist, const string & paddedfilename)");

    //cout << "Saving padded strain contigs to FASTA file: " << paddedfilename << "_<strainname>.fasta" << endl;
    //cout << "(qualities have .qual appended)" << endl;

    vector<bool> allowtrunc(rp.getNumOfStrainInReadpool()+1,true);

    // Save contigs first, then singlets
    for(uint32 savewhat=0; savewhat<2; savewhat++){
      list<Contig>::iterator contigI=clist.begin();
      for(;contigI!=clist.end(); contigI++){
	bool saveme=false;
	if(savewhat==0 && contigI->getNumReadsInContig()>1) saveme=true;
	if(savewhat==1 && contigI->getNumReadsInContig()==1) saveme=true;
	if(saveme){
	  for(int32 strainid=-1; strainid<static_cast<int32>(rp.getNumOfStrainInReadpool()); strainid++){
	  //for(int32 strainid=2; strainid<=2; strainid++){
	    string strainfilename;
	    if(strainid>=0){
	      strainfilename=rp.getStrainOfStrainID(strainid);
	    }else{
	      strainfilename="AllStrains";
	    }
	    try{
	      if(asfastq){
		ofstream fastqpaddedout;
		openFileForAppend((paddedfilename+"_"+strainfilename+".padded.fastq").c_str(), fastqpaddedout, deleteoldfile & allowtrunc[1+strainid]);
		contigI->dumpStrainAsFASTQ(fastqpaddedout, 
					   mincoverage,
					   minqual,
					   strainid,
					   true,
					   fillholesinstrain);
		
		fastqpaddedout.close();
		openFileForAppend((paddedfilename+"_"+strainfilename+".unpadded.fastq").c_str(), fastqpaddedout, deleteoldfile & allowtrunc[1+strainid]);
		contigI->dumpStrainAsFASTQ(fastqpaddedout, 
					   mincoverage,
					   minqual,
					   strainid,
					   false,
					   fillholesinstrain);
		
		fastqpaddedout.close();
		allowtrunc[1+strainid]=false;
	      }else{
		ofstream fastapaddedout;
		ofstream qualpaddedout;
		openFileForAppend((paddedfilename+"_"+strainfilename+".padded.fasta").c_str(), fastapaddedout, deleteoldfile & allowtrunc[1+strainid]);
		openFileForAppend((paddedfilename+"_"+strainfilename+".padded.fasta.qual").c_str(), qualpaddedout, deleteoldfile & allowtrunc[1+strainid]);
		
		contigI->dumpStrainAsFASTAQUAL(fastapaddedout, 
					       qualpaddedout, 
					       mincoverage,
					       minqual,
					       strainid,
					       true,
					       fillholesinstrain);
		
		fastapaddedout.close();
		qualpaddedout.close();
		
		openFileForAppend((paddedfilename+"_"+strainfilename+".unpadded.fasta").c_str(), fastapaddedout, deleteoldfile & allowtrunc[1+strainid]);
		openFileForAppend((paddedfilename+"_"+strainfilename+".unpadded.fasta.qual").c_str(), qualpaddedout, deleteoldfile & allowtrunc[1+strainid]);
		
		contigI->dumpStrainAsFASTAQUAL(fastapaddedout, 
					       qualpaddedout, 
					       mincoverage,
					       minqual,
					       strainid,
					       false,
					       fillholesinstrain);
		fastapaddedout.close();
		qualpaddedout.close();
		allowtrunc[1+strainid]=false;
	      }
	    }
	    catch (Notify n) {
	      cerr << "Error while dumping " << contigI->getContigName() << " for strain " << strainfilename << ".\n";
	      n.handleError(THISFUNC);
	    }
	  }
	}
      }
    }

    FUNCEND();
  }
  



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

  void saveStrainsAsGBF(list<Contig> & clist, const ReadPool & rp, const string & gbfbasename, base_quality_t minqual, bool fillholesinstraingenomes, bool deleteoldfile)
  {
    // Save contigs first, then singlets
    for(uint32 savewhat=0; savewhat<2; savewhat++){
      list<Contig>::iterator contigI=clist.begin();
      for(;contigI!=clist.end(); contigI++){
	bool saveme=false;
	if(savewhat==0 && contigI->getNumReadsInContig()>1) saveme=true;
	if(savewhat==1 && contigI->getNumReadsInContig()==1) saveme=true;
	if(saveme){
	  string consseq;
	  vector<base_quality_t> consqual;
	  vector<string> strain_consseq;
	  vector< vector<base_quality_t> > strain_consqual;
	  strainid2name_t strainnames_in_contig;
	  
	  makeAllStrainGenomes(*contigI,
			       minqual,
			       consseq,
			       consqual,
			       strain_consseq,
			       strain_consqual,
			       strainnames_in_contig,
			       false);
	  
	  if(!fillholesinstraingenomes){
	    consseq.clear();
	    consqual.clear();
	  }

	  // we will look at all GBF features (empty allowed)
	  vector<multitag_t::mte_id_t> allowedfeatures;
	  // but not at Fsrc and Fgen
	  vector<multitag_t::mte_id_t> forbiddenfeatures;
	  forbiddenfeatures.push_back(Read::REA_tagentry_idFsrc);
	  
	  // copy first all Genbank features of all reads into one big vector
	  //  (makes the search loop afterwards easier)
	  
	  list<gbfsummary_t> allGBfeatures;
	  contigI->getGBFSummary(allGBfeatures,
				 allowedfeatures,
				 forbiddenfeatures,
				 false);
	  
	  //// dump contigs only for strains that are in this contig
	  
	  strainid2name_t::const_iterator SIC=strainnames_in_contig.begin();
	  for(; SIC!= strainnames_in_contig.end(); SIC++){
	    int32 actstrainid=SIC->first;
	    CEBUGNPQ("Strainid: " << actstrainid << "\tStrainname:" << SIC->second << endl);
	    
	    ofstream gbfout;
	    openFileForAppend((gbfbasename+"_"+rp.getStrainOfStrainID(actstrainid)+".gbf").c_str(), 
			      gbfout,
			      deleteoldfile);
	    
	    // beware! strain_consseq[actstrainid] will be different after call
	    dumpSequenceAsGBF_priv(contigI->getContigName(),
				   SIC->second,
				   strain_consseq[actstrainid],
				   consseq,
				   allGBfeatures,
				   gbfout);
	    
	    gbfout.close();
	  }
	}
      }
    }

  }


/*************************************************************************
 *
 *  beware! UGLY! (but faster than first copying the whole string):
 *   "seq" will be different after call (lower case and eventually
 *   filled up with altseq if a character in seq is == 'n'
 *
 *************************************************************************/

//#define CEBUG(bla)  {cout << bla; cout.flush();}

  void dumpSequenceAsGBF_priv(const string & seqname, const string & strainname, string & seq, const string & consseq, const list<gbfsummary_t> & allGBfeatures, ostream & fout)
  {
    // position mapping of consensus: padded (in memory) vs unpadded
    // positions with a * get position of last non-* assigned
    uint32 actdepadpos=0;
    vector<uint32> depadposmap;
    depadposmap.resize(seq.size(),0);
    for(uint32 seqi=0; seqi < seq.size();seqi++){
      depadposmap[seqi]=actdepadpos;
      char actbase=static_cast<char>(tolower(seq[seqi]));
      if((actbase=='n' || actbase=='@') && !consseq.empty()) actbase=static_cast<char>(toupper(consseq[seqi]));
      seq[seqi]=actbase;
      if(actbase!='*') actdepadpos++;
    }

    CEBUGF("depadposmap.size(): " << depadposmap.size() << endl);

    fout << "LOCUS       ";
    // dummy
    fout.width(16);
    fout << left << seqname << "  ";
    fout << actdepadpos << " bp    DNA     ENV    ";

    {
      char timestr[80];
      struct tm *timepoint;
      time_t t;

      time(&t);
      timepoint=localtime(&t);

      strftime(&timestr[0],60,"%d-%b-%Y",timepoint);
	
      fout << timestr << "\n";
    }

    fout << "SOURCE      " <<
      "DNA, strain: "<< strainname <<
      "\nACCESSION   " << seqname << 
      "\nCOMMENT     Generated by the MIRA assembler\n"
      "FEATURES             Location/Qualifiers\n"
      "     source          1.." << actdepadpos << endl <<
      "                     /strain=\"" << strainname << "\"\n";

    list<gbfsummary_t>::const_iterator F=allGBfeatures.begin();
    for(; F!=allGBfeatures.end(); F++){
      CEBUGF("\n" << F->locustag << "\t" << F->cfrom << "\t" << F->cto << "\t");
      if(F->cfrom >= depadposmap.size() || F->cto >= depadposmap.size()) continue;

      fout << "     ";
      fout.width(16);

      fout << left << GBF::translateGAP4GBFfeat2GBFfeat(F->identifier);
      CEBUGF("0");
      if(F->direction >=0) {
	fout << depadposmap[F->cfrom]+1 << ".." << depadposmap[F->cto]+1 << "\n";
      }else{
	fout << "complement(" << depadposmap[F->cfrom]+1 << ".." << depadposmap[F->cto]+1 << ")\n";
      }

      CEBUGF("1");

      if(!F->gene.empty()){
	dumpTextAsGBFValueLine_priv("/gene",F->gene,fout);
      }
      if(!F->locustag.empty()){
	dumpTextAsGBFValueLine_priv("/locus_tag",F->locustag,fout);
      }
      if(!F->function.empty()){
	dumpTextAsGBFValueLine_priv("/function",F->function,fout);
      }
      if(!F->ecnumber.empty()){
	dumpTextAsGBFValueLine_priv("/EC_number",F->ecnumber,fout);
      }
      CEBUGF("3");
      if(!F->product.empty()){
	dumpTextAsGBFValueLine_priv("/product",F->product,fout);
      }
      CEBUGF("4");
      if(!F->note.empty()){
	dumpTextAsGBFValueLine_priv("/note",F->note,fout);
      }
      CEBUGF("5");
      if(F->mustbetranslated){
	// dump translation
	string featureprot;
	string featuredna;
	if(F->direction >=0) {
	  dptools::dnaToProtein(seq,
				featureprot,
				featuredna,
				F->cfrom,
				F->cfrom,
				1,
				F->translationtable,
				F->codonstart);
	}else{
	  dptools::dnaToProtein(seq,
				featureprot,
				featuredna,
				F->cto,
				F->cto,
				-1,
				F->translationtable,
				F->codonstart);
	}
	CEBUGF("6");
	if(!featureprot.empty()){
	  // dnaToProtein includes the * stop codon, remove that before output
	  featureprot.resize(featureprot.size()-1);
	  dumpTextAsGBFValueLine_priv("/translation",featureprot,fout);
	}
	CEBUGF("7");
      }
    }

    fout << "ORIGIN\n";

    uint8 posinline=0;
    uint32 depadpos=1;
    bool isnewline=true;
    bool addspace=false;
    for(uint32 seqi=0; seqi<seq.size(); seqi++){
      if(isnewline) {
	isnewline=false;
	fout.width(9);
	fout << right << depadpos;
      }
      if(posinline%10 == 0){
	addspace=true;
      }
      if(seq[seqi]!='*'){
	if(addspace) {
	  addspace=false;
	  fout << " ";
	}
	fout << seq[seqi];
	depadpos++;
	posinline++;
	if(posinline==60) {
	  fout << "\n";
	  posinline=0;
	  isnewline=true;
	}
      }
    }
    if(!isnewline){
      fout << "\n";
    }
    fout << "//\n";

  }

//#define CEBUGF(bla)


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/
  void dumpTextAsGBFValueLine_priv(const string & key, const string & value, ostream & fout)
  {
    //fout << "                     /gene=\"" << F->gene << "\"\n";

    fout << "                     " << key << "=\"";
    //fout << value;

    fout.flush();

    string::size_type posinline=21+key.size()+2;

    string::size_type fpos=0;
    string::size_type tpos=0;

    bool firstwordinline=true;

    int32 runawaystop=100000;

    // print out "words" until the end of the string
    while(fpos<value.size()){

      if(--runawaystop==0) {
	cout << "Abort: runawaystop\n";
	exit(1);
      }

      bool splitit=false;
      bool onnextline=false;

      // find next space (or end of string)
      tpos=value.find(" ",fpos);
      if(tpos==string::npos){
	tpos=value.size();
      }

      string::size_type spaceneeded=tpos-fpos;
      if(!firstwordinline) spaceneeded++;

      // see whether there is enough place remaining in the line
      //  (careful for space when not firstword??)
      if(posinline+spaceneeded > 79) onnextline=true;

      // max 58 chars in one line, if greater, well have to split anyways
      if(spaceneeded > 58) splitit=true;

      if(splitit) {
	for(; fpos<value.size() && posinline<79; fpos++,posinline++) {
	  fout << static_cast<char>(value[fpos]);
	}
	if(posinline==79){
	  fout << "\n                     ";
	  posinline=21;
	  firstwordinline=true;
	}
      } else {
	if (onnextline) {
	  fout << "\n                     ";
	  posinline=21;
	  firstwordinline=true;
	}
	// there is enough place for this word, simply print it
	if(!firstwordinline) {
	  fout << " ";
	  posinline++;
	}
	fout << value.substr(fpos, tpos-fpos);
	posinline+=tpos-fpos;
	fpos+=tpos-fpos+1;  // NOT spaceneeded, +1 to jump over space
	firstwordinline=false;
      }

    }
    if(posinline>=79) fout << "\n                     ";

    fout << "\"\n";
  }



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

  void dumpContigs(list<Contig> & clist, ostream & fout)
  {
    FUNCSTART("void dumpContigs(list<Contig> & clist, ostream & fout)");
    for(uint32 savewhat=0; savewhat<2; savewhat++){
      list<Contig>::iterator contigI=clist.begin();
      for(;contigI!=clist.end(); contigI++){
	bool saveme=false;
	if(savewhat==0 && contigI->getNumReadsInContig()>1) saveme=true;
	if(savewhat==1 && contigI->getNumReadsInContig()==1) saveme=true;
	try{
	  if(saveme) {
	    fout << *contigI;
	  }
	}
	catch (Notify n) {
	  cerr << "Error while dumping " << contigI->getContigName() << ".\n";
	  n.handleError(THISFUNC);
	}
      }
    }
    FUNCEND();
  }


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

  void saveAs_TYPE(list<Contig> & clist, const string & filename, const uint8 type, bool deleteoldfile)
  {
    FUNCSTART("void saveAs_TYPE(list<Contig> & clist, const string & filename, const uint8 type, bool deleteoldfile)");

    ofstream fout;

    // nope, this doesn't speed up simple output
    //
    // char mybuf[1024*1024];
    // fout.rdbuf()->pubsetbuf(mybuf,1024*1024);

    if(!openFileForAppend(filename,fout, deleteoldfile)){
      if(type==Contig::AS_TCS) Contig::dumpTCS_Head(fout);
    }
    Contig::setCoutType(type);
    dumpContigs(clist,fout);
    fout.close();

    FUNCEND();
  }
  void saveAs_TYPE(Contig & con, const string & filename, const uint8 type, bool deleteoldfile)
  {
    FUNCSTART("void saveAs_TYPE(Contig & con, const string & filename, const uint8 type)");

    ofstream fout;

    // nope, this doesn't speed up simple output
    //
    // char mybuf[1024*1024];
    // fout.rdbuf()->pubsetbuf(mybuf,1024*1024);

    if(!openFileForAppend(filename,fout, deleteoldfile)){
      if(type==Contig::AS_TCS) Contig::dumpTCS_Head(fout);
    }
    Contig::setCoutType(type);
    try{
      fout << con;
    }
    catch (Notify n) {
      cerr << "Error while dumping " << con.getContigName() << ".\n";
      n.handleError(THISFUNC);
    }
    fout.close();

    FUNCEND();
  }



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

  void dumpAsACE(list<Contig> & clist, ostream & aceout)
  {
    FUNCSTART("void dumpAsACE(list<Contig> & clist, ostream & aceout)");

    aceout << "AS " << clist.size();
    {
      uint32 sum=0;
      list<Contig>::iterator contigI=clist.begin();
      while(contigI!=clist.end()){
	sum+=contigI->getNumReadsInContig();
	contigI++;
      }
      aceout << " " << sum << endl << endl;
    }

    Contig::setCoutType(Contig::AS_ACE);
    dumpContigs(clist, aceout);

    FUNCEND();
  }


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

  void saveAsACE(list<Contig> & clist, const string & filename, bool deleteoldfile)
  {
    //declare a new file object for appending and overwriting
    fstream fio;

    uint32 oldnumcontigs=0;
    uint32 oldnumreads=0;
    saveAsACE_openACE(fio, 
		      filename,
		      deleteoldfile,
		      oldnumcontigs, 
		      oldnumreads);

    Contig::setCoutType(Contig::AS_ACE);
    dumpContigs(clist, fio);
    
    uint32 newnumreads=oldnumreads;
    {
      list<Contig>::iterator contigI=clist.begin();
      while(contigI!=clist.end()){
	newnumreads+=contigI->getNumReadsInContig();
	contigI++;
      }
    }

    saveAsACE_rewriteHeader(fio, 
			    oldnumcontigs+static_cast<uint32>(clist.size()),
			    newnumreads);

    fio.close();
  }

  void saveAsACE(Contig & con, const string & filename, bool deleteoldfile)
  {
    FUNCSTART("void saveAsACE(Contig & con, const string & filename, bool deleteoldfile)");

    //declare a new file object for appending and overwriting
    fstream fio;

    uint32 oldnumcontigs=0;
    uint32 oldnumreads=0;

    saveAsACE_openACE(fio, 
		      filename,
		      deleteoldfile,
		      oldnumcontigs,
		      oldnumreads);
    Contig::setCoutType(Contig::AS_ACE);
    try{
      fio << con;
    }
    catch (Notify n) {
      cerr << "Error while dumping " << con.getContigName() << ".\n";
      n.handleError(THISFUNC);
    }
    saveAsACE_rewriteHeader(fio,
			    oldnumcontigs+1,
			    oldnumreads+con.getNumReadsInContig());
    fio.close();
    FUNCEND();
  }

  void saveAsACE_openACE(fstream & fio, const string & filename, bool deleteoldfile, uint32 & numcontigs, uint32 & numreads)
  {
    FUNCSTART("void saveAsACE_openACE(fstream & fio, const string & filename, uint32 & numcontigs, uint32 & numreads)");

    numcontigs=0;
    numreads=0;

    struct stat st;
    if(deleteoldfile || stat(filename.c_str(),&st)) {
      // oh, new file
      fio.open(filename.c_str(), ios::out | ios::in | ios::trunc);
      // write a long empty line to reserve space for header
      fio << "                                                                                                                                                                           \n\n";
      return;
    }
    fio.open(filename.c_str(), ios::ate | ios::out | ios::in);

    long fiosize = fio.tellp();
    fio.seekp(0);

    string dummy;
    if(fio.eof()){ 
      MIRANOTIFY(Notify::INTERNAL, "the ace file is present but seems to be empty: " << filename);
    }
    getline(fio,dummy);
    if(dummy.size()<50){
      MIRANOTIFY(Notify::INTERNAL, "first line is too short for rewriting: " << filename);
    }
    fio.seekp(0);
    fio >> dummy >> numcontigs >> numreads;
    fio.seekp(fiosize);
  }

  void saveAsACE_rewriteHeader(fstream & fio, const uint32 numcontigs, const uint32 numreads)
  {
    fio.seekp(0);
    //fio << "AS " << numcontigs << ' ' << numreads << 
    //  "                                                  ";

    string tmp="AS ";
    tmp+=boost::lexical_cast<string>(numcontigs);
    tmp+=" ";
    tmp+=boost::lexical_cast<string>(numreads);
    while(tmp.size()<50) tmp+=" ";
    fio << tmp;
  }

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

  void saveAsGAP4DA(list<Contig> & clist, const string & dirname, bool deleteolddir)
  {
    FUNCSTART("void saveAsGAP4DA(list<Contig> & clist, const string & dirname)");

    if(ensureDirectory(dirname, deleteolddir)){
      MIRANOTIFY(Notify::FATAL, "Could not make sure that directory '" << dirname << "' exists, aborting MIRA.");
    }

    Contig::setCoutType(Contig::AS_GAP4DA);
    ofstream fofnout((dirname+"/fofn").c_str(), ios::out | ios::app);

    for(uint32 savewhat=0; savewhat<2; savewhat++){
      list<Contig>::iterator contigI=clist.begin();
      for(;contigI!=clist.end();contigI++){
	bool saveme=false;
	if(savewhat==0 && contigI->getNumReadsInContig()>1) saveme=true;
	if(savewhat==1 && contigI->getNumReadsInContig()==1) saveme=true;
	if(saveme){
	  try{
	    contigI->saveAsGAP4DA(dirname, fofnout);
	  }
	  catch (Notify n) {
	    cerr << "Error while dumping " << contigI->getContigName() << ".\n";
	    n.handleError(THISFUNC);
	  }
	}
      }
    }
    fofnout.close();

    FUNCEND();
    return;
  }

  void saveAsGAP4DA(Contig & con, const string & dirname, bool deleteolddir)
  {
    FUNCSTART("void saveAsGAP4DA(Contig & con, const string & dirname)");

    if(ensureDirectory(dirname, deleteolddir)){
      MIRANOTIFY(Notify::FATAL, "Could not make sure that directory '" << dirname << "' exists, aborting MIRA.");
    }

    Contig::setCoutType(Contig::AS_GAP4DA);
    ofstream fofnout((dirname+"/fofn").c_str(), ios::out | ios::app);
    try{
      con.saveAsGAP4DA(dirname, fofnout);
    }
    catch (Notify n) {
      cerr << "Error while dumping " << con.getContigName() << ".\n";
      n.handleError(THISFUNC);
    }
    fofnout.close();

    FUNCEND();
    return;
  }



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

  void saveAsWiggle(list<Contig> & clist, const string & filename, bool deleteoldfile)
  {
    FUNCSTART("void saveAsWiggle(list<Contig> & clist, const string & filename, bool deleteoldfile)");

    ofstream fout;
    openFileForAppend(filename,fout, deleteoldfile);
    fout.close();

    for(uint32 savewhat=0; savewhat<2; savewhat++){
      list<Contig>::iterator contigI=clist.begin();
      for(;contigI!=clist.end(); contigI++){
	bool saveme=false;
	if(savewhat==0 && contigI->getNumReadsInContig()>1) saveme=true;
	if(savewhat==1 && contigI->getNumReadsInContig()==1) saveme=true;
	try{
	  if(saveme){
	    saveAsWiggle(*contigI, filename, false);
	  }


	}
	catch (Notify n) {
	  cerr << "Error while dumping " << contigI->getContigName() << ".\n";
	  n.handleError(THISFUNC);
	}
      }
    }

    FUNCEND();
  }

  void saveAsWiggle(Contig & con, const string & filename, bool deleteoldfile)
  {
    FUNCSTART("void saveAsWiggle(Contig & con, const string & filename, bool deleteoldfile)");
  
    ofstream fout;
    openFileForAppend(filename,fout, deleteoldfile);

    try{
      // we just want to know which strain has the GB features
      vector<multitag_t::mte_id_t> allowedfeatures;
      allowedfeatures.push_back(Read::REA_tagentry_idFsrc);
      // but not at Fsrc
      vector<multitag_t::mte_id_t> forbiddenfeatures;
      
      list<gbfsummary_t> allGBfeatures;
      con.getGBFSummary(allGBfeatures,allowedfeatures,forbiddenfeatures, true);
      
      int32 featurestrainid=0;
      if(!allGBfeatures.empty()) featurestrainid=allGBfeatures.front().strainid;
      
      string consseq;
      vector<base_quality_t> dummy;
      
      con.calcConsensi();
      con.newConsensusGet(consseq, 
			  dummy, 
			  featurestrainid);

      con.dumpWiggle_Body(fout, consseq);

    }
    catch (Notify n) {
      cerr << "Error while saving " << con.getContigName() << " as wiggle.\n";
      n.handleError(THISFUNC);
    }
    fout.close();
  
    FUNCEND();
  }



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

  void dumpHTMLHeader(const string & projectname, ostream & htmlout)
  {
    htmlout << "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0//EN\">\n\
<html>\n\
<head>\n\
   <meta http-equiv=\"Content-Type\" content=\"text/html; charset=iso-8859-1\">\n\
   <meta name=\"GENERATOR\" content=\"MIRA (c) Bastien Chevreux & EdIt (c) Thomas Pfisterer;\">\n\
   <meta name=\"Author\" content=\"";

    //char * logname= getenv("LOGNAME");
    //htmlout << getlogin() << "\">\n

    {
      // REMARK:
      // GCC 3.4 tells this
      // : warning: Using 'getpwuid' in statically linked applications 
      //   requires at runtime the shared libraries from the glibc version
      //   used for linking
      //
      //  This might also be the reason for a reported crash of
      //   convert_project on a 2.6 kernel (with another glibc than my
      //   home machine
      //
      // Resolve: get back to getlogin() even if manual says it can be
      //  easily "fooled". This application is not security critical.

      //struct passwd * pws = getpwuid(getuid());
      //bool noname=true;
      //if(pws != NULL) {
      //	if(pws->pw_name != NULL && pws->pw_gecos != NULL) {
      //	  htmlout << pws->pw_name << " (" << pws->pw_gecos << ")";
      //	  noname=false;
      //	}
      //}

      char * namestr=getlogin();
      if(namestr==NULL) {
	namestr=getenv("LOGNAME");
	if(namestr==NULL) {
	  namestr=getenv("USER");
	}
      }
    
      if(namestr!=NULL){
	htmlout << namestr;
      }else{
	htmlout << "unknown";
      }
    }

    htmlout << "\">\n<meta name=\"Description\" content=\"Assembled shotgun project\">\n\
   <title>";
    htmlout << "Project " << projectname << " </title>\n\
  <STYLE TYPE=\"text/css\">\n\
  <!--\n\
  \n\
   .FCDS {color:black;  background-color:#4AA090;}\n\
   .FrRN {color:black;  background-color:#f41e8e;}\n\
   .FtRN {color:black;  background-color:#736cdc;}\n\
   .FmxR {color:black;  background-color:#653BD9;}\n\
   .MISM {color:red;  background-color:#dddddd;}\n\
   .SRMr {color:black;  background-color:#ff5050;}\n\
   .SRMc {color:black;  background-color:#ff5050;}\n\
   .WRMr {color:black;  background-color:orange;}\n\
   .WRMc {color:black;  background-color:orange;}\n\
   .SROr {color:black;  background-color:#00ced1;}\n\
   .SROc {color:black;  background-color:#00ced1;}\n\
   .SAOr {color:black;  background-color:#2e8b57;}\n\
   .SAOc {color:black;  background-color:#2e8b57;}\n\
   .SIOr {color:black;  background-color:#98fb98;}\n\
   .SIOc {color:black;  background-color:#98fb98;}\n\
   .MCVc {color:black;  background-color:#cc3333;}\n\
   .POLY {color:black;  background-color:#ffff99;}\n\
   .EDxD {color:black;  background-color:#db7093;}\n\
   .EDxI {color:black;  background-color:#db7093;}\n\
   .EDxC {color:black;  background-color:#db7093;}\n\
   .IUPC {color:black;  background-color:#cccccc;}\n\
\n\
BODY  { font-family: sans-serif;\n\
  color: #000000 ;\n\
}\n\
\n\
   .jtable1 {\n\
     color : black; \n\
     background-color : #cccccc ;\n\
     font-size: normal ;\n\
     font-style: normal ;\n\
     font-family: sans-serif ; \n\
     font-weight: normal ;\n\
     text-align: left ; \n\
     vertical-align: top ;\n\
     padding: 10px;\n\
   }\n\
   .jtable2 {\n\
     color : black; \n\
     background-color : #eeeeee ;\n\
     font-size: normal ;\n\
     font-style: normal ;\n\
     font-family: sans-serif ; \n\
     font-weight: normal ;\n\
     text-align: left ; \n\
     vertical-align: top ;\n\
     padding: 10px;\n\
   }\n\
   .jtable3 {\n\
     color : black; \n\
     background-color : white ;\n\
     font-size: normal ;\n\
     font-style: normal ;\n\
     font-family: sans-serif ; \n\
     font-weight: normal ;\n\
     text-align: left ; \n\
     vertical-align: top ;\n\
     padding: 10px;\n\
   }\n\
\n\
  -->\n\
</STYLE>\n\
</head>\n\
<body TEXT=\"#000000\" BGCOLOR=\"#FFFFFF\" LINK=\"#FF0000\" VLINK=\"#551A8B\" ALINK=\"#000088\">\n";

    //   .ALUS {color:black;  background-color:#90ee90;}\n
    // #66ffff = azure (pi*daumen)
    
    htmlout << "<h1><center>Tag legend</center></h1>\n"; 
    
    htmlout << "<center>\n";
    htmlout << "<table CELLSPACING=0 CELLPADDING=0 NOSAVE >\n";
    
    htmlout << "<tr align=\"left\"><td><tt><SPAN CLASS=\"FCDS\">&nbsp;</SPAN> = FCDS;</tt></td><td>Feature CDS (coding sequence)</td></tr>\n";
    htmlout << "<tr align=\"left\"><td><tt><SPAN CLASS=\"FtRN\">&nbsp;</SPAN> = FtRN;</tt></td><td>tRNA</td></tr>\n";
    htmlout << "<tr align=\"left\"><td><tt><SPAN CLASS=\"FrRN\">&nbsp;</SPAN> = FrRN;</tt></td><td>rRNA</td></tr>\n";
    htmlout << "<tr align=\"left\"><td><tt><SPAN CLASS=\"FmxR\">&nbsp;</SPAN> = Fm-R;</tt></td><td>misc. RNA</td></tr>\n";
    htmlout << "<tr align=\"left\"><td><tt><SPAN CLASS=\"MISM\">&nbsp;</SPAN> = MISM;</tt></td><td>Mismatch (discrepancy) between reads and consensus</td></tr>\n";
    //htmlout << "<tr><td><tt><SPAN CLASS=\"ALUS\">&nbsp;</SPAN> = ALUS;</tt> Repetitive ALU sequence</td></tr>\n";
    htmlout << "<tr align=\"left\"><td><tt><SPAN CLASS=\"SRMr\">&nbsp;</SPAN> = SRMx;</tt></td><td>Strong Repeat Marker Base set by MIRA</td></tr>\n";
    htmlout << "<tr align=\"left\"><td><tt><SPAN CLASS=\"WRMr\">&nbsp;</SPAN> = WRMx;</tt></td><td>Weak Repeat Marker Base set by MIRA</td></tr>\n";
    htmlout << "<tr align=\"left\"><td><tt><SPAN CLASS=\"SROr\">&nbsp;</SPAN> = SROx;</tt></td><td>SNP inteR Organism (Read/Consensus) set by MIRA</td></tr>\n";
    htmlout << "<tr align=\"left\"><td><tt><SPAN CLASS=\"SAOr\">&nbsp;</SPAN> = SAOx;</tt></td><td>SNP intrA Organism (Read/Consensus) set by MIRA</td></tr>\n";
    htmlout << "<tr align=\"left\"><td><tt><SPAN CLASS=\"SIOr\">&nbsp;</SPAN> = SIOx;</tt></td><td>SNP Inter- and intra-Organism (Read/Consensus) set by MIRA</td></tr>\n";
    htmlout << "<tr align=\"left\"><td><tt><SPAN CLASS=\"MCVc\">&nbsp;</SPAN> = MCVc;</tt></td><td>Missing CoVerage in Consensus (set by MIRA)</td></tr>\n";
    htmlout << "<tr align=\"left\"><td><tt><SPAN CLASS=\"POLY\">&nbsp;</SPAN> = POLY;</tt></td><td>Poly-A signal</td></tr>\n";
    htmlout << "<tr align=\"left\"><td><tt><SPAN CLASS=\"EDxD\">&nbsp;</SPAN> = EDxD;</tt></td><td>Delete operation set by EdIt</td></tr>\n";
    htmlout << "<tr align=\"left\"><td><tt><SPAN CLASS=\"EDxI\">&nbsp;</SPAN> = EDxI;</tt></td><td>Insert operation set by EdIt</td></tr>\n";
    htmlout << "<tr align=\"left\"><td><tt><SPAN CLASS=\"EDxC\">&nbsp;</SPAN> = EDxC;</tt></td><td>Change operation set by EdIt</td></tr>\n";
    htmlout << "<tr align=\"left\"><td><tt><SPAN CLASS=\"IUPC\">&nbsp;</SPAN> = IUPAC;</tt></td><td> IUPAC base (shows only in HTML output)</td></tr>\n";
    
    htmlout<< "</table></center>\n";
  }


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

  void dumpContigListAsHTML(list<Contig> & clist, const string & filename, bool deleteoldfile, const string & projectname)
  {
    FUNCSTART("void dumpContigListAsHTML(list<Contig> & clist, const string & filename, bool deleteoldfile, const string & projectname)");

    ofstream fout;
    if(!openFileForAppend(filename,fout, deleteoldfile)){
      dumpHTMLHeader(projectname, fout);
    }

    // A contig list at the top of the file is currently not possible 
    //  anymore *sigh*
    // maybe splitting into two files and then concatenate at the of the process?
    //
    //list<Contig>::iterator I=clist.begin();
    //htmlout << "<h1><center>Contig List</center></h1>\n"; 
    //while(I!=clist.end()){
    //  
    //  if(I->getContigReads().size() > 1) {
    //	htmlout << "<a href=\"#" << I->getContigName() << "\">Contig " <<  I->getContigID();
    //  }else{
    //	htmlout << "<a href=\"#" << I->getContigName() << "\">Singlet " <<  I->getContigID();
    //  }
    //  htmlout << " (" << I->getContigLength() << ")</a>";
    //  I++;
    //  if(I!=clist.end()){
    //	htmlout << ", ";
    //  }else{
    //	htmlout << "\n<p>\n";
    //  }
    //}
    
    Contig::setCoutType(Contig::AS_HTML);
    dumpContigs(clist,fout);
    
    // This is also bad ... when should the HTML be closed?
    //fout << "\n</body></html>";

    fout.close();
    
    return;

    FUNCEND();
  }


  void dumpContigAsHTML(Contig & con, const string & filename, bool deleteoldfile, const string & projectname)
  {
    FUNCSTART("void dumpContigAsHTML(Contig & con, const string & filename, bool deleteoldfile, const string & projectname)");

    ofstream fout;
    if(!openFileForAppend(filename,fout, deleteoldfile)){
      dumpHTMLHeader(projectname, fout);
    }

    // A contig list at the top of the file is currently not possible 
    //  anymore *sigh*
    // maybe splitting into two files and then concatenate at the of the process?
    
    Contig::setCoutType(Contig::AS_HTML);
    try{
      fout << con;
    }
    catch (Notify n) {
      cerr << "Error while dumping " << con.getContigName() << " as HTML.\n";
      n.handleError(THISFUNC);
    }
    fout.close();

    
    // This is also bad ... when should the HTML be closed?
    //fout << "\n</body></html>";

    fout.close();
    
    return;

    FUNCEND();
  }


}
