; $Id: axiom.scm,v 1.93 2007/09/10 08:22:24 schwicht Exp $
; 8. Assumption variables and axioms
; ==================================
; To be renamed into avars scheme, with the axioms section transferred
; into the new aconst.scm (was globalas.scm)

; 8-1. Assumption variables
; =========================

; Assumption variables are implemented as lists ('avar formula index name).

; To make sure that assumption variables generated by the system are
; different from all user introduced assumption variables, we maintain a
; global counter MAXAVARINDEX.  Whenever the user introduces an
; assumption variable, e.g. by (make-avar formula 27 ""), then MAXAVARINDEX
; is incremented to at least 27.

(define MAXAVARINDEX -1)
(define INITIAL-MAXAVARINDEX MAXAVARINDEX)

; Constructor, accessors and tests for assumption variables:

(define (make-avar formula index name)
  (set! MAXAVARINDEX (max index MAXAVARINDEX))
  (list 'avar formula index name))

(define avar-to-formula cadr)
(define avar-to-index caddr)
(define avar-to-name cadddr)

(define (avar-form? x) (and (pair? x) (eq? 'avar (car x))))

(define (avar? x)
  (and (avar-form? x)
       (list? x)
       (= 4 (length x))
       (let ((formula (cadr x))
	     (index (caddr x))
	     (name (cadddr x)))
	 (and (formula? formula)
	      (<= -1 index)
	      (<= index MAXAVARINDEX)
	      (string? name)))))

(define (avar=? avar1 avar2)
  (or (eq? avar1 avar2)
      (and (avar-form? avar1) (avar-form? avar2)
	   (= (avar-to-index avar1) (avar-to-index avar2))
	   (string=? (avar-to-name avar1) (avar-to-name avar2)))))

; For display we use

(define (avar-to-string avar)
  (let ((name (avar-to-name avar))
	(index (avar-to-index avar)))
    (string-append
     (if (string=? "" name) DEFAULT-AVAR-NAME name)
     (if (= -1 index) "" (number-to-string (+ 1 index))))))

; For automatic generation of assumption variables (e.g. for bound
; renaming) we provide

(define (formula-to-new-avar formula . optional-name)
  (if (null? optional-name)
      (make-avar formula (+ 1 MAXAVARINDEX) "")
      (let ((string (car optional-name)))
	(if (string? string)
	    (make-avar formula (+ 1 MAXAVARINDEX) string)
	    (myerror "formula-to-new-avar" "string expected"
		     (car optional-name))))))

(define DEFAULT-AVAR-NAME "u")

; For convenience we add mk-avar with options.  Options are index (default
; -1) and name (default DEFAULT-AVAR-NAME)

(define (mk-avar formula . options)
  (let ((index -1)
	(name DEFAULT-AVAR-NAME))
    (if (pair? options)
	(begin (set! index (car options))
	       (set! options (cdr  options))))
    (if (pair? options)
	(begin (set! name (car options))
	       (set! options (cdr  options))))
    (if (pair? options)
	 (myerror "make-avar" "unexpected argument" options))
  (cond ((not (and (integer? index) (<= -1 index)))
	 (myerror "make-avar" "index >= -1 expected" index))
	((not (formula-form? formula))
	 (myerror "make-avar" "formula expected" formula))
	((not (string? name))
	 (myerror "make-avar" "string expected" name))
	(else (make-avar formula index name)))))

(define (normalize-avar avar)
  (make-avar (normalize-formula (avar-to-formula avar))
	     (avar-to-index avar)
	     (avar-to-name avar)))


; 8-2. Assumption constants
; =========================

; An assumption constant appears in a proof, as an axiom, a theorem or
; a global assumption.  Its formula is given as an `uninstantiated
; formula', where only type and predicate variables can occur freely;
; these are considered to be bound in the assumption constant. In the
; proof the bound type variables are implicitely instantiated by
; types, and the bound predicate variables by cterms (the arity of a
; cterm is the type-instantiated arity of the corresponding pvar).
; Since we do not have type and predicate quantification in formulas,
; the aconst contains these parts left implicit in the proof: tsubst
; and pinst (will become a psubst, once the arities of pvars are
; type-instantiated with tsubst).

; To normalize a proof we will first translate it into a term, then
; normalize the term and finally translate the normal term back into a
; proof.  To make this work, in case of axioms we pass to the term
; appropriate formulas: all-formulas for induction, an existential
; formula for existence introduction, and an existential formula
; together with a conclusion for existence elimination.  During
; normalization of the term these formulas are passed along.  When the
; normal form is reached, we have to translate back into a proof.  Then
; these formulas are used to reconstruct the axiom in question, via
; all-formulas-to-ind-aconst , all-formula-to-cases-aconst , 
; ex-formula-to-ex-intro-aconst and ex-formula-and-concl-to-ex-elim-aconst .

(define (make-aconst name kind uninst-formula tpinst . repro-formulas)
  (append (list 'aconst name kind uninst-formula tpinst)
	  repro-formulas))

(define aconst-to-name cadr)
(define aconst-to-kind caddr)
(define aconst-to-uninst-formula cadddr)
(define (aconst-to-tpinst x) (car (cddddr x)))
(define (aconst-to-repro-formulas x) (cdr (cddddr x)))

; To construct the formula associated with an aconst, it is useful to
; separate the instantiated formula from the variables to be
; generalized.  The latter can be obtained as free variables in
; inst-formula.

(define (aconst-to-inst-formula aconst)
  (let* ((uninst-formula (aconst-to-uninst-formula aconst))
	 (tpinst (aconst-to-tpinst aconst))
	 (tsubst (list-transform-positive tpinst
		   (lambda (x) (tvar-form? (car x)))))
	 (pinst (list-transform-positive tpinst
		  (lambda (x) (pvar-form? (car x)))))
	 (rename (make-rename tsubst))
	 (prename (make-prename tsubst))
	 (psubst (map (lambda (x) (list (prename (car x)) (cadr x))) pinst)))
    (formula-substitute-aux
     uninst-formula tsubst empty-subst psubst rename prename)))

(define (aconst-to-formula aconst)
  (let* ((inst-formula (aconst-to-inst-formula aconst))
	 (free (formula-to-free inst-formula)))
    (apply mk-allnc (append free (list inst-formula)))))

(define (aconst-form? x) (and (pair? x) (eq? 'aconst (car x))))

(define (aconst? x)
  (and (aconst-form? x)
       (list? x)
       (<= 5 (length x))
       (let ((name (cadr x))
	     (kind (caddr x))
	     (uninst-formula (cadddr x))
	     (tpinst (car (cddddr x))))
	 (and (string? name)
	      (member kind (list 'axiom 'theorem 'global-assumption))
	      (formula? uninst-formula))))) ;need to check tpinst

(define (aconst=? aconst1 aconst2)
  (and (string=? (aconst-to-name aconst1) (aconst-to-name aconst2))
       (eq? (aconst-to-kind aconst1) (aconst-to-kind aconst2))
       (classical-formula=? (aconst-to-formula aconst1)
			    (aconst-to-formula aconst2))))

(define (aconst-without-rules? aconst)
  (let ((name (aconst-to-name aconst))
	(kind (aconst-to-kind aconst)))
    (or
     (eq? 'theorem kind)
     (eq? 'global-assumption kind)
     (and
      (eq? 'axiom kind)
      (not (member name
		   '("Ind" "Cases" "Intro" "Elim" "Ex-Intro" "Ex-Elim")))))))
    
(define (aconst-to-string aconst)
  (let* ((name (aconst-to-name aconst))
	 (repro-formulas (aconst-to-repro-formulas aconst)) ;better repro-data
	 (repro-string
	  (if
	   (string=? "Intro" name)
	   (string-append " " (number-to-string (car repro-formulas))
			  " " (idpredconst-to-string (cadr repro-formulas)))
	   (apply string-append
		  (map (lambda (x) (string-append " " (formula-to-string x)))
		       repro-formulas)))))
    (cond
     ((string=? "Ind" name) (string-append "(Ind" repro-string ")"))
     ((string=? "Cases" name) (string-append "(Cases" repro-string ")"))
     ((string=? "Intro" name) (string-append "(Intro" repro-string ")"))
     ((string=? "Elim" name) (string-append "(Elim" repro-string ")"))
     ((string=? "Ex-Intro" name) (string-append "(Ex-Intro" repro-string ")"))
     ((string=? "Ex-Elim" name) (string-append "(Ex-Elim" repro-string ")"))
     (else name))))

(define (pvar-to-cterm pvar)
  (let* ((arity (pvar-to-arity pvar))
	 (types (arity-to-types arity))
	 (vars (map type-to-new-var types))
	 (varterms (map make-term-in-var-form vars))
	 (formula (apply make-predicate-formula (cons pvar varterms))))
    (apply make-cterm (append vars (list formula)))))

(define truth-aconst (make-aconst "Truth-Axiom" 'axiom truth empty-subst))

(define eq-refl-aconst
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (var (make-var tvar -1 0 name))
	 (varterm (make-term-in-var-form var))
	 (formula-of-eq-refl-aconst
	  (mk-allnc var (make-eq varterm varterm))))
    (make-aconst "Eq-Refl" 'axiom formula-of-eq-refl-aconst empty-subst)))

(define eq-symm-aconst
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (var1 (make-var tvar 1 0 name))
	 (var2 (make-var tvar 2 0 name))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (formula-of-eq-symm-aconst
	  (mk-allnc var1 var2 (mk-imp (make-eq varterm1 varterm2)
				      (make-eq varterm2 varterm1)))))
    (make-aconst "Eq-Symm" 'axiom formula-of-eq-symm-aconst empty-subst)))

(define eq-trans-aconst
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (var1 (make-var tvar 1 0 name))
	 (var2 (make-var tvar 2 0 name))
	 (var3 (make-var tvar 3 0 name))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (varterm3 (make-term-in-var-form var3))
	 (formula-of-eq-trans-aconst
	  (mk-allnc var1 var2 var3 (mk-imp (make-eq varterm1 varterm2)
					   (make-eq varterm2 varterm3)
					   (make-eq varterm1 varterm3)))))
    (make-aconst "Eq-Trans" 'axiom formula-of-eq-trans-aconst empty-subst)))

(define ext-aconst
  (let* ((tvar1 (make-tvar 1 DEFAULT-TVAR-NAME))
	 (tvar2 (make-tvar 2 DEFAULT-TVAR-NAME))
	 (arrow-type (make-arrow tvar1 tvar2))
	 (fname (default-var-name arrow-type))
	 (fvar1 (make-var arrow-type 1 0 fname))
	 (fvar2 (make-var arrow-type 2 0 fname))
	 (name (default-var-name tvar1))
	 (var (make-var tvar1 -1 0 name))
	 (fterm1 (make-term-in-app-form
		  (make-term-in-var-form fvar1)
		  (make-term-in-var-form var)))
	 (fterm2 (make-term-in-app-form
		  (make-term-in-var-form fvar2)
		  (make-term-in-var-form var)))
	 (prem-eq-fla (make-eq fterm1 fterm2))
	 (concl-eq-fla (make-eq (make-term-in-var-form fvar1)
				(make-term-in-var-form fvar2)))
	 (formula-of-ext-aconst
	  (mk-allnc fvar1 fvar2 (mk-imp (mk-allnc var prem-eq-fla)
					concl-eq-fla))))
    (make-aconst "Ext" 'axiom formula-of-ext-aconst empty-subst)))

(define eq-compat-aconst
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (var1 (make-var tvar 1 0 name))
	 (var2 (make-var tvar 2 0 name))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (pvar (make-pvar (make-arity tvar) -1 h-deg-zero n-deg-zero ""))
	 (eq-fla (make-eq varterm1 varterm2))
	 (fla1 (make-predicate-formula pvar varterm1))
	 (fla2 (make-predicate-formula pvar varterm2))
	 (formula-of-eq-compat-aconst
	  (mk-allnc var1 var2 (mk-imp eq-fla fla1 fla2))))
    (make-aconst "Eq-Compat" 'axiom formula-of-eq-compat-aconst empty-subst)))

(define pair-elim-aconst
  (let* ((tvar1 (make-tvar 1 DEFAULT-TVAR-NAME))
	 (tvar2 (make-tvar 2 DEFAULT-TVAR-NAME))
	 (pairtype (make-star tvar1 tvar2))
	 (pairname (default-var-name pairtype))
	 (pairvar (make-var pairtype -1 0 pairname))
	 (pairvarterm (make-term-in-var-form pairvar))
	 (name1 (default-var-name tvar1))
	 (var1 (make-var tvar1 1 0 name1))
	 (name2 (default-var-name tvar2))
	 (var2 (make-var tvar2 2 0 name2))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (pairterm
	  (make-term-in-pair-form varterm1 varterm2))
	 (pvar (make-pvar (make-arity pairtype) -1 h-deg-zero n-deg-zero ""))
	 (fla1 (mk-all var1 var2
		       (make-predicate-formula pvar pairterm)))
	 (fla2 (mk-all pairvar
		       (make-predicate-formula pvar pairvarterm)))
	 (formula-of-pair-elim-aconst (mk-imp fla1 fla2)))
    (make-aconst "Pair-Elim" 'axiom formula-of-pair-elim-aconst empty-subst)))

(define (all-pair-formula-to-pair-elim-aconst all-pair-formula)
  (let* ((var (all-form-to-var all-pair-formula))
	 (kernel (all-form-to-kernel all-pair-formula))
	 (pairtype (var-to-type var))
	 (type1 (star-form-to-left-type pairtype))
	 (type2 (star-form-to-right-type pairtype))
	 (types (list type1 type2))
	 (fixed-tvar1 (make-tvar 1 DEFAULT-TVAR-NAME))
	 (fixed-tvar2 (make-tvar 2 DEFAULT-TVAR-NAME))
	 (fixed-pairtype (make-star fixed-tvar1 fixed-tvar2))
	 (fixed-tvars (list fixed-tvar1 fixed-tvar2))	 
	 (tsubst (make-substitution fixed-tvars types))
	 (cterm (make-cterm var kernel))
	 (fixed-pvar
	  (make-pvar (make-arity fixed-pairtype) -1 h-deg-zero n-deg-zero ""))
	 (pinst (list (list fixed-pvar cterm))))
    (make-aconst (aconst-to-name pair-elim-aconst)
		 (aconst-to-kind pair-elim-aconst)
		 (aconst-to-uninst-formula pair-elim-aconst)
		 (append tsubst pinst))))

(define total-aconst
  (let* ((tvar1 (make-tvar 1 DEFAULT-TVAR-NAME))
	 (tvar2 (make-tvar 2 DEFAULT-TVAR-NAME))
	 (arrow-type (make-arrow tvar1 tvar2))
	 (fname (default-var-name arrow-type))
	 (fvar (make-var arrow-type -1 0 fname))
	 (name (default-var-name tvar1))
	 (var (make-var tvar1 -1 0 name))
	 (varterm (make-term-in-var-form var))
	 (fterm (make-term-in-var-form fvar))
	 (fappterm (make-term-in-app-form fterm varterm))
	 (formula-of-total-aconst
	  (mk-allnc fvar
		    (make-and
		     (mk-imp (make-total fterm)
			     (mk-allnc var (mk-imp (make-total varterm)
						   (make-total fappterm))))
		     (mk-imp (mk-allnc var (mk-imp (make-total varterm)
						   (make-total fappterm)))
			     (make-total fterm))))))
    (make-aconst "Total" 'axiom formula-of-total-aconst empty-subst)))

(define (finalg-to-eq-to-=-1-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var1 (make-var finalg 1 0 name))
	 (var2 (make-var finalg 2 0 name))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (eq-fla (make-eq varterm1 varterm2))
	 (e-fla1 (make-e varterm1))
	 (=-fla (make-= varterm1 varterm2))
	 (formula-of-eq-to-=-1-aconst 
	  (mk-allnc var1 var2 (mk-imp eq-fla e-fla1 =-fla)))
	 (aconst-name (string-append "Eq-to-=-1-" (type-to-string finalg))))
    (make-aconst aconst-name 'axiom formula-of-eq-to-=-1-aconst empty-subst)))

(define (finalg-to-eq-to-=-2-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var1 (make-var finalg 1 0 name))
	 (var2 (make-var finalg 2 0 name))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (eq-fla (make-eq varterm1 varterm2))
	 (e-fla2 (make-e varterm2))
	 (=-fla (make-= varterm1 varterm2))
	 (formula-of-eq-to-=-2-aconst 
	  (mk-allnc var1 var2 (mk-imp eq-fla e-fla2 =-fla)))
	 (aconst-name (string-append "Eq-to-=-2-" (type-to-string finalg))))
    (make-aconst aconst-name 'axiom formula-of-eq-to-=-2-aconst empty-subst)))

(define (finalg-to-=-to-eq-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var1 (make-var finalg 1 0 name))
	 (var2 (make-var finalg 2 0 name))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (=-fla (make-= varterm1 varterm2))
	 (eq-fla (make-eq varterm1 varterm2))
	 (formula-of-=-to-eq-aconst 
	  (mk-allnc var1 var2 (mk-imp =-fla eq-fla)))
	 (aconst-name (string-append "=-to-Eq-" (type-to-string finalg))))
    (make-aconst aconst-name 'axiom formula-of-=-to-eq-aconst empty-subst)))
    
(define (finalg-to-=-to-e-1-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var1 (make-var finalg 1 0 name))
	 (var2 (make-var finalg 2 0 name))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (=-fla (make-= varterm1 varterm2))
	 (e-fla (make-e varterm1))
	 (formula-of-=-to-e-1-aconst 
	  (mk-allnc var1 var2 (mk-imp =-fla e-fla)))
	 (aconst-name (string-append "=-to-E-" (type-to-string finalg))))
    (make-aconst aconst-name 'axiom formula-of-=-to-e-1-aconst empty-subst)))
    
(define (finalg-to-=-to-e-2-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var1 (make-var finalg 1 0 name))
	 (var2 (make-var finalg 2 0 name))
	 (varterm1 (make-term-in-var-form var1))
	 (varterm2 (make-term-in-var-form var2))
	 (=-fla (make-= varterm1 varterm2))
	 (e-fla (make-e varterm2))
	 (formula-of-=-to-e-2-aconst 
	  (mk-allnc var1 var2 (mk-imp =-fla e-fla)))
	 (aconst-name (string-append "=-to-E-" (type-to-string finalg))))
    (make-aconst aconst-name 'axiom formula-of-=-to-e-2-aconst empty-subst)))
    
(define (finalg-to-total-to-e-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var (make-var finalg -1 0 name))
	 (varterm (make-term-in-var-form var))
	 (total-fla (make-total varterm))
	 (e-fla (make-e varterm))
	 (formula-of-total-to-e-aconst 
	  (mk-allnc var (mk-imp total-fla e-fla)))
	 (aconst-name (string-append "Total-to-E-" (type-to-string finalg))))
    (make-aconst aconst-name 'axiom formula-of-total-to-e-aconst empty-subst)))

(define (finalg-to-e-to-total-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var (make-var finalg -1 0 name))
	 (varterm (make-term-in-var-form var))
	 (total-fla (make-total varterm))
	 (e-fla (make-e varterm))
	 (formula-of-e-to-total-aconst 
	  (mk-allnc var (mk-imp e-fla total-fla)))
	 (aconst-name (string-append "E-to-Total-" (type-to-string finalg))))
    (make-aconst aconst-name 'axiom formula-of-e-to-total-aconst empty-subst)))

(define all-allpartial-aconst
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (var (make-var tvar -1 1 name))
	 (varpartial (make-var tvar -1 0 name))
	 (varterm (make-term-in-var-form var))
	 (varpartialterm (make-term-in-var-form varpartial))
	 (pvar (make-pvar (make-arity tvar) -1 h-deg-zero n-deg-zero ""))
	 (all-fla (mk-all var (make-predicate-formula pvar varterm)))
	 (allpartial-fla
	  (mk-all varpartial
		  (mk-imp (make-total varpartialterm)
			  (make-predicate-formula pvar varpartialterm))))
	 (formula-of-all-allpartial-aconst (mk-imp all-fla allpartial-fla)))
    (make-aconst "All-Allpartial"
		 'axiom formula-of-all-allpartial-aconst empty-subst)))

(define allpartial-all-aconst
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (var (make-var tvar -1 1 name))
	 (varpartial (make-var tvar -1 0 name))
	 (varterm (make-term-in-var-form var))
	 (varpartialterm (make-term-in-var-form varpartial))
	 (pvar (make-pvar (make-arity tvar) -1 h-deg-zero n-deg-zero ""))
	 (all-fla (mk-all var (make-predicate-formula pvar varterm)))
	 (allpartial-fla
	  (mk-all varpartial
		  (mk-imp (make-total varpartialterm)
			  (make-predicate-formula pvar varpartialterm))))
	 (formula-of-allpartial-all-aconst (mk-imp allpartial-fla all-fla)))
    (make-aconst "AllPartial-All"
		 'axiom formula-of-allpartial-all-aconst empty-subst)))

(define allnc-allncpartial-aconst
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (var (make-var tvar -1 1 name))
	 (varpartial (make-var tvar -1 0 name))
	 (varterm (make-term-in-var-form var))
	 (varpartialterm (make-term-in-var-form varpartial))
	 (pvar (make-pvar (make-arity tvar) -1 h-deg-zero n-deg-zero ""))
	 (allnc-fla (mk-allnc var (make-predicate-formula pvar varterm)))
	 (allncpartial-fla
	  (mk-allnc varpartial
		    (mk-imp (make-total varpartialterm)
			    (make-predicate-formula pvar varpartialterm))))
	 (formula-of-allnc-allncpartial-aconst
	  (mk-imp allnc-fla allncpartial-fla)))
    (make-aconst "Allnc-Allncpartial"
		 'axiom formula-of-allnc-allncpartial-aconst empty-subst)))

(define allncpartial-allnc-aconst
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (var (make-var tvar -1 1 name))
	 (varpartial (make-var tvar -1 0 name))
	 (varterm (make-term-in-var-form var))
	 (varpartialterm (make-term-in-var-form varpartial))
	 (pvar (make-pvar (make-arity tvar) -1 h-deg-zero n-deg-zero ""))
	 (allnc-fla (mk-allnc var (make-predicate-formula pvar varterm)))
	 (allncpartial-fla
	  (mk-allnc varpartial
		    (mk-imp (make-total varpartialterm)
			    (make-predicate-formula pvar varpartialterm))))
	 (formula-of-allncpartial-allnc-aconst
	  (mk-imp allncpartial-fla allnc-fla)))
    (make-aconst "AllncPartial-Allnc"
		 'axiom formula-of-allncpartial-allnc-aconst empty-subst)))

(define ex-expartial-aconst
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (var (make-var tvar -1 1 name))
	 (varpartial (make-var tvar -1 0 name))
	 (varterm (make-term-in-var-form var))
	 (varpartialterm (make-term-in-var-form varpartial))
	 (pvar (make-pvar (make-arity tvar) -1 h-deg-zero n-deg-zero ""))
	 (ex-fla (mk-ex var (make-predicate-formula pvar varterm)))
	 (expartial-fla
	  (mk-ex varpartial
		 (mk-and (make-total varpartialterm)
			 (make-predicate-formula pvar varpartialterm))))
	 (formula-of-ex-expartial-aconst (mk-imp ex-fla expartial-fla)))
    (make-aconst "Ex-ExPartial"
		 'axiom formula-of-ex-expartial-aconst empty-subst)))

(define expartial-ex-aconst
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (var (make-var tvar -1 1 name))
	 (varpartial (make-var tvar -1 0 name))
	 (varterm (make-term-in-var-form var))
	 (varpartialterm (make-term-in-var-form varpartial))
	 (pvar (make-pvar (make-arity tvar) -1 h-deg-zero n-deg-zero ""))
	 (ex-fla (mk-ex var (make-predicate-formula pvar varterm)))
	 (expartial-fla
	  (mk-ex varpartial
		 (mk-and (make-total varpartialterm)
			 (make-predicate-formula pvar varpartialterm))))
	 (formula-of-expartial-ex-aconst (mk-imp expartial-fla ex-fla)))
    (make-aconst "ExPartial-Ex"
		 'axiom formula-of-expartial-ex-aconst empty-subst)))

(define exnc-exncpartial-aconst
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (var (make-var tvar -1 1 name))
	 (varpartial (make-var tvar -1 0 name))
	 (varterm (make-term-in-var-form var))
	 (varpartialterm (make-term-in-var-form varpartial))
	 (pvar (make-pvar (make-arity tvar) -1 h-deg-zero n-deg-zero ""))
	 (exnc-fla (mk-exnc var (make-predicate-formula pvar varterm)))
	 (exncpartial-fla
	  (mk-exnc varpartial
		   (mk-and (make-total varpartialterm)
			   (make-predicate-formula pvar varpartialterm))))
	 (formula-of-exnc-exncpartial-aconst
	  (mk-imp exnc-fla exncpartial-fla)))
    (make-aconst "Exnc-ExncPartial"
		 'axiom formula-of-exnc-exncpartial-aconst empty-subst)))

(define exncpartial-exnc-aconst
  (let* ((tvar (make-tvar -1 DEFAULT-TVAR-NAME))
	 (name (default-var-name tvar))
	 (var (make-var tvar -1 1 name))
	 (varpartial (make-var tvar -1 0 name))
	 (varterm (make-term-in-var-form var))
	 (varpartialterm (make-term-in-var-form varpartial))
	 (pvar (make-pvar (make-arity tvar) -1 h-deg-zero n-deg-zero ""))
	 (exnc-fla (mk-exnc var (make-predicate-formula pvar varterm)))
	 (exncpartial-fla
	  (mk-exnc varpartial
		   (mk-and (make-total varpartialterm)
			   (make-predicate-formula pvar varpartialterm))))
	 (formula-of-exncpartial-exnc-aconst
	  (mk-imp exncpartial-fla exnc-fla)))
    (make-aconst "ExncPartial-Exnc"
		 'axiom formula-of-exncpartial-exnc-aconst empty-subst)))

(define (finalg-to-all-allpartial-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var (make-var finalg -1 1 name))
	 (varpartial (make-var finalg -1 0 name))
	 (varterm (make-term-in-var-form var))
	 (varpartialterm (make-term-in-var-form varpartial))
	 (pvar (make-pvar (make-arity finalg) -1 h-deg-zero n-deg-zero ""))
	 (all-fla (mk-all var (make-predicate-formula pvar varterm)))
	 (allpartial-fla
	  (mk-all varpartial
		    (mk-imp (make-e varpartialterm)
			    (make-predicate-formula pvar varpartialterm))))
	 (formula-of-all-allpartial-aconst
	  (mk-imp all-fla allpartial-fla))
	 (name (string-append "All-Allpartial-" (type-to-string finalg))))
    (make-aconst name 'axiom formula-of-all-allpartial-aconst empty-subst)))

(define (finalg-to-allnc-allncpartial-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var (make-var finalg -1 1 name))
	 (varpartial (make-var finalg -1 0 name))
	 (varterm (make-term-in-var-form var))
	 (varpartialterm (make-term-in-var-form varpartial))
	 (pvar (make-pvar (make-arity finalg) -1 h-deg-zero n-deg-zero ""))
	 (allnc-fla (mk-allnc var (make-predicate-formula pvar varterm)))
	 (allncpartial-fla
	  (mk-allnc varpartial
		    (mk-imp (make-e varpartialterm)
			    (make-predicate-formula pvar varpartialterm))))
	 (formula-of-allnc-allncpartial-aconst
	  (mk-imp allnc-fla allncpartial-fla))
	 (name (string-append "Allnc-Allncpartial-" (type-to-string finalg))))
    (make-aconst name 'axiom formula-of-allnc-allncpartial-aconst
		 empty-subst)))

(define (finalg-to-expartial-ex-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var (make-var finalg -1 1 name))
	 (varpartial (make-var finalg -1 0 name))
	 (varterm (make-term-in-var-form var))
	 (varpartialterm (make-term-in-var-form varpartial))
	 (pvar (make-pvar (make-arity finalg) -1 h-deg-zero n-deg-zero ""))
	 (ex-fla (mk-ex var (make-predicate-formula pvar varterm)))
	 (expartial-fla
	  (mk-ex varpartial
		 (mk-and (make-e varpartialterm)
			 (make-predicate-formula pvar varpartialterm))))
	 (formula-of-expartial-ex-aconst (mk-imp expartial-fla ex-fla))
	 (name (string-append "ExPartial-Ex-" (type-to-string finalg))))
    (make-aconst name 'axiom formula-of-expartial-ex-aconst empty-subst)))

(define (finalg-to-exncpartial-exnc-aconst finalg)
  (let* ((name (default-var-name finalg))
	 (var (make-var finalg -1 1 name))
	 (varpartial (make-var finalg -1 0 name))
	 (varterm (make-term-in-var-form var))
	 (varpartialterm (make-term-in-var-form varpartial))
	 (pvar (make-pvar (make-arity finalg) -1 h-deg-zero n-deg-zero ""))
	 (exnc-fla (mk-exnc var (make-predicate-formula pvar varterm)))
	 (exncpartial-fla
	  (mk-exnc varpartial
		   (mk-and (make-e varpartialterm)
			   (make-predicate-formula pvar varpartialterm))))
	 (formula-of-exncpartial-exnc-aconst (mk-imp exncpartial-fla exnc-fla))
	 (name (string-append "ExncPartial-Exnc-" (type-to-string finalg))))
    (make-aconst name 'axiom formula-of-exncpartial-exnc-aconst empty-subst)))

; Now for induction

(define (all-formulas-to-uninst-imp-formulas-and-tpinst . all-formulas)
  (if
   (null? all-formulas)
   (list '() empty-subst)
   (let* ((free (apply union (map formula-to-free all-formulas)))
	  (vars (map all-form-to-var all-formulas))
	  (partial-flag (t-deg-zero? (var-to-t-deg (car vars))))
	  (kernels (map (if partial-flag
			    (lambda (x) (imp-form-to-conclusion
					 (all-form-to-kernel x)))
			    all-form-to-kernel)
			all-formulas))
	  (types (map var-to-type vars))
	  (alg-names
	   (map (lambda (type)
		  (if (alg-form? type)
		      (alg-form-to-name type)
		      (myerror
		       "all-formulas-to-uninst-imp-formulas-and-tpinst"
		       "alg expected" type)))
		types))
	  (tparam-lists (map alg-form-to-types types))
	  (all-formula (car all-formulas))
	  (type (car types))
	  (alg-name (car alg-names))
	  (orig-tvars (alg-name-to-tvars alg-name))
	  (tvars (map (lambda (x) (new-tvar)) orig-tvars))
	  (tparams (car tparam-lists))
	  (tsubst (make-substitution tvars tparams))
	  (uninst-types (map (lambda (x) (apply make-alg (cons x tvars)))
			     alg-names)) 
	  (uninst-arities (map (lambda (x) (make-arity x)) uninst-types))
	  (cterms (map (lambda (x y) (make-cterm x y)) vars kernels))
	  (pinst (map (lambda (x y) (list (arity-to-new-general-pvar x) y))
		      uninst-arities cterms))
	  (pvars (map car pinst))
	  (uninst-vars (map (lambda (x y) (type-to-new-var x y))
			    uninst-types vars))
	  (uninst-stotal-prems (map (lambda (x y)
				      (if partial-flag
					  (make-predicate-formula
					   (make-predconst x '() -1 "STotal")
					   (make-term-in-var-form y))
					  #f))
				    uninst-arities uninst-vars))
	  (uninst-all-formulas
	   (map (lambda (x y z)
		  (make-all x (if partial-flag
				  (make-imp y (make-predicate-formula
					       z (make-term-in-var-form x)))
				  (make-predicate-formula
				   z (make-term-in-var-form x)))))
		uninst-vars uninst-stotal-prems pvars))
	  (uninst-kernel-formulas
	   (map (lambda (x y)
                  (make-predicate-formula
                   y (make-term-in-var-form x)))
                uninst-vars pvars))
	  (alg-names-with-uninst-all-formulas
	   (map (lambda (x y) (list x y)) alg-names uninst-all-formulas))
	  (simalg-names (alg-name-to-simalg-names alg-name)))
     (if (not (equal? alg-names (remove-duplicates alg-names)))
	 (myerror "all-formulas-to-uninst-imp-formulas-and-tpinst"
		  "distinct algs expected" alg-names))
     (if (pair? (set-minus alg-names simalg-names))
	 (myerror "all-formulas-to-uninst-imp-formulas-and-tpinst"
		  "too many alg names" (set-minus alg-names simalg-names)))
     (if (< 1 (length (remove-duplicates tparam-lists)))
	 (myerror "all-formulas-to-uninst-imp-formulas-and-tpinst"
		  "lists expected" tparam-lists))
     (let* ((relevant-simalg-names (list-transform-positive simalg-names
				     (lambda (x) (member x alg-names))))
	    (orig-typed-constr-names
	     (apply append (map alg-name-to-typed-constr-names
				relevant-simalg-names)))
	    (renaming-tsubst (make-substitution orig-tvars tvars))
	    (typed-constr-names
	     (map (lambda (x)
		    (list (car x) (type-substitute (cadr x) renaming-tsubst)))
		  orig-typed-constr-names))
	    (uninst-step-formulas
	     (map (lambda (x) (typed-constr-name-to-step-formula
			       x alg-names-with-uninst-all-formulas
			       renaming-tsubst))
		  typed-constr-names))
            (uninst-imp-formulas
             (map (lambda (uninst-var uninst-stotal-prem uninst-kernel-formula)
                    (make-all uninst-var
                              (apply mk-imp
                                     (append (if partial-flag
                                                 (list uninst-stotal-prem)
                                                 '())
                                             uninst-step-formulas
                                             (list uninst-kernel-formula)))))
                  uninst-vars uninst-stotal-prems uninst-kernel-formulas)))
       (list uninst-imp-formulas (append tsubst pinst))))))

(define (all-formulas-to-uninst-imp-formula-and-tpinst . all-formulas)
  (let* ((uninst-imp-formulas-and-tpinst
	  (apply all-formulas-to-uninst-imp-formulas-and-tpinst all-formulas))
	 (uninst-imp-formulas (car uninst-imp-formulas-and-tpinst))
	 (pinst (cadr uninst-imp-formulas-and-tpinst)))
    (list (car uninst-imp-formulas) pinst)))

(define (typed-constr-name-to-step-formula
	 typed-constr-name alg-names-with-all-formulas renaming-tsubst)
  (let* ((constr-name (typed-constr-name-to-name typed-constr-name))
	 (type (typed-constr-name-to-type typed-constr-name))
	 (alg-name (alg-form-to-name (arrow-form-to-final-val-type type)))
	 (all-formula (cadr (assoc alg-name alg-names-with-all-formulas)))
	 (var (all-form-to-var all-formula))
	 (partial-flag (t-deg-zero? (var-to-t-deg var)))
	 (stotal-prem
	  (if partial-flag
	      (imp-form-to-premise (all-form-to-kernel all-formula))
	      #f))
	 (kernel
	  (if partial-flag
	      (imp-form-to-conclusion (all-form-to-kernel all-formula))
	      (all-form-to-kernel all-formula)))
	 (argtypes (arrow-form-to-arg-types type))
	 (orig-tvars (alg-name-to-tvars alg-name))
	 (subst-tvars (map (lambda (type)
			     (type-substitute type renaming-tsubst))
			   orig-tvars))
	 (argvars (if partial-flag
		      (map type-to-new-partial-var argtypes)
		      (map type-to-new-var argtypes)))
	 (constr (const-substitute (constr-name-to-constr constr-name)
				   renaming-tsubst #t))
	 (constr-app-term
	  (apply mk-term-in-app-form
		 (cons (make-term-in-const-form constr)
		       (map make-term-in-var-form argvars))))
	 (concl-of-step (formula-subst kernel var constr-app-term))
	 (non-param-argvars
	  (list-transform-positive argvars
	    (lambda (var)
	      (not (member (arrow-form-to-final-val-type (var-to-type var))
			   subst-tvars)))))

	 (stotal-formulas ;as many as there are non-param-argvars
	  (if
	   partial-flag
	   (map (lambda (argvar)
		  (let* ((argtype (var-to-type argvar))
			 (argargtypes (arrow-form-to-arg-types argtype))
			 (argargvars (map type-to-new-var argargtypes))
			 (argvaltype (arrow-form-to-final-val-type argtype))
			 (app-term (apply mk-term-in-app-form
					  (cons (make-term-in-var-form argvar)
						(map make-term-in-var-form
						     argargvars)))))
		    
		    (make-predicate-formula
		     (make-predconst (make-arity argvaltype) '() -1 "STotal")
		     app-term)))
		non-param-argvars)
	   '()))
	 (pd-formulas
	  (do ((lt argtypes (cdr lt))
	       (lv argvars (cdr lv))
	       (res
		'()
		(let* ((argtype (car lt))
		       (argvar (car lv))
		       (argargtypes (arrow-form-to-arg-types argtype))
		       (argargvars (map type-to-new-var argargtypes))
		       (argvaltype (arrow-form-to-final-val-type argtype))
		       (argvaltype-name (if (alg-form? argvaltype)
					    (alg-form-to-name argvaltype)
					    ""))
		       (info (assoc argvaltype-name
				    alg-names-with-all-formulas)))
		  (if
		   info
		   (let* ((hyp-all-formula (cadr info))
			  (hyp-var (all-form-to-var hyp-all-formula))
			  (hyp-kernel
			   (if partial-flag
			       (imp-form-to-conclusion
				(all-form-to-kernel hyp-all-formula))
			       (all-form-to-kernel hyp-all-formula)))
			  (app-term
			   (apply mk-term-in-app-form
				  (cons (make-term-in-var-form argvar)
					(map make-term-in-var-form
					     argargvars))))
			  (hyp-formula
			   (formula-subst hyp-kernel hyp-var app-term))
			  (pd-formula
			   (apply mk-all
				  (append argargvars (list hyp-formula)))))
		     (cons pd-formula res))
		   res))))
	      ((null? lt) (reverse res)))))
    (apply mk-all
	   (append argvars
		   (list (apply mk-imp
				(append stotal-formulas pd-formulas
					(list concl-of-step))))))))

; We define a procedure that takes all-formulas and returns the
; corresponding induction axiom.

(define (all-formulas-to-ind-aconst . all-formulas)
  (let* ((uninst-imp-formula-and-tpinst
	  (apply all-formulas-to-uninst-imp-formula-and-tpinst all-formulas))
	 (uninst-imp-formula (car uninst-imp-formula-and-tpinst))
	 (tpinst (cadr uninst-imp-formula-and-tpinst)))
    (apply make-aconst (append (list "Ind" 'axiom uninst-imp-formula tpinst)
			       all-formulas))))

; We define a procedure that takes an all-formula and returns the
; corresponding cases axiom.

(define (all-formula-to-cases-aconst all-formula)
  (let* ((uninst-imp-formula-and-tpinst
	  (all-formula-to-uninst-cases-imp-formula-and-tpinst all-formula))
	 (uninst-imp-formula (car uninst-imp-formula-and-tpinst))
	 (tpinst (cadr uninst-imp-formula-and-tpinst)))
    (make-aconst "Cases" 'axiom uninst-imp-formula tpinst all-formula)))

(define (all-formula-to-uninst-cases-imp-formula-and-tpinst all-formula)
  (let* ((free (formula-to-free all-formula))
	 (var (all-form-to-var all-formula))
	 (partial-flag (t-deg-zero? (var-to-t-deg var)))
	 (kernel (if partial-flag
		     (imp-form-to-conclusion
		      (all-form-to-kernel all-formula))
		     (all-form-to-kernel all-formula)))
	 (type (var-to-type var))
	 (alg-name (if (alg-form? type)
		       (alg-form-to-name type)
		       (myerror
			"all-formula-to-uninst-cases-imp-formula-and-tpinst"
			"alg expected" type)))
	 (orig-tvars (alg-name-to-tvars alg-name))
	 (tvars (map (lambda (x) (new-tvar)) orig-tvars))
	 (tparams (alg-form-to-types type))
	 (tsubst (make-substitution tvars tparams))
	 (uninst-type (apply make-alg (cons alg-name tvars)))
	 (uninst-arity (make-arity uninst-type))
	 (cterm (make-cterm var kernel))
	 (pinst
	  (list (list (arity-to-new-general-pvar uninst-arity) cterm)))
	 (pvar (caar pinst))
	 (uninst-var (type-to-new-var uninst-type var))
	 (uninst-stotal-prem (if partial-flag
				 (make-predicate-formula
				  (make-predconst uninst-arity '() -1 "STotal")
				  (make-term-in-var-form uninst-var))
				 #f))
	 (uninst-all-formula
	  (make-all uninst-var
		    (if partial-flag
			(make-imp uninst-stotal-prem
				  (make-predicate-formula
				   pvar (make-term-in-var-form uninst-var)))
			(make-predicate-formula
			 pvar (make-term-in-var-form uninst-var)))))
	 (uninst-kernel-formula
          (make-predicate-formula pvar (make-term-in-var-form uninst-var)))
	 (orig-typed-constr-names (alg-name-to-typed-constr-names alg-name))
	 (renaming-tsubst (make-substitution orig-tvars tvars))
	 (typed-constr-names
	  (map (lambda (x)
		 (list (car x) (type-substitute (cadr x) renaming-tsubst)))
	       orig-typed-constr-names))
	 (uninst-step-formulas
	  (map (lambda (x) (typed-constr-name-to-cases-step-formula
			    x uninst-all-formula renaming-tsubst))
	       typed-constr-names))
	 (uninst-imp-formula
	  (make-all uninst-var
                    (apply mk-imp (append
                                   (if partial-flag
                                       (list uninst-stotal-prem)
                                       '())
                                   uninst-step-formulas
                                   (list uninst-kernel-formula))))))
    (list uninst-imp-formula (append tsubst pinst))))

(define (typed-constr-name-to-cases-step-formula typed-constr-name all-formula
						 renaming-tsubst)
  (let* ((constr-name (typed-constr-name-to-name typed-constr-name))
	 (type (typed-constr-name-to-type typed-constr-name))
	 (alg-name (alg-form-to-name (arrow-form-to-final-val-type type)))
	 (var (all-form-to-var all-formula))
	 (partial-flag (t-deg-zero? (var-to-t-deg var)))
	 (kernel
	  (if partial-flag
	      (imp-form-to-conclusion (all-form-to-kernel all-formula))
	      (all-form-to-kernel all-formula)))
	 (argtypes (arrow-form-to-arg-types type))
	 (orig-tvars (alg-name-to-tvars alg-name))
	 (subst-tvars (map (lambda (type)
			     (type-substitute type renaming-tsubst))
			   orig-tvars))
	 (argvars (if partial-flag
		      (map type-to-new-partial-var argtypes)
		      (map type-to-new-var argtypes)))
	 (constr (const-substitute (constr-name-to-constr constr-name)
				   renaming-tsubst #t))
	 (constr-app-term
	  (apply mk-term-in-app-form
		 (cons (make-term-in-const-form constr)
		       (map make-term-in-var-form argvars))))
	 (concl-of-step (formula-subst kernel var constr-app-term))
	 (non-param-argvars
	  (list-transform-positive argvars
	    (lambda (var)
	      (not (member (arrow-form-to-final-val-type (var-to-type var))
			   subst-tvars)))))
	 (stotal-formulas ;as many as there are non-param-argvars
	  (if
	   partial-flag
	   (map (lambda (argvar)
		  (let* ((argtype (var-to-type argvar))
			 (argargtypes (arrow-form-to-arg-types argtype))
			 (argargvars (map type-to-new-var argargtypes))
			 (argvaltype (arrow-form-to-final-val-type argtype))
			 (app-term (apply mk-term-in-app-form
					  (cons (make-term-in-var-form argvar)
						(map make-term-in-var-form
						     argargvars)))))
		    
		    (make-predicate-formula
		     (make-predconst (make-arity argvaltype) '() -1 "STotal")
		     app-term)))
		non-param-argvars)
	   '())))
    (apply mk-all
	   (append argvars
		   (list (apply mk-imp (append stotal-formulas
					       (list concl-of-step))))))))

; Now the introduction and elimination axioms for the existential quantifier.

; We define a procedure that takes an existential formula and returns the
; corresponding existence introduction axiom:
; ex-intro: all zs,z.A -> ex z A

(define (ex-formula-to-ex-intro-aconst ex-formula)
  (let* ((var (ex-form-to-var ex-formula))
	 (kernel (ex-form-to-kernel ex-formula))
	 (cterm (make-cterm var kernel))
	 (type (var-to-type var))
	 (tvar (new-tvar))
	 (new-var (type-to-new-var tvar var))
	 (arity (make-arity tvar))
	 (pvar (if (nulltype? (cterm-to-formula cterm))
		   (arity-to-new-pvar arity)
		   (arity-to-new-general-pvar arity)))
	 (predicate-formula
	  (make-predicate-formula pvar (make-term-in-var-form new-var)))
	 (imp-formula (make-imp predicate-formula
				(make-ex new-var predicate-formula)))
	 (uninst-ex-intro-formula (make-all new-var imp-formula))
	 (tsubst (make-substitution (list tvar) (list type)))
	 (pinst (list (list pvar cterm))))
    (make-aconst
     "Ex-Intro" 'axiom uninst-ex-intro-formula (append tsubst pinst)
     ex-formula)))

; We define a procedure that takes an existential formula and a
; conclusion, and returns the corresponding existence elimination axiom:
; ex-elim: allnc zs.ex z A -> (all z.A -> B) -> B

(define (ex-formula-and-concl-to-ex-elim-aconst ex-formula concl)
  (let* ((var (ex-form-to-var ex-formula))
	 (kernel (ex-form-to-kernel ex-formula))
	 (cterm1 (make-cterm var kernel))
	 (cterm2 (make-cterm concl))
	 (type (var-to-type var))
	 (tvar (new-tvar))
	 (new-var (type-to-new-var tvar var))
	 (arity1 (make-arity tvar))
	 (pvar1 (if (nulltype? (cterm-to-formula cterm1))
		    (arity-to-new-pvar arity1)
		    (arity-to-new-general-pvar arity1)))
	 (predicate-formula1
	  (make-predicate-formula pvar1 (make-term-in-var-form new-var)))
	 (arity2 (make-arity))
	 (pvar2 (if (nulltype? (cterm-to-formula cterm2))
		    (arity-to-new-pvar arity2)
		    (arity-to-new-general-pvar arity2)))
	 (predicate-formula2 (make-predicate-formula pvar2))
	 (imp-formula
	  (mk-imp
	   (make-ex new-var predicate-formula1)
	   (make-all new-var (make-imp predicate-formula1 predicate-formula2))
	   predicate-formula2))
	 (tsubst (make-substitution (list tvar) (list type)))
	 (pinst (list (list pvar1 cterm1) (list pvar2 cterm2))))
    (make-aconst "Ex-Elim" 'axiom imp-formula (append tsubst pinst)
		 ex-formula concl)))

; Now the introduction and elimination axioms for the exnc quantifier.

; We define a procedure that takes an exnc formula and returns the
; corresponding existence introduction axiom:
; exnc-intro: allnc zs,z.A -> exnc z A

(define (exnc-formula-to-exnc-intro-aconst exnc-formula)
  (let* ((var (exnc-form-to-var exnc-formula))
	 (kernel (exnc-form-to-kernel exnc-formula))
	 (cterm (make-cterm var kernel))
	 (type (var-to-type var))
	 (tvar (new-tvar))
	 (new-var (type-to-new-var tvar var))
	 (arity (make-arity tvar))
	 (pvar (if (nulltype? (cterm-to-formula cterm))
		   (arity-to-new-pvar arity)
		   (arity-to-new-general-pvar arity)))
	 (predicate-formula
	  (make-predicate-formula pvar (make-term-in-var-form new-var)))
	 (imp-formula (make-imp predicate-formula
				(make-exnc new-var predicate-formula)))
	 (uninst-exnc-intro-formula (make-allnc new-var imp-formula))
	 (tsubst (make-substitution (list tvar) (list type)))
	 (pinst (list (list pvar cterm))))
    (make-aconst
     "Exnc-Intro" 'axiom uninst-exnc-intro-formula (append tsubst pinst)
     exnc-formula)))

; We define a procedure that takes an exnc formula and a
; conclusion, and returns the corresponding exnc elimination axiom:
; exnc-elim: allnc zs.exnc z A -> (allnc z.A -> B) -> B

(define (exnc-formula-and-concl-to-exnc-elim-aconst exnc-formula concl)
  (let* ((var (exnc-form-to-var exnc-formula))
	 (kernel (exnc-form-to-kernel exnc-formula))
	 (cterm1 (make-cterm var kernel))
	 (cterm2 (make-cterm concl))
	 (type (var-to-type var))
	 (tvar (new-tvar))
	 (new-var (type-to-new-var tvar var))
	 (arity1 (make-arity tvar))
	 (pvar1 (if (nulltype? (cterm-to-formula cterm1))
		    (arity-to-new-pvar arity1)
		    (arity-to-new-general-pvar arity1)))
	 (predicate-formula1
	  (make-predicate-formula pvar1 (make-term-in-var-form new-var)))
	 (arity2 (make-arity))
	 (pvar2 (if (nulltype? (cterm-to-formula cterm2))
		    (arity-to-new-pvar arity2)
		    (arity-to-new-general-pvar arity2)))
	 (predicate-formula2 (make-predicate-formula pvar2))
	 (imp-formula
	  (mk-imp
	   (make-exnc new-var predicate-formula1)
	   (make-allnc new-var
		       (make-imp predicate-formula1 predicate-formula2))
	   predicate-formula2))
	 (tsubst (make-substitution (list tvar) (list type)))
	 (pinst (list (list pvar1 cterm1) (list pvar2 cterm2))))
    (make-aconst "Exnc-Elim" 'axiom imp-formula (append tsubst pinst)
		 exnc-formula concl)))

; Additional axioms with names "Intro" and "Elim"

; We define a procedure that takes an inductively defined predicate
; constant and a list of comprehension terms, and returns the
; corresponding elimination axiom.  For instance, for the inductively
; defined exid z A we obtain 
; Elim: allnc zs.(allnc z. A -> B) -> exid z A -> B

; We begin with the strengthened elimination axioms.

(define (number-and-idpredconst-to-intro-aconst i idpc)
  (let* ((name (idpredconst-to-name idpc))
	 (types (idpredconst-to-types idpc))
	 (tsubst (idpredconst-name-and-types-to-tsubst name types))
	 (pinst-for-param-pvars (idpredconst-to-pinst idpc))
	 (param-pvars (idpredconst-name-to-param-pvars name))
	 (param-pvar-cterms (map predicate-to-cterm param-pvars))
	 (idpc-names-with-pvars-and-opt-alg-names
	  (idpredconst-name-to-idpc-names-with-pvars-and-opt-alg-names
	   name))
	 (names (map car idpc-names-with-pvars-and-opt-alg-names))
	 (pvars (map cadr idpc-names-with-pvars-and-opt-alg-names))
	 (tvars (idpredconst-name-to-tvars name))
	 (uninst-idpcs (map (lambda (name)
			       (make-idpredconst name tvars param-pvar-cterms))
			     names))
	 (uninst-idpc-cterms (map predicate-to-cterm uninst-idpcs))
	 (psubst-for-pvars (make-substitution-wrt pvar-cterm-equal?
						  pvars uninst-idpc-cterms))
	 (orig-clauses-with-names
	  (idpredconst-name-to-clauses-with-names name))
	 (orig-clauses (map car orig-clauses-with-names))
	 (orig-clause
	  (if (and (integer? i) (< i (length orig-clauses)))
	      (list-ref orig-clauses i)
	      (myerror "number-and-idpredconst-to-intro-aconst" i
		       "should be an index of a clause for" name)))
	 (uninst-clause (formula-substitute orig-clause psubst-for-pvars)))
    (make-aconst "Intro" 'axiom uninst-clause
		 (append tsubst pinst-for-param-pvars) i idpc)))

; Again, now parallel to induction, with repro-formulas.  Reason:
; repro-formulas needed for proof-to-extracted-term-aux , and probably
; also for normalization via terms.

; Now for elimination.  
; imp-formulas is a list of formulas I xs^ -> A[xs^].  uninst-elim-formula
; is Ij xs^ -> K1[Is,Ps] -> .. -> Kk[Is,Ps] -> Pj xs^

; Notice that we can assume that the arg-lists of the premises are
; partial variables new wrt (map idpredconst-to-free idpcs)

(define (imp-formulas-to-uninst-elim-formulas-etc . imp-formulas)
  (if
   (null? imp-formulas)
   (list '() empty-subst empty-subst empty-subst)
   (let* ((prems (map (lambda (x)
			(if (imp-form? x) (imp-form-to-premise x)
			    (myerror
			     "imp-formulas-to-uninst-elim-formulas-etc"
			     "implication expected" x)))
		      imp-formulas))
	  (concls (map imp-form-to-conclusion imp-formulas))
	  (idpcs
	   (map (lambda (prem)
		  (if (and
		       (predicate-form? prem)
		       (idpredconst-form? (predicate-form-to-predicate prem)))
		      (predicate-form-to-predicate prem)
                      (myerror
                       "imp-formulas-to-uninst-elim-formulas-etc"
                       "idpredconst expected" prem)))
		prems))
	  (arg-lists (map predicate-form-to-args prems))
	  (var-lists
	   (map (lambda (args)
		  (map (lambda (arg)
			 (if
			  (and
			   (term-in-var-form? arg)
			   (t-deg-zero?
			    (var-to-t-deg (term-in-var-form-to-var arg))))
			  (term-in-var-form-to-var arg)
			  (myerror
			   "imp-formulas-to-uninst-elim-formulas-etc"
			   "partial variable expected" arg)))
		       args))
		arg-lists))
	  (concl-cterms (map (lambda (vars concl)
			       (apply make-cterm (append vars (list concl))))
			     var-lists concls))
	  (concl-cterm-arities
	   (map (lambda (x) (apply make-arity (map var-to-type x))) var-lists))
	  (idpc-names (map idpredconst-to-name idpcs))
	  (idpc (car idpcs))
	  (idpc-name (car idpc-names))
	  (types (idpredconst-to-types idpc))
	  (param-cterms (idpredconst-to-cterms idpc))
	  (tsubst (idpredconst-name-and-types-to-tsubst idpc-name types))
	  (pinst-for-param-pvars (idpredconst-to-pinst idpc))
	  (idpc-names-with-pvars-and-opt-alg-names
	   (idpredconst-name-to-idpc-names-with-pvars-and-opt-alg-names
	    idpc-name))
	  (simidpc-names (map car idpc-names-with-pvars-and-opt-alg-names))
	  (pvars (map idpredconst-name-to-pvar idpc-names))
	  (subst-pvar-arities
	   (map (lambda (arity)
		  (apply make-arity
			 (map (lambda (x) (type-substitute x tsubst))
			      (arity-to-types arity))))
		(map pvar-to-arity pvars)))
	  (pinst-for-pvars
	   (if (equal? subst-pvar-arities concl-cterm-arities)
	       (make-substitution-wrt pvar-cterm-equal? pvars concl-cterms)
	       (myerror "imp-formulas-to-uninst-elim-formulas-etc"
			"arities do not fit"
			(map arity-to-string subst-pvar-arities)
			(map arity-to-string concl-cterm-arities))))
	  (pvar-formulas
	   (map (lambda (pvar vars)
		  (apply make-predicate-formula
			 (cons pvar (map make-term-in-var-form vars))))
		pvars var-lists))
	  (tvars (idpredconst-name-to-tvars idpc-name))
	  (param-pvars (idpredconst-name-to-param-pvars idpc-name))
	  (param-pvar-cterms (map predicate-to-cterm param-pvars))
	  (uninst-idpcs (map (lambda (name)
			       (make-idpredconst name tvars param-pvar-cterms))
			     idpc-names))
	  (uninst-idpc-formulas
	   (map (lambda (uninst-idpc vars)
		  (apply make-predicate-formula
			 (cons uninst-idpc (map make-term-in-var-form vars))))
		uninst-idpcs var-lists))
	  (relevant-simidpc-names
	   (list-transform-positive simidpc-names
	     (lambda (x) (member x idpc-names))))
	  (orig-clauses-with-opt-constr-names
	   (apply append
		  (map idpredconst-name-to-clauses-with-names
		       relevant-simidpc-names)))
	  (orig-clauses (map car orig-clauses-with-opt-constr-names))
	  (strengthened-clauses
	   (map (lambda (clause) (clause-etc-to-strengthened-clause
				  clause pvars uninst-idpcs))
		orig-clauses))
	  (uninst-elim-formulas
	   (map (lambda (uninst-idpc-formula pvar-formula)
		  (apply mk-imp (cons uninst-idpc-formula
                                      (append strengthened-clauses
                                              (list pvar-formula)))))
		uninst-idpc-formulas pvar-formulas)))
     (if (not (equal? idpc-names (remove-duplicates idpc-names)))
	 (myerror "imp-formulas-to-uninst-elim-formulas-etc"
		  "distinct idpredconsts expected" idpc-names))
     (if (pair? (set-minus idpc-names simidpc-names))
	 (myerror "imp-formulas-to-uninst-elim-formulas-etc"
		  "too many idpredconst names"
		  (set-minus idpc-names simidpc-names)))
     (list uninst-elim-formulas
	   tsubst pinst-for-param-pvars pinst-for-pvars))))

(define (clause-etc-to-strengthened-clause clause pvars uninst-idpcs)
  (let* ((ncvars-and-final-nckernel
	  (allnc-form-to-vars-and-final-kernel clause));added
	 (ncvars (car ncvars-and-final-nckernel));added
	 (nckernel (cadr ncvars-and-final-nckernel));added
	 (vars-and-final-kernel
	  (all-form-to-vars-and-final-kernel nckernel))
	 (vars (car vars-and-final-kernel))
	 (kernel (cadr vars-and-final-kernel))
	 (prems (imp-form-to-premises kernel))
	 (concl (imp-form-to-final-conclusion kernel))
	 (rec-prems
	  (list-transform-positive prems
	    (lambda (prem)
	     (let* ((prem-nckernel (allnc-form-to-final-kernel prem))
		    (prem-kernel (all-form-to-final-kernel prem-nckernel))
		    (prem-concl (imp-form-to-final-conclusion prem-kernel)))
	       (and (predicate-form? prem-concl)
		    (member (predicate-form-to-predicate prem-concl)
			    pvars))))))
	 (psubst (make-substitution-wrt
		  pvar-cterm-equal?
		  pvars (map predicate-to-cterm uninst-idpcs)))
	 (new-prems (append
		     (map (lambda (prem) (formula-substitute prem psubst))
			  prems)
		     rec-prems))
	 (strengthened-clause-kernel
	  (apply mk-imp (append new-prems (list concl)))))
    (apply mk-allnc
	   (append
	    ncvars (list
		    (apply mk-all (append
				   vars (list 
					 strengthened-clause-kernel))))))))

; All clauses should use partial generalized variables.  Hence
; clause-etc-to-strengthened-clause should be changed accordingly, as
; below.  However, this will affect proofs involving inductive
; definitions (Monika Seisenberger's thesis, work of Klaus Thiel,
; Freiric Barral, Stefan Schimanski and Dominik Schlenker), but only
; in a minor way (assume needs to be done with partial variables).
; Postponed.

; (define (clause-etc-to-strengthened-clause clause pvars uninst-idpcs)
;   (let* ((ncvars-and-final-nckernel
; 	  (allnc-form-to-vars-and-final-kernel clause))
; 	 (ncvars (car ncvars-and-final-nckernel))
; 	 (nckernel (cadr ncvars-and-final-nckernel))
; 	 (partial-ncvars (map var-to-new-partial-var ncvars))
; 	 (vars-and-final-kernel
; 	  (all-form-to-vars-and-final-kernel nckernel))
; 	 (vars (car vars-and-final-kernel))
; 	 (kernel (cadr vars-and-final-kernel))
; 	 (partial-vars (map var-to-new-partial-var vars))
; 	 (subst-kernel (formula-substitute
; 			kernel
; 			(map (lambda (x y) (list x y))
; 			     (append ncvars vars)
; 			     (map make-term-in-var-form
; 				  (append partial-ncvars partial-vars)))))
; 	 (prems (imp-form-to-premises subst-kernel))
; 	 (concl (imp-form-to-final-conclusion subst-kernel))
; 	 (rec-prems
; 	  (list-transform-positive prems
; 	    (lambda (prem)
; 	     (let* ((prem-nckernel (allnc-form-to-final-kernel prem))
; 		    (prem-kernel (all-form-to-final-kernel prem-nckernel))
; 		    (prem-concl (imp-form-to-final-conclusion prem-kernel)))
; 	       (and (predicate-form? prem-concl)
; 		    (member (predicate-form-to-predicate prem-concl)
; 			    pvars))))))
; 	 (psubst (make-substitution-wrt
; 		  pvar-cterm-equal?
; 		  pvars (map predicate-to-cterm uninst-idpcs)))
; 	 (new-prems (append
; 		     (map (lambda (prem) (formula-substitute prem psubst))
; 			  prems)
; 		     rec-prems))
; 	 (strengthened-clause-kernel
; 	  (apply mk-imp (append new-prems (list concl)))))
;     (apply mk-allnc
; 	   (append
; 	    partial-ncvars
; 	    (list
; 	     (apply mk-all (append
; 			    partial-vars (list 
; 					  strengthened-clause-kernel))))))))

(define (imp-formulas-to-uninst-elim-formula-etc . imp-formulas)
  (let* ((uninst-elim-formulas-etc
	  (apply imp-formulas-to-uninst-elim-formulas-etc
		 imp-formulas))
	 (uninst-elim-formulas (car uninst-elim-formulas-etc))
	 (rest (cdr uninst-elim-formulas-etc)))
    (cons (car uninst-elim-formulas) rest)))

; We define a procedure that takes imp-formulas and returns the
; corresponding elimination axiom.

(define (imp-formulas-to-elim-aconst . imp-formulas)
  (let* ((uninst-elim-formula-etc
	  (apply imp-formulas-to-uninst-elim-formula-etc
		 imp-formulas))
	 (uninst-elim-formula (car uninst-elim-formula-etc))
	 (tpinst (apply append (cdr uninst-elim-formula-etc))))
    (apply make-aconst (append (list "Elim" 'axiom uninst-elim-formula tpinst)
			       imp-formulas))))

; Theorems

; A theorem is a special assumption constant.  We maintain an
; association list THEOREMS assigning to every name of a theorem the
; assumption constant and its proof.

; Format of THEOREMS 
; ((name aconst proof <extracted-term>) ...)

(define (theorem-name-to-aconst name)
  (let ((info (assoc name THEOREMS)))
    (if info
	(cadr info)
	(myerror "theorem-name-to-aconst" "theorem name expected" name))))

(define (theorem-name-to-proof name)
  (let ((info (assoc name THEOREMS)))
    (if info
	(caddr info)
	(myerror "theorem-name-to-proof" "theorem name expected" name))))

(define (theorem-aconst-to-inst-proof aconst)
  (let* ((name (aconst-to-name aconst))
	 (kind (aconst-to-kind aconst))
	 (proof-of-thm
	  (if (eq? 'theorem kind)
	      (theorem-name-to-proof name)
	      (myerror "theorem-aconst-to-inst-proof" "kind theorem expected"
		       kind (aconst-to-formula aconst))))
	 (tpinst (aconst-to-tpinst aconst))
	 (tsubst (list-transform-positive tpinst
		   (lambda (x) (tvar-form? (car x)))))
	 (pinst (list-transform-positive tpinst
		  (lambda (x) (pvar-form? (car x)))))
	 (rename (make-rename tsubst))
	 (prename (make-prename tsubst))
	 (psubst (map (lambda (x) (list (prename (car x)) (cadr x))) pinst))
	 (arename (make-arename tsubst psubst rename prename)))
    (proof-substitute-aux
     proof-of-thm tsubst empty-subst psubst empty-subst
     rename prename arename)))

(define (theorem-or-global-assumption-name-to-pconst-name string)
  (string-append "c"
		 (list->string (remove-numerals (string->list string)))))

(define (remove-numerals charlist)
  (if (null? charlist)
      '()
      (append (let ((char (car charlist)))
		(cond ((char=? char #\-) (list #\X #\x))
		      ((char=? char #\() (list #\Y #\y)) 
		      ((char=? char #\)) (list #\y #\Y)) 
		      ((char=? char #\1) (list #\O #\n #\e)) 
		      ((char=? char #\2) (list #\T #\w #\o)) 
		      ((char=? char #\3) (list #\T #\h #\r #\e #\e)) 
		      ((char=? char #\4) (list #\F #\o #\u #\r))
		      ((char=? char #\5) (list #\F #\i #\v #\e))
		      ((char=? char #\6) (list #\S #\i #\x))
		      ((char=? char #\7) (list #\S #\e #\v #\e #\n))
		      ((char=? char #\8) (list #\E #\i #\g #\h #\t))
		      ((char=? char #\9) (list #\N #\i #\n #\e))
		      ((char=? char #\0) (list #\Z #\e #\r #\o))
		      (else (list char))))
	      (remove-numerals (cdr charlist)))))

(define (theorem-or-global-assumption-to-pconst thm-or-ga)
  (let* ((thm-or-ga-name (aconst-to-name thm-or-ga))
	 (pconst-name
	  (theorem-or-global-assumption-name-to-pconst-name thm-or-ga-name))
	 (pconst (pconst-name-to-pconst pconst-name))
	 (tpinst (aconst-to-tpinst thm-or-ga))
	 (tsubst (list-transform-positive tpinst
		   (lambda (x) (tvar-form? (car x)))))
	 (pinst (list-transform-positive tpinst
		  (lambda (x) (pvar-form? (car x)))))
	 (new-tsubst
	  (do ((l pinst (cdr l))
	       (res '() (let* ((pvar (caar l))
			       (cterm (cadar l))
			       (cterm-type (formula-to-et-type
					    (cterm-to-formula cterm))))
			  (if (nulltype? cterm-type)
			      res
			      (cons (list (PVAR-TO-TVAR pvar) cterm-type)
				    res)))))
	      ((null? l) (reverse res)))))
    (const-substitute pconst (compose-t-substitutions tsubst new-tsubst) #f)))

(define (add-theorem string . opt-proof)
  (if (and (null? opt-proof)
	   (null? PPROOF-STATE))
      (myerror
       "add-theorem" "proof argument or proof under construction expected"))
  (let ((proof (if (null? opt-proof)
		   (pproof-state-to-proof)
		   (car opt-proof))))
    (if (not (null? (proof-to-free-avars proof)))
	(apply myerror
	       (cons "unexpected free assumptions"
		     (proof-to-free-avars proof))))
    (if
     (is-used? string '() 'theorem)
     *the-non-printing-object*
     (let ((formula (unfold-formula (proof-to-formula proof)))
	   (test (nc-violations proof)))
       (if (pair? test)
	   (apply myerror (cons "allnc-intro with cvars" test)))
       (let ((aconst (make-aconst string 'theorem formula empty-subst)))
	 (set! THEOREMS (cons (list string aconst proof) THEOREMS))
	 (if (not (member string (list "Id")))
	     (comment "ok, " string " has been added as a new theorem."))
	 (if (not (formula-of-nulltype? formula))
	     (let ((pconst-name
		    (theorem-or-global-assumption-name-to-pconst-name string))
		   (type (formula-to-et-type formula)))
	       (add-program-constant pconst-name type 1 'const 0))))))))

(define save add-theorem)

(define (nc-violations proof)
  (if (formula-of-nulltype? (proof-to-formula proof))
      '()
      (nc-violations-aux proof)))

; In nc-violations-aux we can assume that the proved formula has
; computational content.

(define (nc-violations-aux proof)
  (case (tag proof)
    ((proof-in-avar-form proof-in-aconst-form) '())
    ((proof-in-imp-intro-form)
     (nc-violations-aux (proof-in-imp-intro-form-to-kernel proof)))
    ((proof-in-imp-elim-form)
     (let* ((op (proof-in-imp-elim-form-to-op proof))
	    (arg (proof-in-imp-elim-form-to-arg proof))
	    (prevop (nc-violations-aux op))
	    (prevarg (nc-violations arg)))
       (union prevop prevarg)))
    ((proof-in-and-intro-form)
     (let* ((left (proof-in-and-intro-form-to-left proof))
	    (right (proof-in-and-intro-form-to-right proof)))
       (if (formula-of-nulltype? (proof-to-formula left))
	   (nc-violations-aux right)
	   (union (nc-violations-aux left)
		  (nc-violations right)))))
    ((proof-in-and-elim-left-form)
     (nc-violations-aux
      (proof-in-and-elim-left-form-to-kernel proof)))
    ((proof-in-and-elim-right-form)
     (nc-violations-aux
      (proof-in-and-elim-right-form-to-kernel proof)))
    ((proof-in-all-intro-form)
     (nc-violations-aux
      (proof-in-all-intro-form-to-kernel proof)))
    ((proof-in-all-elim-form)
     (nc-violations-aux (proof-in-all-elim-form-to-op proof)))
    ((proof-in-allnc-intro-form)
     (let* ((var (proof-in-allnc-intro-form-to-var proof))
	    (kernel (proof-in-allnc-intro-form-to-kernel proof))
	    (prev (nc-violations-aux kernel)))
       (if (member var (proof-to-cvars proof))
	   (adjoin var prev)
	   prev)))
    ((proof-in-allnc-elim-form)
     (nc-violations-aux
      (proof-in-allnc-elim-form-to-op proof)))
    (else (myerror "nc-violations-aux" "proof expected" proof))))

(define (remove-theorem . strings)
  (define (rthm1 thm-name)
    (let ((info (assoc thm-name THEOREMS)))
      (if info
	  (let* ((proof (theorem-name-to-proof thm-name))
		 (formula (unfold-formula (proof-to-formula proof))))
	    (do ((l THEOREMS (cdr l))
		 (res '() (if (string=? thm-name (caar l))
			      res
			      (cons (car l) res))))
		((null? l) (set! THEOREMS (reverse res))))
	    (comment "ok, theorem " thm-name " is removed")
	    (if (not (formula-of-nulltype? formula))
		(remove-program-constant
		 (theorem-or-global-assumption-name-to-pconst-name
		  thm-name))))
	  (myerror "remove-theorem" "theorem expected"
		   thm-name))))
  (for-each rthm1 strings))

(define (display-theorems . x)
  (if
   COMMENT-FLAG
   (let ((reduced-thms (if (null? x)
			   THEOREMS
			   (do ((l THEOREMS (cdr l))
				(res '() (if (member (caar l) x)
					     (cons (car l) res)
					     res)))
			       ((null? l) res)))))
     (for-each (lambda (thm)
		 (display (car thm))
		 (display tab)
		 (display-formula
		  (fold-formula (aconst-to-formula (cadr thm))))
		 (newline))
	       reduced-thms))))

; Global assumptions

; A global assumption is a special assumption constant.  It provides a
; proposition whose proof does not concern us presently.  We maintain an
; association list GLOBAL-ASSUMPTIONS assigning to every name of a
; global assumption the assumption constant.

; Format of GLOBAL-ASSUMPTIONS 
; ((name aconst) ...)

(define (global-assumption-name-to-aconst name)
  (let* ((info (assoc name GLOBAL-ASSUMPTIONS)))
    (if info
	(cadr info)
	(myerror "global-assumption-name-to-aconst"
		 "global assumption name expected" name))))

(define (add-global-assumption string formula . optional-arity)
  (if (pair? (formula-to-free formula))
      (apply myerror
	     (append (list "add-global-assumption" "unexpected free variables")
		     (formula-to-free formula))))
  (let* ((fla (unfold-formula formula))
	 (aconst (make-aconst string 'global-assumption fla empty-subst)))
    (if
     (is-used? string formula 'global-assumption)
     *the-non-printing-object*
     (begin
       (set! GLOBAL-ASSUMPTIONS (cons (list string aconst) GLOBAL-ASSUMPTIONS))
       (if
	(not (member string (list "Efq-Log" "Stab-Log" "Efq" "Stab")))
	(comment "ok, " string " has been added as a new global assumption."))
       (if (not (formula-of-nulltype? fla))
	   (let* ((pconst-name
		   (theorem-or-global-assumption-name-to-pconst-name string))
		  (type (formula-to-et-type fla))
		  (arity (if (pair? optional-arity)
			     (car optional-arity)
			     0)))
	     (add-program-constant pconst-name type 1 'const arity)))))))

(define aga add-global-assumption)

(define (remove-global-assumption . strings)
  (define (rga1 ga-name)
    (let ((info (assoc ga-name GLOBAL-ASSUMPTIONS)))
      (if info
	  (let* ((aconst (global-assumption-name-to-aconst ga-name))
		 (formula (aconst-to-uninst-formula aconst)))
	    (do ((l GLOBAL-ASSUMPTIONS (cdr l))
		 (res '() (if (string=? ga-name (caar l))
			      res
			      (cons (car l) res))))
		((null? l) (set! GLOBAL-ASSUMPTIONS (reverse res))))
	    (comment "ok, global assumption " ga-name " is removed")
	    (if (not (formula-of-nulltype? formula))
		(remove-program-constant
		 (theorem-or-global-assumption-name-to-pconst-name
		  ga-name))))
	  (myerror "remove-global-assumption" "global assumption expected"
		   ga-name))))
  (for-each rga1 strings))

(define rga remove-global-assumption)

(define (display-global-assumptions . x)
  (if
   COMMENT-FLAG
   (let ((reduced-gas (if (null? x)
			  GLOBAL-ASSUMPTIONS
			  (do ((l GLOBAL-ASSUMPTIONS (cdr l))
			       (res '() (if (member (caar l) x)
					    (cons (car l) res)
					    res)))
			      ((null? l) res)))))
     (for-each (lambda (ga)
		 (display (car ga))
		 (display tab)
		 (display-formula (fold-formula (aconst-to-formula (cadr ga))))
		 (newline))
	       reduced-gas))))

; Added 2004-01-08:

(define (new-global-assumption-name string)
  (new-global-assumption-name-aux string 0))

(define (new-global-assumption-name-aux string n)
  (if (assoc (string-append string (number-to-string n))
	     GLOBAL-ASSUMPTIONS)
      (new-global-assumption-name-aux string (+ n 1))
      (string-append string (number-to-string n))))

; Added 2007-03-09:

; (search-about string) searches in THEOREMS and GLOBAL-ASSUMPTIONS
; for all items whose name contains string.

(define (search-about string)
  (let ((thms (list-transform-positive THEOREMS
		(lambda (x) (substring? string (car x)))))
	(gas (list-transform-positive GLOBAL-ASSUMPTIONS
		(lambda (x) (substring? string (car x))))))
    (if (null? thms)
	(comment "No theorems with name containing " string)
	(begin
	  (comment "Theorems with name containing " string)
	  (for-each (lambda (x)
		      (comment (car x))
		      (display-comment
		       (pretty-print-string
			(string-length COMMENT-STRING)
			(- pp-width (string-length COMMENT-STRING))
			(aconst-to-formula (cadr x))))
		      (newline))
		    thms)))
    (if (null? gas)
	(comment "No global assumptions with name containing " string)
	(begin
	  (comment "Global assumptions with name containing " string)
	  (for-each (lambda (x)
		      (comment (car x))
		      (display-comment
		       (pretty-print-string
			(string-length COMMENT-STRING)
			(- pp-width (string-length COMMENT-STRING))
			(aconst-to-formula (cadr x))))
		      (newline))
		    gas)))))

(define (initial-substring? string1 string2)
  (let ((l1 (string-length string1))
	(l2 (string-length string2)))
    (and (<= l1 l2)
	 (string=? string1 (substring string2 0 l1)))))

; (initial-substring? "abc" "abcde")

(define (substring? string1 string2)
  (do ((s string2 (substring s 1 (string-length s)))
       (res #f (initial-substring? string1 s)))
      ((or res (zero? (string-length s)))
       res)))

; (substring? "bcd" "abcde")

