/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include "proto.h"

extern ForwardSection *forward_section;
extern PrefetchSection *prefetch_section;
extern CacheSection *cache_section;
extern GLOBAL* global;

PrefetchSection::PrefetchSection():
     Section ("prefetch", MUTEX),
     enabled   (field_vec[0].int_value),
     threads   (field_vec[1].int_value),
     queuesize (field_vec[2].int_value)
{
	queue = NULL;
	entries = 0;
	pthread_cond_init(&cond, NULL);
}

void PrefetchSection::update()
{
        prefetch_list.clear();

        ItemList::iterator item;
        for (item = sub_vec[0].item_list.begin(); item != sub_vec[0].item_list.end(); item++)
                prefetch_list.push_back(Prefetch(*item));
}

void PrefetchSection::init_threads() {
	int threads;

	mutex_lock();
	threads = this->threads;
	unlock();

	for (; threads > 0; threads--)
		thread_create((void*)prefetch_thread, prefetch_section);
}

Prefetch::Prefetch(const Item& item):
	enabled   (item.field_vec[0].int_value),
	comment   (item.field_vec[1].string_value),
	profiles  (item.field_vec[2].string_list_value),
	tag       (item.field_vec[3].string_value),
	attribute (item.field_vec[4].string_value),
	pattern   (item.field_vec[5].string_value),
	maxsize   (item.field_vec[6].uint_value),
	recursion (item.field_vec[7].int_value)
{
	pe = (pattern != "") ? reg_compile(pattern.c_str(), REGFLAGS) : NULL;
}

Prefetch::Prefetch(const Prefetch &prefetch):
	enabled   (prefetch.enabled),
	comment   (prefetch.comment),
	profiles  (prefetch.profiles),
	tag       (prefetch.tag),
	attribute (prefetch.attribute),
	pattern   (prefetch.pattern),
	maxsize   (prefetch.maxsize),
	recursion (prefetch.recursion)
{
	pe = (pattern != "") ? reg_compile(pattern.c_str(), REGFLAGS) : NULL;
}	

Prefetch::~Prefetch() 
{
	if (pe != NULL)
		reg_free(pe);
}

void prefetch_thread(PrefetchSection *prefetch) 
{
	struct PREFETCH_QUEUE *queue;

	while(1) {
		prefetch->read_lock();
		if (prefetch->queue == NULL) prefetch->cond_wait(&prefetch->cond);

		if (prefetch->queue != NULL) {
			queue = prefetch->queue;
			prefetch->queue = queue->next;
			prefetch->entries--;

			prefetch->unlock();

			protocol_prefetch(queue->url, queue->header, queue->maxsize, queue->depth);

			prefetch_queue_free(queue);
		} else {
			/* this can happen if another thread steals the entry in the queue without
			   calling pthread_cond_wait */
			prefetch->unlock();
		}
	}
}

int protocol_prefetch(URL *url, HttpHeaderList *header, unsigned int maxsize, int depth)
{
	int x;
	char buf[1024];
	CONNECTION *connection;

	snprintf(buf, sizeof(buf), "%s://%s:%d%s", url->proto, url->host, url->port, url->file);
	if (cache_section->exists(buf) == TRUE) return FALSE;

	/* there's a bit of a race condition here... another thread can create the cache file
	   between this point and when we open it for writing... worst case scenario is we end
	   up downloading some of the file twice */

	connection = connection_new();

	connection->flags = CONNECTION_PREFETCH | CONNECTION_NOCLIENT;
	connection->depth = depth;
	connection->ip = xstrdup("127.0.0.1");
	connection->interface = xstrdup("127.0.0.1");
	connection->transferlimit = maxsize;

	connection->header = header_new();
	connection->header->type = HTTP_PROXY;
	connection->header->version = HTTP_HTTP11;
	connection->header->method = xstrdup("GET");
	connection->header->proto = xstrdup(url->proto);
	connection->header->host = xstrdup(url->host);
	connection->header->file = xstrdup(url->file);
	connection->header->port = url->port;

	snprintf(buf, sizeof(buf), "%s://%s:%d%s", connection->header->proto, connection->header->host, connection->header->port, connection->header->file);
	connection->header->url = xstrdup(buf);
	connection->header->referer = xstrdup(buf);

#ifdef HAVE_ZLIB
	connection->header->accept_encoding = xstrdup(ACCEPTED_ENCODINGS);
#endif /* HAVE_ZLIB */

	if (header != NULL)
		connection->header->header = *header;
	else {
		struct HTTP_HEADER hdr;
		hdr.type = string("User-Agent");
		hdr.value = string(PREFETCH_USERAGENT);
		hdr.send = TRUE;
		connection->header->header.push_back(hdr);
	}

	if (url->username != NULL) connection->header->username = xstrdup(url->username);
	if (url->password != NULL) connection->header->password = xstrdup(url->password);

	forward_section->forward_do(connection);
	
	x = protocol_start(connection);

	if (x >= 0) {
		if (!strcasecmp(url->proto, "http"))
			protocol_http(connection);
		else if (!strcasecmp(url->proto, "ftp"))
			protocol_ftp(connection);
	}

	if (connection->cachemap != NULL)
		cache_section->cache_close(connection->cachemap);

	if (connection->htmlstream != NULL)
		htmlstream_free(connection->htmlstream);

	if (connection->server != NULL) {
		if (connection->keepalive_server == TRUE)
			pool_add(global->pool, connection->server, connection->header->proto, (connection->proxy_type == PROXY_NORMAL && connection->proxy_host != NULL) ? connection->proxy_host : connection->header->host, (connection->proxy_type == PROXY_NORMAL) ? connection->proxy_port : connection->header->port, NULL, NULL);
		else
			xdelete connection->server;
	}

	http_header_free(connection->header);
	connection_free(connection);

	return TRUE;
}

void PrefetchSection::queue_add(URL *url, HttpHeaderList *header, unsigned int maxsize, int depth)
{
	char *ptr;
	struct PREFETCH_QUEUE *queue, *tmp = NULL;

	mutex_lock();

	if (this->entries >= this->queuesize)
		goto error;

	/* this doesn't need to be done now, code elsewhere prevents it from being downloaded again.. but this stops
	   the queue from being filled up with files already cached. */
	ptr = url_create(url);
	if (cache_section->exists(ptr)) {
		xfree(ptr);

		goto error;
	}
	xfree(ptr);

	if (this->queue != NULL) {
		for (tmp = this->queue; tmp->next && !urlcmp(tmp->url, url); tmp = tmp->next);
		if (urlcmp(tmp->url, url))
			/* already in queue. */ 
			goto error;
	}

	this->entries++;

	queue = (PREFETCH_QUEUE*)xmalloc(sizeof(struct PREFETCH_QUEUE));
	queue->url = url;
	if (header != NULL)
		queue->header = new HttpHeaderList(*header);
	else
		queue->header = NULL;

	queue->maxsize = maxsize;
	queue->depth = depth;
	queue->next = NULL;
	

	if (tmp != NULL)
		tmp->next = queue;
	else
		this->queue = queue;

	pthread_cond_signal(&cond);
	unlock();

	return;
error:
	url_free(url);

	unlock();
}

void prefetch_queue_free(struct PREFETCH_QUEUE *queue) {
	url_free(queue->url);
	xdelete queue->header;
	xfree(queue);
}

void PrefetchSection::setup_callbacks(CONNECTION *connection) {
	PrefetchList::const_iterator prefetch;

	mutex_lock();
	if (this->enabled == FALSE) goto out;

	if (connection->rheader->content_type == NULL || strncasecmp(connection->rheader->content_type, "text/html", 9))
		goto out;
	else if (connection->flags & CONNECTION_SSLCLIENT)
		goto out;

	connection->htmlstream = htmlstream_new();


	for (prefetch = prefetch_list.begin(); prefetch != prefetch_list.end(); prefetch++) {
		if (prefetch->tag != "" && prefetch->attribute != "")
			htmlstream_callback_add(connection->htmlstream, prefetch->tag.c_str(), (void*)prefetch_callback, connection);
	}
		
out:
	unlock();
}

void prefetch_callback(HTMLSTREAM *hs, struct htmlstream_node *node, CONNECTION *connection) {
	URL *url = NULL;
	int ret;
	unsigned int maxsize = 0;
	char *ptr;
	struct htmlstream_tag_property *property;
	PrefetchList::const_iterator prefetch;

	prefetch_section->mutex_lock();

	for (prefetch = prefetch_section->prefetch_list.begin(); prefetch != prefetch_section->prefetch_list.end(); prefetch++) {
		if (prefetch->enabled == FALSE)
			continue;

		if (!profile_find(connection->profiles, prefetch->profiles))
			continue;

		if (prefetch->tag != "" && prefetch->attribute != "") {
			if (!strcasecmp(prefetch->tag.c_str(), node->tag->name)) {
				if (prefetch->recursion != 0 && connection->depth >= prefetch->recursion) goto endloop;

				for (property = node->tag->properties; property; property = property->next) {
					if (!strcasecmp(property->name, prefetch->attribute.c_str())) {
						if (strchr(property->value, ':') != NULL)
							url = url_parse(property->value);
						else {
							/* relative URL */
							url = (URL*)xmalloc(sizeof(URL));
							url->username = url->password = NULL;
							url->proto = xstrdup(connection->header->proto);
							url->host = xstrdup(connection->header->host);
							url->port = connection->header->port;
							if (property->value[0] == '/')
								ptr = xstrdup(property->value);
							else {
								ptr = xstrndup(connection->header->file, strrchr(connection->header->file, '/') - connection->header->file + 1);
								ptr = string_append(ptr, property->value);
							}
							url->file = url_path_fix(ptr);
							xfree(ptr);
						}

						if (prefetch->pe != NULL && url != NULL) {
							/* normalize URL for matching */
							ptr = url_create(url);
							ret = reg_exec(prefetch->pe, ptr);
							xfree(ptr);

							if (ret) {
								url_free(url);
								url = NULL;
	
								/* this entry didn't match, but others may. */
								continue;
							}
						}

						maxsize = prefetch->maxsize;

						/* don't allow more than one to match */
						break;
					}
				}
			}
		}

	endloop:
		if (url != NULL) break;
	}

	prefetch_section->unlock();

	if (url != NULL) {
		if (maxsize == 0) maxsize = ~0;
		prefetch_section->queue_add(url, NULL, maxsize, connection->depth + 1);
	}
}
