/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin
    Copyright (C) 2003  Riadh Elloumi

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <stdlib.h>
#include "proto.h"

/*
 * For memory allocation debugging: we log all allocated pointers to marray[0][*]
 * before we start the main loop, and to marray[1][*] after we start main loop (this
 * is controlled by mstart).
 * If you interrupt the program unnder gdb and send signal SIGHUP, the allocated memory
 * in logs, cache and dns will be freed. Then type "print marray[1]", you will find the
 * pointers you didn't free.
 */

#ifdef _DEBUG_MEMORY
extern int mstart;
extern MINFO marray[2][MMAX];
extern pthread_rwlock_t mlock;


void marray_add(void* ptr, const char* file, int line)
{
	int i = 0;

	while (i < MMAX) {
		if (marray[mstart][i].address == 0)
			break;
		i++;
	}
	
	if (i == MMAX) {
		printf("Malloc debugger: too many allocated pointers");
		abort();
	}
	
	marray[mstart][i].address = ptr;
	marray[mstart][i].file = strdup(file);
	marray[mstart][i].line = line;
	marray[mstart][i].function = 0;
}

void marray_delete(void* ptr, const char* file, int line)
{
	int i = 0, j;

	for (j = 0; j < 2; j++)
		for (i = 0; i < MMAX; i++) {
			if (marray[j][i].address == ptr)
				goto add_adress_found;
		}
		
	/* adress was not found */
	printf("Malloc debugger: free a wrong address");
	abort();
	return;
  

 add_adress_found:

	marray[j][i].address = 0;
	free(marray[j][i].file);
	marray[j][i].file = 0;
	marray[j][i].line = 0;
	marray[j][i].function = 0;
}

void marray_replace(void* ptr, void* newptr, char* file, int line)
{
	int i = 0, j;

	for (j = 0; j < 2; j++)
		for (i = 0; i < MMAX; i++) {
			if (marray[j][i].address == ptr)
				goto replace_adress_found;
		}
			
	/* adress was not found */
	printf("Malloc debugger: realloc with a wrong address!");
	abort();

 replace_adress_found:
				
	free(marray[j][i].file);
			
	marray[j][i].address = newptr;
	marray[j][i].file = strdup(file);
	marray[j][i].line = line;
	marray[j][i].function = 0;
}

/* New operator for library allocations */
void * operator new (size_t size)
{
	return xmalloc_(size, "library allocation", 0);  
}

void *xmalloc_(int size, const char* file, int line)
{
	void *ret;

	ASSERT(size > 0);

	ret = malloc(size);

	ASSERT(ret);

	pthread_rwlock_wrlock(&mlock);
		
	marray_add(ret, file, line);

	pthread_rwlock_unlock(&mlock);		
	
	return ret;
}

void xfree_(void *ptr, const char* file, int line)
{
	ASSERT(ptr);

	pthread_rwlock_wrlock(&mlock);
	
	marray_delete(ptr, file, line);
	
	pthread_rwlock_unlock(&mlock);		

	free(ptr);

}

void *xrealloc_(void *ptr, int newsize, char* file, int line)
{
	void *newptr;

	ASSERT(newsize > 0);

	newptr = realloc(ptr, newsize);

	ASSERT(newptr);

	pthread_rwlock_wrlock(&mlock);
	
	if (ptr == NULL) {
		/* equivalent to malloc */
		marray_add(newptr, file, line);
	} else {			
		marray_replace(ptr, newptr, file, line);
	}
	pthread_rwlock_unlock(&mlock);		

	return newptr;
}

/*
strdup which uses xmalloc
*/
char *xstrdup_(const char *s, char* file, int line)
{
	if (s == NULL) return NULL;

	char *dest = (char*)xmalloc_(strlen(s) + 1, file, line);
	return strcpy(dest, s);
}

/*
strndup which uses xmalloc
*/
char *xstrndup_(const char *s, size_t len, char* file, int line)
{
	size_t x = s_strnlen(s, len);

	return s_strncpy((char*)xmalloc_(x + 1, file, line), s, x + 1);
}

pcre *pcre_compile_(const char *pattern, int options,
		    const char **errptr, int *erroffset,
		    const unsigned char *tableptr, const char* file, int line)
{
	pcre* ret = pcre_compile(pattern, options, errptr, erroffset, tableptr);

	pthread_rwlock_wrlock(&mlock);
		
	marray_add(ret, file, line);

	pthread_rwlock_unlock(&mlock);		
	
	return ret;

}

pcre_extra *pcre_study_(const pcre *code, int options,
			const char **errptr, const char* file, int line)
{
	pcre_extra *ret = pcre_study(code, options, errptr);
	
	pthread_rwlock_wrlock(&mlock);
		
	marray_add(ret, file, line);

	pthread_rwlock_unlock(&mlock);		
	
	return ret;
}

#else

/*
basic checking for malloc/realloc/free, better to die gracefully than run into problems
later on.
*/

void *xmalloc(int size)
{
	void *ret;

	ASSERT(size > 0);

	ret = malloc(size);

	ASSERT(ret);

	return ret;
}

void xfree(void *ptr)
{
	ASSERT(ptr);

	free(ptr);
}

void *xrealloc(void *ptr, int newsize)
{
	void *newptr;

	ASSERT(newsize > 0);

	newptr = realloc(ptr, newsize);

	ASSERT(newptr);

	return newptr;
}
/*
strdup which uses xmalloc
*/
char *xstrdup(const char *s)
{
	if (s == NULL) return NULL;

	char *dest = (char*)xmalloc(strlen(s) + 1);
	return strcpy(dest, s);
}

/*
strndup which uses xmalloc
*/
char *xstrndup(const char *s, size_t len)
{
	size_t x = s_strnlen(s, len);

	return s_strncpy((char*)xmalloc(x + 1), s, x + 1);
}

#endif // _DEBUG_MEMORY

/* 
 * Function called when new doesn't find enough memory
 */
void out_of_memory()
{
      putlog(MMLOG_ERROR, "Out of memory: %s:%d in function %s", __FILE__, __LINE__, __FUNCTION__); 
      abort();
}
