/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin
    Copyright (C) 2003  Riadh Elloumi

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <errno.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/tcp.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "proto.h"

extern CacheSection *cache_section;
extern GeneralSection *general_section;
extern GLOBAL *global;
extern char proxyhost[];

ForwardSection::ForwardSection():
     Section("forward", RWLOCK),
     enabled  (field_vec[0].int_value),
     carp     (field_vec[1].int_value)
{
}

void ForwardSection::update()
{
	forward_list.clear();

	ItemList::iterator item;
	for (item = sub_vec[0].item_list.begin(); item != sub_vec[0].item_list.end(); item++)
		forward_list.push_back(Forward(*item));
	
}


Forward::Forward(const Item& item):
     enabled  (item.field_vec[0].int_value),
     comment  (item.field_vec[1].string_value),
     profiles (item.field_vec[2].string_list_value),
     proxy    (item.field_vec[3].string_value),
     username (item.field_vec[4].string_value),
     password (item.field_vec[5].string_value),
     domain   (item.field_vec[6].string_value),
     port     (item.field_vec[7].int_value),
     icptype  (item.field_vec[8].int_value),
     icpport  (item.field_vec[9].int_value),
     type     (item.field_vec[10].int_value),
     which    (item.field_vec[11].int_value)
{
	/* Nothing to do ! */
}

int ForwardSection::forward_do(CONNECTION * connection)
{
	Plist plist;
	struct proxy_candidate *pc;
	string myhost;

	if (connection->bypass & FEATURE_FORWARD)
		return FALSE;

	read_lock();

	if (this->enabled == FALSE) {
		unlock();

		return FALSE;
	}

	general_section->read_lock();
	myhost = general_section->hostname;
	general_section->unlock();

	ForwardList::const_iterator forward;
	for (forward = forward_list.begin(); forward != forward_list.end(); forward++) {
		if (forward->enabled == FALSE)
			continue;

		/* we want to allow ourself in the configuration to easy managing multiple proxy servers
		   (so they can use the same configuration file). */
		if ((proxyhost[0] && forward->proxy == proxyhost) || (!proxyhost[0] && forward->proxy == myhost)) continue;

		if (connection->header->type == HTTP_CONNECT) {
			if (!(forward->which & FORWARD_CONNECT))
				continue;
		} else if (!strcasecmp(connection->header->proto, "HTTP") && !(forward->which & FORWARD_HTTP))
			continue;
		else if (!strcasecmp(connection->header->proto, "FTP") && !(forward->which & FORWARD_FTP))
			continue;

		if (!profile_find(connection->profiles, forward->profiles))
			continue;

		/* allow entries with no proxy host to match, but don't do anything */
		if (forward->proxy == "")
			break;

		/* cached content that needs validation isn't forwarded through a peer proxy since
		   it may cause a recursion if the other proxy needs to validate it too. */
		if ((!(connection->flags & CONNECTION_VALIDATE) || forward->icptype == ICP_NONE) && (!(connection->flags & CONNECTION_NOPROXY) || forward->type == PROXY_SOCKS4)) {
			pc = xnew proxy_candidate;

			pc->proxy = &*forward;
			pc->priority = 0;
			plist.push_back(*pc);
		}
	}

	const Forward* fwd;
	if (!plist.empty()) {
		/* yup.. we hold the lock through the whole ICP selection process :/ */
		fwd = proxy_select(connection, plist);
		if (fwd != NULL) {
			connection->proxy_type = fwd->type;
			connection->proxy_host = xstrdup(fwd->proxy.c_str());
			if (fwd->username != "")
				connection->proxy_username = xstrdup(fwd->username.c_str());
			if (fwd->password != "")
				connection->proxy_password = xstrdup(fwd->password.c_str());
			if (fwd->domain != "")
				connection->proxy_domain = xstrdup(fwd->domain.c_str());
			connection->proxy_port = (fwd->port != -1) ? fwd->port : (fwd->type == PROXY_SOCKS4) ? 443 : 3128;

			putlog(MMLOG_FORWARD, "forwarding to %s:%d", connection->proxy_host, connection->proxy_port);
		}
	}


	unlock();

	return FALSE;
}

const Forward *ForwardSection::proxy_select(CONNECTION * connection, Plist &plist)
{
	int hpc = 0, randpeer;
	unsigned int hp = 0, op;
	Plist::iterator plist_item;

	if (connection->header->type != HTTP_CONNECT) {
		if (carp)
			peer_select_carp(connection, plist);
		else
			peer_select_icp(connection, plist);
	}

	/* find highest priority peer(s). */
	for (plist_item = plist.begin(); plist_item != plist.end(); plist_item++) {
		/* a priority of 0 is reserved for no ICP hits on sibling proxies. */
		if (!carp && plist_item->proxy->icptype == ICP_NONE) plist_item->priority = 1;

		if (plist_item->priority > hp) hp = plist_item->priority;
	}

	if (carp) {
		/* if we score the highest, then don't forward. */
		if (proxyhost[0]) {
			op = carp_hash(connection->header->url, proxyhost);
		} else {
			general_section->read_lock();
			op = carp_hash(connection->header->url, general_section->hostname.c_str());
			general_section->unlock();
		}

		if (op >= hp)
			return NULL;
	} else if (hp == 0) return NULL;

	/* count peer(s) with highest priority. */
	for (plist_item = plist.begin(); plist_item != plist.end(); plist_item++)
		if (plist_item->priority == hp) hpc++;


	randpeer = rand() % hpc;

	/* seek to randomly chosen peer. */
	for (plist_item = plist.begin(); plist_item != plist.end(); plist_item++) {
		if (plist_item->priority == hp) {
			if (!randpeer) break;
			randpeer--;
		}
	}

	return plist_item->proxy;
}

void ForwardSection::peer_select_carp(CONNECTION *connection, Plist &plist) {
	Plist::iterator plist_item;

	for (plist_item = plist.begin(); plist_item != plist.end(); plist_item++)
		plist_item->priority = carp_hash(connection->header->url, plist_item->proxy->proxy.c_str());
		
}

unsigned int ForwardSection::carp_hash(const char *url, const char *peer) {
	unsigned int urlhash = 0, peerhash = 0, combinedhash;

	for (; *url; url++)
		urlhash += ROTATELEFT(urlhash, 19) + (unsigned int)*url;

	for (; *peer; peer++)
		peerhash += ROTATELEFT(peerhash, 19) + (unsigned int)*peer;
	peerhash += peerhash * 0x62531965;
	peerhash = ROTATELEFT(peerhash, 21);

	combinedhash = urlhash ^ peerhash;
	combinedhash += combinedhash * 0x62531965;
	combinedhash = ROTATELEFT(combinedhash, 21);

	return combinedhash;
}

void ForwardSection::peer_select_icp(CONNECTION * connection, Plist &plist)
{
	int x, icpcount = 0, fd = -1, timeout;
	char buf[1024];
	struct icp_packet icppacket;
	struct sockaddr_in saddr;
	fd_set fdlist;
	struct timeval tv;

	Plist::iterator plist_item;

	icppacket.opcode = ICP_OP_QUERY;
	icppacket.version = 3;
	icppacket.mlen = htons(ICP_PACKET_LEN + strlen(connection->header->url) + 5);
	icppacket.options = 0;
	icppacket.optiondata = 0;

	/* this proxy's address */
	icppacket.hostaddr = 0;
	if (connection->interface != NULL)
		inet_pton(AF_INET, connection->interface, (struct addr_in *) &icppacket.hostaddr);

	/* client's address */
	memset(icppacket.payload, 0, 4);
	if (connection->ip != NULL)
		inet_pton(AF_INET, connection->ip, (struct addr_in *) &icppacket.payload);

	s_strncpy((char *) &icppacket.payload[4], connection->header->url, ICP_MESSAGE_MAX - 4);

	for (plist_item = plist.begin(); plist_item != plist.end(); plist_item++) {
		if (plist_item->proxy->icptype != ICP_NONE) {
			plist_item->icppacket.opcode = 255;

			if (fd == -1) {
				fd = socket(AF_INET, SOCK_DGRAM, 0);
				if (fd == -1)
					continue;

				saddr.sin_family = AF_INET;
				saddr.sin_addr.s_addr = INADDR_ANY;
				saddr.sin_port = 0;

				x = bind(fd, (struct sockaddr *) &saddr, sizeof(struct sockaddr));
				if (x == -1) {
					close(fd);
					continue;
				}
			}

			plist_item->reqnum = rand() % ~0;
			icppacket.rnum = htonl(plist_item->reqnum);

			saddr.sin_family = AF_INET;
			inet_pton(AF_INET, plist_item->proxy->proxy.c_str(), &saddr.sin_addr);
			saddr.sin_port = htons(plist_item->proxy->icpport);

			icp_packet_send(fd, &saddr, &icppacket);
			icpcount++;
		}
	}

	timeout = cache_section->icptimeout_get();

	tv.tv_sec = timeout / 1000;
	tv.tv_usec = (timeout % 1000) * 1000;

	while (icpcount) {
		FD_ZERO(&fdlist);
		FD_SET(fd, &fdlist);

		x = select(fd + 1, &fdlist, NULL, NULL, &tv);
		if (x == 1) {
			x = icp_packet_receive(fd, &saddr, &icppacket);
			if (x != -1) {
				inet_ntop(AF_INET, &saddr.sin_addr, buf, sizeof(buf));
				putlog(MMLOG_ICP, "response opcode from %s:%d: %s", buf, ntohs(saddr.sin_port), icp_opcode_name(icppacket.opcode));

				for (plist_item = plist.begin(); plist_item != plist.end(); plist_item++) {
					if (plist_item->proxy->icptype != ICP_NONE && plist_item->reqnum == icppacket.rnum) {
						memcpy(&plist_item->icppacket, &icppacket, icppacket.mlen);
						icpcount--;
						break;
					}
				}

				if (plist_item == plist.end())
					putlog(MMLOG_ICP, "received unknown ICP response number");
			} else
				putlog(MMLOG_ICP, "failed to receive packet");
		} else if (x == 0 || errno != EINTR)
			break;

	}

	if (fd != -1)
		close(fd);

	for (plist_item = plist.begin(); plist_item != plist.end(); plist_item++) {
		if (plist_item->proxy->icptype == ICP_SIBLING) {
			if (plist_item->icppacket.opcode == ICP_OP_HIT)
				plist_item->priority = 1000;
			continue;
		} else if (plist_item->proxy->icptype == ICP_PARENT) {
			if (plist_item->icppacket.opcode == ICP_OP_HIT)
				plist_item->priority = 100;
			else if (plist_item->icppacket.opcode == ICP_OP_MISS)
				plist_item->priority = 10;
			continue;
		}
	}
}



int icp_packet_send(int fd, struct sockaddr_in *dest, struct icp_packet *icppacket)
{
	return sendto(fd, icppacket, ntohs(icppacket->mlen), 0, (struct sockaddr *) dest, sizeof(struct sockaddr_in));
}

int icp_packet_receive(int fd, struct sockaddr_in *saddr, struct icp_packet *icppacket)
{
	int x, len = 0, gotlength = FALSE;
	socklen_t sl = sizeof(struct sockaddr_in);

	/* I know this may not always work... but since this algorithm is designed to select
	   the best ICP peer, receiving a fragmented packet can be an indication
	   that this peer isn't a good choice anyways */
	do {
		x = recvfrom(fd, icppacket + len, (gotlength == FALSE) ? ICP_MESSAGE_MAX - len : icppacket->mlen - len, MSG_DONTWAIT, (struct sockaddr *) saddr, &sl);
		if (x <= 0)
			return -1;

		if (gotlength == FALSE && x >= 4) {
			gotlength = TRUE;

			icppacket->mlen = ntohs(icppacket->mlen);

			/* there's no need to check if the given size is too large.. since it can't be due to the maximum
			   value of icppacket->mlen. */
			if (icppacket->mlen < ICP_PACKET_LEN) {
				putlog(MMLOG_ICP, "ICP packet too short");
				return -1;
			}
		}

		len += x;
	} while ((gotlength == FALSE && len < ICP_MESSAGE_MAX) || (gotlength == TRUE && len < icppacket->mlen));

	icppacket->rnum = ntohl(icppacket->rnum);

	return len;
}

char *icp_opcode_name(int opcode)
{
	switch (opcode) {
	case 0:
		return "ICP_OP_INVALID";
	case 1:
		return "ICP_OP_QUERY";
	case 2:
		return "ICP_OP_HIT";
	case 3:
		return "ICP_OP_MISS";
	case 4:
		return "ICP_OP_ERR";
	case 10:
		return "ICP_OP_SECHO";
	case 11:
		return "ICP_OP_DECHO";
	case 21:
		return "ICP_OP_MISS_NOFETCH";
	case 22:
		return "ICP_OP_DENIED";
	case 23:
		return "ICP_OP_HIT_OBJ";
	default:
		return "INVALID";
	}
}

void icp_thread()
{
	int fd = -1, x, port;
	struct sockaddr_in saddr;
	struct icp_packet icppacket;
	fd_set fdlist;
	char *newurl;
	URL *url;

	port = cache_section->icpport_get();
	if (port == 0)
		return;

	fd = socket(AF_INET, SOCK_DGRAM, 0);
	if (fd == -1)
		goto error;

	saddr.sin_family = AF_INET;
	saddr.sin_addr.s_addr = INADDR_ANY;
	saddr.sin_port = htons(port);

	x = bind(fd, (struct sockaddr *) &saddr, sizeof(struct sockaddr_in));
	if (x == -1)
		goto error;

	while (1) {
		FD_ZERO(&fdlist);
		FD_SET(fd, &fdlist);

		x = select(fd + 1, &fdlist, NULL, NULL, NULL);
		if (x == 1) {
			x = icp_packet_receive(fd, &saddr, &icppacket);
			if (x != -1) {
				icppacket.rnum = htonl(icppacket.rnum);

				if (icppacket.opcode == ICP_OP_QUERY) {
					if (icppacket.mlen <= 24 || memchr(icppacket.payload + 4, '\0', icppacket.mlen - ICP_PACKET_LEN - 4) == NULL) {
						icppacket.opcode = ICP_OP_INVALID;
						icppacket.mlen = htons(ICP_PACKET_LEN);
						icp_packet_send(fd, &saddr, &icppacket);
						putlog(MMLOG_ICP, "received ICP QUERY with no payload");
						continue;
					}

					url = url_parse((char *) icppacket.payload + 4);
					if (url == NULL) {
						icppacket.opcode = ICP_OP_INVALID;
						icppacket.mlen = htons(ICP_PACKET_LEN);
						icp_packet_send(fd, &saddr, &icppacket);
						putlog(MMLOG_ICP, "received invalid URL in query");
						continue;
					}

					/* get rid of the client host in the response payload 
					   (probably unnecessary) */
					memmove(icppacket.payload, icppacket.payload + 4, icppacket.mlen - ICP_PACKET_LEN - 4);
					icppacket.mlen = htons(icppacket.mlen - 4);

					newurl = url_create(url);
					if (cache_section->exists(newurl)) {
						putlog(MMLOG_ICP, "hit: %s", newurl);
						icppacket.opcode = ICP_OP_HIT;
					} else {
						putlog(MMLOG_ICP, "miss: %s", newurl);
						icppacket.opcode = ICP_OP_MISS;
					}
					xfree(newurl);

					icp_packet_send(fd, &saddr, &icppacket);
				} else {
					putlog(MMLOG_ICP, "unhandled ICP opcode: %s", icp_opcode_name(icppacket.opcode));
					icppacket.opcode = ICP_OP_INVALID;
					icppacket.mlen = htons(ICP_PACKET_LEN);

					icp_packet_send(fd, &saddr, &icppacket);
				}
			}
		}
	}

      error:
	if (fd != -1)
		close(fd);

	putlog(MMLOG_ERROR, "failed to start ICP thread");
}
