#ifndef TYPES_H
#define TYPES_H

typedef pair<string, string> StringPair;
typedef list<StringPair> StringPairList;
typedef map<string, string> StringPairMap;

typedef struct STACK_T {
	struct STACK_T *next, *prev;
	char *data;
};

typedef list<struct proxy_candidate> Plist;

struct tm_partial {
	int m_year, tm_year;
	int m_mon, tm_mon;
	int m_mday, tm_mday;
	int m_wday, tm_wday;
	int m_hour, tm_hour;
	int m_min, tm_min;
};

class Filebuf {
   public:
        unsigned int size, realsize;
        char *data, *type;

        Filebuf();
        ~Filebuf();
        Filebuf *Dup();
        void Add(const char *, int);
        void Resize(int);
        void Addf(const char *, ...);
        void Replace(int, int, char *);
        void Shorten();
        int Compress(int);
        int Uncompress(int);
        int Save(char *);
        int Read(char *);
        struct STACK_T *to_stack();
        int Decode(CONNECTION *);
        int Encode(CONNECTION *);
};

struct HASH_LIST {
	struct HASH_LIST *next;
	time_t age;
	char *ref;
	void *data;
	int flags;
};

typedef struct {
	int size;
	struct HASH_LIST **hash_list;
} HASH_TABLE;

struct htmlstream_tag_property {
	struct htmlstream_tag_property *next, *prev;
	char *name;
	char *value;
};

struct htmlstream_tag {
	char *name;
	struct htmlstream_tag_property *properties;
};

struct htmlstream_node {
	struct htmlstream_node *down, *prev, *next, *up;
	int level;
	char *text;
	struct htmlstream_tag *tag;
};

typedef struct {
	HASH_TABLE *callbacks;
	struct htmlstream_node *tree;
	struct htmlstream_node *tail;
	unsigned int startpos;
	int incomment;
	Filebuf *buffer;
} HTMLSTREAM;

struct htmlstream_callback {
	void (*func)(HTMLSTREAM *, struct htmlstream_node *, void *);
	void *arg;
};

class SocketCallBack {
   private:
	int (*func)(void *, int, char *);
   public:
	int id;
	int events;
	void *arg;
	SocketCallBack(int (*)(void *, int, char *), void *);
	int Call(int, char *);
};

typedef list<SocketCallBack> SocketCallBackList;

class Socket {
   private:
	int type;
	unsigned int cbid, eventmask;
        bool frozen;
        size_t inbuf_reallen, outbuf_reallen;
        char *inbuf, *outbuf;
	SocketCallBackList callback_list;

#ifdef HAVE_SSL
	SSL *ssl;
	SSL_CTX *ctx;
#endif
   public:
	// these should be private
        int fd;
        size_t inbuf_len, outbuf_len;
	size_t bytesread, byteswritten;

        static const int READEVENT = 1;
        static const int WRITEEVENT = (1<<1);
        static const int ERROREVENT = (1<<2);

	static const int SOCK_NORMAL = 0;
	static const int SOCK_SSLCLIENT = 1;
	static const int SOCK_SSLSERVER = 2;

        static const int INBUF = 0;
        static const int OUTBUF = 1;

	Socket();
        Socket(int);
        ~Socket();
	bool Encrypt(int);
        void Flush();
        void Resize(size_t, int);
        int Write(char *, int);
        int Read(char *, int);
        int Read(char *, int, int);
        void Clear(int);
        int GetLine(char *, int);
        int GetLine(char *, int, int);
        int PushFront(char *, int);
	int PutSock(char *, ...);
	int Connect(const char *, int);
	bool Connected();
	int WriteFilebuf(const Filebuf *);

	int CallBackAdd(int (*)(void *, int, char *), void *, int);
	bool CallBackRemove(int);
	int CallBackMaskGet(int);
	int CallBackMaskSet(int, int);
	int CallBackEvent(int, int, char *);
};

typedef struct {
	char *username;
	char *password;
	char *proto;
	char *host;
	char *file;
	int port;
} URL;

struct url_command_t {
	char *command;
	char *options;
};

typedef struct {
	pcre *pattern;
	pcre_extra *extra;
} regex_sub_t;

typedef struct {
	int rm_so, rm_eo, subcount;
	const char **substrings;
} regmatch_sub_t;

typedef struct XML_LIST_T {
	struct XML_LIST_T *prev, *next;
	int type;
	char *item;
} XML_LIST;

struct HTTP_HEADER {
	int send;
	string type;
	string value;
};


typedef list<struct HTTP_HEADER> HttpHeaderList;

struct LOGBUFFER_LIST {
	struct LOGBUFFER_LIST *next, *prev;
	char *msg;
};

typedef struct {
	pthread_rwlock_t lock;
	int entries, size;
	struct LOGBUFFER_LIST *head, *tail;
} LOGBUFFER;	

typedef struct {
	char *raw;
	char *url;
	int type;
	int version;
	int port;
	int code;
	unsigned int content_length;
	int chunked;
	int keepalive;
	int proxy_keepalive;
	int null_body;
	char *username;
	char *password;
	char *method;
	char *proto;
	char *host;
	char *file;
	char *referer;
	char *location;
	char *content_type;
	char *content_encoding;
	char *accept_encoding;
	char *authorization;
	char *range;
	char *proxy_authenticate;
	char *proxy_authorization;
	char *host_header;
	char *cache_control;
	char *expires;
	char *if_modified_since;
	char *last_modified;
	char *warning;
	char *age;
	char *x_cache;
	char *xforwardedfor;
	char *via;
	HttpHeaderList header;
	HttpHeaderList header_filtered;
	int flags;
} HEADER;

typedef struct {
	HEADER *header;
	Filebuf *body;
} HTTPRESPONSE;

class Store {
 public:
	/* Should be private */
	const int& enabled;
	const string& comment;
	const StringList& profiles;
	const string& path;
	const size_t& maxdisksize;
	const size_t& diskextra;

	int journalfd, journalentries, diskentries;
	size_t disksize;
	
 public:
	Store(const Item&);	
};

class Refresh {
 public:
	/* should be private */
	const int& enabled;
	const string& comment;
	const StringList& profiles;
	const int &cachable;	
	const unsigned int& minage;
	const unsigned int& maxage;
	const unsigned int& validate;
	const unsigned int& lmfactor;
 public:
	Refresh(const Item&);
};


typedef list<Store> StoreList;
typedef list<Refresh> RefreshList;

typedef struct {
	pthread_cond_t writecompletecond;
	int fd;
        char *key;
        char *data;
        int refcount;
        size_t size, realsize;
	size_t offset;
	int flags;
	HEADER *header;
	Store *store;
	time_t atime, mtime, etime, ftime, lmtime;
	time_t minage, maxage, validate;
	int lmfactor;
} CACHEMAP;

struct CACHEMAPLIST_T {
        struct CACHEMAPLIST_T *prev, *next;
        CACHEMAP *cachemap;
};

class CacheSection: public Section {
 	void update();

 public:
	/* should be private */
	int& enabled;
	int& violaterfc;
        unsigned int& maxmemsize;
	unsigned int& memextra;
	unsigned int& minsize;
	unsigned int& maxsize;
	unsigned int& maxwaitsize;	
	int& prefetchwindow;
	int& icpport;
	int& icptimeout;
	int& sbalancemethod;

	int hashsize;
	unsigned int memsize;
	int mementries;
	time_t lastclean;
	StoreList store_list;
	RefreshList refresh_list;
        struct CACHEMAPLIST_T **cachemap;

	void select_store(CONNECTION *, CACHEMAP *);

	void select_refresh(CONNECTION *, CACHEMAP *);
 public:
	CacheSection();
	void stores_update();
	CACHEMAP *hash_find(char *);
	int filename(CACHEMAP *, char *, int);
	void hash_add(CACHEMAP *);
	void hash_remove(CACHEMAP *);
	CACHEMAP *cache_open(CONNECTION *connection, char *, char *, int);
	void cache_close(CACHEMAP *);
	void close_unlocked(CACHEMAP *);
	int add_unlocked(CACHEMAP *, void *, size_t);
	int add(CACHEMAP *, void *, size_t);
	void scandisk(Store*);
	void clean();
	void check(HEADER *);
	void invalidate(CACHEMAP *);
	void invalidate_unlocked(CACHEMAP *);
	void disk_delete(CACHEMAP *);
	void mem_delete(CACHEMAP *);
	void cache_free(CACHEMAP *);
	int flag_set(CACHEMAP *, int);
	int flag_unset(CACHEMAP *, int);
	void touch(CACHEMAP *, int);
	void touch_unlocked(CACHEMAP *, int);
	void check_header(CACHEMAP *);
	void validated(CACHEMAP *);
	int invalidate_key(char *);
	size_t resize(CACHEMAP *, size_t);
	int exists(char *);
	void journal_write(Store *);
	void journal_read(Store *);
	int journal_add(CACHEMAP *);
	CACHEMAP *cachemap_new(bool);
	int icptimeout_get() const;
	int icpport_get() const;
	void force_clean();
};


typedef struct {
	int status;
	int fd;
	int port;
	char *ip;
} SOCKLIST;

struct CONNECTION_T {
	int thread;
	int flags;
	Socket *client, *server;
	int depth;
	char *ip;
	int port;
	char *interface;
	int request;
	int keepalive_client, keepalive_server;
	int authenticate;
	size_t transferred, transferlimit;
	char *proxy_auth;
	char *proxy_host;
	int proxy_port;
	int proxy_type;
	char *proxy_username;
	char *proxy_password;
	char *proxy_domain;
	int access;
	int bypass;
	struct url_command_t **url_command;
	StringList profiles, oprofiles;
	CACHEMAP *cachemap;
	HEADER *header, *rheader;
	Filebuf *postbody;
	HTMLSTREAM *htmlstream;
	StringPairMap variables;
};

class CookieHeader {
  public:
	bool has_expires;
	int version;
	StringPairList name_pairs;
        string domain;
        string path;
        struct tm expires;

        CookieHeader(string);
        string Build();
};

class Cookie {
 public:
	const int enabled;
	const string comment;
	const StringList profiles;

	const int limityears;
	const int year1;
	const int year2;

	const int limitmonths;
	const int month1;
	const int month2;

	const int limitdays;
	const int day1;
	const int day2;

	const int limitweekdays;
	const int weekday1;
	const int weekday2;

	const int limithours;
	const int hour1;
	const int hour2;

	const int limitminutes;
	const int minute1;
	const int minute2;
	
	string domain;
	string path;

	const int direction;
	const int matchmode;

	struct tm_partial from, to;
	regex_t *dp, *pp;

	Cookie(const Item&);
	Cookie(const Cookie &);
	~Cookie();
};

typedef list<Cookie> CookieList;

class CookieSection : public Section {
 private:
	void update();
	const int& enabled;
	const int& policy;
	CookieList allow_list, deny_list;

 public:
	CookieSection();
	bool check(int, CONNECTION *, CookieHeader &);
};

class Filter : public Encapsulator{
 public:
	const int enabled;
	const string comment;
	const StringList profiles;
	const string host;
	const string file;
	const IpRangeList iprange;
	const string templ;

	regex_t *he, *fe;

 public:
	Filter();
	Filter(const Item&);
	Filter(const Filter&);
	~Filter();
};

typedef list <Filter> FilterList;

class FilterSection : public Section {
 private:
	const int& enabled;
	const int& policy;
	const string& dtempl;
	FilterList allow_list, deny_list;
	Filter empty;

 	void update();
	const Filter* check(CONNECTION*);

 public:
	FilterSection();
	int check_and_block(CONNECTION *);
	void check_show(CONNECTION *);
};

class Keyword {
 public:
	const int enabled;
	const string comment;
	const StringList profiles;
	const string mime;
	const int size;
	const string keyword;
	const int score;

	regex_t *me, *ke;	

	Keyword(const Item&);
	Keyword(const Keyword&);
	~Keyword();
};

typedef list<Keyword> KeywordList;

class KeywordSection : public Section {
 private:
	void update();
	/* should be private */
	const int& enabled;
	KeywordList keyword_list;
	const int& threshold;
	const string& templ;

 public:
	KeywordSection();
	bool check_and_block(CONNECTION *, Filebuf *);
	int check(CONNECTION *, Filebuf *, int);
};

class Redirect {
 public:
	const int enabled;
	const string comment;
	const StringList profiles;
	const string url;
	const string redirect;
	const int port;
	const int send302;
	const int options;
	const int which;
	
	regex_sub_t *up;

	Redirect(const Item&);
	Redirect(const Redirect&);
	~Redirect();
};

typedef list<Redirect> RedirectList;

class RedirectSection : public Section {
 private:
	void update();
	const int& enabled;
	RedirectList redirect_list;

 public:
	RedirectSection();
	int redirect_do(CONNECTION *, int);
};

class Forward {
 public:
	const int enabled;
	const string comment;
	const StringList profiles;
	const string proxy;
	const string username;
	const string password;
	const string domain;
	const int port;
	const int icptype;
	const int icpport;
	const int type;
	const int which;
	
	Forward(const Item&);
};

typedef list<Forward> ForwardList;

class ForwardSection : public Section {
 private:
	void update();
	const int& enabled;
	const int& carp;
	ForwardList forward_list;
	const Forward* proxy_select(CONNECTION *, Plist &);
	void peer_select_icp(CONNECTION *, Plist &);
	void peer_select_carp(CONNECTION *, Plist &);
	unsigned int carp_hash(const char *, const char *);
 public:
	ForwardSection();
	int forward_do(CONNECTION *);
};

class Mime: public Encapsulator {
 public:
	const int enabled;
	const string comment;
	const StringList profiles;
	const string host;
	const string file;
	const string mime;
	const string templ;

	regex_t *me, *he, *fe;

 public:
	Mime();
	Mime(const Item&);
	Mime(const Mime&);
	~Mime();
};

typedef list <Mime> MimeList;

class MimeSection : public Section {
 private:
 	void update();

 public:
	/* should be private */
	const int& enabled;
	const int& policy;
	const string& dtempl;

	MimeList allow_list, deny_list;
	Mime empty;
	const Mime* check(CONNECTION *);
 public:
	MimeSection();
	bool check_and_block(CONNECTION *);
	void check_show(CONNECTION *);
};

class Header {
 public:
	const int enabled;
	const string comment;
	const StringList profiles;
	const string type;
	const string value;
	const int which;

	regex_t *te, *ve;

 public:
	Header(const Item&);
	Header(const Header&);
	~Header();
};

typedef list <Header> HeaderList;

class HeaderSection: public Section {
	void update();	
	HeaderList allow_list, deny_list, insert_list;
	
	const int& enabled;
	const int& policy;

 public:
	HeaderSection();
	void filter(CONNECTION *, int);

};


class Access {
 public:
	const int enabled;
	const string comment;
	const StringList profiles;
	const string ip;
	const int pamauth;
	const string username;
	const string password;
	const int access;
	const int bypass;
	
	regex_t *ue, *ie;

	Access();
	Access(const Item&);
	Access(const Access&);
	~Access();
	int has_ip(char*) const;
	int has_port(int) const;
	void setup(CONNECTION*);
};


typedef list <Access> AccessList;

class AccessSection : public Section {
 private:
	AccessList allow_list, deny_list;
	Access empty;
	const int& policy;

 	void update();
 	const Access* check(CONNECTION*, const char*, const char*);
	void setup(CONNECTION *connection, const Access *al);

public:
	AccessSection();
	bool check_and_setup(CONNECTION* connection);
	void check_and_setup(CONNECTION* connection, char*, char*);
};

class Listen {
 public:
	const int enabled;
	const string comment;
	const int port;
	const string ip;
	
	Listen (const Item&);
};

typedef list<Listen> ListenList;

class NetworkSection : public Section {
 private:
	void update();
	ListenList listen_list;

 public:
	NetworkSection();
	void check();
	int has_port(int);
};

class Template {
 public:
	const int enabled;
	const string comment;
	const StringList profiles;
	const string name;
	const string file;
	const string mime;
	const int code;
	const int type;
	const int parsable;
	
	Template(const Item&);
};

typedef list<Template> TemplateList;

class TemplateSection : public Section {
 private:
	void update();
	const string& path;
	TemplateList template_list;

 public:
	TemplateSection();
	Filebuf *get(CONNECTION *, const char *, int *);
	int send(const char *, CONNECTION *, int);
	int insert(CONNECTION *, Filebuf *, const char *);
};

class External {
 public:
	const int enabled;
	const string comment;
	const StringList profiles;
	const string exec;
	const string mime;
	const int size;
	const string newmime;
	const int type;

	regex_t *me;
	
	External(const Item&);
	External(const External&);
	~External();
};

typedef list<External> ExternalList;

class ExternalSection : public Section {
 private:
	void update();
	const int& enabled;
	ExternalList external_list;

 public:
	ExternalSection();
	const External* find(CONNECTION *) const;
	void process(CONNECTION *, Filebuf *) const;
	bool check_match(CONNECTION *) const;

};

class Rewrite {
 public:
	const int enabled;
	const string comment;
	const StringList profiles;
	const string mime;
	const int size;
	const string pattern;
	const string replace;
	const int which;

	regex_t *me;
	regex_sub_t *pe;
	
	Rewrite(const Item&);
	Rewrite(const Rewrite&);
	~Rewrite();
};

typedef list<Rewrite> RewriteList;

class RewriteSection : public Section {
 private:
	void update();
	const int& enabled;
	RewriteList rewrite_list;

 public:
	RewriteSection();
	int rewrite_do(CONNECTION *, Filebuf *, int, int);
};

class Profile {
 public:
	const int enabled;
	const string comment;
	const StringList profiles;
	const string protocol;
	const string host;
	const string file;
	const string mime;
	const PortRangeList portrange;
	const StringList addprofiles;
	const StringList removeprofiles;
	const StringList urlcommand;

	regex_t *pe, *he, *fe, *me;
	
	Profile(const Item&);
	Profile(const Profile&);
	~Profile();
};

typedef list<Profile> ProfileList;

class ProfileSection : public Section {
 private:
	void update();
	ProfileList profile_list;

 public:
	ProfileSection();
	void profiles_update(CONNECTION *);
};

class Limit {
 public:
	const int& enabled;
	const string& comment;
	const StringList& profiles;
	const int& action;
	const string& templ;

	const int& limitmonths;
	const int& month1;
	const int& month2;

	const int& limitdays;
	const int& day1;
	const int& day2;

	const int& limitweekdays;
	const int& weekday1;
	const int& weekday2;

	const int& limithours;
	const int& hour1;
	const int& hour2;

	const int& limitminutes;
	const int& minute1;
	const int& minute2;

	struct tm_partial from, to;

	const unsigned int& maxbytes;
	unsigned int& n_bytes;
	const unsigned int& maxrequests;
	unsigned int& n_requests;
	
	const int& matchmode;
	const int& flags;
	
	Limit(Item&);
};

typedef list<Limit> LimitList;

class LimitSection : public Section {
 private:
	LimitList limit_list;
	const int& enabled;

	void update();
	const Limit* check(CONNECTION *) const;

 public:	
	LimitSection();
	int check_and_block(CONNECTION *) const;
	void update(CONNECTION *);
	void reset();
};


class Antivirus {
 public:
	const int enabled;
	const string comment;
	const StringList profiles;
	const string templ;
	
	Antivirus(const Item&);
};

typedef list<Antivirus> AntivirusList;

class AntivirusSection : public Section {
 private:
	AntivirusList antivirus_list;
	const int &enabled;
	const string &clamavhost;
	const int &clamavport;
	const string &dtempl;
	void update();
	const Antivirus *find(CONNECTION *) const;
 public:
	AntivirusSection();
	int check_and_block(CONNECTION *, Filebuf *) const;
	bool check_match(CONNECTION *) const;
};	


typedef struct {
	struct in_addr *addr;
	int len;
} HOSTENT;

typedef struct {
	pthread_mutex_t lock;
	pthread_cond_t cond;
	void *arg;
	int flags, pid;
	int requests;
	char *ip, *host, *file, *method;
	int port;
} THREADLIST;
 
class GeneralSection : public Section {
 	void update();

 public:
	/* should be private */
	string& hostname;
	int& ctimeout;
	int& timeout;
	int& keeptimeout;
	unsigned int& maxbuffer;
	string& tempdir;
	int& linelength;
	PortRangeList& cportrange;
	int& poolsize;
	int& pooltimeout;
	string& encodemime;
	int& compressout;  
	int& compressin;
	int& xforwardedfor;
	int& via;
	regex_t *emp;

 public:
	GeneralSection();
	
	int ctimeout_get() const;
	int timeout_get() const;
	int keeptimeout_get() const;
	int maxbuffer_get() const;
	int linelength_get() const;
	int cportrange_check(int) const;
	int poolsize_get() const;
	int pooltimeout_get() const;
	int compressout_get() const;
	int compressin_get() const;
	int xforwardedfor_get() const;
	int via_get() const;
};

class FtpSection : public Section {
 	void update();

 public:
	/* should be private */
	int& passive;
	int& timeout; 
	string& anonlogin;
	string& anonpass;
	int& sortorder;
	int& sortfield;

 public:
	FtpSection();
	int passive_get() const;
	int timeout_get() const;
	int sortorder_get() const;
	int sortfield_get() const;
};

struct pool_t {
	struct pool_t *next, *prev;
	Socket *sock;
	time_t age;
	char *proto;
	char *host;
	char *username, *password;
	int port;
};

typedef struct {
	int entries;
	pthread_mutex_t lock;
	struct pool_t *pool_list;
} POOL;

class DnsblSection: public Section {
 private:
 	void update();
	/* should be private */
	int& enabled;
	string& templ;
	string& domain;
	IpRangeList& badips;

 public:
	bool check_and_block(CONNECTION *) const;
	DnsblSection();
};

struct TIMERS_LIST {
	struct TIMERS_LIST *prev, *next;
	int tid;
	int interval;
	time_t etime;
	int flags;
	void (*callback)(void *);
	void *arg;
};

typedef struct {
	pthread_mutex_t lock;
	int tid;
	struct TIMERS_LIST *timers;
} TIMERS;

struct PREFETCH_QUEUE {
	struct PREFETCH_QUEUE *next;
	int depth;
	unsigned int maxsize;
	URL *url;
	HttpHeaderList *header;
};	

class Prefetch {
public:
	const int enabled;
	const string comment;
	const StringList profiles;
	const string tag;
	const string attribute;
	const string pattern;
	const unsigned int maxsize;
	const int recursion;

	regex_t *pe;

	Prefetch(const Item&);
	Prefetch(const Prefetch&);
	~Prefetch();
};

typedef list<Prefetch> PrefetchList;

class PrefetchSection: public Section {
 public:
	/* should be private */
	int& enabled;
	int& threads;
	int& queuesize; 
	
	pthread_cond_t cond;
	int entries, maxdepth;
	struct PREFETCH_QUEUE *queue;

 public:
	PrefetchSection();
	void init_threads();
	void thread();
	void queue_add(URL *, HttpHeaderList *, unsigned int, int);
	void setup_callbacks(CONNECTION *);

	friend void prefetch_callback(HTMLSTREAM *hs, struct htmlstream_node *node, CONNECTION *connection);

 private:
	PrefetchList prefetch_list;
	void update();
};


class StatCounter {
	public:
	static const int UINT = 1;
	static const int FILESIZE = (1<<1);
	static const int STRING = (1<<2);

	static const int CALLBACK= (1<<3);

	int type;
	string name;

	union {
		void *(*callback)();
		unsigned int count;
	}  value;

	char *Get();
};

typedef list<struct StatCounter> StatCounterList;

class Stat;

class StatGroup {
        private:
                int type;
                unsigned int total;
		string name;

                StatCounterList counter_list;
        public:
                static const int NORMAL = 0;
                static const int RATIO = 1;

                StatGroup(string, int);
                bool Add(string, int type);
		bool Add(string, unsigned int (*)(), int);
		bool Add(string, char *(*)(), int);
                unsigned int Increment(string);
                unsigned int Decrement(string);
                unsigned int Increment(string, unsigned int);
                unsigned int Decrement(string, unsigned int);

        friend class Stats;
};


typedef list<StatGroup> StatGroupList;

class Stats {
        private:
                StatGroupList group_list;
        public:
                bool AddGroup(string, int);
                bool AddCounter(string, string, int);
                bool AddCounter(string, string, unsigned int (*)(), int);
                bool AddCounter(string, string, char *(*)(), int);
                unsigned int Increment(string, string);
                unsigned int Decrement(string, string);
                unsigned int Increment(string, string, unsigned int);
                unsigned int Decrement(string, string, unsigned int);
                void Display(Filebuf *);
};

typedef list<string> UnlinkList;

typedef struct {
	pthread_mutex_t rusage_lock;
	struct rusage rusage;

	LOGBUFFER *logbuffer;
	HASH_TABLE *dns_cache;
	POOL *pool;
	TIMERS *timers;
	Stats stats;
	UnlinkList *unlink_list;	
	pthread_mutex_t unlink_lock;
	pthread_cond_t unlink_cond;
	pthread_mutex_t dns_cache_lock;
} GLOBAL;

struct dir_t {
	struct dir_t *next, *prev;
	int isdir;
	mode_t mode;
	char *user;
	char *group;
	size_t size;
	struct tm time;
	time_t t;
	char *name;
};

struct icp_packet {
        unsigned char opcode;
        unsigned char version;
        uint16_t mlen;
        uint32_t rnum;
        uint32_t options;
        uint32_t optiondata;
        uint32_t hostaddr;
        unsigned char payload[ICP_MESSAGE_MAX];
};

struct proxy_candidate {
	unsigned int reqnum;
	unsigned int priority;
	struct icp_packet icppacket;
	const Forward *proxy;
};

struct statm_t {
	unsigned int size;
	unsigned int resident;
	unsigned int share;
	unsigned int trs;
	unsigned int drs;
	unsigned int lrs;
	unsigned int dt;
};

#ifdef _DEBUG
typedef struct {
	void* address;
	char* file;
	int line;
	char* function;
} MINFO;
#endif

#endif				/* TYPES_H */
