/***************************************************************************
 *   Copyright (C) 2005 by Florian Roth   *
 *   florian@synatic.net   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "metabar_widget.h"

#include <qsize.h>
#include <qfont.h>
#include <qcolor.h>
#include <qcursor.h>
#include <qhbox.h>
#include <qfile.h>
#include <qobjectlist.h>
#include <qtooltip.h>
#include <qgrid.h> 
#include <qpainter.h>

#include <klocale.h>
#include <kiconloader.h>
#include <ktrader.h>
#include <kmimetype.h>
#include <krun.h>
#include <kurl.h>
#include <kglobalsettings.h>
#include <kpropertiesdialog.h>
#include <kapplication.h>
#include <dcopref.h>
#include <dcopclient.h>
#include <kinputdialog.h>
#include <klocale.h>
#include <kstandarddirs.h>
#include <kio/jobclasses.h>
#include <kmessagebox.h>
#include <ksycocaentry.h>
#include <kaction.h>
#include <kdesktopfile.h>
#include <kstandarddirs.h>
#include <kpixmap.h>
#include <kpixmapeffect.h>

#include <soundserver.h>
#include <kartsdispatcher.h>

#include <kdebug.h>

MetabarWidget::MetabarWidget(QWidget *parent, const char *name)
    : QWidget( parent, name )
{
  playobj = 0;
  preview_job = 0;
  
  show_more = 0;
  show_more_actions = 0;
  
  currentItems = 0;
  
  player_timer = new QTimer;
  connect(player_timer, SIGNAL(timeout()), this, SLOT(updatePlayerStatus()));
  
  dispatcher = new KArtsDispatcher();
  soundServer = Arts::Reference("global:Arts_SoundServerV2");
  factory = new KDE::PlayObjectFactory(soundServer);
  
  config = new KConfig("metabarrc");
  iconConfig = new KConfig(locate("data", "metabar/iconsrc"));
  
  QColor background = KGlobalSettings::baseColor();
  if(!background.isValid()){
    background = QColor(255, 255, 255);
  }
  
  actionList.setAutoDelete(TRUE);
  openList.setAutoDelete(TRUE);
  linkList.setAutoDelete(TRUE);
  infoList.setAutoDelete(TRUE);
  infoDict.setAutoDelete(TRUE);
    
  dir_watch = new KDirWatch();
  connect(dir_watch, SIGNAL(dirty(const QString&)), this, SLOT(slotUpdateCurrentInfo(const QString&)));
  connect(dir_watch, SIGNAL(created(const QString&)), this, SLOT(slotUpdateCurrentInfo(const QString&)));
  connect(dir_watch, SIGNAL(deleted(const QString&)), this, SLOT(slotDeleteCurrentInfo(const QString&)));
  
  setPaletteBackgroundColor(background);
    
  //top box  
  top_frame = new QFrame(this);
  top_frame->setMargin(10);
  top_frame->setFrameShape(QFrame::StyledPanel);
  top_frame->setFrameShadow(QFrame::Raised);
  top_frame->setLineWidth(1);
  
  config_button = new KPushButton(top_frame);
  config_button->setPixmap(SmallIcon("configure"));
  config_button->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum);
  connect(config_button, SIGNAL(clicked()), this, SLOT(slotShowConfigDialog()));
  
  icon = new KURLLabel(top_frame);
  connect(icon, SIGNAL(leftClickedURL()), this, SLOT(slotShowPropertiesDialog()));
  
  name_label = new KSqueezedTextLabel(top_frame);
  
  QFont font = name_label->font();
  font.setBold(true);
  font.setPointSize(14);
  name_label->setFont(font);
  
  type_label = new KSqueezedTextLabel(top_frame);
    
  QGridLayout *top_frame_layout = new QGridLayout(top_frame, 2, 3, 5, 5);  
  top_frame_layout->addMultiCellWidget(icon, 0, 1, 0, 0);
  top_frame_layout->addWidget(name_label, 0, 1);
  top_frame_layout->addWidget(config_button, 0, 2);
  top_frame_layout->addMultiCellWidget(type_label, 1, 1, 1, 2);
  
  //action box
  action_frame = new MetaFrame(config, this);
  action_frame->setTitle(i18n("Actions"));
  
  serviceLoader = new ServiceLoader(action_frame, config);
  
  //open box
  open_frame = new MetaFrame(config, this);
  open_frame->setTitle(i18n("Open with %1").arg("..."));

  //link box  
  link_frame = new MetaFrame(config, this);
  link_frame->setTitle(i18n("Links"));
  
  loadLinkList();
  
  //info box  
  info_frame = new MetaFrame(config, this);
  info_frame->setTitle(i18n("Information"));
  
  info_table = new QLabel("", info_frame->mainWidget());
  
  info_frame->addWidget(info_table);
  
  //preview box  
  preview_frame = new MetaFrame(config, this);
  preview_frame->setTitle(i18n("Preview"));
  preview_frame->hide();
  connect(preview_frame, SIGNAL(toggled(bool)), this, SLOT(slotCreatePreview(bool)));
  
  preview = new QLabel(0);
  preview->setAlignment(Qt::AlignHCenter);
  preview->setPaletteBackgroundColor(background);
  
  player_frame = new QFrame(preview_frame->mainWidget());
  
  player_start = new KPushButton(player_frame);
  player_start->setPixmap(SmallIcon("player_play"));
  connect(player_start, SIGNAL(clicked()), this, SLOT(startPlayer()));
  
  player_pause = new KPushButton(player_frame);
  player_pause->setPixmap(SmallIcon("player_pause"));
  connect(player_pause, SIGNAL(clicked()), this, SLOT(pausePlayer()));
  
  player_stop = new KPushButton(player_frame);
  player_stop->setPixmap(SmallIcon("player_stop"));
  connect(player_stop, SIGNAL(clicked()), this, SLOT(stopPlayer()));
  
  player_status = new QSlider(Qt::Horizontal, player_frame);
  player_status->setTracking(FALSE);
  connect(player_status, SIGNAL(valueChanged(int)), this, SLOT(seekPlayer(int)));
  
  video = new KVideoWidget();
  
  preview_stack = new QWidgetStack(preview_frame->mainWidget());
  preview_stack->addWidget(preview);
  preview_stack->addWidget(video);
  preview_stack->raiseWidget(preview);
  
  preview_frame->addWidget(preview_stack, false);
  preview_frame->addWidget(player_frame, false);
  
  //layouts  
  QHBoxLayout *player_frame_layout2 = new QHBoxLayout();
  player_frame_layout2->addWidget(player_start);
  player_frame_layout2->addWidget(player_pause);
  player_frame_layout2->addWidget(player_stop);
  
  QVBoxLayout *player_frame_layout = new QVBoxLayout(player_frame, 5, 5);
  player_frame_layout->addLayout(player_frame_layout2);
  player_frame_layout->addWidget(player_status);
  
  QVBoxLayout *main_layout = new QVBoxLayout(this, 5, 5);
  main_layout->addWidget(top_frame);
  main_layout->addWidget(open_frame);
  main_layout->addWidget(action_frame);
  main_layout->addWidget(link_frame);
  main_layout->addWidget(info_frame);
  main_layout->addWidget(preview_frame);
  main_layout->addItem(new QSpacerItem(10, 10, QSizePolicy::Expanding, QSizePolicy::Expanding));
  
  //load config
  loadConfig();
}

MetabarWidget::~MetabarWidget(){
  config->setGroup("General");
  config->writeEntry("CollapseOpenWith", !open_frame->isExpanded());
  config->writeEntry("CollapseActions", !action_frame->isExpanded());
  config->writeEntry("CollapseInfo", !link_frame->isExpanded());
  config->writeEntry("CollapsePreview", !preview_frame->isExpanded());
  config->writeEntry("CollapseLinks", !link_frame->isExpanded());
  config->sync();
  
  deletePlayer();
  
  delete playobj;
  delete factory;
  delete player_timer;
  delete preview_job;
  
  delete config;
  delete dir_watch;
  delete currentItems;
  delete serviceLoader;
}

void MetabarWidget::setFileItems(const KFileItemList &items)
{
  if(currentItems && *currentItems == items){
    return;
  }
  else{
    int count = items.count();
    
    KFileItem *currentItem = 0;
    if(currentItems){
      currentItem = currentItems->isEmpty() ? 0 : currentItems->first();
    }
    
    KURL url = KURL(getCurrentFolder());
  
    if(count == 0){  
      const KFileItem *item = new KFileItem(KFileItem::Unknown, KFileItem::Unknown, url, true);
      KFileItemList list;
      list.append(item);
      
      setFileItems(list);
      return;
    }
    
    if(count == 1){
      bool go = true;
    
      if(items.getFirst()->url().isEmpty()){
        go = false;
      }
      else if(currentItem && items.getFirst()->url() == currentItem->url())
      {
        go = false;
      }
        
      if(go){
        if(currentItem){
          if(dir_watch->contains(currentItem->url().path())){
            if(currentItem->isDir()){
              dir_watch->removeDir(currentItem->url().path());
            }
            else{
              dir_watch->removeFile(currentItem->url().path());
            }
          }
        }
        
        if(currentItems){
          delete currentItems;
        }
        currentItems = new KFileItemList(items);
        
        KFileItem *newItem = currentItems->getFirst();
        
        if(newItem->url().path() == "blank"){
          currentItems->remove(newItem);
          newItem = new KFileItem(KFileItem::Unknown, KFileItem::Unknown, url, true);
          currentItems->insert(0, newItem);
        }
        
        if(newItem->isDir()){
          dir_watch->addDir(newItem->url().path());
        }
        else{
          dir_watch->addFile(newItem->url().path());
        }
        
        updateSingleInfo(*newItem);
      }
    }
    else{
      currentItems = new KFileItemList(items);
      updateMultiInfo(items);
    }
  }
}

void MetabarWidget::updateSingleInfo(const KFileItem &item){
  setUpdatesEnabled(FALSE);
  
  //delete old data
  openList.clear();
  infoDict.clear();
  infoList.clear();
  runlist.clear();
  
  deletePlayer();
  
  preview_frame->hide();
  
  show_more = 0;
  show_more_actions = 0;
  
  //get new info  
  const KURL url = item.url();
  QString name = item.name();
  name = name.isEmpty() ? i18n("Root Folder") : name;
  
  name_label->setText(name);

  KFileItem *that = const_cast<KFileItem *>(&item);
  type_label->setText(that->determineMimeType()->comment());
  
  icon->setPixmap(item.pixmap(0, KIcon::DefaultState));
  
  bool isDir = item.isDir() || item.name().isEmpty();
  
  if(isDir){
    open_frame->hide();
  
    if(linkList.count() > 0){
      link_frame->show();
    }
    else if(link_frame->isVisible()){
      link_frame->hide();
    }
    
    loadActionList();
  }
  else{ //is file
    link_frame->hide();
        
    if(KDesktopFile::isDesktopFile(item.url().path())){
      KDesktopFile desktop(item.url().path(), TRUE);
      
      actionList.clear();
      
      open_frame->hide();
      
      if(desktop.hasApplicationType ()){
        LinkWidget *url_box = new LinkWidget(action_frame->mainWidget(), "action");
        url_box->setText(i18n("Run %1").arg(desktop.readName()));
        url_box->setURL(url.path());
        url_box->setIcon("run");
        
        connect(url_box, SIGNAL(leftClickedURL(const QString&)), this, SLOT(slotRunDesktopFile(const QString&)));
        
        url_box->show();
        
        action_frame->addWidget(url_box);
        actionList.append(url_box);
      }
      
      else if(desktop.hasDeviceType ()){
      }
    }
    
    else{ // not a desktop file    
      KTrader::OfferList offers;
      offers = KTrader::self()->query(item.mimetype(), "Type == 'Application'");
      if(!offers.isEmpty()){
        open_frame->show();
      
        int id = 0;
        
        KTrader::OfferList::ConstIterator it = offers.begin();
        config->setGroup("General");
        
        for(; it != offers.end(); it++){
          QCString nam;
          nam.setNum(id);        
          LinkWidget *url_box = new LinkWidget(open_frame->mainWidget(), nam.prepend("action_"));
          url_box->setText((*it)->name());
          url_box->setURL(url.url());
          url_box->setIcon((*it)->icon());
          
          connect(url_box, SIGNAL(leftClickedURL()), this, SLOT(slotRun()));  
          
          if(id < config->readNumEntry("MaxEntries", 3)){
            url_box->show();
          }
          else{
            url_box->hide();
          }
          
          runlist[id++] = *it;
          
          open_frame->addWidget(url_box);
          openList.append(url_box);
        }
        
        if(id > config->readNumEntry("MaxEntries", 3)){
          show_more = new KURLLabel(open_frame->mainWidget());
          show_more->setURL("more");
          show_more->setPixmap(SmallIcon("1downarrow"));
          show_more->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);
          show_more->setAlignment(Qt::AlignRight);
          show_more->show();
          
          connect(show_more, SIGNAL(leftClickedURL()), this, SLOT(slotShowMore()));          
          
          open_frame->addWidget(show_more);
          openList.append(show_more);
        }
      }
      else{
        open_frame->hide();
      }
      
      loadActionList();
    }
  }
  
  QString info("<table></td></tr><tr><td>" + i18n("Size") + ":</td><td>");
  info = info.append(KIO::convertSize(item.size()));
  info = info.append("</td></tr><tr><td>" + i18n("User") + ":</td><td>");
  info = info.append(item.user());
  info = info.append("</td></tr><tr><td>" + i18n("Group") + ":</td><td>");
  info = info.append(item.group());
  info = info.append("</td></tr><tr><td>" + i18n("Permissions:") + "</td><td>");
  info = info.append(item.permissionsString());
  info = info.append("</td></tr><tr><td>" + i18n("Last modified") + ":</td><td>");
  info = info.append(item.timeString(KIO::UDS_MODIFICATION_TIME));
  info = info.append("</td></tr><tr><td>" + i18n("Accessed") + ":</td><td>");
  info = info.append(item.timeString(KIO::UDS_ACCESS_TIME));
  
  if(item.isLink()){
    info = info.append("<tr><td>" + i18n("Linktarget") + ":</td><td>");
    info = info.append(item.linkDest());
    info = info.append("</td></tr>");
  }
  
  info = info.append("</table>");
  info_table->setText(info);
  
  if(!isDir){
    const KFileMetaInfo &metaInfo = item.metaInfo();
    if(metaInfo.isValid()){
      QStringList groups = metaInfo.supportedGroups();
      
      for(QStringList::ConstIterator it = groups.begin(); it != groups.end(); ++it){
        KFileMetaInfoGroup group = metaInfo.group(*it);
        if(group.isValid()){
          KURLLabel *title_label = new KURLLabel("exec://" + group.name(), group.translatedName(), info_frame->mainWidget());
          title_label->setUnderline(FALSE);
          title_label->setFloat(TRUE);
          title_label->setHighlightedColor(KGlobalSettings::textColor());
          title_label->setSelectedColor(KGlobalSettings::textColor());
          connect(title_label, SIGNAL(leftClickedURL()), this, SLOT(slotPopupInfo()));
          
          QLabel *info_label = new QLabel(info_frame->mainWidget());
          
          infoDict.insert(group.translatedName(), info_label);
          infoList.append(title_label);
          
          info_frame->addWidget(title_label, false);
          info_frame->addWidget(info_label, false);
          
          QStringList keys = group.supportedKeys();
          
          QString str("<table>");
        
          for(QStringList::ConstIterator it = keys.begin(); it != keys.end(); ++it){
            const KFileMetaInfoItem metaInfoItem = group.item(*it);
            
            if(metaInfoItem.isValid()){
              str = str.append("<tr><td>" + metaInfoItem.translatedKey() + ":</td><td>");
              str = str.append(metaInfoItem.string());
              str = str.append("</td></tr>");
            }
          }
          str = str.append("</table>");
          info_label->setText(str);
          QToolTip::add(info_label, str);
          
          title_label->show();
        }
      }
    }
    
    if(preview_frame->isExpanded()){
      killJobs();
      
      preview->clear();
      preview->setText(i18n("Creating preview"));
      
      preview_stack->raiseWidget(preview);
      preview_stack->show();
      
      if(item.mimetype().startsWith("video/") || item.mimetype().startsWith("audio/")){
        player_status->setValue(0);
        player_frame->show();
        
        playobj = factory->createPlayObject(item.url(), true);
  
        if (playobj && !playobj->object().isNull()){
          if(item.mimetype().startsWith("video/")){
            isVideo = TRUE;
            video->embed((Arts::VideoPlayObject)Arts::DynamicCast(playobj->object()));
          }
          else{
            isVideo = FALSE;
          }
        }
        
        if(item.mimetype().startsWith("audio/")){
          preview_stack->hide();
          preview_frame->mainWidget()->adjustSize();
          preview_frame->show();
        }
      }
      else{
        player_frame->hide();
      }
  
      preview_job = KIO::filePreview(KURL::List(item.url()), top_frame->width(), 0, 0, 0, TRUE, TRUE, 0);
      
      connect(preview_job, SIGNAL(gotPreview(const KFileItem*, const QPixmap&)), this, SLOT(slotSetPreview(const KFileItem*, const QPixmap&)));
      connect(preview_job, SIGNAL(result(KIO::Job*)), this, SLOT(slotGotPreview()));
    }
    else{
      preview_frame->mainWidget()->adjustSize();
      preview_frame->show();
    }
  }
  
  setUpdatesEnabled(TRUE);
  
  info_frame->mainWidget()->adjustSize();
  adjustFrameSizes();
}

void MetabarWidget::updateMultiInfo(const KFileItemList &items){
  setUpdatesEnabled(FALSE);
  
  //delete old data
  openList.clear();
  infoDict.clear();
  infoList.clear();
  runlist.clear();
  
  deletePlayer();
  
  show_more = 0;
  show_more_actions = 0;
  
  //get new info
  open_frame->hide();
  
  if(linkList.count() > 0){
    link_frame->show();
  }
  else if(link_frame->isVisible()){
    link_frame->hide();
  }
  
  KIO::filesize_t size = 0;
  int files = 0;
  int dirs  = 0;
  
  for(KFileItemListIterator it(items); it.current(); ++it){
    size += (*it)->size();
    
    if((*it)->isDir()){
      dirs++;
    }
    else{
      files++;
    }
  }
  
  QString name = QString().setNum(items.count()) + " " + i18n("Items");  
  name_label->setText(name);
  type_label->setText(QString().setNum(files) + " " + i18n("Files") + ", " + QString().setNum(dirs) + " " + i18n("Folders"));
  icon->setPixmap(DesktopIcon("kmultiple"));
  
  loadActionList();
  
  QString info("<table><tr><td>" + i18n("Size:") + "</td><td>");
  info = info.append(KIO::convertSize(size));
  info = info.append("</td></tr></table>");
  
  info_table->setText(info);
  
  killJobs();
  preview->clear();
  preview_stack->hide();
  preview_frame->hide();
  
  setUpdatesEnabled(TRUE);
  
  info_frame->mainWidget()->adjustSize();
  adjustFrameSizes();
}

void MetabarWidget::slotUpdateCurrentInfo(const QString &path)
{
  if(currentItems && currentItems->count() == 1){
    const KFileItem item(KFileItem::Unknown, KFileItem::Unknown, KURL(path), true);
    updateSingleInfo(item);
  }
}

void MetabarWidget::slotDeleteCurrentInfo(const QString&)
{
  if(currentItems && currentItems->count() == 1){
    QString url = getCurrentFolder();
    KURL currentURL;
     
    if(currentItems){
      currentURL = currentItems->first()->url();
    }
    
    if(!currentURL.isEmpty() && KURL(url) != currentURL){
      if(dir_watch->contains(currentURL.path())){
        dir_watch->removeDir(currentURL.path());
      }
      dir_watch->addDir(url);
          
      KFileItem *item = new KFileItem(KFileItem::Unknown, KFileItem::Unknown, url, true);
      
      if(currentItems){
        delete currentItems;
      }
      currentItems = new KFileItemList;
      currentItems->append(item);
        
      updateSingleInfo(*item);
    }
  }
}

void MetabarWidget::slotGotPreview()
{
  preview_job = 0;
  
  if(preview_frame->isExpanded()){
    if (preview->pixmap()){
      preview_stack->raiseWidget(preview);
      preview_stack->show();
    }
    else{
      preview->clear();
      preview_stack->hide();
    }
  }
}

void MetabarWidget::slotSetPreview(const KFileItem*, const QPixmap& icon)
{
  preview_stack->resize(icon.size());
  preview_stack->setMaximumHeight(icon.height());
  
  preview->setPixmap(icon);
  
  video->setFixedSize(icon.size());
  
  preview_frame->mainWidget()->adjustSize();
  preview_frame->show();
  preview_frame->resize();
}

void MetabarWidget::killJobs()
{
  if(preview_job){
    preview_job->kill();
    preview_job = 0;
  }
}

void MetabarWidget::slotCreatePreview(bool toggled)
{
  if(!toggled){
    killJobs();
    
    if(currentItems && currentItems->count() > 0){
      KFileItem *item = currentItems->getFirst();
    
      if(!playobj){
        if(item->mimetype().startsWith("video/") || item->mimetype().startsWith("audio/")){
          player_frame->show();
          
          playobj = factory->createPlayObject(item->url(), true);
    
          if (playobj && !playobj->object().isNull()){
            if(item->mimetype().startsWith("video/")){
              isVideo = TRUE;
              video->embed((Arts::VideoPlayObject)Arts::DynamicCast(playobj->object()));
            }
            else{
              isVideo = FALSE;
            }
          }
        }
        else{
          player_frame->hide();
        }
      }
          
      preview->clear();
      preview->setText(i18n("Creating preview"));
        
      preview_stack->raiseWidget(preview);
      preview_stack->show();
    
      preview_job = KIO::filePreview(KURL::List(item->url()), preview_stack->width(), 0, 0, 0, TRUE, FALSE, 0);
      
      connect(preview_job, SIGNAL(gotPreview(const KFileItem*, const QPixmap&)), this, SLOT(slotSetPreview(const KFileItem*, const QPixmap&)));
      connect(preview_job, SIGNAL(result(KIO::Job*)),this, SLOT(slotGotPreview()));
    }
  }
}

void MetabarWidget::slotRun()
{
  const QObject *sender = this->sender();
  if(sender->inherits("LinkWidget")){
    const LinkWidget *label = static_cast<const LinkWidget*>(sender);
    
    QCString sender_name = sender->name();
    int id = sender_name.mid(sender_name.find("_") + 1).toInt();
      
    QMap<int,KService::Ptr>::Iterator it = runlist.find(id);
    if(it != runlist.end()){
      KRun::run( **it, KURL::List(KURL(label->url())));
      return; 
    }
  }
}

void MetabarWidget::slotRunDesktopFile(const QString &url)
{
  KURL::List list;
  KRun::run(new KDesktopFile(url), list);
}

void MetabarWidget::slotShowMore()
{
  if(show_more){
    bool show = show_more->url() == "more";
    
    if(show){
      show_more->setPixmap(SmallIcon("1uparrow"));
      show_more->setURL("less");
    }
    else{
      show_more->setPixmap(SmallIcon("1downarrow"));
      show_more->setURL("more");
    }
  
    config->setGroup("General");
    for(uint i = config->readNumEntry("MaxEntries", 3); i < openList.count() - 1; i++){
      if(show){
        openList.at(i)->show();
      }
      else{
        openList.at(i)->hide();
      }
    }
    
    open_frame->mainWidget()->adjustSize();
    open_frame->resize();
  }
}

void MetabarWidget::slotShowMoreActions()
{
  if(show_more_actions){
    bool show = show_more_actions->url() == "more";
    
    if(show){
      show_more_actions->setPixmap(SmallIcon("1uparrow"));
      show_more_actions->setURL("less");
    }
    else{
      show_more_actions->setPixmap(SmallIcon("1downarrow"));
      show_more_actions->setURL("more");
    }
  
    config->setGroup("General");
    for(uint i = config->readNumEntry("MaxActions", 3); i < actionList.count() - 1; i++){
      if(show){
        actionList.at(i)->show();
      }
      else{
        actionList.at(i)->hide();
      }
    }
    
    action_frame->mainWidget()->adjustSize();
    action_frame->resize();
  }
}

void MetabarWidget::slotShowPropertiesDialog()
{
  if(currentItems){
    new KPropertiesDialog(*currentItems);
  }
}

void MetabarWidget::slotShowSharingDialog()
{
  if(currentItems && currentItems->count() == 1){
    if(currentItems && currentItems->count() > 0){
      KPropertiesDialog *dialog = new KPropertiesDialog(currentItems->first());
      dialog->showFileSharingPage();
    }
  }
}

void MetabarWidget::slotShowConfigDialog()
{
  config_dialog = new ConfigDialog(this->topLevelWidget()->name(), this);
  if(config_dialog->exec() == QDialog::Accepted){
    config->reparseConfiguration();
    
    loadLinkList();
    loadActionList();
  
    if(currentItems && currentItems->count() == 1){
      slotUpdateCurrentInfo(currentItems->getFirst()->url().path());
    }
  }
  
  delete config_dialog;
}

void MetabarWidget::slotPopupInfo()
{
  const KURLLabel *title = static_cast<const KURLLabel*>(sender());
  QLabel *label = infoDict[title->text()];
  
  if(label){
    if(label->isVisible()){
      label->hide();
    }
    else{
      label->setMaximumWidth(info_frame->width() - 10);
      label->show();
    }
  }
  
  info_frame->mainWidget()->adjustSize();
  info_frame->resize();
}

void MetabarWidget::startPlayer()
{  
  if(playobj &&
     !playobj->object().isNull() &&
     playobj->state() != Arts::posPlaying)
  {
    
    setUpdatesEnabled(FALSE);
    
    if(isVideo){
      video->setFixedSize(preview->size());
      preview_stack->setMaximumHeight(preview->height());
      preview_stack->raiseWidget(video);
    }
    
    if(playobj->state() == Arts::posIdle){
      player_status->setValue(0);
      player_status->setMaxValue(playobj->overallTime().seconds);
    }
    
    player_timer->start(1000);
    playobj->play();
    
    setUpdatesEnabled(TRUE);
  }
}

void MetabarWidget::pausePlayer()
{  
  if(playobj &&
     !playobj->object().isNull() &&
     playobj->state() == Arts::posPlaying)
  {
    
    if(player_timer->isActive()){
      player_timer->stop();
    }
    
    playobj->pause();
  }
}

void MetabarWidget::stopPlayer()
{
  if(player_timer->isActive()){
    player_timer->stop();
  }
  player_status->setValue(0);
  
  if(playobj && 
    !playobj->object().isNull() &&
     playobj->state() != Arts::posIdle)
  {
    
    setUpdatesEnabled(FALSE);
    
    playobj->halt();
    
    video->setFixedSize(0, 0);
  
    if(isVideo){
      preview_stack->raiseWidget(preview);
    }
      
    setUpdatesEnabled(TRUE);
  }
}

void MetabarWidget::deletePlayer()
{  
  stopPlayer();
  
  playobj = 0;
  
  if(preview_stack->isVisible()){
    preview_stack->hide();
  }
}

void MetabarWidget::updatePlayerStatus()
{
  if(playobj && !playobj->object().isNull()){
    player_status->blockSignals(TRUE);
    player_status->setValue(playobj->currentTime().seconds);
    player_status->blockSignals(FALSE);
    
    if(playobj->currentTime().ms == playobj->overallTime().ms){
      player_timer->stop();
      stopPlayer();
    }
  }
}

void MetabarWidget::seekPlayer(int time)
{
  if(playobj && !playobj->object().isNull()){
    playobj->seek(Arts::poTime(time, 0, 0, ""));
  }
}

QString MetabarWidget::getCurrentFolder()
{
  DCOPRef ref(kapp->dcopClient()->appId(), this->topLevelWidget()->name());
  DCOPReply reply = ref.call("currentURL()");
    
  if (reply.isValid()) {
    QString url;
    reply.get(url, "QString");
      
    if(!url.isNull() && !url.isEmpty()){
      return url;
    }
  }
  return 0;
}

void MetabarWidget::openURL(const QString &url)
{
  DCOPRef ref(kapp->dcopClient()->appId(), this->topLevelWidget()->name());
  DCOPReply reply = ref.call("openURL", url);
}

void MetabarWidget::openTab(const QString &url)
{
  DCOPRef ref(kapp->dcopClient()->appId(), this->topLevelWidget()->name());
  DCOPReply reply = ref.call("newTab", url);
}

void MetabarWidget::callAction(const QString &action)
{
  DCOPRef ref(kapp->dcopClient()->appId(), QString(this->topLevelWidget()->name()).append("/action/").append(action).utf8());
  if(ref.call("enabled()")){
    ref.call("activate()");
  }
}

void MetabarWidget::showEvent(QShowEvent*)
{  
  KFileItemList list;
  setFileItems(list);
}

void MetabarWidget::resizeEvent(QResizeEvent*)
{
  QDictIterator<QLabel> it(infoDict);
  for(; it.current(); ++it){
    it.current()->setMaximumWidth(info_frame->width() - 10);
  }
}
  
void MetabarWidget::loadConfig()
{  
  config->setGroup("General");
  open_frame->setExpanded(!config->readBoolEntry("CollapseOpenWith"));
  action_frame->setExpanded(!config->readBoolEntry("CollapseActions"));
  link_frame->setExpanded(!config->readBoolEntry("CollapseInfo"));
  preview_frame->setExpanded(!config->readBoolEntry("CollapsePreview"));
  link_frame->setExpanded(!config->readBoolEntry("CollapseLinks"));
}

void MetabarWidget::loadLinkList(){
  linkList.clear();
  
  config->setGroup("General");
  QStringList actions = config->readListEntry("Links");
  
  for(QStringList::Iterator it = actions.begin(); it != actions.end(); ++it){
    config->setGroup("Link_" + (*it));
    
    QString icon = config->readEntry("Icon", "folder");
    
    LinkWidget *link = new LinkWidget(link_frame->mainWidget());
    link->setText(config->readEntry("Name"));
    link->setURL(config->readEntry("URL"));
    link->setIcon(icon);
        
    connect(link, SIGNAL(leftClickedURL(const QString&)), this, SLOT(openURL(const QString&)));
    connect(link, SIGNAL(middleClickedURL(const QString&)), this, SLOT(openTab(const QString&)));
    
    if(link_frame->isExpanded()){
      link->show();
    }
    
    link_frame->addWidget(link);
    linkList.append(link);
  }
}

void MetabarWidget::loadActionList()
{
  actionList.clear();
  
  config->setGroup("General");
  iconConfig->setGroup("Icons");
  
  QStringList actions = config->readListEntry("Actions");
  
  for(QStringList::Iterator it = actions.begin(); it != actions.end(); ++it){
    if((*it).startsWith("metabar:")){
      if((*it).right((*it).length() - 8) == "share"){      
        LinkWidget *box = new LinkWidget(action_frame->mainWidget());
        box->setText(i18n("Share"));
        box->setURL(*it);
        box->setIcon("network");
        
        connect(box, SIGNAL(leftClickedURL()), this, SLOT(slotShowSharingDialog()));   
        
        if(actionList.count() < (uint)config->readNumEntry("MaxActions", 3)){
          box->show();
        }
        else{
          box->hide();
        }

        action_frame->addWidget(box);
        actionList.append(box);
      }
    }
    else{
      DCOPRef action(kapp->dcopClient()->appId(), QCString(this->topLevelWidget()->name()).append("/action/").append(*it));

      if(!action.isNull()){
        QString text = action.call("plainText()");
        QString icon = iconConfig->readEntry(*it, action.call("icon()"));
        
        LinkWidget *box = new LinkWidget(action_frame->mainWidget());
        box->setText(text);
        box->setURL(*it);
        box->setIcon(icon);
        
        connect(box, SIGNAL(leftClickedURL(const QString&)), this, SLOT(callAction(const QString&)));   
        
        if(actionList.count() < (uint)config->readNumEntry("MaxActions", 3)){
          box->show();
        }
        else{
          box->hide();
        }
      
        action_frame->addWidget(box);
        actionList.append(box);
      }
    }
  }
  
  config->setGroup("General");
  if(config->readBoolEntry("ShowServicemenus", true)){
    if(currentItems && currentItems->count() == 1){
      serviceLoader->loadServices(*(currentItems->first()), actionList);
    }
  }
  
  if(((int)actionList.count()) > config->readNumEntry("MaxActions", 3)){
    show_more_actions = new KURLLabel(action_frame->mainWidget());
    show_more_actions->setURL("more");
    show_more_actions->setPixmap(SmallIcon("1downarrow"));
    show_more_actions->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);
    show_more_actions->setAlignment(Qt::AlignRight);
    
    connect(show_more_actions, SIGNAL(leftClickedURL()), this, SLOT(slotShowMoreActions()));
    
    show_more_actions->show();
    
    action_frame->addWidget(show_more_actions);
    actionList.append(show_more_actions);
  }
  
  if(actionList.count() == 0){
    action_frame->hide();
  }
  else if(!action_frame->isVisible()){
    action_frame->show();
  }
}

void MetabarWidget::adjustFrameSizes()
{
  action_frame->resize();
  open_frame->resize();
  info_frame->resize();
  preview_frame->resize();
  link_frame->resize();
}

#include "metabar_widget.moc"
