/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <messaging/qt/tp/text_channel.h>

#include <messaging/qt/tp/connection.h>
#include <messaging/qt/tp/message.h>
#include <messaging/qt/tp/channel_group_change_reason.h>

#include <messaging/qt/variant.h>

#include <messaging/broadcast.h>
#include <messaging/connection.h>
#include <messaging/group.h>
#include <messaging/messenger.h>
#include <messaging/message.h>
#include <messaging/user.h>

#include <TelepathyQt/Constants>

#include <boost/uuid/uuid.hpp>
#include <boost/uuid/uuid_generators.hpp>
#include <boost/uuid/uuid_io.hpp>

#include <boost/lexical_cast.hpp>

#include <QTemporaryFile>

#include <iostream>
#include <fstream>
#include <cstdint>

#include <glog/logging.h>

namespace mqt = messaging::qt;

namespace
{
namespace supported
{
// The default content_types we support for now.
QStringList content_types()
{
    return QStringList() << "text/plain";
}

Tp::UIntList message_types()
{
    auto types = Tp::UIntList() << Tp::ChannelTextMessageTypeNormal << Tp::ChannelTextMessageTypeDeliveryReport;
    return types;
}

uint message_parts()
{
    return 0;
}

uint delivery_reports()
{
    // TODO(tvoss): Make this queryable from the Chat instance.
    return Tp::DeliveryReportingSupportFlagReceiveSuccesses | Tp::DeliveryReportingSupportFlagReceiveFailures;
}
}

QString guess_content_type(const QByteArray &content)
{
    // try to guess content type
    QTemporaryFile file;
    if (file.open()) {
        file.write(content);
        QMimeDatabase db;
        QMimeType type = db.mimeTypeForFile(file.fileName());
        if (type.isValid()) {
            return type.name();
        }
    }
    return QString("application/octet-stream");
}

QByteArray read_file_content(const std::string& filename)
{
    try
    {
        std::ifstream fin(filename, std::ifstream::binary);
        std::ostringstream ostrm;
        ostrm << fin.rdbuf();
        std::string data(ostrm.str());
        return QByteArray(data.data(), data.size());
    }
    catch (...)
    {
        LOG(ERROR) << "An exception has been thrown when reading file " << filename;
        return QByteArray();
    }
}

messaging::Message make_message(const Tp::MessagePartList &message)
{
    using namespace messaging::qt::tp;
    messaging::Message mf_message{boost::uuids::to_string(boost::uuids::random_generator()()),
                                  std::string(), // sender can be empty on outgoing messages
                                  std::string(), // actual message will be filled below
                                  std::chrono::system_clock::now(),
                                  std::vector<messaging::Attachment>()};
    for (auto part : message)
    {
        messaging::Attachment attachment;
        if (!part.contains(message::keys::content))
        {
            continue;
        }
        QString content_type = part[message::keys::content_type].variant().toString();
        attachment.content_type = content_type.toStdString();
        if (content_type.startsWith("text/plain")) {
            mf_message.message = part[message::keys::content].variant().toString().toStdString();
            continue;
        }
        attachment.content = std::make_shared<messaging::qt::Variant>(part[message::keys::content].variant());
        attachment.id = part[message::keys::identifier].variant().toString().toStdString();
        mf_message.attachments.push_back(attachment);
    }

    return mf_message;
}
}

mqt::tp::TextChannel::Observer::Observer(const std::shared_ptr<qt::Runtime>& runtime, const mqt::tp::TextChannel::Ptr& tc)
        : runtime{runtime}
        , tc{tc}
{
}

void mqt::tp::TextChannel::Observer::on_message_delivery_report(const std::string& id, DeliveryStatus status)
{
    auto thiz = shared_from_this();
    runtime->enter_with_task([thiz, id, status] {
            thiz->tc->on_message_delivery_report(id, status);
        });
}

void mqt::tp::TextChannel::Observer::on_message_received(const messaging::Message& message)
{
    auto thiz = shared_from_this();
    runtime->enter_with_task([thiz, message]() {
            thiz->tc->on_message_received(message);
        });
}

void mqt::tp::TextChannel::Observer::on_message_id_changed(const std::string &old_id, const std::string &new_id)
{
    auto thiz = shared_from_this();
    runtime->enter_with_task([thiz, old_id, new_id]() {
            thiz->tc->on_message_id_changed(old_id, new_id);
        });
}

void mqt::tp::TextChannel::Observer::on_group_created()
{
    auto sp = shared_from_this();
    runtime->enter_with_task([sp](){
            sp->tc->on_group_created();
        });
}

void mqt::tp::TextChannel::Observer::on_group_cancelled(CancelGroupReason reason, const std::string &message)
{
    auto thiz = shared_from_this();
    runtime->enter_with_task([thiz, reason, message]() {
            thiz->tc->on_group_cancelled(reason, message);
        });
}

void mqt::tp::TextChannel::Observer::on_group_title_changed(const std::string &new_title)
{
    auto thiz = shared_from_this();
    runtime->enter_with_task([thiz, new_title]() {
            thiz->tc->on_group_title_changed(new_title);
        });
}

void mqt::tp::TextChannel::Observer::on_group_subject_changed(const std::string &new_subject,
                                                              const std::shared_ptr<Member>& actor,
                                                              const std::chrono::system_clock::time_point& when)
{
    auto thiz = shared_from_this();
    runtime->enter_with_task([thiz, new_subject, actor, when]() {
            thiz->tc->on_group_subject_changed(new_subject, actor, when);
        });
}

void mqt::tp::TextChannel::Observer::on_members_updated(const messaging::Members& users)
{
    auto thiz = shared_from_this();
    runtime->enter_with_task([thiz, users]() {
            thiz->tc->on_members_updated(users);
        });
}

void mqt::tp::TextChannel::Observer::on_group_permissions_changed(const Flags<GroupPermissions> &permissions)
{
    auto thiz = shared_from_this();
    runtime->enter_with_task([thiz, permissions]() {
            thiz->tc->on_group_permissions_changed(permissions);
        });
}

void mqt::tp::TextChannel::Observer::on_file_receiving(const std::string& id, uint received_size, uint total_size)
{
    // TODO: notify telepathy about the progress of the file being downloaded
    auto thiz = shared_from_this();
    runtime->enter_with_task([thiz, id, received_size, total_size]() {
            thiz->tc->on_file_receiving(id, received_size, total_size);
        });
}

void mqt::tp::TextChannel::Observer::on_file_received(const std::string& id, uint received_size, uint total_size)
{
    // TODO: notify telepathy about the received file
    auto thiz = shared_from_this();
    runtime->enter_with_task([thiz, id, received_size, total_size]() {
            thiz->tc->on_file_received(id, received_size, total_size);
        });
}

void mqt::tp::TextChannel::Observer::on_file_sending(const std::string& id, uint sent_size, uint total_size)
{
    // TODO: notify telepathy about the progress of the file being sent
    Q_UNUSED(id)
    Q_UNUSED(sent_size)
    Q_UNUSED(total_size)
}
 
void mqt::tp::TextChannel::Observer::on_participant_starts_typing(const std::shared_ptr<User> &user)
{
    auto thiz = shared_from_this();
    runtime->enter_with_task([thiz, user](){
            thiz->tc->on_participant_starts_typing(user);
        });
}

void mqt::tp::TextChannel::Observer::on_participant_ends_typing(const std::shared_ptr<User> &user)
{
    auto thiz = shared_from_this();
    runtime->enter_with_task([thiz, user](){
            thiz->tc->on_participant_ends_typing(user);
        });
}

mqt::tp::TextChannel::Ptr mqt::tp::TextChannel::create(mqt::tp::Connection* tp_connection,
                                                       uint th,
                                                       uint tht,
                                                       const std::shared_ptr<qt::Runtime>& runtime,
                                                       const std::shared_ptr<messaging::Messenger>& messenger,
                                                       const messaging::Recipient::shared_ptr& recipient,
                                                       const std::shared_ptr<messaging::GroupManager>& group_manager)
{
    return mqt::tp::TextChannel::Ptr{new mqt::tp::TextChannel{tp_connection, th, tht, runtime, messenger, recipient, group_manager}};
}

mqt::tp::TextChannel::TextChannel(mqt::tp::Connection* tpc,
                                  uint th,
                                  uint tht,
                                  const std::shared_ptr<qt::Runtime>& runtime,
                                  const std::shared_ptr<messaging::Messenger>& messenger,
                                  const messaging::Recipient::shared_ptr& recipient,
                                  const std::shared_ptr<messaging::GroupManager>& group_manager)
    : Tp::BaseChannel{tpc->dbusConnection(), tpc, TP_QT_IFACE_CHANNEL_TYPE_TEXT, tht, th}
    , text_type_interface{Tp::BaseChannelTextType::create(this)}
    , messages_interface{Tp::BaseChannelMessagesInterface::create(text_type_interface.data(),
                                                    supported::content_types(),
                                                    supported::message_types(),
                                                    supported::message_parts(),
                                                    supported::delivery_reports())}
    , tp_connection{tpc}
    , observer{std::make_shared<TextChannel::Observer>(runtime, TextChannel::Ptr{this})}
    , chat{messenger->initiate_chat_with(recipient, observer)}
{
    //register group observer if chat is a group chat
    if (recipient->type() == RecipientType::group)
    {
        if (not group_manager)
        {
            LOG(INFO) << "Group manager api object is empty when plugin it as interface for the chat";
            return;
        }

        // is this needed, or should we use same observer than the used for the chat when plugged as chat interface?
        try
        {
            group_manager->set_observer(observer);
        }
        catch (...)
        {
            LOG(ERROR) << "An exception has been thrown when setting group manager observer";
            return;
        }

        try
        {
            chat->plug_interface(group_manager);
        }
        catch (...)
        {
            LOG(ERROR) << "An exception has been thrown when pluging in group manager as chat interface";
            return;
        }

        // initialize the telepathy room interface
        messaging::Group::shared_ptr group = std::dynamic_pointer_cast<messaging::Group>(recipient);
        // FIXME: we need to expose the extra information (like server and creator) in the group class.

        Tp::DBusError error;
        uint creator_handle = tp_connection->requestHandles(Tp::HandleTypeContact,
                                                            QStringList() << QString::fromStdString(group->creator()->id()),
                                                            &error)[0];
        if (error.isValid())
        {
            LOG(ERROR) << "Could not get creator handle";
            return;
        }

        // NOTE (rmescandon): Build and populate values for Subject interface before than roomConfig one. This is needed for having
        // available subject Actor before setting the title in room config. That way, same value could be taken as actor for subject
        // and title in upper layers
        subject_interface = interfaces::BaseChannelSubjectInterface::create();
        subject_interface->setActor(QString::fromStdString(group->creator()->id()));
        subject_interface->setActorHandle(creator_handle);
        subject_interface->setTimestamp(static_cast<qlonglong>(std::chrono::system_clock::to_time_t(std::chrono::system_clock::now())));

        try
        {
            subject_interface->setCanSet(group_manager->permissions().is_set(GroupPermissions::CanChangeSubject));
            // the last thing to set is the subject itself. This way, all the properties should be available in client
            // for composing all the information like "%Actor% set subject to %Subject%"
            subject_interface->setSubject(QString::fromStdString(group_manager->group_subject()));
        }
        catch (...)
        {
            LOG(ERROR) << "An exception has been thrown when setting group title as channel subject";
            return;
        }

        // uncomment the following code when room name is part of messaging::Group
        //QString roomName = QString::fromStdString(group->id());
        room_interface = Tp::BaseChannelRoomInterface::create(/* roomName */ QString(),
                                                    /* serverName */ QString(),
                                                    /* creator */ QString::fromStdString(group->creator()->id()),
                                                    /* creatorHandle */ creator_handle,
                                                    /* creationTimestamp */ QDateTime());
        room_config_interface = Tp::BaseChannelRoomConfigInterface::create();

        try
        {
            room_config_interface->setTitle(QString::fromStdString(group_manager->group_title()));
        }
        catch (...)
        {
            LOG(ERROR) << "An exception has been thrown when setting group title";
            return;
        }

        room_config_interface->setConfigurationRetrieved(true);

        // As telepathy does not offer a way to differentiate whether it can be changed one configuration element
        // or other, let's use the flag to enable/dissable all configuration properties permissions
        room_config_interface->setCanUpdateConfiguration(group_manager->permissions().is_set(GroupPermissions::CanChangeTitle));

        // FIXME: check what flags we want by default
        Tp::ChannelGroupFlags groupFlags = Tp::ChannelGroupFlagCanAdd |
                                           Tp::ChannelGroupFlagCanRemove |
                                           Tp::ChannelGroupFlagHandleOwnersNotAvailable |
                                           Tp::ChannelGroupFlagMembersChangedDetailed |
                                           Tp::ChannelGroupFlagProperties;
        group_interface = Tp::BaseChannelGroupInterface::create();
        group_interface->setGroupFlags(groupFlags);

        roles_interface = interfaces::BaseChannelRolesInterface::create();

        // we need to plug this interface here to avoid a crash
        plug_interface_if_available(group_interface);

        destroyable_interface = mqt::tp::interfaces::BaseChannelDestroyableInterface::create();

        // we need to set the initial group members
        on_members_updated(group->initial_invitees());
    }
    else if (recipient->type() == RecipientType::broadcast)
    {
        messaging::Broadcast::shared_ptr broadcast = std::dynamic_pointer_cast<messaging::Broadcast>(recipient);
        // for broadcast channels we just need to plug the group interface with limited capabilities
        Tp::ChannelGroupFlags groupFlags = Tp::ChannelGroupFlagHandleOwnersNotAvailable |
                                           Tp::ChannelGroupFlagMembersChangedDetailed |
                                           Tp::ChannelGroupFlagProperties;
        group_interface = Tp::BaseChannelGroupInterface::create();
        group_interface->setGroupFlags(groupFlags);

        // we need to plug this interface here to avoid a crash
        plug_interface_if_available(group_interface);

        destroyable_interface = mqt::tp::interfaces::BaseChannelDestroyableInterface::create();

        // we need to set the initial group members
        on_members_updated(broadcast->members());
    }

    chat_state_interface = Tp::BaseChannelChatStateInterface::create();

    plug_interfaces_once();
    register_callbacks_once();

    auto tt = Tp::BaseChannelTextTypePtr::dynamicCast(interface(TP_QT_IFACE_CHANNEL_TYPE_TEXT));
    tt->setMessageAcknowledgedCallback(Tp::memFun(this, &TextChannel::message_acknowledged));
}

void mqt::tp::TextChannel::on_message_delivery_report(const std::string& token, DeliveryStatus status)
{
    Tp::MessagePart header;
    if (targetHandleType() == Tp::HandleTypeContact) {
        header["message-sender"] = QDBusVariant(targetHandle());
    }
    header["delivery-status"] = QDBusVariant(static_cast<Tp::DeliveryStatus>(status));
    header["message-type"] = QDBusVariant(Tp::ChannelTextMessageTypeDeliveryReport);
    header["delivery-token"] = QDBusVariant(QString::fromStdString(token));
    text_type_interface->addReceivedMessage(Tp::MessagePartList() << header);
}

void mqt::tp::TextChannel::on_message_received(const messaging::Message& message)
{
    try
    {
        auto time = std::chrono::system_clock::now();

        Tp::MessagePartList parts;
        Tp::MessagePart header;
        header["message-token"] = QDBusVariant(QString::fromStdString(message.id));
        header["message-received"] = QDBusVariant(static_cast<uint>(std::chrono::system_clock::to_time_t(time)));
        if (targetHandleType() == Tp::HandleTypeContact) {
            header["message-sender"] = QDBusVariant((qulonglong)targetHandle());
        } else {
            Tp::DBusError error;
            uint id = tp_connection->requestHandles(Tp::HandleTypeContact, QStringList() << QString::fromStdString(message.sender), &error)[0];
            header["message-sender"] = QDBusVariant(id);
        }
        header["message-type"] = QDBusVariant(Tp::ChannelTextMessageTypeNormal);

        parts << header;

        bool has_pending_file_transfer = false;
        for (auto attachment : message.attachments) {
            Tp::MessagePart part;
            QString contentType;

            // try to get the content as data and as string as fallback
            QByteArray content;
            try{
                content = QByteArray(attachment.content->as_data(), attachment.content->data_size());
            }catch (const messaging::Variant::TypeMismatch &e) {
                // in case type mismatch, try to get the subject as string
                content = QByteArray(attachment.content->as_string().c_str(), attachment.content->as_string().size());
            }

            if (attachment.content_type.empty()) {
                contentType = guess_content_type(content);
            } else {
                contentType = QString::fromStdString(attachment.content_type);
            }
            part[message::keys::content_type] = QDBusVariant(contentType);
            part[message::keys::content] = QDBusVariant(content);
            part[message::keys::identifier] = QDBusVariant(QString::fromStdString(attachment.id));
            if (attachment.status == messaging::AttachmentStatus::AttachmentPending) {
                has_pending_file_transfer = true;
                part[message::keys::size] = QDBusVariant(QVariant::fromValue(attachment.size));
                part[message::keys::needs_retrieval] = QDBusVariant(true);
                // for now we trigger the download ourselves, but we need to
                // allow clients to do that
                part[message::keys::filename] = QDBusVariant(QString::fromStdString((chat->download_file(attachment.id))));
            }
            parts << part;
        }

        // check if comes text into message.message and add a text plain part if so
        if (!message.message.empty()) {
            Tp::MessagePart text_part;
            text_part[message::keys::content_type] = QDBusVariant("text/plain");
            text_part[message::keys::identifier] = QDBusVariant("text_0.txt");
            text_part[message::keys::content] = QDBusVariant(QString::fromStdString(message.message));
            text_part[message::keys::size] = QDBusVariant(QVariant::fromValue((qulonglong)message.message.length()));
            parts << text_part;
        }

        if (has_pending_file_transfer) {
            pending_messages[message.id] = parts;
        } else {
            text_type_interface->addReceivedMessage(parts);
        }

    }
    catch (const std::runtime_error &e)
    {
        LOG(ERROR) << "An exception has happened when evaluating a received message: " << e.what();
    }
    catch (...)
    {
        LOG(ERROR) << "An exception has happened when evaluating a received message";
    }
}

void mqt::tp::TextChannel::on_message_id_changed(const std::string &old_id, const std::string &new_id)
{
    LOG(INFO) << __PRETTY_FUNCTION__ << " old_id:" << old_id << " new_id:" << new_id;

    Tp::MessagePartList message = sent_messages.value(old_id);

    if (message.isEmpty())
    {
        LOG(ERROR) << "Received new id for a message not found amongst sent ones";
        return;
    }

    for (Tp::MessagePart part : message)
    {
        if (part.contains("message-sent"))
        {
            message[0]["original-message-sent"] = part["message-sent"];
        }
    }

    message[0]["supersedes"] = QDBusVariant(QString::fromStdString(old_id));
    message[0]["message-token"] = QDBusVariant(QString::fromStdString(new_id));

    text_type_interface->addReceivedMessage(message);
    sent_messages.remove(old_id);
}

void mqt::tp::TextChannel::on_file_receiving(const std::string& id, uint received_size, uint total_size)
{
    // FIXME: implement
    Q_UNUSED(id)
    Q_UNUSED(received_size)
    Q_UNUSED(total_size)
}

void mqt::tp::TextChannel::on_file_received(const std::string& id, uint received_size, uint total_size)
{
    Q_UNUSED(received_size)
    Q_UNUSED(total_size)
    for (auto messageId : pending_messages.keys()) {
        auto parts = pending_messages[messageId];
        Tp::MessagePartList newParts;
        bool changed = false;
        int needs_retrieval = 0;
        for (auto part : parts) {
            if (part.contains(message::keys::needs_retrieval)) {
               needs_retrieval++;
            }
            if (part.contains(message::keys::identifier) && part[message::keys::identifier].variant().toString().toStdString() == id) {
                changed = true;
                part[message::keys::content] = QDBusVariant(read_file_content(part[message::keys::filename].variant().toString().toStdString()));
                part.remove(message::keys::filename);
                part.remove(message::keys::needs_retrieval);
            }
            // check if this is the header part and update the received timestamp so the messages
            // are ordered correctly in the UI
            if (part.contains("message-token")) {
                auto time = std::chrono::system_clock::now();
                part["message-received"] = QDBusVariant(static_cast<uint>(std::chrono::system_clock::to_time_t(time)));
            }
            newParts << part;
        }
        if (changed) {
            // check if there's more to come
            if (needs_retrieval > 1) {
                pending_messages[messageId] = newParts;
            } else {
                pending_messages.remove(messageId);
                text_type_interface->addReceivedMessage(newParts);
            }
            return;
        }
    }
}

void mqt::tp::TextChannel::on_file_sending(const std::string& id, uint sent_size, uint total_size)
{
    // FIXME: implement
    Q_UNUSED(id)
    Q_UNUSED(sent_size)
    Q_UNUSED(total_size)
}

void mqt::tp::TextChannel::on_participant_starts_typing(const std::shared_ptr<User> &user)
{
    Tp::DBusError error;
    uint sender_handle = tp_connection->requestHandles(Tp::HandleTypeContact, QStringList() << QString::fromStdString(user->id()), &error)[0];
    if (error.isValid()) {
        return;
    }
    chat_state_interface->chatStateChanged(sender_handle, Tp::ChannelChatStateComposing);
}

void mqt::tp::TextChannel::on_participant_ends_typing(const std::shared_ptr<User> &user)
{
    Tp::DBusError error;
    uint sender_handle = tp_connection->requestHandles(Tp::HandleTypeContact, QStringList() << QString::fromStdString(user->id()), &error)[0];
    if (error.isValid()) {
        return;
    }
    //NOTE (rmescandon): see if the good status after composing is active or Tp::ChannelChatStatePaused instead
    chat_state_interface->chatStateChanged(sender_handle, Tp::ChannelChatStateActive);
}

void mqt::tp::TextChannel::on_group_created()
{
    LOG(INFO) << __PRETTY_FUNCTION__;
    try
    {
        if (not group_interface.isNull())
        {
            group_interface->setSelfHandle(tp_connection->selfHandle());
        }
    }
    catch (...)
    {
        LOG(ERROR) << "An exception has been thrown when setting group self handle";
        return;
    }
}

void mqt::tp::TextChannel::close_channel_with_reason(uint reason, const QString &message)
{
    // At this point, self handle should be removed from the group members, providing one of the reasons
    // described at:
    // https://telepathy.freedesktop.org/spec/Channel_Interface_Group.html#Enum:Channel_Group_Change_Reason
    //
    // This should be done using RemoveMembersWithReason method described at spec:
    // https://telepathy.freedesktop.org/spec/Channel_Interface_Group.html#Method:RemoveMembersWithReason
    //
    // after that, let's close the channel
    Tp::UIntList members = group_interface->members();
    members.removeAll(tp_connection->selfHandle());

    QVariantMap details;
    details["change-reason"] = reason;
    details["message"] = message;

    group_interface->setMembers(members, details);

    // It is needed to delay the close channel action to be sure the events related with previous action of removing
    // self handle from members are sent to clients and processed before the channel expires.
    QTimer::singleShot(2000, this, SLOT(delay_close()));
}

void mqt::tp::TextChannel::delay_close()
{
    close();
}

void mqt::tp::TextChannel::on_group_cancelled(CancelGroupReason cancel_reason, const std::string &cancel_message)
{
    LOG(INFO) << __PRETTY_FUNCTION__;

    uint reason;
    QString message;
    switch (cancel_reason)
    {
    case CancelGroupReason::Error:
        reason = ChannelGroupChangeReasonError;
        message = "Group is closed due to an error";
        break;
    case CancelGroupReason::Rejected:
        reason = ChannelGroupChangeReasonRejected; // We take value 13 as reason for group creation rejection, as currently there is no value
                                                   // in related telepathy enumeration to cover this case
        message = "Group creation rejected";
        break;
    case CancelGroupReason::Kicked:
        reason = ChannelGroupChangeReasonKicked;
        message = "You were expelled from group";
        break;
    case CancelGroupReason::Gone:
        reason = ChannelGroupChangeReasonGone; // We take value 12 as reason for group dissolved, as currently there is no value in telepathy
                                               // that we can use only for that reason
        message = "Group not available anymore";
        break;
    case CancelGroupReason::Leave:
        reason = Tp::ChannelGroupChangeReasonNone;
        message = "Quit group";
    case CancelGroupReason::None:
    default:
        reason = Tp::ChannelGroupChangeReasonNone;
        message = "Cancelled group";
        break;
    }

    // cancel message overwrites default one in case it is not empty
    if (not cancel_message.empty()) {
        message = QString::fromStdString(cancel_message);
    }

    close_channel_with_reason(reason, message);
}

void mqt::tp::TextChannel::on_group_title_changed(const std::string &new_title)
{
    // for a flexible management in client, set as title both room_config title and subject
    if (room_config_interface)
    {
        room_config_interface->setTitle(QString::fromStdString(new_title));
    }
}

void mqt::tp::TextChannel::on_group_subject_changed(const std::string &new_subject,
                                                    const std::shared_ptr<Member>& actor,
                                                    const std::chrono::system_clock::time_point& when)
{
    if (subject_interface)
    {
        if (not actor->id().empty())
        {
            Tp::DBusError error;
            uint handle;
            if (actor->id() == "me") {
                handle = tp_connection->selfHandle();
            } else {
                handle = tp_connection->requestHandles(Tp::HandleTypeContact, QStringList() << QString::fromStdString(actor->id()), &error)[0];
            }
            // set always the actor before the subject, for having it available in client when subject is available
            subject_interface->setActor(QString::fromStdString(actor->id()));
            subject_interface->setActorHandle(handle);
        }

        subject_interface->setTimestamp(static_cast<qlonglong>(std::chrono::system_clock::to_time_t(when)));
        // the last thing to set is the subject itself. This way, all the properties should be available in client
        // for composing all the information like "%Actor% set subject to %Subject%"
        subject_interface->setSubject(QString::fromStdString(new_subject));
    }
}

void mqt::tp::TextChannel::on_members_updated(const Members& received_members)
{
    LOG(INFO) << __PRETTY_FUNCTION__;

    Tp::DBusError error;
    Tp::LocalPendingInfoList local_info_list;
    QStringList member_ids;
    QStringList remote_member_ids;
    interfaces::HandleRolesMap roles_map;

    for (auto member : received_members) {

        try {

            uint handle = tp_connection->requestHandles(Tp::HandleTypeContact, QStringList() << QString::fromStdString(member->id()), &error)[0];

            switch (member->pending_status()) {
            case PendingStatus::None:
                member_ids << QString::fromStdString(member->id());
                LOG(INFO) << "member: " << member->id();
                break;
            case PendingStatus::Remote:
                remote_member_ids << QString::fromStdString(member->id());
                LOG(INFO) << "pending remote: " << member->id();
                break;
            case PendingStatus::Local: {
                Tp::LocalPendingInfo local_info;

                local_info.toBeAdded = handle;
                LOG(INFO) << "pending local: " << member->id();

                VariantMap properties = member->properties();
                std::string actor_id = properties.at("actor")->as_string();
                if (not actor_id.empty()) {
                    local_info.actor = tp_connection->requestHandles(Tp::HandleTypeContact, member_ids, &error)[0];
                }
                std::string reason = properties.at("reason")->as_string();
                if (not reason.empty()) {
                    local_info.reason = boost::lexical_cast<uint>(reason);
                }
                std::string message = properties.at("message")->as_string();
                local_info.message = QString::fromStdString(message);

                local_info_list.append(local_info);
                break;
            }
            }

            // add current member roles to map
            roles_map[handle] = member->roles().underlying_type();
        }
        catch (...) {
            LOG(ERROR) << "Could not update member: " << member->id() << " ,an error has happened";
            continue;
        }
    }

    // if not found admin, add admin role to ourselves
    interfaces::ChannelRoles self_roles(interfaces::ChannelRole::ChannelRoleMember);
    if (self_is_admin()) {
        self_roles |= interfaces::ChannelRole::ChannelRoleAdmin;
    }
    roles_map[tp_connection->selfHandle()] = static_cast<uint>(self_roles);

    Tp::UIntList handles = tp_connection->requestHandles(Tp::HandleTypeContact, member_ids, &error);
    Tp::UIntList remote_handles = tp_connection->requestHandles(Tp::HandleTypeContact, remote_member_ids, &error);

    // selfHandle will be set as member while the channel is active, so add it to received members. In case we are
    // expelled or the group is dissolved, let's remove that selfHandle from members
    handles << tp_connection->selfHandle();

    group_interface->setMembers(handles, local_info_list, remote_handles,  /* details */ QVariantMap());
    if (!roles_interface.isNull())
    {
        roles_interface->setRoles(roles_map);
    }
}

/*!
 * Received the permissions of the group after they have changed, not only the ones changed
 */
void mqt::tp::TextChannel::on_group_permissions_changed(const Flags<GroupPermissions> &permissions)
{
    LOG(INFO) << __PRETTY_FUNCTION__;

    // Can update the title?
    // NOTE: in this case we are giving permissions to modify not only to title but any other property the group has.
    // See how this can be fine grained, as for now, telepathy does not have a way to differentiate one configuration
    // property permissions from another
    if (not room_config_interface.isNull())
    {
        room_config_interface->setCanUpdateConfiguration(permissions.is_set(GroupPermissions::CanChangeTitle));
    }

    if (not subject_interface.isNull())
    {
        subject_interface->setCanSet(permissions.is_set(GroupPermissions::CanChangeTitle));
    }

    // Can kick other participants?
    if (not group_interface.isNull())
    {
        if (permissions.is_set(GroupPermissions::CanKick))
        {
            group_interface->setGroupFlags(group_interface->groupFlags() | Tp::ChannelGroupFlagCanRemove
                                                                         | Tp::ChannelGroupFlagCanRescind);
        }
        else
        {
            group_interface->setGroupFlags(group_interface->groupFlags() & ~Tp::ChannelGroupFlagCanRemove
                                                                         & ~Tp::ChannelGroupFlagCanRescind);
        }
    }
}

QString mqt::tp::TextChannel::send_message(const Tp::MessagePartList& message, uint flags, Tp::DBusError* error)
{
    LOG(INFO) << __PRETTY_FUNCTION__;

    Q_UNUSED(flags)

    // Mark the special return type for signalling the invalid id.
    static const QString the_invalid_id{};
    // check if the message contains any valid part
    auto it = std::find_if(message.begin(),
                           message.end(),
                           [](const Tp::MessagePart& part)
                           {
        auto it = part.find(message::keys::content);
        return (it != part.end());
    });

    // Did not find any valid part, bailing out now.
    if (it == message.end())
    {
        error->set(TP_QT_ERROR_INVALID_ARGUMENT, "Could not find any valid part in the message");
        return the_invalid_id;
    }

    // Extract the content, assemble a message and send it out.
    try
    {
        auto msg = make_message(message);
        std::string id = chat->send_message(msg);
        // save a copy of the sent message in the map of sent ones just in case the id is a temp one and must be
        // superseded in future
        sent_messages[id] = message;
        return QString::fromStdString(id);
    }
    catch (const std::exception& e)
    {
        error->set(TP_QT_ERROR_NOT_AVAILABLE, QString::fromStdString(e.what()));
        LOG(ERROR) << "An exception has been thrown when sending a message: " << e.what();
        return the_invalid_id;
    }
    catch (...)
    {
        LOG(ERROR) << "An exception has been thrown when sending a message";
        error->set(TP_QT_ERROR_NOT_AVAILABLE, "");
        return the_invalid_id;
    }
}

void mqt::tp::TextChannel::message_acknowledged(const QString &id)
{
    std::cout << __PRETTY_FUNCTION__ << std::endl;
    try
    {
        chat->mark_message_as_read(id.toStdString());
    }
    catch (...)
    {
        LOG(ERROR) << "An exception has been thrown when marking message as read";
    }
}

void mqt::tp::TextChannel::add_members(const Tp::UIntList& handles, const QString& message, Tp::DBusError* error)
{
    Q_UNUSED(message);

    try
    {
        // avoid including self handle as member to add in server
        Tp::UIntList handles_without_self = handles;
        handles_without_self.removeAll(tp_connection->selfHandle());
        if (handles_without_self.size() == 0) {
            return;
        }

        std::shared_ptr<messaging::GroupManager> group_manager = chat->interface<GroupManager>();

        if (!group_manager) {
            error->set(TP_QT_ERROR_NOT_AVAILABLE, "Text channel is not a room");
            return;
        }

        QStringList ids = tp_connection->inspectHandles(Tp::HandleTypeContact, handles_without_self, error);
        messaging::Members members;
        for (QString id : ids)
        {
            members.push_back(std::make_shared<messaging::Member>(id.toStdString()));
        }

        group_manager->add_members(members);
    }
    catch (...)
    {
        LOG(ERROR) << "An exception has been thrown when trying to add members to a group";
    }
}

void mqt::tp::TextChannel::remove_members(const Tp::UIntList& handles, const QString& message, uint reason, Tp::DBusError* error)
{
    Q_UNUSED(message);
    Q_UNUSED(reason);

    try
    {   
        std::shared_ptr<messaging::GroupManager> group_manager = chat->interface<GroupManager>();

        if (!group_manager) {
            error->set(TP_QT_ERROR_NOT_AVAILABLE, "Text channel is not a room");
            return;
        }

        // if handles have self handle, we are leaving the group
        if (handles.contains(tp_connection->selfHandle()))
        {
            group_manager->leave_group();
            return;
        }

        // otherwise we are trying to remove people from the group
        if (not group_manager->permissions().is_set(GroupPermissions::CanKick))
        {
            error->set(TP_QT_ERROR_PERMISSION_DENIED, "Remove members not allowed");
            return;
        }

        QStringList ids = tp_connection->inspectHandles(Tp::HandleTypeContact, handles, error);
        messaging::Members members;
        for (QString id : ids)
        {
            members.push_back(std::make_shared<messaging::Member>(id.toStdString()));
        }

        group_manager->remove_members(members);
    }
    catch (...)
    {
        LOG(ERROR) << "An exception has been thrown when trying to remove members from a group";
    }
}

void mqt::tp::TextChannel::set_chat_state(uint state, Tp::DBusError *error)
{
    Q_UNUSED(error);

    try
    {
        switch (state) {
        case Tp::ChannelChatStateComposing:
            chat->start_typing();
            break;
        case Tp::ChannelChatStateActive:
        case Tp::ChannelChatStateGone:
        case Tp::ChannelChatStateInactive:
        case Tp::ChannelChatStatePaused:
        default:
            chat->end_typing();
            break;
        }
    }
    catch (...)
    {
        LOG(ERROR) << "An exception has been thrown when updating typing signal";
    }
}

void mqt::tp::TextChannel::update_group_configuration(const QVariantMap &properties, Tp::DBusError *error)
{
    // we can only update the title for now
    if (properties.count() != 1 || !properties.contains("Title")) {
        error->set(TP_QT_ERROR_NOT_IMPLEMENTED, "Only the Title property can be changed");
        return;
    }

    set_subject(properties["Title"].toString(), error);
}

void mqt::tp::TextChannel::set_subject(const QString &subject, Tp::DBusError *error)
{
    try
    {
        std::shared_ptr<messaging::GroupManager> group_manager = chat->interface<GroupManager>();
        if (!group_manager) {
            error->set(TP_QT_ERROR_NOT_AVAILABLE, "Text channel is not a room");
            return;
        }

        if (not group_manager->permissions().is_set(GroupPermissions::CanChangeTitle)) {
            error->set(TP_QT_ERROR_PERMISSION_DENIED, "Title property update not allowed");
            return;
        }

        group_manager->change_group_title(subject.toStdString());
    }
    catch (const std::logic_error &e)
    {
        LOG(ERROR) << e.what();
        error->set(TP_QT_ERROR_INVALID_ARGUMENT, QString::fromStdString(e.what()));
    }
    catch (...)
    {
        LOG(ERROR) << "An exception has been thrown when setting subject";
    }
}

void mqt::tp::TextChannel::update_roles(const interfaces::HandleRolesMap &contact_roles, Tp::DBusError *error)
{
    messaging::Members members;

    std::shared_ptr<messaging::GroupManager> group_manager = chat->interface<GroupManager>();
    if (!group_manager) {
        error->set(TP_QT_ERROR_NOT_AVAILABLE, "Text channel is not a room. You can only update roles for group members");
        return;
    }

    // for every handle create a member with his roles
    for (uint handle : contact_roles.keys())
    {
        // skip self handle
        if (handle == tp_connection->selfHandle()) {
            continue;
        }

        Flags<messaging::Role> roles{static_cast<messaging::Role>(contact_roles.value(handle))};

        if (roles.is_set(messaging::Role::Admin) && (not group_manager->permissions().is_set(GroupPermissions::CanSetAdmin))) {
            error->set(TP_QT_ERROR_PERMISSION_DENIED, "Set role admin operation not allowed");
            return;
        }

        QStringList identifiers = tp_connection->inspect_handles(Tp::HandleTypeContact, Tp::UIntList() << handle, error);
        if (identifiers.size() == 0) {
            continue;
        }

        std::string id = identifiers.at(0).toStdString();
        members.push_back(std::make_shared<Member>(id, PendingStatus::None /* not relevant here, but needed to be provided */, roles));
    }

    // adding existing members will modify their roles
    group_manager->add_members(members);
}

void mqt::tp::TextChannel::destroy(Tp::DBusError* error)
{
    try
    {
        std::shared_ptr<messaging::GroupManager> group_manager = chat->interface<GroupManager>();
        if (!group_manager) {
            error->set(TP_QT_ERROR_NOT_AVAILABLE, "Text channel is not a room");
            return;
        }

        if (not group_manager->permissions().is_set(GroupPermissions::CanDissolve))
        {
            error->set(TP_QT_ERROR_PERMISSION_DENIED, "Destroy channel not allowed");
            return;
        }

        group_manager->dissolve_group();
    }
    catch (const std::exception& e)
    {
        error->set(TP_QT_ERROR_RESOURCE_UNAVAILABLE, QString("Failed to destroy: %1").arg(e.what()));
        LOG(ERROR) << "An exception has been thrown when destroying a group: " << e.what();
    }
    catch (...)
    {
        error->set(TP_QT_ERROR_RESOURCE_UNAVAILABLE, QString("Failed to destroy"));
        LOG(ERROR) << "An exception has been thrown when destroying a group";
    }
}

void mqt::tp::TextChannel::register_callbacks_once()
{
    messages_interface->setSendMessageCallback(Tp::memFun(this, &TextChannel::send_message));
    chat_state_interface->setSetChatStateCallback(Tp::memFun(this, &TextChannel::set_chat_state));

    if (!group_interface.isNull())
    {
        group_interface->setAddMembersCallback(Tp::memFun(this, &TextChannel::add_members));
        group_interface->setRemoveMembersCallback(Tp::memFun(this, &TextChannel::remove_members));
    }

    if (!room_config_interface.isNull())
    {
        room_config_interface->setUpdateConfigurationCallback(Tp::memFun(this, &TextChannel::update_group_configuration));
    }

    // if the room interface is available, we can plug the destroyable one too
    if (!room_interface.isNull())
    {
        destroyable_interface->setDestroyCallback(Tp::memFun(this, &TextChannel::destroy));
    }

    if (!roles_interface.isNull())
    {
        roles_interface->setUpdateRolesCallback(Tp::memFun(this, &TextChannel::update_roles));
    }

    if (!subject_interface.isNull())
    {
        subject_interface->setSetSubjectCallback(Tp::memFun(this, &TextChannel::set_subject));
    }
}

void mqt::tp::TextChannel::plug_interfaces_once()
{
    plug_interface_if_available(text_type_interface);
    plug_interface_if_available(messages_interface);
    plug_interface_if_available(room_interface);
    plug_interface_if_available(room_config_interface);
    plug_interface_if_available(chat_state_interface);
    plug_interface_if_available(destroyable_interface);
    plug_interface_if_available(roles_interface);
    plug_interface_if_available(subject_interface);
}

void mqt::tp::TextChannel::plug_interface_if_available(const Tp::AbstractChannelInterfacePtr &interface)
{
    if (interface.isNull())
    {
        return;
    }
    plugInterface(interface);
}

bool mqt::tp::TextChannel::self_is_admin()
{
    std::shared_ptr<messaging::GroupManager> group_manager = chat->interface<GroupManager>();
    if (!group_manager) {
        return false;
    }
    std::set<std::string> admins = group_manager->group_admins();
    std::string self_id = tp_connection->selfID().toStdString();

    for (const std::string &admin : admins) {
        if (admin == self_id) {
            return true;
        }
    }

    return false;
}
