/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MESSAGING_QT_TP_CONNECTION_H_
#define MESSAGING_QT_TP_CONNECTION_H_

#include <messaging/connector.h>
#include <messaging/visibility.h>
#include <messaging/messenger.h>
#include <messaging/group_starter.h>

#include <messaging/qt/variant_map_facade.h>
#include <messaging/qt/network_monitor.h>

#include <TelepathyQt/BaseChannel>
#include <TelepathyQt/BaseConnection>

#include <boost/bimap.hpp>

#include <memory>

namespace messaging
{
// A handy fwd declaration
class Connection;

namespace qt
{
// A handy fwd declaration
class Runtime;
class NetworkMonitor;

namespace tp
{
/// @brief TelepathyConnection represents a connection with a service provider.
class MESSAGING_FW_PUBLIC Connection : public Tp::BaseConnection
        , public messaging::Connection::Observer
        , public messaging::Messenger::Observer
        , public messaging::PresenceManager::Observer
{
    Q_OBJECT
public:
    /// @brief Creates a new instance for the given parameters, setting up
    /// fixed and optional properties supported on this connection.
    static Tp::SharedPtr<Connection> create(const std::shared_ptr<Connector>& connector,
                                            const std::shared_ptr<qt::Runtime>& runtime,
                                            const QString& connection_manager_name,
                                            const QString& protocol_name,
                                            const QVariantMap& parameters);

    static Tp::ConnectionPresenceType messaging_presence_type_to_telepathy(const messaging::PresenceType& presence_type);

    /// @brief request_handles maps ids to handles.
    Tp::UIntList request_handles(uint handleType, const QStringList& identifiers, Tp::DBusError* error);

    /// @brief inspect_handles returns ids for handles, reporting issues via the given error.
    QStringList inspect_handles(uint handleType, const Tp::UIntList& handles, Tp::DBusError* error);

    ~Connection();

public Q_SLOTS:
    void onDisconnected();

private:
    /// @brief Observer implements messaging::Connection::Observer, taking into account
    /// object lifetimes and bridging between managed shared pointer models.
    class Observer
            : public messaging::Connection::Observer
            , public messaging::Messenger::Observer
            , public messaging::PresenceManager::Observer
            , public std::enable_shared_from_this<Observer>
    {
      public:
        /// @brief Creates a new instance for the given managed connection instance.
        Observer(const std::shared_ptr<qt::Runtime>& runtime, const Tp::SharedPtr<Connection>& connection);

        /// @brief on_status_changed translates incoming status changes to telepathy.
        ///
        /// Implemented from messaging::Connection::Observer.
        void on_status_changed(messaging::Connection::Status new_status, messaging::Connection::StatusChangedReason reason) override;

        /// @brief on_message_without_chat_received called when received a messages that creates a new chat
        void on_message_without_chat_received(const Recipient::shared_ptr& recipient, const Message &message) override;

        /// @brief on_status_changed informs about the presence status changed for a recipient
        void on_presence_changed(const messaging::Recipient::shared_ptr& recipient, const messaging::Presence& presence) override;

        /// @brief on_new_group_invitation_received notifies an invitation to a new group
        void on_new_group_invitation_received(const std::shared_ptr<Group>& new_group) override;

      private:
        std::shared_ptr<qt::Runtime> runtime; ///< The qt::Runtime instance that we dispatch to.
        Tp::SharedPtr<Connection> connection; ///< The managed connection instance.
    };

    /// @brief on_message_without_chat_received implementation of messaging::Messaging::Observer method
    void on_message_without_chat_received(const Recipient::shared_ptr& recipient, const Message &message) override;

    /// @brief on_status_changed translates incoming status changes to telepathy.
    ///
    /// Implemented from messaging::Connection::Observer.
    void on_status_changed(messaging::Connection::Status new_status, messaging::Connection::StatusChangedReason reason) override;

    /// @brief on_status_changed informs about the presence status changed for a recipient
    ///
    /// Implemented from messaging::PresenceManager::Observer
    void on_presence_changed(const messaging::Recipient::shared_ptr& recipient, const messaging::Presence& presence) override;

    /// @brief on_new_group_invitation_received notifies a new group invitation received
    ///
    /// Implemented from messaging::SessionManager::Observer
    void on_new_group_invitation_received(const std::shared_ptr<Group>& new_group) override;

    /// @brief Creates a new instance for the given parameters, setting up
    /// fixed and optional properties supported on this connection.
    Connection(const std::shared_ptr<Connector>& connector,
               const std::shared_ptr<qt::Runtime>& runtime,
               const QString& connection_manager_name,
               const QString& protocol_name,
               const QVariantMap& parameters);

    /// @brief HandleIdMap helps in mapping handles to ids, and vice versa.
    typedef boost::bimap<uint, QString> HandleIdMap;
    /// @brief Safe us some typing when inserting elements into the map.
    typedef typename HandleIdMap::value_type HandleIdPair;

    /// @brief connect calls into the messaging::Connector instance to establish a new messaging::Connection
    ///
    /// Never throws, and communicates back any potential exception via the given DBusError instance.
    void connect(Tp::DBusError* error) noexcept(true);

    /// @brief create_channel creates the requested channel or reports issues via error.
    Tp::BaseChannelPtr create_channel(const QVariantMap& request,
                                      Tp::DBusError* error);

    /// @brief create_text_channel creates the requested text channel.
    Tp::BaseChannelPtr create_text_channel(uint target_handle_type,
                                           uint target_handle,
                                           const QVariantMap& hints,
                                           Tp::DBusError* error);

    /// @brief get_contact_attributes return requested attributes for given contacts
    Tp::ContactAttributesMap  get_contact_attributes(const Tp::UIntList &handles,
                                                     const QStringList &ifaces,
                                                     Tp::DBusError *error);

    QString uniqueName() const;

    // Remember the qt::Runtime instance we are associated with.
    std::shared_ptr<qt::Runtime> runtime;
    // We keep the connector instance around.
    std::shared_ptr<messaging::Connector> connector;
    // Our internal, briding Observer implementation.
    std::shared_ptr<Observer> observer;

    // We have to keep track of the connection, and it is initially empty.
    std::shared_ptr<messaging::Connection> connection;
    // The currently known handles and their associated ids.
    HandleIdMap handles_;
    // The currently known handles and their associated ids only for groups.
    HandleIdMap group_handles_;
    Tp::SimpleContactPresences presences_;
    // Telepathy interfaces go here.
    Tp::BaseConnectionRequestsInterfacePtr requests;
    Tp::BaseConnectionContactsInterfacePtr contacts;
    Tp::BaseConnectionSimplePresenceInterfacePtr simplePresenceIface;
    QString connection_id;

    NetworkMonitor network_monitor_;
};
}
}
}

#endif  // MESSAGING_QT_TP_CONNECTION_H_
