/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MESSAGING_QT_RUNTIME_H_
#define MESSAGING_QT_RUNTIME_H_

#include <messaging/visibility.h>

#include <QtCore/QCoreApplication>
#include <QtDBus/QDBusConnection>

#include <memory>
#include <stdexcept>

namespace messaging
{
namespace qt
{
/// @brief Runtime is the primary point of entry to the qt world, bundling
/// together runtime requirements and functionalities.
class MESSAGING_FW_PUBLIC Runtime
{
public:
    /// @brief AlreadyCreated is thrown if create_once_or_throw is called more than once.
    struct AlreadyCreated : public std::logic_error
    {
        AlreadyCreated();
    };

    /// @brief create_once_or_throw tries to create a new Runtime instance with the given parameters.
    /// @throws AlreadyCreated if there is an active instance of Runtime in use.
    static std::shared_ptr<Runtime> create_once_or_throw(QDBusConnection::BusType bus_type);
    /// @brief create_once_or_throw tries to create a new Runtime instance with the given parameters.
    /// @throws AlreadyCreated if there is an active instance of Runtime in use.
    static std::shared_ptr<Runtime> create_once_or_throw(const QString& dbus_address);

    /// @cond
    Runtime(const Runtime&) = delete;
    Runtime(Runtime&&) = delete;
    virtual ~Runtime() = default;
    Runtime& operator=(const Runtime&) = delete;
    Runtime& operator=(Runtime&&) = delete;
    /// @endcond

    /// @brief enter_with_task takes the given functor and executes it in the
    /// context of the runtime, i.e., on the qt main thread.
    virtual void enter_with_task(const std::function<void()>& f);

    /// @brief run hands a thread of execution to the runtime.
    virtual int run();

    /// @brief stop requests the runtime to shutdown.
    virtual void stop(int code);

    /// @brief dbus_connection returns an immutable instance of the bus connection associated to this
    /// runtime instance.

    virtual const QDBusConnection& dbus_connection() const;

protected:
    /// @brief Constructs a new instance connecting to the given bus.
    Runtime(QDBusConnection::BusType bus_type);
    /// @brief Constructs a new instance connecting to the bus with the given address.
    Runtime(const QString& dbus_address);

private:
    QCoreApplication app_;
    QDBusConnection dbus_connection_;
};
}
}

#endif  // MESSAGING_QT_RUNTIME_H_
