/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MESSAGING_MESSENGER_H
#define MESSAGING_MESSENGER_H

#include <messaging/non_copyable.h>
#include <messaging/non_movable.h>

#include <messaging/chat.h>
#include <messaging/recipient.h>

#include <memory>

namespace messaging
{
struct Message;
class Recipient;
class Group;

/// @brief Messenger abstracts creation of Chat instances.
class MESSAGING_FW_PUBLIC Messenger : NonCopyable, NonMovable
{
public:
    virtual ~Messenger() = default;

    class Observer: NonCopyable, NonMovable
    {
      public:
        /// @brief a message has arrived and there is still not related chat. This event creates a new one and put the message in there.
        virtual void on_message_without_chat_received(const std::shared_ptr<Recipient>& recipient,
                                                      const Message& message) = 0;

        /// @brief a new group invitation is received. This should be managed to create a new related chat
        virtual void on_new_group_invitation_received(const std::shared_ptr<Group>& new_group) = 0;

      protected:
        /// @brief only inherited classes can created an observer
        Observer() = default;
    };

    /// @brief initiate_chat_with creates a new chat instance with the given recipient as the participant
    /// in the textual communication.
    std::shared_ptr<Chat> initiate_chat_with(const std::shared_ptr<Recipient>& recipient,
                                             const std::shared_ptr<Chat::Observer>& observer);

    /// @brief chats returns the map of currently active chats in the plugin
    std::map<std::string, messaging::Chat::SPtr>& chats();

  protected:
    /// @brief Messenger constructs a new instance, installing the given observer instance.
    Messenger(const std::shared_ptr<Observer>& observer);

    /// @brief announces a message creating a new chat is received
    void announce_message_without_chat_received(const std::shared_ptr<Recipient>& recipient,
                                                const Message& message);

    /// @brief announces a new group invitation received
    void announce_new_group_invitation_received(const std::shared_ptr<Group>& new_group);

    virtual Chat::SPtr create_chat_with(const std::shared_ptr<Recipient>& recipient,
                                        const std::shared_ptr<Chat::Observer>& observer) = 0;

    Chat::SPtr get_chat(const std::string& recipient_id);

  private:
    /// @cond
    class Private;
    std::shared_ptr<Private> impl;
    /// @endcond
};
}

#endif  // MESSAGING_MESSENGER_H
