/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MESSAGING_HAS_INTERFACES
#define MESSAGING_HAS_INTERFACES

#include <messaging/interface.h>

namespace messaging
{

class MESSAGING_FW_PUBLIC HasInterfaces : public std::enable_shared_from_this<HasInterfaces>
{
public:
    virtual ~HasInterfaces() = default;

    /// @brief returns all the set of plugged interfaces
    const InterfacesSet& interfaces();

    /// @brief interface<T> returns a valid instance of a plugged interface implementation for the requested T type
    template <typename T>
    std::shared_ptr<T> interface()
    {
        for (auto it = interfaces().begin(); it != interfaces().end(); ++it)
        {
            auto result = std::dynamic_pointer_cast<T>(*it);
            if (result)
            {
                return result;
            }
        }
        return nullptr;
    }

    /// @brief plug_interface plugs a specific interface implementation object
    void plug_interface(const std::shared_ptr<Interface>& interface);

protected:
    HasInterfaces();

    /// @brief on_interface_plugged overwrite this virtual method in subclasses if wanted to perform any action when
    /// certain interface is plugged
    virtual void on_interface_plugged(const std::shared_ptr<Interface>& interface);

private:
    /// @cond
    struct Private;
    std::shared_ptr<Private> impl;
    /// @endcond
};

}
#endif // MESSAGING_HAS_INTERFACES

