/* mg-custom-layout.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "mg-custom-layout.h"
#include "mg-work-layout.h"
#include "mg-ref-base.h"
#include "mg-xml-storage.h"
#include "mg-referer.h"
#include "mg-query.h"
#include "mg-target.h"
#include "mg-field.h"
#include <string.h>

/* 
 * Main static functions 
 */
static void mg_custom_layout_class_init (MgCustomLayoutClass * class);
static void mg_custom_layout_init (MgCustomLayout * srv);
static void mg_custom_layout_dispose (GObject   * object);
static void mg_custom_layout_finalize (GObject   * object);

#ifdef debug
static void mg_custom_layout_dump                     (MgCustomLayout *context, guint offset);
#endif

/* XML storage interface */
static void        mg_custom_layout_xml_storage_init (MgXmlStorageIface *iface);
static gchar      *mg_custom_layout_get_xml_id (MgXmlStorage *iface);
static xmlNodePtr  mg_custom_layout_save_to_xml (MgXmlStorage *iface, GError **error);
static gboolean    mg_custom_layout_load_from_xml (MgXmlStorage *iface, xmlNodePtr node, GError **error);

/* Referer interface */
static void        mg_custom_layout_referer_init        (MgRefererIface *iface);
static gboolean    mg_custom_layout_activate            (MgReferer *iface);
static void        mg_custom_layout_deactivate          (MgReferer *iface);
static gboolean    mg_custom_layout_is_active           (MgReferer *iface);
static GSList     *mg_custom_layout_get_ref_objects     (MgReferer *iface);
static void        mg_custom_layout_replace_refs        (MgReferer *iface, GHashTable *replacements);


/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/*
 * Structure to hold a Glade import spec
 */
typedef struct {
	gchar     *box_name;
	gboolean   show_actions;
	GSList    *dest_list; /* list of MgRefBase objects */
} GladeBox;
#define GLADE_BOX(x) ((GladeBox*)(x))

static GladeBox *glade_box_new (const gchar *name, gboolean show_actions);
static void      glade_box_destroy (GladeBox *box);


/*
 * Structure to hold parameters connections between MgWorkWidget widgets
 */
typedef struct {
	MgRefBase *src_layout;
	MgRefBase *src_field;
	MgRefBase *dest_layout;
	MgRefBase *dest_field;
} LayoutConnect;
#define LAYOUT_CONNECT(x) ((LayoutConnect *)(x))
static LayoutConnect *layout_connect_new (MgCustomLayout *layout, MgConf *conf);
static void           layout_connect_destroy (MgCustomLayout *layout, LayoutConnect *lc);


/* private structure */
struct _MgCustomLayoutPrivate
{
	MgCustomLayoutType type;

	/* contents of the layout */
	union {
		/* sub layouts, for MG_CUSTOM_LAYOUT_LAYOUT */
		struct {
			GSList           *children; /* list of MgCustomLayout objects, owned here */
			GSList           *connects; /* list of LayoutConnect structures */
		}                layout;
		
		struct {
			/* for all the MgWork* widgets */
			MgQuery          *query;       /* owned query, may be NULL */
			MgRefBase        *ref_query;      /* reference to another query, even 'query' or 'query_extra' */
			guint             mode;
			MgRefBase        *ref_modif;

			/* For MgWorkMatrix only */
			MgQuery          *query_extra; /* owned query, may be NULL */
			MgRefBase        *ref_query_extra;/* reference to another query, even 'query' or 'query_extra' */
			MgRefBase        *ref_cols_target;
			MgRefBase        *ref_rows_target;
			MgWorkMatrixType  view_type;
		}                work_iface;
	} contents;
	GSList          *referer_objects; /* list of all the MgRefBase objects owned by the MgCustomLayout */
	
	/* glade layout */
	gchar           *filename;
	gchar           *root_widget;
	GSList          *boxes; /* list of GladeBox structures */
};



/* module error */
GQuark mg_custom_layout_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("mg_custom_layout_error");
	return quark;
}


guint
mg_custom_layout_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MgCustomLayoutClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) mg_custom_layout_class_init,
			NULL,
			NULL,
			sizeof (MgCustomLayout),
			0,
			(GInstanceInitFunc) mg_custom_layout_init
		};

		static const GInterfaceInfo xml_storage_info = {
			(GInterfaceInitFunc) mg_custom_layout_xml_storage_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo referer_info = {
			(GInterfaceInitFunc) mg_custom_layout_referer_init,
			NULL,
			NULL
		};
		
		type = g_type_register_static (MG_BASE_TYPE, "MgCustomLayout", &info, 0);
		g_type_add_interface_static (type, MG_XML_STORAGE_TYPE, &xml_storage_info);
		g_type_add_interface_static (type, MG_REFERER_TYPE, &referer_info);
	}
	return type;
}

static void 
mg_custom_layout_xml_storage_init (MgXmlStorageIface *iface)
{
	iface->get_xml_id = mg_custom_layout_get_xml_id;
	iface->save_to_xml = mg_custom_layout_save_to_xml;
	iface->load_from_xml = mg_custom_layout_load_from_xml;
}

static void
mg_custom_layout_referer_init (MgRefererIface *iface)
{
	iface->activate = mg_custom_layout_activate;
	iface->deactivate = mg_custom_layout_deactivate;
	iface->is_active = mg_custom_layout_is_active;
	iface->get_ref_objects = mg_custom_layout_get_ref_objects;
	iface->replace_refs = mg_custom_layout_replace_refs;
}

static void
mg_custom_layout_class_init (MgCustomLayoutClass *class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = mg_custom_layout_dispose;
	object_class->finalize = mg_custom_layout_finalize;
	
	/* virtual functions */
#ifdef debug
        MG_BASE_CLASS (class)->dump = (void (*)(MgBase *, guint)) mg_custom_layout_dump;
#endif
}

static void
mg_custom_layout_init (MgCustomLayout *mg_custom_layout)
{
	mg_custom_layout->priv = g_new0 (MgCustomLayoutPrivate, 1);
}

/**
 * mg_custom_layout_new
 * @conf: a #MgConf object
 *
 * Creates a new #MgCustomLayout object
 *
 * Returns: a new #MgCustomLayout object
 */
GObject *
mg_custom_layout_new (MgConf *conf)
{
	GObject *obj;
	MgCustomLayout *mg_custom_layout;
	guint id;

	g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);

	obj = g_object_new (MG_CUSTOM_LAYOUT_TYPE, "conf", conf, NULL);
        mg_custom_layout = MG_CUSTOM_LAYOUT (obj);

	g_object_get (G_OBJECT (conf), "layout_serial", &id, NULL);
	mg_base_set_id (MG_BASE (obj), id);

	mg_conf_declare_layout (conf, mg_custom_layout);

	return obj;
}

/**
 * mg_custom_layout_get_layout_type
 * @layout: a #MgCustomLayout object
 *
 * Get the kind of custom layout @layout is.
 *
 * Returns:
 */
MgCustomLayoutType
mg_custom_layout_get_layout_type (MgCustomLayout *layout)
{
	g_return_val_if_fail (layout && IS_MG_CUSTOM_LAYOUT (layout), MG_CUSTOM_LAYOUT_FORM);
	g_return_val_if_fail (layout->priv, MG_CUSTOM_LAYOUT_FORM);

	return layout->priv->type;
}

static void
mg_custom_layout_dispose (GObject *object)
{
	MgCustomLayout *cl;
	GSList *list;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_CUSTOM_LAYOUT (object));

	cl = MG_CUSTOM_LAYOUT (object);

	/* contents related */
	switch (cl->priv->type) {
	case MG_CUSTOM_LAYOUT_LAYOUT:
		if (cl->priv->contents.layout.children) {
			list = cl->priv->contents.layout.children;
			while (list) {
				g_object_unref (list->data);
				list = g_slist_next (list);
			}
			g_slist_free (cl->priv->contents.layout.children);
			cl->priv->contents.layout.children = NULL;
		}
		if (cl->priv->contents.layout.connects) {
			list = cl->priv->contents.layout.connects;
			while (list) {
				layout_connect_destroy (cl, LAYOUT_CONNECT (list->data));
				list = g_slist_next (list);
			}
			g_slist_free (cl->priv->contents.layout.connects);
			cl->priv->contents.layout.connects = NULL;
		}
		break;
	case MG_CUSTOM_LAYOUT_GRID:
	case MG_CUSTOM_LAYOUT_FORM:
	case MG_CUSTOM_LAYOUT_MATRIX:
		if (cl->priv->contents.work_iface.query) {
			g_object_unref (cl->priv->contents.work_iface.query);
			cl->priv->contents.work_iface.query = NULL;
		}
		if (cl->priv->contents.work_iface.query_extra) {
			g_object_unref (cl->priv->contents.work_iface.query_extra);
			cl->priv->contents.work_iface.query_extra = NULL;
		}
		if (cl->priv->contents.work_iface.ref_query) {
			g_object_unref (cl->priv->contents.work_iface.ref_query);
			cl->priv->contents.work_iface.ref_query = NULL;
		}
		if (cl->priv->contents.work_iface.ref_query_extra) {
			g_object_unref (cl->priv->contents.work_iface.ref_query_extra);
			cl->priv->contents.work_iface.ref_query_extra = NULL;
		}
		if (cl->priv->contents.work_iface.ref_modif) {
			g_object_unref (cl->priv->contents.work_iface.ref_modif);
			cl->priv->contents.work_iface.ref_modif = NULL;
		}
		if (cl->priv->contents.work_iface.ref_cols_target) {
			g_object_unref (cl->priv->contents.work_iface.ref_cols_target);
			cl->priv->contents.work_iface.ref_cols_target = NULL;
		}
		if (cl->priv->contents.work_iface.ref_rows_target) {
			g_object_unref (cl->priv->contents.work_iface.ref_rows_target);
			cl->priv->contents.work_iface.ref_rows_target = NULL;
		}
		break;
	}

	/* Glade related */
	if (cl->priv->filename) {
		g_free (cl->priv->filename);
		cl->priv->filename = NULL;
	}

	if (cl->priv->root_widget) {
		g_free (cl->priv->root_widget);
		cl->priv->root_widget = NULL;
	}

	if (cl->priv->boxes) {
		list = cl->priv->boxes;		
		while (list) {
			glade_box_destroy (GLADE_BOX (list->data));
			list = g_slist_next (list);
		}
		g_slist_free (cl->priv->boxes);
	}

	if (cl->priv->referer_objects) {
		g_slist_free (cl->priv->referer_objects);
		cl->priv->referer_objects = NULL;
	}

	/* parent class */
	parent_class->dispose (object);
}


static void
mg_custom_layout_finalize (GObject *object)
{
	MgCustomLayout *cl;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_CUSTOM_LAYOUT (object));

	cl = MG_CUSTOM_LAYOUT (object);
	if (cl->priv) {
		g_free (cl->priv);
		cl->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


#ifdef debug
static void
mg_custom_layout_dump (MgCustomLayout *cl, guint offset)
{
	gchar *str;
        guint i;
	GSList *list;
	
	g_return_if_fail (cl && IS_MG_CUSTOM_LAYOUT (cl));

        /* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

        /* dump */
        if (cl->priv) {
                g_print ("%s" D_COL_H1 "MgCustomLayout" D_COL_NOR " %p (id=%s) ",
                         str, cl, mg_xml_storage_get_xml_id (MG_XML_STORAGE (cl)));
		switch (cl->priv->type) {
		case MG_CUSTOM_LAYOUT_LAYOUT:
			g_print ("MG_CUSTOM_LAYOUT_LAYOUT\n");
			break;
		case MG_CUSTOM_LAYOUT_GRID:
			g_print ("MG_CUSTOM_LAYOUT_GRID\n");
			break;
		case MG_CUSTOM_LAYOUT_FORM:
			g_print ("MG_CUSTOM_LAYOUT_FORM\n");
			break;
		case MG_CUSTOM_LAYOUT_MATRIX:
			g_print ("MG_CUSTOM_LAYOUT_MATRIX\n");
			break;
		}

		switch (cl->priv->type) {
		case MG_CUSTOM_LAYOUT_LAYOUT:
			list = cl->priv->contents.layout.children;
			if (list)
				g_print ("%s" D_COL_H1 "* Sub layouts:\n" D_COL_NOR, str);
			else
				g_print ("%s" D_COL_H1 "* No sub layout.\n" D_COL_NOR, str);
			while (list) {
				mg_base_dump (MG_BASE (list->data), offset+5);
				list = g_slist_next (list);
			}
			list = cl->priv->contents.layout.connects;
			if (list)
				g_print ("%s" D_COL_H1 "* Connections:\n" D_COL_NOR, str);
			else
				g_print ("%s" D_COL_H1 "* No connections.\n" D_COL_NOR, str);
			while (list) {
				g_print ("%s - Source layout:\n", str);
				mg_base_dump (MG_BASE (LAYOUT_CONNECT (list->data)->src_layout), offset+5);
				g_print ("%s - Source field:\n", str);
				mg_base_dump (MG_BASE (LAYOUT_CONNECT (list->data)->src_field), offset+5);
				g_print ("%s - Destination layout:\n", str);
				mg_base_dump (MG_BASE (LAYOUT_CONNECT (list->data)->dest_layout), offset+5);
				g_print ("%s - Destination field:\n", str);
				mg_base_dump (MG_BASE (LAYOUT_CONNECT (list->data)->dest_field), offset+5);
				list = g_slist_next (list);
			}
			break;
		case MG_CUSTOM_LAYOUT_GRID:
		case MG_CUSTOM_LAYOUT_FORM:
		case MG_CUSTOM_LAYOUT_MATRIX:
			if (cl->priv->contents.work_iface.query) {
				g_print ("%s" D_COL_H1 "* Query:\n" D_COL_NOR, str);
				mg_base_dump (MG_BASE (cl->priv->contents.work_iface.query), offset+5);
			}
			if (cl->priv->contents.work_iface.query_extra) {
				g_print ("%s" D_COL_H1 "* Query_extra:\n" D_COL_NOR, str);
				mg_base_dump (MG_BASE (cl->priv->contents.work_iface.query_extra), offset+5);
			}
			if (cl->priv->contents.work_iface.ref_query) {
				g_print ("%s" D_COL_H1 "* Ref Query:\n" D_COL_NOR, str);
				mg_base_dump (MG_BASE (cl->priv->contents.work_iface.ref_query), offset+5);
			}
			if (cl->priv->contents.work_iface.ref_query_extra) {
				g_print ("%s" D_COL_H1 "* Ref Query_extra:\n" D_COL_NOR, str);
				mg_base_dump (MG_BASE (cl->priv->contents.work_iface.ref_query), offset+5);
			}
			g_print ("%s" D_COL_H1 "* Mode:" D_COL_NOR " %d\n", str, cl->priv->contents.work_iface.mode);
			if (cl->priv->contents.work_iface.ref_modif) {
				g_print ("%s" D_COL_H1 "* Modif [table|target]:\n" D_COL_NOR, str);
				mg_base_dump (MG_BASE (cl->priv->contents.work_iface.ref_modif), offset+5);
			}
			break;
		}

		if (cl->priv->filename && cl->priv->root_widget) {
			g_print ("%s" D_COL_H1 "* Glade file:" D_COL_NOR " %s, root_widget=%s\n", 
				 str, cl->priv->filename, cl->priv->root_widget);
			list = cl->priv->boxes;
			while (list) {
				GSList *list2;
				g_print ("%s   " D_COL_H1 "-> box" D_COL_NOR " %s for:\n",
					 str, GLADE_BOX (list->data)->box_name);
				list2 = GLADE_BOX (list->data)->dest_list;
				while (list2) {
					mg_base_dump (MG_BASE (list2->data), offset+6);
					list2 = g_slist_next (list2);
				}
				g_print ("\n");
				list = g_slist_next (list);
			}
		}
		else 
			g_print ("%s" D_COL_H1 "* No Glade file.\n" D_COL_NOR, str);
	}
        else
                g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, cl);
}
#endif

/**
 * mg_custom_layout_get_widget
 * @layout: a #MgCustomLayout object
 * @error: location to store error, or %NULL
 *
 * Creates a widget from the "specifications" stored within @layout.
 *
 * Returns: a new #MgWorkLayout widget, or %NULL if an error occured.
 */
GtkWidget *
mg_custom_layout_get_widget (MgCustomLayout *layout, GError **error)
{
	g_return_val_if_fail (layout && IS_MG_CUSTOM_LAYOUT (layout), NULL);
	g_return_val_if_fail (layout->priv, NULL);

	return mg_work_layout_new (layout);
}

/**
 * mg_custom_layout_get_glade_instance
 * @layout: a #MgCustomLayout object
 * @root_widget: a place to store a pointer to the root widget of the #GladeXml instance
 * @box_widgets: a place to store the GHashTable for the #GtkBox widgets
 * @error: location to store error, or %NULL
 *
 * Builds a #GladeXML object, and if there is no error, creates a new #GHashTable and stores
 * it in @box_widgets, and stores a pointer to the root widget into @root_widget.
 *
 * Returns: a new #GladeXML object, or %NULL if an error occured or no Glade
 * file is to be used in @layout.
 */
GladeXML *
mg_custom_layout_get_glade_instance (MgCustomLayout *layout, GtkWidget **root_widget, 
				     GHashTable **box_widgets, GError **error)
{
	GladeXML *xml = NULL;
	GHashTable *hash;
	GSList *box_list;
	gboolean err = FALSE;
	
	g_return_val_if_fail (layout && IS_MG_CUSTOM_LAYOUT (layout), NULL);
	g_return_val_if_fail (layout->priv, NULL);
	g_return_val_if_fail (root_widget, NULL);
	g_return_val_if_fail (box_widgets, NULL);

	/* chech */
	if (! mg_referer_activate (MG_REFERER (layout))) {
		g_set_error (error,
			     MG_CUSTOM_LAYOUT_ERROR,
			     MG_CUSTOM_LAYOUT_GLADEXML_ERROR,
			     _("Missing required objects!"));
		*root_widget = NULL;
		*box_widgets = NULL;
		return NULL;
	}

	/* making the GladeXML object */
	if (layout->priv->filename && *layout->priv->filename) {
		if (! g_file_test (layout->priv->filename, G_FILE_TEST_EXISTS))
			g_set_error (error,
				     MG_CUSTOM_LAYOUT_ERROR,
				     MG_CUSTOM_LAYOUT_GLADEXML_ERROR,
				     _("XML file '%s' does not exist"), layout->priv->filename);
		else {
			xml = glade_xml_new (layout->priv->filename, layout->priv->root_widget, NULL);
			if (!xml) 
				g_set_error (error,
					     MG_CUSTOM_LAYOUT_ERROR,
					     MG_CUSTOM_LAYOUT_GLADEXML_ERROR,
					     _("Can't use XML file '%s' with root widget '%s'"), 
					     layout->priv->filename, layout->priv->root_widget);
		}
	}

	if (!xml) {
		*root_widget = NULL;
		*box_widgets = NULL;
		return NULL;
	}

	/* creating the box_widgets hash table */
	hash = g_hash_table_new (NULL, NULL);
	if (!(*root_widget = glade_xml_get_widget (xml, layout->priv->root_widget))) {
		err = TRUE;
		g_set_error (error,
			     MG_CUSTOM_LAYOUT_ERROR,
			     MG_CUSTOM_LAYOUT_GLADEXML_ERROR,
			     _("Can't find widget named '%s'"), layout->priv->root_widget);
	}

	box_list = layout->priv->boxes;
	while (box_list && !err) {
		GtkWidget *box = glade_xml_get_widget (xml, GLADE_BOX (box_list->data)->box_name);

		if (box && GTK_IS_BOX (box)) {
			GSList *dest_list = GLADE_BOX (box_list->data)->dest_list;
			while (dest_list) {
				g_hash_table_insert (hash, 
						     mg_ref_base_get_ref_object (MG_REF_BASE (dest_list->data)), box);
				dest_list = g_slist_next (dest_list);
			}
			if (GLADE_BOX (box_list->data)->show_actions)
				g_object_set_data (G_OBJECT (box), "show_actions", GINT_TO_POINTER (TRUE));
		}
		else {
			err = TRUE;
			if (!box)
				g_set_error (error,
					     MG_CUSTOM_LAYOUT_ERROR,
					     MG_CUSTOM_LAYOUT_GLADEXML_ERROR,
					     _("Can't find widget named '%s'"), GLADE_BOX (box_list->data)->box_name);
			else
				g_set_error (error,
					     MG_CUSTOM_LAYOUT_ERROR,
					     MG_CUSTOM_LAYOUT_GLADEXML_ERROR,
					     _("Widget '%s' is not a GtkBox (is a %s)"), GLADE_BOX (box_list->data)->box_name,
					     G_OBJECT_TYPE_NAME (box));
		}

		box_list = g_slist_next (box_list);
	}

	if (!err)
		*box_widgets = hash;
	else {
		*box_widgets = NULL;
		g_hash_table_destroy (hash);
		g_object_unref (xml);
		xml = NULL;
	}

	return xml;
}

/**
 * mg_custom_layout_get_exec_context
 * @layout: a #MgCustomLayout object
 *
 * Use this function to retreive a #MgContext object containing all the
 * #MgParameter parameters required to be able to 'run' a widget obtained using the
 * mg_custom_layout_get_widget() function.
 *
 * Note: the parameters contained within the returned #MgContext won't be used by any
 * widget obtained using the mg_custom_layout_get_widget(); accessing those parameters
 * is done using the methods of the #MgWorkWidget interface.
 *
 * Returns: a new #MgContext object
 */
MgContext *
mg_custom_layout_get_exec_context (MgCustomLayout *layout)
{
	MgContext *retval = NULL;

	g_return_val_if_fail (layout && IS_MG_CUSTOM_LAYOUT (layout), NULL);
	g_return_val_if_fail (layout->priv, NULL);

	TO_IMPLEMENT;
	return retval;
}

/**
 * mg_custom_layout_get_data_context
 * @layout: a #MgCustomLayout object
 *
 * Use this function to retreive a #MgContext object containing all the
 * #MgParameter parameters which get updated when a widget obtained using the
 * mg_custom_layout_get_widget() is used.
 *
 * Note: the parameters contained within the returned #MgContext won't be used by any
 * widget obtained using the mg_custom_layout_get_widget(); accessing those parameters
 * is done using the methods of the #MgWorkWidget interface.
 *
 * Returns: a new #MgContext object
 */
MgContext *
mg_custom_layout_get_data_context (MgCustomLayout *layout)
{
	MgContext *retval = NULL;

	g_return_val_if_fail (layout && IS_MG_CUSTOM_LAYOUT (layout), NULL);
	g_return_val_if_fail (layout->priv, NULL);

	TO_IMPLEMENT;
	return retval;
}

/**
 * mg_custom_layout_get_data
 * @layout: a #MgCustomLayout object
 * @error: place to store the error, or %NULL
 *
 * Retreive all the information stored in @layout to be able to create
 * #MgWorkLayout objects.
 *
 * The caller is responsible to call mg_custom_layout_data_free() on the returned structure, without
 * trying to free its internals, or unref the objects pointer inside it.
 *
 * The returned #MgCustomLayoutData structure holds pointers to the referenced objects, and a check is made for
 * all the pointers which cannot be %NULL (if an error is found, then the function returns %NULL, and %error is set)
 *
 * Returns: a new #MgCustomLayoutData structure, to be free'ed by the caller using mg_custom_layout_data_free()
 */
MgCustomLayoutData *
mg_custom_layout_get_data (MgCustomLayout *layout, GError **error)
{
	MgCustomLayoutData *data;
	MgCustomLayoutConnect *cdata;
	GSList *list;
	gboolean err;

	g_return_val_if_fail (layout && IS_MG_CUSTOM_LAYOUT (layout), NULL);
	g_return_val_if_fail (layout->priv, NULL);

	if (!mg_referer_activate (MG_REFERER (layout))) {
		g_set_error (error,
			     MG_CUSTOM_LAYOUT_ERROR,
			     MG_CUSTOM_LAYOUT_DATA_ERROR,
			     _("Cannot find some objects used by this MgCustomLayout object"));
		return NULL;
	}

	data = g_new0 (MgCustomLayoutData, 1);
	data->type = layout->priv->type;
	switch (layout->priv->type) {
	case MG_CUSTOM_LAYOUT_LAYOUT:
		err = FALSE;
		list = layout->priv->contents.layout.connects;
		while (list) {
			cdata = g_new0 (MgCustomLayoutConnect, 1);
			cdata->src_layout = mg_ref_base_get_ref_object (LAYOUT_CONNECT (list->data)->src_layout);
			cdata->src_field = mg_ref_base_get_ref_object (LAYOUT_CONNECT (list->data)->src_field);
			cdata->dest_layout = mg_ref_base_get_ref_object (LAYOUT_CONNECT (list->data)->dest_layout);
			cdata->dest_field = mg_ref_base_get_ref_object (LAYOUT_CONNECT (list->data)->dest_field);
			data->contents.layout.connects = g_slist_append (data->contents.layout.connects, cdata);

			if (!g_slist_find (layout->priv->contents.layout.children, cdata->src_layout)) {
				g_set_error (error,
					     MG_CUSTOM_LAYOUT_ERROR,
					     MG_CUSTOM_LAYOUT_DATA_ERROR,
					     _("Can't find source layout among sub-layouts"));
				err = TRUE;
			}
			if (!g_slist_find (layout->priv->contents.layout.children, cdata->dest_layout)) {
				g_set_error (error,
					     MG_CUSTOM_LAYOUT_ERROR,
					     MG_CUSTOM_LAYOUT_DATA_ERROR,
					     _("Can't find destination layout among sub-layouts"));
				err = TRUE;
			}
			TO_IMPLEMENT;
			if (0 && !err) {
				/* make sure that:
				 * --> the cdata->src_field has a corresponding parameter in the potential data context
				 *     of a widget created using mg_custom_layout_get_widget(cdata->src_layout)
				 * --> the cdata->dest_field has a corresponding parameter in the potential exec context
				 *     of a widget created using mg_custom_layout_get_widget(cdata->dest_layout) 
				 */
				MgContext *context;

				context = mg_custom_layout_get_data_context (cdata->src_layout);
				if (! mg_context_find_parameter_for_field (context, cdata->src_field)) {
					g_set_error (error,
						     MG_CUSTOM_LAYOUT_ERROR,
						     MG_CUSTOM_LAYOUT_DATA_ERROR,
						     _("Invalid source field for layout connections"));
					err = TRUE;
				}
				g_object_unref (context);

				context = mg_custom_layout_get_exec_context (cdata->dest_layout);
				if (! mg_context_find_parameter_for_field (context, cdata->dest_field)) {
					g_set_error (error,
						     MG_CUSTOM_LAYOUT_ERROR,
						     MG_CUSTOM_LAYOUT_DATA_ERROR,
						     _("Invalid destination field for layout connections"));
					err = TRUE;
				}
				g_object_unref (context);
			}

			if (err) {
				mg_custom_layout_data_free (data);
				return NULL;
			}

			list = g_slist_next (list);
		}
		data->contents.layout.children = layout->priv->contents.layout.children;
		break;
	case MG_CUSTOM_LAYOUT_MATRIX:
		if (layout->priv->contents.work_iface.ref_query_extra)
			data->contents.work_iface.query_extra = 
				MG_QUERY (mg_ref_base_get_ref_object (layout->priv->contents.work_iface.ref_query_extra));
		if (layout->priv->contents.work_iface.ref_cols_target)
			data->contents.work_iface.cols_target = 
				MG_TARGET (mg_ref_base_get_ref_object (layout->priv->contents.work_iface.ref_cols_target));
		if (layout->priv->contents.work_iface.ref_rows_target)
			data->contents.work_iface.rows_target = 
				MG_TARGET (mg_ref_base_get_ref_object (layout->priv->contents.work_iface.ref_rows_target));
		data->contents.work_iface.view_type = layout->priv->contents.work_iface.view_type;
	case MG_CUSTOM_LAYOUT_GRID:
	case MG_CUSTOM_LAYOUT_FORM:
		data->contents.work_iface.mode = layout->priv->contents.work_iface.mode;
		if (layout->priv->contents.work_iface.ref_query)
			data->contents.work_iface.query = 
				MG_QUERY (mg_ref_base_get_ref_object (layout->priv->contents.work_iface.ref_query));
		if (layout->priv->contents.work_iface.ref_modif)
			data->contents.work_iface.modified = 
				mg_ref_base_get_ref_object (layout->priv->contents.work_iface.ref_modif);
		break;
	}

	return data;
}

/**
 * mg_custom_layout_data_free
 * @data: a #MgCustomLayoutData structure
 *
 * Ensures that the @data allocated through mg_custom_layout_get_data() is de-allocated properly.
 */
void
mg_custom_layout_data_free (MgCustomLayoutData *data)
{
	if (data->type == MG_CUSTOM_LAYOUT_LAYOUT) {
		GSList *list = data->contents.layout.connects;
		while (list) {
			g_free (list->data);
			list = g_slist_next (list);
		}
		g_slist_free (data->contents.layout.connects);
	}

	g_free (data);
}

/* 
 * MgXmlStorage interface implementation
 */
static gchar *
mg_custom_layout_get_xml_id (MgXmlStorage *iface)
{
	g_return_val_if_fail (iface && IS_MG_CUSTOM_LAYOUT (iface), NULL);
	g_return_val_if_fail (MG_CUSTOM_LAYOUT (iface)->priv, NULL);

	return g_strdup_printf ("CL%d", mg_base_get_id (MG_BASE (iface)));
}

static xmlNodePtr
mg_custom_layout_save_to_xml (MgXmlStorage *iface, GError **error)
{
	xmlNodePtr node = NULL;
	MgCustomLayout *cl;
	GSList *list;
	MgBase *ref;
	gchar *str;
	const gchar *cstr;
	xmlNodePtr query_ref_node = NULL;

	g_return_val_if_fail (iface && IS_MG_CUSTOM_LAYOUT (iface), NULL);
	g_return_val_if_fail (MG_CUSTOM_LAYOUT (iface)->priv, NULL);

	cl = MG_CUSTOM_LAYOUT (iface);

	/* main XML node */
	switch (cl->priv->type) {
	case MG_CUSTOM_LAYOUT_LAYOUT:
		node = xmlNewNode (NULL, "MG_CUSTOM_LAYOUT");
		break;
	case MG_CUSTOM_LAYOUT_GRID:
		node = xmlNewNode (NULL, "MG_CUSTOM_GRID");
		break;
	case MG_CUSTOM_LAYOUT_FORM:
		node = xmlNewNode (NULL, "MG_CUSTOM_FORM");
		break;
	case MG_CUSTOM_LAYOUT_MATRIX:
		node = xmlNewNode (NULL, "MG_CUSTOM_MATRIX");
		break;
	}
	
	str = mg_xml_storage_get_xml_id (MG_XML_STORAGE (iface));
	xmlSetProp (node, "id", str);
	g_free (str);
	
	cstr = mg_base_get_name (MG_BASE (cl));
	if (cstr && *cstr)
		xmlSetProp (node, "name", cstr);
	cstr = mg_base_get_description (MG_BASE (cl));
	if (cstr && *cstr)
		xmlSetProp (node, "descr", cstr);

	/* contents */
	switch (cl->priv->type) {
	case MG_CUSTOM_LAYOUT_LAYOUT:
		list = cl->priv->contents.layout.children;
		while (list) {
			xmlNodePtr child = mg_xml_storage_save_to_xml (MG_XML_STORAGE (list->data), error);
			if (!child)
				return NULL;
			xmlAddChild (node, child);
			list = g_slist_next (list);
		}
		break;
	case MG_CUSTOM_LAYOUT_MATRIX:
		if (cl->priv->contents.work_iface.query_extra) {
			xmlNodePtr child = mg_xml_storage_save_to_xml (MG_XML_STORAGE (cl->priv->contents.work_iface.query_extra),
								       error);
			if (!child)
				return NULL;
			xmlAddChild (node, child);
		}
		ref = mg_ref_base_get_ref_object (cl->priv->contents.work_iface.ref_query_extra);
		if (!ref) {
			g_set_error (error,
				     MG_CUSTOM_LAYOUT_ERROR,
				     MG_CUSTOM_LAYOUT_XML_SAVE_ERROR,
				     _("Can't find referenced query"));
			return NULL;
		}
		if (ref != (MgBase *) cl->priv->contents.work_iface.query_extra) {
			xmlNodePtr child = xmlNewChild (node, NULL, "MG_QUERY_REF", NULL);
			xmlSetProp (child, "id_query", mg_xml_storage_get_xml_id (MG_XML_STORAGE (ref)));
			query_ref_node = child;
		}

		ref = mg_ref_base_get_ref_object (cl->priv->contents.work_iface.ref_rows_target);
		str = mg_xml_storage_get_xml_id (MG_XML_STORAGE (ref));
		xmlSetProp (node, "rows_target", str);
		g_free (str);

		ref = mg_ref_base_get_ref_object (cl->priv->contents.work_iface.ref_cols_target);
		str = mg_xml_storage_get_xml_id (MG_XML_STORAGE (ref));
		xmlSetProp (node, "cols_target", str);
		g_free (str);

		switch (cl->priv->contents.work_iface.view_type) {
		case MG_WORK_MATRIX_TABULAR_SYNTHETIC:
			str = "TS";
			break;
		case MG_WORK_MATRIX_LIST_DETAILLED:
			str = "LD";
			break;
		case MG_WORK_MATRIX_LIST_SYNTHETIC:
			str = "LS";
			break;
		default:
			g_assert_not_reached ();
			break;
		}
		xmlSetProp (node, "view_type", str);
	case MG_CUSTOM_LAYOUT_GRID:
	case MG_CUSTOM_LAYOUT_FORM:
		if (cl->priv->contents.work_iface.query) {
			xmlNodePtr child = mg_xml_storage_save_to_xml (MG_XML_STORAGE (cl->priv->contents.work_iface.query),
								       error);
			if (!child)
				return NULL;
			xmlAddChild (node, child);
		}
		ref = mg_ref_base_get_ref_object (cl->priv->contents.work_iface.ref_query);
		if (!ref) {
			g_set_error (error,
				     MG_CUSTOM_LAYOUT_ERROR,
				     MG_CUSTOM_LAYOUT_XML_SAVE_ERROR,
				     _("Can't find referenced query"));
			return NULL;
		}
		if (ref != (MgBase *) cl->priv->contents.work_iface.query) {
			xmlNodePtr child;
			str = mg_xml_storage_get_xml_id (MG_XML_STORAGE (ref));
			if (query_ref_node) {
				child = xmlNewNode (NULL, "MG_QUERY_REF");
				xmlSetProp (child, "id_query", str);
				xmlAddPrevSibling (query_ref_node, child);
			}
			else {
				child = xmlNewChild (node, NULL, "MG_QUERY_REF", NULL);
				xmlSetProp (child, "id_query", str);
			}
			g_free (str);
		}
		
		if (cl->priv->contents.work_iface.ref_modif) {
			ref = mg_ref_base_get_ref_object (cl->priv->contents.work_iface.ref_modif);
			if (!ref) {
				g_set_error (error,
					     MG_CUSTOM_LAYOUT_ERROR,
					     MG_CUSTOM_LAYOUT_XML_SAVE_ERROR,
					     _("Can't find referenced modified target"));
				return NULL;
			}
			if (cl->priv->type == MG_CUSTOM_LAYOUT_MATRIX)
				xmlSetProp (node, "modif_table", mg_xml_storage_get_xml_id (MG_XML_STORAGE (ref)));
			else
				xmlSetProp (node, "modif_target", mg_xml_storage_get_xml_id (MG_XML_STORAGE (ref)));
		}
		str = g_strdup_printf ("%d", cl->priv->contents.work_iface.mode);
		xmlSetProp (node, "mode", str);
		g_free (str);
		break;
	}

	/* Glade related */
	if (cl->priv->filename && cl->priv->root_widget && cl->priv->boxes) {
		xmlNodePtr glade;
		xmlNodePtr box, dest;
		GSList *dest_list;

		glade = xmlNewChild (node, NULL, "MG_GLADE", NULL);
		xmlSetProp (glade, "filename", cl->priv->filename);
		xmlSetProp (glade, "root_widget", cl->priv->root_widget);
		list = cl->priv->boxes;
		while (list) {
			box =  xmlNewChild (glade, NULL, "MG_GLADE_BOX", NULL);
			xmlSetProp (box, "box_name", GLADE_BOX (list->data)->box_name);
			xmlSetProp (box, "show_actions", GLADE_BOX (list->data)->show_actions ? "t" : "f");
			dest_list = GLADE_BOX (list->data)->dest_list;
			while (dest_list) {
				ref = mg_ref_base_get_ref_object (MG_REF_BASE (dest_list->data));
				if (!ref) {
					g_set_error (error,
						     MG_CUSTOM_LAYOUT_ERROR,
						     MG_CUSTOM_LAYOUT_XML_SAVE_ERROR,
						     _("Can't find referenced object for Glade box"));
					return NULL;
				}
				dest = xmlNewChild (box, NULL, "MG_GLADE_DEST", NULL);
				xmlSetProp (dest, "object", mg_xml_storage_get_xml_id (MG_XML_STORAGE (ref)));
				dest_list = g_slist_next (dest_list);
			}

			list = g_slist_next (list);
		}
	}


	/* Connects if applicable */
	if (cl->priv->type == MG_CUSTOM_LAYOUT_LAYOUT) {
		if (cl->priv->contents.layout.connects) {
			list = cl->priv->contents.layout.connects;
			while (list) {
				xmlNodePtr lcnode;
				lcnode = xmlNewChild (node, NULL, "MG_LAYOUT_CONNECT", NULL);

				xmlSetProp (lcnode, "src_layout", 
					    mg_ref_base_get_ref_name (LAYOUT_CONNECT (list->data)->src_layout, NULL, NULL));
				xmlSetProp (lcnode, "src_field", 
					    mg_ref_base_get_ref_name (LAYOUT_CONNECT (list->data)->src_field, NULL, NULL));
				xmlSetProp (lcnode, "dest_layout", 
					    mg_ref_base_get_ref_name (LAYOUT_CONNECT (list->data)->dest_layout, NULL, NULL));
				xmlSetProp (lcnode, "dest_field", 
					    mg_ref_base_get_ref_name (LAYOUT_CONNECT (list->data)->dest_field, NULL, NULL));
				list = g_slist_next (list);
			}
		}
	}

	return node;
}

static gboolean
mg_custom_layout_load_from_xml (MgXmlStorage *iface, xmlNodePtr node, GError **error)
{
	MgCustomLayout *cl;
	gchar *prop;
	xmlNodePtr children;
	guint mode = 0;
	gboolean mode_found = FALSE;
	MgConf *conf;

	g_return_val_if_fail (iface && IS_MG_CUSTOM_LAYOUT (iface), FALSE);
	g_return_val_if_fail (MG_CUSTOM_LAYOUT (iface)->priv, FALSE);
	g_return_val_if_fail (node, FALSE);

	cl = MG_CUSTOM_LAYOUT (iface);
	conf = mg_base_get_conf (MG_BASE (cl));

	/* type of custom layout */
	if (!strcmp (node->name, "MG_CUSTOM_LAYOUT")) {
		mode = MG_CUSTOM_LAYOUT_LAYOUT;
		mode_found = TRUE;
	}
	if (!mode_found && !strcmp (node->name, "MG_CUSTOM_GRID")) {
		mode = MG_CUSTOM_LAYOUT_GRID;
		mode_found = TRUE;
	}
	if (!mode_found && !strcmp (node->name, "MG_CUSTOM_FORM")) {
		mode = MG_CUSTOM_LAYOUT_FORM;
		mode_found = TRUE;
	}
	if (!mode_found && !strcmp (node->name, "MG_CUSTOM_MATRIX")) {
		mode = MG_CUSTOM_LAYOUT_MATRIX;
		mode_found = TRUE;
	}

	if (!mode_found) {
		g_set_error (error,
			     MG_CUSTOM_LAYOUT_ERROR,
			     MG_CUSTOM_LAYOUT_XML_LOAD_ERROR,
			     _("XML Tag is not <MG_CUSTOM_LAYOUT> or <MG_CUSTOM_GRID> or "
			       "<MG_CUSTOM_FORM> or <MG_CUSTOM_MATRIX>"));
		return FALSE;
	}
	cl->priv->type = mode;

	/* name and description */
	prop = xmlGetProp (node, "name");
	if (prop) {
		mg_base_set_name (MG_BASE (cl), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "descr");
	if (prop) {
		mg_base_set_description (MG_BASE (cl), prop);
		g_free (prop);
	}

	if ((cl->priv->type == MG_CUSTOM_LAYOUT_MATRIX) ||
	    (cl->priv->type == MG_CUSTOM_LAYOUT_GRID) ||
	    (cl->priv->type == MG_CUSTOM_LAYOUT_FORM)) {
		prop = xmlGetProp (node, "mode");
		if (prop) {
			cl->priv->contents.work_iface.mode = atoi (prop);
			g_free (prop);
		}

		if (cl->priv->type == MG_CUSTOM_LAYOUT_MATRIX)
			prop = xmlGetProp (node, "modif_table");
		else
			prop = xmlGetProp (node, "modif_target");
		if (prop) {
			cl->priv->contents.work_iface.ref_modif = MG_REF_BASE (mg_ref_base_new (conf));
			mg_ref_base_set_ref_name (cl->priv->contents.work_iface.ref_modif, 0, REFERENCE_BY_XML_ID, prop);
			g_free (prop);
			cl->priv->referer_objects = g_slist_prepend (cl->priv->referer_objects, 
								     cl->priv->contents.work_iface.ref_modif);
		}

		if (cl->priv->type == MG_CUSTOM_LAYOUT_MATRIX) {
			prop = xmlGetProp (node, "rows_target");
			g_assert (prop); /* normally done through DTD */
			if (prop) {
				cl->priv->contents.work_iface.ref_rows_target = MG_REF_BASE (mg_ref_base_new (conf));	
				mg_ref_base_set_ref_name (cl->priv->contents.work_iface.ref_rows_target, 0, 
							  REFERENCE_BY_XML_ID, prop);
				g_free (prop);
			}
			prop = xmlGetProp (node, "cols_target");
			g_assert (prop); /* normally done through DTD */
			if (prop) {
				cl->priv->contents.work_iface.ref_cols_target = MG_REF_BASE (mg_ref_base_new (conf));	
				mg_ref_base_set_ref_name (cl->priv->contents.work_iface.ref_cols_target, 0, 
							  REFERENCE_BY_XML_ID, prop);
				g_free (prop);
			}
			prop = xmlGetProp (node, "view_type");
			if (prop) {
				if ((*prop == 'T') && (*(prop+1) == 'S'))
					cl->priv->contents.work_iface.view_type = MG_WORK_MATRIX_TABULAR_SYNTHETIC;
				if ((*prop == 'L') && (*(prop+1) == 'S'))
					cl->priv->contents.work_iface.view_type = MG_WORK_MATRIX_LIST_SYNTHETIC;
				if ((*prop == 'L') && (*(prop+1) == 'D'))
					cl->priv->contents.work_iface.view_type = MG_WORK_MATRIX_LIST_DETAILLED;
				g_free (prop);
			}
		}
	}

	/* children nodes */
	children = node->children;
	while (children) {
		gboolean done = FALSE;

		/* a query */
		if (!strcmp (children->name, "MG_QUERY")) {
			MgQuery *query = (MgQuery *) mg_query_new (conf);
			if (!mg_xml_storage_load_from_xml (MG_XML_STORAGE (query), children, error))
				return FALSE;
			if (! cl->priv->contents.work_iface.query) {
				cl->priv->contents.work_iface.query = query;
				cl->priv->contents.work_iface.ref_query = MG_REF_BASE (mg_ref_base_new (conf));
				mg_ref_base_set_ref_object (cl->priv->contents.work_iface.ref_query, MG_BASE (query));
				cl->priv->referer_objects = g_slist_prepend (cl->priv->referer_objects, 
									     cl->priv->contents.work_iface.ref_query);    
			}
			else {
				cl->priv->contents.work_iface.query_extra = query;
				cl->priv->contents.work_iface.ref_query_extra =  MG_REF_BASE (mg_ref_base_new (conf));
				mg_ref_base_set_ref_object (cl->priv->contents.work_iface.ref_query_extra, MG_BASE (query));
				cl->priv->referer_objects = g_slist_prepend (cl->priv->referer_objects, 
									     cl->priv->contents.work_iface.ref_query_extra);   
			}
			done = TRUE;
		}

		/* a query reference */
		if (!done && !strcmp (children->name, "MG_QUERY_REF")) {
			prop = xmlGetProp (children, "id_query");
			if (!prop) {
				g_set_error (error,
					     MG_CUSTOM_LAYOUT_ERROR,
					     MG_CUSTOM_LAYOUT_XML_LOAD_ERROR,
					     _("XML Tag <MG_QUERY_REF> has no \"id_query\" property"));
				return FALSE;
			}
			if (! cl->priv->contents.work_iface.ref_query) {
				cl->priv->contents.work_iface.ref_query = MG_REF_BASE (mg_ref_base_new (conf));
				mg_ref_base_set_ref_name (cl->priv->contents.work_iface.ref_query, 0, 
							  REFERENCE_BY_XML_ID, prop);
				cl->priv->referer_objects = g_slist_prepend (cl->priv->referer_objects, 
									     cl->priv->contents.work_iface.ref_query);	     
			}
			else {
				cl->priv->contents.work_iface.ref_query_extra = MG_REF_BASE (mg_ref_base_new (conf));
				mg_ref_base_set_ref_name (cl->priv->contents.work_iface.ref_query_extra, 0, 
							  REFERENCE_BY_XML_ID, prop);
				cl->priv->referer_objects = g_slist_prepend (cl->priv->referer_objects, 
									     cl->priv->contents.work_iface.ref_query_extra);
			}
			g_free (prop);
			done = TRUE;
		}

		/* Glade */
		if (!done && !strcmp (children->name, "MG_GLADE")) {
			xmlNodePtr boxes;

			prop = xmlGetProp (children, "filename");
			cl->priv->filename = prop;
			prop = xmlGetProp (children, "root_widget");
			cl->priv->root_widget = prop;
			
			boxes = children->children;
			while (boxes) {
				if (!strcmp (boxes->name, "MG_GLADE_BOX")) {
					GladeBox *gb;
					xmlNodePtr dests;
					
					prop = xmlGetProp (boxes, "box_name");
					gb = glade_box_new (prop, TRUE);
					cl->priv->boxes = g_slist_prepend (cl->priv->boxes, gb);
					g_free (prop);
					prop = xmlGetProp (boxes, "show_actions");
					if (prop) {
						gb->show_actions = (*prop == 't');
						g_free (prop);
					}
					
					dests = boxes->children;
					while (dests) {
						if (!strcmp (dests->name, "MG_GLADE_DEST")) {
							MgRefBase *ref = (MgRefBase *) mg_ref_base_new (conf);
							prop = xmlGetProp (dests, "object");
							mg_ref_base_set_ref_name (ref, 0, REFERENCE_BY_XML_ID, prop);
							g_free (prop);
							gb->dest_list = g_slist_prepend (gb->dest_list, ref);
							cl->priv->referer_objects = 
								g_slist_prepend (cl->priv->referer_objects, ref);
						}
						dests = dests->next;
					}
					gb->dest_list = g_slist_reverse (gb->dest_list);
				}
				boxes = boxes->next;
			}
			cl->priv->boxes = g_slist_reverse (cl->priv->boxes);
			
			done = TRUE;
		}

		/* connects */
		if (!done && !strcmp (children->name, "MG_LAYOUT_CONNECT")) {
			LayoutConnect *lc;

			if (cl->priv->type != MG_CUSTOM_LAYOUT_LAYOUT) {
				g_set_error (error,
					     MG_CUSTOM_LAYOUT_ERROR,
					     MG_CUSTOM_LAYOUT_XML_LOAD_ERROR,
					     _("This Custom Layout can't have a <MG_LAYOUT_CONNECT> child"));
				return FALSE;
			}
			
			lc = layout_connect_new (cl, conf);
			prop = xmlGetProp (children, "src_layout");
			g_assert (prop); /* normally done through DTD */
			mg_ref_base_set_ref_name (lc->src_layout, MG_CUSTOM_LAYOUT_TYPE, REFERENCE_BY_XML_ID, prop);
			g_free (prop);

			prop = xmlGetProp (children, "src_field");
			g_assert (prop); /* normally done through DTD */
			mg_ref_base_set_ref_name (lc->src_field, MG_FIELD_TYPE, REFERENCE_BY_XML_ID, prop);
			g_free (prop);

			prop = xmlGetProp (children, "dest_layout");
			g_assert (prop); /* normally done through DTD */
			mg_ref_base_set_ref_name (lc->dest_layout, MG_CUSTOM_LAYOUT_TYPE, REFERENCE_BY_XML_ID, prop);
			g_free (prop);

			prop = xmlGetProp (children, "dest_field");
			g_assert (prop); /* normally done through DTD */
			mg_ref_base_set_ref_name (lc->dest_field, MG_FIELD_TYPE, REFERENCE_BY_XML_ID, prop);
			g_free (prop);

			cl->priv->contents.layout.connects = g_slist_append (cl->priv->contents.layout.connects, lc);

			done = TRUE;
		}

		/* sub layouts */
		if (!done && 
		    (!strcmp (children->name, "MG_CUSTOM_LAYOUT") ||
		     !strcmp (children->name, "MG_CUSTOM_GRID") ||
		     !strcmp (children->name, "MG_CUSTOM_FORM") ||
		     !strcmp (children->name, "MG_CUSTOM_MATRIX"))) {
			MgCustomLayout *scl = (MgCustomLayout *) mg_custom_layout_new (conf);
			if (!mg_xml_storage_load_from_xml (MG_XML_STORAGE (scl), children, error))
				return FALSE;
			cl->priv->contents.layout.children = g_slist_append (cl->priv->contents.layout.children, scl);
			done = TRUE;
		}
		
		children = children->next;
	}

	mg_referer_activate (MG_REFERER (iface));
	return TRUE;
}


/*
 * Functions dealing with GladeBox structures
 */
static GladeBox *
glade_box_new (const gchar *name, gboolean show_actions)
{
	GladeBox *gb = g_new0 (GladeBox, 1);
	gb->box_name = g_strdup (name);
	gb->show_actions = show_actions;

	return gb;
}

static void
glade_box_destroy (GladeBox *box)
{
	GSList *list;

	g_free (box->box_name);
	list = box->dest_list;
	while (list) {
		g_object_unref (list->data);
		list = g_slist_next (list);
	}
	g_slist_free (box->dest_list);
	g_free (box);
}

/* 
 * MgReferer interface implementation
 */
static gboolean
mg_custom_layout_activate (MgReferer *iface)
{
	GSList *list;
	gboolean active = TRUE;

	g_return_val_if_fail (iface && IS_MG_CUSTOM_LAYOUT (iface), FALSE);
	g_return_val_if_fail (MG_CUSTOM_LAYOUT (iface)->priv, FALSE);

	list = MG_CUSTOM_LAYOUT (iface)->priv->referer_objects;
	while (list) {
		active = mg_ref_base_activate (MG_REF_BASE (list->data)) && active;
		list = g_slist_next (list);
	}

	return active;
}

static void
mg_custom_layout_deactivate (MgReferer *iface)
{
	GSList *list;

	g_return_if_fail (iface && IS_MG_CUSTOM_LAYOUT (iface));
	g_return_if_fail (MG_CUSTOM_LAYOUT (iface)->priv);

	list = MG_CUSTOM_LAYOUT (iface)->priv->referer_objects;
	while (list) {
		mg_ref_base_deactivate (MG_REF_BASE (list->data));
		list = g_slist_next (list);
	}
}

static gboolean
mg_custom_layout_is_active (MgReferer *iface)
{
	GSList *list;
	gboolean active = TRUE;

	g_return_val_if_fail (iface && IS_MG_CUSTOM_LAYOUT (iface), FALSE);
	g_return_val_if_fail (MG_CUSTOM_LAYOUT (iface)->priv, FALSE);

	list = MG_CUSTOM_LAYOUT (iface)->priv->referer_objects;
	while (list && active) {
		active = mg_ref_base_is_active (MG_REF_BASE (list->data));
		list = g_slist_next (list);
	}

	return active;
}

static GSList *
mg_custom_layout_get_ref_objects (MgReferer *iface)
{
	GSList *list = NULL;
	MgBase *base;

	g_return_val_if_fail (iface && IS_MG_CUSTOM_LAYOUT (iface), NULL);
	g_return_val_if_fail (MG_CUSTOM_LAYOUT (iface)->priv, NULL);

	list = MG_CUSTOM_LAYOUT (iface)->priv->referer_objects;
	while (list) {
		base = mg_ref_base_get_ref_object (MG_REF_BASE (list->data));
		if (base)
			list = g_slist_append (list, base);
		list = g_slist_next (list);
	}

	return list;
}

static void
mg_custom_layout_replace_refs (MgReferer *iface, GHashTable *replacements)
{
	GSList *list = NULL;

	g_return_if_fail (iface && IS_MG_CUSTOM_LAYOUT (iface));
	g_return_if_fail (MG_CUSTOM_LAYOUT (iface)->priv);

	list = MG_CUSTOM_LAYOUT (iface)->priv->referer_objects;
	while (list) {
		mg_ref_base_replace_ref_object (MG_REF_BASE (list->data), replacements);
		list = g_slist_next (list);
	}
}


/*
 * Functions dealing with LayoutConnect structures
 */

static LayoutConnect *
layout_connect_new (MgCustomLayout *layout, MgConf *conf)
{
	LayoutConnect *lc;

	lc = g_new0 (LayoutConnect, 1);
	lc->src_layout = (MgRefBase *) mg_ref_base_new (conf);
	lc->src_field = (MgRefBase *) mg_ref_base_new (conf);
	lc->dest_layout = (MgRefBase *) mg_ref_base_new (conf);
	lc->dest_field = (MgRefBase *) mg_ref_base_new (conf);

	layout->priv->referer_objects = g_slist_prepend (layout->priv->referer_objects, lc->dest_field);
	layout->priv->referer_objects = g_slist_prepend (layout->priv->referer_objects, lc->dest_layout);
	layout->priv->referer_objects = g_slist_prepend (layout->priv->referer_objects, lc->src_field);
	layout->priv->referer_objects = g_slist_prepend (layout->priv->referer_objects, lc->src_layout);

	return lc;
}

static void
layout_connect_destroy (MgCustomLayout *layout, LayoutConnect *lc)
{
	layout->priv->referer_objects = g_slist_remove (layout->priv->referer_objects, lc->src_layout);
	layout->priv->referer_objects = g_slist_remove (layout->priv->referer_objects, lc->src_field);
	layout->priv->referer_objects = g_slist_remove (layout->priv->referer_objects, lc->dest_layout);
	layout->priv->referer_objects = g_slist_remove (layout->priv->referer_objects, lc->dest_field);

	g_object_unref (lc->src_layout);
	g_object_unref (lc->src_field);
	g_object_unref (lc->dest_layout);
	g_object_unref (lc->dest_field);
	
	g_free (lc);
}
