// DoubleFormatterTest.cs - NUnit Test Cases for System.DoubleFormatter
//
// Author:
//     Patrick Kalkman  kalkman@cistron.nl
//
// (C) 2003 Patrick Kalkman
// 
using NUnit.Framework;
using System;
using System.Threading;
using System.Globalization;

namespace MonoTests.System {
	
	[TestFixture]
	public class DoubleFormatterTest : Assertion
	{
		[Test]
		[ExpectedException(typeof(FormatException))]
		public void TestToDecimal()
		{
			Double x = 1.0000001F;
			string Result = x.ToString ("D2"); //To Decimal is for integral types only. 
		}
		
		[Test]
		[ExpectedException(typeof(FormatException))]
		public void TestToHex()
		{
			Double x = 1.212121F;
			string Result = x.ToString ("X2"); //To Hex is for integral types only. 
		}

		[Test]
		[ExpectedException(typeof(FormatException))]
		public void TestToUnknown()
		{
			Double x = 1.212121F;
			string Result = x.ToString ("L2"); //Invalid format. 
		}
		
		private void FormatStringTest(string TestNumber, NumberFormatInfo NumberFormat, double Number, string Format, string ExpectedResult)
		{
			Assertion.AssertEquals (TestNumber, ExpectedResult, Number.ToString(Format, NumberFormat));                                
		}

		private NumberFormatInfo GetNumberFormat1()
		{
			NumberFormatInfo format = new NumberFormatInfo();
			
			format.NaNSymbol = "NaN";
			format.PositiveSign = "+";
			format.NegativeSign = "-";
			format.PerMilleSymbol = "x";
			format.PositiveInfinitySymbol = "Infinity";
			format.NegativeInfinitySymbol = "-Infinity";
			
			format.NumberDecimalDigits = 5; 
			format.NumberDecimalSeparator = ",";
			format.NumberGroupSeparator = ".";
			format.NumberGroupSizes = new int[] {3};
			format.NumberNegativePattern = 2;
			
			format.CurrencyDecimalDigits = 2;
			format.CurrencyDecimalSeparator = ",";
			format.CurrencyGroupSeparator = ".";
			format.CurrencyGroupSizes = new int[] {3};
			format.CurrencyNegativePattern = 8;
			format.CurrencyPositivePattern = 3;
			format.CurrencySymbol = "EUR";
			
			format.PercentDecimalDigits = 5; 
			format.PercentDecimalSeparator = ",";
			format.PercentGroupSeparator = ".";
			format.PercentGroupSizes = new int[] {3};
			format.PercentNegativePattern = 0;
			format.PercentPositivePattern = 0;
			format.PercentSymbol = "%";
			
			return format;
		}
		
		private NumberFormatInfo GetNumberFormat2()
		{
			NumberFormatInfo format = new NumberFormatInfo();
			
			format.NaNSymbol = "Geen";
			format.PositiveSign = "+";
			format.NegativeSign = "-";
			format.PerMilleSymbol = "x";
			format.PositiveInfinitySymbol = "Oneindig";
			format.NegativeInfinitySymbol = "-Oneindig";
			
			format.NumberDecimalDigits = 2; 
			format.NumberDecimalSeparator = ".";
			format.NumberGroupSeparator = ",";
			format.NumberGroupSizes = new int[] {3};
			format.NumberNegativePattern = 1;
			
			format.CurrencyDecimalDigits = 1;
			format.CurrencyDecimalSeparator = ".";
			format.CurrencyGroupSeparator = ",";
			format.CurrencyGroupSizes = new int[] {3};
			format.CurrencyNegativePattern = 3;
			format.CurrencyPositivePattern = 1;
			format.CurrencySymbol = "$";
			
			format.PercentDecimalDigits = 2; 
			format.PercentDecimalSeparator = ".";
			format.PercentGroupSeparator = ",";
			format.PercentGroupSizes = new int[] {3};
			format.PercentNegativePattern = 1;
			format.PercentPositivePattern = 2;
			format.PercentSymbol = "##";
			
			return format;
		}

		[Test]
		public void TestFormatStringsN1()
		{
			NumberFormatInfo NumberFormat = GetNumberFormat1();			
			
			FormatStringTest ("DblFn1 #0", NumberFormat, 99999.9999996, "C", "100.000,00 EUR");
			FormatStringTest ("DblFn1 #1", NumberFormat, 99999.9999996, "C0", "100.000 EUR");  
			FormatStringTest ("DblFn1 #2", NumberFormat, 99999.9999996, "C1", "100.000,0 EUR");  
			FormatStringTest ("DblFn1 #3", NumberFormat, 99999.9999996, "C3", "100.000,000 EUR");  
			FormatStringTest ("DblFn1 #4", NumberFormat, 99999.9999996, "C4", "100.000,0000 EUR");  
			FormatStringTest ("DblFn1 #5", NumberFormat, 99999.9999996, "C6", "100.000,000000 EUR");  
			FormatStringTest ("DblFn1 #6", NumberFormat, 99999.9999996, "C9", "99.999,999999600 EUR");  
			FormatStringTest ("DblFn1 #7", NumberFormat, 99999.9999996, "C67", "99.999,9999996000000000000000000000000000000000000000000000000000000000000 EUR");  
			FormatStringTest ("DblFn1 #8", NumberFormat, 99999.9999996, "E", "1,000000E+005");  
			FormatStringTest ("DblFn1 #9", NumberFormat, 99999.9999996, "E0", "1E+005");  
			FormatStringTest ("DblFn1 #10", NumberFormat, 99999.9999996, "E1", "1,0E+005");  
			FormatStringTest ("DblFn1 #11", NumberFormat, 99999.9999996, "E3", "1,000E+005");  
			FormatStringTest ("DblFn1 #12", NumberFormat, 99999.9999996, "E4", "1,0000E+005");  
			FormatStringTest ("DblFn1 #13", NumberFormat, 99999.9999996, "E6", "1,000000E+005");  
			FormatStringTest ("DblFn1 #14", NumberFormat, 99999.9999996, "E9", "1,000000000E+005");  
			FormatStringTest ("DblFn1 #15", NumberFormat, 99999.9999996, "E67", "9,9999999999599997000000000000000000000000000000000000000000000000000E+004");  
			FormatStringTest ("DblFn1 #16", NumberFormat, 99999.9999996, "F", "100000,00000");  
			FormatStringTest ("DblFn1 #17", NumberFormat, 99999.9999996, "F0", "100000");  
			FormatStringTest ("DblFn1 #18", NumberFormat, 99999.9999996, "F1", "100000,0");  
			FormatStringTest ("DblFn1 #19", NumberFormat, 99999.9999996, "F3", "100000,000");  
			FormatStringTest ("DblFn1 #20", NumberFormat, 99999.9999996, "F4", "100000,0000");  
			FormatStringTest ("DblFn1 #21", NumberFormat, 99999.9999996, "F6", "100000,000000");  
			FormatStringTest ("DblFn1 #22", NumberFormat, 99999.9999996, "F9", "99999,999999600");  
			FormatStringTest ("DblFn1 #23", NumberFormat, 99999.9999996, "F67", "99999,9999996000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn1 #24", NumberFormat, 99999.9999996, "G", "99999,9999996");  
			FormatStringTest ("DblFn1 #25", NumberFormat, 99999.9999996, "G0", "99999,9999996");  
			FormatStringTest ("DblFn1 #26", NumberFormat, 99999.9999996, "G1", "1E+05");  
			FormatStringTest ("DblFn1 #27", NumberFormat, 99999.9999996, "G3", "1E+05");  
			FormatStringTest ("DblFn1 #28", NumberFormat, 99999.9999996, "G4", "1E+05");  
			FormatStringTest ("DblFn1 #29", NumberFormat, 99999.9999996, "G6", "100000");  
			FormatStringTest ("DblFn1 #30", NumberFormat, 99999.9999996, "G9", "100000");  
			FormatStringTest ("DblFn1 #31", NumberFormat, 99999.9999996, "G67", "99999,999999599997");  
			FormatStringTest ("DblFn1 #32", NumberFormat, 99999.9999996, "N", "100.000,00000");  
			FormatStringTest ("DblFn1 #33", NumberFormat, 99999.9999996, "N0", "100.000");  
			FormatStringTest ("DblFn1 #34", NumberFormat, 99999.9999996, "N1", "100.000,0");  
			FormatStringTest ("DblFn1 #35", NumberFormat, 99999.9999996, "N3", "100.000,000");  
			FormatStringTest ("DblFn1 #36", NumberFormat, 99999.9999996, "N4", "100.000,0000");  
			FormatStringTest ("DblFn1 #37", NumberFormat, 99999.9999996, "N6", "100.000,000000");  
			FormatStringTest ("DblFn1 #38", NumberFormat, 99999.9999996, "N9", "99.999,999999600");  
			FormatStringTest ("DblFn1 #39", NumberFormat, 99999.9999996, "N67", "99.999,9999996000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn1 #40", NumberFormat, 99999.9999996, "P", "9.999.999,99996 %");  
			FormatStringTest ("DblFn1 #41", NumberFormat, 99999.9999996, "P0", "10.000.000 %");  
			FormatStringTest ("DblFn1 #42", NumberFormat, 99999.9999996, "P1", "10.000.000,0 %");  
			FormatStringTest ("DblFn1 #43", NumberFormat, 99999.9999996, "P3", "10.000.000,000 %");  
			FormatStringTest ("DblFn1 #44", NumberFormat, 99999.9999996, "P4", "10.000.000,0000 %");  
			FormatStringTest ("DblFn1 #45", NumberFormat, 99999.9999996, "P6", "9.999.999,999960 %");  
			FormatStringTest ("DblFn1 #46", NumberFormat, 99999.9999996, "P9", "9.999.999,999960000 %");  
			FormatStringTest ("DblFn1 #47", NumberFormat, 99999.9999996, "P67", "9.999.999,9999600000000000000000000000000000000000000000000000000000000000000 %");  
			FormatStringTest ("DblFn1 #48", NumberFormat, 1.79769313486231E+307, "C", "17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,00 EUR");  
			FormatStringTest ("DblFn1 #49", NumberFormat, 1.79769313486231E+307, "C0", "17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000 EUR");  
			FormatStringTest ("DblFn1 #50", NumberFormat, 1.79769313486231E+307, "C1", "17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,0 EUR");  
			FormatStringTest ("DblFn1 #51", NumberFormat, 1.79769313486231E+307, "C3", "17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,000 EUR");  
			FormatStringTest ("DblFn1 #52", NumberFormat, 1.79769313486231E+307, "C4", "17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,0000 EUR");  
			FormatStringTest ("DblFn1 #53", NumberFormat, 1.79769313486231E+307, "C6", "17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,000000 EUR");  
			FormatStringTest ("DblFn1 #54", NumberFormat, 1.79769313486231E+307, "C9", "17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,000000000 EUR");  
			FormatStringTest ("DblFn1 #55", NumberFormat, 1.79769313486231E+307, "C67", "17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,0000000000000000000000000000000000000000000000000000000000000000000 EUR");  
			FormatStringTest ("DblFn1 #56", NumberFormat, 1.79769313486231E+307, "E", "1,797693E+307");  
			FormatStringTest ("DblFn1 #57", NumberFormat, 1.79769313486231E+307, "E0", "2E+307");  
			FormatStringTest ("DblFn1 #58", NumberFormat, 1.79769313486231E+307, "E1", "1,8E+307");  
			FormatStringTest ("DblFn1 #59", NumberFormat, 1.79769313486231E+307, "E3", "1,798E+307");  
			FormatStringTest ("DblFn1 #60", NumberFormat, 1.79769313486231E+307, "E4", "1,7977E+307");  
			FormatStringTest ("DblFn1 #61", NumberFormat, 1.79769313486231E+307, "E6", "1,797693E+307");  
			FormatStringTest ("DblFn1 #62", NumberFormat, 1.79769313486231E+307, "E9", "1,797693135E+307");  
			FormatStringTest ("DblFn1 #63", NumberFormat, 1.79769313486231E+307, "E67", "1,7976931348623100000000000000000000000000000000000000000000000000000E+307");  
			FormatStringTest ("DblFn1 #64", NumberFormat, 1.79769313486231E+307, "F", "17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,00000");  
			FormatStringTest ("DblFn1 #65", NumberFormat, 1.79769313486231E+307, "F0", "17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn1 #66", NumberFormat, 1.79769313486231E+307, "F1", "17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,0");  
			FormatStringTest ("DblFn1 #67", NumberFormat, 1.79769313486231E+307, "F3", "17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,000");  
			FormatStringTest ("DblFn1 #68", NumberFormat, 1.79769313486231E+307, "F4", "17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,0000");  
			FormatStringTest ("DblFn1 #69", NumberFormat, 1.79769313486231E+307, "F6", "17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,000000");  
			FormatStringTest ("DblFn1 #70", NumberFormat, 1.79769313486231E+307, "F9", "17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,000000000");  
			FormatStringTest ("DblFn1 #71", NumberFormat, 1.79769313486231E+307, "F67", "17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,0000000000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn1 #72", NumberFormat, 1.79769313486231E+307, "G", "1,79769313486231E+307");  
			FormatStringTest ("DblFn1 #73", NumberFormat, 1.79769313486231E+307, "G0", "1,79769313486231E+307");  
			FormatStringTest ("DblFn1 #74", NumberFormat, 1.79769313486231E+307, "G1", "2E+307");  
			FormatStringTest ("DblFn1 #75", NumberFormat, 1.79769313486231E+307, "G3", "1,8E+307");  
			FormatStringTest ("DblFn1 #76", NumberFormat, 1.79769313486231E+307, "G4", "1,798E+307");  
			FormatStringTest ("DblFn1 #77", NumberFormat, 1.79769313486231E+307, "G6", "1,79769E+307");  
			FormatStringTest ("DblFn1 #78", NumberFormat, 1.79769313486231E+307, "G9", "1,79769313E+307");  
			FormatStringTest ("DblFn1 #79", NumberFormat, 1.79769313486231E+307, "G67", "1,79769313486231E+307");  
			FormatStringTest ("DblFn1 #80", NumberFormat, 1.79769313486231E+307, "N", "17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,00000");  
			FormatStringTest ("DblFn1 #81", NumberFormat, 1.79769313486231E+307, "N0", "17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000");  
			FormatStringTest ("DblFn1 #82", NumberFormat, 1.79769313486231E+307, "N1", "17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,0");  
			FormatStringTest ("DblFn1 #83", NumberFormat, 1.79769313486231E+307, "N3", "17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,000");  
			FormatStringTest ("DblFn1 #84", NumberFormat, 1.79769313486231E+307, "N4", "17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,0000");  
			FormatStringTest ("DblFn1 #85", NumberFormat, 1.79769313486231E+307, "N6", "17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,000000");  
			FormatStringTest ("DblFn1 #86", NumberFormat, 1.79769313486231E+307, "N9", "17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,000000000");  
			FormatStringTest ("DblFn1 #87", NumberFormat, 1.79769313486231E+307, "N67", "17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,0000000000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn1 #88", NumberFormat, 1.79769313486231E+307, "P", "1.797.693.134.862.310.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,00000 %");  
			FormatStringTest ("DblFn1 #89", NumberFormat, 1.79769313486231E+307, "P0", "1.797.693.134.862.310.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000 %");  
			FormatStringTest ("DblFn1 #90", NumberFormat, 1.79769313486231E+307, "P1", "1.797.693.134.862.310.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,0 %");  
			FormatStringTest ("DblFn1 #91", NumberFormat, 1.79769313486231E+307, "P3", "1.797.693.134.862.310.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,000 %");  
			FormatStringTest ("DblFn1 #92", NumberFormat, 1.79769313486231E+307, "P4", "1.797.693.134.862.310.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,0000 %");  
			FormatStringTest ("DblFn1 #93", NumberFormat, 1.79769313486231E+307, "P6", "1.797.693.134.862.310.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,000000 %");  
			FormatStringTest ("DblFn1 #94", NumberFormat, 1.79769313486231E+307, "P9", "1.797.693.134.862.310.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,000000000 %");  
			FormatStringTest ("DblFn1 #95", NumberFormat, 1.79769313486231E+307, "P67", "1.797.693.134.862.310.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,0000000000000000000000000000000000000000000000000000000000000000000 %");  
			FormatStringTest ("DblFn1 #96", NumberFormat, -1.79769313486231E+307, "C", "-17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,00 EUR");  
			FormatStringTest ("DblFn1 #97", NumberFormat, -1.79769313486231E+307, "C0", "-17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000 EUR");  
			FormatStringTest ("DblFn1 #98", NumberFormat, -1.79769313486231E+307, "C1", "-17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,0 EUR");  
			FormatStringTest ("DblFn1 #99", NumberFormat, -1.79769313486231E+307, "C3", "-17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,000 EUR");  
			FormatStringTest ("DblFn1 #100", NumberFormat, -1.79769313486231E+307, "C4", "-17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,0000 EUR");  
			FormatStringTest ("DblFn1 #101", NumberFormat, -1.79769313486231E+307, "C6", "-17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,000000 EUR");  
			FormatStringTest ("DblFn1 #102", NumberFormat, -1.79769313486231E+307, "C9", "-17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,000000000 EUR");  
			FormatStringTest ("DblFn1 #103", NumberFormat, -1.79769313486231E+307, "C67", "-17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,0000000000000000000000000000000000000000000000000000000000000000000 EUR");  
			FormatStringTest ("DblFn1 #104", NumberFormat, -1.79769313486231E+307, "E", "-1,797693E+307");  
			FormatStringTest ("DblFn1 #105", NumberFormat, -1.79769313486231E+307, "E0", "-2E+307");  
			FormatStringTest ("DblFn1 #106", NumberFormat, -1.79769313486231E+307, "E1", "-1,8E+307");  
			FormatStringTest ("DblFn1 #107", NumberFormat, -1.79769313486231E+307, "E3", "-1,798E+307");  
			FormatStringTest ("DblFn1 #108", NumberFormat, -1.79769313486231E+307, "E4", "-1,7977E+307");  
			FormatStringTest ("DblFn1 #109", NumberFormat, -1.79769313486231E+307, "E6", "-1,797693E+307");  
			FormatStringTest ("DblFn1 #110", NumberFormat, -1.79769313486231E+307, "E9", "-1,797693135E+307");  
			FormatStringTest ("DblFn1 #111", NumberFormat, -1.79769313486231E+307, "E67", "-1,7976931348623100000000000000000000000000000000000000000000000000000E+307");  
			FormatStringTest ("DblFn1 #112", NumberFormat, -1.79769313486231E+307, "F", "-17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,00000");  
			FormatStringTest ("DblFn1 #113", NumberFormat, -1.79769313486231E+307, "F0", "-17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn1 #114", NumberFormat, -1.79769313486231E+307, "F1", "-17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,0");  
			FormatStringTest ("DblFn1 #115", NumberFormat, -1.79769313486231E+307, "F3", "-17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,000");  
			FormatStringTest ("DblFn1 #116", NumberFormat, -1.79769313486231E+307, "F4", "-17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,0000");  
			FormatStringTest ("DblFn1 #117", NumberFormat, -1.79769313486231E+307, "F6", "-17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,000000");  
			FormatStringTest ("DblFn1 #118", NumberFormat, -1.79769313486231E+307, "F9", "-17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,000000000");  
			FormatStringTest ("DblFn1 #119", NumberFormat, -1.79769313486231E+307, "F67", "-17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000,0000000000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn1 #120", NumberFormat, -1.79769313486231E+307, "G", "-1,79769313486231E+307");  
			FormatStringTest ("DblFn1 #121", NumberFormat, -1.79769313486231E+307, "G0", "-1,79769313486231E+307");  
			FormatStringTest ("DblFn1 #122", NumberFormat, -1.79769313486231E+307, "G1", "-2E+307");  
			FormatStringTest ("DblFn1 #123", NumberFormat, -1.79769313486231E+307, "G3", "-1,8E+307");  
			FormatStringTest ("DblFn1 #124", NumberFormat, -1.79769313486231E+307, "G4", "-1,798E+307");  
			FormatStringTest ("DblFn1 #125", NumberFormat, -1.79769313486231E+307, "G6", "-1,79769E+307");  
			FormatStringTest ("DblFn1 #126", NumberFormat, -1.79769313486231E+307, "G9", "-1,79769313E+307");  
			FormatStringTest ("DblFn1 #127", NumberFormat, -1.79769313486231E+307, "G67", "-1,79769313486231E+307");  
			FormatStringTest ("DblFn1 #128", NumberFormat, -1.79769313486231E+307, "N", "- 17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,00000");  
			FormatStringTest ("DblFn1 #129", NumberFormat, -1.79769313486231E+307, "N0", "- 17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000");  
			FormatStringTest ("DblFn1 #130", NumberFormat, -1.79769313486231E+307, "N1", "- 17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,0");  
			FormatStringTest ("DblFn1 #131", NumberFormat, -1.79769313486231E+307, "N3", "- 17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,000");  
			FormatStringTest ("DblFn1 #132", NumberFormat, -1.79769313486231E+307, "N4", "- 17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,0000");  
			FormatStringTest ("DblFn1 #133", NumberFormat, -1.79769313486231E+307, "N6", "- 17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,000000");  
			FormatStringTest ("DblFn1 #134", NumberFormat, -1.79769313486231E+307, "N9", "- 17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,000000000");  
			FormatStringTest ("DblFn1 #135", NumberFormat, -1.79769313486231E+307, "N67", "- 17.976.931.348.623.100.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,0000000000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn1 #136", NumberFormat, -1.79769313486231E+307, "P", "-1.797.693.134.862.310.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,00000 %");  
			FormatStringTest ("DblFn1 #137", NumberFormat, -1.79769313486231E+307, "P0", "-1.797.693.134.862.310.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000 %");  
			FormatStringTest ("DblFn1 #138", NumberFormat, -1.79769313486231E+307, "P1", "-1.797.693.134.862.310.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,0 %");  
			FormatStringTest ("DblFn1 #139", NumberFormat, -1.79769313486231E+307, "P3", "-1.797.693.134.862.310.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,000 %");  
			FormatStringTest ("DblFn1 #140", NumberFormat, -1.79769313486231E+307, "P4", "-1.797.693.134.862.310.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,0000 %");  
			FormatStringTest ("DblFn1 #141", NumberFormat, -1.79769313486231E+307, "P6", "-1.797.693.134.862.310.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,000000 %");  
			FormatStringTest ("DblFn1 #142", NumberFormat, -1.79769313486231E+307, "P9", "-1.797.693.134.862.310.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,000000000 %");  
			FormatStringTest ("DblFn1 #143", NumberFormat, -1.79769313486231E+307, "P67", "-1.797.693.134.862.310.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000.000,0000000000000000000000000000000000000000000000000000000000000000000 %");  
			FormatStringTest ("DblFn1 #144", NumberFormat, 1E-10, "C", "0,00 EUR");  
			FormatStringTest ("DblFn1 #145", NumberFormat, 1E-10, "C0", "0 EUR");  
			FormatStringTest ("DblFn1 #146", NumberFormat, 1E-10, "C1", "0,0 EUR");  
			FormatStringTest ("DblFn1 #147", NumberFormat, 1E-10, "C3", "0,000 EUR");  
			FormatStringTest ("DblFn1 #148", NumberFormat, 1E-10, "C4", "0,0000 EUR");  
			FormatStringTest ("DblFn1 #149", NumberFormat, 1E-10, "C6", "0,000000 EUR");  
			FormatStringTest ("DblFn1 #150", NumberFormat, 1E-10, "C9", "0,000000000 EUR");  
			FormatStringTest ("DblFn1 #151", NumberFormat, 1E-10, "C67", "0,0000000001000000000000000000000000000000000000000000000000000000000 EUR");  
			FormatStringTest ("DblFn1 #152", NumberFormat, 1E-10, "E", "1,000000E-010");  
			FormatStringTest ("DblFn1 #153", NumberFormat, 1E-10, "E0", "1E-010");  
			FormatStringTest ("DblFn1 #154", NumberFormat, 1E-10, "E1", "1,0E-010");  
			FormatStringTest ("DblFn1 #155", NumberFormat, 1E-10, "E3", "1,000E-010");  
			FormatStringTest ("DblFn1 #156", NumberFormat, 1E-10, "E4", "1,0000E-010");  
			FormatStringTest ("DblFn1 #157", NumberFormat, 1E-10, "E6", "1,000000E-010");  
			FormatStringTest ("DblFn1 #158", NumberFormat, 1E-10, "E9", "1,000000000E-010");  
			FormatStringTest ("DblFn1 #159", NumberFormat, 1E-10, "E67", "1,0000000000000000000000000000000000000000000000000000000000000000000E-010");  
			FormatStringTest ("DblFn1 #160", NumberFormat, 1E-10, "F", "0,00000");  
			FormatStringTest ("DblFn1 #161", NumberFormat, 1E-10, "F0", "0");  
			FormatStringTest ("DblFn1 #162", NumberFormat, 1E-10, "F1", "0,0");  
			FormatStringTest ("DblFn1 #163", NumberFormat, 1E-10, "F3", "0,000");  
			FormatStringTest ("DblFn1 #164", NumberFormat, 1E-10, "F4", "0,0000");  
			FormatStringTest ("DblFn1 #165", NumberFormat, 1E-10, "F6", "0,000000");  
			FormatStringTest ("DblFn1 #166", NumberFormat, 1E-10, "F9", "0,000000000");  
			FormatStringTest ("DblFn1 #167", NumberFormat, 1E-10, "F67", "0,0000000001000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn1 #168", NumberFormat, 1E-10, "G", "1E-10");  
			FormatStringTest ("DblFn1 #169", NumberFormat, 1E-10, "G0", "1E-10");  
			FormatStringTest ("DblFn1 #170", NumberFormat, 1E-10, "G1", "1E-10");  
			FormatStringTest ("DblFn1 #171", NumberFormat, 1E-10, "G3", "1E-10");  
			FormatStringTest ("DblFn1 #172", NumberFormat, 1E-10, "G4", "1E-10");  
			FormatStringTest ("DblFn1 #173", NumberFormat, 1E-10, "G6", "1E-10");  
			FormatStringTest ("DblFn1 #174", NumberFormat, 1E-10, "G9", "1E-10");  
			FormatStringTest ("DblFn1 #175", NumberFormat, 1E-10, "G67", "1E-10");  
			FormatStringTest ("DblFn1 #176", NumberFormat, 1E-10, "N", "0,00000");  
			FormatStringTest ("DblFn1 #177", NumberFormat, 1E-10, "N0", "0");  
			FormatStringTest ("DblFn1 #178", NumberFormat, 1E-10, "N1", "0,0");  
			FormatStringTest ("DblFn1 #179", NumberFormat, 1E-10, "N3", "0,000");  
			FormatStringTest ("DblFn1 #180", NumberFormat, 1E-10, "N4", "0,0000");  
			FormatStringTest ("DblFn1 #181", NumberFormat, 1E-10, "N6", "0,000000");  
			FormatStringTest ("DblFn1 #182", NumberFormat, 1E-10, "N9", "0,000000000");  
			FormatStringTest ("DblFn1 #183", NumberFormat, 1E-10, "N67", "0,0000000001000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn1 #184", NumberFormat, 1E-10, "P", "0,00000 %");  
			FormatStringTest ("DblFn1 #185", NumberFormat, 1E-10, "P0", "0 %");  
			FormatStringTest ("DblFn1 #186", NumberFormat, 1E-10, "P1", "0,0 %");  
			FormatStringTest ("DblFn1 #187", NumberFormat, 1E-10, "P3", "0,000 %");  
			FormatStringTest ("DblFn1 #188", NumberFormat, 1E-10, "P4", "0,0000 %");  
			FormatStringTest ("DblFn1 #189", NumberFormat, 1E-10, "P6", "0,000000 %");  
			FormatStringTest ("DblFn1 #190", NumberFormat, 1E-10, "P9", "0,000000010 %");  
			FormatStringTest ("DblFn1 #191", NumberFormat, 1E-10, "P67", "0,0000000100000000000000000000000000000000000000000000000000000000000 %");  
			FormatStringTest ("DblFn1 #192", NumberFormat, 4.94065645841247E-324, "C", "0,00 EUR");  
			FormatStringTest ("DblFn1 #193", NumberFormat, 4.94065645841247E-324, "C0", "0 EUR");  
			FormatStringTest ("DblFn1 #194", NumberFormat, 4.94065645841247E-324, "C1", "0,0 EUR");  
			FormatStringTest ("DblFn1 #195", NumberFormat, 4.94065645841247E-324, "C3", "0,000 EUR");  
			FormatStringTest ("DblFn1 #196", NumberFormat, 4.94065645841247E-324, "C4", "0,0000 EUR");  
			FormatStringTest ("DblFn1 #197", NumberFormat, 4.94065645841247E-324, "C6", "0,000000 EUR");  
			FormatStringTest ("DblFn1 #198", NumberFormat, 4.94065645841247E-324, "C9", "0,000000000 EUR");  
			FormatStringTest ("DblFn1 #199", NumberFormat, 4.94065645841247E-324, "C67", "0,0000000000000000000000000000000000000000000000000000000000000000000 EUR");  
			FormatStringTest ("DblFn1 #200", NumberFormat, 4.94065645841247E-324, "E", "4,940656E-324");  
			FormatStringTest ("DblFn1 #201", NumberFormat, 4.94065645841247E-324, "E0", "5E-324");  
			FormatStringTest ("DblFn1 #202", NumberFormat, 4.94065645841247E-324, "E1", "4,9E-324");  
			FormatStringTest ("DblFn1 #203", NumberFormat, 4.94065645841247E-324, "E3", "4,941E-324");  
			FormatStringTest ("DblFn1 #204", NumberFormat, 4.94065645841247E-324, "E4", "4,9407E-324");  
			FormatStringTest ("DblFn1 #205", NumberFormat, 4.94065645841247E-324, "E6", "4,940656E-324");  
			FormatStringTest ("DblFn1 #206", NumberFormat, 4.94065645841247E-324, "E9", "4,940656458E-324");  
			FormatStringTest ("DblFn1 #207", NumberFormat, 4.94065645841247E-324, "E67", "4,9406564584124654000000000000000000000000000000000000000000000000000E-324");  
			FormatStringTest ("DblFn1 #208", NumberFormat, 4.94065645841247E-324, "F", "0,00000");  
			FormatStringTest ("DblFn1 #209", NumberFormat, 4.94065645841247E-324, "F0", "0");  
			FormatStringTest ("DblFn1 #210", NumberFormat, 4.94065645841247E-324, "F1", "0,0");  
			FormatStringTest ("DblFn1 #211", NumberFormat, 4.94065645841247E-324, "F3", "0,000");  
			FormatStringTest ("DblFn1 #212", NumberFormat, 4.94065645841247E-324, "F4", "0,0000");  
			FormatStringTest ("DblFn1 #213", NumberFormat, 4.94065645841247E-324, "F6", "0,000000");  
			FormatStringTest ("DblFn1 #214", NumberFormat, 4.94065645841247E-324, "F9", "0,000000000");  
			FormatStringTest ("DblFn1 #215", NumberFormat, 4.94065645841247E-324, "F67", "0,0000000000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn1 #216", NumberFormat, 4.94065645841247E-324, "G", "4,94065645841247E-324");  
			FormatStringTest ("DblFn1 #217", NumberFormat, 4.94065645841247E-324, "G0", "4,94065645841247E-324");  
			FormatStringTest ("DblFn1 #218", NumberFormat, 4.94065645841247E-324, "G1", "5E-324");  
			FormatStringTest ("DblFn1 #219", NumberFormat, 4.94065645841247E-324, "G3", "4,94E-324");  
			FormatStringTest ("DblFn1 #220", NumberFormat, 4.94065645841247E-324, "G4", "4,941E-324");  
			FormatStringTest ("DblFn1 #221", NumberFormat, 4.94065645841247E-324, "G6", "4,94066E-324");  
			FormatStringTest ("DblFn1 #222", NumberFormat, 4.94065645841247E-324, "G9", "4,94065646E-324");  
			FormatStringTest ("DblFn1 #223", NumberFormat, 4.94065645841247E-324, "G67", "4,9406564584124654E-324");  
			FormatStringTest ("DblFn1 #224", NumberFormat, 4.94065645841247E-324, "N", "0,00000");  
			FormatStringTest ("DblFn1 #225", NumberFormat, 4.94065645841247E-324, "N0", "0");  
			FormatStringTest ("DblFn1 #226", NumberFormat, 4.94065645841247E-324, "N1", "0,0");  
			FormatStringTest ("DblFn1 #227", NumberFormat, 4.94065645841247E-324, "N3", "0,000");  
			FormatStringTest ("DblFn1 #228", NumberFormat, 4.94065645841247E-324, "N4", "0,0000");  
			FormatStringTest ("DblFn1 #229", NumberFormat, 4.94065645841247E-324, "N6", "0,000000");  
			FormatStringTest ("DblFn1 #230", NumberFormat, 4.94065645841247E-324, "N9", "0,000000000");  
			FormatStringTest ("DblFn1 #231", NumberFormat, 4.94065645841247E-324, "N67", "0,0000000000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn1 #232", NumberFormat, 4.94065645841247E-324, "P", "0,00000 %");  
			FormatStringTest ("DblFn1 #233", NumberFormat, 4.94065645841247E-324, "P0", "0 %");  
			FormatStringTest ("DblFn1 #234", NumberFormat, 4.94065645841247E-324, "P1", "0,0 %");  
			FormatStringTest ("DblFn1 #235", NumberFormat, 4.94065645841247E-324, "P3", "0,000 %");  
			FormatStringTest ("DblFn1 #236", NumberFormat, 4.94065645841247E-324, "P4", "0,0000 %");  
			FormatStringTest ("DblFn1 #237", NumberFormat, 4.94065645841247E-324, "P6", "0,000000 %");  
			FormatStringTest ("DblFn1 #238", NumberFormat, 4.94065645841247E-324, "P9", "0,000000000 %");  
			FormatStringTest ("DblFn1 #239", NumberFormat, 4.94065645841247E-324, "P67", "0,0000000000000000000000000000000000000000000000000000000000000000000 %");  
			FormatStringTest ("DblFn1 #240", NumberFormat, Double.NaN,  "C", "NaN");  
			FormatStringTest ("DblFn1 #241", NumberFormat, Double.NaN,  "C0", "NaN");  
			FormatStringTest ("DblFn1 #242", NumberFormat, Double.NaN,  "C1", "NaN");  
			FormatStringTest ("DblFn1 #243", NumberFormat, Double.NaN,  "C3", "NaN");  
			FormatStringTest ("DblFn1 #244", NumberFormat, Double.NaN,  "C4", "NaN");  
			FormatStringTest ("DblFn1 #245", NumberFormat, Double.NaN,  "C6", "NaN");  
			FormatStringTest ("DblFn1 #246", NumberFormat, Double.NaN,  "C9", "NaN");  
			FormatStringTest ("DblFn1 #247", NumberFormat, Double.NaN,  "C67", "NaN");  
			FormatStringTest ("DblFn1 #248", NumberFormat, Double.NaN,  "E", "NaN");  
			FormatStringTest ("DblFn1 #249", NumberFormat, Double.NaN,  "E0", "NaN");  
			FormatStringTest ("DblFn1 #250", NumberFormat, Double.NaN,  "E1", "NaN");  
			FormatStringTest ("DblFn1 #251", NumberFormat, Double.NaN,  "E3", "NaN");  
			FormatStringTest ("DblFn1 #252", NumberFormat, Double.NaN,  "E4", "NaN");  
			FormatStringTest ("DblFn1 #253", NumberFormat, Double.NaN,  "E6", "NaN");  
			FormatStringTest ("DblFn1 #254", NumberFormat, Double.NaN,  "E9", "NaN");  
			FormatStringTest ("DblFn1 #255", NumberFormat, Double.NaN,  "E67", "NaN");  
			FormatStringTest ("DblFn1 #256", NumberFormat, Double.NaN,  "F", "NaN");  
			FormatStringTest ("DblFn1 #257", NumberFormat, Double.NaN,  "F0", "NaN");  
			FormatStringTest ("DblFn1 #258", NumberFormat, Double.NaN,  "F1", "NaN");  
			FormatStringTest ("DblFn1 #259", NumberFormat, Double.NaN,  "F3", "NaN");  
			FormatStringTest ("DblFn1 #260", NumberFormat, Double.NaN,  "F4", "NaN");  
			FormatStringTest ("DblFn1 #261", NumberFormat, Double.NaN,  "F6", "NaN");  
			FormatStringTest ("DblFn1 #262", NumberFormat, Double.NaN,  "F9", "NaN");  
			FormatStringTest ("DblFn1 #263", NumberFormat, Double.NaN,  "F67", "NaN");  
			FormatStringTest ("DblFn1 #264", NumberFormat, Double.NaN,  "G", "NaN");  
			FormatStringTest ("DblFn1 #265", NumberFormat, Double.NaN,  "G0", "NaN");  
			FormatStringTest ("DblFn1 #266", NumberFormat, Double.NaN,  "G1", "NaN");  
			FormatStringTest ("DblFn1 #267", NumberFormat, Double.NaN,  "G3", "NaN");  
			FormatStringTest ("DblFn1 #268", NumberFormat, Double.NaN,  "G4", "NaN");  
			FormatStringTest ("DblFn1 #269", NumberFormat, Double.NaN,  "G6", "NaN");  
			FormatStringTest ("DblFn1 #270", NumberFormat, Double.NaN,  "G9", "NaN");  
			FormatStringTest ("DblFn1 #271", NumberFormat, Double.NaN,  "G67", "NaN");  
			FormatStringTest ("DblFn1 #272", NumberFormat, Double.NaN,  "N", "NaN");  
			FormatStringTest ("DblFn1 #273", NumberFormat, Double.NaN,  "N0", "NaN");  
			FormatStringTest ("DblFn1 #274", NumberFormat, Double.NaN,  "N1", "NaN");  
			FormatStringTest ("DblFn1 #275", NumberFormat, Double.NaN,  "N3", "NaN");  
			FormatStringTest ("DblFn1 #276", NumberFormat, Double.NaN,  "N4", "NaN");  
			FormatStringTest ("DblFn1 #277", NumberFormat, Double.NaN,  "N6", "NaN");  
			FormatStringTest ("DblFn1 #278", NumberFormat, Double.NaN,  "N9", "NaN");  
			FormatStringTest ("DblFn1 #279", NumberFormat, Double.NaN,  "N67", "NaN");  
			FormatStringTest ("DblFn1 #280", NumberFormat, Double.NaN,  "P", "NaN");  
			FormatStringTest ("DblFn1 #281", NumberFormat, Double.NaN,  "P0", "NaN");  
			FormatStringTest ("DblFn1 #282", NumberFormat, Double.NaN,  "P1", "NaN");  
			FormatStringTest ("DblFn1 #283", NumberFormat, Double.NaN,  "P3", "NaN");  
			FormatStringTest ("DblFn1 #284", NumberFormat, Double.NaN,  "P4", "NaN");  
			FormatStringTest ("DblFn1 #285", NumberFormat, Double.NaN,  "P6", "NaN");  
			FormatStringTest ("DblFn1 #286", NumberFormat, Double.NaN,  "P9", "NaN");  
			FormatStringTest ("DblFn1 #287", NumberFormat, Double.NaN,  "P67", "NaN"); 
			FormatStringTest ("DblFn1 #288", NumberFormat, Double.NegativeInfinity, "C", "-Infinity");  
			FormatStringTest ("DblFn1 #289", NumberFormat, Double.NegativeInfinity, "C0", "-Infinity");  
			FormatStringTest ("DblFn1 #290", NumberFormat, Double.NegativeInfinity, "C1", "-Infinity");  
			FormatStringTest ("DblFn1 #291", NumberFormat, Double.NegativeInfinity, "C3", "-Infinity");  
			FormatStringTest ("DblFn1 #292", NumberFormat, Double.NegativeInfinity, "C4", "-Infinity");  
			FormatStringTest ("DblFn1 #293", NumberFormat, Double.NegativeInfinity, "C6", "-Infinity");  
			FormatStringTest ("DblFn1 #294", NumberFormat, Double.NegativeInfinity, "C9", "-Infinity");  
			FormatStringTest ("DblFn1 #295", NumberFormat, Double.NegativeInfinity, "C67", "-Infinity");  
			FormatStringTest ("DblFn1 #296", NumberFormat, Double.NegativeInfinity, "E", "-Infinity");  
			FormatStringTest ("DblFn1 #297", NumberFormat, Double.NegativeInfinity, "E0", "-Infinity");  
			FormatStringTest ("DblFn1 #298", NumberFormat, Double.NegativeInfinity, "E1", "-Infinity");  
			FormatStringTest ("DblFn1 #299", NumberFormat, Double.NegativeInfinity, "E3", "-Infinity");  
			FormatStringTest ("DblFn1 #300", NumberFormat, Double.NegativeInfinity, "E4", "-Infinity");  
			FormatStringTest ("DblFn1 #301", NumberFormat, Double.NegativeInfinity, "E6", "-Infinity");  
			FormatStringTest ("DblFn1 #302", NumberFormat, Double.NegativeInfinity, "E9", "-Infinity");  
			FormatStringTest ("DblFn1 #303", NumberFormat, Double.NegativeInfinity, "E67", "-Infinity");  
			FormatStringTest ("DblFn1 #304", NumberFormat, Double.NegativeInfinity, "F", "-Infinity");  
			FormatStringTest ("DblFn1 #305", NumberFormat, Double.NegativeInfinity, "F0", "-Infinity");  
			FormatStringTest ("DblFn1 #306", NumberFormat, Double.NegativeInfinity, "F1", "-Infinity");  
			FormatStringTest ("DblFn1 #307", NumberFormat, Double.NegativeInfinity, "F3", "-Infinity");  
			FormatStringTest ("DblFn1 #308", NumberFormat, Double.NegativeInfinity, "F4", "-Infinity");  
			FormatStringTest ("DblFn1 #309", NumberFormat, Double.NegativeInfinity, "F6", "-Infinity");  
			FormatStringTest ("DblFn1 #310", NumberFormat, Double.NegativeInfinity, "F9", "-Infinity");  
			FormatStringTest ("DblFn1 #311", NumberFormat, Double.NegativeInfinity, "F67", "-Infinity");  
			FormatStringTest ("DblFn1 #312", NumberFormat, Double.NegativeInfinity, "G", "-Infinity");  
			FormatStringTest ("DblFn1 #313", NumberFormat, Double.NegativeInfinity, "G0", "-Infinity");  
			FormatStringTest ("DblFn1 #314", NumberFormat, Double.NegativeInfinity, "G1", "-Infinity");  
			FormatStringTest ("DblFn1 #315", NumberFormat, Double.NegativeInfinity, "G3", "-Infinity");  
			FormatStringTest ("DblFn1 #316", NumberFormat, Double.NegativeInfinity, "G4", "-Infinity");  
			FormatStringTest ("DblFn1 #317", NumberFormat, Double.NegativeInfinity, "G6", "-Infinity");  
			FormatStringTest ("DblFn1 #318", NumberFormat, Double.NegativeInfinity, "G9", "-Infinity");  
			FormatStringTest ("DblFn1 #319", NumberFormat, Double.NegativeInfinity, "G67", "-Infinity");  
			FormatStringTest ("DblFn1 #320", NumberFormat, Double.NegativeInfinity, "N", "-Infinity");  
			FormatStringTest ("DblFn1 #321", NumberFormat, Double.NegativeInfinity, "N0", "-Infinity");  
			FormatStringTest ("DblFn1 #322", NumberFormat, Double.NegativeInfinity, "N1", "-Infinity");  
			FormatStringTest ("DblFn1 #323", NumberFormat, Double.NegativeInfinity, "N3", "-Infinity");  
			FormatStringTest ("DblFn1 #324", NumberFormat, Double.NegativeInfinity, "N4", "-Infinity");  
			FormatStringTest ("DblFn1 #325", NumberFormat, Double.NegativeInfinity, "N6", "-Infinity");  
			FormatStringTest ("DblFn1 #326", NumberFormat, Double.NegativeInfinity, "N9", "-Infinity");  
			FormatStringTest ("DblFn1 #327", NumberFormat, Double.NegativeInfinity, "N67", "-Infinity");  
			FormatStringTest ("DblFn1 #328", NumberFormat, Double.NegativeInfinity, "P", "-Infinity");  
			FormatStringTest ("DblFn1 #329", NumberFormat, Double.NegativeInfinity, "P0", "-Infinity");  
			FormatStringTest ("DblFn1 #330", NumberFormat, Double.NegativeInfinity, "P1", "-Infinity");  
			FormatStringTest ("DblFn1 #331", NumberFormat, Double.NegativeInfinity, "P3", "-Infinity");  
			FormatStringTest ("DblFn1 #332", NumberFormat, Double.NegativeInfinity, "P4", "-Infinity");  
			FormatStringTest ("DblFn1 #333", NumberFormat, Double.NegativeInfinity, "P6", "-Infinity");  
			FormatStringTest ("DblFn1 #334", NumberFormat, Double.NegativeInfinity, "P9", "-Infinity");  
			FormatStringTest ("DblFn1 #335", NumberFormat, Double.NegativeInfinity, "P67", "-Infinity");  
			FormatStringTest ("DblFn1 #336", NumberFormat, Double.PositiveInfinity, "C", "Infinity");  
			FormatStringTest ("DblFn1 #337", NumberFormat, Double.PositiveInfinity, "C0", "Infinity");  
			FormatStringTest ("DblFn1 #338", NumberFormat, Double.PositiveInfinity, "C1", "Infinity");  
			FormatStringTest ("DblFn1 #339", NumberFormat, Double.PositiveInfinity, "C3", "Infinity");  
			FormatStringTest ("DblFn1 #340", NumberFormat, Double.PositiveInfinity, "C4", "Infinity");  
			FormatStringTest ("DblFn1 #341", NumberFormat, Double.PositiveInfinity, "C6", "Infinity");  
			FormatStringTest ("DblFn1 #342", NumberFormat, Double.PositiveInfinity, "C9", "Infinity");  
			FormatStringTest ("DblFn1 #343", NumberFormat, Double.PositiveInfinity, "C67", "Infinity");  
			FormatStringTest ("DblFn1 #344", NumberFormat, Double.PositiveInfinity, "E", "Infinity");  
			FormatStringTest ("DblFn1 #345", NumberFormat, Double.PositiveInfinity, "E0", "Infinity");  
			FormatStringTest ("DblFn1 #346", NumberFormat, Double.PositiveInfinity, "E1", "Infinity");  
			FormatStringTest ("DblFn1 #347", NumberFormat, Double.PositiveInfinity, "E3", "Infinity");  
			FormatStringTest ("DblFn1 #348", NumberFormat, Double.PositiveInfinity, "E4", "Infinity");  
			FormatStringTest ("DblFn1 #349", NumberFormat, Double.PositiveInfinity, "E6", "Infinity");  
			FormatStringTest ("DblFn1 #350", NumberFormat, Double.PositiveInfinity, "E9", "Infinity");  
			FormatStringTest ("DblFn1 #351", NumberFormat, Double.PositiveInfinity, "E67", "Infinity");  
			FormatStringTest ("DblFn1 #352", NumberFormat, Double.PositiveInfinity, "F", "Infinity");  
			FormatStringTest ("DblFn1 #353", NumberFormat, Double.PositiveInfinity, "F0", "Infinity");  
			FormatStringTest ("DblFn1 #354", NumberFormat, Double.PositiveInfinity, "F1", "Infinity");  
			FormatStringTest ("DblFn1 #355", NumberFormat, Double.PositiveInfinity, "F3", "Infinity");  
			FormatStringTest ("DblFn1 #356", NumberFormat, Double.PositiveInfinity, "F4", "Infinity");  
			FormatStringTest ("DblFn1 #357", NumberFormat, Double.PositiveInfinity, "F6", "Infinity");  
			FormatStringTest ("DblFn1 #358", NumberFormat, Double.PositiveInfinity, "F9", "Infinity");  
			FormatStringTest ("DblFn1 #359", NumberFormat, Double.PositiveInfinity, "F67", "Infinity");  
			FormatStringTest ("DblFn1 #360", NumberFormat, Double.PositiveInfinity, "G", "Infinity");  
			FormatStringTest ("DblFn1 #361", NumberFormat, Double.PositiveInfinity, "G0", "Infinity");  
			FormatStringTest ("DblFn1 #362", NumberFormat, Double.PositiveInfinity, "G1", "Infinity");  
			FormatStringTest ("DblFn1 #363", NumberFormat, Double.PositiveInfinity, "G3", "Infinity");  
			FormatStringTest ("DblFn1 #364", NumberFormat, Double.PositiveInfinity, "G4", "Infinity");  
			FormatStringTest ("DblFn1 #365", NumberFormat, Double.PositiveInfinity, "G6", "Infinity");  
			FormatStringTest ("DblFn1 #366", NumberFormat, Double.PositiveInfinity, "G9", "Infinity");  
			FormatStringTest ("DblFn1 #367", NumberFormat, Double.PositiveInfinity, "G67", "Infinity");  
			FormatStringTest ("DblFn1 #368", NumberFormat, Double.PositiveInfinity, "N", "Infinity");  
			FormatStringTest ("DblFn1 #369", NumberFormat, Double.PositiveInfinity, "N0", "Infinity");  
			FormatStringTest ("DblFn1 #370", NumberFormat, Double.PositiveInfinity, "N1", "Infinity");  
			FormatStringTest ("DblFn1 #371", NumberFormat, Double.PositiveInfinity, "N3", "Infinity");  
			FormatStringTest ("DblFn1 #372", NumberFormat, Double.PositiveInfinity, "N4", "Infinity");  
			FormatStringTest ("DblFn1 #373", NumberFormat, Double.PositiveInfinity, "N6", "Infinity");  
			FormatStringTest ("DblFn1 #374", NumberFormat, Double.PositiveInfinity, "N9", "Infinity");  
			FormatStringTest ("DblFn1 #375", NumberFormat, Double.PositiveInfinity, "N67", "Infinity");  
			FormatStringTest ("DblFn1 #376", NumberFormat, Double.PositiveInfinity, "P", "Infinity");  
			FormatStringTest ("DblFn1 #377", NumberFormat, Double.PositiveInfinity, "P0", "Infinity");  
			FormatStringTest ("DblFn1 #378", NumberFormat, Double.PositiveInfinity, "P1", "Infinity");  
			FormatStringTest ("DblFn1 #379", NumberFormat, Double.PositiveInfinity, "P3", "Infinity");  
			FormatStringTest ("DblFn1 #380", NumberFormat, Double.PositiveInfinity, "P4", "Infinity");  
			FormatStringTest ("DblFn1 #381", NumberFormat, Double.PositiveInfinity, "P6", "Infinity");  
			FormatStringTest ("DblFn1 #382", NumberFormat, Double.PositiveInfinity, "P9", "Infinity");  
			FormatStringTest ("DblFn1 #383", NumberFormat, Double.PositiveInfinity, "P67", "Infinity");  
		}
		[Test]
		public void TestFormatStringsN2()
		{
			NumberFormatInfo NumberFormat = GetNumberFormat2();			
			
			FormatStringTest ("DblFn2 #0", NumberFormat, 99999.9999996, "C", "100,000.0$");
			FormatStringTest ("DblFn2 #1", NumberFormat, 99999.9999996, "C0", "100,000$");  
			FormatStringTest ("DblFn2 #2", NumberFormat, 99999.9999996, "C1", "100,000.0$");  
			FormatStringTest ("DblFn2 #3", NumberFormat, 99999.9999996, "C3", "100,000.000$");  
			FormatStringTest ("DblFn2 #4", NumberFormat, 99999.9999996, "C4", "100,000.0000$");  
			FormatStringTest ("DblFn2 #5", NumberFormat, 99999.9999996, "C6", "100,000.000000$");  
			FormatStringTest ("DblFn2 #6", NumberFormat, 99999.9999996, "C9", "99,999.999999600$");  
			FormatStringTest ("DblFn2 #7", NumberFormat, 99999.9999996, "C67", "99,999.9999996000000000000000000000000000000000000000000000000000000000000$");  
			FormatStringTest ("DblFn2 #8", NumberFormat, 99999.9999996, "E", "1.000000E+005");  
			FormatStringTest ("DblFn2 #9", NumberFormat, 99999.9999996, "E0", "1E+005");  
			FormatStringTest ("DblFn2 #10", NumberFormat, 99999.9999996, "E1", "1.0E+005");  
			FormatStringTest ("DblFn2 #11", NumberFormat, 99999.9999996, "E3", "1.000E+005");  
			FormatStringTest ("DblFn2 #12", NumberFormat, 99999.9999996, "E4", "1.0000E+005");  
			FormatStringTest ("DblFn2 #13", NumberFormat, 99999.9999996, "E6", "1.000000E+005");  
			FormatStringTest ("DblFn2 #14", NumberFormat, 99999.9999996, "E9", "1.000000000E+005");  
			FormatStringTest ("DblFn2 #15", NumberFormat, 99999.9999996, "E67", "9.9999999999599997000000000000000000000000000000000000000000000000000E+004");  
			FormatStringTest ("DblFn2 #16", NumberFormat, 99999.9999996, "F", "100000.00");  
			FormatStringTest ("DblFn2 #17", NumberFormat, 99999.9999996, "F0", "100000");  
			FormatStringTest ("DblFn2 #18", NumberFormat, 99999.9999996, "F1", "100000.0");  
			FormatStringTest ("DblFn2 #19", NumberFormat, 99999.9999996, "F3", "100000.000");  
			FormatStringTest ("DblFn2 #20", NumberFormat, 99999.9999996, "F4", "100000.0000");  
			FormatStringTest ("DblFn2 #21", NumberFormat, 99999.9999996, "F6", "100000.000000");  
			FormatStringTest ("DblFn2 #22", NumberFormat, 99999.9999996, "F9", "99999.999999600");  
			FormatStringTest ("DblFn2 #23", NumberFormat, 99999.9999996, "F67", "99999.9999996000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn2 #24", NumberFormat, 99999.9999996, "G", "99999.9999996");  
			FormatStringTest ("DblFn2 #25", NumberFormat, 99999.9999996, "G0", "99999.9999996");  
			FormatStringTest ("DblFn2 #26", NumberFormat, 99999.9999996, "G1", "1E+05");  
			FormatStringTest ("DblFn2 #27", NumberFormat, 99999.9999996, "G3", "1E+05");  
			FormatStringTest ("DblFn2 #28", NumberFormat, 99999.9999996, "G4", "1E+05");  
			FormatStringTest ("DblFn2 #29", NumberFormat, 99999.9999996, "G6", "100000");  
			FormatStringTest ("DblFn2 #30", NumberFormat, 99999.9999996, "G9", "100000");  
			FormatStringTest ("DblFn2 #31", NumberFormat, 99999.9999996, "G67", "99999.999999599997");  
			FormatStringTest ("DblFn2 #32", NumberFormat, 99999.9999996, "N", "100,000.00");  
			FormatStringTest ("DblFn2 #33", NumberFormat, 99999.9999996, "N0", "100,000");  
			FormatStringTest ("DblFn2 #34", NumberFormat, 99999.9999996, "N1", "100,000.0");  
			FormatStringTest ("DblFn2 #35", NumberFormat, 99999.9999996, "N3", "100,000.000");  
			FormatStringTest ("DblFn2 #36", NumberFormat, 99999.9999996, "N4", "100,000.0000");  
			FormatStringTest ("DblFn2 #37", NumberFormat, 99999.9999996, "N6", "100,000.000000");  
			FormatStringTest ("DblFn2 #38", NumberFormat, 99999.9999996, "N9", "99,999.999999600");  
			FormatStringTest ("DblFn2 #39", NumberFormat, 99999.9999996, "N67", "99,999.9999996000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn2 #40", NumberFormat, 99999.9999996, "P", "##10,000,000.00");  
			FormatStringTest ("DblFn2 #41", NumberFormat, 99999.9999996, "P0", "##10,000,000");  
			FormatStringTest ("DblFn2 #42", NumberFormat, 99999.9999996, "P1", "##10,000,000.0");  
			FormatStringTest ("DblFn2 #43", NumberFormat, 99999.9999996, "P3", "##10,000,000.000");  
			FormatStringTest ("DblFn2 #44", NumberFormat, 99999.9999996, "P4", "##10,000,000.0000");  
			FormatStringTest ("DblFn2 #45", NumberFormat, 99999.9999996, "P6", "##9,999,999.999960");  
			FormatStringTest ("DblFn2 #46", NumberFormat, 99999.9999996, "P9", "##9,999,999.999960000");  
			FormatStringTest ("DblFn2 #47", NumberFormat, 99999.9999996, "P67", "##9,999,999.9999600000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn2 #48", NumberFormat, 1.79769313486231E+307, "C", "17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0$");  
			FormatStringTest ("DblFn2 #49", NumberFormat, 1.79769313486231E+307, "C0", "17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000$");  
			FormatStringTest ("DblFn2 #50", NumberFormat, 1.79769313486231E+307, "C1", "17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0$");  
			FormatStringTest ("DblFn2 #51", NumberFormat, 1.79769313486231E+307, "C3", "17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.000$");  
			FormatStringTest ("DblFn2 #52", NumberFormat, 1.79769313486231E+307, "C4", "17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0000$");  
			FormatStringTest ("DblFn2 #53", NumberFormat, 1.79769313486231E+307, "C6", "17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.000000$");  
			FormatStringTest ("DblFn2 #54", NumberFormat, 1.79769313486231E+307, "C9", "17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.000000000$");  
			FormatStringTest ("DblFn2 #55", NumberFormat, 1.79769313486231E+307, "C67", "17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0000000000000000000000000000000000000000000000000000000000000000000$");  
			FormatStringTest ("DblFn2 #56", NumberFormat, 1.79769313486231E+307, "E", "1.797693E+307");  
			FormatStringTest ("DblFn2 #57", NumberFormat, 1.79769313486231E+307, "E0", "2E+307");  
			FormatStringTest ("DblFn2 #58", NumberFormat, 1.79769313486231E+307, "E1", "1.8E+307");  
			FormatStringTest ("DblFn2 #59", NumberFormat, 1.79769313486231E+307, "E3", "1.798E+307");  
			FormatStringTest ("DblFn2 #60", NumberFormat, 1.79769313486231E+307, "E4", "1.7977E+307");  
			FormatStringTest ("DblFn2 #61", NumberFormat, 1.79769313486231E+307, "E6", "1.797693E+307");  
			FormatStringTest ("DblFn2 #62", NumberFormat, 1.79769313486231E+307, "E9", "1.797693135E+307");  
			FormatStringTest ("DblFn2 #63", NumberFormat, 1.79769313486231E+307, "E67", "1.7976931348623100000000000000000000000000000000000000000000000000000E+307");  
			FormatStringTest ("DblFn2 #64", NumberFormat, 1.79769313486231E+307, "F", "17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.00");  
			FormatStringTest ("DblFn2 #65", NumberFormat, 1.79769313486231E+307, "F0", "17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn2 #66", NumberFormat, 1.79769313486231E+307, "F1", "17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.0");  
			FormatStringTest ("DblFn2 #67", NumberFormat, 1.79769313486231E+307, "F3", "17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.000");  
			FormatStringTest ("DblFn2 #68", NumberFormat, 1.79769313486231E+307, "F4", "17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.0000");  
			FormatStringTest ("DblFn2 #69", NumberFormat, 1.79769313486231E+307, "F6", "17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.000000");  
			FormatStringTest ("DblFn2 #70", NumberFormat, 1.79769313486231E+307, "F9", "17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.000000000");  
			FormatStringTest ("DblFn2 #71", NumberFormat, 1.79769313486231E+307, "F67", "17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.0000000000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn2 #72", NumberFormat, 1.79769313486231E+307, "G", "1.79769313486231E+307");  
			FormatStringTest ("DblFn2 #73", NumberFormat, 1.79769313486231E+307, "G0", "1.79769313486231E+307");  
			FormatStringTest ("DblFn2 #74", NumberFormat, 1.79769313486231E+307, "G1", "2E+307");  
			FormatStringTest ("DblFn2 #75", NumberFormat, 1.79769313486231E+307, "G3", "1.8E+307");  
			FormatStringTest ("DblFn2 #76", NumberFormat, 1.79769313486231E+307, "G4", "1.798E+307");  
			FormatStringTest ("DblFn2 #77", NumberFormat, 1.79769313486231E+307, "G6", "1.79769E+307");  
			FormatStringTest ("DblFn2 #78", NumberFormat, 1.79769313486231E+307, "G9", "1.79769313E+307");  
			FormatStringTest ("DblFn2 #79", NumberFormat, 1.79769313486231E+307, "G67", "1.79769313486231E+307");  
			FormatStringTest ("DblFn2 #80", NumberFormat, 1.79769313486231E+307, "N", "17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.00");  
			FormatStringTest ("DblFn2 #81", NumberFormat, 1.79769313486231E+307, "N0", "17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000");  
			FormatStringTest ("DblFn2 #82", NumberFormat, 1.79769313486231E+307, "N1", "17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0");  
			FormatStringTest ("DblFn2 #83", NumberFormat, 1.79769313486231E+307, "N3", "17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.000");  
			FormatStringTest ("DblFn2 #84", NumberFormat, 1.79769313486231E+307, "N4", "17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0000");  
			FormatStringTest ("DblFn2 #85", NumberFormat, 1.79769313486231E+307, "N6", "17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.000000");  
			FormatStringTest ("DblFn2 #86", NumberFormat, 1.79769313486231E+307, "N9", "17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.000000000");  
			FormatStringTest ("DblFn2 #87", NumberFormat, 1.79769313486231E+307, "N67", "17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0000000000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn2 #88", NumberFormat, 1.79769313486231E+307, "P", "##1,797,693,134,862,310,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.00");  
			FormatStringTest ("DblFn2 #89", NumberFormat, 1.79769313486231E+307, "P0", "##1,797,693,134,862,310,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000");  
			FormatStringTest ("DblFn2 #90", NumberFormat, 1.79769313486231E+307, "P1", "##1,797,693,134,862,310,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0");  
			FormatStringTest ("DblFn2 #91", NumberFormat, 1.79769313486231E+307, "P3", "##1,797,693,134,862,310,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.000");  
			FormatStringTest ("DblFn2 #92", NumberFormat, 1.79769313486231E+307, "P4", "##1,797,693,134,862,310,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0000");  
			FormatStringTest ("DblFn2 #93", NumberFormat, 1.79769313486231E+307, "P6", "##1,797,693,134,862,310,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.000000");  
			FormatStringTest ("DblFn2 #94", NumberFormat, 1.79769313486231E+307, "P9", "##1,797,693,134,862,310,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.000000000");  
			FormatStringTest ("DblFn2 #95", NumberFormat, 1.79769313486231E+307, "P67", "##1,797,693,134,862,310,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0000000000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn2 #96", NumberFormat, -1.79769313486231E+307, "C", "$17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0-");  
			FormatStringTest ("DblFn2 #97", NumberFormat, -1.79769313486231E+307, "C0", "$17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000-");  
			FormatStringTest ("DblFn2 #98", NumberFormat, -1.79769313486231E+307, "C1", "$17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0-");  
			FormatStringTest ("DblFn2 #99", NumberFormat, -1.79769313486231E+307, "C3", "$17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.000-");  
			FormatStringTest ("DblFn2 #100", NumberFormat, -1.79769313486231E+307, "C4", "$17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0000-");  
			FormatStringTest ("DblFn2 #101", NumberFormat, -1.79769313486231E+307, "C6", "$17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.000000-");  
			FormatStringTest ("DblFn2 #102", NumberFormat, -1.79769313486231E+307, "C9", "$17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.000000000-");  
			FormatStringTest ("DblFn2 #103", NumberFormat, -1.79769313486231E+307, "C67", "$17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0000000000000000000000000000000000000000000000000000000000000000000-");  
			FormatStringTest ("DblFn2 #104", NumberFormat, -1.79769313486231E+307, "E", "-1.797693E+307");  
			FormatStringTest ("DblFn2 #105", NumberFormat, -1.79769313486231E+307, "E0", "-2E+307");  
			FormatStringTest ("DblFn2 #106", NumberFormat, -1.79769313486231E+307, "E1", "-1.8E+307");  
			FormatStringTest ("DblFn2 #107", NumberFormat, -1.79769313486231E+307, "E3", "-1.798E+307");  
			FormatStringTest ("DblFn2 #108", NumberFormat, -1.79769313486231E+307, "E4", "-1.7977E+307");  
			FormatStringTest ("DblFn2 #109", NumberFormat, -1.79769313486231E+307, "E6", "-1.797693E+307");  
			FormatStringTest ("DblFn2 #110", NumberFormat, -1.79769313486231E+307, "E9", "-1.797693135E+307");  
			FormatStringTest ("DblFn2 #111", NumberFormat, -1.79769313486231E+307, "E67", "-1.7976931348623100000000000000000000000000000000000000000000000000000E+307");  
			FormatStringTest ("DblFn2 #112", NumberFormat, -1.79769313486231E+307, "F", "-17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.00");  
			FormatStringTest ("DblFn2 #113", NumberFormat, -1.79769313486231E+307, "F0", "-17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn2 #114", NumberFormat, -1.79769313486231E+307, "F1", "-17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.0");  
			FormatStringTest ("DblFn2 #115", NumberFormat, -1.79769313486231E+307, "F3", "-17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.000");  
			FormatStringTest ("DblFn2 #116", NumberFormat, -1.79769313486231E+307, "F4", "-17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.0000");  
			FormatStringTest ("DblFn2 #117", NumberFormat, -1.79769313486231E+307, "F6", "-17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.000000");  
			FormatStringTest ("DblFn2 #118", NumberFormat, -1.79769313486231E+307, "F9", "-17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.000000000");  
			FormatStringTest ("DblFn2 #119", NumberFormat, -1.79769313486231E+307, "F67", "-17976931348623100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.0000000000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn2 #120", NumberFormat, -1.79769313486231E+307, "G", "-1.79769313486231E+307");  
			FormatStringTest ("DblFn2 #121", NumberFormat, -1.79769313486231E+307, "G0", "-1.79769313486231E+307");  
			FormatStringTest ("DblFn2 #122", NumberFormat, -1.79769313486231E+307, "G1", "-2E+307");  
			FormatStringTest ("DblFn2 #123", NumberFormat, -1.79769313486231E+307, "G3", "-1.8E+307");  
			FormatStringTest ("DblFn2 #124", NumberFormat, -1.79769313486231E+307, "G4", "-1.798E+307");  
			FormatStringTest ("DblFn2 #125", NumberFormat, -1.79769313486231E+307, "G6", "-1.79769E+307");  
			FormatStringTest ("DblFn2 #126", NumberFormat, -1.79769313486231E+307, "G9", "-1.79769313E+307");  
			FormatStringTest ("DblFn2 #127", NumberFormat, -1.79769313486231E+307, "G67", "-1.79769313486231E+307");  
			FormatStringTest ("DblFn2 #128", NumberFormat, -1.79769313486231E+307, "N", "-17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.00");  
			FormatStringTest ("DblFn2 #129", NumberFormat, -1.79769313486231E+307, "N0", "-17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000");  
			FormatStringTest ("DblFn2 #130", NumberFormat, -1.79769313486231E+307, "N1", "-17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0");  
			FormatStringTest ("DblFn2 #131", NumberFormat, -1.79769313486231E+307, "N3", "-17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.000");  
			FormatStringTest ("DblFn2 #132", NumberFormat, -1.79769313486231E+307, "N4", "-17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0000");  
			FormatStringTest ("DblFn2 #133", NumberFormat, -1.79769313486231E+307, "N6", "-17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.000000");  
			FormatStringTest ("DblFn2 #134", NumberFormat, -1.79769313486231E+307, "N9", "-17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.000000000");  
			FormatStringTest ("DblFn2 #135", NumberFormat, -1.79769313486231E+307, "N67", "-17,976,931,348,623,100,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0000000000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn2 #136", NumberFormat, -1.79769313486231E+307, "P", "-1,797,693,134,862,310,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.00##");  
			FormatStringTest ("DblFn2 #137", NumberFormat, -1.79769313486231E+307, "P0", "-1,797,693,134,862,310,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000##");  
			FormatStringTest ("DblFn2 #138", NumberFormat, -1.79769313486231E+307, "P1", "-1,797,693,134,862,310,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0##");  
			FormatStringTest ("DblFn2 #139", NumberFormat, -1.79769313486231E+307, "P3", "-1,797,693,134,862,310,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.000##");  
			FormatStringTest ("DblFn2 #140", NumberFormat, -1.79769313486231E+307, "P4", "-1,797,693,134,862,310,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0000##");  
			FormatStringTest ("DblFn2 #141", NumberFormat, -1.79769313486231E+307, "P6", "-1,797,693,134,862,310,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.000000##");  
			FormatStringTest ("DblFn2 #142", NumberFormat, -1.79769313486231E+307, "P9", "-1,797,693,134,862,310,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.000000000##");  
			FormatStringTest ("DblFn2 #143", NumberFormat, -1.79769313486231E+307, "P67", "-1,797,693,134,862,310,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000.0000000000000000000000000000000000000000000000000000000000000000000##");  
			FormatStringTest ("DblFn2 #144", NumberFormat, 1E-10, "C", "0.0$");  
			FormatStringTest ("DblFn2 #145", NumberFormat, 1E-10, "C0", "0$");  
			FormatStringTest ("DblFn2 #146", NumberFormat, 1E-10, "C1", "0.0$");  
			FormatStringTest ("DblFn2 #147", NumberFormat, 1E-10, "C3", "0.000$");  
			FormatStringTest ("DblFn2 #148", NumberFormat, 1E-10, "C4", "0.0000$");  
			FormatStringTest ("DblFn2 #149", NumberFormat, 1E-10, "C6", "0.000000$");  
			FormatStringTest ("DblFn2 #150", NumberFormat, 1E-10, "C9", "0.000000000$");  
			FormatStringTest ("DblFn2 #151", NumberFormat, 1E-10, "C67", "0.0000000001000000000000000000000000000000000000000000000000000000000$");  
			FormatStringTest ("DblFn2 #152", NumberFormat, 1E-10, "E", "1.000000E-010");  
			FormatStringTest ("DblFn2 #153", NumberFormat, 1E-10, "E0", "1E-010");  
			FormatStringTest ("DblFn2 #154", NumberFormat, 1E-10, "E1", "1.0E-010");  
			FormatStringTest ("DblFn2 #155", NumberFormat, 1E-10, "E3", "1.000E-010");  
			FormatStringTest ("DblFn2 #156", NumberFormat, 1E-10, "E4", "1.0000E-010");  
			FormatStringTest ("DblFn2 #157", NumberFormat, 1E-10, "E6", "1.000000E-010");  
			FormatStringTest ("DblFn2 #158", NumberFormat, 1E-10, "E9", "1.000000000E-010");  
			FormatStringTest ("DblFn2 #159", NumberFormat, 1E-10, "E67", "1.0000000000000000000000000000000000000000000000000000000000000000000E-010");  
			FormatStringTest ("DblFn2 #160", NumberFormat, 1E-10, "F", "0.00");  
			FormatStringTest ("DblFn2 #161", NumberFormat, 1E-10, "F0", "0");  
			FormatStringTest ("DblFn2 #162", NumberFormat, 1E-10, "F1", "0.0");  
			FormatStringTest ("DblFn2 #163", NumberFormat, 1E-10, "F3", "0.000");  
			FormatStringTest ("DblFn2 #164", NumberFormat, 1E-10, "F4", "0.0000");  
			FormatStringTest ("DblFn2 #165", NumberFormat, 1E-10, "F6", "0.000000");  
			FormatStringTest ("DblFn2 #166", NumberFormat, 1E-10, "F9", "0.000000000");  
			FormatStringTest ("DblFn2 #167", NumberFormat, 1E-10, "F67", "0.0000000001000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn2 #168", NumberFormat, 1E-10, "G", "1E-10");  
			FormatStringTest ("DblFn2 #169", NumberFormat, 1E-10, "G0", "1E-10");  
			FormatStringTest ("DblFn2 #170", NumberFormat, 1E-10, "G1", "1E-10");  
			FormatStringTest ("DblFn2 #171", NumberFormat, 1E-10, "G3", "1E-10");  
			FormatStringTest ("DblFn2 #172", NumberFormat, 1E-10, "G4", "1E-10");  
			FormatStringTest ("DblFn2 #173", NumberFormat, 1E-10, "G6", "1E-10");  
			FormatStringTest ("DblFn2 #174", NumberFormat, 1E-10, "G9", "1E-10");  
			FormatStringTest ("DblFn2 #175", NumberFormat, 1E-10, "G67", "1E-10");  
			FormatStringTest ("DblFn2 #176", NumberFormat, 1E-10, "N", "0.00");  
			FormatStringTest ("DblFn2 #177", NumberFormat, 1E-10, "N0", "0");  
			FormatStringTest ("DblFn2 #178", NumberFormat, 1E-10, "N1", "0.0");  
			FormatStringTest ("DblFn2 #179", NumberFormat, 1E-10, "N3", "0.000");  
			FormatStringTest ("DblFn2 #180", NumberFormat, 1E-10, "N4", "0.0000");  
			FormatStringTest ("DblFn2 #181", NumberFormat, 1E-10, "N6", "0.000000");  
			FormatStringTest ("DblFn2 #182", NumberFormat, 1E-10, "N9", "0.000000000");  
			FormatStringTest ("DblFn2 #183", NumberFormat, 1E-10, "N67", "0.0000000001000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn2 #184", NumberFormat, 1E-10, "P", "##0.00");  
			FormatStringTest ("DblFn2 #185", NumberFormat, 1E-10, "P0", "##0");  
			FormatStringTest ("DblFn2 #186", NumberFormat, 1E-10, "P1", "##0.0");  
			FormatStringTest ("DblFn2 #187", NumberFormat, 1E-10, "P3", "##0.000");  
			FormatStringTest ("DblFn2 #188", NumberFormat, 1E-10, "P4", "##0.0000");  
			FormatStringTest ("DblFn2 #189", NumberFormat, 1E-10, "P6", "##0.000000");  
			FormatStringTest ("DblFn2 #190", NumberFormat, 1E-10, "P9", "##0.000000010");  
			FormatStringTest ("DblFn2 #191", NumberFormat, 1E-10, "P67", "##0.0000000100000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn2 #192", NumberFormat, 4.94065645841247E-324, "C", "0.0$");  
			FormatStringTest ("DblFn2 #193", NumberFormat, 4.94065645841247E-324, "C0", "0$");  
			FormatStringTest ("DblFn2 #194", NumberFormat, 4.94065645841247E-324, "C1", "0.0$");  
			FormatStringTest ("DblFn2 #195", NumberFormat, 4.94065645841247E-324, "C3", "0.000$");  
			FormatStringTest ("DblFn2 #196", NumberFormat, 4.94065645841247E-324, "C4", "0.0000$");  
			FormatStringTest ("DblFn2 #197", NumberFormat, 4.94065645841247E-324, "C6", "0.000000$");  
			FormatStringTest ("DblFn2 #198", NumberFormat, 4.94065645841247E-324, "C9", "0.000000000$");  
			FormatStringTest ("DblFn2 #199", NumberFormat, 4.94065645841247E-324, "C67", "0.0000000000000000000000000000000000000000000000000000000000000000000$");  
			FormatStringTest ("DblFn2 #200", NumberFormat, 4.94065645841247E-324, "E", "4.940656E-324");  
			FormatStringTest ("DblFn2 #201", NumberFormat, 4.94065645841247E-324, "E0", "5E-324");  
			FormatStringTest ("DblFn2 #202", NumberFormat, 4.94065645841247E-324, "E1", "4.9E-324");  
			FormatStringTest ("DblFn2 #203", NumberFormat, 4.94065645841247E-324, "E3", "4.941E-324");  
			FormatStringTest ("DblFn2 #204", NumberFormat, 4.94065645841247E-324, "E4", "4.9407E-324");  
			FormatStringTest ("DblFn2 #205", NumberFormat, 4.94065645841247E-324, "E6", "4.940656E-324");  
			FormatStringTest ("DblFn2 #206", NumberFormat, 4.94065645841247E-324, "E9", "4.940656458E-324");  
			FormatStringTest ("DblFn2 #207", NumberFormat, 4.94065645841247E-324, "E67", "4.9406564584124654000000000000000000000000000000000000000000000000000E-324");  
			FormatStringTest ("DblFn2 #208", NumberFormat, 4.94065645841247E-324, "F", "0.00");  
			FormatStringTest ("DblFn2 #209", NumberFormat, 4.94065645841247E-324, "F0", "0");  
			FormatStringTest ("DblFn2 #210", NumberFormat, 4.94065645841247E-324, "F1", "0.0");  
			FormatStringTest ("DblFn2 #211", NumberFormat, 4.94065645841247E-324, "F3", "0.000");  
			FormatStringTest ("DblFn2 #212", NumberFormat, 4.94065645841247E-324, "F4", "0.0000");  
			FormatStringTest ("DblFn2 #213", NumberFormat, 4.94065645841247E-324, "F6", "0.000000");  
			FormatStringTest ("DblFn2 #214", NumberFormat, 4.94065645841247E-324, "F9", "0.000000000");  
			FormatStringTest ("DblFn2 #215", NumberFormat, 4.94065645841247E-324, "F67", "0.0000000000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn2 #216", NumberFormat, 4.94065645841247E-324, "G", "4.94065645841247E-324");  
			FormatStringTest ("DblFn2 #217", NumberFormat, 4.94065645841247E-324, "G0", "4.94065645841247E-324");  
			FormatStringTest ("DblFn2 #218", NumberFormat, 4.94065645841247E-324, "G1", "5E-324");  
			FormatStringTest ("DblFn2 #219", NumberFormat, 4.94065645841247E-324, "G3", "4.94E-324");  
			FormatStringTest ("DblFn2 #220", NumberFormat, 4.94065645841247E-324, "G4", "4.941E-324");  
			FormatStringTest ("DblFn2 #221", NumberFormat, 4.94065645841247E-324, "G6", "4.94066E-324");  
			FormatStringTest ("DblFn2 #222", NumberFormat, 4.94065645841247E-324, "G9", "4.94065646E-324");  
			FormatStringTest ("DblFn2 #223", NumberFormat, 4.94065645841247E-324, "G67", "4.9406564584124654E-324");  
			FormatStringTest ("DblFn2 #224", NumberFormat, 4.94065645841247E-324, "N", "0.00");  
			FormatStringTest ("DblFn2 #225", NumberFormat, 4.94065645841247E-324, "N0", "0");  
			FormatStringTest ("DblFn2 #226", NumberFormat, 4.94065645841247E-324, "N1", "0.0");  
			FormatStringTest ("DblFn2 #227", NumberFormat, 4.94065645841247E-324, "N3", "0.000");  
			FormatStringTest ("DblFn2 #228", NumberFormat, 4.94065645841247E-324, "N4", "0.0000");  
			FormatStringTest ("DblFn2 #229", NumberFormat, 4.94065645841247E-324, "N6", "0.000000");  
			FormatStringTest ("DblFn2 #230", NumberFormat, 4.94065645841247E-324, "N9", "0.000000000");  
			FormatStringTest ("DblFn2 #231", NumberFormat, 4.94065645841247E-324, "N67", "0.0000000000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn2 #232", NumberFormat, 4.94065645841247E-324, "P", "##0.00");  
			FormatStringTest ("DblFn2 #233", NumberFormat, 4.94065645841247E-324, "P0", "##0");  
			FormatStringTest ("DblFn2 #234", NumberFormat, 4.94065645841247E-324, "P1", "##0.0");  
			FormatStringTest ("DblFn2 #235", NumberFormat, 4.94065645841247E-324, "P3", "##0.000");  
			FormatStringTest ("DblFn2 #236", NumberFormat, 4.94065645841247E-324, "P4", "##0.0000");  
			FormatStringTest ("DblFn2 #237", NumberFormat, 4.94065645841247E-324, "P6", "##0.000000");  
			FormatStringTest ("DblFn2 #238", NumberFormat, 4.94065645841247E-324, "P9", "##0.000000000");  
			FormatStringTest ("DblFn2 #239", NumberFormat, 4.94065645841247E-324, "P67", "##0.0000000000000000000000000000000000000000000000000000000000000000000");  
			FormatStringTest ("DblFn2 #240", NumberFormat, Double.NaN,  "C", "Geen");  
			FormatStringTest ("DblFn2 #241", NumberFormat, Double.NaN,  "C0", "Geen");  
			FormatStringTest ("DblFn2 #242", NumberFormat, Double.NaN,  "C1", "Geen");  
			FormatStringTest ("DblFn2 #243", NumberFormat, Double.NaN,  "C3", "Geen");  
			FormatStringTest ("DblFn2 #244", NumberFormat, Double.NaN,  "C4", "Geen");  
			FormatStringTest ("DblFn2 #245", NumberFormat, Double.NaN,  "C6", "Geen");  
			FormatStringTest ("DblFn2 #246", NumberFormat, Double.NaN,  "C9", "Geen");  
			FormatStringTest ("DblFn2 #247", NumberFormat, Double.NaN,  "C67", "Geen");  
			FormatStringTest ("DblFn2 #248", NumberFormat, Double.NaN,  "E", "Geen");  
			FormatStringTest ("DblFn2 #249", NumberFormat, Double.NaN,  "E0", "Geen");  
			FormatStringTest ("DblFn2 #250", NumberFormat, Double.NaN,  "E1", "Geen");  
			FormatStringTest ("DblFn2 #251", NumberFormat, Double.NaN,  "E3", "Geen");  
			FormatStringTest ("DblFn2 #252", NumberFormat, Double.NaN,  "E4", "Geen");  
			FormatStringTest ("DblFn2 #253", NumberFormat, Double.NaN,  "E6", "Geen");  
			FormatStringTest ("DblFn2 #254", NumberFormat, Double.NaN,  "E9", "Geen");  
			FormatStringTest ("DblFn2 #255", NumberFormat, Double.NaN,  "E67", "Geen");  
			FormatStringTest ("DblFn2 #256", NumberFormat, Double.NaN,  "F", "Geen");  
			FormatStringTest ("DblFn2 #257", NumberFormat, Double.NaN,  "F0", "Geen");  
			FormatStringTest ("DblFn2 #258", NumberFormat, Double.NaN,  "F1", "Geen");  
			FormatStringTest ("DblFn2 #259", NumberFormat, Double.NaN,  "F3", "Geen");  
			FormatStringTest ("DblFn2 #260", NumberFormat, Double.NaN,  "F4", "Geen");  
			FormatStringTest ("DblFn2 #261", NumberFormat, Double.NaN,  "F6", "Geen");  
			FormatStringTest ("DblFn2 #262", NumberFormat, Double.NaN,  "F9", "Geen");  
			FormatStringTest ("DblFn2 #263", NumberFormat, Double.NaN,  "F67", "Geen");  
			FormatStringTest ("DblFn2 #264", NumberFormat, Double.NaN,  "G", "Geen");  
			FormatStringTest ("DblFn2 #265", NumberFormat, Double.NaN,  "G0", "Geen");  
			FormatStringTest ("DblFn2 #266", NumberFormat, Double.NaN,  "G1", "Geen");  
			FormatStringTest ("DblFn2 #267", NumberFormat, Double.NaN,  "G3", "Geen");  
			FormatStringTest ("DblFn2 #268", NumberFormat, Double.NaN,  "G4", "Geen");  
			FormatStringTest ("DblFn2 #269", NumberFormat, Double.NaN,  "G6", "Geen");  
			FormatStringTest ("DblFn2 #270", NumberFormat, Double.NaN,  "G9", "Geen");  
			FormatStringTest ("DblFn2 #271", NumberFormat, Double.NaN,  "G67", "Geen");  
			FormatStringTest ("DblFn2 #272", NumberFormat, Double.NaN,  "N", "Geen");  
			FormatStringTest ("DblFn2 #273", NumberFormat, Double.NaN,  "N0", "Geen");  
			FormatStringTest ("DblFn2 #274", NumberFormat, Double.NaN,  "N1", "Geen");  
			FormatStringTest ("DblFn2 #275", NumberFormat, Double.NaN,  "N3", "Geen");  
			FormatStringTest ("DblFn2 #276", NumberFormat, Double.NaN,  "N4", "Geen");  
			FormatStringTest ("DblFn2 #277", NumberFormat, Double.NaN,  "N6", "Geen");  
			FormatStringTest ("DblFn2 #278", NumberFormat, Double.NaN,  "N9", "Geen");  
			FormatStringTest ("DblFn2 #279", NumberFormat, Double.NaN,  "N67", "Geen");  
			FormatStringTest ("DblFn2 #280", NumberFormat, Double.NaN,  "P", "Geen");  
			FormatStringTest ("DblFn2 #281", NumberFormat, Double.NaN,  "P0", "Geen");  
			FormatStringTest ("DblFn2 #282", NumberFormat, Double.NaN,  "P1", "Geen");  
			FormatStringTest ("DblFn2 #283", NumberFormat, Double.NaN,  "P3", "Geen");  
			FormatStringTest ("DblFn2 #284", NumberFormat, Double.NaN,  "P4", "Geen");  
			FormatStringTest ("DblFn2 #285", NumberFormat, Double.NaN,  "P6", "Geen");  
			FormatStringTest ("DblFn2 #286", NumberFormat, Double.NaN,  "P9", "Geen");  
			FormatStringTest ("DblFn2 #287", NumberFormat, Double.NaN,  "P67", "Geen");  
			FormatStringTest ("DblFn2 #288", NumberFormat, Double.NegativeInfinity, "C", "-Oneindig");  
			FormatStringTest ("DblFn2 #289", NumberFormat, Double.NegativeInfinity, "C0", "-Oneindig");  
			FormatStringTest ("DblFn2 #290", NumberFormat, Double.NegativeInfinity, "C1", "-Oneindig");  
			FormatStringTest ("DblFn2 #291", NumberFormat, Double.NegativeInfinity, "C3", "-Oneindig");  
			FormatStringTest ("DblFn2 #292", NumberFormat, Double.NegativeInfinity, "C4", "-Oneindig");  
			FormatStringTest ("DblFn2 #293", NumberFormat, Double.NegativeInfinity, "C6", "-Oneindig");  
			FormatStringTest ("DblFn2 #294", NumberFormat, Double.NegativeInfinity, "C9", "-Oneindig");  
			FormatStringTest ("DblFn2 #295", NumberFormat, Double.NegativeInfinity, "C67", "-Oneindig");  
			FormatStringTest ("DblFn2 #296", NumberFormat, Double.NegativeInfinity, "E", "-Oneindig");  
			FormatStringTest ("DblFn2 #297", NumberFormat, Double.NegativeInfinity, "E0", "-Oneindig");  
			FormatStringTest ("DblFn2 #298", NumberFormat, Double.NegativeInfinity, "E1", "-Oneindig");  
			FormatStringTest ("DblFn2 #299", NumberFormat, Double.NegativeInfinity, "E3", "-Oneindig");  
			FormatStringTest ("DblFn2 #300", NumberFormat, Double.NegativeInfinity, "E4", "-Oneindig");  
			FormatStringTest ("DblFn2 #301", NumberFormat, Double.NegativeInfinity, "E6", "-Oneindig");  
			FormatStringTest ("DblFn2 #302", NumberFormat, Double.NegativeInfinity, "E9", "-Oneindig");  
			FormatStringTest ("DblFn2 #303", NumberFormat, Double.NegativeInfinity, "E67", "-Oneindig");  
			FormatStringTest ("DblFn2 #304", NumberFormat, Double.NegativeInfinity, "F", "-Oneindig");  
			FormatStringTest ("DblFn2 #305", NumberFormat, Double.NegativeInfinity, "F0", "-Oneindig");  
			FormatStringTest ("DblFn2 #306", NumberFormat, Double.NegativeInfinity, "F1", "-Oneindig");  
			FormatStringTest ("DblFn2 #307", NumberFormat, Double.NegativeInfinity, "F3", "-Oneindig");  
			FormatStringTest ("DblFn2 #308", NumberFormat, Double.NegativeInfinity, "F4", "-Oneindig");  
			FormatStringTest ("DblFn2 #309", NumberFormat, Double.NegativeInfinity, "F6", "-Oneindig");  
			FormatStringTest ("DblFn2 #310", NumberFormat, Double.NegativeInfinity, "F9", "-Oneindig");  
			FormatStringTest ("DblFn2 #311", NumberFormat, Double.NegativeInfinity, "F67", "-Oneindig");  
			FormatStringTest ("DblFn2 #312", NumberFormat, Double.NegativeInfinity, "G", "-Oneindig");  
			FormatStringTest ("DblFn2 #313", NumberFormat, Double.NegativeInfinity, "G0", "-Oneindig");  
			FormatStringTest ("DblFn2 #314", NumberFormat, Double.NegativeInfinity, "G1", "-Oneindig");  
			FormatStringTest ("DblFn2 #315", NumberFormat, Double.NegativeInfinity, "G3", "-Oneindig");  
			FormatStringTest ("DblFn2 #316", NumberFormat, Double.NegativeInfinity, "G4", "-Oneindig");  
			FormatStringTest ("DblFn2 #317", NumberFormat, Double.NegativeInfinity, "G6", "-Oneindig");  
			FormatStringTest ("DblFn2 #318", NumberFormat, Double.NegativeInfinity, "G9", "-Oneindig");  
			FormatStringTest ("DblFn2 #319", NumberFormat, Double.NegativeInfinity, "G67", "-Oneindig");  
			FormatStringTest ("DblFn2 #320", NumberFormat, Double.NegativeInfinity, "N", "-Oneindig");  
			FormatStringTest ("DblFn2 #321", NumberFormat, Double.NegativeInfinity, "N0", "-Oneindig");  
			FormatStringTest ("DblFn2 #322", NumberFormat, Double.NegativeInfinity, "N1", "-Oneindig");  
			FormatStringTest ("DblFn2 #323", NumberFormat, Double.NegativeInfinity, "N3", "-Oneindig");  
			FormatStringTest ("DblFn2 #324", NumberFormat, Double.NegativeInfinity, "N4", "-Oneindig");  
			FormatStringTest ("DblFn2 #325", NumberFormat, Double.NegativeInfinity, "N6", "-Oneindig");  
			FormatStringTest ("DblFn2 #326", NumberFormat, Double.NegativeInfinity, "N9", "-Oneindig");  
			FormatStringTest ("DblFn2 #327", NumberFormat, Double.NegativeInfinity, "N67", "-Oneindig");  
			FormatStringTest ("DblFn2 #328", NumberFormat, Double.NegativeInfinity, "P", "-Oneindig");  
			FormatStringTest ("DblFn2 #329", NumberFormat, Double.NegativeInfinity, "P0", "-Oneindig");  
			FormatStringTest ("DblFn2 #330", NumberFormat, Double.NegativeInfinity, "P1", "-Oneindig");  
			FormatStringTest ("DblFn2 #331", NumberFormat, Double.NegativeInfinity, "P3", "-Oneindig");  
			FormatStringTest ("DblFn2 #332", NumberFormat, Double.NegativeInfinity, "P4", "-Oneindig");  
			FormatStringTest ("DblFn2 #333", NumberFormat, Double.NegativeInfinity, "P6", "-Oneindig");  
			FormatStringTest ("DblFn2 #334", NumberFormat, Double.NegativeInfinity, "P9", "-Oneindig");  
			FormatStringTest ("DblFn2 #335", NumberFormat, Double.NegativeInfinity, "P67", "-Oneindig");  
			FormatStringTest ("DblFn2 #336", NumberFormat, Double.PositiveInfinity, "C", "Oneindig");  
			FormatStringTest ("DblFn2 #337", NumberFormat, Double.PositiveInfinity, "C0", "Oneindig");  
			FormatStringTest ("DblFn2 #338", NumberFormat, Double.PositiveInfinity, "C1", "Oneindig");  
			FormatStringTest ("DblFn2 #339", NumberFormat, Double.PositiveInfinity, "C3", "Oneindig");  
			FormatStringTest ("DblFn2 #340", NumberFormat, Double.PositiveInfinity, "C4", "Oneindig");  
			FormatStringTest ("DblFn2 #341", NumberFormat, Double.PositiveInfinity, "C6", "Oneindig");  
			FormatStringTest ("DblFn2 #342", NumberFormat, Double.PositiveInfinity, "C9", "Oneindig");  
			FormatStringTest ("DblFn2 #343", NumberFormat, Double.PositiveInfinity, "C67", "Oneindig");  
			FormatStringTest ("DblFn2 #344", NumberFormat, Double.PositiveInfinity, "E", "Oneindig");  
			FormatStringTest ("DblFn2 #345", NumberFormat, Double.PositiveInfinity, "E0", "Oneindig");  
			FormatStringTest ("DblFn2 #346", NumberFormat, Double.PositiveInfinity, "E1", "Oneindig");  
			FormatStringTest ("DblFn2 #347", NumberFormat, Double.PositiveInfinity, "E3", "Oneindig");  
			FormatStringTest ("DblFn2 #348", NumberFormat, Double.PositiveInfinity, "E4", "Oneindig");  
			FormatStringTest ("DblFn2 #349", NumberFormat, Double.PositiveInfinity, "E6", "Oneindig");  
			FormatStringTest ("DblFn2 #350", NumberFormat, Double.PositiveInfinity, "E9", "Oneindig");  
			FormatStringTest ("DblFn2 #351", NumberFormat, Double.PositiveInfinity, "E67", "Oneindig");  
			FormatStringTest ("DblFn2 #352", NumberFormat, Double.PositiveInfinity, "F", "Oneindig");  
			FormatStringTest ("DblFn2 #353", NumberFormat, Double.PositiveInfinity, "F0", "Oneindig");  
			FormatStringTest ("DblFn2 #354", NumberFormat, Double.PositiveInfinity, "F1", "Oneindig");  
			FormatStringTest ("DblFn2 #355", NumberFormat, Double.PositiveInfinity, "F3", "Oneindig");  
			FormatStringTest ("DblFn2 #356", NumberFormat, Double.PositiveInfinity, "F4", "Oneindig");  
			FormatStringTest ("DblFn2 #357", NumberFormat, Double.PositiveInfinity, "F6", "Oneindig");  
			FormatStringTest ("DblFn2 #358", NumberFormat, Double.PositiveInfinity, "F9", "Oneindig");  
			FormatStringTest ("DblFn2 #359", NumberFormat, Double.PositiveInfinity, "F67", "Oneindig");  
			FormatStringTest ("DblFn2 #360", NumberFormat, Double.PositiveInfinity, "G", "Oneindig");  
			FormatStringTest ("DblFn2 #361", NumberFormat, Double.PositiveInfinity, "G0", "Oneindig");  
			FormatStringTest ("DblFn2 #362", NumberFormat, Double.PositiveInfinity, "G1", "Oneindig");  
			FormatStringTest ("DblFn2 #363", NumberFormat, Double.PositiveInfinity, "G3", "Oneindig");  
			FormatStringTest ("DblFn2 #364", NumberFormat, Double.PositiveInfinity, "G4", "Oneindig");  
			FormatStringTest ("DblFn2 #365", NumberFormat, Double.PositiveInfinity, "G6", "Oneindig");  
			FormatStringTest ("DblFn2 #366", NumberFormat, Double.PositiveInfinity, "G9", "Oneindig");  
			FormatStringTest ("DblFn2 #367", NumberFormat, Double.PositiveInfinity, "G67", "Oneindig");  
			FormatStringTest ("DblFn2 #368", NumberFormat, Double.PositiveInfinity, "N", "Oneindig");  
			FormatStringTest ("DblFn2 #369", NumberFormat, Double.PositiveInfinity, "N0", "Oneindig");  
			FormatStringTest ("DblFn2 #370", NumberFormat, Double.PositiveInfinity, "N1", "Oneindig");  
			FormatStringTest ("DblFn2 #371", NumberFormat, Double.PositiveInfinity, "N3", "Oneindig");  
			FormatStringTest ("DblFn2 #372", NumberFormat, Double.PositiveInfinity, "N4", "Oneindig");  
			FormatStringTest ("DblFn2 #373", NumberFormat, Double.PositiveInfinity, "N6", "Oneindig");  
			FormatStringTest ("DblFn2 #374", NumberFormat, Double.PositiveInfinity, "N9", "Oneindig");  
			FormatStringTest ("DblFn2 #375", NumberFormat, Double.PositiveInfinity, "N67", "Oneindig");  
			FormatStringTest ("DblFn2 #376", NumberFormat, Double.PositiveInfinity, "P", "Oneindig");  
			FormatStringTest ("DblFn2 #377", NumberFormat, Double.PositiveInfinity, "P0", "Oneindig");  
			FormatStringTest ("DblFn2 #378", NumberFormat, Double.PositiveInfinity, "P1", "Oneindig");  
			FormatStringTest ("DblFn2 #379", NumberFormat, Double.PositiveInfinity, "P3", "Oneindig");  
			FormatStringTest ("DblFn2 #380", NumberFormat, Double.PositiveInfinity, "P4", "Oneindig");  
			FormatStringTest ("DblFn2 #381", NumberFormat, Double.PositiveInfinity, "P6", "Oneindig");  
			FormatStringTest ("DblFn2 #382", NumberFormat, Double.PositiveInfinity, "P9", "Oneindig");  
			FormatStringTest ("DblFn2 #383", NumberFormat, Double.PositiveInfinity, "P67", "Oneindig");  
		}

		[Test]
		public void TestNumberDecimals ()
		{
			NumberFormatInfo nfi = new NumberFormatInfo ();
			nfi.NumberDecimalDigits = 5;
			FormatStringTest ("#1", nfi, 70.50, "N", "70.50000");
		}

		// "G" format doesn't use the banker rounding (like Math.Round)
		// see bug #60111 for more details
		[Test]
		public void NonStandardRounding () 
		{
			// banker's rounding would results in 2.71828182845904
			AssertEquals ("G-E", "2.71828182845905", Math.E.ToString (CultureInfo.InvariantCulture));
		}

		// see bug #60110 for more details
		[Test]
		public void Roundtrip_ExactStringFormat () 
		{
			// here we check that the "R" output is "extactly" the same as MS implementation
			AssertEquals ("R-E", "2.7182818284590451", Math.E.ToString ("R", CultureInfo.InvariantCulture));
		}

		// see bug #60110 for more details
		[Test]
		public void Roundtrip () 
		{
			// here we check that we can recreate the "extact" same double from the "R" format
			string se = Math.E.ToString ("R", CultureInfo.InvariantCulture);
			Double de = Double.Parse (se);
			AssertEquals ("E==E", Math.E, de);

			// we try Mono "long" R format
			se = "2.718281828459045090795598298427648842334747314453125";
			de = Double.Parse (se);
			AssertEquals ("Mono==E", Math.E, de);

			// we try MS "short" R format
			se = "2.7182818284590451";
			de = Double.Parse (se);
			AssertEquals ("Microsoft==E", Math.E, de);
		}

		[Test]
		public void NegativeRoundtrip () 
		{
			Double value = -Math.E;
			// here we check that we can recreate the "extact" same double from the "R" format
			string se = value.ToString ("R", CultureInfo.InvariantCulture);
			Double de = Double.Parse (se);
			AssertEquals ("-E==-E", value, de);
		}
	}
}
		
