//
// X509SecurityTokenTest.cs - NUnit Test Cases for X509SecurityToken
//
// Author:
//	Sebastien Pouliot (spouliot@motus.com)
//
// (C) 2003 Motus Technologies Inc. (http://www.motus.com)
//

using NUnit.Framework;
using Microsoft.Web.Services.Security;
using Microsoft.Web.Services.Security.X509;
using System;
using System.ComponentModel; // for Win32Exception
using System.Reflection;
using System.Security.Cryptography;
using System.Xml;

namespace MonoTests.MS.Web.Services.Security {

	[TestFixture]
	public class X509SecurityTokenTest : Assertion {

		// current but not trusted (test certificate)
		private static byte[] cert1 = { 0x30, 0x82, 0x02, 0x2A, 0x30, 0x82, 0x01, 0x97, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x91, 0xC4, 0x4B, 0x0D, 0xB7, 0xD8, 0x10, 0x84, 0x42, 0x26, 0x71, 0xB3, 0x97, 0xB5, 0x00, 0x97, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1D, 0x05, 0x00, 0x30, 0x28, 0x31, 0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x1D, 0x4D, 0x6F, 0x74, 0x75, 0x73, 0x20, 0x54, 0x65, 0x63, 0x68, 0x6E, 0x6F, 0x6C, 0x6F, 0x67, 0x69, 0x65, 0x73, 0x20, 0x69, 0x6E, 0x63, 0x2E, 0x28, 0x74, 0x65, 0x73, 0x74, 0x29, 0x30, 0x1E, 0x17, 0x0D, 0x30, 0x33, 0x30, 0x38, 0x31, 0x33, 0x30, 0x30, 0x34, 0x33, 0x34, 0x37, 0x5A, 0x17, 0x0D, 0x33, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 
			0x39, 0x35, 0x39, 0x5A, 0x30, 0x13, 0x31, 0x11, 0x30, 0x0F, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x08, 0x46, 0x41, 0x52, 0x53, 0x43, 0x41, 0x50, 0x45, 0x30, 0x81, 0x9F, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8D, 0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xD2, 0xCB, 0x47, 0x21, 0xF5, 0x62, 0xDD, 0x35, 0xBF, 0x1D, 0xEC, 0x9A, 0x4C, 0x07, 0x2C, 0x01, 0xF0, 0x28, 0xC2, 0x82, 0x17, 0x8E, 0x58, 0x32, 0xD5, 0x4C, 0xAC, 0x86, 0xB4, 0xC9, 0xEB, 0x21, 0x26, 0xF3, 0x22, 0x30, 0xC5, 0x7A, 0xA3, 0x5A, 0xDD, 0x53, 0xAB, 0x1C, 0x06, 0x3E, 0xB2, 0x13, 0xC4, 0x05, 0x1D, 0x95, 0x8B, 0x0A, 0x71, 0x71, 0x11, 0xA7, 0x47, 0x26, 0x61, 0xF1,
			0x76, 0xBE, 0x35, 0x72, 0x32, 0xC5, 0xCB, 0x47, 0xA4, 0x22, 0x41, 0x1E, 0xAD, 0x29, 0x11, 0x0D, 0x39, 0x22, 0x0C, 0x79, 0x90, 0xC6, 0x52, 0xA1, 0x10, 0xF6, 0x55, 0x09, 0x4E, 0x51, 0x26, 0x47, 0x0E, 0x94, 0xE6, 0x81, 0xF5, 0x18, 0x6B, 0x99, 0xF0, 0x76, 0xF3, 0xB2, 0x4C, 0x91, 0xE9, 0xBA, 0x3B, 0x3F, 0x6E, 0x63, 0xDA, 0x12, 0xD1, 0x0B, 0x73, 0x0E, 0x12, 0xC7, 0x70, 0x77, 0x22, 0x03, 0x9D, 0x5D, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x72, 0x30, 0x70, 0x30, 0x13, 0x06, 0x03, 0x55, 0x1D, 0x25, 0x04, 0x0C, 0x30, 0x0A, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x30, 0x59, 0x06, 0x03, 0x55, 0x1D, 0x01, 0x04, 0x52, 0x30, 0x50, 0x80, 0x10, 0xAE, 0xD7, 0x80, 0x88, 0xA6, 0x3D, 0xBA,
			0x50, 0xA1, 0x7E, 0x57, 0xE5, 0x40, 0xC9, 0x6F, 0xC5, 0xA1, 0x2A, 0x30, 0x28, 0x31, 0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x1D, 0x4D, 0x6F, 0x74, 0x75, 0x73, 0x20, 0x54, 0x65, 0x63, 0x68, 0x6E, 0x6F, 0x6C, 0x6F, 0x67, 0x69, 0x65, 0x73, 0x20, 0x69, 0x6E, 0x63, 0x2E, 0x28, 0x74, 0x65, 0x73, 0x74, 0x29, 0x82, 0x10, 0x9D, 0xAE, 0xA3, 0x39, 0x47, 0x0E, 0xD4, 0xA2, 0x49, 0x78, 0xEA, 0x6C, 0xBA, 0x0D, 0xDE, 0x9C, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1D, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00, 0x32, 0x8A, 0x7E, 0xAD, 0xE7, 0x67, 0x9E, 0x5C, 0x4C, 0xD8, 0x33, 0x59, 0x68, 0xCF, 0x94, 0xC0, 0x36, 0x47, 0x7A, 0xA7, 0x85, 0xC2, 0xDD, 0xD8, 0xDA, 0x11, 0x3C, 0x66, 0xC1, 0x83,
			0xE3, 0xAB, 0x33, 0x06, 0x7C, 0xE3, 0x6A, 0x15, 0x72, 0xB8, 0x83, 0x3D, 0x0B, 0xAB, 0x3C, 0xEE, 0x75, 0x13, 0xBD, 0x5C, 0x96, 0x25, 0x56, 0x36, 0x05, 0xFA, 0xAE, 0xD4, 0xF4, 0xCF, 0x52, 0xEC, 0x11, 0xB5, 0xEA, 0x9F, 0x20, 0xA3, 0xC8, 0x34, 0x72, 0x59, 0x09, 0x51, 0xE7, 0x36, 0x87, 0x86, 0x86, 0x98, 0xB5, 0x30, 0x7B, 0xFB, 0x3D, 0xCC, 0x5E, 0xE8, 0xC9, 0x49, 0xE0, 0xC6, 0xEA, 0x02, 0x76, 0x01, 0xE0, 0xBB, 0x8A, 0x70, 0xEB, 0x07, 0x86, 0xE8, 0x04, 0xE7, 0x48, 0xE4, 0x6C, 0x90, 0xE6, 0x16, 0x42, 0xB4, 0xBB, 0xC0, 0xC4, 0x82, 0x5F, 0xF8, 0xFB, 0x7E, 0xB2, 0x9E, 0xC2, 0x78, 0x26, 0x86 };

		// this certificate isn't current
		private static byte[] cert2 = { 0x30,0x82,0x09,0xB9,0x30,0x82,0x09,0x22,0xA0,0x03,0x02,0x01,0x02,0x02,0x10,0x20,0x0B,0x35,0x5E,0xCE,0xC4,0xB0,0x63,0xB7,0xDE,0xC6,0x34,0xB9,0x70,0x34,0x44,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x04,0x05,0x00,0x30,0x62,0x31,0x11,0x30,0x0F,0x06,0x03,0x55,0x04,0x07,0x13,0x08,0x49,0x6E,0x74,0x65,0x72,0x6E,0x65,0x74,0x31,0x17,0x30,0x15,0x06,0x03,0x55,0x04,0x0A,0x13,0x0E,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x31,0x34,0x30,0x32,0x06,0x03,0x55,0x04,0x0B,
			0x13,0x2B,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x20,0x43,0x6C,0x61,0x73,0x73,0x20,0x31,0x20,0x43,0x41,0x20,0x2D,0x20,0x49,0x6E,0x64,0x69,0x76,0x69,0x64,0x75,0x61,0x6C,0x20,0x53,0x75,0x62,0x73,0x63,0x72,0x69,0x62,0x65,0x72,0x30,0x1E,0x17,0x0D,0x39,0x36,0x30,0x38,0x32,0x31,0x30,0x30,0x30,0x30,0x30,0x30,0x5A,0x17,0x0D,0x39,0x37,0x30,0x38,0x32,0x30,0x32,0x33,0x35,0x39,0x35,0x39,0x5A,0x30,0x82,0x01,0x0A,0x31,0x11,0x30,0x0F,0x06,0x03,0x55,0x04,0x07,0x13,0x08,0x49,0x6E,0x74,0x65,0x72,0x6E,0x65,0x74,
			0x31,0x17,0x30,0x15,0x06,0x03,0x55,0x04,0x0A,0x13,0x0E,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x31,0x34,0x30,0x32,0x06,0x03,0x55,0x04,0x0B,0x13,0x2B,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x20,0x43,0x6C,0x61,0x73,0x73,0x20,0x31,0x20,0x43,0x41,0x20,0x2D,0x20,0x49,0x6E,0x64,0x69,0x76,0x69,0x64,0x75,0x61,0x6C,0x20,0x53,0x75,0x62,0x73,0x63,0x72,0x69,0x62,0x65,0x72,0x31,0x46,0x30,0x44,0x06,0x03,0x55,0x04,0x0B,0x13,0x3D,0x77,0x77,0x77,0x2E,0x76,0x65,0x72,0x69,0x73,0x69,
			0x67,0x6E,0x2E,0x63,0x6F,0x6D,0x2F,0x72,0x65,0x70,0x6F,0x73,0x69,0x74,0x6F,0x72,0x79,0x2F,0x43,0x50,0x53,0x20,0x49,0x6E,0x63,0x6F,0x72,0x70,0x2E,0x20,0x62,0x79,0x20,0x52,0x65,0x66,0x2E,0x2C,0x4C,0x49,0x41,0x42,0x2E,0x4C,0x54,0x44,0x28,0x63,0x29,0x39,0x36,0x31,0x26,0x30,0x24,0x06,0x03,0x55,0x04,0x0B,0x13,0x1D,0x44,0x69,0x67,0x69,0x74,0x61,0x6C,0x20,0x49,0x44,0x20,0x43,0x6C,0x61,0x73,0x73,0x20,0x31,0x20,0x2D,0x20,0x4E,0x65,0x74,0x73,0x63,0x61,0x70,0x65,0x31,0x16,0x30,0x14,0x06,0x03,0x55,0x04,0x03,
			0x13,0x0D,0x44,0x61,0x76,0x69,0x64,0x20,0x54,0x2E,0x20,0x47,0x72,0x61,0x79,0x31,0x1E,0x30,0x1C,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x09,0x01,0x16,0x0F,0x64,0x61,0x76,0x69,0x64,0x40,0x66,0x6F,0x72,0x6D,0x61,0x6C,0x2E,0x69,0x65,0x30,0x5C,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x4B,0x00,0x30,0x48,0x02,0x41,0x00,0xC5,0x81,0x07,0xA2,0xEB,0x0F,0xB8,0xFF,0xF8,0xF8,0x1C,0xEE,0x32,0xFF,0xBF,0x12,0x35,0x6A,0xF9,0x6B,0xC8,0xBE,0x2F,0xFB,0x3E,0xAF,0x04,0x51,
			0x4A,0xAC,0xDD,0x10,0x29,0xA8,0xCD,0x40,0x5B,0x66,0x1E,0x98,0xEF,0xF2,0x4C,0x77,0xFA,0x8F,0x86,0xD1,0x21,0x67,0x92,0x44,0x4A,0xC4,0x89,0xC9,0x83,0xCF,0x88,0x9F,0x6F,0xE2,0x32,0x35,0x02,0x03,0x01,0x00,0x01,0xA3,0x82,0x07,0x08,0x30,0x82,0x07,0x04,0x30,0x09,0x06,0x03,0x55,0x1D,0x13,0x04,0x02,0x30,0x00,0x30,0x82,0x02,0x1F,0x06,0x03,0x55,0x1D,0x03,0x04,0x82,0x02,0x16,0x30,0x82,0x02,0x12,0x30,0x82,0x02,0x0E,0x30,0x82,0x02,0x0A,0x06,0x0B,0x60,0x86,0x48,0x01,0x86,0xF8,0x45,0x01,0x07,0x01,0x01,0x30,0x82,
			0x01,0xF9,0x16,0x82,0x01,0xA7,0x54,0x68,0x69,0x73,0x20,0x63,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x20,0x69,0x6E,0x63,0x6F,0x72,0x70,0x6F,0x72,0x61,0x74,0x65,0x73,0x20,0x62,0x79,0x20,0x72,0x65,0x66,0x65,0x72,0x65,0x6E,0x63,0x65,0x2C,0x20,0x61,0x6E,0x64,0x20,0x69,0x74,0x73,0x20,0x75,0x73,0x65,0x20,0x69,0x73,0x20,0x73,0x74,0x72,0x69,0x63,0x74,0x6C,0x79,0x20,0x73,0x75,0x62,0x6A,0x65,0x63,0x74,0x20,0x74,0x6F,0x2C,0x20,0x74,0x68,0x65,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x20,0x43,
			0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6F,0x6E,0x20,0x50,0x72,0x61,0x63,0x74,0x69,0x63,0x65,0x20,0x53,0x74,0x61,0x74,0x65,0x6D,0x65,0x6E,0x74,0x20,0x28,0x43,0x50,0x53,0x29,0x2C,0x20,0x61,0x76,0x61,0x69,0x6C,0x61,0x62,0x6C,0x65,0x20,0x61,0x74,0x3A,0x20,0x68,0x74,0x74,0x70,0x73,0x3A,0x2F,0x2F,0x77,0x77,0x77,0x2E,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6E,0x2E,0x63,0x6F,0x6D,0x2F,0x43,0x50,0x53,0x3B,0x20,0x62,0x79,0x20,0x45,0x2D,0x6D,0x61,0x69,0x6C,0x20,0x61,0x74,0x20,0x43,0x50,0x53,0x2D,
			0x72,0x65,0x71,0x75,0x65,0x73,0x74,0x73,0x40,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6E,0x2E,0x63,0x6F,0x6D,0x3B,0x20,0x6F,0x72,0x20,0x62,0x79,0x20,0x6D,0x61,0x69,0x6C,0x20,0x61,0x74,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x2C,0x20,0x32,0x35,0x39,0x33,0x20,0x43,0x6F,0x61,0x73,0x74,0x20,0x41,0x76,0x65,0x2E,0x2C,0x20,0x4D,0x6F,0x75,0x6E,0x74,0x61,0x69,0x6E,0x20,0x56,0x69,0x65,0x77,0x2C,0x20,0x43,0x41,0x20,0x39,0x34,0x30,0x34,0x33,0x20,0x55,0x53,0x41,0x20,0x54,0x65,
			0x6C,0x2E,0x20,0x2B,0x31,0x20,0x28,0x34,0x31,0x35,0x29,0x20,0x39,0x36,0x31,0x2D,0x38,0x38,0x33,0x30,0x20,0x43,0x6F,0x70,0x79,0x72,0x69,0x67,0x68,0x74,0x20,0x28,0x63,0x29,0x20,0x31,0x39,0x39,0x36,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x20,0x20,0x41,0x6C,0x6C,0x20,0x52,0x69,0x67,0x68,0x74,0x73,0x20,0x52,0x65,0x73,0x65,0x72,0x76,0x65,0x64,0x2E,0x20,0x43,0x45,0x52,0x54,0x41,0x49,0x4E,0x20,0x57,0x41,0x52,0x52,0x41,0x4E,0x54,0x49,0x45,0x53,0x20,0x44,0x49,0x53,0x43,
			0x4C,0x41,0x49,0x4D,0x45,0x44,0x20,0x61,0x6E,0x64,0x20,0x4C,0x49,0x41,0x42,0x49,0x4C,0x49,0x54,0x59,0x20,0x4C,0x49,0x4D,0x49,0x54,0x45,0x44,0x2E,0xA0,0x0E,0x06,0x0C,0x60,0x86,0x48,0x01,0x86,0xF8,0x45,0x01,0x07,0x01,0x01,0x01,0xA1,0x0E,0x06,0x0C,0x60,0x86,0x48,0x01,0x86,0xF8,0x45,0x01,0x07,0x01,0x01,0x02,0x30,0x2C,0x30,0x2A,0x16,0x28,0x68,0x74,0x74,0x70,0x73,0x3A,0x2F,0x2F,0x77,0x77,0x77,0x2E,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6E,0x2E,0x63,0x6F,0x6D,0x2F,0x72,0x65,0x70,0x6F,0x73,0x69,0x74,0x6F,
			0x72,0x79,0x2F,0x43,0x50,0x53,0x20,0x30,0x11,0x06,0x09,0x60,0x86,0x48,0x01,0x86,0xF8,0x42,0x01,0x01,0x04,0x04,0x03,0x02,0x07,0x80,0x30,0x36,0x06,0x09,0x60,0x86,0x48,0x01,0x86,0xF8,0x42,0x01,0x08,0x04,0x29,0x16,0x27,0x68,0x74,0x74,0x70,0x73,0x3A,0x2F,0x2F,0x77,0x77,0x77,0x2E,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6E,0x2E,0x63,0x6F,0x6D,0x2F,0x72,0x65,0x70,0x6F,0x73,0x69,0x74,0x6F,0x72,0x79,0x2F,0x43,0x50,0x53,0x30,0x82,0x04,0x87,0x06,0x09,0x60,0x86,0x48,0x01,0x86,0xF8,0x42,0x01,0x0D,0x04,0x82,0x04,
			0x78,0x16,0x82,0x04,0x74,0x43,0x41,0x55,0x54,0x49,0x4F,0x4E,0x3A,0x20,0x54,0x68,0x65,0x20,0x43,0x6F,0x6D,0x6D,0x6F,0x6E,0x20,0x4E,0x61,0x6D,0x65,0x20,0x69,0x6E,0x20,0x74,0x68,0x69,0x73,0x20,0x43,0x6C,0x61,0x73,0x73,0x20,0x31,0x20,0x44,0x69,0x67,0x69,0x74,0x61,0x6C,0x20,0x0A,0x49,0x44,0x20,0x69,0x73,0x20,0x6E,0x6F,0x74,0x20,0x61,0x75,0x74,0x68,0x65,0x6E,0x74,0x69,0x63,0x61,0x74,0x65,0x64,0x20,0x62,0x79,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x2E,0x20,0x49,0x74,0x20,0x6D,0x61,0x79,0x20,0x62,
			0x65,0x20,0x74,0x68,0x65,0x0A,0x68,0x6F,0x6C,0x64,0x65,0x72,0x27,0x73,0x20,0x72,0x65,0x61,0x6C,0x20,0x6E,0x61,0x6D,0x65,0x20,0x6F,0x72,0x20,0x61,0x6E,0x20,0x61,0x6C,0x69,0x61,0x73,0x2E,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x20,0x64,0x6F,0x65,0x73,0x20,0x61,0x75,0x74,0x68,0x2D,0x0A,0x65,0x6E,0x74,0x69,0x63,0x61,0x74,0x65,0x20,0x74,0x68,0x65,0x20,0x65,0x2D,0x6D,0x61,0x69,0x6C,0x20,0x61,0x64,0x64,0x72,0x65,0x73,0x73,0x20,0x6F,0x66,0x20,0x74,0x68,0x65,0x20,0x68,0x6F,0x6C,0x64,0x65,0x72,0x2E,
			0x0A,0x0A,0x54,0x68,0x69,0x73,0x20,0x63,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x20,0x69,0x6E,0x63,0x6F,0x72,0x70,0x6F,0x72,0x61,0x74,0x65,0x73,0x20,0x62,0x79,0x20,0x72,0x65,0x66,0x65,0x72,0x65,0x6E,0x63,0x65,0x2C,0x20,0x61,0x6E,0x64,0x20,0x0A,0x69,0x74,0x73,0x20,0x75,0x73,0x65,0x20,0x69,0x73,0x20,0x73,0x74,0x72,0x69,0x63,0x74,0x6C,0x79,0x20,0x73,0x75,0x62,0x6A,0x65,0x63,0x74,0x20,0x74,0x6F,0x2C,0x20,0x74,0x68,0x65,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x20,0x0A,0x43,0x65,0x72,
			0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6F,0x6E,0x20,0x50,0x72,0x61,0x63,0x74,0x69,0x63,0x65,0x20,0x53,0x74,0x61,0x74,0x65,0x6D,0x65,0x6E,0x74,0x20,0x28,0x43,0x50,0x53,0x29,0x2C,0x20,0x61,0x76,0x61,0x69,0x6C,0x61,0x62,0x6C,0x65,0x0A,0x69,0x6E,0x20,0x74,0x68,0x65,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x20,0x72,0x65,0x70,0x6F,0x73,0x69,0x74,0x6F,0x72,0x79,0x20,0x61,0x74,0x3A,0x20,0x0A,0x68,0x74,0x74,0x70,0x73,0x3A,0x2F,0x2F,0x77,0x77,0x77,0x2E,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6E,0x2E,
			0x63,0x6F,0x6D,0x3B,0x20,0x62,0x79,0x20,0x45,0x2D,0x6D,0x61,0x69,0x6C,0x20,0x61,0x74,0x0A,0x43,0x50,0x53,0x2D,0x72,0x65,0x71,0x75,0x65,0x73,0x74,0x73,0x40,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6E,0x2E,0x63,0x6F,0x6D,0x3B,0x20,0x6F,0x72,0x20,0x62,0x79,0x20,0x6D,0x61,0x69,0x6C,0x20,0x61,0x74,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x2C,0x0A,0x49,0x6E,0x63,0x2E,0x2C,0x20,0x32,0x35,0x39,0x33,0x20,0x43,0x6F,0x61,0x73,0x74,0x20,0x41,0x76,0x65,0x2E,0x2C,0x20,0x4D,0x6F,0x75,0x6E,0x74,0x61,0x69,0x6E,
			0x20,0x56,0x69,0x65,0x77,0x2C,0x20,0x43,0x41,0x20,0x39,0x34,0x30,0x34,0x33,0x20,0x55,0x53,0x41,0x0A,0x0A,0x43,0x6F,0x70,0x79,0x72,0x69,0x67,0x68,0x74,0x20,0x28,0x63,0x29,0x31,0x39,0x39,0x36,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x20,0x20,0x41,0x6C,0x6C,0x20,0x52,0x69,0x67,0x68,0x74,0x73,0x20,0x0A,0x52,0x65,0x73,0x65,0x72,0x76,0x65,0x64,0x2E,0x20,0x43,0x45,0x52,0x54,0x41,0x49,0x4E,0x20,0x57,0x41,0x52,0x52,0x41,0x4E,0x54,0x49,0x45,0x53,0x20,0x44,0x49,0x53,0x43,
			0x4C,0x41,0x49,0x4D,0x45,0x44,0x20,0x41,0x4E,0x44,0x20,0x0A,0x4C,0x49,0x41,0x42,0x49,0x4C,0x49,0x54,0x59,0x20,0x4C,0x49,0x4D,0x49,0x54,0x45,0x44,0x2E,0x0A,0x0A,0x57,0x41,0x52,0x4E,0x49,0x4E,0x47,0x3A,0x20,0x54,0x48,0x45,0x20,0x55,0x53,0x45,0x20,0x4F,0x46,0x20,0x54,0x48,0x49,0x53,0x20,0x43,0x45,0x52,0x54,0x49,0x46,0x49,0x43,0x41,0x54,0x45,0x20,0x49,0x53,0x20,0x53,0x54,0x52,0x49,0x43,0x54,0x4C,0x59,0x0A,0x53,0x55,0x42,0x4A,0x45,0x43,0x54,0x20,0x54,0x4F,0x20,0x54,0x48,0x45,0x20,0x56,0x45,0x52,0x49,
			0x53,0x49,0x47,0x4E,0x20,0x43,0x45,0x52,0x54,0x49,0x46,0x49,0x43,0x41,0x54,0x49,0x4F,0x4E,0x20,0x50,0x52,0x41,0x43,0x54,0x49,0x43,0x45,0x0A,0x53,0x54,0x41,0x54,0x45,0x4D,0x45,0x4E,0x54,0x2E,0x20,0x20,0x54,0x48,0x45,0x20,0x49,0x53,0x53,0x55,0x49,0x4E,0x47,0x20,0x41,0x55,0x54,0x48,0x4F,0x52,0x49,0x54,0x59,0x20,0x44,0x49,0x53,0x43,0x4C,0x41,0x49,0x4D,0x53,0x20,0x43,0x45,0x52,0x54,0x41,0x49,0x4E,0x0A,0x49,0x4D,0x50,0x4C,0x49,0x45,0x44,0x20,0x41,0x4E,0x44,0x20,0x45,0x58,0x50,0x52,0x45,0x53,0x53,0x20,
			0x57,0x41,0x52,0x52,0x41,0x4E,0x54,0x49,0x45,0x53,0x2C,0x20,0x49,0x4E,0x43,0x4C,0x55,0x44,0x49,0x4E,0x47,0x20,0x57,0x41,0x52,0x52,0x41,0x4E,0x54,0x49,0x45,0x53,0x0A,0x4F,0x46,0x20,0x4D,0x45,0x52,0x43,0x48,0x41,0x4E,0x54,0x41,0x42,0x49,0x4C,0x49,0x54,0x59,0x20,0x4F,0x52,0x20,0x46,0x49,0x54,0x4E,0x45,0x53,0x53,0x20,0x46,0x4F,0x52,0x20,0x41,0x20,0x50,0x41,0x52,0x54,0x49,0x43,0x55,0x4C,0x41,0x52,0x0A,0x50,0x55,0x52,0x50,0x4F,0x53,0x45,0x2C,0x20,0x41,0x4E,0x44,0x20,0x57,0x49,0x4C,0x4C,0x20,0x4E,0x4F,
			0x54,0x20,0x42,0x45,0x20,0x4C,0x49,0x41,0x42,0x4C,0x45,0x20,0x46,0x4F,0x52,0x20,0x43,0x4F,0x4E,0x53,0x45,0x51,0x55,0x45,0x4E,0x54,0x49,0x41,0x4C,0x2C,0x0A,0x50,0x55,0x4E,0x49,0x54,0x49,0x56,0x45,0x2C,0x20,0x41,0x4E,0x44,0x20,0x43,0x45,0x52,0x54,0x41,0x49,0x4E,0x20,0x4F,0x54,0x48,0x45,0x52,0x20,0x44,0x41,0x4D,0x41,0x47,0x45,0x53,0x2E,0x20,0x53,0x45,0x45,0x20,0x54,0x48,0x45,0x20,0x43,0x50,0x53,0x0A,0x46,0x4F,0x52,0x20,0x44,0x45,0x54,0x41,0x49,0x4C,0x53,0x2E,0x0A,0x0A,0x43,0x6F,0x6E,0x74,0x65,0x6E,
			0x74,0x73,0x20,0x6F,0x66,0x20,0x74,0x68,0x65,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x20,0x72,0x65,0x67,0x69,0x73,0x74,0x65,0x72,0x65,0x64,0x0A,0x6E,0x6F,0x6E,0x76,0x65,0x72,0x69,0x66,0x69,0x65,0x64,0x53,0x75,0x62,0x6A,0x65,0x63,0x74,0x41,0x74,0x74,0x72,0x69,0x62,0x75,0x74,0x65,0x73,0x20,0x65,0x78,0x74,0x65,0x6E,0x73,0x69,0x6F,0x6E,0x20,0x76,0x61,0x6C,0x75,0x65,0x20,0x73,0x68,0x61,0x6C,0x6C,0x20,0x0A,0x6E,0x6F,0x74,0x20,0x62,0x65,0x20,0x63,0x6F,0x6E,0x73,0x69,0x64,0x65,0x72,0x65,0x64,0x20,
			0x61,0x73,0x20,0x61,0x63,0x63,0x75,0x72,0x61,0x74,0x65,0x20,0x69,0x6E,0x66,0x6F,0x72,0x6D,0x61,0x74,0x69,0x6F,0x6E,0x20,0x76,0x61,0x6C,0x69,0x64,0x61,0x74,0x65,0x64,0x20,0x0A,0x62,0x79,0x20,0x74,0x68,0x65,0x20,0x49,0x41,0x2E,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x04,0x05,0x00,0x03,0x81,0x81,0x00,0x2B,0x3D,0x44,0xC7,0x32,0x59,0xAE,0xF1,0x5F,0x8F,0x3F,0x87,0xE3,0x3E,0xEB,0x81,0x30,0xF8,0xA9,0x96,0xDB,0x01,0x42,0x0B,0x04,0xEF,0x37,0x02,0x3F,0xD4,0x20,0x61,0x58,0xC4,0x4A,0x3A,
			0x39,0xB3,0xFB,0xD9,0xF8,0xA5,0xC4,0x5E,0x33,0x5A,0x0E,0xFA,0x93,0x56,0x2F,0x6F,0xD6,0x61,0xA2,0xAF,0xA5,0x0C,0x1D,0xE2,0x41,0x65,0xF3,0x40,0x75,0x66,0x83,0xD2,0x5A,0xB4,0xB7,0x56,0x0B,0x8E,0x0D,0xA1,0x33,0x13,0x7D,0x49,0xC3,0xB1,0x00,0x68,0x83,0x7F,0xB5,0x66,0xD4,0x32,0x32,0xFE,0x8B,0x9A,0x5A,0xD6,0x01,0x72,0x31,0x5D,0x85,0x91,0xBC,0x93,0x9B,0x65,0x60,0x25,0xC6,0x1F,0xBC,0xDD,0x69,0x44,0x62,0xC2,0xB2,0x6F,0x46,0xAB,0x2F,0x20,0xA5,0x6F,0xDA,0x48,0x6C,0x9C };

		public const string xml = "<wsse:BinarySecurityToken ValueType=\"wsse:X509v3\" EncodingType=\"wsse:Base64Binary\" xmlns:wsu=\"http://schemas.xmlsoap.org/ws/2002/07/utility\" wsu:Id=\"SecurityToken-9a43da12-9834-4119-a3c4-c042bdc56b25\" xmlns:wsse=\"http://schemas.xmlsoap.org/ws/2002/07/secext\">MIICKjCCAZegAwIBAgIQkcRLDbfYEIRCJnGzl7UAlzAJBgUrDgMCHQUAMCgxJjAkBgNVBAMTHU1vdHVzIFRlY2hub2xvZ2llcyBpbmMuKHRlc3QpMB4XDTAzMDgxMzAwNDM0N1oXDTM5MTIzMTIzNTk1OVowEzERMA8GA1UEAxMIRkFSU0NBUEUwgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJAoGBANLLRyH1Yt01vx3smkwHLAHwKMKCF45YMtVMrIa0yeshJvMiMMV6o1rdU6scBj6yE8QFHZWLCnFxEadHJmHxdr41cjLFy0ekIkEerSkRDTkiDHmQxlKhEPZVCU5RJkcOlOaB9RhrmfB287JMkem6Oz9uY9oS0QtzDhLHcHciA51dAgMBAAGjcjBwMBMGA1UdJQQMMAoGCCsGAQUFBwMBMFkGA1UdAQRSMFCAEK7XgIimPbpQoX5X5UDJb8WhKjAoMSYwJAYDVQQDEx1Nb3R1cyBUZWNobm9sb2dpZXMgaW5jLih0ZXN0KYIQna6jOUcO1KJJeOpsug3enDAJBgUrDgMCHQUAA4GBADKKfq3nZ55cTNgzWWjPlMA2R3qnhcLd2NoRPGbBg+OrMwZ842oVcriDPQurPO51E71cliVWNgX6rtT0z1LsEbXqnyCjyDRyWQlR5zaHhoaYtTB7+z3MXujJSeDG6gJ2AeC7inDrB4boBOdI5GyQ5hZCtLvAxIJf+Pt+sp7CeCaG</wsse:BinarySecurityToken>";

		private void CheckValueType (X509SecurityToken xst) 
		{
			AssertNotNull ("ValueType!=null", xst);
			Assert ("ValueType.IsEmpty", !xst.ValueType.IsEmpty);
			AssertEquals ("ValueType.Name", "X509v3", xst.ValueType.Name);
			AssertEquals ("ValueType.Namespace", WSSecurity.NamespaceURI, xst.ValueType.Namespace);
		}

		[Test]
		public void ConstructorX509 ()
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			AssertNotNull ("X509SecurityToken(X509Certificate)", xst);
			AssertNotNull ("X509SecurityToken.Certificate", xst.Certificate);
			AssertNotNull ("X509SecurityToken.RawData", xst.RawData);
			// check "static" properties (no need to do it for every certificate)
			Assert ("EncodingType.IsEmpty", !xst.EncodingType.IsEmpty);
			AssertEquals ("EncodingType.Name", "Base64Binary", xst.EncodingType.Name);
			AssertEquals ("EncodingType.Namespace", WSSecurity.NamespaceURI, xst.EncodingType.Namespace);
			CheckValueType (xst);
		}

		[Test]
		[ExpectedException (typeof (ArgumentNullException))] 
		public void ConstructorX509Null () 
		{
			X509Certificate x = null;
			X509SecurityToken xst = new X509SecurityToken (x);
		}

		[Test]
		public void ConstructorXmlElement () 
		{
			XmlDocument doc = new XmlDocument ();
			doc.LoadXml (xml);
			X509SecurityToken xst = new X509SecurityToken (doc.DocumentElement);
			AssertNotNull ("X509SecurityToken(XmlElement)", xst);
			AssertNotNull ("X509SecurityToken.Certificate", xst.Certificate);
			// check "static" properties (no need to do it for every certificate)
			Assert ("EncodingType.IsEmpty", !xst.EncodingType.IsEmpty);
			AssertEquals ("EncodingType.Name", "Base64Binary", xst.EncodingType.Name);
			AssertEquals ("EncodingType.Namespace", WSSecurity.NamespaceURI, xst.EncodingType.Namespace);
			Assert ("ValueType.IsEmpty", !xst.ValueType.IsEmpty);
			AssertEquals ("ValueType.Name", "X509v3", xst.ValueType.Name);
			AssertEquals ("ValueType.Namespace", WSSecurity.NamespaceURI, xst.ValueType.Namespace);
		}

		[Test]
		[ExpectedException (typeof (ArgumentNullException))] 
		public void ConstructorXmlElementNull () 
		{
			XmlElement xel = null;
			X509SecurityToken xst = new X509SecurityToken (xel);
		}

		[Test]
		public void Certificate () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			AssertEquals ("X509SecurityToken.Certificate", BitConverter.ToString (cert1), BitConverter.ToString (xst.Certificate.GetRawCertData ()));
		}

		[Test]
		// LAMESPEC: [ExpectedException (typeof (ArgumentNullException))] 
		public void CertificateNull ()
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			xst.Certificate = null;
			// oops - many things can go wrong if the certificate is null!
			// so I think it's a bug, not just some lame spec
			AssertNull (xst.Certificate);
		}

		[Test]
		public void NullCertificateRawData () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			xst.Certificate = null;
			byte[] rd = xst.RawData;
			AssertNull ("(Certificate==null).RawData", rd);
		}

		[Test]
		// supports both DigitalSignature and DataEncryption
		public void TestCertificate1 () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			// check properties
			AssertEquals ("RawData.Length", cert1.Length, xst.RawData.Length);
			Assert ("Certificate1.SupportsDigitalSignature", xst.SupportsDigitalSignature);
			AssertNotNull ("Certificate1.AuthenticationKey", xst.AuthenticationKey);
			Assert ("Certificate1.SupportsDataEncryption", xst.SupportsDataEncryption);
			AssertNotNull ("Certificate1.EncryptionKey", xst.EncryptionKey);
		}

		[Test]
		// supports both DigitalSignature and DataEncryption
		public void TestCertificate2 () 
		{
			X509Certificate x = new X509Certificate (cert2);
			X509SecurityToken xst = new X509SecurityToken (x);
			// check properties
			AssertEquals ("RawData.Length", cert2.Length, xst.RawData.Length);
			Assert ("Certificate2.SupportsDigitalSignature", xst.SupportsDigitalSignature);
			AssertNotNull ("Certificate2.AuthenticationKey", xst.AuthenticationKey);
			Assert ("Certificate2.SupportsDataEncryption", xst.SupportsDataEncryption);
			AssertNotNull ("Certificate2.EncryptionKey", xst.EncryptionKey);
		}

		[Test]
		public void GetXml () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			XmlDocument doc = new XmlDocument ();
			XmlElement xel = xst.GetXml (doc);
			AssertEquals ("XML length", xml.Length, xel.OuterXml.Length);
		}

		[Test]
		[ExpectedException (typeof (ArgumentNullException))] 
		public void GetXmlNull () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			xst.GetXml (null);
		}

		[Test]
		public void LoadXml () 
		{
			X509Certificate x = new X509Certificate (cert2);
			X509SecurityToken xst = new X509SecurityToken (x);
			XmlDocument doc = new XmlDocument ();
			doc.LoadXml (xml);
			xst.LoadXml (doc.DocumentElement);
			CheckValueType (xst);
			AssertEquals ("Same cert", BitConverter.ToString (cert1), BitConverter.ToString (xst.RawData));
		}

		[Test]
		[ExpectedException (typeof (ArgumentNullException))] 
		public void LoadXmlNull () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			xst.LoadXml (null);
		}

		[Test]
		public void AuthenticationKey () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			AssertNotNull ("Certificate1.AuthenticationKey", xst.AuthenticationKey);
			Assert ("X509Certificate.SupportsDigitalSignature", x.SupportsDigitalSignature);
		}

		[Test]
		[ExpectedException (typeof (InvalidOperationException))]
		public void NullCertificateAuthenticationKey () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			xst.Certificate = null;
			AssertNotNull ("(Certificate==null).AuthenticationKey", xst.AuthenticationKey);
		}

		[Test]
		[Ignore ("need to complete X509Store")]
		public void DecryptionKey () 
		{
			// TODO with a certificate associated with a private key
		}

		[Test]
		[ExpectedException (typeof (InvalidOperationException))]
		public void NullCertificateDecryptionKey () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			xst.Certificate = null;
			AssertNotNull ("(Certificate==null).DecryptionKey", xst.DecryptionKey);
		}

		[Test]
		[ExpectedException (typeof (Win32Exception))]
		public void CertificateWithoutPrivateKeyDecryptionKey () 
		{
			X509Certificate x = new X509Certificate (cert2);
			X509SecurityToken xst = new X509SecurityToken (x);
			AssertNotNull ("Certificate2.DecryptionKey", xst.DecryptionKey);
			Assert ("X509Certificate.SupportsDataEncryption", x.SupportsDataEncryption);
		}

		[Test]
		public void EncryptionKey () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			AssertNotNull ("Certificate1.EncryptionKey", xst.EncryptionKey);
			Assert ("X509Certificate.SupportsDataEncryption", x.SupportsDataEncryption);
		}

		[Test]
		[ExpectedException (typeof (InvalidOperationException))]
		public void NullCertificateEncryptionKey () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			xst.Certificate = null;
			AssertNotNull ("(Certificate==null).EncryptionKey", xst.EncryptionKey);
		}

		[Test]
		public void RawData () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			AssertEquals ("RawData1", BitConverter.ToString (cert1), BitConverter.ToString (xst.RawData));
			// kind of indirect null
			xst.Certificate = null;
			AssertNull ("(Certificate==null).RawData", xst.RawData);
			xst.RawData = cert1;
			AssertEquals ("RawData2", BitConverter.ToString (cert1), BitConverter.ToString (xst.RawData));
		}

		[Test]
		[ExpectedException (typeof (ArgumentNullException))] 
		public void RawDataNull () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			xst.RawData = null;
		}

		[Test]
		[Ignore ("need to complete X509Store")]
		public void SignatureKey () 
		{
			// TODO with a certificate associated with a private key
		}

		[Test]
		[ExpectedException (typeof (InvalidOperationException))]
		public void NullCertificateSignatureKey () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			xst.Certificate = null;
			AssertNotNull ("(Certificate==null).SignatureKey", xst.SignatureKey);
		}

		[Test]
		[ExpectedException (typeof (Win32Exception))]
		public void CertificateWithoutPrivateKeySignatureKey () 
		{
			X509Certificate x = new X509Certificate (cert2);
			X509SecurityToken xst = new X509SecurityToken (x);
			AssertNotNull ("Certificate2.SignatureKey", xst.SignatureKey);
			Assert ("X509Certificate.SupportsDigitalSignature", x.SupportsDigitalSignature);
		}

		[Test]
		public void SupportsDataEncryption () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			Assert ("SupportsDataEncryption", xst.SupportsDataEncryption);
		}

		[Test]
		[ExpectedException (typeof (InvalidOperationException))]
		public void NullCertificateSupportsDataEncryption () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			xst.Certificate = null;
			Assert ("(Certificate==null).SupportsDataEncryption", xst.SupportsDataEncryption);
		}

		[Test]
		public void SupportsDigitalSignature () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			Assert ("SupportsDigitalSignature", xst.SupportsDigitalSignature);
		}

		[Test]
		[ExpectedException (typeof (InvalidOperationException))]
		public void NullCertificateSupportsDigitalSignature () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			xst.Certificate = null;
			Assert ("(Certificate==null).SupportsDigitalSignature", xst.SupportsDigitalSignature);
		}

		[Test]
		public void NullCertificateValueType () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			// is ValueType related to certificate presence ?
			xst.Certificate = null;
			CheckValueType (xst);
		}

		[Test]
		public void ValueType () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			XmlQualifiedName qn = new XmlQualifiedName ("X509v3", WSSecurity.NamespaceURI);
			xst.ValueType = qn;
		}

		[Test]
		[ExpectedException (typeof (SecurityFormatException))]
		public void ValueTypeInvalidName () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			XmlQualifiedName qn = new XmlQualifiedName ("?", WSSecurity.NamespaceURI);
			xst.ValueType = qn;
		}

		[Test]
		[ExpectedException (typeof (SecurityFormatException))]
		public void ValueTypeInvalidNamespace () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			XmlQualifiedName qn = new XmlQualifiedName ("X509v3", "?");
			xst.ValueType = qn;
		}

		[Test]
		[Ignore ("strangely MS code signing cert seems invalid - need another candidate")]
		public void Verify () 
		{
			X509Certificate x = new X509Certificate (null);
			X509SecurityToken xst = new X509SecurityToken (x);
			xst.Verify ();
			// ok if not exception is thrown
		}

		[Test]
		// [ExpectedException (typeof (SecurityFault))] SecurityFault is internal
		public void VerifyNullCertificate () 
		{
			X509Certificate x = new X509Certificate (cert1);
			X509SecurityToken xst = new X509SecurityToken (x);
			xst.Certificate = null;
			try {
				xst.Verify ();
			}
			catch (Exception e) {
				if (!e.ToString ().StartsWith ("Microsoft.Web.Services.Security.SecurityFault"))
					Fail ("expected SecurityFault");
			}
		}

		[Test]
		// [ExpectedException (typeof (SecurityFault))] SecurityFault is internal
		public void VerifyCertificateNotCurrent ()
		{
			X509Certificate x = new X509Certificate (cert2);
			X509SecurityToken xst = new X509SecurityToken (x);
			try {
				xst.Verify ();
			}
			catch (Exception e) {
				if (!e.ToString ().StartsWith ("Microsoft.Web.Services.Security.SecurityFault"))
					Fail ("expected SecurityFault");
			}
		}
	}
}