#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2016 Canonical, Ltd.
# 
# This library is free software; you can redistribute it and/or modify it under
# the terms of version 3 of the GNU Lesser General Public License as published
# by the Free Software Foundation.
#
# This library is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
# details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Author: Gary Wang  <gary.wang@canonical.com>

import json
import os
import sys
import tornado.httpserver
import tornado.ioloop
import tornado.netutil
import tornado.web
import xml.dom.minidom
import base64
import string
import random
import re
import time

ACCESS_TOKEN = 'the_access_token'

ROOT_FOLDER_ID = '1811asktx23a00019700101000000001'

SOCK_PORT = 0

AUTHORIZATION_BEARER = 'Bearer %s' % ACCESS_TOKEN

def read_file(path):
    file = os.path.join(os.path.dirname(__file__), path)
    if os.path.isfile(file):
        with open(file, 'r') as fp:
            content = fp.read()
    else:
        raise Exception("File '%s' not found\n" % file)
    return content

def create_file(path):
    file = os.path.join(os.path.dirname(__file__), path)
    with open(file, "wb") as fp:
        size = random.randint(1024, 1024 * 1024 * 50)
        fp.truncate(size)

def replace_folder_id(self, folder_id, resource_id):
    old_str = ('/?<parentCatalogId>.*</parentCatalogId>')
    new_str = ('<parentCatalogId>%s</parentCatalogId>') % folder_id
    path = ('resources/%s.xml') % resource_id
    file = os.path.join(os.path.dirname(__file__), path)
    if os.path.isfile(file):
        with open(file, 'r') as f:
            newlines = []
            for line in f.readlines():
                newlines.append(re.sub(old_str, new_str, line))
        with open(file, 'w') as f:
            for line in newlines:
                f.write(line)

def generate_content_info(path, id, name, size, folder_id):
    t_file = os.path.join(os.path.dirname(__file__), 'resources/content_info_template.xml')
    with open(t_file, 'r') as f:
        newlines = []
        for line in f.readlines():
            result = line.replace('[content_id]', id)
            result = result.replace('[content_name]', name)
            result = result.replace('[content_size]', size)
            newlines.append(result.replace('[parent_folder_id]', folder_id))
    file = os.path.join(os.path.dirname(__file__), path)
    with open(file, 'w') as f:
        for line in newlines:
            f.write(line)

def generate_download_info(path, id):
    d_file = os.path.join(os.path.dirname(__file__), 'resources/download_info_template.xml')
    with open(d_file, 'r') as f:
        newlines = []
        for line in f.readlines():
            newlines.append(line.replace('[content_id]', id))
    file = os.path.join(os.path.dirname(__file__), path)
    with open(file, 'w') as f:
        for line in newlines:
            f.write(line)

def generate_folder_info(path, name, folder_id):
    d_file = os.path.join(os.path.dirname(__file__), 'resources/create_folder_template.xml')
    with open(d_file, 'r') as f:
        newlines = []
        for line in f.readlines():
            results = line.replace('[folder_name]', name)
            newlines.append(results.replace('[parent_folder_id]', folder_id))
    file = os.path.join(os.path.dirname(__file__), path)
    with open(file, 'w') as f:
        for line in newlines:
            f.write(line)

def generate_move_info(path, resource_id, mode='content'):
    d_file = os.path.join(os.path.dirname(__file__), ('resources/move_%s_template.xml') % mode)
    with open(d_file, 'r') as f:
        newlines = []
        for line in f.readlines():
            newlines.append(line.replace('[%s_id]' % mode, resource_id))
    file = os.path.join(os.path.dirname(__file__), path)
    with open(file, 'w') as f:
        for line in newlines:
            f.write(line)

def generate_copy_info(path, resource_id):
    d_file = os.path.join(os.path.dirname(__file__), ('resources/copy_template.xml'))
    with open(d_file, 'r') as f:
        newlines = []
        for line in f.readlines():
            newlines.append(line.replace('[id]', resource_id))
    file = os.path.join(os.path.dirname(__file__), path)
    with open(file, 'w') as f:
        for line in newlines:
            f.write(line)

def generate_random_id(size=17, chars=string.ascii_uppercase + string.digits):
    num = ''.join(random.choice(chars) for _ in range(size))
    return '181111111111111' + num

def is_valid_id(id):
    if len(id) == 32 and id.startswith('1811'):
        return True
    return False

def is_valid_file_name(name):
    regexp = re.compile(r'[\\/:"*?<>|]+')
    if regexp.search(name) is None:
        return True
    return False
          
def is_valid_token(self):
    actual = self.request.headers.get('Authorization', '')
    if actual == 'Bearer':
        return False
    return True

def upload_response(self, uploadRequests):
    uploadContentList  = uploadRequests[0].getElementsByTagName('uploadContentList')[0]
    uploadContentInfo = uploadContentList.getElementsByTagName('uploadContentInfo')[0]
    contentName = uploadContentInfo.getElementsByTagName('contentName')[0].firstChild.data
    contentSize = uploadContentInfo.getElementsByTagName('contentSize')[0].firstChild.data
    parentCatalog = uploadRequests[0].getElementsByTagName('parentCatalogID')[0].firstChild
    parentCatalogID = ROOT_FOLDER_ID
    if parentCatalog:
        parentCatalogID = parentCatalog.data

    contentId = None
    if (contentName == 'file_not_exists_on_cloud' or contentName == 'file_exists_on_cloud'):
        file = 'resources/%s.xml' % (contentName)
    elif not is_valid_file_name(contentName):
        file = 'resources/illegal_content_name.xml'
    else:
        if contentName == "time_out":
            time.sleep(10)
        file = 'resources/upload-file-request-template.xml'
        contentId = generate_random_id()
        content_file = 'resources/content_%s.xml' % (contentId)
        generate_content_info(content_file, contentId, contentName, contentSize, parentCatalogID)

    content = read_file(file)
    content = content.replace('[port]', str(SOCK_PORT))
    content = content.replace('[upload_file_name]', contentName)
    if contentId:
        content = content.replace('[content_id]', contentId)

    self.write(content)
    self.finish()

def download_response(self, downloadRequests):
    contentId = downloadRequests[0].getElementsByTagName('contentID')[0].firstChild.data

    if is_valid_id(contentId):
        file = 'resources/download-file-request-template.xml'
    else:
        file = 'resources/invalid_content_id.xml'

    content = read_file(file)
    content = content.replace('[port]', str(SOCK_PORT))
    content = content.replace('[download_file_id]', contentId)

    self.write(content)
    self.finish()

class ErrorHandler(tornado.web.RequestHandler):
    def write_error(self, status_code, **kwargs):
        self.write(json.dumps({'error': '%s: %d' % (kwargs['exc_info'][1], status_code)}))

class DiskInfo(ErrorHandler):
    def post(self):
        if not is_valid_token(self):
            file = 'resources/invalid_token.xml'
        else:
            file = 'resources/diskinfo.xml'
        self.write(read_file(file))
        self.finish()

class Catalog(ErrorHandler):
    def post(self):
        if not is_valid_token(self):
            catalogId = 'invalid_token'
        else:
            dom = xml.dom.minidom.parseString(self.request.body)
            disk = dom.getElementsByTagName("getDisk")
            if disk:
                catalogId = disk[0].getElementsByTagName('catalogID')[0].firstChild.data
                pageIndex = disk[0].getElementsByTagName('startNumber')[0].firstChild.data
                if int(pageIndex) > 1:
                    catalogId += "_index_" + pageIndex

            catalog = dom.getElementsByTagName('createCatalogExt')
            if catalog:
                catalogName = catalog[0].getElementsByTagName('newCatalogName')[0].firstChild.data
                parentCatalogID = catalog[0].getElementsByTagName('parentCatalogID')[0].firstChild.data
                if is_valid_id(parentCatalogID):
                    catalogId = generate_random_id()
                    folder_file = 'resources/%s.xml' % (catalogId)
                    generate_folder_info(folder_file, catalogName, parentCatalogID)
                else:
                    catalogId = "invalid_folder_id"

            items = dom.getElementsByTagName('moveContentCatalog')
            if items: 
                newCatalogID = items[0].getElementsByTagName('newCatalogID')[0].firstChild.data
                catalogInfoList = items[0].getElementsByTagName('catalogInfoList')[0]
                contentInfoList = items[0].getElementsByTagName('contentInfoList')[0]
                if int(catalogInfoList.attributes["length"].value) > 1:
                    catalogId = 'move_folder_multiple'
                elif int(catalogInfoList.attributes["length"].value) > 0:
                    resourceId = catalogInfoList.getElementsByTagName('ID')[0].firstChild.data
                    catalogId = 'move_' + resourceId
                    folderId =  'folder_' + resourceId
                    move_file = 'resources/%s.xml' % (catalogId)
                    generate_move_info(move_file , resourceId , "folder");
                    replace_folder_id(self, newCatalogID, folderId)
                if int(contentInfoList.attributes['length'].value) > 1:
                    catalogId = 'move_content_multiple'
                elif int(contentInfoList.attributes['length'].value) > 0:
                    resourceId = contentInfoList.getElementsByTagName('ID')[0].firstChild.data
                    catalogId = 'move_' + resourceId
                    contentId = 'content_' + resourceId
                    move_file = 'resources/%s.xml' % (catalogId)
                    generate_move_info(move_file , resourceId, "content");
                    replace_folder_id(self, newCatalogID, contentId)

            updates = dom.getElementsByTagName('updateCatalogInfo')
            if updates:
                catalogId = 'update_folder' 

            copies = dom.getElementsByTagName('copyContentCatalog')
            if copies:
                newCatalogID = copies[0].getElementsByTagName('newCatalogID')[0].firstChild.data
                catalogInfoList = copies[0].getElementsByTagName('catalogInfoList')[0]
                contentInfoList = copies[0].getElementsByTagName('contentInfoList')[0]
                if int(catalogInfoList.attributes["length"].value) > 1:
                    catalogId = 'copy_multiple_folder'
                elif int(catalogInfoList.attributes["length"].value) > 0:
                    resourceId = catalogInfoList.getElementsByTagName('ID')[0].firstChild.data
                    catalogId = 'copy_' + resourceId
                    folderId =  'folder_' + resourceId
                    copy_file = 'resources/%s.xml' % (catalogId)
                    generate_copy_info(copy_file, resourceId);
                    replace_folder_id(self, newCatalogID, folderId)
                if int(contentInfoList.attributes['length'].value) > 1:
                    catalogId = 'copy_multiple_content'
                elif int(contentInfoList.attributes['length'].value) > 0:
                    resourceId = contentInfoList.getElementsByTagName('ID')[0].firstChild.data
                    catalogId = 'copy_' + resourceId
                    contentId = 'content_' + resourceId
                    copy_file = 'resources/%s.xml' % (catalogId)
                    generate_copy_info(copy_file, resourceId);
                    replace_folder_id(self, newCatalogID, contentId)

        file = 'resources/%s.xml' % catalogId
        self.write(read_file(file))
        self.finish()

class Content(ErrorHandler):
    def post(self):
        if not is_valid_token(self):
            file = 'resources/invalid_token.xml'
        else:
            dom = xml.dom.minidom.parseString(self.request.body)
            disk = dom.getElementsByTagName('delCatalogContent')
            if disk:
                catalogIdList = disk[0].getElementsByTagName('catalogIDs')
                contentIdList = disk[0].getElementsByTagName('contentIDs')
                if (int(catalogIdList[0].attributes['length'].value) > 0 
                        or int(contentIdList[0].attributes['length'].value) > 0):
                    file = 'resources/delete_result.xml'

            content = dom.getElementsByTagName('getContentInfo')
            if content:
                contentId = content[0].getElementsByTagName('ContentID')[0].firstChild.data
                if is_valid_id(contentId):
                    file = 'resources/content_%s.xml' % (contentId)
                else:
                    file = 'resources/invalid_content_id.xml'

        self.write(read_file(file))
        self.finish()

class Outlinks(ErrorHandler):
    def post(self):
        if not is_valid_token(self):
            file = 'resources/invalid_token.xml'
        else:
            dom = xml.dom.minidom.parseString(self.request.body)
            outLink = dom.getElementsByTagName('getOutLink')[0]
            outLinkReq  = outLink.getElementsByTagName('getOutLinkReq')[0]
            catalogIdList = outLinkReq.getElementsByTagName('caIDLst')
            contentIdList = outLinkReq.getElementsByTagName('coIDLst')

            if int(catalogIdList[0].attributes['length'].value) > 0:
                category = "folder"
                folderId = catalogIdList[0].getElementsByTagName('item')[0].firstChild.data
            if int(contentIdList[0].attributes['length'].value) > 0:
                category = "content"
                folderId = contentIdList[0].getElementsByTagName('item')[0].firstChild.data
            file = 'resources/outlink_%s_%s.xml' % (category, folderId)

        self.write(read_file(file))
        self.finish()

class UploadAndDownload(ErrorHandler):
    def post(self):
        if not is_valid_token(self):
            file = 'resources/invalid_token.xml'
            self.write(read_file(file))
            self.finish()
        else:
            dom = xml.dom.minidom.parseString(self.request.body)
            uploadRequests = dom.getElementsByTagName('pcUploadFileRequest')
            if uploadRequests:
                upload_response(self, uploadRequests)
            
            downloadRequest = dom.getElementsByTagName('downloadRequest')
            if downloadRequest:
                download_response(self, downloadRequest)

class DownloadFile(ErrorHandler):
    def get(self):
        data = self.request.arguments['filename'][0]
        data = data.replace(' ', '+') 
        contentName = str(data).decode('base64')

        file = 'files/%s' % (contentName)
        create_file(file)
        self.write(read_file(file))
        self.finish()

class UploadFile(ErrorHandler):
    def post(self):
        file = 'resources/upload_response.xml'
        self.write(read_file(file))
        self.finish()

def new_app():
    application = tornado.web.Application([
        (r'/richlifeApp/devapp/IUser', DiskInfo),
        (r'/richlifeApp/devapp/ICatalog', Catalog),
        (r'/richlifeApp/devapp/IContent', Content),
        (r'/richlifeApp/devapp/IOutLink', Outlinks),
        (r'/richlifeApp/devapp/IUploadAndDownload', UploadAndDownload),
        (r'/storageWeb/servlet/pcDownloadFile', DownloadFile),
        (r'/storageWeb/servlet/uploadFileServlet', UploadFile)
    ])
    sockets = tornado.netutil.bind_sockets(0, '127.0.0.1')
    server = tornado.httpserver.HTTPServer(application)
    server.add_sockets(sockets)

    global SOCK_PORT
    SOCK_PORT = sockets[0].getsockname()[1]
    sys.stdout.write('%d\n' % SOCK_PORT)
    sys.stdout.flush()

    return application

if __name__ == "__main__":
    application = new_app()
    tornado.ioloop.IOLoop.instance().start()
