/*
 * Copyright (C) 2016 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors: Gary Wang <gary.wang@canonical.com>
 */

#include "McloudProvider.h"
#include <unity/storage/provider/ProviderBase.h>
#include <unity/storage/provider/TempfileUploadJob.h>
#include <unity/storage/provider/UploadJob.h>
#include <unity/storage/provider/DownloadJob.h>
#include <unity/storage/provider/metadata_keys.h>
#include <unity/storage/provider/Exceptions.h>

#include <boost/thread.hpp>
#include <boost/thread/future.hpp>
#include <boost/filesystem.hpp>
#include <boost/make_shared.hpp>

#include <unistd.h>
#include <sys/socket.h>
#include <iostream>
#include <memory>
#include <mutex>
#include <stdexcept>

#include <mcloud/api/client.h>
#include <mcloud/api/uploadtask.h>
#include <mcloud/api/syncmanager.h>
#include <mcloud/api/cloudresource.h>
#include <mcloud/api/cloudcontent.h>
#include <mcloud/api/cloudfolder.h>
#include <mcloud/api/exceptions.h>

using namespace std;
using namespace unity::storage;
using namespace unity::storage::provider;
using namespace mcloud::api;

using boost::make_ready_future;
using boost::make_exceptional_future;

namespace {
    static const int   TIME_OUT     = 10;
    static const int   CLIENT_COUNT = 3;

    static const char *FOLDER_TYPE[]  = {"normal", "pictures", "music", "videos", "message", "docs", "app", "sync"};  
    static const char *CONTENT_TYPE[] = {"all", "image", "audio", "video", "other", "doc", "spreadsheet", "ppt"};
    static const char *STATUS[]       = {"unstart", "running", "canceled", "paused", "broken", "complete"};

    string make_job_id()
    {
        static int last_upload_id = 0;
        return to_string(++last_upload_id);
    }

    string status_to_string(Task::Status status)
    {
        return STATUS[int(status)];
    }

    string folder_type_to_string(CloudFolder::Type type)
    {
        return FOLDER_TYPE[int(type)];
    }

    string content_type_to_string(CloudContent::Type type)
    {
        return CONTENT_TYPE[int(type)];
    }

    string time_to_iso(time_t t)
    {
        char buf[sizeof "2016-08-11T15:10:07+0000"];
        strftime(buf, sizeof buf, "%FT%T%z", gmtime(&t));
        return buf; 
    }

    int pending_jobs_count(Client::Ptr client, McloudProvider::JobMode mode)
    {
        int count = 0;
        auto syncmanager = client->syncmanager();
        if (mode == McloudProvider::JobMode::Download) {
            auto download_task_queue = syncmanager->download_queue();
            for (auto & item_ptr : download_task_queue) {
                if (item_ptr->status() == Task::Status::Running
                    || item_ptr->status() == Task::Status::Paused
                    || item_ptr->status() == Task::Status::Unstart) {
                    count++;
                }
            }
        } else if (mode == McloudProvider::JobMode::Upload) {
            auto upload_task_queue = syncmanager->upload_queue();
            for (auto & item_ptr : upload_task_queue) {
                if (item_ptr->status() == Task::Status::Running
                    || item_ptr->status() == Task::Status::Paused
                    || item_ptr->status() == Task::Status::Unstart) {
                    count++;
                }
            }
        }

        return count;
    }

    Item content_to_item(CloudResource::Ptr resource)
    {
        Item item;
        item.item_id = resource->id();
        item.parent_ids = {resource->parent_catalog_id()};
        item.name = resource->name();
        item.etag = resource->etag();
        item.metadata["owner"] = resource->owner();
        item.metadata[provider::CREATION_TIME] = time_to_iso(resource->created_date());
        item.metadata[provider::LAST_MODIFIED_TIME] = time_to_iso(resource->updated_date());
        if (resource->property() == CloudResource::Property::Folder) {
            item.type = ItemType::folder;
            auto folder = std::static_pointer_cast<mcloud::api::CloudFolder>(resource);
            item.metadata["folder_type"] = folder_type_to_string(folder->folder_type());
            item.metadata["folder_path"] = folder->folder_path();
        } else if (resource->property() == CloudResource::Property::Content){
            item.type = ItemType::file;
            auto file = std::static_pointer_cast<mcloud::api::CloudContent>(resource);
            item.metadata["suffix"] = file->suffix();
            item.metadata["content_type"] = content_type_to_string(file->type());

            item.metadata[provider::SIZE_IN_BYTES] = file->content_size();
            item.metadata["description"] = file->description();
            item.metadata["thumbnail_url"] = file->thumbnail_url();
            item.metadata["big_thumbnail_url"] = file->big_thumbnail_url();
            item.metadata["present_url"] = file->big_thumbnail_url();
        }

        return item;
    }
}

class McloudUploadJob : public UploadJob
{
public:
    McloudUploadJob(string const& upload_id,
                    string const& parent_id, 
                    string const& file_name,
                    int64_t       size,
                    bool allow_overwrite,
                    Client::Ptr   client);

    boost::future<void> cancel() override;
    boost::future<Item> finish() override;
private:
    boost::future<tuple<bool, string>> upload_data(); 
    void   stop_and_cancel();
private:
    ssize_t     feed_size;
    string  parent_id_;
    string  file_name_;
    int64_t  upload_size_;
    bool    allow_overwrite_;

    boost::future<tuple<bool, string>> upload_future_;

    std::mutex    mutex_;

    UploadTask::Ptr task_; 
    Client::Ptr     client_;
};

class McloudDownloadJob : public DownloadJob
{
public:
    McloudDownloadJob(string const& download_id,
                      string const& item_id,
                      Client::Ptr   client);

    boost::future<void> cancel() override;
    boost::future<void> finish() override;
private:
    boost::future<tuple<bool, string>> download_data();
    void send_data();
    void stop_and_cancel();
private:
    string  item_id_;
    boost::future<tuple<bool, string>>  download_future_;

    std::mutex           mutex_;
	
    DownloadTask::Ptr    task_;
    Client::Ptr          client_;
};

McloudProvider::McloudProvider() 
    : client_list_(CLIENT_COUNT, std::make_shared<Client>(TIME_OUT))
{
}

boost::future<ItemList> McloudProvider::roots(Context const& ctx)
{
    return boost::async([this, ctx](){
        try {
            auto client = assign_client(ctx);
            auto root_folder_id = client->cloud_root_folder_id();

            ItemList roots = {
                {root_folder_id, {}, "root", "", ItemType::root, {}}
            };

            return make_ready_future<ItemList>(roots);
        } catch (runtime_error &e) {
            return make_exceptional_future<ItemList>(
                    RemoteCommsException(string("McloudProvider::roots(): failed: ") + e.what())); 
        }
    });
}

boost::future<tuple<ItemList,string>> McloudProvider::list(
    string const& item_id, string const& page_token,
    Context const& ctx)
{
    int page_token_index = 0;
    if (!page_token.empty()) {
        try {
            page_token_index = stoi(page_token);
        } catch (invalid_argument &e) {
            return make_exceptional_future<tuple<ItemList,string>>(
                    InvalidArgumentException(string("McloudProvider::list(): invalid page token: ") + e.what()));
        }
    }

    return boost::async([this, item_id, page_token_index, ctx](){
        try {
            const int count_per_page = 50;
            int start_index = 1 + page_token_index * count_per_page;
            auto client = assign_client(ctx);
            auto content_list = client->cloud_content_list(start_index, count_per_page,
                                                           CloudContent::Type::All, item_id);

            ItemList roots;
            for (const auto & content: content_list) {
                roots.push_back(content_to_item(content));
            }

            int offset = 0;
            boost::promise<tuple<ItemList,string>> prom;
            if (content_list.size() >= count_per_page) {
                offset++;
            }
            prom.set_value(make_tuple(roots, std::to_string(page_token_index + offset)));
            
            return prom.get_future();
        } catch (InvalidIDException &e) {
            return make_exceptional_future<tuple<ItemList,string>>(
                    NotExistsException(string("McloudProvider::list(): failed: ") + e.what(), item_id)); 
        } catch (runtime_error &e) {
            return make_exceptional_future<tuple<ItemList,string>>(
                    RemoteCommsException(string("McloudProvider::list(): failed: ") + e.what())); 
        }
    });
}

boost::future<ItemList> McloudProvider::lookup(
    string const& parent_id, string const& name,
    Context const& ctx)
{
    return boost::async([this, parent_id, name, ctx](){
        try {
            const int count_per_page = 50;
            int page_token_index = 0;
            int start_index = 0;

            ItemList roots;
            do {
                page_token_index++;
                start_index = 1 + (page_token_index - 1) * count_per_page;
                auto client = assign_client(ctx);
                auto content_list = client->cloud_content_list(start_index, count_per_page, 
                                                               CloudContent::Type::All, parent_id);

                for (const auto & content: content_list) {
                    if (content->name().find(name) != string::npos) {
                        roots.push_back(content_to_item(content));
                    }
                }

                if (content_list.size() < count_per_page)
                    break;
            } while (true);

            return make_ready_future<ItemList>(roots);
        } catch (InvalidIDException &e) {
            return make_exceptional_future<ItemList>(
                    NotExistsException(string("McloudProvider::lookup(): folder not exists: ") + e.what(), parent_id)); 
        } catch (runtime_error &e) {
            return make_exceptional_future<ItemList>(
                    RemoteCommsException(string("McloudProvider::lookup(): failed: ") + e.what())); 
        }
    });
}

boost::future<Item> McloudProvider::metadata(string const& item_id,
                                             Context const& ctx)
{
    return boost::async([this, item_id, ctx](){
        try {
            auto client = assign_client(ctx);
            auto content = client->content_info(item_id);
            return make_ready_future<Item>(content_to_item(content));
        } catch (NonExistentException &e) {
            return make_exceptional_future<Item>(
                    NotExistsException(string("McloudProvider::metadata(): content not exists: ") + e.what(), item_id)); 
        } catch (runtime_error &e) {
            return make_exceptional_future<Item>(
                    RemoteCommsException(string("McloudProvider::metadata(): failed: ") + e.what())); 
        }
    });
}

boost::future<Item> McloudProvider::create_folder(
        string const& parent_id, string const& name,
        Context const& ctx)
{
    return boost::async([this, parent_id, name, ctx](){
        try {
            auto client = assign_client(ctx);
            auto content = client->create_folder(name, parent_id);
            return make_ready_future<Item>(content_to_item(content));
        } catch (NonExistentException &e) {
            return make_exceptional_future<Item>(
                    NotExistsException(string("McloudProvider::create_folder(): name: ") + e.what(), name)); 
        } catch (runtime_error &e) {
            return make_exceptional_future<Item>(
                    RemoteCommsException(string("McloudProvider::create_folder(): failed: ") + e.what())); 
        }
    });
}

boost::future<unique_ptr<UploadJob>> McloudProvider::create_file(
    string const& parent_id, string const& name,
    int64_t size, string const& /*content_type*/, bool allow_overwrite,
    Context const& ctx)
{
    auto client = assign_client(ctx, JobMode::Upload);
    return make_ready_future(unique_ptr<UploadJob>(new McloudUploadJob(make_job_id(), parent_id, name, 
                                                                       size, allow_overwrite, client)));
}

boost::future<unique_ptr<UploadJob>> McloudProvider::update(
    string const& item_id, int64_t size, string const& /*old_etag*/, 
    Context const& ctx)
{
    //mcloud doesn't have update API so do some workarounds.
    //1.fetch metadata to get the content name and parent folder id;
    //2.delete that file
    //3.create a new file in the original folder with the same file name.
    //side effect: the original item id is changed after updated.

    return boost::async([this, item_id, size, ctx](){
        try {
            auto client = assign_client(ctx);
            auto content = client->content_info(item_id);
            auto file_name = content->name();
            auto parent_id = content->parent_catalog_id();

            delete_item(item_id, ctx);

            return create_file(parent_id, file_name, size, "", true, ctx);
        } catch (runtime_error &e) {
            return make_exceptional_future<unique_ptr<UploadJob>>(
                    RemoteCommsException(string("McloudProvider::update(): failed: ") + e.what()));
        }
    });
}

boost::future<unique_ptr<DownloadJob>> McloudProvider::download(
    string const& item_id, Context const& ctx)
{
    auto client = assign_client(ctx, JobMode::Download);
    return make_ready_future(unique_ptr<DownloadJob>(new McloudDownloadJob(make_job_id(), item_id, client)));
}

boost::future<void> McloudProvider::delete_item(
    string const& item_id, Context const& ctx)
{
    return boost::async([this, item_id, ctx](){
        try {
            auto client = assign_client(ctx);
            Client::Stringlist content_id_list{{item_id}};
            client->delete_contents(content_id_list);
            return make_ready_future();
        } catch (ParameterInvalidException &e) {
            return make_exceptional_future<void>(
                    NotExistsException(string("McloudProvider::delete_item(): failed: ") + e.what(), item_id)); 
        } catch (runtime_error &e) {
            return make_exceptional_future<void>(
                    RemoteCommsException(string("McloudProvider::delete_item(): failed: ") + e.what())); 
        }
    });
}

boost::future<Item> McloudProvider::move(
    string const& item_id, string const& new_parent_id,
    string const& /*new_name*/, Context const& ctx)
{
    //Mcloud doens't support changing content name when moving content
    return boost::async([this, item_id, new_parent_id, ctx](){
        try {
            auto client = assign_client(ctx);
            Client::Stringlist folderlist;
            Client::Stringlist contentlist{{item_id}};
            client->move_items(folderlist, contentlist, new_parent_id);
            auto content = client->content_info(item_id);
            return make_ready_future<Item>(content_to_item(content));
        } catch (NonExistentException &e) {
            return make_exceptional_future<Item>(
                    NotExistsException(string("McloudProvider::move(): content or folder not exist: ") + e.what(), "")); 
        } catch (runtime_error &e) {
            return make_exceptional_future<Item>(
                    RemoteCommsException(string("McloudProvider::move(): failed: ") + e.what())); 
        }
    });
}

boost::future<Item> McloudProvider::copy(
    string const& item_id, string const& new_parent_id,
    string const& /*new_name*/, Context const& ctx)
{
    //Mcloud doens't support changing content name when copying content
    return boost::async([this, item_id, new_parent_id, ctx](){
        try {
            Client::Stringlist content_list{{item_id}};
            auto client = assign_client(ctx);
            auto content_id = client->copy_contents(content_list, new_parent_id)[0];
            auto content = client->content_info(content_id);
            return make_ready_future<Item>(content_to_item(content));
        } catch (NonExistentException &e) {
            return make_exceptional_future<Item>(
                    NotExistsException(string("McloudProvider::copy(): content or folder not exist: ") + e.what(), item_id)); 
        } catch (runtime_error &e) {
            return make_exceptional_future<Item>(
                    RemoteCommsException(string("McloudProvider::copy(): failed: ") + e.what())); 
        }
    });
}

Client::Ptr McloudProvider::assign_client(Context const& ctx, JobMode mode) {
    auto access_token = boost::get<OAuth2Credentials>(ctx.credentials).access_token;
    if (getenv("MCLOUD_LOCAL_SERVER_URL")) {
        access_token = "valid_token";
    }
             
    std::cout <<"mcloud access_token:  " << access_token << std::endl;

    int index = 0;
    if (mode == JobMode::Unknown) {
        static int c_index = 0; c_index++; c_index %= CLIENT_COUNT;
        index = c_index;
    } else {
        int min_count = 0;
        for (size_t i = 0; i < client_list_.size(); i++) {
            auto client_ptr = client_list_[i];
            auto pending_count = pending_jobs_count(client_ptr, mode);
            if (i == 0 || pending_count < min_count) {
                min_count = pending_count;
                index = i;
            }
        }
    }

    auto client = client_list_[index];
    client->set_access_token(access_token);
    return client;
}

McloudUploadJob::McloudUploadJob(string const &upload_id,
                                 string const& parent_id, 
                                 string const& file_name,
                                 int64_t     size,
                                 bool allow_overwrite,
                                 Client::Ptr  client)
    : UploadJob(upload_id)
    , parent_id_(parent_id)
    , file_name_(file_name)
    , upload_size_(size)
    , allow_overwrite_(allow_overwrite)
    , task_(nullptr)
    , client_(client) 
{
    upload_future_ = upload_data();
}

boost::future<void> McloudUploadJob::cancel()
{
    printf("cancel_upload('%s')\n", upload_id().c_str());
    stop_and_cancel();

    return make_ready_future();
}

boost::future<Item> McloudUploadJob::finish()
{
    printf("finish_upload('%s')\n", upload_id().c_str());
    return boost::async([this](){
        try {
            auto upload_ctx = upload_future_.get();
            auto upload_ok = get<0>(upload_ctx);
            auto error_str = get<1>(upload_ctx);
            if (upload_ok){
                auto content = client_->content_info(task_->content_id());
                return make_ready_future<Item>(content_to_item(content));
            }

            return make_exceptional_future<Item>(
                    RemoteCommsException(string("McloudUploadJob::finish(): failed: ") + error_str)); 
        } catch (runtime_error &e) {
            return make_exceptional_future<Item>(
                    RemoteCommsException(string("McloudUploadJob::finish(): failed: ") + e.what())); 
        }
    });
}

boost::future<tuple<bool, string>> McloudUploadJob::upload_data()
{
    auto prom = boost::make_shared<boost::promise<tuple<bool, string>>>();

    return boost::async([this, prom](){
        std::lock_guard<std::mutex> guard(mutex_);

        int socket_fd = read_socket(); 
        UploadBufferCb buffer_cb{[socket_fd](void *dest, size_t buf_size) -> size_t {
            //data reading callback.
            size_t read_size =  read(socket_fd, dest, buf_size);			
            //explicitly send EOF(-1) if read_size == 0 to notify net-cpp to throw an exception if
            //sth wrong with data reading to avoid hangs here.
            if (read_size == 0) { 
                return -1;
            }
            return read_size;
        }, (size_t)upload_size_, parent_id_, file_name_};
        
        try {
            task_ = client_->syncmanager()->add_upload_task(buffer_cb);
            task_->status_changed() = [this, prom](Task::Status status) {
                std::cout << " status: " << status_to_string(status) << std::endl;
                if (status == Task::Status::Complete){
                    prom->set_value(make_tuple(true, string()));
                } else if (status == Task::Status::Canceled || 
                        status == Task::Status::Broken) {
                    prom->set_value(make_tuple(false, task_->error_string()));
                }
            };
            task_->progress_changed() = [this](float percent) {
                cout<< "uploading: " << task_->content_name() << "  " << percent << endl;
            };
        } catch (runtime_error & e) {
            prom->set_value(make_tuple(false, e.what()));
        }

        return prom->get_future();
    });
}

void McloudUploadJob::stop_and_cancel()
{
    std::lock_guard<std::mutex> guard(mutex_);

    if (task_ != nullptr && 
            (task_->status() != Task::Status::Broken && 
             task_->status() != Task::Status::Complete)) {
        task_->cancel();
    }
}

McloudDownloadJob::McloudDownloadJob(string const &download_id,
                                     string const &item_id,
                                     Client::Ptr  client)
    : DownloadJob(download_id)
    , item_id_(item_id)
    , task_(nullptr)
    , client_(client)
{
    download_future_ = download_data();
}

boost::future<void> McloudDownloadJob::cancel()
{
    stop_and_cancel();

    printf("cancel_download('%s')\n", download_id().c_str());
    return make_ready_future();
}

boost::future<void> McloudDownloadJob::finish()
{
    return boost::async([this]() {
        auto download_ctx  = download_future_.get();
        auto download_ok = get<0>(download_ctx);
        auto error_str   = get<1>(download_ctx);
        if (download_ok){
            return make_ready_future();
        }

        return make_exceptional_future<void>(
                RemoteCommsException(string("failed to download from mcloud: ") + error_str));
    });
}

boost::future<tuple<bool, string>> McloudDownloadJob::download_data()
{
    auto prom = boost::make_shared<boost::promise<tuple<bool, string>>>();

    return boost::async([this, prom](){
        std::lock_guard<std::mutex> guard(mutex_);

        int socket_fd = write_socket();
        DownloadBufferCb buffer_cb{item_id_,
            [socket_fd](void *dest, size_t buf_size) -> size_t {
                //data writing callback.
                return write(socket_fd, dest, buf_size);
            }
        };

        try {
            task_ = client_->syncmanager()->add_download_task(buffer_cb);
            task_->status_changed() = [this, prom](Task::Status status) {
                std::cout << " status: " << status_to_string(status) << std::endl;
                if (status == Task::Status::Complete) {
                    report_complete();
                    prom->set_value(make_tuple(true, string()));
                } else if (status == Task::Status::Canceled || 
                        status == Task::Status::Broken) {
                    prom->set_value(make_tuple(false, task_->error_string()));
                }
            };
            task_->progress_changed() = [this](float percent) {
                cout<< "downloading: " << task_->content_name() << "  " << percent << endl;
            };
        } catch (runtime_error & e) {
            prom->set_value(make_tuple(false, e.what()));
        }

        return prom->get_future();
    });
}

void McloudDownloadJob::stop_and_cancel()
{
    std::lock_guard<std::mutex> guard(mutex_);

    if (task_ != nullptr && 
            (task_->status() != Task::Status::Broken && 
             task_->status() != Task::Status::Complete)) {
       task_->cancel();
    }
}

