/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Gary Wang <gary.wang@canonical.com>
 */

#ifndef MCLOUD_API_CLOUDCONTENT_H_
#define MCLOUD_API_CLOUDCONTENT_H_

#include <mcloud/api/cloudresource.h>
#include <mcloud/api/visibility.h>

namespace tinyxml2 {
    class XMLElement;
}

namespace mcloud {
namespace api {

/*!
    \class CloudContent
    \brief CloudContent represents a content object on mcloud.
     the supported content type can be found \sa CloudContent::Type
 */

class MCLOUD_API_DLL_PUBLIC CloudContent: public CloudResource {
public:
    typedef std::shared_ptr<CloudContent> Ptr;

    /*!
     * \brief All Content type .
     */
    enum class Type {
        All,        ///< all kinds of content type
        Image,      ///< png, gif, jpg content
        Audio,      ///< mp3, ogg, wav content
        Video,      ///< mp4, avi, rmvb, mkv content
        Other,      ///< except explicit declared content type in this enum
        Doc,        ///< doc, dox content
        Speadsheet, ///< xls, xlsx content
        PPT         ///< ppt content
    };

    virtual ~CloudContent() = default;

    /*!
     * \brief Returns a unique id of cloud content object.
     */
    const std::string &id() const override;

    /*!
     * \brief Returns the name of cloud content object.
     */
    const std::string &name() const override;

    /*!
     * \brief Returns the datetime when content was created.
     */
    const std::time_t &created_date() const override;

    /*!
     * \brief Returns the datetime when content was updated.
     */
    const std::time_t &updated_date() const override;

    /*!
     * \brief Returns a unique parent folder id of cloud content object.
     */
    const std::string &parent_catalog_id() const override;

    /*!
     * \brief Returns etag of a content object.
     */
    const std::string &etag() const override;

    /*!
     * \brief Returns owner id of a content object.
     */
    const std::string &owner() const override;

    /*!
     * \brief Returns resource type of an object.
     */
    CloudResource::Property property() const override;

    /*!
     * \brief Returns extension of a content object.
     */
    const std::string &suffix() const;

    /*!
     * \brief Returns content type of an object.
     */
    Type type() const;

    /*!
     * \brief Returns content size of an object in bytes.
     */
    int64_t content_size() const;

    /*!
     * \brief Returns the description of a content object.
     */
    const std::string &description() const;

    /*!
     * \brief Returns thumbnail link of a content object.
     * It will be empty if there is no such a field in response.
     */
    const std::string &thumbnail_url() const;

    /*!
     * \brief Returns big thumbnail link of a content object,
     * It will be empty if there is no such a field in response.
     */
    const std::string &big_thumbnail_url() const;

    /*!
     * \brief Returns a external playable link of a media content object,
     * It will be empty if there is no such a field in response.
     */
    const std::string &present_url() const;

private:
    CloudContent(const tinyxml2::XMLElement *root);

    friend class ClientPriv;

    class Priv;
    std::shared_ptr<Priv> p;
};

}
}

#endif // MCLOUD_API_CLOUDCONTENT_H_
