""" The default perspective menu for a workbench window. """


# Standard library imports.
import logging

# Enthought library imports.
from enthought.pyface.action.api import Group, MenuManager
from enthought.traits.api import Delegate, Instance

# Local imports.
from setattr_action import SetattrAction
from new_user_perspective_action     import NewUserPerspectiveAction
from rename_user_perspective_action  import RenameUserPerspectiveAction
from save_as_user_perspective_action import SaveAsUserPerspectiveAction
from delete_user_perspective_action  import DeleteUserPerspectiveAction


# Setup a logger for this module.
logger=logging.getLogger(__name__)


# fixme: We should unify this with the one in 'perspective_action.py'! That one
# is used by Dave in the CAD plugins.
class PerspectiveAction(SetattrAction):
    """ An action that sets the active perspective. """

    #### 'Action' interface ###################################################

    # Is the action enabled?
    enabled = Delegate('value')

    ###########################################################################
    # 'object' interface.
    ###########################################################################

    def __init__(self, **traits):
        """ Creates a new perspective action. """

        # Inherited constructor.
        super(PerspectiveAction, self).__init__(**traits)

        # Listen for the perspective being enabled/disabled.
        self.value.on_trait_change(self._on_enabled_changed, 'enabled')

        return

    ###########################################################################
    # Private interface.
    ###########################################################################

    #### Trait change handlers ################################################

    def _on_enabled_changed(self, old, new):
        """ Dynamic trait change handler. """

        self.enabled = new

        return


class PerspectiveMenu(MenuManager):
    """ The default perspective menu for a workbench window. """

    #### 'ActionManager' interface ############################################

    # The manager's unique identifier.
    id = 'PerspectivesMenu'

    #### 'MenuManager' interface ##############################################

    # The menu manager's name.
    name = 'Perspectives'

    #### 'PerspectiveMenu' interface ##########################################

    # The workbench window that the menu is part of.
    window = Instance('enthought.envisage.workbench.Window')

    # The Group that all PerspectiveAction objects are added to.
    perspective_group = Instance( Group )

    ###########################################################################
    # 'PerspectiveMenu' interface.
    ###########################################################################

    def refresh(self):
        """ Refreshes the checked state of the actions in the menu. """

        logger.debug('refreshing perspective menu')

        if self.window is not None:
            perspective = self.window.active_perspective

            # Find the action associated with this perspective.
            action_item = self.find_item(perspective.id)

            # If the action doesn't exist, then create one:
            if action_item is None:
                action = self._create_action(self.window, perspective)
                self.perspective_group.append(action)
                self.changed = True
            else:
                action = action_item.action

            # The perspective actions are all in a radio group so we only have
            # to set the one that we want, and the menu takes care of the
            # others.
            action.checked = True

        return

    def rename(self, id, name):
        """ Renames the action specified by id to name.
        """
        action_item = self.find_item( id )
        if action_item is not None:
            action_item.action.name = name
            self.changed = True

    def add(self, perspective):
        """ Adds 'switch to perspective' action for the specified perspective
            if it is not already in the menu.
        """
        # Try to find an action associated with the perspective's id.
        action_item = self.find_item(perspective.id)

        # If the action doesn't exist, then create one:
        if action_item is None:
            action = self._create_action(self.window, perspective)
            self.perspective_group.append(action)
            self.changed = True

    def remove(self, id):
        """ Removes the 'switch to perspective' action for a specified
            perspective id.
        """
        # Find the action associated with this perspective.
        action_item = self.find_item(id)

        # If the action exists, then remove it:
        if action_item is not None:
            self.perspective_group.remove(action_item)
            self.changed = True

    def update(self, id, perspective):
        """ Updates the action corresponding to 'id' to refer to the
            perspective specified by 'perspective'.
        """
        # Find the action associated with this perspective.
        action_item = self.find_item(id)

        # If the action exists, then update it:
        if action_item is not None:
            action_item.action.value = perspective


    ###########################################################################
    # Private interface.
    ###########################################################################

    def _initialize(self, window):
        """ Initializes the menu. """

        # fixme: Not sure if alphabetic sorting is appropriate in all cases,
        # but it will do for now!
        perspectives = window.perspectives[:]
        perspectives.sort(lambda x, y: cmp(x.name, y.name))

        # Create all of the actions for switching to individual perspectives:
        self.perspective_group = group = Group()
        for perspective in perspectives:
            group.append(self._create_action(window, perspective))

        self.append(group)

        # Create the group for all predefined actions:
        group = Group(
            NewUserPerspectiveAction(
                id     = 'enthought.envisage.workbench.action.new_user_'
                         'perspective_action.NewUserPerspectiveAction',
                name   = 'New Perspective...',
                window = window,
                _manager = self
            ),
            SaveAsUserPerspectiveAction(
                id       = 'enthought.envisage.workbench.action.save_as_user_'
                           'perspective_action.SaveAsUserPerspectiveAction',
                name     = 'Save Perspective As...',
                window   = window,
                _manager = self
            ),
            RenameUserPerspectiveAction(
                id       = 'enthought.envisage.workbench.action.rename_user_'
                           'perspective_action.RenameUserPerspectiveAction',
                name     = 'Rename Perspective...',
                window   = window,
                _manager = self
            ),
            DeleteUserPerspectiveAction(
                id       = 'enthought.envisage.workbench.action.delete_user_'
                           'perspective_action.DeleteUserPerspectiveAction',
                name     = 'Delete Perspective',
                window   = window,
                _manager = self
            )
        )
        self.append(group)

        return

    def _create_action(self, window, perspective):
        """ Creates an action that sets the active perspective. """

        action = PerspectiveAction(
            id             = perspective.id,
            name           = perspective.name,
            style          = 'radio',
            obj            = window,
            attribute_name = 'active_perspective',
            value          = perspective
        )

        if window.active_perspective is not None:
            if perspective.id == window.active_perspective.id:
                action.checked = True

        return action

    #### Trait change handlers ################################################

    def _window_changed(self, old, new):
        """ Static trait change handler. """

        if old is not None:
            old.on_trait_change(self.refresh,'active_perspective',remove=True)

        if new is not None:
            new.on_trait_change(self.refresh, 'active_perspective')

            # Initialize the menu.
            self._initialize(self.window)

        return

#### EOF ######################################################################
