/*
 * SysInfo.xs
 *
 * $Header: //sapdb/V74/develop/sys/src/install/perl/SysInfo.xs#17 $
 * $DateTime: 2002/06/10 11:34:37 $
 * $Change: 22112 $
 */

static char szWhatHeader[] = 
	"@(#) $Header: //sapdb/V74/develop/sys/src/install/perl/SysInfo.xs#17 $";

#ifndef __cplusplus
#ifndef HAS_BOOL
typedef char bool;
#endif
#endif

#ifndef WIN32
#define UNIX UNIX
#endif

#ifdef UNIX
#include <sys/utsname.h>
#endif

#ifdef WIN32
#include <malloc.h>
#ifndef alloca
#define alloca(size) _alloca(size)
#endif
#define snprintf _snprintf
#endif

#ifdef __hpux
#define DONT_DECLARE_STD
#endif

#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

#ifdef PERL_OBJECT
#define FILE struct _PerlIO
#endif

#if defined WIN32
#define NEED_TYPEDEF_UCHAR_T NEED_TYPEDEF_UCHAR_T
#define NEED_TYPEDEF_UNIT16_T NEED_TYPEDEF_UNIT16_T
#define NEED_TYPEDEF_UNIT32_T NEED_TYPEDEF_UNIT32_T
#define NEED_TYPEDEF_UNIT64_T NEED_TYPEDEF_UNIT64_T
#elif defined linux
#define NEED_TYPEDEF_UCHAR_T NEED_TYPEDEF_UCHAR_T
#elif defined __hpux
#define NEED_TYPEDEF_UCHAR_T NEED_TYPEDEF_UCHAR_T
#elif defined SNI
#define NEED_TYPEDEF_UNIT16_T NEED_TYPEDEF_UNIT16_T
#define NEED_TYPEDEF_UNIT32_T NEED_TYPEDEF_UNIT32_T
#define NEED_TYPEDEF_UNIT64_T NEED_TYPEDEF_UNIT64_T
#elif defined __osf__
#define NEED_TYPEDEF_UNIT16_T NEED_TYPEDEF_UNIT16_T
#define NEED_TYPEDEF_UNIT32_T NEED_TYPEDEF_UNIT32_T
#define NEED_TYPEDEF_UNIT64_T NEED_TYPEDEF_UNIT64_T
#define NEED_SNPRINTF NEED_SNPRINTF
#define HAS_SPRINTF_INTRINSIC HAS_SPRINTF_INTRINSIC
#endif

#ifdef NEED_TYPEDEF_UCHAR_T
typedef unsigned char uchar_t;
#endif
#ifdef NEED_TYPEDEF_UNIT16_T
typedef unsigned short uint16_t;
#endif
#ifdef NEED_TYPEDEF_UNIT32_T
typedef unsigned int   uint32_t;
#endif
#ifdef NEED_TYPEDEF_UNIT64_T
typedef unsigned long  uint64_t;
#endif

#ifdef _AIX
#define NEED_PRAGMA_ALLOCA NEED_PRAGMA_ALLOCA
#endif
#if defined UNIX && !defined _AIX && !defined SNI
#define NEED_ALLOCA_H NEED_ALLOCA_H
#endif

#ifdef NEED_ALLOCA_H
#include <alloca.h>
#endif

#ifdef NEED_PRAGMA_ALLOCA
#pragma alloca
#endif

#if defined __hpux || defined sun || defined _AIX
#define NEED_SYS_ERRLIST_PROTOTYPE NEED_SYS_ERRLIST_PROTOTYPE
#endif

/*
 * expect compiler warning 4113 when using Microsoft C
 * in c code generated by xs preprocessor
 */
#if _MSC_VER >= 1200
#pragma warning(disable : 4113)
#endif

/*
 * prototype and dummy function to find out libc version
 */
#if defined linux
char *gnu_get_libc_version ();
#else
static char *
gnu_get_libc_version () {
	return 0;
}
#endif

#include "SysInfo.h"

/*
 * common used type definitions
 */
typedef struct {
	int value;
	char name[64];
} names_by_value_t;

typedef struct {
	char their_name[64];
	char my_name[64];
} names_by_name_t;

typedef struct {
	char *system;
	char *version;
	char *subversion;
	char *architecture;
	char *c_runtime;
	char *cpp_runtime;
	char *cpp_runtime64;
} sysinfo_t;

/*
 * function prototypes
 */
static char *get_archname ();
static sysinfo_t *get_sysinfo ();

/*
 * common used helper funtions
 */
static char *
get_name_by_value (int value, names_by_value_t *ptr) {
	while (ptr->value >= 0) {
		if (ptr->value == value) {
			return (ptr->name);
		}
		ptr++;
	}
	return 0;
}

static char *
get_name_by_name (char *their_name, names_by_name_t *ptr) {
	if (ptr == 0) {
		return 0;
	}

	while (strlen (ptr->their_name) > 0) {
		if (strcmp (ptr->their_name, their_name) == 0) {
			return (ptr->my_name);
		}
		ptr++;
	}
	return 0;
}

#ifdef WIN32
/*
 * MS Windows PE (portable executable)
 */
typedef struct {
	uint16_t e_magic;    /* Magic number */
	uint16_t e_cblp;     /* Bytes on last page of file */
	uint16_t e_cp;       /* Pages in file */
	uint16_t e_crlc;     /* Relocations */
	uint16_t e_cparhdr;  /* Size of header in paragraphs */
	uint16_t e_minalloc; /* Minimum extra paragraphs needed */
	uint16_t e_maxalloc; /* Maximum extra paragraphs needed */
	uint16_t e_ss;       /* Initial (relative) SS value */
	uint16_t e_sp;       /* Initial SP value */
	uint16_t e_csum;     /* Checksum */
	uint16_t e_ip;       /* Initial IP value */
	uint16_t e_cs;       /* Initial (relative) CS value */
	uint16_t e_lfarlc;   /* File address of relocation table */
	uint16_t e_ovno;     /* Overlay number */
	uint16_t e_res[4];   /* Reserved words */
	uint16_t e_oemid;    /* OEM identifier (for e_oeminfo) */
	uint16_t e_oeminfo;  /* OEM information; e_oemid specific */
	uint16_t e_res2[10]; /* Reserved words */
	uint16_t e_lfanew;   /* File address of new exe header */
} win_doshdr_t;

typedef struct {
	uint16_t Machine;
	uint16_t NumberOfSections;
	uint32_t TimeDateStamp;
	uint32_t PointerToSymbolTable;
	uint32_t NumberOfSymbols;
	uint16_t SizeOfOptionalHeaders;
	uint16_t Characteristics;
} win_imagefilehdr_t;

typedef struct {
	uchar_t Signatur[4];
} win_signatur_t;

static uchar_t win_dosmagic[2] = {'M', 'Z'};
static uchar_t win_ntmagic[4] = {'P', 'E', '\0', '\0'};

static names_by_value_t win_machine_names[] = {
	{0x014c, "I386"},
	{0x0200, "IA64"},
	{-1, "unknown"}
};

static int
get_pe_offset (char *buff) {
	int rc;
	win_doshdr_t *doshdr;

	doshdr = (win_doshdr_t *) buff;
	
	if (memcmp 
	(&doshdr->e_magic, win_dosmagic, sizeof (win_dosmagic)) != 0) {
		return -1;
	}

	return (doshdr->e_lfanew);
}

static char *
get_pe_arch (char *buff) {
	win_imagefilehdr_t *winnthdr;
	win_signatur_t *signature;
	char *machine;
	names_by_value_t *ptr;
	
	signature = (win_signatur_t *) buff;
	winnthdr = (win_imagefilehdr_t *) (buff + sizeof (win_signatur_t));

	if (memcmp 
	(&signature->Signatur, win_ntmagic, sizeof (win_ntmagic)) != 0) {
		return 0;
	}

	return (get_name_by_value (winnthdr->Machine, win_machine_names));
}

#define BLOCKSIZE 512

static char *
get_arch_by_magic (char *filename) {
	FILE *fp;
	int rc;
	int offset;
	char buff[BLOCKSIZE];
	char *sz_text;

	fp = fopen (filename, "rb");
	if (fp == 0) {
		return 0;
	}

	offset = 0;

	rc = fseek (fp, offset, SEEK_SET);
	if (rc != 0) {
		fclose (fp);
		return 0;
	}

	rc = fread (buff, 1, BLOCKSIZE, fp);
	if (rc != BLOCKSIZE) {
		fclose (fp);
		return 0;
	}
		
	offset = get_pe_offset (buff);
	if (offset < 0) {
		fclose (fp);
		return 0;
	} 

	rc = fseek (fp, offset, SEEK_SET);
	if (rc != 0) {
		fclose (fp);
		return 0;
	}


	rc = fread (buff, 1, BLOCKSIZE, fp);
	if (rc != BLOCKSIZE) {
		fclose (fp);
		return 0;
	}

	sz_text = get_pe_arch (buff);
	fclose (fp);
	return (sz_text);
}

static char *
get_systemroot (char *buffer, int len) {
	int rc;
	HANDLE h_key;
	char *ptr;

	rc = RegOpenKeyEx (
	HKEY_LOCAL_MACHINE, "Software\\Microsoft\\Windows NT\\CurrentVersion",
	0, KEY_READ, &h_key);

	if (rc == ERROR_SUCCESS) {
		DWORD type;
		DWORD size;

		size = len - 1;
		rc = RegQueryValueEx (h_key, "SystemRoot", 0, &type,
		(uchar_t *) buffer, &size); 

		RegCloseKey (h_key);

		if (rc == ERROR_SUCCESS)
			return (buffer);
	}

	ptr = getenv ("SystemRoot");
	if (ptr != 0) {
		strncpy (buffer, ptr, len -1);
		return (buffer);
	}

	strncpy (buffer, "C:\\WINDOWS", len - 1);
	return (buffer);
}

static int
is_same_file (char *filename0, char *filename1) {
	HANDLE h_file0, h_file1;
	BY_HANDLE_FILE_INFORMATION fi0, fi1;
	
	h_file0 = CreateFile (filename0, 0, 0, 0, OPEN_EXISTING, 0, 0);
	if (GetLastError () != ERROR_SUCCESS) {
		return -1;
	} 

	h_file1 = CreateFile (filename1, 0, 0, 0, OPEN_EXISTING, 0, 0);
	if (GetLastError() != ERROR_SUCCESS) {
		CloseHandle (h_file0);
		return -1;
	}

	GetFileInformationByHandle (h_file0, &fi0);
	if (GetLastError() != ERROR_SUCCESS) {
		CloseHandle (h_file0);
		CloseHandle (h_file1);
		return -1;
	}

	GetFileInformationByHandle (h_file1, &fi1);
	if (GetLastError() != ERROR_SUCCESS) {
		CloseHandle (h_file0);
		CloseHandle (h_file1);
		return -1;
	}

	CloseHandle (h_file0);
	CloseHandle (h_file1);

	if (fi0.dwVolumeSerialNumber != fi1.dwVolumeSerialNumber)
		return 0;

	if (fi0.nFileIndexHigh != fi1.nFileIndexHigh)
		return 0;

	if (fi0.nFileIndexLow != fi1.nFileIndexLow)
		return 0;

	return 1;
}

static names_by_value_t os_names[] = {
	{VER_PLATFORM_WIN32_WINDOWS, "WIN32"},
	{VER_PLATFORM_WIN32_NT,      "NT"},
	{-1, "unknown"}
};

static char *
get_version () {
	OSVERSIONINFO os_info[1];
	char vers[4096];
	char *name;
	char *system;
	
	os_info->dwOSVersionInfoSize = sizeof (OSVERSIONINFO);
	GetVersionEx (os_info);

	name = get_name_by_value (os_info->dwPlatformId, os_names);
	snprintf (vers, sizeof (vers),
	          "%d.%d", os_info->dwMajorVersion, os_info->dwMinorVersion);

	system = (char *) malloc (strlen (name) + 1 + strlen (vers) + 1);
	if (system == 0)
		return 0;
	
	strcpy (system , name);
	strcat (system, " ");
	strcat (system, vers);

	return (system);
}

static char *
get_subversion () {
	OSVERSIONINFO os_info[1];
	char *subversion;
	
	os_info->dwOSVersionInfoSize = sizeof (OSVERSIONINFO);
	GetVersionEx (os_info);

	if (os_info->szCSDVersion == 0 || 
    strlen (os_info->szCSDVersion) == 0)
		return 0;

	subversion = (char *) malloc (strlen (os_info->szCSDVersion) + 1);
	if (subversion == 0)
		return 0;
	
	strcpy (subversion, os_info->szCSDVersion);
	return (subversion);
}
#endif

#ifdef UNIX
static names_by_name_t os_names[] = {
	{"AIX",           "AIX"},
	{"HP-UX",         "HPUX"},
	{"Linux",         "Linux"},
	{"OSF1",          "Tru64"},
	{"SunOS",         "Solaris"},
	{"ReliantUNIX-N", "Reliant"},
	{"ReliantUNIX-Y", "Reliant"},
	{"SINIX-N",       "Reliant"},
	{"SINIX-Y",       "Reliant"},
	{"",              "unknown"}
};

static names_by_name_t arch_names[] = {
	{"alpha",  "ALPHA"},
	{"i386",   "I386"},
	{"i486",   "I386"},
	{"i586",   "I386"},
	{"i686",   "I386"},
	{"i786",   "I386"},
	{"i886",   "I386"},
	{"i986",   "I386"},
	{"ia64",   "IA64"},
	{"IA64",   "IA64"},
	{"sun4u",  "SPARC"},
	{"",       "unknown"}
};

static names_by_name_t solaris_release_names[] = {
	{"5.4",   "2.4"},
	{"5.5.1", "2.5.1"},
	{"5.6",   "2.6"},
	{"",      "unknown"}
};

/*
 * get_sysinfo ()
 */
static sysinfo_t *
get_sysinfo () {
	struct utsname un[1];
	int rc;

	sysinfo_t *sysinfo;
	
	char *os_name;
	char *arch_name;
	char *version;
	char *subversion;
	char *c_runtime;
	char *cpp_runtime;
	char *cpp_runtime64;

	rc = uname (un);
	if (rc < 0) {
		return 0;
	}

	os_name = get_name_by_name (un->sysname, os_names);	
	if (os_name == 0) {
		return 0;
	}

	sysinfo = (sysinfo_t *) malloc (sizeof (sysinfo_t));
	if (sysinfo == 0) {
		return 0;
	}

	memset (sysinfo, 0, sizeof (sysinfo_t));

	subversion = 0;
	c_runtime = 0;
	cpp_runtime = 0;
	cpp_runtime64 = 0;

	sysinfo->system = malloc (strlen (os_name) + 1);
	if (sysinfo->system == 0) {
		free (sysinfo);
		return 0;
	}
	strcpy (sysinfo->system, os_name);

	if (strcmp (os_name, "AIX") == 0) {
		arch_name = "PowerPC";
	} else if (strcmp (os_name, "HPUX") == 0) {
		if (memcmp (un->machine, "9000", 4) == 0) {
			arch_name = "PA-RISC";
		} else {
			arch_name = get_name_by_name (un->machine, arch_names);
		}
	} else if (strcmp (os_name, "Reliant") == 0) {
		arch_name = "MIPS";
	} else {
		arch_name = get_name_by_name (un->machine, arch_names);
	}

	if (arch_name == 0) {
		sysinfo->architecture = malloc (1);
		if (sysinfo->architecture == 0) {
			free (sysinfo->system);
			free (sysinfo);
			return 0;
		}
		strcpy (sysinfo->architecture, "");
	} else {
		sysinfo->architecture = malloc (strlen (arch_name) + 1);
		if (sysinfo->architecture == 0) {
			free (sysinfo->system);
			free (sysinfo);
			return 0;
		}
		strcpy (sysinfo->architecture, arch_name);
	}

	if (strcmp (os_name, "AIX") == 0) {
		/*
		 * AIX
		 * try to get aix_release.level
		 */
		version = 0;

		{
		int rc;
		char *ptr;

		dSP;
		ENTER;
		SAVETMPS;
		PUSHMARK (sp);
		
		rc = perl_call_pv (
		"SAPDB::Install::SysInfo::get_aix_version", G_SCALAR);

		SPAGAIN;
		if (rc == 1) {
			ptr = POPp;
			if (ptr != 0) {
				version = alloca (strlen (ptr) + 1);
				strcpy (version, ptr);
			}
		}

		PUTBACK;
		FREETMPS;
		LEAVE;
		}

		/*
		 * try uname when we missed first chance
		 * we have to concat version and release
		 */
		if (version == 0) {
			version = alloca (strlen (un->version) + 1 +
			                  strlen (un->release) + 1);	

			strcpy (version, un->version);
			strcat (version, ".");
			strcat (version, un->release);
		}
	} else if (strcmp (os_name, "HPUX") == 0) {
		/*
		 * HP-UX
		 * release name looks like 'B.<dd>.<dd>'
		 * so we only have to get rid of 'B.'
		 */
		if (memcmp (un->release, "B.", 2) != 0) {
			version = alloca (1);
			strcpy (version, "");
		} else {
			version =
			alloca (strlen ((char *)(un->release) + 2) + 1);
			strcpy (version, (char *)(un->release) + 2); 
		}
	} else if (strcmp (os_name, "Linux") == 0) {
		/*
		 * Linux
		 * release name looks like '<d>.<d>.<d>'
		 * or '<d>.<d>.<d>-<some _text>
		 * so we have to get rid of the trailing text
		 */
		char *ptr;
		char *glibc_version;

		ptr = strchr (un->release, '-');
		if (ptr != 0) {
			int len;

			len = ptr - (char *)(un->release);
			version = alloca (len + 1);
			memcpy (version, (char *)(un->release), len);
			version[len] = '\0';
			
			subversion = alloca (strlen (ptr + 1) + 1);
			strcpy (subversion, ptr + 1);

		} else {
			version = alloca (strlen ((char *)(un->release)) + 1);
			strcpy (version, (char *)(un->release));
		}

		glibc_version = gnu_get_libc_version ();
		if (glibc_version != 0) {
			char *glibc_name = "GLIBC";

			c_runtime = alloca (strlen (glibc_name) + 1 +
			                    strlen (glibc_version) + 1);
			strcpy (c_runtime, glibc_name);
			strcat (c_runtime, " ");
			strcat (c_runtime, glibc_version);
		}
	} else if (strcmp (os_name, "Reliant") == 0) {
		/*
		 * Reliant
		 * really amazing, we an use the output of uname
		 */
		version = alloca (strlen ((char *)(un->release)) + 1);
		strcpy (version, (char *)(un->release));
	} else if (strcmp (os_name, "Solaris") == 0) {
		/*
		 * Solaris
		 * use mapping table for known releases or
		 * get rid of the heading '5.'
		 */
		version = get_name_by_name (un->release, solaris_release_names);
		if (version == 0 || strlen (version) == 0) {
			if (memcmp (un->release, "5.", 2) == 0) {
				version =
				alloca (strlen ((char *)(un->release) + 2) + 1);
				strcpy (version, (char *)(un->release) + 2); 
			} else {
				version = alloca (1);
				strcpy (version, "");
			}
		}

		/*
		 * get applied patch for C++-runtime
		 * using 'showrev -p'
		 */
		{
			int rc;
			char *ptr;

			dSP;
			ENTER;
			SAVETMPS;
			PUSHMARK (sp);
		
			XPUSHs (sv_2mortal (newSVpv (version, strlen (version))));
			PUTBACK;
		
			rc = perl_call_pv (
			"SAPDB::Install::SysInfo::get_solaris_cppruntime", G_ARRAY);

			SPAGAIN;
			if (rc == 1) {
				ptr = POPp;
				if (ptr != 0) {
					cpp_runtime = alloca (strlen (ptr) + 1);
					strcpy (cpp_runtime, ptr);
				}
			}

			if (rc == 2) {
				ptr = POPp;
				if (ptr != 0) {
					cpp_runtime64 = alloca (strlen (ptr) + 1);
					strcpy (cpp_runtime64, ptr);
				}

				ptr = POPp;
				if (ptr != 0) {
					cpp_runtime = alloca (strlen (ptr) + 1);
					strcpy (cpp_runtime, ptr);
				}
			}

			PUTBACK;
			FREETMPS;
			LEAVE;
		}
	} else if (strcmp (os_name, "Tru64") == 0) {
		/*
		 * get version of running kernel using 'sizer -v'
		 */
		{
		int rc;
		char *ptr;

		dSP;
		ENTER;
		SAVETMPS;
		PUSHMARK (sp);
		
		rc = perl_call_pv (
		"SAPDB::Install::SysInfo::get_tru64_version", G_SCALAR);

		SPAGAIN;
		if (rc == 1) {
			ptr = POPp;
			if (ptr != 0) {
				version = alloca (strlen (ptr) + 1);
				strcpy (version, ptr);
			} else {
				version = alloca (1);
				strcpy (version, "");
			}
		} else {
			version = alloca (1);
			strcpy (version, "");
		}

		PUTBACK;
		FREETMPS;
		LEAVE;
		}

		/*
		 * get latest applied patch kit from
		 * /var/log/patch/log/event.log
		 */
		{
		int rc;
		char *ptr;

		dSP;
		ENTER;
		SAVETMPS;
		PUSHMARK (sp);
		
		rc = perl_call_pv (
		"SAPDB::Install::SysInfo::get_tru64_subversion", G_SCALAR);

		SPAGAIN;
		if (rc == 1) {
			ptr = POPp;
			if (ptr != 0) {
				subversion = alloca (strlen (ptr) + 1);
				strcpy (subversion, ptr);
			}
		}

		PUTBACK;
		FREETMPS;
		LEAVE;
		}
	}

	if (version == 0) {
		sysinfo->version = malloc (1);
		if (sysinfo->version == 0) {
			free (sysinfo->system);
			free (sysinfo->architecture);
			free (sysinfo);
			return 0;
		}
		strcpy (sysinfo->version, "");
	} else {
		sysinfo->version = malloc (strlen (version) + 1);
		if (sysinfo->version == 0) {
			free (sysinfo->system);
			free (sysinfo->architecture);
			free (sysinfo);
			return 0;
		}
		strcpy (sysinfo->version, version);
	}

	if (subversion != 0) {
		sysinfo->subversion = malloc (strlen (subversion) + 1);
		if (sysinfo->subversion == 0) {
			free (sysinfo->version);
			free (sysinfo->system);
			free (sysinfo->architecture);
			free (sysinfo);
			return 0;
		}
		strcpy (sysinfo->subversion, subversion);
	}

	if (c_runtime != 0) {
		sysinfo->c_runtime = malloc (strlen (c_runtime) + 1);
		if (sysinfo->c_runtime == 0) {
			free (sysinfo->version);
			free (sysinfo->system);
			free (sysinfo->architecture);
			if (sysinfo->subversion) {
				free (sysinfo->subversion);
			}
			free (sysinfo);
			return 0;
		}
		strcpy (sysinfo->c_runtime, c_runtime);
	}

	if (cpp_runtime != 0) {
		sysinfo->cpp_runtime = malloc (strlen (cpp_runtime) + 1);
		if (sysinfo->cpp_runtime == 0) {
			free (sysinfo->version);
			free (sysinfo->system);
			free (sysinfo->architecture);
			if (sysinfo->subversion) {
				free (sysinfo->subversion);
			}
			if (sysinfo->c_runtime) {
				free (sysinfo->c_runtime);
			}
			free (sysinfo);
			return 0;
		}
		strcpy (sysinfo->cpp_runtime, cpp_runtime);
	}

	if (cpp_runtime64 != 0) {
		sysinfo->cpp_runtime64 = malloc (strlen (cpp_runtime64) + 1);
		if (sysinfo->cpp_runtime64 == 0) {
			free (sysinfo->version);
			free (sysinfo->system);
			free (sysinfo->architecture);
			if (sysinfo->subversion) {
				free (sysinfo->subversion);
			}
			if (sysinfo->c_runtime) {
				free (sysinfo->c_runtime);
			}
			if (sysinfo->cpp_runtime) {
				free (sysinfo->cpp_runtime);
			}
			free (sysinfo);
			return 0;
		}
		strcpy (sysinfo->cpp_runtime64, cpp_runtime64);
	}

	return (sysinfo);
}
#endif

#ifdef WIN32                                                 
/*
 * get_archname ()
 *
 * What kind of nonsense are we doing here?
 * We want to know what kind of processor architecture
 * our operating system is running
 *
 * When we are running on a windows for itanium systen
 * and we are a 32-bit application, we are inside of an
 * emulator.
 * That means we cannot see the 64-bit part of the registry,
 * we cannot see the 64-bit GetSystemInfo (), and we cannot
 * see the 64-bit %SYSTEMROT%\system32 directory.
 *
 * What  can we do now?
 * 1.) find out archname of %SYSTEMROOT%\system32\kernel32.dll
 * 2.) if this is "i386"
 *         we are in an emulation or on an i386 machine
 *     else
 *         we are on the named arhitecture
 * 3.) if we are we are in an emulation or on an i386 machine
 *        find out archname of %SYSTEMROOT%\system32\kernel32.dll
 * 4.) if this not exists
 *        we are on an i386 machine
 * 5.) if this is "i386"
 *        we are in an emulation
 */

static char *
get_archname () {
	char sz_systemroot[1024 + 1];
	char *sz_kernelpath;
	char *sz_wow64_kernelpath;
	char *sz_archname;
	char *sz_wow64_archname;

	char sz_kernelname[] = "\\system32\\kernel32.dll";
	char sz_wow64_kernelname[] = "\\syswow64\\kernel32.dll";

	/*
	 * get %SYSTEMROOT%
	 */
	get_systemroot (sz_systemroot, sizeof (sz_systemroot));;

	sz_kernelpath = (char *) alloca
	(strlen (sz_systemroot) + strlen (sz_kernelname) + 1);

	strcpy (sz_kernelpath, sz_systemroot);
	strcat (sz_kernelpath, sz_kernelname);

	sz_archname = get_arch_by_magic (sz_kernelpath);
	/*
	 * error: no file found
	 */
	if (sz_archname == 0)
		return 0;

	/*
	 * anything else then I386
	 */
	if (strcmp (sz_archname, "I386") != 0)
		return (sz_archname);

	sz_wow64_kernelpath = (char *) alloca
	(strlen (sz_systemroot) + strlen (sz_kernelname) + 1);

	strcpy (sz_wow64_kernelpath, sz_systemroot);
	strcat (sz_wow64_kernelpath, sz_wow64_kernelname);

	sz_wow64_archname = get_arch_by_magic (sz_wow64_kernelpath);

	/* 
	 * no emulation directory found
	 * return arch from real systemroot
	 */
	if (sz_wow64_archname == 0)
		return ("I386");

	/*
	 * other then i386 emulation ???
	 * return arch from real systemroot
	 */
	if (strcmp (sz_wow64_archname, "I386") != 0)
		return ("I386");

	/*
	 * not the same file, may be somone made a copy
	 * we are not in an emulation
	 */
	if (is_same_file (sz_wow64_kernelpath, sz_kernelpath) <= 0)
		return ("I386");

	return ("IA64");
}

/*
 * get_sysinfo ()
 */
static sysinfo_t *
get_sysinfo () {
	sysinfo_t *sysinfo;
	char *system = "Windows";
	char *version = 0; 
	char *architecture;
	
	sysinfo = (sysinfo_t *) malloc (sizeof (sysinfo_t));
	if (sysinfo == 0) {
		return 0;
	}

	memset (sysinfo, 0, sizeof (sysinfo_t));

	sysinfo->system = (char *) malloc (strlen (system) + 1);
	if (sysinfo->system == 0) {
		free (sysinfo);
		return 0;
	}
	strcpy (sysinfo->system, system);

	architecture = get_archname ();
	if (architecture == 0) {
		sysinfo->architecture = (char *) malloc (1);
		if (sysinfo->architecture == 0) {
			free (sysinfo->system);
			free (sysinfo);
			return 0;
		} else {
			strcpy (sysinfo->architecture, "");
		}		
	} else {
		sysinfo->architecture = (char *) malloc (strlen (architecture) + 1);
		if (sysinfo->architecture == 0) {
			free (sysinfo->system);
			free (sysinfo);
			return 0;
		} else {
			strcpy (sysinfo->architecture, architecture);
		}
	}

	sysinfo->version = get_version ();
	if (sysinfo->version == 0) {
		sysinfo->version = (char *) malloc (1);
		if (sysinfo->version == 0) {
			free (sysinfo->system);
			free (sysinfo->architecture);
			free (sysinfo);
			return 0;
		} else {
			strcpy (sysinfo->version, "");
		}
	}
 
	sysinfo->subversion = get_subversion ();
	if (sysinfo->subversion == 0) {
		sysinfo->subversion = (char *) malloc (1);
		if (sysinfo->subversion == 0) {
			free (sysinfo->system);
			free (sysinfo->version);
			free (sysinfo->architecture);
			free (sysinfo);
			return 0;
		} else {
			strcpy (sysinfo->subversion, "");
		}
	}
 
	return (sysinfo);
}
#endif

MODULE = SAPDB::Install::SysInfo	PACKAGE = SAPDB::Install::SysInfo

PROTOTYPES: DISABLE

BOOT:
	perl_eval_pv ((char *) sz_text, 1);

SV *
GetSystemInfo (...)
PREINIT:
	sysinfo_t *sysinfo;
	HV *hv;
	SV *rv;
PPCODE:
	sysinfo = get_sysinfo ();
	
	if (sysinfo == 0) {
		XSRETURN_UNDEF;
	}

	hv = newHV ();
	
	if (sysinfo->system != 0) {
		char *key = "system";

		hv_store (hv, key, strlen (key), 
		newSVpv (sysinfo->system, strlen (sysinfo->system)), 0);

		free (sysinfo->system);
	}

	if (sysinfo->version != 0) {
		char *key = "version";

		hv_store (hv, key, strlen (key), 
		newSVpv (sysinfo->version, strlen (sysinfo->version)), 0);

		free (sysinfo->version);
	}

	if (sysinfo->architecture != 0) {
		char *key = "architecture";

		hv_store (hv, key, strlen (key), 
		newSVpv (sysinfo->architecture,
		strlen (sysinfo->architecture)), 0);

		free (sysinfo->architecture);
	}

	if (sysinfo->subversion != 0) {
		char *key = "subversion";

		hv_store (hv, key, strlen (key), 
		newSVpv (sysinfo->subversion,
		strlen (sysinfo->subversion)), 0);

		free (sysinfo->subversion);
	}

	if (sysinfo->c_runtime != 0) {
		char *key = "c_runtime";

		hv_store (hv, key, strlen (key), 
		newSVpv (sysinfo->c_runtime,
		strlen (sysinfo->c_runtime)), 0);

		free (sysinfo->c_runtime);
	}

	if (sysinfo->cpp_runtime != 0) {
		char *key = "cpp_runtime";

		hv_store (hv, key, strlen (key), 
		newSVpv (sysinfo->cpp_runtime,
		strlen (sysinfo->cpp_runtime)), 0);

		free (sysinfo->cpp_runtime);
	}

	if (sysinfo->cpp_runtime64 != 0) {
		char *key = "cpp_runtime64";

		hv_store (hv, key, strlen (key), 
		newSVpv (sysinfo->cpp_runtime64,
		strlen (sysinfo->cpp_runtime64)), 0);

		free (sysinfo->cpp_runtime64);
	}

	free (sysinfo);
	rv = sv_2mortal (newRV ((SV *) hv));
	SvREFCNT_dec (hv);

	XPUSHs (rv);
	XSRETURN (1);

