#! /usr/bin/perl
#
#    LCEnv.pl - Creates and syncs a complete SAPDB make environment.
#
#    @(#)LCEnv     2003-04-22
#
#    U. Jansen, SAP AG
#
#    ========== licence begin LGPL
#    Copyright (C) 2002 SAP AG
#
#    This library is free software; you can redistribute it and/or
#    modify it under the terms of the GNU Lesser General Public
#    License as published by the Free Software Foundation; either
#    version 2.1 of the License, or (at your option) any later version.
#
#    This library is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    Lesser General Public License for more details.
#
#    You should have received a copy of the GNU Lesser General Public
#    License along with this library; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#    ========== licence end
#
##############################################################################
# Static includes:

unless ($ENV{'TOOL'})
{
    if (-e "/devtool/TOOL/tool")
    {
        $ENV{'TOOL'} = "/devtool/TOOL/tool";
    }
    elsif(-e "/rel/TOOL/tool")
    {
        $ENV{'TOOL'} = "/rel/TOOL/tool";
    }
    else
    {
        print "%TOOL% is not set in the environment!\n";
        exit (10);
    }
}

unshift @INC, "$ENV{'TOOL'}/lib/perl5";

use strict;

use Getopt::Long;
require IO::SimpleCfg;
import  IO::SimpleCfg;
require IO::FilePath;
import  IO::FilePath;
require IO::CurrentDir;
import  IO::CurrentDir;
require Tool::HashRef;
import  Tool::HashRef;
require Tool::RequestInput;
import  Tool::RequestInput;
require Env::LCEnv;
import  Env::LCEnv;

##############################################################################
# Console Preparation (redirect STDERR to STDOUT and disable buffering):

close   STDERR;
open    STDERR, ">&STDOUT";
select  STDOUT;

$| = 1;

##############################################################################
# Global variables:

my $ProgramName     = "LCEnv";
my $Version         = "1";
my $Revision        = "33";
my $Date            = "2003-10-23";

my %Opts            = ();
my $Glob            = {};
my $PathSep         = ($^O =~ /MSWin32/i) ? "\\" : "/";

$Glob->{'CFG'}      = {};
$Glob->{'DEF_CFG'}  = {};

##############################################################################
# Main routine:
##############################################################################

print "\n$ProgramName v$Version.$Revision ($Date), (c)2003 SAP AG\n\n";

# Check command line parameters for correctness:

if (!GetOptions( \%Opts,
                'h|help|?',
                'v|verbose',
                'y|yes',
                'b|batch',
                'ccl|check_cl=i',
                'd|default',
                'resync=i',
                'own=s',
                'rel=s',
                'cl=i',
                'p4u=s',
                'p4c=s',
                'p4r=s',
                'p4pw=s',
                'prod=i',
                'prod_cfg=s',
                'm=s',
                'cfg=s',
                'scratch=i',
                'c|clear',
                'wi|win_icon',
                'ws|win_share',
                'win|win_icon_name=s',
                'wsn|win_share_name=s',
                'tool=s',
                'X64',
                '64=i',
                'sync=i',
                'l|list',
                'shell',
                'rm|remove',
                'p4=i',
                'save',
                'grp=s',
                'cmd=s',
                'low=i',
                'color=s',
                'ed|edit_defaults') || $Opts{'h'} || @ARGV)
{
    Usage();
}

# First, try loading the system defaults.
my $sel = "";

if (-e "$ENV{'TOOL'}/config/LCEnv.default")
{
    $Glob->{'DEF_CFG'} = LoadSimpleCfg("$ENV{'TOOL'}/config/LCEnv.default");
}
elsif (-e "$ENV{'TOOL'}/data/LCEnv.default")
{
    $Glob->{'DEF_CFG'} = LoadSimpleCfg("$ENV{'TOOL'}/data/LCEnv.default");
}
else
{
    if ($Opts{'shell'} or $Opts{'cfg'})
    {
        $sel = "n";
    }
    else
    {
        $sel = RequestInput("Could not find any system defaults. Do you want to set them up now ? (Y/n) ", "y", "[yYnNjJ]", "Input incorrect! Please, just answer 'y' or 'n'.\n");
    }
}

if (($sel =~ /^[yj]$/i) or $Opts{'ed'})
{
    LCESetDefaults($Glob->{'DEF_CFG'});
    SaveSimpleCfg("$ENV{'TOOL'}/config/LCEnv.default", $Glob->{'DEF_CFG'});
    print "\nGlobal Configuration saved under '$ENV{'TOOL'}" . $PathSep . "config" . $PathSep . "LCEnv.default'.\n";
    print "\nExiting...\n";
    exit(0);
}

# Then, load the configuration file:
if ($Opts{'cfg'})
{

    if ($Opts{'cfg'} =~ /[\\\/]/)
    {
        $Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'} = $Opts{'cfg'};
    }
    elsif (-e $Opts{'cfg'})
    {
        $Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'} = GetCurrentDir() . $PathSep . $Opts{'cfg'};
    }
    elsif ($Glob->{'DEF_CFG'}->{'GLOBAL'}->{'CFG_PATH'})
    {
        $Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'} = $Glob->{'DEF_CFG'}->{'GLOBAL'}->{'CFG_PATH'} . $PathSep . $Opts{'cfg'};
    }
    else
    {
        $Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'} = $Opts{'cfg'};
    }

    if ($Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'})
    {
        unless ($Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'} =~ /\.cfg$/)
        {
            $Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'} .= ".cfg";
        }

        if (-e $Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'})
        {

	        my $cfg_file = $Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'};
	
	        $Glob->{'CFG'} = LoadSimpleCfg($cfg_file);
	        $Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'} = $cfg_file;
	
	        print "Configration '$Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'}' loaded.\n\n" if ($Opts{'v'});
        }
        else
        {
            print "Warning: Configuration '$Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'}' not found.\n\n";
            $Glob->{'CFG'}->{'GLOBAL'}->{'INIT'} = 1;
        }
    }
}

# Then, overload settings by given command line parameters:
ParseCmdLineParams();

# Check, if this process has to have low priority:
if ($Glob->{'CFG'}->{'GLOBAL'}->{'LOW'})
{
    if ($^O =~ /MSWin32/i)
    {
        system("$ENV{'TOOL'}\\pgm\\lowprio.exe $$ >nul");
        $ENV{'LOW_PRIO'} = 1;
    }
    else
    {
        print "Warning: Setting '-low' ignored under UNIX.\n\n";
    }
}

if ($Opts{'rm'})
{
    #=========================================================================
    # REMOVE INSTANCE
    #=========================================================================

    unless ($Opts{'cfg'})
    {
        print "Sorry, you can only remove EXISTING environments. Please, use '-cfg'!\n";
        exit(1);
    }

    print "Removing environment for '$Glob->{'CFG'}->{'RELEASE'}->{'VERSION'}'. Please be patient...\n\n";

    LCERemEnv($Glob->{'CFG'});
}
elsif ($Opts{'l'})
{
    #=========================================================================
    # LIST ALL INSTANCES
    #=========================================================================

    LCEListRels();
}
else
{
    #=========================================================================
    # INSTALL INSTANCE
    #=========================================================================

    #-------------------------------------------------------------------------
    # First, check configuration integrity and ask for missing parameters:
    #-------------------------------------------------------------------------
    LCECheckEnv($Glob->{'CFG'});

    if ((!$Glob->{'CFG'}->{'PERFORCE'}->{'USE'}) && !(-d $Glob->{'CFG'}->{'RELEASE'}->{'OWN'}))
    {
        print "\nWARNING! Since you don't want to use Perforce and the OWN-Directory doen't\n";
        print "exist, the profile generation may fail! If this happens, you have to sync your\n";
        print "sources, first.\n\n";
    }

    #-------------------------------------------------------------------------
    # Set path and name for configuration file:
    #-------------------------------------------------------------------------

    unless ($Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'})
    {
        if ($Glob->{'DEF_CFG'}->{'GLOBAL'}->{'CFG_PATH'})
        {
            $Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'} = $Glob->{'DEF_CFG'}->{'GLOBAL'}->{'CFG_PATH'} . $PathSep;
            my $tmpstr = $Glob->{'CFG'}->{'RELEASE'}->{'VERSION'};
            $tmpstr =~ s/\.//g;
            $Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'} .= $tmpstr . ".cfg";
        }
        else
        {
            $Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'} = $Glob->{'CFG'}->{'RELEASE'}->{'OWN'} . $PathSep . "LCEnv.cfg";
        }
    }

    #-------------------------------------------------------------------------
    # Print out generated configuration:
    #-------------------------------------------------------------------------

    if ($Glob->{'CFG'}->{'GLOBAL'}->{'VERBOSE'})
    {
        print "\nThe following configuration will be used:\n\n";

        ShowHashRef($Glob->{'CFG'});

        unless (!$Glob->{'CFG'}->{'GLOBAL'}->{'INIT'} || $Glob->{'CFG'}->{'GLOBAL'}->{'ASSUME_YES'} || $Glob->{'CFG'}->{'GLOBAL'}->{'BATCH_MODE'} || (RequestInput("Is this correct (Y/n)?", "y", "[yYnN]", "Input incorrect. Please, use just [y or n].\n") =~ /^y$/i))
        {
            print "Aborting...\n";
            exit (0);
        }

        print "\n";
    }

    #-------------------------------------------------------------------------
    # Now test / create / update the environment:
    #-------------------------------------------------------------------------

    LCECreateEnv($Glob->{'CFG'});

    #-------------------------------------------------------------------------
    # Save the generated configuration including all changes made:
    #-------------------------------------------------------------------------

    # First, delete the backup entries again:

    if ($Opts{'save'} or $Glob->{'CFG'}->{'GLOBAL'}->{'INIT'})
    {
        if ($Glob->{'CFG'}->{'GLOBAL'}->{'INIT'})
        {
            $Glob->{'CFG'}->{'PERFORCE'}->{'USE'} = (defined $Opts{'p4'}) ? $Opts{'p4'} : $Glob->{'DEF_CFG'}->{'PERFORCE'}->{'USE'};
        }
        undef $Glob->{'CFG'}->{'GLOBAL'}->{'INIT'};
        my $cfg_file = $Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'};
        undef $Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'};
        SaveSimpleCfg($cfg_file, $Glob->{'CFG'});
        $Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'} = $cfg_file;
        print "Release settings saved under '$Glob->{'CFG'}->{'GLOBAL'}->{'CFG_FILE'}'.\n\n";
    }

    #-------------------------------------------------------------------------
    # Make any given target:
    #-------------------------------------------------------------------------

    LCEDoMake($Glob->{'CFG'});

    #-------------------------------------------------------------------------
    # Check for CLEAR flag and eventually remove everything again:
    #-------------------------------------------------------------------------
    if ($Glob->{'CFG'}->{'ACTIONS'}->{'CLEAR'})
    {
        LCERemEnv($Glob->{'CFG'});
    }
    elsif ($Opts{'shell'})
    {
        $ENV{'P4USER'} = $Glob->{'CFG'}->{'PERFORCE'}->{'USER'};
        $ENV{'P4CLIENT'} = $Glob->{'CFG'}->{'PERFORCE'}->{'CLIENT'};
        $ENV{'P4PASSWD'} = $Glob->{'CFG'}->{'PERFORCE'}->{'PASSWD'} if ($Glob->{'CFG'}->{'PERFORCE'}->{'PASSWD'});
        $ENV{'P4PORT'} = "perforce3006.wdf.sap.corp:3006";

        if ($^O =~ /MSWin32/i)
        {
            if ($ENV{'COMSPEC'})
            {
                system("$ENV{'COMSPEC'} /T:$Glob->{'CFG'}->{'GLOBAL'}->{'COLOR'} /K $Glob->{'CFG'}->{'RELEASE'}->{'OWN'}\\iprofile.bat");
            }
            else
            {
                system("cmd.exe /T:$Glob->{'CFG'}->{'GLOBAL'}->{'COLOR'} /K $Glob->{'CFG'}->{'RELEASE'}->{'OWN'}\\iprofile.bat");
            }
        }
    }

}

##############################################################################
# ParseCmdLineParams() - Checks and copies the command line parameters.
##############################################################################

sub ParseCmdLineParams
{
    # Global options:
    $Glob->{'CFG'}->{'GLOBAL'}->{'VERBOSE'}         = $Opts{'v'}    ? 1 : 0;
    $Glob->{'CFG'}->{'GLOBAL'}->{'ASSUME_YES'}      = $Opts{'y'}    ? 1 : 0;
    $Glob->{'CFG'}->{'GLOBAL'}->{'BATCH_MODE'}      = $Opts{'b'}    ? 1 : 0;
    $Glob->{'CFG'}->{'GLOBAL'}->{'USE_DEFAULT'}     = $Opts{'d'}    ? 1 : 0;
    $Glob->{'CFG'}->{'GLOBAL'}->{'PROD_MAKE'}       = (defined $Opts{'prod'}) ? $Opts{'prod'} : ($Glob->{'CFG'}->{'GLOBAL'}->{'PROD_MAKE'} ? 1 : 0);
    $Glob->{'CFG'}->{'GLOBAL'}->{'SCRATCH'}         = (defined $Opts{'scratch'}) ? $Opts{'scratch'} : $Glob->{'CFG'}->{'GLOBAL'}->{'SCRATCH'} ? 1 : 0;
    $Glob->{'CFG'}->{'GLOBAL'}->{'TOOL_DIR'}        = $Opts{'tool'}    ? $Opts{'tool'} : ($Glob->{'CFG'}->{'GLOBAL'}->{'TOOL_DIR'} ? $Glob->{'CFG'}->{'GLOBAL'}->{'TOOL_DIR'} : "");
    $Glob->{'CFG'}->{'GLOBAL'}->{'X64'}             = $Opts{'X64'}  ? 1 : ($Glob->{'CFG'}->{'GLOBAL'}->{'X64'} ? 1 : 0);
    $Glob->{'CFG'}->{'GLOBAL'}->{'64'}              = (defined $Opts{'64'}) ? $Opts{'64'} : ($Glob->{'CFG'}->{'GLOBAL'}->{'64'}  ? 1 : 0);
    $Glob->{'CFG'}->{'GLOBAL'}->{'ICON_PATH'}       = $Glob->{'DEF_CFG'}->{'GLOBAL'}->{'ICON_PATH'};
    $Glob->{'CFG'}->{'GLOBAL'}->{'LOW'}             = (defined $Opts{'low'}) ? $Opts{'low'} : ($Glob->{'CFG'}->{'GLOBAL'}->{'LOW'}  ? 1 : 0);
    $Glob->{'CFG'}->{'GLOBAL'}->{'INIT'}            = (defined $Glob->{'CFG'}->{'GLOBAL'}->{'INIT'} ? $Glob->{'CFG'}->{'GLOBAL'}->{'INIT'} : (defined $Opts{'cfg'} ? 0 : 1));
    $Glob->{'CFG'}->{'GLOBAL'}->{'COLOR'}           = (defined $Opts{'color'}) ? $Opts{'color'} : ((defined $Glob->{'CFG'}->{'GLOBAL'}->{'COLOR'}) ? $Glob->{'CFG'}->{'GLOBAL'}->{'COLOR'} : "00");
    $Glob->{'CFG'}->{'GLOBAL'}->{'CHECK_CL'}        = (defined $Opts{'ccl'}) ? $Opts{'ccl'} : ($Glob->{'CFG'}->{'GLOBAL'}->{'CCL'}  ? 1 : 0);
    $Glob->{'CFG'}->{'GLOBAL'}->{'PROD_CFG'}        = (defined $Opts{'prod_cfg'}) ? $Opts{'prod_cfg'} : ($Glob->{'CFG'}->{'GLOBAL'}->{'PROD_CFG'} ? $Glob->{'CFG'}->{'GLOBAL'}->{'PROD_CFG'} : "");

    # Release specific options:
    $Glob->{'CFG'}->{'RELEASE'}->{'OWN'}            = $Opts{'own'}      ? $Opts{'own'} : ($Glob->{'CFG'}->{'RELEASE'}->{'OWN'} ? $Glob->{'CFG'}->{'RELEASE'}->{'OWN'} : "");
    $Glob->{'CFG'}->{'RELEASE'}->{'GRP'}            = $Opts{'grp'}      ? $Opts{'grp'} : ($Glob->{'CFG'}->{'RELEASE'}->{'GRP'} ? $Glob->{'CFG'}->{'RELEASE'}->{'GRP'} : "");
    $Glob->{'CFG'}->{'RELEASE'}->{'VERSION'}        = $Opts{'rel'}      ? $Opts{'rel'} : ($Glob->{'CFG'}->{'RELEASE'}->{'VERSION'} ? $Glob->{'CFG'}->{'RELEASE'}->{'VERSION'} : "");
    $Glob->{'CFG'}->{'RELEASE'}->{'CHANGELIST'}     = (defined $Opts{'cl'}) ? $Opts{'cl'}  : ($Glob->{'CFG'}->{'RELEASE'}->{'CHANGELIST'} ? $Glob->{'CFG'}->{'RELEASE'}->{'CHANGELIST'} : 0);

    # Perforce specific options:

    $Glob->{'CFG'}->{'PERFORCE'}->{'USE'}           = (defined $Opts{'p4'}) ? $Opts{'p4'} : ((!$Glob->{'CFG'}->{'GLOBAL'}->{'INIT'} && !$Opts{'rm'}) ? (defined $Glob->{'CFG'}->{'PERFORCE'}->{'USE'} ? $Glob->{'CFG'}->{'PERFORCE'}->{'USE'} : $Glob->{'DEF_CFG'}->{'PERFORCE'}->{'USE'}) : 1);
    $Glob->{'CFG'}->{'PERFORCE'}->{'USER'}          = $Opts{'p4u'} ? $Opts{'p4u'} : ($Glob->{'CFG'}->{'PERFORCE'}->{'USER'} ? $Glob->{'CFG'}->{'PERFORCE'}->{'USER'} : (($Glob->{'DEF_CFG'}->{'PERFORCE'}->{'ALWAYS_ASK'} or !$Glob->{'DEF_CFG'}->{'PERFORCE'}->{'USER'}) ? "" : $Glob->{'DEF_CFG'}->{'PERFORCE'}->{'USER'}));
    $Glob->{'CFG'}->{'PERFORCE'}->{'CLIENT'}        = $Opts{'p4c'} ? $Opts{'p4c'} : ($Glob->{'CFG'}->{'PERFORCE'}->{'CLIENT'} ? $Glob->{'CFG'}->{'PERFORCE'}->{'CLIENT'} : (($Glob->{'DEF_CFG'}->{'PERFORCE'}->{'ALWAYS_ASK'} or !$Glob->{'DEF_CFG'}->{'PERFORCE'}->{'CLIENT'}) ? "" : $Glob->{'DEF_CFG'}->{'PERFORCE'}->{'CLIENT'}));
    $Glob->{'CFG'}->{'PERFORCE'}->{'ROOT'}          = $Opts{'p4r'} ? $Opts{'p4r'} : ($Glob->{'CFG'}->{'PERFORCE'}->{'ROOT'} ? $Glob->{'CFG'}->{'PERFORCE'}->{'ROOT'} : (($Glob->{'DEF_CFG'}->{'PERFORCE'}->{'ALWAYS_ASK'} or !$Glob->{'DEF_CFG'}->{'PERFORCE'}->{'ROOT'}) ? "" : $Glob->{'DEF_CFG'}->{'PERFORCE'}->{'ROOT'}));
    $Glob->{'CFG'}->{'PERFORCE'}->{'_MAN_ROOT'}     = $Opts{'p4r'} ? 1 : 0;
    $Glob->{'CFG'}->{'PERFORCE'}->{'PASSWD'}        = $Opts{'p4pw'} ? $Opts{'p4pw'} : ($Glob->{'CFG'}->{'PERFORCE'}->{'PASSWD'} ? $Glob->{'CFG'}->{'PERFORCE'}->{'PASSWD'} : (($Glob->{'DEF_CFG'}->{'PERFORCE'}->{'ALWAYS_ASK'} or !$Glob->{'DEF_CFG'}->{'PERFORCE'}->{'PASSWD'}) ? "" : $Glob->{'DEF_CFG'}->{'PERFORCE'}->{'PASSWD'}));
    $Glob->{'CFG'}->{'PERFORCE'}->{'SERVER_PATH'}   = $Glob->{'CFG'}->{'PERFORCE'}->{'SERVER_PATH'} ? $Glob->{'CFG'}->{'PERFORCE'}->{'SERVER_PATH'} : "";
    $Glob->{'CFG'}->{'PERFORCE'}->{'CLIENT_PATH'}   = $Glob->{'CFG'}->{'PERFORCE'}->{'CLIENT_PATH'} ? $Glob->{'CFG'}->{'PERFORCE'}->{'CLIENT_PATH'} : "";
    $Glob->{'CFG'}->{'PERFORCE'}->{'RESYNC'}        = (defined $Opts{'resync'}) ? $Opts{'resync'} : ($Glob->{'CFG'}->{'PERFORCE'}->{'RESYNC'} ? 1 : 0);
    $Glob->{'CFG'}->{'PERFORCE'}->{'SYNC'}          = (defined $Opts{'sync'}) ? $Opts{'sync'} : ($Glob->{'CFG'}->{'PERFORCE'}->{'SYNC'} ? 1 : $Glob->{'DEF_CFG'}->{'PERFORCE'}->{'SYNC'});

    # Actions to be performed:
    $Glob->{'CFG'}->{'ACTIONS'}->{'CLEAR'}          = $Opts{'c'}  ? 1 : 0;
    $Glob->{'CFG'}->{'ACTIONS'}->{'CMD'}            = $Opts{'cmd'} ? $Opts{'cmd'} : ($Glob->{'CFG'}->{'ACTIONS'}->{'CMD'} ? $Glob->{'CFG'}->{'ACTIONS'}->{'CMD'} : "");
    $Glob->{'CFG'}->{'ACTIONS'}->{'TARGET'}         = $Opts{'m'}  ? $Opts{'m'} : ($Glob->{'CFG'}->{'ACTIONS'}->{'TARGET'} ? $Glob->{'CFG'}->{'ACTIONS'}->{'TARGET'} : "");
    $Glob->{'CFG'}->{'ACTIONS'}->{'MAKE'}           = ($Glob->{'CFG'}->{'GLOBAL'}->{'PROD_MAKE'} || $Glob->{'CFG'}->{'ACTIONS'}->{'TARGET'} || $Glob->{'CFG'}->{'ACTIONS'}->{'CMD'}) ? 1 : 0;
    $Glob->{'CFG'}->{'ACTIONS'}->{'CREATE_ICON'}    = $Opts{'wi'} ? 1 : (!$Glob->{'CFG'}->{'GLOBAL'}->{'INIT'} ? 0 : $Glob->{'DEF_CFG'}->{'ACTIONS'}->{'CREATE_ICON'});
    $Glob->{'CFG'}->{'ACTIONS'}->{'ICON_NAME'}      = ($Opts{'win'} ? $Opts{'win'} : ($Glob->{'CFG'}->{'ACTIONS'}->{'ICON_NAME'} ? $Glob->{'CFG'}->{'ACTIONS'}->{'ICON_NAME'} : ""));
    $Glob->{'CFG'}->{'ACTIONS'}->{'CREATE_SHARE'}   = $Opts{'ws'} ? 1 : (!$Glob->{'CFG'}->{'GLOBAL'}->{'INIT'} ? 0 : $Glob->{'DEF_CFG'}->{'ACTIONS'}->{'CREATE_SHARE'});
    $Glob->{'CFG'}->{'ACTIONS'}->{'SHARE_NAME'}     = ($Opts{'wsn'} ? $Opts{'wsn'} : ($Glob->{'CFG'}->{'ACTIONS'}->{'SHARE_NAME'} ? $Glob->{'CFG'}->{'ACTIONS'}->{'SHARE_NAME'} : ""));

    if ($Glob->{'CFG'}->{'ACTIONS'}->{'PROD_MAKE'} && $Glob->{'CFG'}->{'ACTIONS'}->{'TARGET'})
    {
        print "WARNING! Setting a specific target in productive mode is prohibited.\nThe target '$Glob->{'CFG'}->{'ACTIONS'}->{'TARGET'}' will be ignored.\n\n";
        $Glob->{'CFG'}->{'ACTIONS'}->{'TARGET'} = "";
    }

    if (!$Glob->{'CFG'}->{'PERFORCE'}->{'SYNC'} && $Glob->{'CFG'}->{'PERFORCE'}->{'RESYNC'})
    {
        print "WARNING! options -resync=1 AND -sync=0 are set! -resync will be ignored!\n\n";
        $Glob->{'CFG'}->{'PERFORCE'}->{'RESYNC'} = 0;
    }
}

##############################################################################
# Usage() - Shows the program usage and exits the script.
##############################################################################

sub Usage
{
    print <<USAGE_END;

Usage: LCEnv.pl [-64 <0|1>] [-(b|batch)] [-(c|clear)] [-cfg <CONFIG_FILE>]
                [-(ccl|check_cl) <1|0>] [-cl <CHANGELIST>] [-cmd <COMMAND>]
                [-color <COLOR_CODE>] [-(d|default)] [-(ed|edit_defaults)]
                [-low <1|0>] [-grp <PATH>] [-(h|help|?)] [-(l|list)]
                [-m <MAKE_TARGET>] [-own <OWN_DIR>] [-p4c <P4CLIENT>]
                [-p4pw <P4PASSWD>] [-p4r <P4ROOT>] [-p4u <P4USER>]
                [-prod <0|1>] [-rel <RELEASE>] [-resync <0|1>] [-(rm|remove)]
                [-save] [-scratch <0|1>] [-shell] [-sync <0|1>]
                [-tool <TOOL_DIR>] [-(v|verbose)] [-(wi|win_icon)]
                [-(win|win_icon_name) <ICON_NAME>] [-(ws|win_share)] 
                [-(wsn|win_share_name) <SHARE_NAME>] [-X64] [-(y|yes)]

Where:
        -64 <0|1>
            Set 64-bit-flag in environment, if set to 1. NOTE: This is not
            the same as the '-X64' option which enables 'CROSS'-Compilation!

        -b or -batch
            Start program in batch mode. This option prohibits any
            interactive action by users - if any option is missing,
            it causes the program to fail.

        -c or -clear
            Clears the generated environment again after make.

        -cfg <CONFIG_FILE>
            Uses a specific configuration preset.

        -ccl <0|1> or check_cl <0|1>
            With this option activated, LCEnv will check if the changelist
            has changed since last run. If not, LCEnv will not start any
            makes.

        -cl <CHANGELIST>
            Sets a specific change list you want to use.

        -cmd <COMMAND>
            A command to be executed after the make command.

        -color <COLOR_CODE>
            Automatically sets background and foreground colors of the
            shell window. For more information, please take a look into
            the "color manual" by typing 'color /?'.

        -d or -default
            Use default settings whenever possible. This option is not
            compatible with -d.

        -ed
            Edit default parameters. If this option is given, you will be
            asked for the global program defaults. The program will exit
            after that.

        -grp <PATH>
            Inserts a second path into vmake's source paths. This may
            be any path allowed by vmake (including 'remote'-paths).

        -h or -help or -?
            Displays this help file and ends program execution.

        -l or -list
            Lists all installed instances and exits.

        -low <1|0>
            Sets process to low priority under Windows OS, if set to 1.

        -m <MAKE_TARGET>
            Makes a specific target by calling 'imf <MAKE_TARGET>' after
            finishing the sync.

        -save
            Enforces a manual save of options.

        -own <OWN_DIR>
            Sets the new OWN directory to be used.

        -p4 <1|0>
            Use Perforce source control. If set to 1, it enables LCEnv to
            sync sources, determine changelists and automatically create
            and remove complete source tree's.

        -p4c <P4CLIENT>
            Sets the perforce client to be used.

        -p4pw <P4PASSWD>
            Sets the perforce passwd to be used (if needed).

        -p4r <P4ROOT>
            Sets the perforce root to be used.

        -p4u <P4USER>
            Sets the perfoce user to be used.

        -prod <0|1>
            Initiates a productive make by calling 'NACHT.ALL', if set to 1.

        -rel <RELEASE>
            Sets the release to be installed.
            This release must be given in one of the following formats:

            - 7403(DEV|COR|DEVELOP)
            - 740320
            - 7.4.03.(dev|cor|develop)
            - 7.4.03.20

            NOTE:
            1. If the release string is incomplete or missing, any
               missing parts will be asked for. You have then the
               opportunity to browse the perforce tree.
            2. There won't be a check for the given parts (except
               for the build prefix). If you give an incorrect
               version/correction level, it will lead to an error.

        -resync <0|1>
            Switch that tuns complete re-synchronisation on/off.

        -rm or -remove
            Just removes the environment without syncing/making etc.

        -scratch <0|1>
            Initiates a scratch make, if set to 1. If you've used the -m
            option to compile a single target, a -U will be appended to
            the imf command.

        -shell
            Causes LCEnv to open a shell environment at the end rather than
            exiting. (Should not be called manually)

        -sync <0|1>
            Automatically syncs all sources, if set to 1.

        -tool <TOOL_DIR>
            Use another TOOL installation rather than the default one.

        -v or -verbose
            Produces a more detailed output (mostly for debugging).

        -wi or -win_icon
            If this option is set, a shortcut icon will be created.
            NOTE: This option works only with Windows systems!

        -win <ICON_NAME> or -win_icon_name <ICON_NAME>
            With this option you can override the default windows icon name.

        -ws or -win_share
            If this option is set, a network share will be created.
            NOTE: This option works only with Windows systems!
            
        -wsn <SHARE_NAME> or -win_share_name <SHARE_NAME>
            With this setting you can override the default windows share
            name.

        -X64
            Sets environment for 64bit ITANIUM cross-compilation.

        -y or -yes
            Assume 'yes' on all 'yes/no' questions.

For more detailed information feel free to take a look into the documentation
embedded into the program. You can do this by calling the Perl inline
documentation tool 'perlpod'. If you have problems finding the perl script:
'perldoc \$TOOL/bin/LCEnv.pl' (Unix) or 'perldoc \%TOOL\%\\bin\\LCEnv.pl' (Win)
should do it.

\If you want to extract the documentation into HTML format, you can use the
perl tool 'pod2html'. Just call 'pod2html \$TOOL/bin/LCEnv.pl > LCEnv.html'
(UNIX) or 'pod2html \%TOOL\%\\bin\\LCEnv.pl > LCEnv.html' (Win).

\If the programs 'perldoc' and 'pod2html' are not already in your path, you can
find them under '<PERL_INSTROOT>/bin' (just as the Perl-exe itself).

USAGE_END

    exit(1);
}

__END__

=head1 NAME

LCEnv - a tool to handle liveCache / SAP DB environments.

 Current Version:  1.33 from 10/23/03

=head1 NOTE

This tool is a 'strictly internal' program since it uses internal Perforce
settings to gather release information. There is no confidential data like
passwords in the code - therefore it could be released but it wouldn't
work outside SAP.

=head1 ABSTRACT

This tool is used to install and manage local development environments. There
are many different usage aspects which include:

=over 4

=item *
setting / editing global (machine) environment defaults

=item *
creating new environments

=item *
editing development environment configurations at any time

=item *
removing complete environments again

=item *
syncronizing source content

=item *
automatically adapting Perforce structure changes

=item *
linking specific tasks directly to the environment (e.g. auto-make on open)

=item *
...and many more

=back

=head1 NEWS / CHANGES

 1.33:  New Features:
 
            - New parameter -prod_cfg introduced which enables NACHT.ALL to
              handle multiple configurations.

 1.32:  Bug fixes:
 
            - Check for P4PASSWD was too late, resp. env variable was not set
              in time.

 1.31:  New Features:
            
            - If option 'cfg' is given but file doesn't exist, LCEnv will
              start in 'initial' mode and use the cfg file name as template.
              
        Bug fixes:
            
            - If option p4c is given, it will also reset p4r to the new
              client's root.
 
 1.30:  New Features:
 
            - New options 'win' and 'wsn' to manually name windows icon
              and share names.
 
        Bug fixes:
 
            - When installing a 76dev release, LCEnv failed to set correction
              level and build prefix correctly.
              
            - ReleaseDB entries are now alwys updated again since they
              disappeared sometimes.

 1.29:  New Features:

            - Added option 'ccl'. With this option turned on, a make will
              only start, if the change list has changed since last run.

 1.28:  Bug fixes:

            - LCEnv asked for a "missing correction level" by mistake when
              using Perforce support in conjunction with an "always latest
              development branch" window (no real correction level). Affected
              only V75.(00).develop so far and has now been removed.

            - Fixed problem with "\\" path separators under UNIX systems.

 1.27:  Started this section.

        Bug fixes:

            - Configuration file name got lost while installation of new
              Studios. Therefore, all new shortcuts got empty configuration
              names.

        New Features:

            - Built-in color control. Use -color=<2-digit-color-code> to
              colour your windows. (Check command 'color /?' for color
              codes.)

            - Info string '(low prio)' is now shown in windows title when
              having set a window to low priority.

=head1 SYNOPSIS

Since there's many different ways of using this program, i'll give only a
short description of all parameters here. In the following sections
"Description" and "Examples" i'll go into more detail.

 Usage: LCEnv.pl [-64 <0|1>] [-(b|batch)] [-(c|clear)] [-cfg <CONFIG_FILE>]
                 [-(ccl|check_cl) <1|0>] [-cl <CHANGELIST>] [-cmd <COMMAND>]
                 [-color <COLOR_CODE>] [-(d|default)] [-(ed|edit_defaults)]
                 [-low <1|0>] [-grp <PATH>] [-(h|help|?)] [-(l|list)]
                 [-m <MAKE_TARGET>] [-own <OWN_DIR>] [-p4c <P4CLIENT>]
                 [-p4pw <P4PASSWD>] [-p4r <P4ROOT>] [-p4u <P4USER>]
                 [-prod <0|1>] [-rel <RELEASE>] [-resync <0|1>] [-(rm|remove)]
                 [-save] [-scratch <0|1>] [-shell] [-sync <0|1>]
                 [-tool <TOOL_DIR>] [-(v|verbose)] [-(wi|win_icon)]
                 [-(win|win_icon_name) <ICON_NAME>] [-(ws|win_share)] 
                 [-(wsn|win_share_name) <SHARE_NAME>] [-X64] [-(y|yes)]

 Where:
         -64 <0|1>
             Set 64-bit-flag in environment, if set to 1. NOTE: This is not
             the same as the '-X64' option which enables 'CROSS'-Compilation!

         -b or -batch
             Start program in batch mode. This option prohibits any
             interactive action by users - if any option is missing,
             it causes the program to fail.

         -c or -clear
             Clears the generated environment again after make.

         -cfg <CONFIG_FILE>
             Uses a specific configuration preset.

         -ccl <0|1> or check_cl <0|1>
             With this option activated, LCEnv will check if the changelist
             has changed since last run. If not, LCEnv will not start any
             makes.

         -cl <CHANGELIST>
             Sets a specific change list you want to use.

         -cmd <COMMAND>
             A command to be executed after the make command.

        -color <COLOR_CODE>
            Automatically sets background and foreground colors of the
            shell window. For more information, please take a look into
            the "color manual" by typing 'color /?'.

         -d or -default
             Use default settings whenever possible. This option is not
             compatible with -d.

         -ed or -edit_defaults
             Edit default parameters. If this option is given, you will be
             asked for the global program defaults. The program will exit
             after that.

         -grp <PATH>
             Inserts a second path into vmake's source paths. This may
             be any path allowed by vmake (including 'remote'-paths).

         -h or -help or -?
             Displays this help file and ends program execution.

         -l or -list
             Lists all installed instances and exits.

         -low <1|0>
             Sets process to low priority under Windows OS, if set to 1.

         -m <MAKE_TARGET>
             Makes a specific target by calling 'imf <MAKE_TARGET>' after
             finishing the sync.

         -own <OWN_DIR>
             Sets the new OWN directory to be used.

         -p4 <1|0>
             Use Perforce source control. If set to 1, it enables LCEnv to
             sync sources, determine changelists and automatically create
             and remove complete source tree's.

         -p4c <P4CLIENT>
             Sets the perforce client to be used.

         -p4pw <P4PASSWD>
             Sets the perforce passwd to be used (if needed).

         -p4r <P4ROOT>
             Sets the perforce root to be used.

         -p4u <P4USER>
             Sets the perfoce user to be used.

         -prod <0|1>
             Initiates a productive make by calling 'NACHT.ALL', if set to 1.

         -rel <RELEASE>
             Sets the release to be installed.
             This release must be given in one of the following formats:

             - 7403(DEV|COR|DEVELOP)
             - 740320
             - 7.4.03.(dev|cor|develop)
             - 7.4.03.20

             NOTE:
             1. If the release string is incomplete or missing, any
                missing parts will be asked for. You have then the
                opportunity to browse the perforce tree.
             2. There won't be a check for the given parts (except
                for the build prefix). If you give an incorrect
                version/correction level, it will lead to an error.

         -resync <0|1>
             Switch that tuns complete re-synchronisation on/off.

         -rm or -remove
             Just removes the environment without syncing/making etc.

         -save
             Enforces a manual save of options.

         -scratch <0|1>
             Initiates a scratch make, if set to 1. If you've used the -m
             option to compile a single target, a -U will be appended to
             the imf command.

         -shell
             Causes LCEnv to open a shell environment at the end rather than
             exiting. (Should not be called manually)

         -sync <0|1>
             Automatically syncs all sources, if set to 1.

         -tool <TOOL_DIR>
             Use another TOOL installation rather than the default one.

         -v or -verbose
             Produces a more detailed output (mostly for debugging).

         -wi or -win_icon
             If this option is set, a shortcut icon will be created.
             NOTE: This option works only with Windows systems!

         -win <ICON_NAME> or -win_icon_name <ICON_NAME>
             With this option you can override the default windows icon name.
 
         -ws or -win_share
             If this option is set, a network share will be created.
             NOTE: This option works only with Windows systems!
             
         -wsn <SHARE_NAME> or -win_share_name <SHARE_NAME>
             With this setting you can override the default windows share
             name.

         -X64
             Sets environment for 64bit ITANIUM cross-compilation.

         -y or -yes
             Assume 'yes' on all 'yes/no' questions.

=head1 DESCRIPTION

LCEnv was originally developed to be a redesigned and cleaned-up version
of DevStudio. However, there were so many new functions and external
requirements that the final outcome became a completely new tool which
has not much to do with the original any more.
It was designed to be able to do anything the old tool could do and much
more - the focus there was to implement functions which are needed by the
developers and are implemented e.g. as shell scripts by every developer himself
so far. Additionally, the intention was to tie DevStudio to Perforce to be
able to install a Studio for everything you can see under Perforce //sapdb
root. When we finally decided to implement an iprofile generation mechnism,
the way was free to create a completely new version of DevStudio which is
now 100% dynamic and does not need any patch when introducing new Releases.

So far, so good...Now let's take a deeper look into its usage.

Usage of this tool can divided into 4 groups: Initial setup, managing releases,
using environments and performing automatic tasks.

First comes the initial setup. This step is a bit annoying but quite important
for LCEnv and sould be done at first use - if you don't, LCEnv will not stop
asking you lot's of questions which normally only have to be answered once.
A detailed walkthrough is described in the next section 'Initial Configuration'.

When initial setup is done, you can proceed with generating new studio's.
To learn more about how to do this, please take a look into the examples and
read the section 'Creating New Environments'.

=head2 Initial Configuration

When starting LCEnv for the very first time, it'll inform you that global
system defaults were not found:

    C:\>lcenv

    LCEnv v1.23 (2003-08-20), (c)2003 SAP AG

    Could not find any system defaults. Do you want to set them up now ? (Y/n) y

Alternatively, you can force LCEnv to setup system defaults by calling it with
the '-ed' option.

Now you'll be faced with the following questions:

=over 4

=item *
Configuration file location [default = 'c']

    Where do you want your Release configurations to be stored:

    [o] = Each in its release's own directory.
    [c] = Centrally, under C:\SAPDevelop\Devtool\data\LCEnv. (default)
    [s] = Centrally, i'd like to choose a path manually.

    NOTE: If you store configurations centrally, you don't have to give a complete
          path when selecting a configuration.

    Please select :

Here you can decide where the configuration file for one specific SAP DB
release should be stored. This configuration is needed every time you want to
open a new Devstudio window or change some settings ('-cfg <config file>').
Since this option is needed quite often, it may be useful to store these files
centrally. This has the advantage, that LCEnv knows where to look for these
files - in other words: If you have to type the command manually for any
reasons, you don't have to type the complete path to the configuration file.
Additionally, it may make managing configurations easier when everything is
stored centrally.

But, don't be too serious about this question - LCEnv will work fine
regardless what option you choose here. That's what i did:

    Please select : s
    Configuration dir : c:\sapdevelop\profiles

=item *
Always check Perforce settings [default = 'n']

    Do you want all your settings checked / compared with Perforce every time you
    open a new development window (makes LCEnv robust against changes but slow
    when starting) ? (y/N)

If this question is answered with 'y', LCEnv will check all your Perforce
settings every time you open a development window. These settings include:

=over 4

=item *
P4 User

=item *
P4 Client

=item *
Release's position within the Perforce tree

=item *
P4 Server path for this release

=item *
Actual P4 change list for this release

=back

Please note that this option makes LCEnv more robust against environment
changes and misconfiguration from outside: If a P4 client is deleted, it
will be re-created; if - for example - somebody decides to introduce a new
development tree 'V7405dev', LCEnv would notice that and would move the
actual 'V7404dev' tree from '//sapdb/V74/develop' to '//sapdb/V74/c_04/develop'.
On the other hand, this option makes starting devstudio's quite slow -
opening a window would take between 5 and 10 seconds at least. For this
reason, this option is deactivated by default.

=item *
Sync on window open [default = 'n']

    Do you want Perforce to sync your files when opening a new window ? (y/N)

With this option, you can tell LCEnv to synchronize all sources of a release
when opening a new development window. Please note, that this option has no
effect when you answered 'n' at the question above.

Since synchronizing can take quite some time and is therefore mostly done at
night time, this option is deactivated by default.

=item *
Always ask for Perforce settings [default = 'n']

    Do you want to be asked for your perforce settings every time when creating
    a new instance (to manage multiple configurations) ? (y/N)

This option allowes you to handle multiple Perforce settings for each release.
Some users like to have for example different clients for each release. These
users should answer this question with 'y', all others should say 'n' here to
avoid answering stupid questions every time when creating a new studio.

=item *
Setting up Perforce defaults [default = 'y']

    Do you want to set up your default perforce settings now ? (Y/n)

Here you can set your Perforce default settings which are used to generate new
Studios unless you explicitly tell LCEnv to use different settings.

NOTE: It is quite important that you setup your defaults here since LCEnv
needs these information to generate new Studio's. Not setting them up is like
answering the question above (Always aks for Perforce settings) with 'y'.

=over 4

=item *
Perforce user [default = 'l']

    Please select which Perforce user you want to use:

    [l] = use your login name (d040173), default
    [s] = select a different user name manually

    Please select:

Here you can enter your Perforce user. Normally, 'l' should do it *BUT BE
VERY CAREFUL HERE* since Perforce names are CASE SENSITIVE! If the given
login name doesn't fit, *DON'T USE IT* - take 's' instead and enter your
user manually.

HINT: You can check your actual Perforce user and client by looking into
the window title of the Perforce window. Please check here, whether your
user and client are written lower- or upper-case.

=item *
Perforce client [default = 'h']

    Please select which Perforce client you want to use:

    [h] = use the local host name (P73709), default
    [s] = select a different client name manually

    Please select:

Now, you have to enter the Perforce client you want to use. Again, this
name ist CASE-SENSITIVE. Read section above to check for the correct case -
if the host name doesn't fit, enter the client name manually by selecting 's'.

=item *
Perforce client root [default = 'p']

    Please select which client root you want to use for your P4 client:

    [c] = use the current directory (C:\)
    [p] = use current P4 client root (c:\SAPDevelop)
    [s] = select a different client name manually

    Please select:

Here, you can select the Perforce client root you want to use. Please note,
that all releases have to be installed into a sub directory of this root in
order to use Perforce correctly. In fact, LCEnv will deny installing any
releases outside this root.

HINT: If the second line ('[p] = use current P4 client root (c:\SAPDevelop)')
appears, the Perforce user and client you entered above were correct. If not,
LCEnv couldn't either connect to Perforce with the settings you entered, or
you just created a new client.

NOTE: Normally, you should always use 'p' here since any other option will
permanently cheange your Perforce client.

=back

=item *
Create shortcuts [default = 'y'], Windows only

    Do you want a new shortcut to be created by default ? (Y/n)

Here you can decide wether you want a shortcut to be created by default or
not. Most users answer 'y' here.

=over 4

=item *
Shortcut storage place [default = 'm'], Windows only

    Where do you want to store your shortcuts:

    [m] = In Windows start menu. (default)
    [d] = On desktop.
    [e] = At a place stored in %DEVSTUDIO_SHORTCUTS%
    [s] = Select a different place manually.

    Please select :

Now, you can tell LCEnv where to create the shortcuts. If using 'm', LCEnv
will create a folder in the start menu called "SAPDevelop D<d-user>" where
all shortcuts are stored. Alternatively, you can store your shortcuts on
desktop ('d') or at any other place ('e', 's').

=back

=item *
Create network shares [default = 'y'], Windows only

    Do you want a new windows network share to be created by default ? (Y/n)

If you want to have a network share to be created with every new studio,
answer 'y' here.

NOTE: To meet earlier specifications, LCEnv creates share names in the format
'V<RELVER>' e.g. 'V75' for the latest development branch of a release. For any
other release it takes the format 'V<RELVER><CORR><LC_STATE>' e.g. 'V7404COR'

=back

When you have answered all these questions, the initial setup is complete.
The program will exit then and has to be restarted.

The generated configuration file can be found under:
'%TOOL%\data\LCEnv.default'.

=head2 Creating New Environments

To create a new environment you just have to call LCEnv without parameters.
If you do so, you'll be asked for a release, correction level, build prefix
and OWN directory. After that, all directories are created and at least a
minimal sync was performed to be able to create a iprofile.bat.

Alternatively, you can call LCEnv with the parameters '-rel <RELEASE>' and
'-own <OWN-Directory>' - the order is not important. You can also add any
other parameter given above to override default parameters or to perform
default tasks.
But be careful here: Any parameter given at this time will be stored
permanently. Example: If you give a make target here, it will be executed
every time, you open a window!

After creating a new instance, you'll find a release configuration file
which is either stored centrally with the release as name, e.g. 'V7600DEV.cfg'
or within the OWN-Directory, stored as 'LCEnv.cfg'. To have a deeper look
into these files, please take a look into the section 'Release Configuration'.

Instead of starting the studio by clicking the Icon, you can also call it by
using LCEnv from command line (any development window): 'LCEnv -cfg <Cfg-File>'.
If you want to change any settings later, you adding them as parameter in
the command line. Note: If you want to have them saved permanently, you
also have to enter '-save'. Take a look into the 'EXAMPLES' section for more
information.

=head2 Configuration Files

Since configuration files should normally not be edited by hand, i only show
some example files here instead of completely explaining all parameters.

=over 4

=item *
Default configuration [%TOOL%\data\LCEnv.default]

    # File generated by SimpleCfg v1.00
    # Generation time: Wed Aug 20 16:53:07 2003

    [GLOBAL]
    ICON_PATH=C:\Documents and Settings\All Users\Start Menu\SAPDevelop D010101
    CFG_PATH=c:\sapdevelop\profiles

    [PERFORCE]
    ROOT=c:\SAPDevelop
    USE=0
    USER=d010101
    ALWAYS_ASK=0
    CLIENT=P10101
    PORT=perforce3006.wdf.sap.corp:3006
    SYNC=0

    [ACTIONS]
    CREATE_SHARE=1
    CREATE_ICON=1

=item *
Release Configuration [Example here: V7600DEV.cfg]

    # File generated by SimpleCfg v1.00
    # Generation time: Tue Aug 19 18:15:00 2003

    [RELEASE]
    GRP=
    OWN=c:\SAPDevelop\V76\develop
    VERSION=76.00.DEV
    _CHANGELIST=50253
    CHANGELIST=0

    [GLOBAL]
    LOW=0
    BATCH_MODE=0
    VERBOSE=0
    64=0
    PROD_MAKE=0
    ASSUME_YES=0
    CFG_FILE=c:\sapdevelop\Profiles\7600DEV.cfg
    ICON_PATH=C:\Documents and Settings\All Users\Start Menu\SAPDevelop D010101
    USE_DEFAULT=0
    INIT=
    TOOL_DIR=
    X64=0
    SCRATCH=0

    [PERFORCE]
    _VIEW_REMOVED=
    ROOT=c:\SAPDevelop
    USE=0
    USER=d010101
    _USER_CREATED=0
    CLIENT_PATH=//P10101/V76/develop
    _ROOT_ADDED=
    RESYNC=0
    PASSWD=
    _ROOT_REMOVED=
    _CLIENT_CREATED=0
    SERVER_PATH=//sapdb/V76/develop
    _VIEW_ADDED=
    CLIENT=P10101
    PORT=perforce3006.wdf.sap.corp:3006
    SYNC=0

    [ACTIONS]
    CREATE_SHARE=1
    CREATE_ICON=1
    _ICON_CREATED=C:\Documents and Settings\All Users\Start Menu\SAPDevelop D010101\Developer Studio 76.00.DEV.lnk
    MAKE=0
    CMD=
    CLEAR=0
    _SHARE_CREATED=V76
    TARGET=

As you can see, some settings overlap with the default settings. This is,
because they were copied from the defaults at creation time. Some of them
are even not really used (such as 'GLOBAL/ICON_PATH') but they were
essential when creating the environment...

=back

=head1 EXAMPLES

Here you can find some usage examples for LCEnv. Please note that this section
implicates that you have set up the default settings before. If you don't,
please do this first.

=head2 Interactive Mode

Scenario: I want to create a new Studio for Release 7.6 using default settings.

Screenshot:

    C:\>LCEnv

    LCEnv v1.23 (2003-08-20), (c)2003 SAP AG


    The LC release is not set. Please select one:

    [1] = 72
    [2] = 73
    [3] = 74
    [4] = 75
    [5] = 76

    Please select: 5

    Checking perforce path for LC V76.00.develop....OK

    Checking release OWN....NOT SET!

    Please select which $Own directory you want to use:

    [c] = use the current directory (C:\)
    [p] = use P4 client root (c:\SAPDevelop)
    [r] = use releases default directory (c:\SAPDevelop\V76\develop), default
    [s] = select a different client name manually

    Please select: r

    Checking Perforce client path....OK

    Retrieving actual changelist....OK (50356)

    Updating Perforce client 'P73709'....OK

    Doing minimal sync to be able to create profile...

    Creating iprofile....OK

    Creating Windows shortcut....OK

    Creating Windows share....OK

    Updating Release DB entries....OK

    Release settings saved.

As you can see above, LCEnv didn't ask for a correction level and a build
prefix - it realised that there exists only one release '76' within the
Perforce tree.

NOTE: Even though i told LCEnv NOT to use Perforce by default, it still
uses Perforce to create new studios. That's why you really should setup
your defaults, first.

Scenario: Same as above, but i want to have a low priority window now.

Screenshot:

    C:\>LCEnv -low=1

    LCEnv v1.23 (2003-08-20), (c)2003 SAP AG


    The LC release is not set. Please select one:

    [1] = 72
    [2] = 73
    [3] = 74
    [4] = 75
    [5] = 76

    Please select: 5

    Checking perforce path for LC V76.00.develop....OK

    Checking release OWN....NOT SET!

    Please select which $Own directory you want to use:

    [c] = use the current directory (C:\)
    [p] = use P4 client root (c:\SAPDevelop)
    [r] = use releases default directory (c:\SAPDevelop\V76\develop), default
    [s] = select a different client name manually

    Please select: r

    Checking Perforce client path....OK

    Retrieving actual changelist....OK (50356)

    Updating Perforce client 'P73709'....OK

    Doing minimal sync to be able to create profile...

    Creating iprofile....OK

    Creating Windows shortcut....OK

    Creating Windows share....OK

    Updating Release DB entries....OK

    Release settings saved.

=head2 Default Use / Creating a New Instance

Scenario: I want to create new window for release 7.6

Screenshot:

    C:\>LCEnv -rel 7600dev -own c:\sapdevelop\V76\develop

    LCEnv v1.23 (2003-08-20), (c)2003 SAP AG

    Checking perforce path for LC V76.00.develop....OK

    Checking release OWN....OK

    Checking Perforce client path....OK

    Retrieving actual changelist....OK (50356)

    Updating Perforce client 'P73709'....OK

    Doing minimal sync to be able to create profile...

    Creating iprofile....OK

    Creating Windows shortcut....OK

    Creating Windows share....OK

    Updating Release DB entries....OK

    Release settings saved.

And again: Since there's only one release 7.6, it'S enough to say:

    C:\>LCEnv -rel 76 -own c:\sapdevelop\V76\develop

    LCEnv v1.23 (2003-08-20), (c)2003 SAP AG

    Checking perforce path for LC V76.00.develop....OK

    Checking release OWN....OK

    Checking Perforce client path....OK

    Retrieving actual changelist....OK (50356)

    Updating Perforce client 'P73709'....OK

    Doing minimal sync to be able to create profile...

    Creating iprofile....OK

    Creating Windows shortcut....OK

    Creating Windows share....OK

    Updating Release DB entries....OK

    Release settings saved.

=head2 Stating a Studio From Command Line Prompt

Scenario: I want to start my release 7.6 now (using central configurations).

Screenshot:

    C:\>LCEnv -cfg 7600dev -shell

    LCEnv v1.23 (2003-08-20), (c)2003 SAP AG

    Checking release OWN....OK

    Creating iprofile....OK

    Updating Release DB entries....OK

    Welcome to 7.6.00.00 SAP DB Developer Studio (DEV) !

    ---- ! Microsoft SDK is used   ! ----
    Targeting Windows 2000 and IE 5.0 RETAIL


    %OWN%: c:\sapdevelop\V76\develop
    %GRP%: \\P79775\V76\develop
    %ALL%:
    %REL%:

    %VMAKE_PATH%: c:\sapdevelop\V76\develop


    C:\SAPDevelop\V76\develop>

Scenario: I want to start my release 7.6 now (using local configurations).

Screenshot:

    C:\>LCEnv -cfg c:\sapdevelop\V76\develop\LCEnv.cfg -shell

    LCEnv v1.23 (2003-08-20), (c)2003 SAP AG

    Checking release OWN....OK

    Creating iprofile....OK

    Updating Release DB entries....OK

    Welcome to 7.6.00.00 SAP DB Developer Studio (DEV) !

    ---- ! Microsoft SDK is used   ! ----
    Targeting Windows 2000 and IE 5.0 RETAIL


    %OWN%: c:\sapdevelop\V76\develop
    %GRP%: \\P79775\V76\develop
    %ALL%:
    %REL%:

    %VMAKE_PATH%: c:\sapdevelop\V76\develop


    C:\SAPDevelop\V76\develop>

=head2 Temporarily Using Different Parameters

Scenario: I want to sync my freshly created release 7.6.

Screenshot:

    C:\>LCEnv -cfg 7600dev -p4=1 -sync=1

    LCEnv v1.23 (2003-08-20), (c)2003 SAP AG

    Checking perforce path for LC V76.00.develop....OK

    Checking release OWN....OK

    Checking Perforce client path....OK

    Retrieving actual changelist....OK (50356)

    Sync files (incremental)....OK

    Creating iprofile....OK

    Updating Release DB entries....OK

Note: To be able to use the 'sync' option, you have to temporarily activate
Perforce support by also giving '-p4=1'. Otherwise '-sync' wouldn't have
any effect.

=head2 Permanently Changing Parameters

Scenario: I liked that autosync thingy, let's make it permanent!

Screenshot:

    C:\>LCEnv -cfg 7600dev -p4=1 -sync=1 -save

    LCEnv v1.23 (2003-08-20), (c)2003 SAP AG

    Checking perforce path for LC V76.00.develop....OK

    Checking release OWN....OK

    Checking Perforce client path....OK

    Retrieving actual changelist....OK (50356)

    Sync files (incremental)....OK

    Creating iprofile....OK

    Updating Release DB entries....OK

    Release settings saved.

Note here that the program said that it saved the setting at the end. They're
now permanent.

=head2 Removing an Instance

Scenario: I want to remove my old release 7.4.02 DEV

Screenshot:

    C:\>LCEnv -cfg 7402dev -rm

    LCEnv v1.23 (2003-08-20), (c)2003 SAP AG

    Removing environment for '74.02.DEV'. Please be patient...

    Removing Release DB entry....OK

    Perforce client restored.

    Directory tree removed.

=head2 Performing a Nightly Sync and Make

To perform a nightly sync and make, you could create a new scheduled task
which calls this command:

    perl c:\sapdevelop\devtool\bin\LCEnv.pl -cfg 7600dev -p4=1 -sync=1 -m all

This commandy temporarily activates Perforce support, synchronizes all files
and then call imf with target 'all'.

=head1 COPYRIGHT

Copyright 2003 SAP AG