/*!***************************************************************************

  module	  : LVCSim_liveCacheSink.hpp

  -------------------------------------------------------------------------

  responsible : IvanS

  special area: liveCache Simulator
  description : liveCache Sink implementation

  -------------------------------------------------------------------------





    ========== licence begin  GPL
    Copyright (c) 2002-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end




*****************************************************************************/


#ifndef LVCSIM_LIVECACHESINK_HPP
#define LVCSIM_LIVECACHESINK_HPP

/*===========================================================================*
 *	INCLUDES																 *
 *===========================================================================*/

#include "LVCSimulator/LVCSim_Internal.hpp"
#include "LVCSimulator/LVCSim_FastHash.hpp"
#include "SAPDBCommon/MemoryManagement/SAPDBMem_RawAllocator.hpp"
#include "LVCSimulator/LVCSim_CheckpointIO.hpp"

#include "liveCache/LVC_IliveCacheSink.hpp"
#include "geo07.h"


/// Maximum count of sink pointers (and thus auto-assigned OMS handles).
#define	LVCSIM_MAX_SINK_PTR 	1024


/*===========================================================================*
 *	CLASSES, STRUCTURES, TYPES, UNIONS ...									 *
 *===========================================================================*/


class LVCSim_TXList;
class LVCSim_CheckedAllocator;
class LVCSim_CheckedAllocatorInst;
class LVCSim_GlobalAllocator;
class LVCSim_ContainerID;

class LVC_Dispatcher;

/*!
 * \brief LiveCache session-specific kernel sink.
 *
 * Definition of those functions that can be called from the liboms to obtain
 * kernel support. In this case, kernel means simulator.
 *
 * An instance of this class practically represents a transaction.
 */
class LVCSim_liveCacheSink : public IliveCacheSink
{
public:
	class	SchemaName;
	class	ConsViewList;
    class   UndoList;
    class   ActiveVersion;
	class	SubTxList;

private:
	/// Count of defined schemas.
	static int									schemaCount;
	/// Schema directory.
	static LVCSim_FastHash<SchemaName,
		int, LVCSim_SchemaMeta, true>			*schemaDir;
	/// Spinlock protecting schema directory.
	static RTESync_Spinlock						*schemaLock;

	/// Count of registered containers.
	static int									containerCount;
	/// Container directory.
	static LVCSim_FastHash<LVCSim_ContainerID,
		int, LVCSim_ContainerMeta, false>		*containerDir;
	/// Spinlock protecting container directory.
	static RTESync_Spinlock						*containerLock;

	/// Spinlocks for emulating regions.
	static RTESync_Spinlock						*regionLocks;

	/// Spinlock protecting temporary sink pointers.
	static RTESync_Spinlock						sinkPtrLock;
	/// Temporary sink pointers used in LVCSim::Attach().
	static LVCSim_liveCacheSink*				sinkPtr[LVCSIM_MAX_SINK_PTR];
	/// Temporary sink cancel flags used in LVCSim::Attach().
	static bool									sinkPtrC[LVCSIM_MAX_SINK_PTR];
	/// Free pointer for temporary sinks used in LVCSim::Attach().
	static int									sinkPtrFree;

	/// Simulator state dump sequence counter.
	static int									dumpSequence;

	/// Dummy default context.
	void										*m_defaultContext;
	/// Dummy cancel byte.
	pasbool										m_cancelByte;
	/// Sink reference count.
	int 										m_cRef;

	/// Current read view.
	LVCSim_TID									m_readView;
	/// Current undo list (for active subtransaction).
	UndoList                                    *m_undoList;
	/// Current subtransaction list.
	SubTxList									*m_subTxList;
	/// Current lock chain.
    class LVCSim_ObjectHandle                   *m_lockChain;

	/// Return current undo list, creating it, if necessary.
    UndoList *undoList();
	/// Unlock all objects in lock chain (after COMMIT/ROLLBACK).
    void unlockAll();
    
	/// Create underlying raw allocator for session-specific allocators.
    void createRawAllocator();

    tsp00_TaskId         m_taskId;
    static tsp00_TaskId  m_taskIdCnt;

protected:
	/// Lock for transaction-specific variables.
	static RTESync_Spinlock						txCounterLock; 

	/// Current TX counter.
	static LVCSim_TID							txCounter;
	/// List of existing consistent views.
	static ConsViewList 						*consViewList;
	/// Oldest running TX.
	static LVCSim_TID							txOldest;
	/// First session in linked list of all sessions.
    static LVCSim_liveCacheSink                 *firstSession; 

	/// Next session in sessions chain.
    LVCSim_liveCacheSink                        *m_next;

	friend class LVCSim_Initer;
	friend class LVCSim_TXList;
	friend class LVCSim;
	friend class LVCSim_ObjectHandle;
    friend class ConsViewList;
    friend class LVCSim_ContainerMeta;

	/// Initialize the static variables.
	void init();

	/// Reset the sink (e.g., on simulator reset).
	void reset();

	/// Shut down the sink.
	void done();

	/// Internal commit processing.
	void SimCommit();
	/// Internal rollback processing.
	void SimRollback();

	/// Get new temporary sink pointer for pure OmsHandle.
	static LVCSim_liveCacheSink **getSinkPtr(bool **cancelFlag);
	/// Return temporary sink pointer for pure OmsHandle.
	static void returnSinkPtr(LVCSim_liveCacheSink **ptr);

	/// Write contents of the simulator persistent state (checkpoint).
	static void writeToStream(LVCSim_CheckpointWriter &o);
	/// Read contents of the simulator persistent state.
	static void readFromStream(LVCSim_CheckpointReader &i);

	/// Dump internal simulator state for given handle. TODO: params.
	void omsSimDump(OmsHandle &handle, int param);
	/// Dump single container into dump file. TODO: params.
	static long omsSimDumpContainer(OmsHandle &handle, FILE *out, 
		LVCSim_liveCacheSink *tid, const char *datetime, 
		OMS_LibOmsInterfaceClassInfo &cinfo);
	/// Count and calculate the size of all non-var and var objects in all containers of a specified schema.
	static void countObjects(OmsHandle &handle, const OmsSchemaHandle schema, int &objCount, size_t &objSize, int &varObjCount, size_t &varObjSize);
	/// Set file to use as VTRACE.
	static void SetVTraceFile(FILE *f);

  /// Set dump count.
  static void SetDumpCount(int param);

	/// Create new sink.
	LVCSim_liveCacheSink(bool appendToList = true);
	/// Destroy the sink.
	~LVCSim_liveCacheSink();

	/// Commit (write) transaction timestamp.
	LVCSim_TID      m_commitTime;
	/// Reference into list of consistent views.
    ConsViewList    *m_commitTimeRef;

	/// Get or create read consistent view.
    void getReadConsistentView(void *lpView, short *lpb_error);
	/// Get or create write consistent view.
    void getWriteConsistentView(void *lpView, short *lpb_error);
	/// Return true, if transaction sees changes made by otherTX.
    bool sees(LVCSim_TID otherTX);
	/// Return true, if current transaction holds a read view.
    inline bool hasReadView() const { return m_otherTXs != NULL; }
	/// Return commit transaction timestamp.
    LVCSim_TID getCommitTime();
	/// Check if we see timestamp tx as committed.
    static bool checkCommited(LVCSim_TID tx);

	/// Timestamps of parallel transactions that ran as we started.
    LVCSim_TXList           *m_otherTXs;
	/// Currently active persistent version or NULL.
    ActiveVersion           *m_activeVersion;   

public:

	STDMETHODIMP QueryInterface(const IID& iid, LPVOID* ppv) 
	{	 
		if (iid == IID_IUnknown)
			*ppv = STATIC_CAST(IUnknown*, this); 
		else {
			*ppv = NULL ;
			return E_NOINTERFACE ;
		}
		(REINTERPRET_CAST(IUnknown*,*ppv))->AddRef() ;
		return S_OK ;
	}

	STDMETHODIMP_(ULONG) AddRef() 
	{
		return ++m_cRef;
	}

	STDMETHODIMP_(ULONG) GetRefCount() 
	{
		return m_cRef;
	}
  
	STDMETHODIMP_(ULONG) Release()
	{
		if (--m_cRef == 0) {
			delete this;
			return 0;
		}
		return m_cRef;
	}

	/// Get container metadata for given container ID.
	static class LVCSim_ContainerMeta *getContainer(unsigned int id);

	/// Get oldest running transaction timestamp.
	static inline LVCSim_TID getOldestTID() { return txOldest; }


	/*----------------------------------------------------------------**
  ** Documentation for the sink-methods is given in IliveCacheSink  **
	**----------------------------------------------------------------*/

	virtual HRESULT STDMETHODCALLTYPE ABAPRead( 
		OmsTypeInt4   &tabHandle,
		OmsTypeInt4   &count,
		OmsTypeInt4   &bufSize,
		OmsTypeByte  *&buf,
		short         &error);
	
	virtual HRESULT STDMETHODCALLTYPE ABAPWrite( 
		OmsTypeInt4    *TabHandle,
		tsp00_Int4      row_size,
		tsp00_Int4      row_count,
		unsigned char  *lpv,
		short          *lpb_error);
		
	virtual HRESULT STDMETHODCALLTYPE AllocPacket( 
		unsigned char  **p,
		tsp00_Int4	    *sz);

	virtual HRESULT STDMETHODCALLTYPE CloseVersion( 
		short        *lpb_error);
	  
	virtual HRESULT STDMETHODCALLTYPE ConsoleMsg( 
		tsp00_Int4      buflen,
		unsigned char  *lpv);

	virtual HRESULT STDMETHODCALLTYPE CreateVersion( 
 		unsigned char  *lpConsistentView,
    unsigned char  *lpVersionContext,
		short          *lpb_error);

   virtual HRESULT STDMETHODCALLTYPE CreateKeyRangeIterator(
    tgg91_TransNo            &consistentView,
    tgg00_FileId             &objFile,
    tgg01_OmsVersionContext  *lpVersionContext,
    void                    **pKeyRangeIterator,
    tsp00_Int4                bodySize,
    tsp00_Int4                keyLen,
    const void               *pStartKey,              
    const void               *pLowerKey,             
    const void               *pUpperKey,              
    tsp00_Int4               &noOfOid,
    OmsTypeOid               *pResultOId,  
    tgg91_PageRef            *pObjSeq,
    unsigned char            *pBody,
    tsp00_Int4               &LogReadAccesses,
    tgg00_BasisError         &e,
    OmsIterDirection          direction = OMS_ASCENDING);

  virtual HRESULT STDMETHODCALLTYPE StartUnloadedVersionIter(
    tgg91_TransNo           &consistentView,
    tgg01_OmsVersionContext *lpVersionContext,
    tgg00_FileId            &objFile,
    tsp00_Int4              keyLen,
    void                    *pStartKey,            
    const void              *pLowerKey,
    const void              *pUpperKey,
    tsp00_Int4              ObjBodySize,           
    void                    *pObjBody,             
    OmsTypeOid              &ResultOId,            
    short                   &e,
    OmsIterDirection        direction);             
	
  virtual HRESULT STDMETHODCALLTYPE DeleteAllObj(
    tgg91_TransNo&  consistentView,
    unsigned char  *lpContainerId,
    unsigned char  *lpVersionFile,
    int            &lpDeleted,
    short          &lpb_error,
    OmsTypeOid     &errorOid);

	virtual HRESULT STDMETHODCALLTYPE DeleteObj( 
		unsigned char  *lpConsistentView,
		unsigned char  *lpContainerId,
		unsigned char  *lpVersionContext,
		unsigned char  *lpoid,
		unsigned char  *lpobjseq,
		short          *lpb_error);

  virtual HRESULT STDMETHODCALLTYPE DeleteObjMass(
    tsp00_Int4               cnt,
    tgg91_TransNo           *pConsistentView,
    tgg01_ContainerId      **ppContainerId,
    unsigned char           *pVersionContext,
    OmsTypeOid              *pOid,
    tgg91_PageRef           *pObjSeq,
    tsp00_Int4              &errCnt,
    tgg00_BasisError        *pDBError);

	virtual HRESULT STDMETHODCALLTYPE DestroyKeyRangeIterator(
		void* pKeyRangeIterator);
	
	virtual HRESULT STDMETHODCALLTYPE EndConsistentView( 
		unsigned char  *lpConsistentView,
		short          *lpb_error);

	virtual HRESULT STDMETHODCALLTYPE EnterCriticalSection(
    short regionId);
 
  virtual HRESULT STDMETHODCALLTYPE GetContainerId(		
    tgg01_ContainerId              *pcontainerId, 
    const tgg01_OmsVersionContext  *pVersionContext,
    tsp00_Int4                      noOfOid,
    const OmsTypeOid               *pOid,
    tgg00_BasisError               *pDBError);

  virtual HRESULT STDMETHODCALLTYPE GetObj( 
    tgg91_TransNo                 &consistentView,
    tgg01_ContainerId             &containerId, 
    const tgg01_OmsVersionContext *pVersionContext,
    const OmsTypeOid              &oid,
    bool                           doLock,
    tgg91_PageRef                 &objSeq,
    tsp00_Int4                     bodySize,
    unsigned char                 *pBody,
    tsp00_Int4                    &bodyLen,
    tsp00_Int4                    &histLogCount,
    tgg00_BasisError              &e); 

  virtual HRESULT STDMETHODCALLTYPE GetObjects(
    tgg91_TransNo                 &consistentView,
    tgg01_ContainerId             *pContainerId, 
    const tgg01_OmsVersionContext *pVersionContext,
    tsp00_Int4                     noOfOid,
    const OmsTypeOid              *pOid,
    tgg91_PageRef                 *pObjVers,
    tsp00_Int4                    *pBodySize,
    bool                           doLock,
    unsigned char                **ppObj,
    tsp00_Int4                    &histLogCount,
    tsp00_Int4                    &errorCount,
    tgg00_BasisError              *e);

	virtual HRESULT STDMETHODCALLTYPE GetObjWithKey( 
		unsigned char  *lpConsistentView,
		unsigned char  *lptree,
		unsigned char  *lpVersionContext,
		tsp00_Int4      keylen,
		unsigned char  *lpkey,
		boolean         doLock,
		tsp00_Int4      bodysize,
		unsigned char  *lpbody,
		OmsTypeOid	   *lpoid,
		unsigned char  *lpobjseq,
		int            *hist_log_count,
		short          *lpb_error);

  virtual HRESULT STDMETHODCALLTYPE GetObjectsWithKey( 
    tgg91_TransNo            *pConsistentView,
    tgg01_ContainerId        *pContainerId,
    tgg01_OmsVersionContext  *pVersionContext,
    tsp00_Int4               *pKeyLen,
    tsp00_Int4                noOfObj,
    unsigned char           **ppKey,
    boolean                   doLock,
    tsp00_Int4               *pBodySize,
    unsigned char           **ppBody,
    OmsTypeOid               *pOid,
    tgg91_PageRef            *pObjSeq,
    tsp00_Int4               &histLogCnt,
    tsp00_Int4               &errorCnt,
    tgg00_BasisError         *pDBError);

	virtual HRESULT STDMETHODCALLTYPE HResultError( 
		HRESULT h_error);

	virtual HRESULT STDMETHODCALLTYPE IsLocked( 
		unsigned char  *lpContainerId,
		OmsTypeOid     *lpoid,
		 short         *lpb_error);
	
	virtual HRESULT STDMETHODCALLTYPE LeaveCriticalSection(
    short regionId);

	virtual HRESULT STDMETHODCALLTYPE LockObj( 
		unsigned char  *lpConsistentView,
		unsigned char  *lpContainerId,
		OmsTypeOid     *lpoid,
		unsigned char  *lpobjseq,
		short          *lpb_error);

  virtual HRESULT STDMETHODCALLTYPE LockObjMass(
    tsp00_Int4              &cnt,
    tgg91_TransNo           *pConsistentView,
    tgg01_ContainerId      **ppContainerId,
    OmsTypeOid              *pOid,
    tgg91_PageRef           *pObjSeq,
    tsp00_Int8              &timeout,
    tsp00_Int4              &errCnt,
    tgg00_BasisError        *pDBError);
 
	virtual HRESULT STDMETHODCALLTYPE MultiPurpose( 
		unsigned char   MessType,
		unsigned char   MessType2,
		tsp00_Int4     *lpbuflen,
		unsigned char  *lpv,
		short          *lpb_error);

	virtual HRESULT STDMETHODCALLTYPE NewConsistentView( 
		unsigned char  *lpConsistentView,
		short          *lpb_error);

	virtual HRESULT STDMETHODCALLTYPE NewObj( 
		unsigned char  *lpConsistentView,
		unsigned char  *lpContainerId,
		unsigned char  *lpVersionContext,
		tsp00_Int4      KeyPos,
		tsp00_Int4      KeyColSize,
		unsigned char  *KeyColumn,
		OmsTypeOid	   *lpoid,
		unsigned char  *lpobjseq,
		short          *lpb_error);

  virtual HRESULT STDMETHODCALLTYPE NewObjMass( 
    tsp00_Int4               cnt,
    tgg91_TransNo           *pConsistentView,
    tgg01_ContainerId      **ppContainerId,
    unsigned char           *pVersionContext,
    tsp00_Int4              *pKeyPos,
    tsp00_Int4              *pKeyColSize,
    unsigned char          **ppKeyColumn, 
    OmsTypeOid              *pOid,
    tgg91_PageRef           *pObjSeq,
    tsp00_Int4              &errCnt,
    tgg00_BasisError        *pDBError);

  virtual HRESULT STDMETHODCALLTYPE NextOids(
    tgg91_TransNo           &ConsistentView,
    tgg00_FileId            &ContainerId,
    tgg01_OmsVersionContext *pVersionContext,
    OmsTypeOid               CurrOid,
    tsp00_Int4              &noOfOid,
    OmsTypeOid              *pOid,
    tgg91_PageRef           *pObjVers,
    tsp00_Int4              &LogReadAccesses,
    tgg00_BasisError        *pb_error);

	virtual HRESULT STDMETHODCALLTYPE NextObjFromKeyIterator (
    const tgg91_TransNo     &ConsistentView,  // PTS 1127520 
    tgg01_ContainerId       &ObjFile,
    tgg01_OmsVersionContext *VersionContext,
    void                    *pKeyIterator,
    tsp00_Int4               bodySize,
    tsp00_Int4               KeyLen,
    void                    *pRestartKey, 
    tsp00_Int4              &NoOfOId,
    OmsTypeOid              *pResultOId, 
    tgg91_PageRef           *pObjVers,
    unsigned char           *pBody,
    tsp00_Int4              &objHistReadCount,
    tgg00_BasisError        &e);

  virtual HRESULT STDMETHODCALLTYPE VersionNextKey(
		tgg01_OmsVersionContext  *VersionContext,
		const tgg01_ContainerId  &ObjFile,
		void                     *pKey,
		tsp00_Int4                KeyLen,
		bool                      Inclusive,
		const void               *pStopKey,
		OmsTypeOid               &Oid,
		tgg00_BasisError         &e);

	virtual HRESULT STDMETHODCALLTYPE PrevObjFromKeyIterator ( 
    const tgg91_TransNo     &ConsistentView,  // PTS 1127520 
    tgg01_ContainerId       &ObjFile,
    tgg01_OmsVersionContext *VersionContext,
    void                    *pKeyIterator,
    tsp00_Int4               bodySize,
    tsp00_Int4               KeyLen,
    void                    *pRestartKey, 
    tsp00_Int4              &NoOfOId,
    OmsTypeOid              *pResultOId, 
    tgg91_PageRef           *pObjVers,
    unsigned char           *pBody,
    tsp00_Int4              &objHistReadCount,
    tgg00_BasisError        &e);

	virtual HRESULT STDMETHODCALLTYPE VersionPrevKey(   
		tgg01_OmsVersionContext  *VersionContext,
		const tgg01_ContainerId  &ObjFile,
		void                     *pKey,
		tsp00_Int4                KeyLen,
		bool                      Inclusive,
		const void               *pStopKey,
		OmsTypeOid               &Oid,
		tgg00_BasisError         &e);

	virtual HRESULT STDMETHODCALLTYPE RegisterClass(
		tsp00_Int4           SchemaId,
		GUID                *lpGUID,
		tsp00_KnlIdentifier *lpClassName,
		tsp00_Int4           size,
		OmsContainerNo       ContainerNo,
		tsp00_Int4           keyPos,          // PTS 1122540
		tsp00_Int4           keyLen,          // PTS 1122540
		bool                 keyPartitioned,  // PTS 1122540
		unsigned char       *lpContainerId,
		short               *lpb_error);

	virtual HRESULT STDMETHODCALLTYPE ResetVersion( 
		unsigned char *lpVersionContext,
		short	        &error);
 
	virtual HRESULT STDMETHODCALLTYPE SetError( 
		short                 errorno,
		tsp00_Int4            buflen,
		const unsigned char  *lpv,
		tsp00_CodeType        codeType );
	
	virtual HRESULT STDMETHODCALLTYPE SQLRequest( 
		tsp00_Int4      buflen,
		unsigned char  *lpv);

	virtual HRESULT STDMETHODCALLTYPE SubtransCommit( 
		short  *lpb_error);

	virtual HRESULT STDMETHODCALLTYPE SubtransRollback( 
		short  *lpb_error);

	virtual HRESULT STDMETHODCALLTYPE SubtransStart( 
		short  *lpb_error);

	virtual HRESULT STDMETHODCALLTYPE UnlockObj( 
		unsigned char  *lpContainerId,
		OmsTypeOid     *lpoid,
		short          *lpb_error);
	
	virtual HRESULT STDMETHODCALLTYPE UpdateObj( 
		unsigned char  *lpConsistentView,
		unsigned char  *lpContainerId,
		unsigned char  *lpVersionContext,
		unsigned char  *lpoid,
		unsigned char  *lpobjseq,
		tsp00_Int4      bodysize,
		unsigned char  *lpbody,
		tsp00_Int4      keyPos,            // PTS 1122540
		tsp00_Int4      keyLen,            // PTS 1122540
    bool            contObj,           // PTS 1124278
		short          *lpb_error);
  
  virtual HRESULT STDMETHODCALLTYPE UpdateObjMass(
    tsp00_Int4               cnt,
    tgg91_TransNo           *pConsistentView,
    tgg01_ContainerId      **ppContainerId,
    unsigned char           *pVersionContext,
    OmsTypeOid              *pOid,
    tgg91_PageRef           *pObjSeq,
    tsp00_Int4              *pBodySize,
    unsigned char          **ppBody,
    tsp00_Int4              *pKeyPos,       // PTS 1122540
    tsp00_Int4              *pKeyLen,       // PTS 1122540
    tsp00_Int4              &errCnt,
    tgg00_BasisError        *pDBError);

  virtual HRESULT STDMETHODCALLTYPE UpdateVarObjMass(
    tsp00_Int4               cnt,
    tgg91_TransNo           *pConsistentView,
    tgg01_ContainerId      **ppContainerId,
    unsigned char           *pVersionContext,
    OmsTypeOid              *pOid,
    tgg91_PageRef           *pObjSeq,
    tsp00_Int4              *pObjSize,
    tsp00_Int4              *pBodySize,
    unsigned char          **ppBody,
    OmsTypeOid              *pNextOid,
    bool                    *pContObj,  // PTS 1124278
    tsp00_Int4              &errCnt,
    tgg00_BasisError        *pDBError);

	virtual HRESULT STDMETHODCALLTYPE Vtrace( 
		tsp00_Int4      buflen,
		unsigned char  *lpv);

	virtual HRESULT STDMETHODCALLTYPE GetDefaultContext(
		void         **p, 
		pasbool      **pCancelByte,
		tsp00_TaskId  &taskId /* PTS 1110315 */);
	
	virtual HRESULT STDMETHODCALLTYPE SetDefaultContext(
    void* p);

  virtual HRESULT STDMETHODCALLTYPE GetSqlSessionContext(
    void** p);
        
 virtual HRESULT STDMETHODCALLTYPE SetSqlSessionContext(
   void* p);

	virtual HRESULT STDMETHODCALLTYPE OpenFile(
    char	      *lpFileName,
    tsp00_Uint4  FileSize,
    boolean      doAppend, 
    tsp00_Int4	*FileHandle,
    short       *lpb_error);
	  
	virtual HRESULT STDMETHODCALLTYPE WriteFile(
    tsp00_Int4 FileHandle,
    char      *msg,
    short     *lpb_error); 

	virtual HRESULT STDMETHODCALLTYPE CloseFile(
    tsp00_Int4  FileHandle,
    short      *lpb_error);

	virtual HRESULT STDMETHODCALLTYPE SyncFile(
    char	     *lpFileName,
    tsp00_Int4  FileHandle,
    short      *lpb_error);

	virtual HRESULT STDMETHODCALLTYPE GetDateTime(
    tsp00_Date* Date,
    tsp00_Time* Time);
	
	virtual HRESULT STDMETHODCALLTYPE GetClock(
    tsp00_Int4* Sec,
    tsp00_Int4* MSec);

	virtual HRESULT STDMETHODCALLTYPE DropVersion( 
    unsigned char  *lpVersionContext,
    short          *lpb_error);
	
	virtual HRESULT STDMETHODCALLTYPE CreateSchema(
    const OmsTypeWyde *lpSchemaName,
    tsp00_Int4        *lpSchemaId,
    short             *lpb_error);

	virtual HRESULT STDMETHODCALLTYPE ExistsSchema(
		const OmsTypeWyde *lpSchemaName,
		tsp00_Int4        *lpSchemaId,
		short             *lpb_error);

	virtual HRESULT STDMETHODCALLTYPE DropSchema(
    tsp00_Int4  SchemaId,
    short      *lpb_error);

	virtual HRESULT STDMETHODCALLTYPE GetSchemaName(
    tsp00_Int4    SchemaId,
    OmsTypeWyde  *lpSchemaName,
    short        *lpb_error);

	virtual HRESULT STDMETHODCALLTYPE ExistsFile(
    char    *lpFileName,
    bool    &existsFile,
    bool    &isDirectory,
    short   &error);
	
	virtual HRESULT STDMETHODCALLTYPE DropFile(
    char	*lpFileName,
    short &error); 

	virtual HRESULT STDMETHODCALLTYPE GetWorkingDirectory(
    tsp00_Pathc &pathName,
    short       &error); 

	virtual HRESULT STDMETHODCALLTYPE GetDbrootSapPath(
    tsp00_Pathc &pathName,
    short       &error);
	  
	virtual HRESULT STDMETHODCALLTYPE LoadLibraryEx( 
    const char    *path, 
		bool           loadGlobal,
		HANDLE		    *hd,
		tsp00_ErrText &errtext,
		int	           errtext_size);

	virtual HRESULT STDMETHODCALLTYPE FreeLibrary( 
    HANDLE         hd,
		bool          &ok,
		tsp00_ErrText &errtext,
		int            errtext_size);

	virtual HRESULT STDMETHODCALLTYPE DropContainer(
    OmsSchemaHandle  schema,
		const GUID       &guid,
		OmsContainerNo    cno,
		tgg00_BasisError &error);

	virtual HRESULT STDMETHODCALLTYPE GetContainerInfo(
    tsp00_Uint4      containerHandle,
		GUID            &guid,
		OmsSchemaHandle &schema,
		OmsContainerNo  &cno,
		short           &error);

	virtual HRESULT STDMETHODCALLTYPE IsObjHistoryUsed( 
		const OmsTypeOid  &oid,
		bool              &isUsed,
		tgg00_BasisError  &e);

	virtual HRESULT STDMETHODCALLTYPE Reschedule(); /* PTS 1107849 */
	
	virtual HRESULT STDMETHODCALLTYPE LockUpdObj( 
		unsigned char   *lpConsistentView,
		unsigned char   *lpContainerId,
		OmsTypeOid      *lpoid,
		unsigned char   *lpobjseq,
		short           *lpb_error);

  virtual HRESULT STDMETHODCALLTYPE LockUpdObjMass(
    tsp00_Int4               cnt,
    tgg91_TransNo           *pConsistentView,
    tgg01_ContainerId      **ppContainerId,
    OmsTypeOid              *pOid,
    tgg91_PageRef           *pObjSeq,
    tsp00_Int4              &errCnt,
    tgg00_BasisError        *pDBError);

	virtual HRESULT STDMETHODCALLTYPE Signal(
    tsp00_TaskId TaskId, 
    tsp00_Bool   ok); // PTS 1110315

	virtual HRESULT STDMETHODCALLTYPE Wait(
    tsp00_Bool& ok); // PTS 1110315

	virtual HRESULT STDMETHODCALLTYPE ExistsContainer(
		tsp00_Int4       SchemaId,
		const GUID      &Guid,
		OmsContainerNo   ContainerNo,
		short           &error); // PTS 1110334

	virtual HRESULT STDMETHODCALLTYPE GetSessionHeapUsage(
		tsp00_8ByteCounter& heapUsage); 

	virtual HRESULT STDMETHODCALLTYPE EvalCallStatistics(
		 bool            AllocatorOnly,
		 OmsMonitorData &MonitorData); 

	virtual HRESULT STDMETHODCALLTYPE SimCtlDumpToHtml(
		OmsHandle                 &handle,
		const OmsObjectId         &obj,
		class OmsSimDumpCollector &str); 

	virtual HRESULT STDMETHODCALLTYPE SimCtlSetObjectName(
		OmsHandle         &handle,
		const OmsObjectId &obj,
		const char        *name); 

	virtual HRESULT STDMETHODCALLTYPE SimCtlGetObjectName(
		OmsHandle          &handle,
		const OmsObjectId  &obj,
		const char        **name);

	virtual HRESULT STDMETHODCALLTYPE SimCtlDestroyHandleCB(
		class OmsHandle          &handle,
		struct IliveCacheSink   **session); 

  virtual HRESULT STDMETHODCALLTYPE GetSessionAllocator(
    SAPDBMem_IRawAllocator *&allocator); 
  
  virtual HRESULT STDMETHODCALLTYPE GetOmsSessionAllocator(
    SAPDBMem_IRawAllocator *&allocator); 
  
  virtual HRESULT STDMETHODCALLTYPE SimCtlGetHTMLRef(
		class OmsHandle           &handle,
		const struct OmsObjectId &obj,
		const char              **data,
		int                      flags); 

    virtual HRESULT STDMETHODCALLTYPE IsInRegion(
      int   regionId,
      bool &inRegion); 
    
    virtual HRESULT STDMETHODCALLTYPE GetSqlClsAllocator(
      SAPDBMem_IRawAllocator* &allocator); 

   virtual HRESULT STDMETHODCALLTYPE AttachSequence(
     const tsp00_KnlIdentifier &sequenceName,
     tgg00_Surrogate           &sequenceId,
     tgg00_SeqInfo             &sequenceInfo,
     short                     &sqlError); 

   virtual HRESULT STDMETHODCALLTYPE NextVal(
     const tgg00_Surrogate &sequenceId,
     tgg00_SeqInfo         &sequenceInfo,
     tsp00_Uint4           &nextVal,
     short                 &sqlError);

    virtual HRESULT STDMETHODCALLTYPE GetGlobalUserAllocator(
      OmsAllocatorInterface* &allocator, 
      const char *name);

    virtual HRESULT STDMETHODCALLTYPE GetSessionUserAllocator(
      OmsAllocatorInterface* &allocator);

    virtual HRESULT STDMETHODCALLTYPE GetTransactionUserAllocator(
      OmsAllocatorInterface* &allocator);

    virtual HRESULT STDMETHODCALLTYPE GetCOMRoutineUserAllocator(
      OmsAllocatorInterface* &allocator);

    virtual HRESULT STDMETHODCALLTYPE UserAllocatorCtl(
      int   msg, 
      void *param);

    virtual HRESULT STDMETHODCALLTYPE IsUnicodeInstance(
      bool  &isUnicodeInstance,
      short &sqlError);

    virtual int STDMETHODCALLTYPE GetSinkType();

    virtual HRESULT STDMETHODCALLTYPE ResetMonitor();

private:
    LVCSim_CheckedAllocator       *m_pLLSessionAllocator;
    LVCSim_CheckedAllocatorInst   *m_pSessionAllocator;
    LVCSim_CheckedAllocatorInst   *m_pTransactionAllocator;
    LVCSim_CheckedAllocatorInst   *m_pCOMRoutineAllocator;
    static LVCSim_GlobalAllocator *m_pGlobalNamedAllocator;
};

/*! endclass: IliveCacheSink */

class LVCSim_TXList {
public:
    LVCSim_TXList() {}  // no init, will be inited later
    LVCSim_TXList(LVCSim_TID &ourTx);
    ~LVCSim_TXList();
    bool sees(LVCSim_TID otherTx) const;

    void AddRef();
    void Release(bool doLock);
private:
    int         m_refCnt;
    int         m_otherTxCount;                   // count of other transactions
    LVCSim_TID  m_ourTx;                          // our TX ID (read view)
    LVCSim_liveCacheSink::ConsViewList*  m_otherTxList[MAX_CONCURRENT_TX]; // start timestamps of concurrent TXs at start time
};


#endif 
