/*


    ========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/


package com.sap.dbtech.powertoys;

import java.util.Properties;
import com.sap.dbtech.util.StructuredMem;
import com.sap.dbtech.rte.comm.JdbcCommFactory;
import com.sap.dbtech.rte.comm.JdbcCommunication;
import com.sap.dbtech.rte.comm.SocketComm;
import com.sap.dbtech.rte.comm.SecureCommunication;
import com.sap.dbtech.rte.comm.RTEException;
/**
 * Executes administration commands for SAP DB
 */
public class DBM
{
    private static final String hostKeyC = "host";
    private static final String hostDefaultC = "localhost";
    private static final String dbnameKeyC = "dbname";
    private static final String dbnameDefaultC = "";
    private static final String dbrootKeyC = "dbroot";
    private static final String dbrootDefaultC = "";
    private static final String userKeyC = "user";
    private static final String pgmNameC = "dbmsrv";
    private static final String transportC = "transport";

    private static final int alignSizeC = 8;
    static final int indicatorLengthC = 4;

    private JdbcCommunication connection;
    /**
       creates a new DBM.

       The following properties are used by this class:
        <UL>
        <LI>"host": start dbmserver on this host
        <LI>"dbname": start dbmserver for this database
        <LI>"dbroot": start dbmserver for this release/diretory
        <LI>"user": user,pwd implicit dbm connect (not yet implemented)
        <UL>
     */
    public DBM (Properties properties)
        throws RTEException
    {
        String host = properties.getProperty (hostKeyC, hostDefaultC);
        String dbname = properties.getProperty (dbnameKeyC, dbnameDefaultC);
        String dbroot = properties.getProperty (dbrootKeyC , dbrootDefaultC);
        // String commtimeout = properties.getProperty(communicationtimeout_C, null);



        String transport = properties.getProperty(transportC, "socket");
        JdbcCommFactory factory = com.sap.dbtech.jdbc.DriverSapDB.getCommunicationFactory ();
        if(transport.equalsIgnoreCase("secure")) {
            factory = SecureCommunication.factory;
        }

        this.connection = factory.xopen(host, dbname,
                                        dbroot, pgmNameC, properties);

        // check for user and do implicit connect
    }

    /**
     * closes the connection
     */
    public void release ()
        throws RTEException
    {
        if (this.connection != null) {
            try {
              try {
                this.cmd("release");
              }
              catch (Exception ignore) {}
              this.connection.release ();
            }
            finally {
                this.connection = null;
            }
        }
    }
    /**
     *
     */
    public void finalize ()
        throws RTEException
    {
        this.release ();
    }
    /**
     * executes <i>cmdString</i> in the current dbm session.
     *
     * @exception DBMException
     * @exception RTEException
     */
    public String cmd (String cmdString)
        throws RTEException, DBMException
    {
        StructuredMem request;
        int alignedLen;
        StructuredMem reply;
        String result;

        /*
         * send request
         */
        request = this.connection.getRequestPacket ();
        alignedLen = (cmdString.length () + alignSizeC - 1)
            / alignSizeC * alignSizeC;
        request.putString (cmdString, 0);
        this.connection.request (request, alignedLen);
        reply = this.connection.receive ();
        /*
         * parse result and throw exception
         */
        String errorIndicator = reply.getString (0,
            Math.min (indicatorLengthC, reply.size ()));
        if (errorIndicator.startsWith ("OK")) {
            int dataStart = errorIndicator.indexOf ('\n') + 1;
            result = reply.getString (dataStart, reply.size () - dataStart);
        }
        else {
            throw DBMException.create (reply);
        }
        return result;
    }
    /**
     * creates a new DBM by specifying the host and the dbname
     */
    public static DBM dbDBM (String host, String dbname)
        throws RTEException
    {
        Properties props = new Properties ();

        if (host != null) {
            props.put (hostKeyC, host);
        }
        if (dbname != null) {
            props.put (dbnameKeyC, dbname);
        }
        return new DBM (props);
    }
    /**
     * create a new DBM by specifying the host and the dbroot
     */
    public static DBM dbrootDBM (String host, String dbroot)
        throws RTEException
    {
        Properties props = new Properties ();

        if (host != null) {
            props.put (hostKeyC, host);
        }
        props.put (dbrootKeyC, dbroot);
        return new DBM (props);
    }

}
