/*!
    \file    DBMSrvBLabel_Label.cpp
    \author  TiloH
    \ingroup backup label handling for DBM Server
    \brief   a class handling labels of backups

\if EMIT_LICENCE

    ========== licence begin LGPL
    Copyright (c) 1998-2005 SAP AG

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
    ========== licence end

\endif
*/

#include "DBM/Srv/Backup/Label/DBMSrvBLabel_Label.hpp"
#include "gcn00.h"
#include "hcn36.h"
#include "hcn90.h"
#include "heo02.h"
#include "heo06.h"
#include "gkb03.h"

int DBMSrvBLabel_Label::m_MaxAllowedBlockSize=32*1024*1024; //a maximal allowed block size of 32 MB should be sufficient for a long time

DBMSrvBLabel_Label::DBMSrvBLabel_Label()
  : m_Label(0),
    m_StartDate(0),
    m_StartTime(0),
    m_Server(0),
    m_StopDate(0),
    m_StopTime(0),
    m_DB(0),
    m_DBStamp1Date(0),
    m_DBStamp1Time(0),
    m_DBStamp2Date(0),
    m_DBStamp2Time(0)
{
}

DBMSrvBLabel_Label::~DBMSrvBLabel_Label()
{
    cn36_StrDealloc(m_Label);
    cn36_StrDealloc(m_StartDate);
    cn36_StrDealloc(m_StartTime);
    cn36_StrDealloc(m_Server);
    cn36_StrDealloc(m_StopDate);
    cn36_StrDealloc(m_StopTime);
    cn36_StrDealloc(m_DB);
    cn36_StrDealloc(m_DBStamp1Date);
    cn36_StrDealloc(m_DBStamp1Time);
    cn36_StrDealloc(m_DBStamp2Date);
    cn36_StrDealloc(m_DBStamp2Time);
}

bool DBMSrvBLabel_Label::readLabelFrom(
    const char * deviceName,
    size_t       blockSize,
    char       * replyData,
    int        * replyLen,
    int          replyLenMax)
{
    bool                rc=true;
    tsp00_Int4          bufferSize=(int)blockSize;
    void              * pBuffer;
    tsp00_Int4          fileHandle;
    tsp05_RteFileError  rteFileErr;
    
    /* first try to open with given block size */
    sqlfsaveopenc(deviceName, sp5vf_read, &bufferSize, &fileHandle, &pBuffer, &rteFileErr);

    if(rteFileErr.sp5fe_result!=vf_ok &&    //if first try was unsuccessful ...
       bufferSize!=blockSize &&             //but returned an alternative block size ...
       0<bufferSize &&                      //that is ...
       bufferSize<=m_MaxAllowedBlockSize)   //plausible ...
    {
        sqlfsaveopenc(deviceName, sp5vf_read, &bufferSize, &fileHandle, &pBuffer, &rteFileErr); //... try open with new block size
    }

    if(rteFileErr.sp5fe_result!=vf_ok)
    {
        rc=false;
        cn90AnswerRTEError(replyData, replyLen, ERR_RTE_CN00, rteFileErr.sp5fe_text,
            rteFileErr.sp5fe_text.length(), rteFileErr.sp5fe_result);
    }
    else
    {
        tsp00_Longint nOut;

        sqlfreadc(fileHandle, pBuffer, bufferSize,
                  &nOut, &rteFileErr);

        if(rteFileErr.sp5fe_result!=vf_ok)
        {
            rc=false;
            cn90AnswerRTEError(replyData, replyLen, ERR_RTE_CN00, rteFileErr.sp5fe_text,
                rteFileErr.sp5fe_text.length(), rteFileErr.sp5fe_result );
        }
        else
        {
            tkb3_info_stuff info;

            memcpy(&info, pBuffer, sizeof(info));

            if(!createCopy(m_Label,        info.inf_label,         sizeof(info.inf_label),         replyData, replyLen) ||
               !createCopy(m_StartDate,    info.inf_start_date,    sizeof(info.inf_start_date),    replyData, replyLen) ||
               !createCopy(m_StartTime,    info.inf_start_time,    sizeof(info.inf_start_time),    replyData, replyLen) ||
               !createCopy(m_Server,       info.inf_servernode,    sizeof(info.inf_servernode),    replyData, replyLen) ||
               !createCopy(m_StopDate,     info.inf_end_date,      sizeof(info.inf_end_date),      replyData, replyLen) ||
               !createCopy(m_StopTime,     info.inf_end_time,      sizeof(info.inf_end_time),      replyData, replyLen) ||   
               !createCopy(m_DB,           info.inf_serverdb,      sizeof(info.inf_serverdb),      replyData, replyLen) ||
               !createCopy(m_DBStamp1Date, info.inf_dbstamp1_date, sizeof(info.inf_dbstamp1_date), replyData, replyLen) ||
               !createCopy(m_DBStamp1Time, info.inf_dbstamp1_time, sizeof(info.inf_dbstamp1_time), replyData, replyLen) ||
               !createCopy(m_DBStamp2Date, info.inf_dbstamp2_date, sizeof(info.inf_dbstamp2_date), replyData, replyLen) ||
               !createCopy(m_DBStamp2Time, info.inf_dbstamp2_time, sizeof(info.inf_dbstamp2_time), replyData, replyLen)    )
            {
                rc=false;
            }
        }

        sqlfclosec(fileHandle, sp5vf_close_normal, &rteFileErr);
    }

    return rc;
}

const char * DBMSrvBLabel_Label::getLabel() const
{
    return m_Label?m_Label:"";
}

const char * DBMSrvBLabel_Label::getStartDate() const
{
    return m_StartDate?m_StartDate:"";
}

const char * DBMSrvBLabel_Label::getStartTime() const
{
    return m_StartTime?m_StartTime:"";
}

const char * DBMSrvBLabel_Label::getStopDate() const
{
    return m_StopDate?m_StopDate:"";
}

const char * DBMSrvBLabel_Label::getStopTime() const
{
    return m_StopTime?m_StopTime:"";
}

const char * DBMSrvBLabel_Label::getServer() const
{
    return m_Server?m_Server:"";
}

const char * DBMSrvBLabel_Label::getDB() const
{
    return m_DB?m_DB:"";
}

const char * DBMSrvBLabel_Label::getDBStamp1Date() const
{
    return m_DBStamp1Date?m_DBStamp1Date:"";
}

const char * DBMSrvBLabel_Label::getDBStamp1Time() const
{
    return m_DBStamp1Time?m_DBStamp1Time:"";
}

const char * DBMSrvBLabel_Label::getDBStamp2Date() const
{
    return m_DBStamp2Date?m_DBStamp2Date:"";
}

const char * DBMSrvBLabel_Label::getDBStamp2Time() const
{
    return m_DBStamp2Time?m_DBStamp2Time:"";
}

bool DBMSrvBLabel_Label::getStartDateTime(tcn36_DateTimeString & time) const
{
    bool rc=true;

    if(0<strlen(getStartDate()) && 0<strlen(getStartTime()))
    {
        char * h=0;

        rc=cn36_StrAlloc(h, strlen(getStartDate())+1+strlen(getStartTime()))?true:false;

        if(rc)
        {
            strcpy(h, getStartDate());
            strcat(h, " ");
            strcat(h, getStartTime());

            time.SetTo(h, "yyyymmdd 00HHMMSS");
        }

        cn36_StrDealloc(h);
    }

    return rc;
}

bool DBMSrvBLabel_Label::equals(const DBMSrvBLabel_Label & otherLabel) const
{
    return (0==strcmp(getLabel(),        otherLabel.getLabel()) &&
            0==strcmp(getStartDate(),    otherLabel.getStartDate()) &&
            0==strcmp(getStartTime(),    otherLabel.getStartTime()) &&
            0==strcmp(getServer(),       otherLabel.getServer()) &&
            0==strcmp(getStopDate(),     otherLabel.getStopDate()) &&
            0==strcmp(getStopTime(),     otherLabel.getStopTime()) &&
            0==strcmp(getDB(),           otherLabel.getDB()) &&
            0==strcmp(getDBStamp1Date(), otherLabel.getDBStamp1Date()) &&
            0==strcmp(getDBStamp1Time(), otherLabel.getDBStamp1Time()) &&
            0==strcmp(getDBStamp2Date(), otherLabel.getDBStamp2Date()) &&
            0==strcmp(getDBStamp2Time(), otherLabel.getDBStamp2Time())    );
}

bool DBMSrvBLabel_Label::createCopy(
        char       *& dest,
        const char *  src,
        size_t        maxLengthSrc,
        char       *  replyData,
        int        *  replyLen)
{
    bool rc=cn36_StrAlloc(dest, maxLengthSrc)?true:false;

    if(rc)
        cn90StringPtoC(dest, src, (int)maxLengthSrc);
    else
        cn90AnswerIError(replyData, replyLen, ERR_MEM_CN00);

    return rc;
}
