/*!
  @file           Converter_IBackUp.hpp
  @author         TorstenS
  @author         AlexanderK
  @ingroup        Converter
  @brief          abstract back up interface class of the converter
*/

/*
    ========== licence begin  GPL
    Copyright (c) 2001-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

*/



#ifndef CONVERTER_IBACKUP_HPP
#define CONVERTER_IBACKUP_HPP


/*===========================================================================*
 *  INCLUDES                                                                 *
 **/

#include "gsp00.h"
#include "DataAccess/Data_Types.hpp"

class IOMan_BlockAddress;
class Converter_LeafPage;
class Converter_BitMapPage;
class Converter_UsedPageNoContainer;

/*!
  @interface    Converter_IBackUp
  @brief        Abstract back up interface class of the converter
 */

class Converter_IBackUp
{
public:
    /*!
       @brief   Returns the backup interface of the converter. This static
                function returns the interface of the converter which allows
                access to all methods defined in this interface class
       @return  (Converter_IBackUp) instance of the converter
     */

    static   Converter_IBackUp & Instance();

    /*!
       @brief   Starts a total backup (also called save data). All currently
                occupied blocks of the database are marked within the FBM to
                be included in the back up
       @param   taskId [in] task id
       @return  none
     */

    virtual void BeginSaveData( const tsp00_TaskId taskId ) = 0;

    /*!
       @brief   Starts a differential backup (also called save pages). All
                currently occupied blocks of the database which are marked as
                changed since the last total backup are marked within the FBM
                to be included in the back up. A list off all page numbers
                used in the current data base is created
       @param   taskId [in] task id
       @return  none
     */

    virtual void BeginSavePages( const tsp00_TaskId taskId ) = 0;

    /*!
        @brief  Ends the backup of data or pages. The back up active flag in
                the converter is reset. If  the current back up  did not finish
                successfully all blocks still marked as to be written to the
                back up files are unmarked.
        @param  taskId                      [in] current task id
        @param  bBackUpSuccessfullyFinished [in] indicates whether the current back
                                                 up finished successfully
        @return none
    */

    virtual void  EndSave(
        const tsp00_TaskId  taskId,
        const SAPDB_Bool    bBackUpSuccessfullyFinished ) = 0;

    /*!
        @brief  Returns the number of of pages marked for back up
        @return none
    */

    virtual SAPDB_UInt GetNumPagesForBackUp() = 0 ;

    /*!
        @brief  Returns the number of converter pages required to address all
                data pages which belong to the current backup.
        @return SAPDB_UInt
    */

    virtual SAPDB_UInt GetNumConvPagesForBackUp() = 0;

    /*!
       @brief   Checks if the current back up finished successfully. Checks if
                all pages marked as to be included in the back up were fetched
                by the back up manager
       @param   taskId [in] current task id
       @return  true if the back was successful else false
     */

    virtual SAPDB_Bool CheckBackUp( const tsp00_TaskId taskId ) const = 0;

    /*!
       @brief   Signals the converter that a given page was written into the back
                up file. Resets the flag in the FBM which displays that the given
                page still has to be written to the back up file. Removes the page
                number from the list of page numbers still to inlclude in the back up
       @param   TaskId         [in] current task id
       @param   BlockAddress   [in] block address for which the back up file is to
                                    remove in the FBM
       @param   PageNo         [in] page number which is to remove from the list of
                                    pages still to write to the back up device
       @param   AddressingMode [in] addressing mode of the page for which the back up
                                    flags have to be removed
       @return  true if the page number was still in the list of pages still to write
                to the back up device and if the corresponding block was still marked
                for back up
     */

    virtual SAPDB_Bool  ResetDataPageAfterBackUp(
        const tsp00_TaskId          &TaskId,
        const IOMan_BlockAddress    &BlockAddress,
        const Data_PageNo           &PageNo,
        const Data_AddressingMode   &AddressingMode ) = 0;

    /*!
       @brief   Sets a block address in a converter entry. After a page was restored
                this command updates the converter entry, i.e the block address of
                the restored page is updated in the converter entry
       @param   TaskId         [in] current task id
       @param   BlockAddress   [in] new block address which hast to be inserted in
                                    the converter entry of the page defined by PageNo
                                    and PageRecoveryMode
       @param   PageNo         [in] page number for which the converter entry has to
                                    be updated
       @param   AddressingMode [in] addressing mode of the page for which the back up
                                    flags have to be removed
       @return  none
     */

    virtual void SetBlockAddressOfRestoredPage(
        const tsp00_TaskId          &TaskId,
        const IOMan_BlockAddress    &BlockAddress,
        const Data_PageNo           &PageNo,
        const Data_AddressingMode   &AddressingMode ) = 0;

    /*!
       @brief   This method fills the given page frame with a the copy
                of a static converter leaf page. After the page frame
                is copied the original page frame is deallocated.
       @param   leafPage [out] leaf page to be filled
       @return  SAPDB_Bool
     */

    virtual SAPDB_Bool GetStaticArchiveLeafPage( Converter_LeafPage &leafPage ) = 0;

    /*!
       @brief   This method fills the given page frame with a the copy
                of a dynamic converter leaf page. After the page frame
                is copied the original page frame is deallocated.
       @param   leafPage [out] leaf page to be filled
       @return  SAPDB_Bool
     */

    virtual SAPDB_Bool GetDynamicArchiveLeafPage( Converter_LeafPage &leafPage ) = 0;

    /*!
       @brief   Clears the containers with copied converter leafpages. Removes
                all copied converter leafpages which belong to a backup
       @return  none
     */

    virtual void RemoveArchiveLeafPages() = 0;

    /*!
       @brief   This method is used to write the content of a static
                converter bit map page into a given page frame. After
                the page frame is copied the original page frame is deallocated.
                The copied page frame will be copied into the backup file.
       @param   bitMapPage [in|out] page to be copied into
       @return  (SAPDB_Bool) true if page is available; else false
     */

    virtual SAPDB_Bool GetStaticBitMapPage( Converter_BitMapPage   &bitMapPage ) = 0;

    /*!
       @brief   This method is used to write the content of a dynamic
                converter bit map page into a given page frame. After
                the page frame is copied the original page frame is deallocated.
                The copied page frame will be copied into the backup file.
       @param   bitMapPage [in|out] page to be copied into
       @return  (SAPDB_Bool) true if page is available; else false
     */

    virtual SAPDB_Bool GetDynamicBitMapPage( Converter_BitMapPage   &bitMapPage ) = 0;

     /*!
       @brief   This method is used to clean up the internal static and
                dynamic converter bit map vector, which is used for data
                backup's (save pages)
       @return  none
     */

    virtual void RemoveBitMapPages() = 0;

    /*!
       @brief   Starts recovery of used pagenos. This method must be called prior
                to the first call to RestoreUsedPageNos() to allocated the needed
                memory.
       @return  (SAPDB_Bool) true if successful
     */

    virtual SAPDB_Bool BeginRestoreUsedPageNos() = 0;

    /*!
        @brief  Releases page numbers which do not belong to a restored data base.
                All page number which are not marked as used at the UsedPageNoBitMapPage
                the corresponding converter entry is set to free and the block is released.
                All page numbers which are marked as used in the UsedPageNoBitMapPage but
                not in the converter the correspondung page entry is reserved. All
                converter pages which contain any used page number are marked as used
                (see also FreePageNosNotInConverterBitMapPages)
        @param  TaskId               [in] task id
        @param  UsedPageNoBitMapPage [in] page which stores the page numbers of
                                      used pages in a bit map
        @return none
    */

    virtual void RestoreUsedPageNos(
        const tsp00_TaskId          TaskId,
        const Converter_BitMapPage &UsedPageNoBitMapPage ) = 0;

    /*!
        @brief  integrate converter pages from backup into the converter.
        @param  taskId [in] own task id
        @param  leafPage [in] converter page from backup medium
        @return none
    */

    virtual void RestoreConverterPage(
        const tsp00_TaskId  taskId,
        Converter_LeafPage  &leafPage ) = 0;

    /*!
        @brief  Releases page numbers which do not belong to a restored data base.
                After all UsedPageNoBitMapPage of a back up are read this function
                scans the converter and removes all those converter leaf pages
                (and their entries) which are not marked as used. (see also FreeUnusedPageNo)
                This means all those converter entries which were neither marked
                as used nor as free in the UsedPageNoBitMapPage's (means not at
                all mentioned) are removed.
        @param  TaskId [in] task id
        @return none
    */

    virtual void FinishRestoreUsedPageNos( const tsp00_TaskId TaskId ) = 0;
};

#endif  /* CONVERTER_IBACKUP_HPP */
