/*
  -----------------------------------------------------------------------------
 
  module: vls18.cpp
 
  -----------------------------------------------------------------------------
 
  responsible:  SteffenS and d033893

  special area: Replication Server
 
  description:  Implementation of data file row and value functions
                
  version:      7.5.

  -----------------------------------------------------------------------------
 
  copyright:    (c) 1999-2004 SAP AG-2004
 
  -----------------------------------------------------------------------------



    ========== licence begin  GPL
    Copyright (c) 1999-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



*/


#include "gls00.h"
#include "hls05.h"
#include "hls07.h"
#include "hls18.h"
#include "hls19_check.h"
#include "hls98msg.h"
#include "hls99.h"

#include "SAPDBCommon/SAPDB_UTF8Basis.hpp"

#define _INTERNAL_FILE_ID_    _T("vls18.cpp")

#define NL_UNDEFINED_LS18                           0
#define NL_SINGLE_DELIMITER_LS18                    1
#define NL_DELIMITER_PAIR_LS18                      2
#define NL_SEPARATOR_FOLLOWED_BY_DELIMITER_LS18     3

tsp77encoding const *pEncoding_ls18[8] = {sp77encodingAscii,
                                          sp77encodingAscii,
                                          sp77encodingAscii,
                                          sp77encodingUTF8,
                                          sp77encodingAscii,
                                          sp77encodingUCS2,
                                          sp77encodingUCS2Swapped,
                                          sp77encodingUCS2Native};
/*
  -----------------------------------------------------------------------------
  Chapter: Private functions declaration 
  -----------------------------------------------------------------------------
*/
/*
  -----------------------------------------------------------------------------
  function:     ls18_RemoveDelimiter
  -----------------------------------------------------------------------------

  description:  special function only used for support of ls18GetUnformattedValueEx:
                Removes first delimiter from first value and last delimiter
                from last value.

  arguments:    pszStr        [OUT]  - string to remove delimiter from
                lStrLen       [OUT]  - length from string above
                pszDelimiter  [IN]   - delimiter string to remove
                lDelimiterLen [IN]   - length of delimiter above
                bFirst        [IN]   - first or last value

  returnvalue:  No
  -----------------------------------------------------------------------------
*/
static void
ls18_RemoveDelimiter(tsp00_Addr     &pszStr,
                     tsp00_Longint  &lStrLen,
                     tsp00_Addr      pszDelimiter,
                     tsp00_Longint   lDelimiterLen,
                     bool            bFirst = true);

/*
  -----------------------------------------------------------------------------
  EndChapter: Private functions declaration 
  -----------------------------------------------------------------------------
*/


/*
  -----------------------------------------------------------------------------
  Chapter: Public functions implementation
  -----------------------------------------------------------------------------
*/

/*
  -----------------------------------------------------------------------------
  function:     ls18GetRowLength
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls18GetRowLength(tls00_MultColSpec      &Columns,
                 tls00_TableDescription &TableInfo,
                 //bool                    bBinary,    // defaults to false
                 bool                    bUseLastColPos,
                 bool                    bParam)     // defaults to false
{
    ROUTINE_DBG_MEO00 ("ls18GetRowLength");

    tsp00_Int4 lBytes = 0;    // This will be the return value
            
    if (true == bUseLastColPos)
    {
        //  Formatted (text and binary) --> Max position defines the count of bytes to read
        TRACE_PRNF_MLS99(("ls18GetRowLength", "using last column position"));
        
        int i = 0;
        for (i; i < Columns.mcsCount_ls00; i++)
        {
            if ( Columns.mcsColumn_ls00[i]->colFPos_ls00.fpoEnd_ls00 > lBytes )
            {
                lBytes = Columns.mcsColumn_ls00[i]->colFPos_ls00.fpoEnd_ls00;
            }
        }
        TRACE_PRNF_MLS99(("ls18GetRowLength", "Bytes per row: %d\n", lBytes));
    }
    else
    { 
        // Compressed in text mode
        // Special treatment for LONG columns: DEF_MAX_LONG_VALUE_LS00 Bytes is max. length
        
        // Data is saved in ASCII. We count 1 byte per character/digit.
        // The character count = sp7fi_len.
        // characters:              sp7fi_len = sp7fi_in_out_l - 1
        // numbers (integers ...):  sp7fi_len = number of digits.
        // If we rely in case of text files on the position specification in the load command
        // the read function would need to reallocate space any time the positions give a
        // wrong length. To prevent this we use the internal length.
        tls00_FieldBasics   *pBasicInfo;        //PTS 1107309 (S.S.)

        int i = 0;
        for (i; i < TableInfo.tdFieldCount_ls00; ++i)
        {
            if (true == bParam)
            {
                pBasicInfo = &TableInfo.tdParamArray_ls00[i]->piBasicInfo_ls00;
                TRACE_PRNF_MLS99(("ls18GetRowLength", "Params used\n"));
            }
            else
            {
                pBasicInfo = &TableInfo.tdFieldsArray_ls00[i]->fiBasics_ls00;
                TRACE_PRNF_MLS99(("ls18GetRowLength", "Fields used\n"));
            }

            if (1 == szMap_LongDataType_ls00[pBasicInfo->fbDataType_ls00])    // LONG column
            {
                lBytes += DEF_MAX_LONG_VALUE_LS00;  // Add delimiters and separators
                
                TRACE_PRNF_MLS99(("ls18GetRowLength", "Long column: + %d bytes\n", DEF_MAX_LONG_VALUE_LS00));
            }
            else
            {
                lBytes += pBasicInfo->fbLength_ls00;

                TRACE_PRNF_MLS99(("ls18GetRowLength", "Column: + %d bytes\n", pBasicInfo->fbLength_ls00));
            }
        }
        lBytes += (Columns.mcsCount_ls00*3 - 1);    // Add delimiters and separators
        
        TRACE_PRNF_MLS99(("ls18GetRowLength", "Including dels and seps bytes per row: %d\n", lBytes));
    }

    return lBytes;
}
//  ls18GetRowLength()

/*
  -----------------------------------------------------------------------------
  function:     ls18GetRowLengthForExtract
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls18GetRowLengthForExtract(tls00_MultColSpec      *Columns,
                           tls00_ColumnData       *pColumnData,
                           tls00_TableDescription *TableInfo,
                           tls00_FileFormat       *FileFormat)
{
    ROUTINE_DBG_MEO00 ("ls18GetRowLengthForExtract");

    tsp00_Int4 lBytes = 0;  // the return value
    int        i      = 0;

    if (true == FileFormat->ffoFormatted_ls00)
    {
        lBytes = ls18GetRowLength(*Columns, *TableInfo, true, true);
    }
    else
    {
        lBytes = ls18GetRowLength(*Columns, *TableInfo, false, true);

        for (i=0; i < Columns->mcsCount_ls00; i++)
        {
            tls00_Column *myColumn = Columns->mcsColumn_ls00[i];
            if (true == myColumn->colIsLit_ls00)        // literal is null terminated
            {
                // add bytes for delimiters und separators
                lBytes += 3 + STATIC_CAST( tsp00_Int4, myColumn->colName_ls00.length() );
                continue;
            }

            // We use the length check somewhat incorrectly: we specify a value length of 0
            // and get the right length back
            tsp00_Int4 rc = errOK_els00;
            tsp00_Int4 ValLen = 0;
            tsp00_Int4 NILValLen = 0;
        
            rc = ls19CheckLengthForExtract(0,
                                           myColumn,
                                           &TableInfo->tdParamArray_ls00[pColumnData[i].cdPosInStmt_ls00]->piBasicInfo_ls00,
                                           FileFormat,
                                           ValLen);     // needed length

            rc = ls19CheckNILValLength(0,
                                       myColumn,
                                       TableInfo->tdParamArray_ls00[pColumnData[i].cdPosInStmt_ls00]->piMode_ls00,
                                       FileFormat,
                                       NILValLen);
            
            lBytes += (NILValLen > ValLen) ? NILValLen: ValLen;
        }
    }

    return lBytes;
}
//  ls18GetRowLengthForExtract()


/*
  -----------------------------------------------------------------------------
  function:     ls18GetRecordLength
  -----------------------------------------------------------------------------
*/
tsp00_Int2
ls18GetRecordLength(tls00_TableDescription & TableInfo, bool bParam)
{
    ROUTINE_DBG_MEO00 ("ls18GetRecordLength");

    tsp00_Int2 Length = 0;
    tsp00_Int4 i      = 0;

    if (true == bParam)     // work with params
    {
        //* record length corresponds to the buffer position plus
        //* length of parameter for this position. But the buffer position
        //* are not sorted ascending --> find max
        tsp00_Int2 Max = 0;
        
        for(i=0; i < TableInfo.tdFieldCount_ls00; ++i)
        {
            tls00_ParamInfo *Param = TableInfo.tdParamArray_ls00[i];

            Max = Param->piBasicInfo_ls00.fbInOutLength_ls00 + Param->piBufpos_ls00 - 1;
            if(Length < Max)
            {
                Length = Max;
            }
        }
    }
    else        // work with fields; THIS IS MAINLY (ONLY) FOR FASTLOAD !!!!
    {
        Length = 8;     // 8 byte header in every record

        for(i=0; i < TableInfo.tdFieldCount_ls00; ++i)
        {
            tls00_FieldInfo *FI = TableInfo.tdFieldsArray_ls00[i];
            
            Length += FI->fiBasics_ls00.fbInOutLength_ls00;

            // depending on length type of column some bytes need to be added
            // the following length types exist: sp7_fix_len, sp7_var_len, sp7_var_len_long
            switch (FI->fiLengthType_ls00)
            {
                case sp7_var_len:
                {
                    //  syskey or last column of key are variable long ===> skip this column
                    if ( (mod_key    != FI->fiColMode_ls00) &&
                         (mod_syskey != FI->fiColMode_ls00) )
                    {
                        Length += 1;        // add the length byte to the record length
                    }
                    break;
                }
                case sp7_var_len_long:
                {
                    Length += 2;            // add length bytes to the record length
                    break;
                }
                default:    // corresponds to sp7_fix_len
                {
                    break;      // nothing to add
                }
            } //switch (FI->fiLengthType_ls00)

        } //for(i=0; i < TableInfo.tdFieldCount_ls00; ++i)
    }

    return Length;
}
// ls18GetRecordLength()


/*
  -----------------------------------------------------------------------------
  function:     ls18RecordInit
  -----------------------------------------------------------------------------
*/
void
ls18RecordInit(tsp00_Int2 ColumnCount, tls00_String *& DataArray, tls00_String & pszString)
{
    ROUTINE_DBG_MEO00 ("ls18RecordInit");

    tsp00_Int4 i = 0;

    for (i; i<ColumnCount; ++i)
    {
        if (NULL != DataArray[i].strAddr_ls00)
        {
            // If the pointer is within string (e.g. &pszString[0] < ptr < &pszString[lLength]),
            // set simply to NULL. Otherwise also delete the pointer.
            if ( (DataArray[i].strAddr_ls00 < &pszString.strAddr_ls00[0]) ||
                 (DataArray[i].strAddr_ls00 > &pszString.strAddr_ls00[pszString.strLeng_ls00]) )
            {
                delete [] DataArray[i].strAddr_ls00;
            }
            DataArray[i].strAddr_ls00 = NULL;
        }
        DataArray[i].strLeng_ls00 = INSERT_EMPTY_VALUE;
    }
}
//  ls18RecordInit()


/*
  -----------------------------------------------------------------------------
  function:     ls18RecordInit - overloaded
  -----------------------------------------------------------------------------
*/
void
ls18RecordInit(tsp00_Int2      ColumnCount,
               tls00_String *& DataArray,
               tsp00_Addr      pszString,
               tsp00_Int4      lStrLen)
{
    ROUTINE_DBG_MEO00 ("ls18RecordInit");

    tsp00_Int4 i = 0;

    for (i; i<ColumnCount; ++i)
    {
        if (NULL != DataArray[i].strAddr_ls00)
        {
            // If the pointer is within string (e.g. &pszString[0] < ptr < &pszString[lLength]),
            // set simply to NULL. Otherwise also delete the pointer.
            if ( (DataArray[i].strAddr_ls00 < &pszString[0]) ||
                 (DataArray[i].strAddr_ls00 > &pszString[lStrLen]) )
            {
                delete [] DataArray[i].strAddr_ls00;
            }
            DataArray[i].strAddr_ls00 = NULL;
        }
        DataArray[i].strLeng_ls00 = INSERT_EMPTY_VALUE;
    }
}
//  ls18RecordInit()


/*
  -----------------------------------------------------------------------------
  function:     ls18RecordConstructor
  -----------------------------------------------------------------------------
*/
tsp00_Int2
ls18RecordConstructor(tsp00_Int2 ColumnCount, tls00_String *& DataArray)
{
    ROUTINE_DBG_MEO00 ("ls18RecordConstructor");

    tsp00_Int2  rc  = errOK_els00;

    DataArray = new tls00_String[ColumnCount];
    if (NULL == DataArray)
    {
        rc = errNoMoreMemory_els98;
    }
    else
    {
        tsp00_Int4 i = 0;
            
        for (i; i< ColumnCount; i++)
        {
            DataArray[i].strLeng_ls00 = INSERT_EMPTY_VALUE;
            DataArray[i].strAddr_ls00 = NULL;
        }
    }
    return rc;
}
//  ls20RecordConstructor()

/*
  -----------------------------------------------------------------------------
   function:     ls18RecordDestructor
  -----------------------------------------------------------------------------
*/
void
ls18RecordDestructor(tsp00_Int2 ColumnCount, tls00_String *& DataArray)
{
    ROUTINE_DBG_MEO00 ("ls18RecordDestructor");

    delete [] DataArray;
    DataArray = NULL;
}


/*
  -----------------------------------------------------------------------------
  function:     ls18GetUnformattedValue
  -----------------------------------------------------------------------------
*/
int
ls18GetUnformattedValue(tsp00_Addr      pszLine,
                        tsp00_Int4      LineLength,
                        tsp00_Int4      lIndex,
                        char            szSep[],    // 2 byte char array
                        char            szQuot[],   // 2 byte char array
                        tls00_String*   Value)
{
    ROUTINE_DBG_MEO00 ("ls18GetUnformattedValue");
    
    tsp00_Int4 rc = errOK_els00;

    char                szFind[3];                  // Null terminated string containing ',"'
    static tsp00_Int4   lBeginPos;
    static tsp00_Int4   lFoundPos = -1;
    static tsp00_Int4   lQuotPos1 = 0;
    static tsp00_Int4   lQuotPos2 = 0;
    static tsp00_Int4   HitNo;
    static tsp00_Int4   lLength;

    if(-1 != LineLength)
    {
        lBeginPos = lQuotPos1 = lQuotPos2 = HitNo = 0;
        lFoundPos = -1;
        lLength = LineLength;
    }

    // lIndex may not be LT HitNo; we assume ascending sorting
    if (HitNo != lIndex)
    {
        sprintf(szFind, "%c%c", szSep[0], szQuot[0]);
        lQuotPos2 = 0;
        
        while ( (HitNo != lIndex) && (lBeginPos < lLength) )
        {
            lBeginPos += lFoundPos + 1;
            if (lBeginPos > lLength)    // If value is GT length of input string --> stop
            {
                break;
            }            
            // Find separator or Delimiter in pszLine
            lFoundPos = STATIC_CAST( tsp00_Int4, strcspn(&pszLine[lBeginPos], szFind) );
            
            if (szQuot[0] == pszLine[lFoundPos+lBeginPos])      // No delimiter found so far, sure!
            {
                lBeginPos += lFoundPos + 1;
                
                lFoundPos = STATIC_CAST( tsp00_Int4, strcspn(&pszLine[lBeginPos], szQuot) );   // Find 2nd delimiter
                if ((lFoundPos + lBeginPos) == lLength)
                {
                    rc = errCorruptDataFile_els98;
                    break;
                }
                
                lQuotPos1 = lBeginPos;
                lQuotPos2 = lFoundPos;
                continue;
            }
            else                // found char is no delimiter so it's a separator --> increase number of found values.
            {
                HitNo += 1;
                if (HitNo != lIndex)
                {
                    lQuotPos1 = lQuotPos2 = 0;
                }
            }
        }   // end while ( (HitNo != lIndex) && (lBeginPos < lLength) )
    }   // end if (HitNo != lIndex)

    // There are less than lIndex values in the line
    if (errOK_els00 == rc)
    {
        if (lIndex > HitNo)         
        {
            rc = errMissingData_els98;
        }
        else    // (lIndex == HitNo)
        {
            // if no value found (= the length positions are 0): Don't do anything.
            // This should be handled by the caller. The value contains a length
            // indicating to insert the default value.
            if (0 != lQuotPos2)     // Delimiters found in the value.
            {
                Value->strLeng_ls00 = lQuotPos2;
                Value->strAddr_ls00 = &pszLine[lQuotPos1];
            }
            else
            {
                if (0 != lFoundPos)
                {
                    Value->strLeng_ls00 = lFoundPos;
                    Value->strAddr_ls00 = &pszLine[lBeginPos];
                }
            }
        }
    }
    
    return rc;
}
//  ls18GetUnformattedValue()


/*
  -----------------------------------------------------------------------------
  function:     ls18GetUnformattedValueEx
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls18GetUnformattedValueEx(char          *pString,
                          tsp00_Uint4    lStrLen,
                          tls00_Pattern *pPattern,
                          int            lCount,
                          tsp00_Uint4    lPosToRead[],
                          tls00_CodeType CodeType,
                          tls00_String   pValue[],
                          int           &lMissingIndex)
{
    ROUTINE_DBG_MEO00 ("ls18GetUnformattedValueEx");

    // This function uses a pattern searching algorithm from Knuth, Morris and Pratt

    tsp00_Int4 rc = errOK_els00;
    
    tsp00_Uint4 i = 0;
    tsp00_Int4  j = 0;

    tsp00_Uint4 m = 0;  // count of found patterns
    int         n = 0;  // count of values to read (lCount) and running index for read positions

    //*
    //* Variables for checking for masked delimiters within the data
    //*
    int         k = 0;
    int         v = 0;

    char          *pStr       = pString;
    tsp00_Longint  tmpStrLen  = 0;
    bool           bLastValue = false;

    for (; ( (n < lCount) && (errOK_els00 == rc) && (false == bLastValue)); ++m)
    {
        while (i < lStrLen && j < pPattern->lpatPatternLen_ls00)
        {
            if (pStr[i] == pPattern->ppatPattern_ls00[j])
            {
                ++i;
                ++j;

                // empty values should be not considered as data including the separator
                if (j == pPattern->lpatPatternLen_ls00)
                {
                    if ( (*pPattern->ppatDelimit_ls00 == pStr[i]) &&
                         (*pPattern->ppatDelimit_ls00 == pStr[i - pPattern->lpatPatternLen_ls00 - 1]) )
                    {
                        if (0 != memcmp(pPattern->ppatPattern_ls00, &pStr[i-2*pPattern->lpatPatternLen_ls00], pPattern->lpatPatternLen_ls00))
                        {
                            j = 0;
                        }
                    }
                }
            }
            else if (0 == j)
            {
                ++i;
            }
            else
            {
                j = pPattern->plpatFailure_ls00[j-1] + 1;       // pattern failure function
            }
        }

        if (j == pPattern->lpatPatternLen_ls00)
        {
            tmpStrLen = i - pPattern->lpatPatternLen_ls00;
        }
        else
        {   // stop - no more pattern found; assign the last value
            bLastValue = true;

            if ((lPosToRead[n] - 1) > m)            // check if the read position is still within the data line
            {
                lMissingIndex = n;
                rc = errMissingData_els98;
                break;
            }
            else
            {
                tmpStrLen = i;  // no pattern found -> no need to decrease by pattern length
            }
        }

        // We have to handle the situation that the user specified the same read positions several times
        // user given positions start with 1 but we count beginning with 0
        while ( (n < lCount) && (m == (lPosToRead[n] - 1)) )
        {
            if (tmpStrLen > 0)        // leave the string undefined and string length = INSERT_EMPTY_VALUE
            {                         // if the length is 0
                pValue[n].strLeng_ls00 = STATIC_CAST(tsp00_Int4, tmpStrLen);
                pValue[n].strAddr_ls00 = pStr;      // assign the value
            }
            ++n;
        }

        pStr    += i;
        lStrLen -= i;
        j = 0;      // j must be reinitialized every time
        i = 0;
    }   // end for (; ( (n < lCount) && (errOK_els00 == rc) && (false == bLastValue)); ++m)


    // Check if we found all patterns but should have read some more values
    if (errOK_els00 == rc)
    {
        if (n < lCount)
        {
            lMissingIndex = n;
            rc = errMissingData_els98;
        }
    }

    // Some special handling for first and last value if a delimiter is defined:
    // - they may be delimited by only szDelimiter at the beginning (first value) or
    //   at the end (last value) which have to be removed; those are not found by the pattern search
    // - additionally we must make sure that there are no whitespace found before/after first/last delimiter
    if ( (errOK_els00 == rc)  && (0 != pPattern->lpatDelimiterLen_ls00) )
    {
        // first found value; if no delimiter defined here's nothing left to do
        if (NULL != pValue[0].strAddr_ls00)
        {
            tmpStrLen = pValue[0].strLeng_ls00;       // implicit conversion
            
            ls05StrTrimLeft(pValue[0].strAddr_ls00, tmpStrLen, BLANK_LS00, CodeType);

            ls18_RemoveDelimiter(pValue[0].strAddr_ls00,
                                 tmpStrLen,
                                 pPattern->ppatDelimit_ls00,
                                 pPattern->lpatDelimiterLen_ls00);
            pValue[0].strLeng_ls00 = (tmpStrLen > 0) ? STATIC_CAST(tsp00_Int4, tmpStrLen) : INSERT_EMPTY_VALUE;

            // Check if there are more fields pointing to the same (first) position and remove leading delimiter, too
            m = lPosToRead[0];
            n = 1;
            while ( (m == (lPosToRead[n])) && (n <= lCount) )
            {
                pValue[n].strAddr_ls00 = pValue[0].strAddr_ls00;
                pValue[n].strLeng_ls00 = pValue[0].strLeng_ls00;
                ++n;
            }
        }


        // last found value; make sure we found something thats worth analyzing
        n = lCount - 1;
        if (NULL != pValue[n].strAddr_ls00)
        {
            tmpStrLen = pValue[n].strLeng_ls00;
            
            ls05StrTrimRight(pValue[n].strAddr_ls00, tmpStrLen, BLANK_LS00, CodeType);

            ls18_RemoveDelimiter(pValue[n].strAddr_ls00,
                                 tmpStrLen,
                                 pPattern->ppatDelimit_ls00,
                                 pPattern->lpatDelimiterLen_ls00,
                                 false);
            pValue[n].strLeng_ls00 = (tmpStrLen > 0) ? STATIC_CAST(tsp00_Int4, tmpStrLen) : INSERT_EMPTY_VALUE;

            // Check if there are more fields pointing to the same (last) position and remove trailing delimiter, too
            m = lPosToRead[lCount-1];
            n = lCount - 2;
            while ( (m == (lPosToRead[n])) && (n >= 0) )
            {
                pValue[n].strAddr_ls00 = pValue[lCount - 1].strAddr_ls00;
                pValue[n].strLeng_ls00 = pValue[lCount - 1].strLeng_ls00;
                --n;
            }
        }
    }
    return rc;
}
// ls18GetUnformattedValueEx()


/*
  -----------------------------------------------------------------------------
  function:     ls18PutUnformattedValue
  -----------------------------------------------------------------------------
*/
tsp00_Longint
ls18PutUnformattedValue(tsp00_Addr        pszBuffer,
                        tsp00_Longint     lBufLen,
                        tsp00_Int4        lColumnCount,
                        tls00_ColumnData *pColumnData,
                        tls00_Pattern    *pPattern)
{
    ROUTINE_DBG_MEO00 ("ls18PutUnformattedValue");
    
    tsp00_Longint rc = errOK_els00;

    tsp00_Longint lBytesWritten = 0;
    tsp00_Addr    pszData       = NULL;
    tsp00_Addr    pszBufPtr     = pszBuffer;
    tsp00_Int4    Cnt           = 0;

    SAPDB_Int4           k           = 0;
    // external encoding is the same for all values
    const tsp77encoding* _pEncoding  = pEncoding_ls18[pColumnData->cdExtEncoding_ls00]; 
    SAPDB_Int4           _lCharSize  = _pEncoding->fixedCharacterSize;

    //*
    //*     Insert delimiter at the begin of the string to write
    //*
    if (0 != pPattern->lpatDelimiterLen_ls00)           // no delimiter -> nothing to insert here
    {
        memcpy(pszBufPtr, pPattern->ppatDelimit_ls00, pPattern->lpatDelimiterLen_ls00);
        lBytesWritten += pPattern->lpatDelimiterLen_ls00;
        pszBufPtr     += pPattern->lpatDelimiterLen_ls00;
    }

    //*
    //*     Write values to buffer
    //*
    for (SAPDB_Int4 i = 0; i < lColumnCount; ++i)
    {
        if (true == pColumnData[i].cdPtrUsed_ls00)
        {
            pszData = pColumnData[i].cdPtrToData_ls00;
        }
        else
        {
            pszData = pColumnData[i].cdDataBuffer_ls00;
        }

        //*
        //*     Insert value in right code type and hex if requested
        //* In case of char data
        //*  - double delimiters found in data
        //*  - write new line dependent on operating system running on
//TODOTODO BUFFERLAENGE MUSS NOCH UEBERPRUEFT WERDEN

        if ( (ioChar_els00  == pColumnData[i].cdExtDataType_ls00)  &&
             (0             != pPattern->lpatDelimiterLen_ls00)    &&
             ( (1 == szMap_CharDataType_ls00[pColumnData[i].cdSQLDataType_ls00]) &&
               (1 != szMap_BlobDataType_ls00[pColumnData[i].cdSQLDataType_ls00]) ) )
        {
            for (k = 0; (k < pColumnData[i].cdDataLen_ls00); k += _lCharSize)
            {
                if (ctUTF8_els00 == pColumnData[i].cdExtEncoding_ls00)
                {
                    _lCharSize = SAPDB_UTF8Basis::CharacterSize(REINTERPRET_CAST(SAPDB_UTF8*, &pszData[k]));
                }

                if ( 0 == memcmp(&pszData[k], pPattern->ppatDelimit_ls00, pPattern->lpatDelimiterLen_ls00) )
                {
                    memcpy(pszBufPtr, &pszData[k], pPattern->lpatDelimiterLen_ls00);
                    pszBufPtr     += pPattern->lpatDelimiterLen_ls00;
                    lBytesWritten += pPattern->lpatDelimiterLen_ls00;
                }
                else if (0 == memcmp(&pszData[k], _pEncoding->charTable->nl.bytes, _pEncoding->charTable->nl.byteCount))
                {
#if defined(_WIN32)
                    // Put a additional carriage return in for Windows
                    memcpy(pszBufPtr, _pEncoding->charTable->cr.bytes, _pEncoding->charTable->cr.byteCount);
                    pszBufPtr     += _pEncoding->charTable->cr.byteCount;
                    lBytesWritten += _pEncoding->charTable->cr.byteCount;
#endif
                }
                memcpy(pszBufPtr, &pszData[k], _lCharSize);
                pszBufPtr += _lCharSize;
            }
            lBytesWritten += pColumnData[i].cdDataLen_ls00;
        }
        else
        {
            memcpy(pszBufPtr, pszData, pColumnData[i].cdDataLen_ls00);

            lBytesWritten += pColumnData[i].cdDataLen_ls00;
            pszBufPtr     += pColumnData[i].cdDataLen_ls00;
        }

        //*
        //*     Insert value separator(pattern)
        //*
        if (i < (lColumnCount - 1) )
        {
            memcpy(pszBufPtr, pPattern->ppatPattern_ls00, pPattern->lpatPatternLen_ls00);
            lBytesWritten += pPattern->lpatPatternLen_ls00;
            pszBufPtr     += pPattern->lpatPatternLen_ls00;
        }
    }   // end for (int i = 0; i < lColumnCount; ++i)

    //*
    //*     Insert delimiter at the end of the data record
    //*
    if (0 != pPattern->lpatDelimiterLen_ls00)       // no delimiter -> nothing to insert
    {
        memcpy(pszBufPtr, pPattern->ppatDelimit_ls00, pPattern->lpatDelimiterLen_ls00);

        lBytesWritten += pPattern->lpatDelimiterLen_ls00;
    }
    return lBytesWritten;
}
//  ls18PutUnformattedValue()

/*
  -----------------------------------------------------------------------------
  function:     ls18GetCompactValue
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls18GetCompactValue(char              *pString,
                    tls00_MetaData    *pMeta,
                    tsp00_Uint4        lCount,
                    tsp00_Uint4        lPosToRead[],
                    tls00_String       pValue[],
                    tls00_MultColSpec *pMultCol,
                    int               &lMissingIndex)
{
    ROUTINE_DBG_MEO00 ("ls18GetCompactValue");

    tsp00_Int4 rc = errOK_els00;

    tsp00_Uint4 i         = 0;
    size_t      lOffset   = 0;          // defines start of string values
    tsp00_Int4  len       = 0;          // used for handling of long value lengths
    tsp00_Uint4 lMaxPos   = lPosToRead[lCount-1];
    tsp00_Uint4 i_longpos = 0;          // position index of long column pos
    tsp00_Uint4 n_longpos = 0;          // new position index of long column pos used within loop
    tsp00_Uint4 i_pospos  = 0;          // index of positions to read
    
    tls00_ParamInfo_LONG *pLCI    = NULL;
    bool                  bIsLong = false;


    for (i; ( (i < lMaxPos) && (errOK_els00 == rc) ); ++i)
    {
        if (i > pMeta->mdFieldCount_ls00)
        {   // trying to read data fields with position numbers higher
            // than count of fields in data file must fail
            lMissingIndex = i;
            rc = errMissingData_els98;
        }
        else
        {
            // Assign length to value length member; be careful with longs
            if ( 0         != pMeta->mdLongFieldCount_ls00 &&
                 i_longpos <  pMeta->mdLongFieldCount_ls00  )   // http://pts:1080/webpts?wptsdetail=yes&ErrorType=0&ErrorID=1124387
            {
                if ((i+1) == pMeta->pmdLongFieldPos_ls00[i_longpos])
                {
                    // Attention: long length is built using 2 entries of pmdLengthArray_ls00
                    //            to be able to represent values gt 32767 bytes
                    memcpy(&len, &pMeta->pmdLengthArray_ls00[i + i_longpos], sizeof(tsp00_Int4));
                    bIsLong = true;
                }
                else
                {
                    len = pMeta->pmdLengthArray_ls00[i + i_longpos];
                }
            }
            else
            {
                len = pMeta->pmdLengthArray_ls00[i + i_longpos];
            }

            if ((i+1) == lPosToRead[i_pospos])
            {
                pValue[i].strLeng_ls00 = ( (0 == len) && (bIsLong == false) ) ? INSERT_EMPTY_VALUE : len;

                if (bIsLong == false)
                {
                    if ( (INSERT_NULL != pValue[i].strLeng_ls00) && (INSERT_EMPTY_VALUE != pValue[i].strLeng_ls00) ) 
                    {
                        pValue[i].strAddr_ls00 = &pString[lOffset];
                    }
                }
                else
                {
                    // Adapt long file info for this column!
                    // index i corresponds to index in multcol structure
                    pLCI = &pMultCol->mcsColumn_ls00[i]->colLongInfo;

                    if (INSERT_NULL != pValue[i].strLeng_ls00)
                    {
                        pLCI->loiLength_ls00          = len;
                        pLCI->loiPart_ls00            = true;   // Long value is part of the data file
                        pLCI->loiCountOfSendData_ls00 = 0L;     // count of already send data (to kernel)
                    }

                    if (0 == i_longpos)
                    {
                        pLCI->loiFileOffset_ls00 = pLCI->loiFileStruct_ls00->vfFilePos_ls00;
                    }
                    else
                    {
                        n_longpos = pMeta->pmdLongFieldPos_ls00[i_longpos - 1] - 1;

                        pLCI->loiFileOffset_ls00 =
                                pMultCol->mcsColumn_ls00[n_longpos]->colLongInfo.loiFileOffset_ls00;
                        if (INSERT_NULL != pValue[n_longpos].strLeng_ls00)
                        {
                            ls07AddLargeNumbers(pLCI->loiFileOffset_ls00, pValue[n_longpos].strLeng_ls00);
                        }
                    }
                }

                ++i_pospos;
            }

            if ( (pValue[i].strLeng_ls00 > 0) && (bIsLong == false) )
            {
                lOffset += pValue[i].strLeng_ls00;
            }
            else if (true == bIsLong)
            {
                ++i_longpos;
                bIsLong = false;
            }
        }
    }   // end for (i; ( (i < lMaxPos) && (errOK_els00 == rc) ); ++i)

    return rc;
}
// ls18GetCompactValue()


/*
  -----------------------------------------------------------------------------
  function:     ls18PutCompactValue
  -----------------------------------------------------------------------------
*/
tsp00_Longint
ls18PutCompactValue(tls00_Buffer2     *pszMeta,
                    tsp00_Addr         pszBuffer,
                    tsp00_Longint     lBufLen,
                    tls00_ColumnData  *pColumnData,
                    tls00_MultColSpec *pColumnDesc)
{
    ROUTINE_DBG_MEO00 ("ls18PutCompactValue");

    tsp00_Addr    pszData = NULL;
    tsp00_Int4    vlen    = 0;      // value length = 4 Bytes if LONGs and 2 Bytes if not LONGs
    tsp00_Int2    auxlen  = 0;      // used to write value length to meta buffer
    tsp00_Longint rlen    = 0;      // record length


    for (tsp00_Int i = 0; i < pColumnDesc->mcsCount_ls00; ++i, vlen = 0)
    {
        if (true == pColumnData[i].cdPtrUsed_ls00)
        {
            pszData = pColumnData[i].cdPtrToData_ls00;
        }
        else
        {
            pszData = pColumnData[i].cdDataBuffer_ls00;
        }


        if (true == pColumnData[i].cdIsNILValue_ls00)
        {
            // No value to write to buffer; length of INSERT_NULL indicates NULL value
            vlen = INSERT_NULL;
        }
        else
        {
            if (1 == szMap_LongDataType_ls00[pColumnData[i].cdSQLDataType_ls00])
            {
                vlen = STATIC_CAST(tsp00_Int4, pColumnDesc->mcsColumn_ls00[i]->colLongInfo.loiLength_ls00);
            }
            else
            {
                vlen = STATIC_CAST(tsp00_Int4, pColumnData[i].cdDataLen_ls00);
                memcpy(pszBuffer, pszData, vlen);

                // Adapt record buffer values
                rlen      += vlen;
                pszBuffer += vlen;
            }
        }

        if (1 == szMap_LongDataType_ls00[pColumnData[i].cdSQLDataType_ls00])
        {
            memcpy(&(STATIC_CAST(_TCHAR*, pszMeta->pBufPtr_ls00)[pszMeta->lBufPos_ls00]), &vlen, sizeof(tsp00_Int4));
            pszMeta->lBufPos_ls00 += sizeof(tsp00_Int4);
        }
        else
        {
            auxlen = STATIC_CAST(tsp00_Int2, vlen);
            memcpy(&(STATIC_CAST(_TCHAR*, pszMeta->pBufPtr_ls00)[pszMeta->lBufPos_ls00]), &auxlen, sizeof(tsp00_Int2));
            pszMeta->lBufPos_ls00 += sizeof(tsp00_Int2);
        }
    }

    return rlen;
}
// ls18PutCompactValue()


/*
  -----------------------------------------------------------------------------
   function:     ls18GetFormattedValue
  -----------------------------------------------------------------------------
*/
int
ls18GetFormattedValue(tsp00_Addr    pszLine,
                      tsp00_Int4    LineLength,
                      tsp00_Int2&   lStartPos,
                      tsp00_Int2&   lEndPos,
                      tls00_String* Value)
{
    ROUTINE_DBG_MEO00 ("ls18GetFormattedValue");
    
    // The start position starts at 1 in the user file. So we have to subtract 1 from
    // the start position to be on the right position, because C starts counting with 0.
    // In case the StartPos exceeds LineLength simply return. The caller has 
    // to handle this case.

    int rc = errOK_els00;
    int MyStartPos = lStartPos - 1;

    if (MyStartPos <= LineLength)
    {
        // If lEndPos is 0 the length of the value evaluates to 1.
        if (0 == lEndPos)
        {
            Value->strLeng_ls00 = 1;
        }
        else
        {
            Value->strLeng_ls00 = (lEndPos > LineLength) ? (LineLength - MyStartPos) : (lEndPos - MyStartPos);
        }
        Value->strAddr_ls00 = &pszLine[MyStartPos];
    }
    else
    {
        rc = errMissingData_els98;
    }

    return rc;
}
//  ls18GetFormattedValue()

                      
/*
  -----------------------------------------------------------------------------
   function:     ls18GetFormattedValueEx
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls18GetFormattedValueEx(tsp00_Addr         pszLine,
                        tsp00_Int4         LineLength,
                        tls00_MultColSpec *pMultCol,
                        tsp00_Uint4        lReadFieldsCnt,
                        tsp00_Uint4        lValToColIndex[],
                        tls00_String       pValue[],
                        tsp00_Int4        &lMissingIndex)
{
    ROUTINE_DBG_MEO00 ("ls18GetFormattedValueEx");
    
    // The start position starts at 1 in the user file. So we have to subtract 1 from
    // the start position to be on the right position, because C starts counting with 0.
    // In case the StartPos exceeds LineLength simply return. The caller has 
    // to handle this case.

    int rc = errOK_els00;
    int spos = 0;
    int epos = 0;

    tls00_Column *pCol = NULL;

    for (tsp00_Uint4 i=0; ( (errOK_els00 == rc) && (i < lReadFieldsCnt) ); ++i)
    {
        pCol = pMultCol->mcsColumn_ls00[lValToColIndex[i]];

        spos = pCol->colFPos_ls00.fpoStart_ls00 - 1;
        epos = pCol->colFPos_ls00.fpoEnd_ls00;
        if (spos <= LineLength)
        {
            // If lEndPos is 0 the length of the value evaluates to 1.
            if (0 == epos)
            {
                pValue[i].strLeng_ls00 = 1;
            }
            else
            {
                pValue[i].strLeng_ls00 = (epos > LineLength) ? (LineLength - spos) : (epos - spos);
            }
            pValue[i].strAddr_ls00 = &pszLine[spos];
        }
        else
        {
            rc = errMissingData_els98;
            lMissingIndex = i;
        }
    }

    return rc;
}
// ls18GetFormattedValueEx()



// Kann auch binaere daten ausgeben
tsp00_Longint
ls18PutFormattedValue(tsp00_Addr         pszBuffer,        // zeigt auf pszBuffer[StartPos]
                      tsp00_Longint      lBufLen,          // definiert die zulaessige laenge EndPos - StartPos + 1
                      tls00_ColumnData  *pColumnData, //tsp00_Addr      pszData,
                      tls00_MultColSpec *pColumnDesc)
                      
{
    ROUTINE_DBG_MEO00 ("ls18PutFormattedValue");

    tsp00_Addr           pszData  = NULL;
    tsp00_Longint        spos     = 0;      // starting position of single data value in record built here
    tsp00_Longint        vlen     = 0;      // length of single data value
    tsp00_Longint        rlen     = 0;      // overall length of record built here
    tsp00_SwapKind_Enum  CurrSwap = ls07DefineSwap(NULL);

    // In case the file format is FORMATTED we initialize the output buffer will fill
    // bytes to prevent later (time consuming) filling for each value.
    tsp00_DataType tmpDataType;
    tmpDataType.becomes(dcha);

    ls05InsertFillBytes(REINTERPRET_CAST(unsigned char*, pszBuffer),
                        0,
                        tmpDataType,
                        lBufLen,
                        (ctIgnore_els00 < pColumnData->cdExtEncoding_ls00),
                        CurrSwap);


    for (tsp00_Int i = 0; i < pColumnDesc->mcsCount_ls00; ++i)
    {
        if (true == pColumnData[i].cdPtrUsed_ls00)
        {
            pszData = pColumnData[i].cdPtrToData_ls00;
        }
        else
        {
            pszData = pColumnData[i].cdDataBuffer_ls00;
        }

        //*
        //*     Define starting position to write in buffer
        //*
        //* Take into account that strings have to be written left-justified
        //* but numbers right-justified
        //* Additionally define the maximum allowed length for this single value
        //*
        spos = pColumnDesc->mcsColumn_ls00[i]->colFPos_ls00.fpoStart_ls00 - 1;
        vlen = pColumnDesc->mcsColumn_ls00[i]->colFPos_ls00.fpoEnd_ls00 - spos;

        if ( (1 == szMap_NumberDataType_ls00[pColumnData[i].cdSQLDataType_ls00]) &&
             (false == pColumnData[i].cdIsNILValue_ls00)                          )
        {
            spos = pColumnDesc->mcsColumn_ls00[i]->colFPos_ls00.fpoEnd_ls00 - pColumnData[i].cdDataLen_ls00;
        }

        //*
        //*     Insert value
        //*
        if (pColumnData[i].cdDataLen_ls00 < vlen)
        {
            memcpy(&pszBuffer[spos], pszData, pColumnData[i].cdDataLen_ls00);

            // In case of binary fields (like truly hexadecimal) fill the remaining
            // Bytes of the field with null
            if ( (1     == szMap_BlobDataType_ls00[pColumnData[i].cdSQLDataType_ls00]) &&
                 (0     == szMap_LongDataType_ls00[pColumnData[i].cdSQLDataType_ls00]) &&
                 (false == pColumnData[i].cdExtHexRepresentation_ls00)                  )
            {
                tmpDataType.becomes(dfixed);

                ls05InsertFillBytes(REINTERPRET_CAST(unsigned char*, &pszBuffer[spos]),
                                    pColumnData[i].cdDataLen_ls00,
                                    tmpDataType,
                                    vlen,
                                    (ctAscii_els00 != pColumnData->cdExtEncoding_ls00),
                                    CurrSwap);
            }
        }
        else
        {
            memcpy(&pszBuffer[spos], pszData, vlen);
        }

        if (rlen < pColumnDesc->mcsColumn_ls00[i]->colFPos_ls00.fpoEnd_ls00 )
        {
            rlen = pColumnDesc->mcsColumn_ls00[i]->colFPos_ls00.fpoEnd_ls00;
        }
    }   // end for (int i = 0; i < lColumnCount; ++i)

    return rlen;
}
// ls18PutFormattedValue()


/*
  -----------------------------------------------------------------------------
  function:     ls18BindDataToCol
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls18BindDataToCol(tsp00_Addr              pszBuffer,
                   tsp00_Int4              BufLength,
                   tls00_TableDescription *pTableDesc,
                   tls00_ColumnData        ColData[],
                   tsp00_Int4              ColCount,
                   bool                    bUnicodeOI,
                   tsp00_Addr              pszErrText)
{
    ROUTINE_DBG_MEO00 ("ls18BindDataToCol");
    
    tsp00_Longint rc = errOK_els00;

    // The start position starts at 1 in the packet buffer. So we have to subtract 1 from
    // the start position to be on the right position.
    // In case the StartPos exceeds LineLength simply return. The caller has 
    // to handle this case.

    // InOutLength includes the def-byte, too.
    // Because numbers are always given binary the value buffer is not NULL terminated
    tsp00_Int4       undef_pos = 0;
    tls00_ParamInfo *pPI       = NULL;

    for (tsp00_Int4 i = 0; ( (i < ColCount) && (errOK_els00 == rc) ); ++i)
    {
        //*
        //*     Define parameter description position in table description structure
        //*
        pPI = pTableDesc->tdParamArray_ls00[ColData[i].cdPosInStmt_ls00];

        undef_pos = pPI->piBufpos_ls00 - 1;
        if (undef_pos <= BufLength)             // Check if position is still in delivered buffer
        {
            if ( UNDEF_BYTE_LS00 == STATIC_CAST(unsigned char, pszBuffer[undef_pos]) )
            {
                //*
                //*     NULL value found
                //*
                ColData[i].cdDataLen_ls00 = INSERT_NULL;
            }
            else
            {
                //*
                //*     REAL value found
                //*
                //* In case of long values we find only the descriptor at this place. The long descriptor
                //* MUST not be further processed here but simply stored for later processing.
                //*

                ColData[i].cdDataLen_ls00   = ColData[i].cdSQLLen_ls00 - 1;
                ColData[i].cdPtrToData_ls00 = &pszBuffer[pPI->piBufpos_ls00];
                ColData[i].cdPtrUsed_ls00   = true;

                //*
                //*     FIRST TRANSFORMATION of delivered value: bring it into 'internal'
                //*     encoding (ascii so far) to process it if the order interface used
                //*     is UNICODE (i.e. UCS2).
                //*
                //* Transformation to 'internal' encoding for values of type
                //*  - date, time, timestamp (value must be brought into the right format and
                //*    transformation routines may only handle ascii data)
                //* No transformation for values of all other types because
                //*  - boolean - delivered as binary 0 or 1 and thus not depending on encoding
                //*  - long - real long values gotten later when processing the descriptor
                //*  - numeric - delivered in internal (db number) representation
                //*  - char/varchar - transformed later with regards to the external representation

                if ( (true == bUnicodeOI)                                                           &&
                     ( (1        == szMap_SpecConstDataType_ls00[ColData[i].cdSQLDataType_ls00]) &&
                       (dboolean != ColData[i].cdSQLDataType_ls00)                                )  )
                {
                    // to indicate in the function call the length of the available buffer for the data
                    ColData[i].cdDataLen_ls00 = ColData[i].cdDataBufferLen_ls00;
                    rc = ls05ConvertFromUCS2(REINTERPRET_CAST(unsigned char*, ColData[i].cdDataBuffer_ls00),
                                             ColData[i].cdDataLen_ls00,
                                             ctAscii_els00,
                                             REINTERPRET_CAST(unsigned char*, ColData[i].cdPtrToData_ls00),
                                             ColData[i].cdSQLLen_ls00 - 1,
                                             ColData[i].cdSQLDataType_ls00,
                                             pszErrText);
                    if (errOK_els00 == rc)
                    {
                        ColData[i].cdPtrUsed_ls00 = false;
                    }
                }
            }

        }   // end if (undef_pos <= BufLength)

    }   // end for (i; ( (i < ColCount) && (errOK_els00 == rc) ); ++i)
    
    return STATIC_CAST(tsp00_Int4, rc);
}
//  ls18BindDataToCol()


/*
  -----------------------------------------------------------------------------
  function:     ls18ConstructPattern
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls18ConstructPattern(tls00_Pattern  *pPattern,
                     tsp00_Addr      pszSeparator,
                     tsp00_Addr      pszDelimiter,
                     tls00_CodeType  ExtEncoding,
                     tsp00_Addr      pszErrText)
{
    ROUTINE_DBG_MEO00 ("ls18ConstructPattern");

    tsp00_Longint rc = errOK_els00;
    // pattern is build this way 'szDelimiter_szSeparator_szDelimiter';
    // example could be ","

    tsp00_Int4 lDelLen = STATIC_CAST(tsp00_Int4, _tcslen(pszDelimiter));
    tsp00_Int4 lPatLen = STATIC_CAST(tsp00_Int4,_tcslen(pszSeparator)) + 2 * lDelLen;


    pPattern->lpatDelimiterLen_ls00 = lDelLen;  // fits for ascii, too
    pPattern->lpatPatternLen_ls00   = lPatLen;  // fits for ascii, too

    if ( (ctUCS2_els00 == ExtEncoding) || (ctUCS2Swapped_els00 == ExtEncoding))
    {
        pPattern->lpatPatternLen_ls00   = 2*lPatLen;
        pPattern->lpatDelimiterLen_ls00 = 2*lDelLen;
    }
    else if (ctUTF8_els00 == ExtEncoding)
    {
        pPattern->lpatPatternLen_ls00   = 6*lPatLen;        // to be safe
        pPattern->lpatDelimiterLen_ls00 = 6*lDelLen;        // to be safe
    }


    pPattern->ppatPattern_ls00 = new char[pPattern->lpatPatternLen_ls00 + 1];   // terminating 0; makes no sense for ucs2
    pPattern->ppatDelimit_ls00 = new char[pPattern->lpatDelimiterLen_ls00 + 1]; // in case of length 0 only to be sure not working with a null ptr

    //*
    //*     Initialize pattern and delimiter to ascii encoding
    //*
    if (0 != pPattern->lpatDelimiterLen_ls00)
    {
        pPattern->ppatPattern_ls00[0] = pszDelimiter[0];
        pPattern->ppatPattern_ls00[1] = pszSeparator[0];
        pPattern->ppatPattern_ls00[2] = pszDelimiter[0];
        pPattern->ppatPattern_ls00[3] = 0;

        pPattern->ppatDelimit_ls00[0] = pszDelimiter[0];
        pPattern->ppatDelimit_ls00[1] = 0;
    }
    else
    {
        pPattern->ppatPattern_ls00[0] = pszSeparator[0];
        pPattern->ppatPattern_ls00[1] = 0;

        pPattern->ppatDelimit_ls00[0] = 0;
    }

    //*
    //*     Convert to UCS2 or UTF8 if necessary
    //*
    if ( (ctUCS2_els00 == ExtEncoding) || (ctUCS2Swapped_els00 == ExtEncoding))
    {
        tsp00_Longuint lDestLen = 0;

        tsp00_Addr pszTmp = new _TCHAR[lPatLen + 1];
        memcpy(pszTmp, pPattern->ppatPattern_ls00, lPatLen);
        pszTmp[lPatLen] = 0;

        lDestLen = pPattern->lpatPatternLen_ls00;
        rc = ls05ConvertToUCS2Simple(REINTERPRET_CAST(unsigned char*, pPattern->ppatPattern_ls00),
                                     lDestLen,
                                     REINTERPRET_CAST(unsigned char*, pszTmp),
                                     lPatLen,
                                     ctAscii_els00,
                                     pszErrText);
        if (errOK_els00 == rc)
        {
            pPattern->lpatPatternLen_ls00 = STATIC_CAST(tsp00_Longint, lDestLen);

            if (0 != pPattern->lpatDelimiterLen_ls00)
            {
                // Reuse of lDestLen!
                lDestLen = pPattern->lpatDelimiterLen_ls00;
            
                rc = ls05ConvertToUCS2Simple(REINTERPRET_CAST(unsigned char*, pPattern->ppatDelimit_ls00),
                                             lDestLen,
                                             REINTERPRET_CAST(unsigned char*, pszDelimiter),
                                             lDelLen,
                                             ctAscii_els00,
                                             pszErrText);
                if (errOK_els00 == rc)
                {
                    pPattern->lpatDelimiterLen_ls00 = STATIC_CAST(tsp00_Longint, lDestLen);
                }
            }
        }

        if (NULL != pszTmp)
        {
            delete [] pszTmp;
            pszTmp = NULL;
        }
    }
    else if (ctUTF8_els00 == ExtEncoding)
    {
        Tools_DynamicUTF8String Tmp;

        rc = ls05ConvertToUTF8(Tmp,
                               REINTERPRET_CAST(unsigned char*, pPattern->ppatPattern_ls00),
                               lPatLen,
                               ctAscii_els00,
                               pszErrText);
        if (errOK_els00 == rc)
        {
            pPattern->lpatPatternLen_ls00 = STATIC_CAST(tsp00_Longint, Tmp.BasisSize());
            memcpy(pPattern->ppatPattern_ls00, Tmp.StrPtr(), pPattern->lpatPatternLen_ls00);
            pPattern->ppatPattern_ls00[pPattern->lpatPatternLen_ls00] = 0;

            if (0 != pPattern->lpatDelimiterLen_ls00)
            {
                // Reuse of Tmp!
                rc = ls05ConvertToUTF8(Tmp,
                                       REINTERPRET_CAST(unsigned char*, pszDelimiter),
                                       lDelLen,
                                       ctAscii_els00,
                                       pszErrText);
                if (errOK_els00 == rc)
                {
                    pPattern->lpatDelimiterLen_ls00 = STATIC_CAST(tsp00_Longint, Tmp.BasisSize());
                    memcpy(pPattern->ppatDelimit_ls00, Tmp.StrPtr(), pPattern->lpatDelimiterLen_ls00);
                    pPattern->ppatDelimit_ls00[pPattern->lpatDelimiterLen_ls00] = 0;
                }
            }
        }
    }
    //else      // supposing ascii; it's done with initialization of pattern


    //*
    //*     Build pattern fail function
    //*
    if (errOK_els00 == rc)
    {
        pPattern->plpatFailure_ls00 = new tsp00_Int4[pPattern->lpatPatternLen_ls00];

        ls18ComputePatFailFunc(pPattern);
    }
    return STATIC_CAST(tsp00_Int4, rc);
}
// ls18ConstructPattern()


/*
  -----------------------------------------------------------------------------
  function:     ls18ComputePatFailFunc
  -----------------------------------------------------------------------------
*/
void
ls18ComputePatFailFunc(tls00_Pattern *pPattern)
{
    ROUTINE_DBG_MEO00 ("ls18ComputePatFailFunc");

    // Now compute pattern failure function
    tsp00_Addr  pPat   = pPattern->ppatPattern_ls00;
    tsp00_Int4 *plFail = pPattern->plpatFailure_ls00;
    
    plFail[0] = -1;
    tsp00_Int4 i = 0;
    int        j = 1;

    for(j; j < pPattern->lpatPatternLen_ls00; ++j)
    {
        i = plFail[j-1];
        while ( (pPat[j] != pPat[i+1]) && (i >= 0) )
        {
            i = plFail[i];
        }

        if (pPat[j] == pPat[i+1])
        {
            plFail[j] = i+1;
        }
        else
        {
            plFail[j] = -1;
        }
    }

    return;
}
// ls18ComputePatFailFunc()


/*
  -----------------------------------------------------------------------------
  function:     ls18DestructPattern
  -----------------------------------------------------------------------------
*/
void
ls18DestructPattern(tls00_Pattern *pPattern)
{
    ROUTINE_DBG_MEO00 ("ls18DestructPattern");

    if (NULL != pPattern)
    {
        if (NULL != pPattern->ppatPattern_ls00)
        {
            delete [] pPattern->ppatPattern_ls00;
            pPattern->ppatPattern_ls00 = NULL;
        }

        if (NULL != pPattern->ppatDelimit_ls00)
        {
            delete [] pPattern->ppatDelimit_ls00;
            pPattern->ppatDelimit_ls00 = NULL;
        }
        
        if (NULL != pPattern->plpatFailure_ls00)
        {
            delete [] pPattern->plpatFailure_ls00;
            pPattern->plpatFailure_ls00 = NULL;
        }
    }

    return;
}
// ls18DestructPattern()

/*
  -----------------------------------------------------------------------------
  function:     ls18UndoubleDelimitersInCharData
  -----------------------------------------------------------------------------
*/
void
ls18UndoubleDelimitersInCharData(tls00_String*   pValue,
                                 tls00_Pattern*  pPattern,
                                 tls00_CodeType  CodeType)
{
    //*
    //*     Check for delimiters and 'un'double them - this should be of any
    //*     interest only for data meant for char columns of any encoding
    //*
    if (0 != pPattern->lpatDelimiterLen_ls00) 
    {
        SAPDB_UInt  _lCharSize     = sizeof(char);              // initialize for ASCII
        SAPDB_UInt  v              = 0;
        SAPDB_UInt  k              = 0;
        SAPDB_UInt  _lBufLen       = pValue->strLeng_ls00;
        char*       _pszWrkBuffer  = pValue->strAddr_ls00;   // Coming here pValue->strAddr_ls00 is still a ptr into
                                                        // the line read from data file; in case of found delimiters
                                                        // within the data the function will allocate memory for
                                                        // pValue->strAddr_ls00
        if (ctUTF8_els00 != CodeType)
        {
            if (CodeType > ctIgnore_els00)
            {
                _lCharSize = sp77encodingUCS2->fixedCharacterSize;
            }

            // step size is 2 chars
            for (k = _lCharSize; (k < _lBufLen); k += 2*_lCharSize)        // find first delimter
            {
                if (0 == memcmp(&_pszWrkBuffer[k], pPattern->ppatDelimit_ls00, pPattern->lpatDelimiterLen_ls00))
                {
                    if (0 == memcmp(&_pszWrkBuffer[k - _lCharSize], pPattern->ppatDelimit_ls00, pPattern->lpatDelimiterLen_ls00))
                    {
                        v = k;
                        k += _lCharSize;
                    }
                    else if ((k + _lCharSize) <= _lBufLen)
                    {
                        if (0 == memcmp(&_pszWrkBuffer[k + _lCharSize], pPattern->ppatDelimit_ls00, pPattern->lpatDelimiterLen_ls00))
                        {
                            v  = k + _lCharSize;                  // initialize for next for loop
                            k += 2 * _lCharSize;                  // skip 2nd delimiter
                        }

                    }   // end if ((k+1) <= _lBufLen)

                    if (0 != v)
                    {
                        // As we have found a doubled delimiter we need to allocate space for
                        // this value to copy as the origin (the read line from data file) shouldn't be changed
                        pValue->strAddr_ls00 = new char[pValue->strLeng_ls00];
                        memcpy(pValue->strAddr_ls00, _pszWrkBuffer, v);
                        pValue->strLeng_ls00 -= _lCharSize;
                        break;
                    }
                }   // end if (*pPattern->ppatDelimit_ls00 == _pszWrkBuffer[k])
            }   // end for (k=1; (k < _lBufLen); k+=2)

            // If there was a first delimiter we have to copy the rest, too
            while (k < _lBufLen)
            {
                memcpy(&pValue->strAddr_ls00[v], &_pszWrkBuffer[k], _lCharSize);
                
                if (0 == memcmp(&_pszWrkBuffer[k], pPattern->ppatDelimit_ls00, pPattern->lpatDelimiterLen_ls00))
                {
                    if ( (k + _lCharSize) <= _lBufLen)
                    {
                        if (0 == memcmp(&_pszWrkBuffer[k + _lCharSize], pPattern->ppatDelimit_ls00, pPattern->lpatDelimiterLen_ls00))
                        {
                            pValue->strLeng_ls00 -= _lCharSize;
                            k                    += _lCharSize;    // skip 2nd delimiter
                        }
                    }
                }
                k += _lCharSize;
                v += _lCharSize;
            }   // end for (k, v; k < _lBufLen; ++k, ++v)

        }
        else    //if (ctUTF8_els00 == CodeType)
        {
            _lCharSize = SAPDB_UTF8Basis::CharacterSize(REINTERPRET_CAST(SAPDB_UTF8*, _pszWrkBuffer));

            // step size is 2 chars
            k = _lCharSize; 
            while (k < _lBufLen)
            {
                if (0 == memcmp(&_pszWrkBuffer[k], pPattern->ppatDelimit_ls00, pPattern->lpatDelimiterLen_ls00))
                {
                    if (0 == memcmp(&_pszWrkBuffer[k - _lCharSize], pPattern->ppatDelimit_ls00, pPattern->lpatDelimiterLen_ls00))
                    {
                        v  = k;                // initialize for next for loop
                        k += pPattern->lpatDelimiterLen_ls00;       // skip 2nd (just evaluated char) delimiter
                    }
                    else
                    {
                        k += pPattern->lpatDelimiterLen_ls00;
                        if (k < _lBufLen)
                        {
                            if (0 == memcmp(&_pszWrkBuffer[k], pPattern->ppatDelimit_ls00, pPattern->lpatDelimiterLen_ls00))
                            {
                                v  = k;      // initialize for next for loop
                                k += pPattern->lpatDelimiterLen_ls00;       // skip 2nd (just evaluated char) delimiter
                            }
                        }
                    }   // end if ((k+1) <= _lBufLen)
                    
                    if (0 != v)
                    {
                        // As we have found a doubled delimiter we need to allocate space for
                        // this value to copy as the origin (the read line from data file) shouldn't be changed
                        pValue->strAddr_ls00 = new char[pValue->strLeng_ls00];
                        memcpy(pValue->strAddr_ls00, _pszWrkBuffer, v);
                        pValue->strLeng_ls00 -= pPattern->lpatDelimiterLen_ls00;
                        break;
                    }
                    else
                    {
                        _lCharSize  = SAPDB_UTF8Basis::CharacterSize(REINTERPRET_CAST(SAPDB_UTF8*, &_pszWrkBuffer[k]));
                        k          += _lCharSize;
                    }
                }   // end if (0 == memcmp(&_pszWrkBuffer[k], pPattern->ppatDelimit_ls00, pPattern->lpatDelimiterLen_ls00))
                else
                {
                    _lCharSize = SAPDB_UTF8Basis::CharacterSize(REINTERPRET_CAST(SAPDB_UTF8*, &_pszWrkBuffer[k]));
                    k += _lCharSize;
                    _lCharSize = SAPDB_UTF8Basis::CharacterSize(REINTERPRET_CAST(SAPDB_UTF8*, &_pszWrkBuffer[k]));
                    k += _lCharSize;
                }
            }   // end for (k=1; (k < _lBufLen); k+=2)

            // If there was a first delimiter we have to copy the rest, too
            while (k < _lBufLen)
            {
                _lCharSize = SAPDB_UTF8Basis::CharacterSize(REINTERPRET_CAST(SAPDB_UTF8*, &_pszWrkBuffer[k]));

                memcpy(&pValue->strAddr_ls00[v], &_pszWrkBuffer[k], _lCharSize);

                v += _lCharSize;

                if (0 == memcmp(&_pszWrkBuffer[k], pPattern->ppatDelimit_ls00, pPattern->lpatDelimiterLen_ls00))
                {
                    k += pPattern->lpatDelimiterLen_ls00;

                    if (k < _lBufLen)
                    {
                        if (0 == memcmp(&_pszWrkBuffer[k], pPattern->ppatDelimit_ls00, pPattern->lpatDelimiterLen_ls00))
                        {
                            pValue->strLeng_ls00 -= pPattern->lpatDelimiterLen_ls00;
                            k                    += pPattern->lpatDelimiterLen_ls00;    // skip 2nd delimiter
                        }
                    }
                }
                else
                {
                    k += _lCharSize;
                }
            }   // end while (k < _lBufLen)


/*          This is an alternative implementation for UTF8 using the UTF8 string class; the shortcoming is
            only: it is way too slow!

            Tools_DynamicUTF8String _delimUTF8;
            _delimUTF8.AssignRaw(REINTERPRET_CAST(unsigned char*, pPattern->ppatDelimit_ls00), pPattern->lpatDelimiterLen_ls00);


            Tools_DynamicUTF8String srcUTF8;
            srcUTF8.AssignRaw(REINTERPRET_CAST(unsigned char*, _pszWrkBuffer), _lBufLen);

            Tools_DynamicUTF8String destUTF8;

            SAPDB_UInt4 size = srcUTF8.Size();

            // step size is 2 chars
            for (k = 1; (k < size); k += 2)        // find first delimter
            {
                if (0 == _delimUTF8.Compare(srcUTF8.SubStrElem(k, 1)))
                {
                    if (0 == _delimUTF8.Compare(srcUTF8.SubStrElem(k-1, 1)))
                    {
                        v = k;
                        k += 1;
                    }
                    else if ((k + 1) <= size)
                    {
                        if (0 == _delimUTF8.Compare(srcUTF8.SubStrElem(k+1, 1)))
                        {
                            v  = k + 1;                  // initialize for next for loop
                            k += 2;                  // skip 2nd delimiter
                        }

                    }   // end if ((k+1) <= _lBufLen)

                    if (0 != v)
                    {
                        // As we have found a doubled delimiter we need to allocate space for
                        // this value to copy as the origin (the read line from data file) shouldn't be changed
                        pValue->strAddr_ls00 = new char[pValue->strLeng_ls00];

                        destUTF8.Append(srcUTF8, 0, v);
                        pValue->strLeng_ls00 -= _delimUTF8.Size();
                        break;
                    }
                }   // end if (*pPattern->ppatDelimit_ls00 == _pszWrkBuffer[k])
            }   // end for (k=1; (k < _lBufLen); k+=2)



            for (k, v; k < size; ++k, ++v)
            {
                destUTF8.Append(srcUTF8.GetIteratorAt(k), srcUTF8.GetIteratorAt(k+1));
                
                if (0 == _delimUTF8.Compare(srcUTF8.SubStrElem(k, 1)))
                {
                    if ( (k + 1) <= size)
                    {
                        if (0 == _delimUTF8.Compare(srcUTF8.SubStrElem(k+1, 1)))
                        {
                            //pValue->strLeng_ls00 -= 1;
                            ++k;    // skip 2nd delimiter
                        }
                    }
                }
            }   // end for (k, v; k < _lBufLen; ++k, ++v)

            if (0 != v)
            {
                pValue->strLeng_ls00 = destUTF8.BasisSize();
                memcpy(pValue->strAddr_ls00, destUTF8.StrPtr(), pValue->strLeng_ls00);              
            } 
*/

        }   // end else of if (ctUTF8_els00 == CodeType)

    }   // end if (0 != pPattern->lpatDelimiterLen_ls00) 
}
//  ls18UndoubleDelimitersInCharData()

/*!
  -----------------------------------------------------------------------------
  function:     ls18CheckForNLInData
  -----------------------------------------------------------------------------
*/
SAPDB_Bool
ls18CheckForNLInData(tls00_Buffer2* pBuffer, tls00_Pattern* pPattern, tls00_FileFormat* pFileFormat)
{
    SAPDB_Bool           bNLInData   = SAPDB_FALSE;
    const tsp77encoding *_pEncoding  = pEncoding_ls18[pFileFormat->ffoCodeType_ls00];
  
    if (true == pFileFormat->ffoFormatted_ls00)
    {
        bNLInData = (pFileFormat->ffoBytesPerLine_ls00 > (pBuffer->lBufPos_ls00 + _pEncoding->charTable->nl.byteCount));
    }
    else
    {
        // Coming here means ALWAYS the user defined a delimiter. So. The last char
        // (excluding some whitespaces) must be - yes, the delimter. Check it.
        short           _state             = NL_UNDEFINED_LS18;
        char*           _pszWrkBuffer      = STATIC_CAST(char*, pBuffer->pBufPtr_ls00);
        tsp00_Int4      _lBufLen           = pBuffer->lBufPos_ls00;      // This is the length in Bytes!
        tsp77charConst  _Separator;
        SAPDB_Bool      _bDelimiterIsSpace = (SAPDB_FALSE != _pEncoding->isSpace(pPattern->ppatDelimit_ls00));
        SAPDB_UInt      _lCharSize         = _pEncoding->fixedCharacterSize;            // Initialize for ASCII

        if (csp_utf8 == _pEncoding->EncodingType)
        {
            _lCharSize = SAPDB_UTF8Basis::ReverseCharacterSize(REINTERPRET_CAST(SAPDB_UTF8*, _pszWrkBuffer)+_lBufLen);
        }
        
        // As the pattern consists of the sequence 'delimiter_separator-delimiter' we may conclude
        // from the pattern the separator
        _Separator.byteCount = pPattern->lpatPatternLen_ls00 - 2*pPattern->lpatDelimiterLen_ls00;
        memcpy(_Separator.bytes, pPattern->ppatPattern_ls00 + pPattern->lpatDelimiterLen_ls00, _Separator.byteCount);

        // Check for newlines eventually
        while (_lBufLen > 0)
        {
            // We have to make sure it is not checked for blank if the delimiter itself is a space!

            if ( (SAPDB_TRUE == _bDelimiterIsSpace) ||
                 ( (SAPDB_FALSE == _bDelimiterIsSpace) &&
                 //(0 == _pEncoding->isBlank(&_pszWrkBuffer[_lBufLen - _lCharSize])) &&
                   (0 == _pEncoding->isSpace(&_pszWrkBuffer[_lBufLen - _lCharSize])) ) )
/*            if ( (0 == _pEncoding->isBlank(&_pszWrkBuffer[_lBufLen - _lCharSize])) &&
                 (0 == _pEncoding->isSpace(&_pszWrkBuffer[_lBufLen - _lCharSize]))  )
*/
            {
                if (0 == memcmp(pPattern->ppatDelimit_ls00, &_pszWrkBuffer[_lBufLen - _lCharSize], pPattern->lpatDelimiterLen_ls00))
                {
                    switch (_state)
                    {
                        case NL_SINGLE_DELIMITER_LS18:
                        {
                            _state = NL_DELIMITER_PAIR_LS18;         // 2 adjacent delimiters define one single masked delimiter!
                            break;                                  // OR they denote an empty data value
                        }
                        case NL_DELIMITER_PAIR_LS18:
                        {
                            _state = NL_SINGLE_DELIMITER_LS18;       // well, already 2 found delimters -> declare the next as (next) first found
                            break;
                        }
                        case NL_SEPARATOR_FOLLOWED_BY_DELIMITER_LS18:
                        {
                            bNLInData = SAPDB_TRUE;
                            break;
                        }
                        default:        // NL_UNDEFINED_LS18
                        {
                            _state = NL_SINGLE_DELIMITER_LS18;
                        }
                    }   // end switch (_state)
                }
                else if (0 == memcmp(_Separator.bytes, &_pszWrkBuffer[_lBufLen - _lCharSize], _Separator.byteCount))
                {
                    // "data1","data2",                 _state = NL_UNDEFINED_LS18
                    // "data3"
                    //
                    // "data1","data2","                _state = NL_SINGLE_DELIMITER_LS18
                    // data3"
                    //
                    // "data1","data2","data,""         _state = NL_DELIMITER_PAIR_LS18
                    // 3"""
                    //
                    // "data1","data2","data3,"         _state = NL_SINGLE_DELIMITER_LS18
                    //
                    // "data1","data2","data""3,"""     _state = NL_SINGLE_DELIMITER_LS18
                    //
                    // Attention with empty values:
                    // "data1","data2","data3","",""
                    // but
                    // "data1","data2","data3"",""      here spreads content of value3 over 2 lines (rather unlikely)
                    // more to data3"


                    //if ( (NL_UNDEFINED_LS18 == _state) || (NL_DELIMITER_PAIR_LS18 == _state) || (NL_SEPARATOR_FOLLOWED_BY_DELIMITER_LS18 == _state))
                    if (NL_SINGLE_DELIMITER_LS18 == _state)
                    {
                        _state = NL_SEPARATOR_FOLLOWED_BY_DELIMITER_LS18;
                    }
                    else
                    {   
                        if (NL_DELIMITER_PAIR_LS18 != _state)    // in case of separator followed by delimiter pair we assume empty values at the end of 
                        {                                       // a data line -> no NL in data
                            bNLInData = SAPDB_TRUE;
                        }
                        break;
                    }
                }
                else
                {
                    if ( (NL_UNDEFINED_LS18 == _state) || (NL_DELIMITER_PAIR_LS18 == _state) )
                    {
                        bNLInData = SAPDB_TRUE;
                    }
//                  else //if ( (NL_SINGLE_DELIMITER_LS18 == _state) || (NL_SEPARATOR_FOLLOWED_BY_DELIMITER_LS18 == _state) ) --> no NL in data
//                    handles beliebige count of delimiter pairs + concluding delimiter
                    break;
                }
            }   // end if ( (0 == _pEncoding->isBlank(&_pszWrkBuffer[_lBufLen - _lCharSize])) && ...

            _lBufLen -= _lCharSize;

            if (csp_utf8 == _pEncoding->EncodingType)
            {
                _lCharSize = SAPDB_UTF8Basis::ReverseCharacterSize(REINTERPRET_CAST(SAPDB_UTF8*, _pszWrkBuffer)+_lBufLen);;
            }
        }   // end while (_lBufLen  >= 0)
    }   // end else of if (true == pFileFormat->ffoFormatted_ls00)


    if (SAPDB_TRUE == bNLInData)
    {   // 'Repair' record: put new line in ('\x0A')
        memcpy (&(STATIC_CAST(char*, pBuffer->pBufPtr_ls00)[pBuffer->lBufPos_ls00]),
                _pEncoding->charTable->nl.bytes, _pEncoding->charTable->nl.byteCount);
        pBuffer->lBufPos_ls00 += _pEncoding->charTable->nl.byteCount;
    }

    return bNLInData;
}
// ls18CheckForNLInData()

/*
  -----------------------------------------------------------------------------
  EndChapter: Public functions implementation
  -----------------------------------------------------------------------------
*/

//////////////////////////////////////////////////////////////////////////////////////////////
/*
  -----------------------------------------------------------------------------
  Chapter: Private functions implementation
  -----------------------------------------------------------------------------
*/

/*
  -----------------------------------------------------------------------------
  function:     ls18_RemoveDelimiter
  -----------------------------------------------------------------------------
*/
static void
ls18_RemoveDelimiter(tsp00_Addr     &pszStr,
                     tsp00_Longint  &lStrLen,
                     tsp00_Addr      pszDelimiter,
                     tsp00_Longint   lDelimiterLen,
                     bool            bFirst)
{
    if (0 != lDelimiterLen)
    {
        if (true == bFirst)
        {
            if (0 != lStrLen)       // the trimmed string is not empty
            {
                if (0 == memcmp(pszDelimiter, pszStr, lDelimiterLen) )
                {
                    lStrLen -= lDelimiterLen;
                    if (0 != lStrLen)          
                    {
                        pszStr += lDelimiterLen;
                    }
                }
            }

            if (0 == lStrLen)   // leave the string undefined if the length is 0
            {
                pszStr = NULL;
            }
        }
        else
        {
            if (0 != lStrLen)
            {
//TODOTODO WAS WENN DER STRING KUERZER IST ALS DIE DELIMITERLAENGE
                if (0 == memcmp(pszDelimiter, &pszStr[lStrLen - lDelimiterLen], lDelimiterLen) )
                {
                    lStrLen -= lDelimiterLen;
                }
            }

            if (0 == lStrLen)   // leave the string undefined if the length is 0
            {
                pszStr = NULL;
            }
        }

    }   // end if (0 != lDelimiterLen)
    return;
}
// ls18_RemoveDelimiter()



// Funktion zum lesen von daten mit \n
/*
SAPDB_Int4
ls18_ReadNextRecord(tls00_VFile*    pFile)
{
    SAPDB_Int4 rc = errOK_els00;


    ReadBuffer aReadBuffer(pFile, 32*1024, 3);
    
    tls00_Buffer2 myBuffer;
    myBuffer.lBufLen_ls00 = 32*1024;
    myBuffer.pBufPtr_ls00 = STATIC_CAST(char*, aReadBuffer.ls51GetNext());


    return rc;
}
*/

/*
  -----------------------------------------------------------------------------
  EndChapter: Private functions implementation
  -----------------------------------------------------------------------------
*/
