/*!
    @file     SharedSQL_SQLCache.hpp
    @ingroup  SharedSQL
    @author   DirkT
    @brief    SQLCache for calling SharedSQL from SQLManager
    @see            

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
\endif
*/

#ifndef SharedSQL_SQLCache_HPP
#define SharedSQL_SQLCache_HPP

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "SQLManager/SharedSQL/SharedSQL_ISQLCache.hpp"
#include "SQLManager/SharedSQL/SharedSQL_ICommandCache.hpp"
#include "SQLManager/SharedSQL/SharedSQL_CommandCacheStatistics.hpp"
#include "SQLManager/SharedSQL/SharedSQL_CommandCacheIterator.hpp"
#include "SQLManager/SharedSQL/SharedSQL_ParseIDCache.hpp"
#include "SQLManager/SharedSQL/SharedSQL_IHandles.hpp"
#include "SQLManager/SharedSQL/SharedSQL_Types.hpp"

#include "SAPDBCommon/SAPDB_Types.hpp"

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

class SharedSQL_CommandCache;
class SAPDBMem_RawAllocator;

/*---------------------------------------------------------------------------*/
/*!
    @class          SharedSQL_SQLCache
    @brief          Used by transactions to handle the sigleton commandcache and a session specific parseid cache
    @copydoc        SharedSQL_ISQLCache
 */
class SharedSQL_SQLCache : public SharedSQL_ISQLCache
{
public:
    /*!
        @brief constructor
        @param Alloc [in] - Allocator to be used by the session specific parseid cache
    */
    SharedSQL_SQLCache( SAPDBMem_IRawAllocator& Alloc );

    /*!
        @brief destructor - does nothing
    */
    ~SharedSQL_SQLCache( void );

	virtual SharedSQL_IPrepareHandle* NewPrepareHandle( SAPDBMem_IRawAllocator& Alloc, SharedSQL_SQLContext& Context, SharedSQL_SQLStmt& Stmt, SAPDB_Bool Internal );
	virtual void DestroyPrepareHandle( SharedSQL_IPrepareHandle* PH );
	virtual SharedSQL_IExecuteHandle* NewExecuteHandle( SAPDBMem_IRawAllocator& Alloc, const SharedSQL_ParseID& PID, bool CountExecuting, SAPDB_Bool& Valid, SAPDB_Bool& M9 );
	virtual void DestroyExecuteHandle( SharedSQL_IExecuteHandle* EH );
    //
    virtual SAPDB_Bool PutParseID( const SharedSQL_IPrepareHandle* PH, const SharedSQL_ParseID& PID, const SharedSQL_ModuleName& ModuleName, const SAPDB_Bool M9 );
	virtual SAPDB_Bool DropParseID( const SharedSQL_ParseID& PID );
    virtual const SharedSQL_ModuleName* GetModuleName( const SharedSQL_ParseID& PID );
    //
    virtual void CleanUpAll( void );
    virtual void InvalidateAll( void );
    //
    virtual SAPDB_Bool SetExecuteStatisticData( const SharedSQL_ParseID& PID,
                                        SAPDB_UInt8 ExecuteTimeSec, SAPDB_UInt8 ExecuteTimeMicroSec,
                                        SAPDB_Int8 ReadRowCount, SAPDB_Int8 QualifiedRowCount,
                                        SAPDB_Int8 VirtualReadCount, SAPDB_Int8 PhysicalReadCount,
                                        SAPDB_Int8 FetchRowCount, 
                                        SAPDB_Int8 SuspendCount, SAPDB_Int8 WaitCount );
    //
    virtual void GetCommandCacheInfo( SharedSQL_CommandCacheInfo& S );
    virtual void GetCommandCacheInfoReset( SharedSQL_CommandCacheInfo& S );
    virtual void ResetCommandCacheInfo( void );
    virtual void ResetCommandInfos( void );
    virtual SharedSQL_CommandCacheIterator GetCommandCacheIterator( void );
    //
    // methods derived from SAPDBMem_RawAllocator
    virtual void* Allocate( SAPDB_ULong ByteCount );
    virtual void* Allocate( SAPDB_ULong ByteCount, const void* Hint );
    virtual void Deallocate( void* p) ;
    virtual void GetBaseAllocatorCallStatistics( SAPDB_ULong &CountAlloc, SAPDB_ULong &CountDealloc ) const;
    virtual void GetCallStatistics( SAPDB_ULong &CountAlloc, SAPDB_ULong &CountDealloc ) const;
    virtual int  GetErrorCount( void ) const;
    //
    virtual SAPDBMem_RawAllocator* GetCommandCacheBaseRawAllocator( void ); // just for Testing !!!
    //
private:
    /// Session specific ParseIDCache
	SharedSQL_ParseIDCache	    mParseIDCache;
    //
};  // SharedSQL_SQLCache

#endif  /* SharedSQL_SQLCache_HPP */

