/*!
  @file           SAPDBFields_IFieldSequences.hpp
  @author         StefanB, TillL, UweH
  @ingroup        Field Container classes
  @brief          Declarations of interfaces for field containers.

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
\endif
*/


#ifndef SAPDBFields_IFieldSequences_HPP
#define SAPDBFields_IFieldSequences_HPP

#include "SAPDBCommon/SAPDB_Types.hpp"
#include "SAPDBCommon/ErrorsAndMessages/SAPDBErr_Assertions.hpp"
#include "SAPDBCommon/Fields/SAPDBFields_Field.hpp"
#include "SAPDBCommon/SAPDB_OStream.hpp"

/*!
@interface      SAPDBFields_IFieldSequence
@brief          An interface to an ordered set of fields.
*/
class SAPDBFields_IFieldSequence
{
public:
    /// The range for the number of elements in field Containers
    typedef SAPDB_UInt IndexType;
    /*!
    @brief          Returns the number of fields in the field sequence.
    @return         IndexType
    */
    virtual IndexType GetFieldCount() const = 0;
    /*!
    @brief          Returns the &lt;N>-th field within the field sequence.
    @param          FieldNo [in] The index of the field [0...GetFieldCount()-1]
    @param          Field [in/out] The field to be assigned to the &lt;N>-th component
    @return         SAPDBFields_Field
    */
    virtual SAPDBFields_Field& GetField ( IndexType          FieldNo,
                                          SAPDBFields_Field &Field ) const = 0;
    /*!
    @brief          Returns the length of the field with index &lt;N>.
    @return         SAPDBFields_Field::SizeType:
    */
    virtual SAPDBFields_Field::SizeType GetFieldLength(IndexType FieldNo) const = 0;
    /*!
    @brief          Returns the sum of the length of all fields in the field sequence in bytes.
    @return         SAPDBFields_Field::SizeType:
    */
    virtual SAPDBFields_Field::SizeType GetTotalLength() const = 0;
    /*!
       @brief          Writes all fields from Begin up to End sequentially to
                       Destination End is not included.
       @param          Destination [in] The buffer into which the fields should be written.
       @param          Offset [in] The offset withion the buffer where the writing starts.
       @param          Begin [in] The index of the fields from where the write starts.
       @param          End [in] The index of the field up to which the write works.
                       End is not included in writing.
       @return         SAPDBFields_Field::SizeType: The number of bytes copied.
    */
    virtual SAPDBFields_Field::SizeType Write(SAPDBFields_Field            &Destination,
                                              SAPDBFields_Field::IndexType  Offset,
                                              IndexType                     Begin,
                                              IndexType                     End) const = 0;
    /*!
    @brief   Outputs the content of the sequence and returns the modified output stream.
    @param   os [in] The output stream
    @param   Seq [in] The field sequence to be printed.
    @return  SAPDB_OStream&amp;

	This operator ist pseudo virtual, because it calls the
    virtual method Print(). So every derived class can define its
    own output method.
    */
      
    friend SAPDB_OStream& operator <<(SAPDB_OStream& os, const SAPDBFields_IFieldSequence& Seq)
    {
        Seq.Print(os);
        return os;
    }

protected:
    /*!
    @brief   Outputs the content of the sequence to an output stream.
    @param   os [in] The output stream
    */
    virtual void Print(SAPDB_OStream& os) const = 0;
};
#endif // SAPDBFields_IFieldSequences_HPP

