/*!
 * \file    OMS_VersionDictionary.hpp
 * \author  MarkusSi
 * \brief   OMS context.
 */
/*

    ========== licence begin  GPL
    Copyright (c) 2002-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/

#ifndef __OMS_VERSIONDICTIONARY_HPP
#define __OMS_VERSIONDICTIONARY_HPP

#include "Oms/OMS_ContextDef.hpp"
#include "Oms/OMS_UnloadableVersionList.hpp"
#include "Oms/OMS_Defines.h"                     
#include "Oms/OMS_LibOmsInterfaceInstance.hpp"    

class OMS_VersionDictionary {
public :
  /// Iterator for the version dictionary  
  /*! 
  ** This class implements an iterator to read the version dictionary.
  **
  ** \attention Even if the access is synchronized, the iterator might not return a 'snapshort'
  ** as always only a single lock on the current slot is hold while iterating over the entries
  ** and not a global lock on all slots. This is done because of performance and as it is not
  ** considered as so important to get a real 'snapshot'.
  */
  class Iter {
  private:
    /// Slot of the hashbucket, where the iterator is currently positioned to. 
    int                            m_slot;
    /// How should the entries be locked while iteration over them?
    OMS_LockMode                   m_lockMode;
    /// The version the iteration is currently pointing to.
    OMS_Context                   *m_pCurr;
    /*! Pointer, to identify the iterator in the kernel, if the ProcServer is active.
    **  \attention Do not deref this pointer, it is only useful as parameter to call
    **  ProcServer-methods
    */
    void                          *m_pIter;

    /// Enters the region (either reader-writer of critical section) which belongs to the current slot  
    void enterRgn();
    /// Leaves the region (either reader-writer of critical section) which belongs to the current slot  
    void leaveRgn();

    Iter(const Iter&);                  ///< Not allowed, therefore private
    Iter& operator=(const Iter &iter);  ///< Not allowed, therefore private
  
  public :
    /// Default-constructor
    Iter();

    /// Constructor
    /*! 
    ** A iterator is created and the iterator is positioned on the first entry.
    ** if the lockMode does not equal NO_LOCK, then a lock on the slot of the first
    ** entry is aquired and is hold.
    **
    ** \param lockMode          [in] should the access to the dictionary be synchronized.
    */
    Iter(OMS_LockMode lockMode) {first(lockMode);}
    
    /// Destructor
    /*! 
    ** The iterator is destructed. If it still holds some locks, than these will be
    ** released.
    */
    ~Iter();

    /// Increment
    /*! 
    ** Increments the iterator to the next entry. If this entry is locacted in another
    ** hash-bucket, then the old lock will be released an the lock for the next slot
    ** will be aquired. This means, that a pointer to a version, which is remembered
    ** might not be locked anymore, after the iterator has been incremented. 
    **
    ** \attention Do not drop a version while iterating over the version dictionary
    **    with this operator. If the deletion of the current version is needed use
    **    the method CVersionDictionary_co17::getNext instead.
    */
    void operator++() { getNext(false); }

    /// Increment with possibility to delete the current version
    /*! 
    ** Increments the iterator to the next entry and if specified the entry of the
    ** current version is deleted. If this entry is locacted in another hash-bucket, 
    ** then the old lock will be released an the lock for the next slot will be aquired.
    **
    ** \param dropCurr [in] If this parameter is specified, then the version, the iterator
    **        is currently pointing to, is deleted and the iterator is then moved to the 
    **        next entry. 
    */
    void getNext(bool dropCurr=false);

    /// Returns true if the iterator does not have reached its end.
    operator bool() const;

    /// Returns the version the iterator is currently pointing to.
    OMS_Context* operator()() const;

    /// Positions an existing iterator onto the frist entry.
    /*! 
    ** Positions an existing iterator onto the frist entry.
    ** 
    ** \param lockMode          [in] should the access to the dictionary be synchronized.
    */
    void first(OMS_LockMode lockMode);
    
    /// Stops an iterator
    /*! 
    **  Stops an iterator. This function is needed, if an iterator should stop processing
    **  without having run to the end. In this case the iterator might hold a lock which must
    **  be released to prevent lock contention.
    **  A call of this method set the iterator to invalid mode. Therefore it is not possible 
    **  to continue iterating after having called 'stop'.
    */
    void stop();
  };

  /// Constructor
  OMS_VersionDictionary();

  /// Returns the version corrsponding to the specified version-id
  OMS_Context* FindVersion  (const OmsVersionId&);

  /// Inserts a new version in the directory
  /*!
  ** Inserts a new version in the directory.
  **
  ** \param versionId [in] Name of the version (key in the search-structure), 
  **                       which should be inserted
  ** \param pContext  [in] Pointer to the version, which should be inserted
  **
  ** \return If a version with the same name already exists, then the error code
  **         e_duplicate_key is returned; otherwise e_ok
  */
  tgg00_BasisError InsertVersion(const OmsVersionId& versionId, OMS_Context* pContext); 

  /// Deletes a version out of the directory
  /*!
  ** Deletes a version out of the directory.
  ** 
  ** \param versionId [in] Name of the version, which which should be deleted
  **
  ** \return If no version with the specified name is found, then e_unknown_version
  **         is returned; otherwise e_ok
  */
  tgg00_BasisError DropVersion  (const OmsVersionId& versionId);
 
  void             MarkUnloadable(OMS_Context* pContext, bool callFromDestructor);
  void             MarkNotUnloadable(IliveCacheSink* lcSink, OMS_Context* pContext);
  void             Shutdown();
  bool             UnloadOldestVersion(IliveCacheSink* lcSink);

  /// Determines and returns the hash-slot of a given version-id  
  int              GetSingleLockId(const OmsVersionId &versionId) const;

  /// Determines and returns the hash-slots of all given version-ids
  /*! 
  ** Determines and returns the hash-slots of all given version-ids.
  **
  ** \param ppVersionId [in]  Array with the given version-ids.
  ** \param size        [in,out] in: number of version-ids in the array ppVersionId
  **                             out: number of slot-ids returned in pLockId
  **  \param pLockId    [out] Array with the hash-slots which corresponds to the given version-ids
  **
  ** \attention The entries in the array pLockId are sorted in ascending order
  ** to prevent deadlocks. Therefore does not change this sequence manually!
  */
  void             GetMultiLockIds(const OmsVersionId **ppVersionId, int &size, int  *pLockId) const;
  
  /// Returns all lock-ids of the version-dictionary
  void             GetAllLockIds(int *pLockId, int &size) const;

  /// Are reader-writer-locks or critical section used for synchonization
  bool             UseRWLocks() const { return m_useRWLocks; }

private :
  /// Compute the hash-slot for a given version-id
  inline int       HashValue (const OmsVersionId&) const;
  /// Return the offset of the lock-ids.
  /*! 
  ** In the version-dictionary coding it is assumed that the lock-ids start with
  ** the number zero, but in the kernel these lock-ids might be mapped to other 
  ** ids. Therefore this offset is needed.
  */
  inline int       GetVDirStartCnt() const;

  /// Does the own session own a lock on the given slot
  bool isLockedByOwn(int slot) const;   

  /// Hash-array. Collisions are resolved by chaining 
  OMS_Context           *m_pVersionDictionary[OMS_VDIR_SIZE];
  /// Datastructure which contains all versions which could be unloaded
  OMS_UnloadableVersionList m_UnloadableVersions;
  /// Flag which indicates whether to use reader-writer-locks or critical sections for synchonization
  bool                   m_useRWLocks;
  Iter                  *m_pIter;

  friend void OMS_LibOmsInterfaceInstance::AdviseKernelInterface(LVC_KernelInterface& knlInterface, bool isKernel); // Initializes the variable m_useRWLocks
  friend class Iter;
};

/*---------------------------------------------------------------------------*/

inline int OMS_VersionDictionary::GetVDirStartCnt() const {
  return m_useRWLocks ? OMS_VDIR_RW_START_CNT : OMS_VDIR_CS_START_CNT; 
}

/*---------------------------------------------------------------------------*/

inline int OMS_VersionDictionary::HashValue (const OmsVersionId &versionId) const{
  unsigned long sum = 0;
  for (int i=0; i<sizeof(OmsVersionId); ++i){
    sum ^= ((sum << 5) ^ versionId[i]);
  }
  return (sum % OMS_VDIR_SIZE);
}

#endif  // __OMS_VERSIONDICTIONARY_HPP
