/*!
  @file           Log_WriterTaskSynchronizer.hpp
  @author         UweH
  @author         TillL
  @ingroup        Logging
  @brief          
  @see            

\if EMIT_LICENCE
  ========== licence begin  GPL
  Copyright (c) 2000-2004 SAP AG

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  ========== licence end
\endif
*/



#ifndef LOG_WRITERTASKSYNCHRONIZER_HPP
#define LOG_WRITERTASKSYNCHRONIZER_HPP


#include "gsp00.h"
#include "ggg00.h"
#include "heo56.h"

#include "SAPDBCommon/SAPDB_Types.hpp"
#include "RunTime/Synchronisation/RTESync_NamedSpinlock.hpp"


/// The Log_Writer is suspended with this reason because of an empty queue
#define LOG_QUEUE_EMPTY             235

/// The Log_Writer is initially suspended with this suspend reason
#define LOG_WRITER_NOT_INITIALIZED  242


/*!
  @class          Log_WriterTaskSynchronizer
  @brief          Corodinates the suspend/resume of the Log_Writer
 */

class Log_WriterTaskSynchronizer
{
public:

	/// identifier of kernel task
    typedef tsp00_TaskId                                        TaskID;

    /// suspend reason id
    typedef SAPDB_Int2                                          SuspendReason;

    /// constructor
    Log_WriterTaskSynchronizer
        (TaskID                         taskid)
        :
         m_TaskID    (taskid),
         m_Suspended (false),
         m_Spinlock  ((SAPDB_UTF8*)"Log_WriterTaskSynchronizer::m_Spinlock")
    {}

    /// set the flag, that the writer will be suspended in the future
    void WillBeSuspended ()
    {
        RTESync_LockedScope Lock(m_Spinlock);
        m_Suspended = true;
    }

    ///  after WillBeSuspend() must follow DoSuspend(), which definitely suspends the caller
    void DoSuspend  (SuspendReason reason)
    {
        vsuspend(m_TaskID, reason);
    }

    /// Suspends writer-task if necessary
    void Suspend (SuspendReason reason)
    {
        m_Spinlock.Lock();
        m_Suspended = true;
        m_Spinlock.Unlock();
        vsuspend(m_TaskID, reason);
    }

    /// Resumes task if necessary (if still running).
    void Resume()
    {
        SAPDB_Bool do_resume = false;
        {
            RTESync_LockedScope Lock(m_Spinlock);

            if (m_Suspended)
            {
                do_resume   = true;
                m_Suspended = false;
            }
        }
        if (do_resume)
            vresume(m_TaskID);
    }

    /// Returns the task id of the writer task
    TaskID GetTaskID() const
    {
        return m_TaskID;
    }

private:

    /// user task id
    TaskID                          m_TaskID;

    /// task suspend state
    SAPDB_Bool                      m_Suspended;

    /// lock for synchronized code parts
    mutable RTESync_NamedSpinlock   m_Spinlock;

};




#endif // LOG_WRITERTASKSYNCHRONIZER_HPP

