#ifndef JOIN_IRESULTSETAGGREGATOR_HPP
#define JOIN_IRESULTSETAGGREGATOR_HPP
/*!
 * @file
 * @brief interface to result set operator which inserts record and 
 * sort them corresponding to key
 *
 * @author GertG
 * @ingroup Join
 *
 * @par last changed by:
 * <br>
 * $Author: d024980 $ $DateTime: 2004/05/03 16:16:26 $
 *
 * @sa Join_Execute.cpp, Join_IResultSetOperator.hpp
 */
/*

    ========== licence begin  GPL
    Copyright (c) 2003-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/

#include "SAPDB/Container/Container_Vector.hpp"
#include "SAPDB/Join/Join_IResultSetOperator.hpp"
#include "SAPDB/Join/Join_JoinOperator.hpp"
#include "hgg04.h"
#include "gsp00.h"
#include "hsp30.h"
#include "hsp40.h"
#include "hsp41.h"
#include "hsp51.h"
#include "hta01.h"
#include "hbd73.h"
#include "gkb07.h"

//! interface to result set operator which aggregates record corresponding to key
class Join_IResultSetAggregator : public Join_IResultSetOperator {
public:
    //! @name constructor / destructor
    //@{
    //! constructor
    /*! 
     * @param acv [in] global context
     * @param rec_desc [in] pointer to result record description
     * @param result_id [in] name of result file
     * @param rescnt [in] amount of records already in result set
     * @param max_recs_accepted [in] maximal amount of records accepted for result set
     * @param rec_producer [in] operator delivers records
     */

    
    Join_IResultSetAggregator(
            tak_all_command_glob&          acv, 
            const tgg00_StackList * const  rec_desc,
            tgg00_FileId&                  result_id, 
            const SAPDB_Int4&              rescnt,
            const SAPDB_Int4&              max_recs_accepted,
            Join_JoinOperator&             rec_producer ) :
    Join_IResultSetOperator(acv, rec_desc, rescnt, max_recs_accepted), 
    m_GroupCnt(0), m_CallPostProcessing(sfh_none), 
    m_LaterOutputPos(0), m_RecordProducer(rec_producer),
    m_DistinctFileRoot( *((SAPDBMem_IRawAllocator*) acv.a_transinf.tri_trans.trAllocator_gg00) ),
    m_ResultTree( result_id )
    {
        SAPDBTRACE_METHOD_DEBUG( "Join_IResultSetAggregator::ctor", Join_Trace, 5 );
        g04build_temp_tree_id( m_DistinctFileId, acv.a_transinf.tri_trans );
        m_DistinctFileId.fileTfnTemp_gg00().becomes( ttfnDistinct_egg00 );
        m_DistinctFileRoot.Reserve(2);
        b73cmd_count( iins_sorts );
    }

    //! destructor
    virtual ~Join_IResultSetAggregator()
    {
        SAPDBTRACE_METHOD_DEBUG( "Join_IResultSetAggregator::dtor", Join_Trace, 5 );
        if ( ! m_DistinctFileRoot.IsEmpty() )
        {
            // destroy distinct files
            b01prefix_destroy_files(m_acv.a_transinf.tri_trans, 
                    m_DistinctFileId.fileName_gg00(), TEMP_PREFIX_MXGG00 + 1 );
        }
        
    }

    //! initializes operator
    virtual void Open();

    //! finalize operator
    virtual void Close()
    {
        SAPDBTRACE_METHOD_DEBUG( "Join_IResultSetAggregator::Close", Join_Trace, 3 );
        if ( this->ResultCount() > 0 )
        {
            if ( (sfh_none != m_CallPostProcessing) || !m_RecordCounterPositions.IsEmpty() )
                post_processing();
        }
        b73cmds_count( iins_sorts_rows_ins, this->ResultCount() );
    }
    //@}

    //! @name properties
    //@{
    //! return amount of records in result set
    virtual SAPDB_Int4 ResultCount() const { return m_GroupCnt; }
    //@}

protected:

    //! common code to add a record to result set
    /*!
     * @param record [in] record to be added
     * @return information code <tt>[e_ok, e_no_next_record]</tt> / error code
     */
    tgg00_BasisError add( tgg00_Rec& record )
    {
        SAPDBTRACE_METHOD_DEBUG( "Join_IResultSetAggregator::add", Join_Trace, 3 );

        tgg00_BasisError _b_err = e_ok;
        
        set_result_counter( record, 1 );
        
        if ( e_key_not_found == this->get_group_record( record ) )
            _b_err = this->new_group( record );
        else
            _b_err = this->update_group( record );

        if ( e_ok == _b_err )
        {
            if ( m_acv.a_transinf.tri_trans.trRteCommPtr_gg00->to_cancel )
                _b_err = e_cancelled;

            if ( ++m_ResultCnt == m_MaxResultCnt ) 
            {
                _b_err = e_no_next_record;
                if ( m_SetWarning )
                {
                    m_acv.a_transinf.tri_trans.trWarning_gg00.addElement(warn0_exist);
                    m_acv.a_transinf.tri_trans.trWarning_gg00.addElement(warn12_rowno_used);
                }
            }
        }
        SAPDBTRACE_WRITELN( Join_Trace, 3, "accepted recs: " << m_ResultCnt << "\tresult cnt: " << m_GroupCnt );
        SAPDBTRACE_WRITELN( Join_Trace, 3, "return: " << SAPDBTrace::BasisError(_b_err) );
        return _b_err;
    }

    void build_group_key( tgg00_Rec& record )
    {
        SAPDBTRACE_IF( Join_Trace, 5, memset( &m_GroupKey, ':', sizeof( m_GroupKey ) ) );
        m_GroupKey.len() = record.recKeyLen_gg00();
        memcpy( &m_GroupKey.k(), (SAPDB_Byte *)&record.recBody_gg00(), record.recKeyLen_gg00() );
        SAPDBTRACE_IF( Join_Trace, 5, t01key( td_always, "m_GroupKey  ", m_GroupKey ) );
    }
    
    //! get group record for a given record into buffer behind m_GroupRec
    /*!
     * @param record [in] record for a group
     */
    virtual tgg00_BasisError get_group_record( tgg00_Rec& record )=0;

    //! update group record for a given record
    virtual tgg00_BasisError update_group_record()=0;

    //! insert group record for a given record
    /*!
     * @param record [in] record for a group
     */
    virtual tgg00_BasisError add_group_record( tgg00_Rec& record )=0;

    //! build new group record in m_GroupRec
    /*!
     * @param record [in] record for new group
     */
    tgg00_BasisError new_group( tgg00_Rec& record );

    
    //! build an updated group record in m_GroupRec
    /*!
     * @param record [in] record for group update
     */
    tgg00_BasisError update_group( tgg00_Rec& record );

    //! build an updated group record in m_GroupRec
    /*!
     * @param record [in] record for group update
     */
    virtual tgg00_BasisError is_new_group_allowed() =0 ;
    
    virtual void init_result_scan()=0;
    
    virtual tgg00_BasisError fetch_next_result( tgg00_Rec& record )=0;

    virtual tgg00_BasisError update_result( tgg00_Rec& record )=0;
        
    // member variables
    SAPDB_Int4 m_GroupCnt;            //!< result count == group count
    tgg00_Rec  m_GroupRec;            //!< aux. buffer for group record
    tgg00_FilePos  m_TreePos; //!< treepos of group record
    tgg00_Lkey     m_GroupKey;        //!< group key
    SAPDB_UInt4    m_LaterOutputPos;
    tgg00_FileId   m_ResultTree;      //!< name of result file
    
    Join_JoinOperator& m_RecordProducer;

private:

    //! calculate functions and check SUM overflow
    void post_processing();
    
    struct mc_RootPair { 
        mc_RootPair() :
            m_Root(0), m_RootCheck(0) {}
        mc_RootPair( SAPDB_Int4 root, tsp00_PageNo rootcheck) :
            m_Root(root), m_RootCheck(rootcheck) {}
        SAPDB_Int4 m_Root; tsp00_PageNo m_RootCheck; 
    };

    // member variables
    tkb07_setfunc_handling_Enum m_CallPostProcessing;

    tgg00_FileId    m_DistinctFileId; //!< template name of distinct files
    Container_Vector<mc_RootPair>  m_DistinctFileRoot;
};

#endif // JOIN_IRESULTSETAGGREGATOR_HPP
