/*!
 * @file
 * @brief implements Join_IResultSetAggregator
 *
 * @author GertG
 * @ingroup Join
 *
 * @par last changed by:
 * <br>
 * $Author: d024980 $ $DateTime: 2004/05/03 16:16:26 $
 *
 * @sa Join_IResultSetAggregator.hpp
 */
/*

    ========== licence begin  GPL
    Copyright (c) 2003-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/
#include "SAPDB/Join/Join_IResultSetAggregator.hpp"
#include "SAPDB/SAPDBCommon/SAPDB_RangeCode.hpp"
#include "hak07.h"
#include "hbd07.h"
#include "hkb721.h"
#include "hbd01.h"
#include "hgg01_3.h"
#include "hta01.h"
#include "hsp51.h"

/* ******************** PUBLIC MEMBERS ********************* */
/*!
  * analyses result record and detect ROWNO/ROWNUM positions
  */
void Join_IResultSetAggregator::Open() 
{
    SAPDBTRACE_METHOD_DEBUG( "Join_IResultSetAggregator::Open", Join_Trace, 1 );

    Join_IResultSetOperator::Open();
    
    tgg00_BasisError _b_err = e_ok;
    SAPDB_UInt4 _i = 1;
    SAPDB_UInt4 _func_pos = (*m_RecordDesc)[ 0 ].epos() - 1;
    SAPDB_UInt2 _func_cnt = 0, _out_pos;
    SAPDB_UInt4 _disfile_cnt = 0;
   
    // check for LASTFUNCTION
    if ( !( st_func == (*m_RecordDesc)[ _func_pos - 1 ].etype() && 
          op_f_none == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ))
        _b_err = e_stack_type_illegal;
    
    while (( _func_pos - (*m_RecordDesc)[ _func_pos - 1 ].epos() > 1) 
            && ( e_ok == _b_err ))
    {
        _func_pos -= (*m_RecordDesc)[ _func_pos - 1 ].epos();
        SAPDBTRACE_WRITELN( Join_Trace, 3, "func pos: " << _func_pos );
        if ( st_func == (*m_RecordDesc)[ _func_pos - 1 ].etype() )
        {
            if ( op_f_check_null != (*m_RecordDesc)[ _func_pos - 1 ].eop_func() )
            {
                ++_func_cnt; 
                if  (
                    (op_f_avg         == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                    op_f_dis_avg      == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                    op_f_stddev       == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                    op_f_dis_stddev   == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                    op_f_variance     == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                    op_f_dis_variance == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() )
                    &&
                    (
                    (1 == (*m_RecordDesc)[ _func_pos - 1 ].ecol_tab()[0]) ||
                    /* selrec.selr_union OR       check for union has to be included */
                    ((ttfnUserResult_egg00 != m_ResultTree.fileTfnTemp_gg00()) &&
                    ( ttfnInto_egg00       != m_ResultTree.fileTfnTemp_gg00()))
                    )
                    )
                {
                    m_CallPostProcessing = sfh_all;
                }
                else
                {
                    if (op_f_stddev       == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                        op_f_dis_stddev   == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                        op_f_variance     == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                        op_f_dis_variance == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() )
                    { 
                        m_CallPostProcessing = sfh_stddev_variance;
                    }
                }
                if ( op_f_dis_avg     == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                    op_f_dis_sum      == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                    op_f_dis_count    == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                    op_f_stddev       == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                    op_f_dis_stddev   == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                    op_f_variance     == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                    op_f_dis_variance == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() )
                {
                    tgg00_FileId _htree( m_DistinctFileId );
                    
                    _htree.fileTempCnt_gg00()   = ++_disfile_cnt;
                    _htree.fileRootCheck_gg00() = ROOT_CHECK_GG00;
                    b01tcreate_file( m_acv.a_transinf.tri_trans, _htree );
                    _b_err = m_acv.a_transinf.tri_trans.trError_gg00;
                    if ( m_DistinctFileRoot.IsFull() ) 
                        m_DistinctFileRoot.Reserve( 2*m_DistinctFileRoot.GetSize() );
                    m_DistinctFileRoot.InsertEnd( mc_RootPair(_htree.fileRoot_gg00(), _htree.fileRootCheck_gg00()) );
                }
                _out_pos = k721out_entry( const_cast<tgg00_StackList*>(m_RecordDesc), _func_pos );
                if ( op_o_output_hold == (*m_RecordDesc)[ _out_pos - 1 ].eop_out() &&
                    op_o_output_later == (*m_RecordDesc)[ _out_pos - 1 + 1 ].eop_out() )
                    m_LaterOutputPos = _out_pos + 1;
           }
        }
        else
            _b_err = e_stack_type_illegal;
    }
    SAPDBTRACE_WRITELN( Join_Trace, 3, "m_CallPostProcessing: " << m_CallPostProcessing );

    if ( e_ok != _b_err )
        a07_b_put_error( m_acv, _b_err, 1 );
}

/* ******************* PROTECTED MEMBERS ******************* */
/*!
 *
 */
tgg00_BasisError Join_IResultSetAggregator::new_group( tgg00_Rec& record )
{
    SAPDBTRACE_METHOD_DEBUG( "Join_IResultSetAggregator::new_group", Join_Trace, 3 );
    tgg00_BasisError _b_err = e_ok;

    SAPDBTRACE_WRITELN( Join_Trace, 3, "input record:" );
    SAPDBTRACE_IF( Join_Trace, 3, 
    t01buf( td_always, &record, 1, record.recLen_gg00() <= 300 ? record.recLen_gg00() : 300 ));
    _b_err = this->is_new_group_allowed();
    
    if ( e_ok == _b_err )
    {
        SAPDB_UInt4 _func_pos = (*m_RecordDesc)[ 0 ].epos() - 1;
        SAPDB_UInt4 _out_pos, _check_null_pos = 0, _disfile_cnt = 0;
        bool _null_found = false;
        

        // loop over functions
        while (( _func_pos - (*m_RecordDesc)[ _func_pos - 1 ].epos() > 1) 
                && ( e_ok == _b_err ))
        {
            _func_pos -= (*m_RecordDesc)[ _func_pos - 1 ].epos();
            // loop over LASTFUNCTION
            while (( st_func == (*m_RecordDesc)[ _func_pos - 1 ].etype() ) 
                    && ( op_f_none == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ))
                _func_pos -= (*m_RecordDesc)[ _func_pos - 1 ].epos();
            _out_pos = k721out_entry( const_cast<tgg00_StackList*>(m_RecordDesc), _func_pos );
            if ( st_func == (*m_RecordDesc)[ _func_pos - 1 ].etype() )
            {
                SAPDBTRACE_WRITELN( Join_Trace, 3, "func on: " << _func_pos );
                if ( op_f_check_null == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() )
                {
                    _check_null_pos = (*m_RecordDesc)[ _out_pos - 1 ].epos();
                    SAPDBTRACE_WRITELN( Join_Trace, 3, "_check_null_pos: " << _check_null_pos );
                }
                else
                {
                    if ( op_f_dis_avg   == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                         op_f_dis_sum   == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                         op_f_dis_count == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                         op_f_stddev    == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                         op_f_dis_stddev == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                         op_f_variance  == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                         op_f_dis_variance == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() )
                    {
                        if ( csp_undef_byte != record.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1 ] && 
                             csp_oflw_byte  != record.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1 ] )
                        {
                            SAPDBTRACE_WRITELN( Join_Trace, 3, "create dist entries" );
                            tgg00_FileId _htree( m_DistinctFileId );
                            tgg00_Rec _dist_buffer;
                            SAPDBTRACE_IF( Join_Trace, 3, memset( &_dist_buffer, ':', sizeof( _dist_buffer ) ) );

                            // move key=<group value> to _dist_buffer
                            SAPDB_RangeMove( __FILE__, 1,
                                    sizeof( m_GroupKey ), sizeof(_dist_buffer),
                                    &m_GroupKey.k(), 1,
                                    &_dist_buffer.recBody_gg00(), 1,
                                    m_GroupKey.len(), _b_err );
                            // move value to aggregate to _dist_buffer
                            SAPDB_RangeMove( __FILE__, 2,
                                    sizeof( record ), sizeof( _dist_buffer ),
                                    &record, (*m_RecordDesc)[ _out_pos - 1 ].epos(),
                                    &_dist_buffer.recBuf_gg00(), 
                                    m_GroupKey.len() - RESCNT_MXGG04 + cgg_rec_key_offset + 1,
                                    (*m_RecordDesc)[ _out_pos - 1 ].elen_var(), 
                                    _b_err );
                            // set key = <group value, value>
                            _dist_buffer.recKeyLen_gg00() = m_GroupKey.len() - RESCNT_MXGG04 + (*m_RecordDesc)[ _out_pos - 1 ].elen_var();
                            // set record length
                            _dist_buffer.recLen_gg00() = _dist_buffer.recKeyLen_gg00() + cgg_rec_key_offset;
                            _dist_buffer.recVarcolOffset_gg00() = 0;
                            _dist_buffer.recVarcolCnt_gg00() = 0;
                            
                            if ( op_f_stddev        == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                                 op_f_dis_stddev    == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                                 op_f_variance      == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                                 op_f_dis_variance  == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() )
                            {
                                // write 1 with swapping 
                                SAPDBTRACE_WRITELN( Join_Trace, 3, "write 1" );
                                s20ch4sw( 1, g01code.kernel_swap,
                                      _dist_buffer.recBody_gg00(),
                                      _dist_buffer.recKeyLen_gg00() + 1,
                                      sw_normal);
                                _dist_buffer.recLen_gg00() += RESCNT_MXGG04;
                            }

                            _htree.fileRoot_gg00() = m_DistinctFileRoot[ _disfile_cnt ].m_Root;
                            _htree.fileRootCheck_gg00() = m_DistinctFileRoot[ _disfile_cnt ].m_RootCheck;
                            _htree.fileTempCnt_gg00() = ++_disfile_cnt;
                            SAPDBTRACE_IF( Join_Trace, 3, 
                            t01buf( td_always, &_dist_buffer, 1, _dist_buffer.recLen_gg00() <= 300 ? _dist_buffer.recLen_gg00() : 300 ));
                            b07cadd_record( m_acv.a_transinf.tri_trans, _htree, _dist_buffer );
                            _b_err = m_acv.a_transinf.tri_trans.trError_gg00;
                        }
                    }
                    else
                    {
                        // do nothing for op_f_all_count, op_f_count, 
                        // op_f_avg. op_f_min/max, op_f_sum
                    }
                }
                if (( csp_undef_byte == record.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1 ] && 
                     op_f_all_count != (*m_RecordDesc)[ _func_pos - 1 ].eop_func() )
                    ||
                    ( csp_oflw_byte == record.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1 ] && 
                      !( op_f_count     == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() || 
                         op_f_dis_count == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                         op_f_all_count == (*m_RecordDesc)[ _func_pos - 1 ].eop_func())))
                {
                    SAPDBTRACE_WRITELN( Join_Trace, 3, "NULL found: " << _out_pos );
                    _null_found = true;
                    m_acv.a_transinf.tri_trans.trWarning_gg00.addElement(warn0_exist);
                    m_acv.a_transinf.tri_trans.trWarning_gg00.addElement(warn2_null_in_builtin_func);
                }
                if ( op_f_avg        == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() || 
                     op_f_dis_avg    == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                     op_f_count      == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                     op_f_dis_count  == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                     op_f_all_count  == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                     op_f_stddev     == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                     op_f_dis_stddev == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                     op_f_variance   == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                     op_f_dis_variance == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() )
                {
                    SAPDB_UInt4 _pos;
                    tsp00_NumError _numerr;
                    _numerr.becomes( num_ok );
                    
                    if ( op_f_avg        == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() || 
                         op_f_dis_avg    == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                         op_f_stddev     == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                         op_f_dis_stddev == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                         op_f_variance   == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                         op_f_dis_variance == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() )
                        _pos = (*m_RecordDesc)[ _out_pos - 1 ].epos() +  NUMBER_MXGG04;
                    else
                        // op_f_*_count
                        _pos = (*m_RecordDesc)[ _out_pos - 1 ].epos();
                    SAPDBTRACE_WRITELN( Join_Trace, 3, "pos: " << _pos );
                    if (( csp_undef_byte == record.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1 ] && 
                         op_f_all_count != (*m_RecordDesc)[ _func_pos - 1 ].eop_func() )
                        ||
                        ( csp_oflw_byte == record.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1 ] && 
                          !( op_f_count     == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() || 
                             op_f_dis_count == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                             op_f_all_count == (*m_RecordDesc)[ _func_pos - 1 ].eop_func())))
                        // int(0) --> VDN number
                        s41p4int( record.recBuf_gg00(), _pos + 1, 0, _numerr );
                    else
                        // int(1) --> VDN number
                        s41p4int( record.recBuf_gg00(), _pos + 1, 1, _numerr );
                    record.recBuf_gg00()[ _pos - 1 ] = csp_defined_byte;
                    if (( op_f_count     == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() || 
                          op_f_dis_count == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ) 
                          &&
                          (*m_RecordDesc)[ _out_pos - 1 ].elen_var() > RESNUM_MXSP00 )
                    {
                        SAPDB_RangeFill( __FILE__,   3, 
                              sizeof( record ), &record, _pos + RESNUM_MXSP00,
                              (*m_RecordDesc)[ _out_pos - 1 ].elen_var() - RESNUM_MXSP00,
                              csp_defined_byte, _b_err );
                    }
                }
                else
                {
                    // op_f_min/max, op_f_[dis_]sum 
                    // op_f_check_null
                }
            }
        }
        // result record processed
        if ( e_ok == _b_err )
        {
            if ( _check_null_pos > 0 )
            {
                if ( _null_found )
                    record.recBuf_gg00()[ _check_null_pos - 1 ] = csp_undef_byte;
                else
                    record.recBuf_gg00()[ _check_null_pos - 1 ] = csp_defined_byte;
            }
            SAPDBTRACE_IF( Join_Trace, 3, 
            t01buf( td_always, &record, 1, record.recLen_gg00() <= 300 ? record.recLen_gg00() : 300 ));
            _b_err = this->add_group_record( record );
        }
    }
    return _b_err;
}

/*!
 *
 */
tgg00_BasisError Join_IResultSetAggregator::update_group( tgg00_Rec& record )
{
    SAPDBTRACE_METHOD_DEBUG( "Join_IResultSetAggregator::update_group", Join_Trace, 3 );
    tgg00_BasisError _b_err = e_ok;
    SAPDB_UInt4 _func_pos = (*m_RecordDesc)[ 0 ].epos() - 1;
    SAPDB_UInt4 _out_pos, _check_null_pos = 0, _disfile_cnt = 0;
    bool _null_found = false, _replace_rec = false;
    tgg00_StackOpFunc _operator;
    SAPDBTRACE_WRITELN( Join_Trace, 7, "new record " );
    SAPDBTRACE_IF( Join_Trace, 7, 
    t01buf( td_always, &record, 1, record.recLen_gg00() <= 300 ? record.recLen_gg00() : 300 ));
    SAPDBTRACE_WRITELN( Join_Trace, 7, "group record " );
    SAPDBTRACE_IF( Join_Trace, 7, 
    t01buf( td_always, &m_GroupRec, 1, m_GroupRec.recLen_gg00() <= 300 ? m_GroupRec.recLen_gg00() : 300 ));

    // loop over functions
    while (( _func_pos - (*m_RecordDesc)[ _func_pos - 1 ].epos() > 1) 
            && ( e_ok == _b_err ))
    {
        _func_pos -= (*m_RecordDesc)[ _func_pos - 1 ].epos();
        // loop over LASTFUNCTION
        while (( st_func == (*m_RecordDesc)[ _func_pos - 1 ].etype() ) 
                && ( op_f_none == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ))
            _func_pos -= (*m_RecordDesc)[ _func_pos - 1 ].epos();
        _out_pos = k721out_entry( const_cast<tgg00_StackList*>(m_RecordDesc), _func_pos );
        if ( st_func == (*m_RecordDesc)[ _func_pos - 1 ].etype() )
        {
            _operator = (*m_RecordDesc)[ _func_pos - 1 ].eop_func();
            if ( op_f_check_null == _operator )
                _check_null_pos = (*m_RecordDesc)[ _out_pos - 1 ].epos();
            if ( csp_undef_byte != record.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1 ]  || 
                 op_f_all_count == _operator )
            {
                bool _get_act_value = false;

                if ( csp_oflw_byte == m_GroupRec.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1 ] &&
                     op_f_all_count != _operator )
                {
                    // overflow byte in group record
                    _replace_rec = false;
                    // do nothing more
                }
                else
                {
                    // no overflow byte in group record || op_f_all_count
                    if ( csp_oflw_byte == record.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1 ] &&
                         !( op_f_all_count == _operator ||
                            op_f_count     == _operator ||
                            op_f_dis_count == _operator ) ) 
                    {
                        // overflow byte w/o COUNT operator
                        _replace_rec = true;
                        m_GroupRec.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1 ] = csp_oflw_byte;
                        // do nothing more
                    }
                    else
                    {
                        // no null/overflow byte || op_f_*_count
                        if ( op_f_min == _operator || op_f_max == _operator )
                        {
                            if ( csp_undef_byte == m_GroupRec.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1 ])
                                _get_act_value = true;
                            else
                            {
                                tsp00_LcompResult _lres;
                                s30cmp( &record,  
                                        (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1, 
                                        (*m_RecordDesc)[ _out_pos - 1 ].elen_var() - 1,
                                        &m_GroupRec,
                                        (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1, 
                                        (*m_RecordDesc)[ _out_pos - 1 ].elen_var() - 1,
                                        _lres );
                                SAPDBTRACE_IF( Join_Trace, 3, 
                                t01buf( td_always, &record, (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1,  (*m_RecordDesc)[ _out_pos - 1 ].epos() + (*m_RecordDesc)[ _out_pos - 1 ].elen_var() - 1 ));
                                SAPDBTRACE_WRITELN( Join_Trace, 3, "comp res: " << _lres );
                                SAPDBTRACE_IF( Join_Trace, 3, 
                                t01buf( td_always, &m_GroupRec, (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1,  (*m_RecordDesc)[ _out_pos - 1 ].epos() + (*m_RecordDesc)[ _out_pos - 1 ].elen_var() - 1 ));
                                if (( op_f_min == _operator && l_less == _lres ) ||
                                   ( op_f_max == _operator && l_greater == _lres ))
                                    _get_act_value = true;
                            }
                        }
                        else
                        {
                            if ( op_f_dis_avg   == _operator ||
                                 op_f_dis_sum   == _operator ||
                                 op_f_dis_count == _operator ||
                                 op_f_stddev    == _operator ||
                                 op_f_dis_stddev == _operator ||
                                 op_f_variance  == _operator ||
                                 op_f_dis_variance == _operator )
                            {
                                tgg00_Rec     _dist_buffer;
                                tgg00_FileId  _htree( m_DistinctFileId );
                                
                                // move key=<group value> to _dist_buffer
                                SAPDB_RangeMove( __FILE__, 4,
                                        sizeof( m_GroupKey ), sizeof(_dist_buffer),
                                        &m_GroupKey.k(), 1,
                                        &_dist_buffer.recBody_gg00(), 1,
                                        m_GroupKey.len(), _b_err );
                                // move value to aggregate to _dist_buffer
                                SAPDB_RangeMove( __FILE__, 5,
                                        sizeof( record ), sizeof( _dist_buffer ),
                                        &record, (*m_RecordDesc)[ _out_pos - 1 ].epos(),
                                        &_dist_buffer.recBuf_gg00(), 
                                        m_GroupKey.len() - RESCNT_MXGG04 + cgg_rec_key_offset + 1,
                                        (*m_RecordDesc)[ _out_pos - 1 ].elen_var(), 
                                        _b_err );
                                // set key = <group value, value>
                                _dist_buffer.recKeyLen_gg00() = m_GroupKey.len() - RESCNT_MXGG04 + (*m_RecordDesc)[ _out_pos - 1 ].elen_var();
                                // set record length
                                _dist_buffer.recLen_gg00() = _dist_buffer.recKeyLen_gg00() + cgg_rec_key_offset;
                                _dist_buffer.recVarcolOffset_gg00() = 0;
                                _dist_buffer.recVarcolCnt_gg00() = 0;
                                
                                if ( op_f_stddev        == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                                     op_f_dis_stddev    == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                                     op_f_variance      == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||
                                     op_f_dis_variance  == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() )
                                {
                                    // write 1 with swapping 
                                    s20ch4sw( 1, g01code.kernel_swap,
                                          _dist_buffer.recBody_gg00(),
                                          _dist_buffer.recKeyLen_gg00() + 1,
                                          sw_normal);
                                    _dist_buffer.recLen_gg00() += RESCNT_MXGG04;
                                }

                                _htree.fileRoot_gg00() = m_DistinctFileRoot[ _disfile_cnt ].m_Root;
                                _htree.fileRootCheck_gg00() = m_DistinctFileRoot[ _disfile_cnt ].m_RootCheck;
                                _htree.fileTempCnt_gg00() = ++_disfile_cnt;
                                
                                if ( csp_oflw_byte != record.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1 ] )
                                {
                                    if ( e_ok == _b_err )
                                    {
                                        b07cadd_record( m_acv.a_transinf.tri_trans, _htree, _dist_buffer );
                                        _b_err = m_acv.a_transinf.tri_trans.trError_gg00;
                                        if ( e_duplicate_key == m_acv.a_transinf.tri_trans.trError_gg00 )
                                            m_acv.a_transinf.tri_trans.trError_gg00 = e_ok;
                                    }

                                    if ( e_duplicate_key == _b_err && 
                                       ( op_f_stddev   == _operator || 
                                         op_f_variance == _operator ))
                                    {
                                        tgg00_Lkey    _stddev_key;
                                        tgg00_FilePos _htreepos;
                                        _htreepos.tpsPno_gg00 = NIL_PAGE_NO_GG00; 
                                        _b_err = m_acv.a_transinf.tri_trans.trError_gg00 = e_ok;
                                        _stddev_key.len() = _dist_buffer.recKeyLen_gg00();
                                        
                                        SAPDB_RangeMove( __FILE__, 6,
                                                sizeof( _dist_buffer ), sizeof( _stddev_key.k() ),
                                                &_dist_buffer, cgg_rec_key_offset + 1,
                                                &_stddev_key.k(), 1,
                                                _dist_buffer.recKeyLen_gg00(), 
                                                _b_err );
                                        if ( e_ok == _b_err )
                                            b07ctget_record( m_acv.a_transinf.tri_trans, _htree, _htreepos, _stddev_key, _dist_buffer );
                                        // update counter
                                        s20ch4sw( 1 +
                                              s20or4sw( &_dist_buffer,
                                              _dist_buffer.recKeyLen_gg00() + cgg_rec_key_offset + 1,
                                              sw_normal, g01code.kernel_swap ),
                                              g01code.kernel_swap,
                                              _dist_buffer.recBody_gg00(),
                                              _dist_buffer.recKeyLen_gg00() + 1, sw_normal);
                                        if ( e_ok == _b_err )
                                            b07ctrepl_record( m_acv.a_transinf.tri_trans, _htree, _htreepos, _dist_buffer );
                                    }
                                }
                                else
                                    // overflow byte found
                                    _b_err = e_duplicate_key;
                                if ( e_duplicate_key == _b_err )
                                {
                                    _b_err = m_acv.a_transinf.tri_trans.trError_gg00 = e_ok;
                                }
                                else
                                {
                                    if ( e_ok == _b_err )
                                    {
                                        switch ( _operator )
                                        {
                                            case op_f_dis_avg:
                                            case op_f_stddev:
                                            case op_f_dis_stddev:
                                            case op_f_variance:
                                            case op_f_dis_variance:
                                                       _operator.becomes(  op_f_avg ); break;
                                            case op_f_dis_sum:
                                                       _operator.becomes( op_f_sum ); break;
                                            case op_f_dis_count:
                                                       _operator.becomes( op_f_count ); break;
                                        }
                                    }
                                }
                            }
                            else
                            {
                                // op_f_avg, op_f_sum, op_f_count
                                // op_f_all_count, op_f_check_null
                            }
                        }
                        tsp00_NumError _numerr;
                        _numerr.becomes( num_ok );

                        if (( op_f_avg == _operator || 
                              op_f_sum == _operator ) 
                           && e_ok == _b_err )
                        {
                            if ( csp_undef_byte == m_GroupRec.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1 ] )
                            {
                                // value for group is null
                                _get_act_value = true;
                            }
                            else
                            {
                                tsp00_NumError _numerr;
                                SAPDB_Int4 _length;
                                
                                _replace_rec = true;
                                if ( _out_pos + 1 != k721out_entry( const_cast<tgg00_StackList*>(m_RecordDesc), _out_pos ) )
                                {
                                    s51add ( &record,
                                          /* new value to aggregate */
                                          (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1,
                                          (*m_RecordDesc)[ _out_pos - 1 ].elen_var() - 1,
                                          /* old aggregate value */
                                          &m_GroupRec,
                                          (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1,
                                          (*m_RecordDesc)[ _out_pos - 1 ].elen_var(),
                                          /* new aggregate value */
                                          &m_GroupRec,
                                          (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1,
                                          (*m_RecordDesc)[ _out_pos - 1 ].epos(),
                                          (*m_RecordDesc)[ _out_pos - 1 ].elen_var(),
                                          _length, _numerr );
                                }
                                else
                                {
                                    if (st_sum_length == (*m_RecordDesc)[ _func_pos - 2 ].etype())
                                    {
                                        s51add ( &record,
                                            /* new value to aggregate */
                                            (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1,
                                            (*m_RecordDesc)[ _out_pos - 1 ].elen_var() - 1,
                                            /* old aggregate value */
                                            &m_GroupRec,
                                            (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1,
                                            (*m_RecordDesc)[ _out_pos - 1 ].elen_var() - 1,
                                            /* new aggregate value */
                                            &m_GroupRec,
                                            (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1,
                                            (*m_RecordDesc)[ _func_pos- 2 ].epos(),
                                            (*m_RecordDesc)[ _func_pos- 2 ].elen_var(),
                                            _length, _numerr );
                                    }
                                    else
                                    {
                                        s51add ( &record,
                                        /* new value to aggregate */
                                        (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1,
                                        (*m_RecordDesc)[ _out_pos - 1 ].elen_var() - 1,
                                        /* old aggregate value */
                                        &m_GroupRec,
                                        (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1,
                                        (*m_RecordDesc)[ _out_pos - 1 ].elen_var() - 1,
                                        /* new aggregate value */
                                        &m_GroupRec,
                                        (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1,
                                        csp_fixed /* whole amount of numbers*/,
                                        csp_float_frac /* amount of fraction numbers of above */,
                                        _length, _numerr );
                                    }
                                }
                                if ( _numerr != num_ok )
                                {
                                    // not all overflowes become num_overflow !!! 
                                    // i.e.:
                                    // 9.9999999999999999999999999999999999999E+62 
                                    // + 11 will fail !!! 
                                    // it produces pattern shown further down
                                    // try to sum up to the end 
                                    // don't stop other aggregation functions
                                    m_GroupRec.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1     ] = csp_oflw_byte;
                                    m_GroupRec.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1 + 1 ] = 0xFF;
                                    for ( SAPDB_Int1 _i = 1; _i <= (*m_RecordDesc)[ _out_pos - 1 ].elen_var() - 2; ++_i )
                                        m_GroupRec.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1 + 1 + _i ] = 0x99;
                                }
                            }
                        }
                        if (( op_f_avg      == _operator || 
                              op_f_count    == _operator || 
                              op_f_all_count == _operator ) 
                           && e_ok == _b_err )
                        {
                            SAPDB_UInt4 _pos;
                            SAPDB_Int4  _cnt; 
                            
                            _replace_rec = true;
                            if ( op_f_avg == _operator )
                                _pos = (*m_RecordDesc)[ _out_pos - 1 ].epos() + NUMBER_MXGG04 + 1;
                            else
                                _pos = (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1;
                            // get old aggregate value
                            s40g4int( &m_GroupRec, _pos, _cnt, _numerr );
                            if ( num_ok == _numerr )
                                s41p4int( &m_GroupRec, _pos, _cnt + 1, _numerr );
                        }
                        if ( _get_act_value )
                        {
                            SAPDB_RangeMove( __FILE__, 7,
                                    sizeof( record ), sizeof( m_GroupRec ),
                                    &record, 
                                    (*m_RecordDesc)[ _out_pos - 1 ].epos(),
                                    &m_GroupRec, 
                                    (*m_RecordDesc)[ _out_pos - 1 ].epos(),
                                    (*m_RecordDesc)[ _out_pos - 1 ].elen_var(), 
                                    _b_err );
                            _replace_rec = true;
                        }
                    }
                }
            }
            else
            {
                // null value found && op != op_f_all_count
                _null_found = true;
                m_acv.a_transinf.tri_trans.trWarning_gg00.addElement(warn0_exist);
                m_acv.a_transinf.tri_trans.trWarning_gg00.addElement(warn2_null_in_builtin_func);
            }
        }
    }
    // result record processed
    if ( _null_found && _check_null_pos > 0 )
    {
        if ( csp_defined_byte == m_GroupRec.recBuf_gg00()[ _check_null_pos - 1 ] )
        {
            // previous NULL not found
            m_GroupRec.recBuf_gg00()[ _check_null_pos - 1 ] = csp_undef_byte;
            _replace_rec = true;
        }
    }
    if ( _replace_rec && e_ok == _b_err )
    {
        SAPDBTRACE_IF( Join_Trace, 3, 
        t01buf( td_always, &m_GroupRec, 1, m_GroupRec.recLen_gg00() <= 300 ? m_GroupRec.recLen_gg00() : 300 ));
        _b_err = this->update_group_record();
    }
    return _b_err;
}

/* ******************** PRIVATE MEMBERS ******************** */
/*!
 *
 */
void Join_IResultSetAggregator::post_processing()
{
    SAPDBTRACE_METHOD_DEBUG( "Join_IResultSetAggregator::post_processing", Join_Trace, 3 );

    tgg00_BasisError _b_err = e_ok;
    bool _repl_needed;
    SAPDB_UInt4 _cnt_repl = 0;
    SAPDB_UInt4 _func_pos, _out_pos;
    SAPDB_UInt4 _disfile_cnt = 0;
    SAPDB_UInt4 _res_cnt = 0;
    tgg00_Rec _buffer;
    Container_Vector<SAPDB_Int4>::Iterator _iter;

    this->init_result_scan();
    do {
        SAPDBTRACE_IF( Join_Trace, 5, memset( &_buffer, ':', sizeof( _buffer ) ) );
        _func_pos = (*m_RecordDesc)[ 0 ].epos() - 1;
        _b_err = this->fetch_next_result( _buffer );
        
        if ( m_acv.a_transinf.tri_trans.trRteCommPtr_gg00->to_cancel )
            _b_err = e_cancelled;
        if ( e_ok == _b_err )
        {
            ++_res_cnt;
            _repl_needed = false;
            
            if ( !m_RecordCounterPositions.IsEmpty() )
            {
                tsp00_NumError _ret;
        
                // put ROWNO
                for ( _iter = m_RecordCounterPositions.Begin(); _iter != m_RecordCounterPositions.End(); ++_iter) 
                {
                    _buffer.buf()[ *_iter - 1 ] = csp_defined_byte;
                    s41plint( _buffer.buf(), *_iter + 1,
                          csp_resnum_deflen, 0, _res_cnt, _ret );
                }
                _repl_needed = true;
            }

            while (( _func_pos - (*m_RecordDesc)[ _func_pos - 1 ].epos() > 1) 
                    && ( e_ok == _b_err ))
            {
                _func_pos -= (*m_RecordDesc)[ _func_pos - 1 ].epos();
                _out_pos = k721out_entry( const_cast<tgg00_StackList*>(m_RecordDesc), _func_pos );
                if ( st_func == (*m_RecordDesc)[ _func_pos - 1 ].etype() )
                {
                    if  (
                        (
                        (op_f_avg       == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() || 
                        op_f_dis_avg    == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                        op_f_sum        == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                        op_f_dis_sum    == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() )
                        &&
                        (sfh_all == m_CallPostProcessing)
                        )                                                              ||
                        op_f_stddev     == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                        op_f_dis_stddev == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                        op_f_variance   == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                        op_f_dis_variance == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() 
                        )
                    {
                        if ( csp_undef_byte != _buffer.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1 ] && 
                            csp_undef_byte == _buffer.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1 + 1 ] )
                        {
                            // check for s51add() errors 
                            SAPDB_UInt4 _i = (*m_RecordDesc)[ _out_pos - 1 ].epos() + 2;
                            bool _overflow = true;
                            do {
                                if ( 0x99 != _buffer.recBuf_gg00()[ _i - 1 ] )
                                    _overflow = false;
                                ++_i;
                            } while (!( _i == (*m_RecordDesc)[ _out_pos - 1 ].epos() + 10 || !_overflow ));
                            if ( _overflow )
                            {
                                _buffer.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 + 1 ].epos() - 1 ] = csp_oflw_byte;
                                _repl_needed = true;
                            }
                        }
                    }
                    if ( op_f_avg        == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() || 
                         op_f_dis_avg    == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                         op_f_stddev     == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                         op_f_dis_stddev == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                         op_f_variance   == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                         op_f_dis_variance == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() )
                    {
                        if ( csp_undef_byte != _buffer.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1 ] && 
                             csp_oflw_byte  != _buffer.recBuf_gg00()[ (*m_RecordDesc)[ _out_pos - 1 ].epos() - 1 ] )
                        {
                            tsp00_NumError _numerr;
                            SAPDB_Int4 _length;
                            SAPDBTRACE_WRITELN( Join_Trace, 3, "s51div " );
#ifdef SAPDB_SLOW
                            t01buf( td_always, &_buffer, 
                                    (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1, 
                                    (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1 + NUMBER_MXGG04 - 1 );
                            t01buf( td_always, &_buffer, 
                                    (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1 + NUMBER_MXGG04, 
                                    (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1 + NUMBER_MXGG04 + RESNUM_MXSP00 - 1 );
#endif
                            s51div( &_buffer,
                                  (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1, 
                                  NUMBER_MXGG04 - 1,
                                  &_buffer, 
                                  (*m_RecordDesc)[ _out_pos - 1 ].epos() + NUMBER_MXGG04 + 1,
                                  RESNUM_MXSP00 - 1,
                                  &_buffer, 
                                  (*m_RecordDesc)[ _out_pos - 1 ].epos() + 1,
                                  csp_fixed, csp_float_frac, _length, _numerr );
                            _repl_needed = true;
                            if ( op_f_stddev     == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                                 op_f_dis_stddev == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                                 op_f_variance   == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() ||    
                                 op_f_dis_variance == (*m_RecordDesc)[ _func_pos - 1 ].eop_func() )
                            {
                                tgg00_FileId _htree( m_DistinctFileId );
                                tgg00_Rec _buf1, _buf2;
                                SAPDBTRACE_IF( Join_Trace, 5, memset( &_buf1, ':', sizeof( _buf1 ) ) );
                                SAPDBTRACE_IF( Join_Trace, 5, memset( &_buf2, ':', sizeof( _buf2 ) ) );
                                _htree.fileRoot_gg00() = m_DistinctFileRoot[ _disfile_cnt ].m_Root;
                                _htree.fileRootCheck_gg00() = m_DistinctFileRoot[ _disfile_cnt ].m_RootCheck;
                                _htree.fileTempCnt_gg00() = ++_disfile_cnt;
                                
                                k721stddev( &m_acv.a_transinf.tri_trans, _htree, 
                                        _buffer, _buf1, &_buf2, 
                                        (*m_RecordDesc)[ _func_pos - 1 ].eop_func(),
                                        (*m_RecordDesc)[ _out_pos - 1 ].epos() );
                            }
                        }
                    }
                }
            }
            // call later output
            if ( m_LaterOutputPos > 0 )
            {
                SAPDBTRACE_WRITELN( Join_Trace, 3, "put 'later out': " 
                        << (*m_RecordDesc)[ m_LaterOutputPos - 1 - 1 ].epos() 
                        << "[" << (*m_RecordDesc)[ m_LaterOutputPos - 1 ].elen_var() << "]" 
                        << " --> " << (*m_RecordDesc)[ m_LaterOutputPos - 1 ].epos() );
                SAPDB_RangeMove( __FILE__, 8,
                        sizeof( _buffer ), sizeof( _buffer ),
                        &_buffer, 
                        (*m_RecordDesc)[ m_LaterOutputPos - 1 - 1 ].epos(),
                        &_buffer, 
                        (*m_RecordDesc)[ m_LaterOutputPos - 1 ].epos(),
                        (*m_RecordDesc)[ m_LaterOutputPos - 1 ].elen_var(), 
                        m_acv.a_transinf.tri_trans.trError_gg00 );
            }

            SAPDBTRACE_IF( Join_Trace, 5, 
            t01buf( td_always, &_buffer, 1, _buffer.recLen_gg00() <= 300 ? _buffer.recLen_gg00() : 300 ));
            if ( _repl_needed && ( e_ok == (_b_err = update_result( _buffer)) )) ++_cnt_repl;
        }
    } while ( e_ok == _b_err && e_ok == m_acv.a_transinf.tri_trans.trError_gg00 );
    
    if ( e_no_next_record == m_acv.a_transinf.tri_trans.trError_gg00 )
        m_acv.a_transinf.tri_trans.trError_gg00 = e_ok;
    b73cmds_count( iins_sorts_rows_ins, _cnt_repl );
}

