/*!
 * @file
 * @brief implements Join_AccessOperator
 *
 * @author GertG
 * @ingroup Join
 *
 * @par last changed by:
 * <br>
 * $Author: d024980 $ $DateTime: 2004/05/03 16:16:26 $
 *
 * @sa Join_AccessOperator.hpp
 */
/*

    ========== licence begin  GPL
    Copyright (c) 2002-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/
#include "SAPDB/Join/Join_IAccessOperator.hpp"
#include "ggg00.h"
#include "gsp00.h"
#include "hak07.h"
#include "hgg04.h"
#include "hta01.h"
#include "hta01_3.h"
#include "SAPDB/Trace/Trace_Entry.hpp"
#include "Trace/Trace_MessBlockEntry.hpp"
#include "SAPDBCommon/SAPDB_Types.hpp"

#include "SAPDB/Join/Join_AccessOperator.hpp"

#include <stdlib.h>

/* ******************** PUBLIC MEMBERS ********************* */
/*!
 * @param acv [in] global context
 * @param buffersize [in] size of table buffer
 */
Join_AccessOperator::Join_AccessOperator(
        tak_all_command_glob& acv,
        SAPDB_Byte*           bufferPtr,
        const SAPDB_UInt4&    buffersize)
    :
    m_allocator(*(REINTERPRET_CAST( SAPDBMem_IRawAllocator*, acv.a_transinf.tri_trans.trAllocator_gg00) )),
    Join_IAccessOperator(acv),
    m_RecBuf(bufferPtr),
    m_RecBufSize(buffersize),
    m_Record(),
    m_foreignRecBuf(0 != bufferPtr),
    m_AllRecsInBuf(0),
    m_FirstBufferFill(true),
    m_BufferReusagePossible(false)
{
    SAPDBTRACE_METHOD_DEBUG( "Join_AccessOperator::ctor", Join_Trace, 1 );

    if ( 0 != m_acv.a_returncode )
        return;

    SAPDBTRACE_WRITELN( Join_Trace, 5, "m_RecBufSize: " << m_RecBufSize );

    tgg00_BasisError _b_err = e_ok;

    m_AccessDesc.MBlock().mb_trns()->trRteCommPtr_gg00->file_record_cnt = 1;

    if ( ! m_foreignRecBuf ) {
        if ( m_RecBufSize < GetRecordLength() )
        {
            m_RecBufSize = GetRecordLength();
            SAPDBTRACE_WRITELN(
                Join_Trace, 3, "m_RecBufSize adjusted: " << m_RecBufSize );
        }

        if (!( m_RecBuf = GetAlloc().Allocate( m_RecBufSize )))
            _b_err = e_no_more_memory;
    }

    if ( _b_err != e_ok )
        a07_b_put_error( acv, _b_err, 1 );
}

/*!
 *
 */
Join_AccessOperator::~Join_AccessOperator()
{
    SAPDBTRACE_METHOD_DEBUG( "Join_AccessOperator::dtor", Join_Trace, 1 );
    if ( ! m_foreignRecBuf && m_RecBuf )
        GetAlloc().Deallocate( m_RecBuf );
}

/*!
 *
 */
void Join_AccessOperator::Open()
{
    SAPDBTRACE_METHOD_DEBUG( "Join_AccessOperator::Open", Join_Trace, 1 );
    tgg00_BasisError _b_err;
    Join_TwoKeys _old_startkeys(m_Startkeys), _old_stopkeys(m_Stopkeys);

    IOperator::Open();

    _b_err = m_AccessDesc.CreateKeys( m_Startkeys, m_Stopkeys );
    if ( e_ok == _b_err ) _b_err = base_init();

    if ( _b_err != e_ok )
        a07_b_put_error( m_acv, _b_err, 1 );
    else
        check_buffer_reusage( _old_startkeys, _old_stopkeys );
}

/*!
 * @param startkeys [in] restrict record stream to records with key greater/equal than startkey
 * @param stopkeys [in] restrict record stream to records with key lower/equal than stopkey
 */
void Join_AccessOperator::Open(
            const Join_TwoKeys& startkeys,
            const Join_TwoKeys& stopkeys )
{
    SAPDBTRACE_METHOD_DEBUG( "Join_AccessOperator::Open(startkeys,stopkeys)", Join_Trace, 1 );
    tgg00_BasisError _b_err;

    Join_TwoKeys _old_startkeys(m_Startkeys), _old_stopkeys(m_Stopkeys);

    IOperator::Open( startkeys, stopkeys );
    _b_err = base_init();

    if ( _b_err != e_ok )
        a07_b_put_error( m_acv, _b_err, 1 );
    else
        check_buffer_reusage( _old_startkeys, _old_stopkeys );

#ifdef SAPDB_SLOW
    if ( Join_Trace.TracesLevel( 3 ) )
    {
        t01key( td_always, "start.rec   ", m_Startkeys.reckey );
        t01key( td_always, "stop.rec    ", m_Stopkeys.reckey );
        t01key( td_always, "start.inv   ", m_Startkeys.listkey );
        t01key( td_always, "stop.inv    ", m_Stopkeys.listkey );
    }
#endif
}

/*!
 * @param record [in/out] pointer to memory should be filled with record
 * @return information code <tt>[e_ok, e_no_next_record]</tt> / error code
 */
tgg00_BasisError Join_AccessOperator::Next( tgg00_Rec*& recptr )
{
    SAPDBTRACE_METHOD_DEBUG( "Join_AccessOperator::Next", Join_Trace, 1 );

    m_AccessDesc.MBlock().mb_trns()->trError_gg00 = e_ok;

    if ( is_empty_buffer() )
    {
        if ( e_no_next_record == m_BDState )
        {
            SAPDBTRACE_WRITELN( Join_Trace, 3, "file empty!" );
            m_AccessDesc.MBlock().mb_trns()->trError_gg00 = e_no_next_record;
            m_BDState = e_ok;
        }

        if ( e_ok == m_AccessDesc.MBlock().mb_trns()->trError_gg00 )
        {
            SAPDBTRACE_WRITELN( Join_Trace, 3, "fill buffer" );
            SAPDBTRACE_IF( Join_Trace, 3, memset(m_RecBuf, ':', m_RecBufSize) );
            m_AccessDesc.MBlock().mb_trns()->trRteCommPtr_gg00->file_root = NIL_PAGE_NO_GG00;

            m_SelFields.sfp_result_length() = 0;
            m_SelFields.sfp_m_result_len()  = 0;
            m_SelFields.sfp_m_result_size() = m_RecBufSize;
            m_SelFields.sfp_m_result_addr() = reinterpret_cast<tsp00_MoveObjPtr>( m_RecBuf );
            m_SelFields.sfp_m_result_cnt()  = MAX_INT4_SP00; // set buffer count

            // child classes need to implement this:
            this->SelectRecords();


            // bd delivers e_buffer_limit if one record fits not at all
            // we can't get this error because auf m_RecBufSize >= BUF8L_MXSP00
            m_BDState = m_AccessDesc.MBlock().mb_trns()->trError_gg00;

            SAPDBTRACE_WRITELN( Join_Trace, 5, "BD call: " << SAPDBTrace::BasisError(m_AccessDesc.MBlock().mb_trns()->trError_gg00) );
            if ( ( e_no_next_record == m_BDState && m_SelFields.sfp_m_result_cnt() > 0 ))
                m_AccessDesc.MBlock().mb_trns()->trError_gg00 = e_ok;

            SAPDBTRACE_WRITELN( Join_Trace, 5, "records from BD: " << m_SelFields.sfp_m_result_cnt() );

            if ( e_no_next_record == m_BDState && m_FirstBufferFill )
            {
                SAPDBTRACE_WRITELN( Join_Trace, 5, "all recs in buffer " );
                m_AllRecsInBuf = m_SelFields.sfp_m_result_cnt();
                m_BufferReusagePossible = true;
            }
            m_FirstBufferFill = false;


            // get first record from buffer
            m_Record = reinterpret_cast<SAPDB_Byte*>( m_RecBuf );
        }
    }
    else
    {
        SAPDBTRACE_WRITELN( Join_Trace, 3, "get record from buffer" );
        SAPDBTRACE_WRITELN( Join_Trace, 5, "records in buffer: " << m_SelFields.sfp_m_result_cnt() );
        // get next record from buffer
        m_Record += m_SelFields.sfp_resrec_maxlen();
    }
    SAPDBTRACE_WRITELN(
        Join_Trace, 3,
        "Next(): "
        << SAPDBTrace::BasisError(
            m_AccessDesc.MBlock().mb_trns()->trError_gg00) );

#ifdef SAPDB_SLOW
    if ( Join_Trace.TracesLevel( 5 )
         && e_ok == m_AccessDesc.MBlock().mb_trns()->trError_gg00 )
    {
        t01surrogate(
            td_always, "sel TABID   ",
            m_AccessDesc.MBlock().mb_qual()->mtree().fileTabId_gg00());
        SAPDBTRACE_WRITELN(
            Join_Trace, 5,
            "records (reclen="
            << m_SelFields.sfp_resrec_maxlen()
            << "[" << GetRecordLength()-cgg_rec_key_offset << "]) are: " );
        t01buf( td_always, m_Record, 1,
                m_SelFields.sfp_m_result_cnt()
                * m_SelFields.sfp_resrec_maxlen()
                <= 300
                ? (m_SelFields.sfp_m_result_cnt()
                   * m_SelFields.sfp_resrec_maxlen() + cgg_rec_key_offset)
                : 300 );
    }
#endif
    --m_SelFields.sfp_m_result_cnt();
    recptr = reinterpret_cast<tgg00_Rec*>( m_Record );

    return m_AccessDesc.MBlock().mb_trns()->trError_gg00;
}

/*!
 *
 */
void Join_AccessOperator::Close()
{
    SAPDBTRACE_METHOD_DEBUG( "Join_AccessOperator::Close", Join_Trace, 1 );

    // possibly release table lock

    m_AccessDesc.ResetFileID();
    IOperator::Close();
}

SAPDB_Bool Join_AccessOperator::IsResultFullyBuffered() const
{
    return m_BufferReusagePossible;
}


SAPDB_UInt4 Join_AccessOperator::GetBufferedRecordCount() const
{
    return m_AllRecsInBuf;
}

/* ******************* PROTECTED MEMBERS ******************* */


/* ******************** PRIVATE MEMBERS ******************** */
/*!
 *
 */
tgg00_BasisError Join_AccessOperator::init()
{
    SAPDBTRACE_METHOD_DEBUG( "Join_AccessOperator::init", Join_Trace, 3 );

    // initialize select fields parameter
    tgg00_BasisError _b_err = e_ok;

    m_Actualkeys = m_Startkeys;

    m_RangeSet.clear();

    m_BDState = e_ok;

    if ( m_Startkeys.reckey.len() != 0 )
        m_RangeSet.addElement( primary_start );

    if ( m_Stopkeys.reckey.len() != 0
         && m_Stopkeys.reckey.k()[0] != csp_undef_byte )
        m_RangeSet.addElement( primary_stop );

    g04init_select_fields( m_SelFields, 
            &m_AccessDesc.MBlock().mb_data()->mbp_buf(), 
            m_AccessDesc.MBlock().mb_data_size(), 
            m_AccessDesc.MBlock().mb_valuearr(), 
            m_AccessDesc.MBlock().mb_validx_max(), 
            m_AccessDesc.MBlock().mb_work_st(), 
            m_AccessDesc.MBlock().mb_work_st_max(), 
            m_AccessDesc.MBlock().mb_workbuf(), 
            m_AccessDesc.MBlock().mb_workbuf_size(),
            m_AccessDesc.SQLMode() );

    // siehe vkb720 kb720sel_first_part()
    m_SelFields.sfp_bd_mess_type().becomes( m_select );
    m_SelFields.sfp_bd_mess2_type().becomes( mm_first );
    m_SelFields.sfp_result_wanted() = true;
    m_SelFields.sfp_is_first_record() = true;
    m_SelFields.sfp_bd_use_stopkey() = m_RangeSet.includes( primary_stop );
    m_SelFields.sfp_bd_inv_only().becomes( m_AccessDesc.Strategy()->str_qual_kind );
    m_SelFields.sfp_check_for_result() = false; // siehe k75_fetch
    m_SelFields.sfp_rows_read() = 0;
    m_SelFields.sfp_rows_qual() = 0;
    m_SelFields.sfp_act_cntresult() = 0;
    m_SelFields.sfp_m_result_cnt() = 0;
    m_SelFields.sfp_acv_addr() = reinterpret_cast<tsp00_Addr>( &m_acv );
    // we overlap records with cgg_rec_key_offset
    // (because of sfp_resrec_maxlen = reclen without cgg_rec_key_offset)
    m_SelFields.sfp_resrec_maxlen() = GetRecordLength() - cgg_rec_key_offset;
    if ( 1 == m_SelFields.sfp_resrec_maxlen() % 2 ) ++m_SelFields.sfp_resrec_maxlen();

    // possibly request table lock

    return _b_err;
}

/* ******************** PRIVATE MEMBERS ******************** */
/*!
 *
 */
inline
bool Join_AccessOperator::is_same_key(
        const Join_TwoKeys& old_key,
        const Join_TwoKeys& new_key) const
{
    if ( old_key.reckey.keyLen_gg00() == new_key.reckey.keyLen_gg00() )
    {
        short _diff = memcmp(&old_key.reckey.keyVal_gg00(), &new_key.reckey.keyVal_gg00(), new_key.reckey.keyLen_gg00());
#ifdef SAPDB_SLOW
        if ( _diff )
            SAPDBTRACE_WRITELN( Join_Trace, 7, "different key content" );
#endif
        if ( (!_diff) && m_AccessDesc.IsIndexAccess() )
        {
            if ( old_key.listkey.keyLen_gg00() == new_key.listkey.keyLen_gg00() )
            {
                _diff += memcmp(&old_key.listkey.keyVal_gg00(), &new_key.listkey.keyVal_gg00(), new_key.listkey.keyLen_gg00());
#ifdef SAPDB_SLOW
                if ( _diff )
                    SAPDBTRACE_WRITELN( Join_Trace, 7, "different index content" );
#endif
            }
            else
                // diff length of secondary keys
                return false;
        }
        return !_diff;
    }
    // diff length of primary keys
    return false;
}

/*!
 *
 */
void Join_AccessOperator::check_buffer_reusage(
        const Join_TwoKeys& old_startkeys,
        const Join_TwoKeys& old_stopkeys )
{
    SAPDBTRACE_METHOD_DEBUG( "Join_BufferedAccessOperator::check_buffer_reusage", Join_Trace, 5 );
    if ( m_BufferReusagePossible )
    {
        if ( is_same_key( old_startkeys, m_Startkeys ) &&
             is_same_key( old_stopkeys , m_Stopkeys ) )
        {
            SAPDBTRACE_WRITELN( Join_Trace, 5, "reuse previous buffer content" );
            // reset content of buffer
            m_SelFields.sfp_m_result_cnt() = this->GetBufferedRecordCount();
            m_BDState = e_no_next_record;
            // step virtually one record befor first record
            m_Record = reinterpret_cast<SAPDB_Byte*>( m_RecBuf ) -
                m_SelFields.sfp_resrec_maxlen();
        }
        else
            m_BufferReusagePossible = false;
    }
}
