/*!
  @file           SQLDBC_ProxyRuntime.cpp
  @author         D039759
  @ingroup        SQLDBC
  @brief          Special Runtime for MySQL proxy.
  @see            

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2001-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
\endif
*/

#include "Interfaces/SQLDBC/SQLDBC_ProxyRuntime.h"
#include "Interfaces/Runtime/Packet/IFRPacket_RequestPacket.h"
#include "Interfaces/Runtime/Packet/IFRPacket_RequestSegment.h"
#include "Interfaces/Runtime/Packet/IFRPacket_ReplyPacket.h"
#include "Interfaces/Runtime/Packet/IFRPacket_ReplySegment.h"
#include "Interfaces/Runtime/Packet/IFRPacket_Part.h"

#ifdef WIN32
#  define STRICMP _stricmp
#else
#  define STRICMP strcasecmp
#endif

// Type definitions for C initialisation functions. Sun complains
// about a cast because it wants a C function type.
extern "C" {

typedef void (*SQLGETTLS_InitFuncType)(void*);
typedef void* (*SQLONCE_InitFuncType)(void*);

}


// The MySQL hashed password used to connect is 9 bytes long:
//  - 1 byte algorithm (00 = old, 01 = new)
//  - 8 byte hash (computed by MySQL client) 
#define MYSQL_HASHED_PASSWORD_LENGTH  9

#define OOM_ERROR(messagelist)                                          \
SAPDBErr_MessageList __tmp__messagelist("SQLDBC",                       \
                                         __FILE__,                      \
                                         __LINE__,                      \
                                         SAPDBErr_MessageList::Error,   \
                                         -10760,                        \
                                         "Memory allocation failed.",   \
                                         0);                            \
messagelist.AppendNewMessage(__tmp__messagelist);

//----------------------------------------------------------------------
SQLDBC_ProxyRuntime::SQLDBC_ProxyRuntime()
:pendingSessionsCount(0)
{
    memset(pendingSessions, 0, sizeof(pendingSessions));
    for(SQLDBC_size_t i=0; i< sizeof(pendingSessions)/sizeof(SessionInfo); ++i) {
        pendingSessions[i].sessionid = -1;
    }
    SAPDBErr_MessageList ignored;
    createMutex(pendingSessionsMutex, getGlobalAllocator(), ignored);
}

//----------------------------------------------------------------------
SQLDBC_ProxyRuntime::~SQLDBC_ProxyRuntime()
{
    // NEVER CALLED
}


//----------------------------------------------------------------------
const char *
SQLDBC_ProxyRuntime::getIdentfier() const
{
    return "SQLDBC Proxy Runtime";
}

//----------------------------------------------------------------------
SAPDB_Bool 
SQLDBC_ProxyRuntime::checkKeyOption(const char *keyoption,
                                    char *servernode,
                                    SAPDB_Int4& servernodelength,
                                    char *serverdb,
                                    SAPDB_Int4& serverdblength,
                                    SAPDB_Int4& isolationlevel,
                                    SAPDB_Int4& cachelimit,
                                    SAPDB_Int4& sqlmode,
                                    SAPDB_Int4& timeout,
                                    char *username,
                                    SAPDB_Int4& usernamelength,
                                    SQLDBC_StringEncoding& usernameEncoding,
                                    SAPDBErr_MessageList& errorMessages)
{
    return SQLDBC_ClientRuntime::checkKeyOption(keyoption, 
                                                servernode,
                                                servernodelength,
                                                serverdb,
                                                serverdblength,
                                                isolationlevel,
                                                cachelimit,
                                                sqlmode,
                                                timeout,
                                                username,
                                                usernamelength,
                                                usernameEncoding,
                                                errorMessages);
}

//----------------------------------------------------------------------
SAPDBMem_IRawAllocator& 
SQLDBC_ProxyRuntime::getGlobalAllocator()
{
    return SQLDBC_ClientRuntime::getGlobalAllocator();
}
//----------------------------------------------------------------------
SAPDB_Bool 
SQLDBC_ProxyRuntime::getSession(const char *connectUrl,
                                const char *connectCommand,
                                const char *password,
                                SAPDB_Int4  passwordLength,
                                const SQLDBC_StringEncoding commandEncoding,
                                SAPDB_Int8&  sessionID,
                                SAPDB_UInt4& packetSize,
                                SAPDB_Int4& packetListSize,
                                void **packetList,  
                                void **sessionInfoReply,
                                SAPDBErr_MessageList& errorMessages,
                                SAPDBMem_IRawAllocator* allocator)
{
    // Use global allocator if no allocator passed.
    if(allocator == 0) {
        allocator = &getGlobalAllocator();
    }
    IFR_Bool memory_ok=true;
    IFR_ConnectProperties properties(*allocator, memory_ok);
    if(!memory_ok) {
        return false;
    }
    if(!parseConnectURL(connectUrl, 
                        properties,
                        errorMessages)) {
        return false;
    }
    
    const char *sessionid = properties.getProperty("SESSIONID");
    if(sessionid == 0) {
        SAPDBErr_MessageList tmpMessageList("SQLDBC",
                                            __FILE__,
                                            __LINE__,
                                            SAPDBErr_MessageList::Error,
                                            0,
                                            "No session id found.",
                                            0);
        errorMessages.AppendNewMessage(tmpMessageList);
        return false;
    }
    char *errp;
    SAPDB_Int8 prop_sessionid = strtol(sessionid, &errp, 0);
    if(*errp && errno!=ERANGE) {
        char errmesg[80];
        sp77sprintf(errmesg, 80, "Invalid session number %s specified.", sessionid);
        SAPDBErr_MessageList tmpMessageList("SQLDBC",
                                            __FILE__,
                                            __LINE__,
                                            SAPDBErr_MessageList::Error,
                                            0,
                                            errmesg,
                                            0);
        errorMessages.AppendNewMessage(tmpMessageList);
        return false;
    }
    SessionInfo *session=getSessionInfo(prop_sessionid);
    if(!session) {
        char errmesg[80];
        sp77sprintf(errmesg, 80, "Invalid session number %s specified.", sessionid);
        SAPDBErr_MessageList tmpMessageList("SQLDBC",
                                            __FILE__,
                                            __LINE__,
                                            SAPDBErr_MessageList::Error,
                                            0,
                                            errmesg,
                                            0);
        errorMessages.AppendNewMessage(tmpMessageList);
        return false;
    }

    packetList[0]  = session->packetlist[0];
    packetList[1]  = session->packetlist[1];
    packetSize     = session->packetsize;
    packetListSize = 1;
    sessionID = session->sessionid;

    if(!packetList[0]) {
        char errmesg[80];
        sp77sprintf(errmesg, 80, "No request packet available.");
        SAPDBErr_MessageList tmpMessageList("SQLDBC",
                                            __FILE__,
                                            __LINE__,
                                            SAPDBErr_MessageList::Error,
                                            0,
                                            errmesg,
                                            0);
        errorMessages.AppendNewMessage(tmpMessageList);
        return false;
    }

    int  client_unicode=0;
    const char *unicode_str=properties.getProperty(IFR_CONNECTPROPERTY_UNICODE, "0");
    if(STRICMP(unicode_str, "TRUE")==0 || atoi(unicode_str)) {
        client_unicode=1;
    } else {
        client_unicode=0;
    }
    
    // no applencrypt, the password that is provided is already what we need ...
    // password is 
    if(passwordLength != MYSQL_HASHED_PASSWORD_LENGTH) {
        SAPDBErr_MessageList tmpMessageList("SQLDBC",
                                            __FILE__,
                                            __LINE__,
                                            SAPDBErr_MessageList::Error,
                                            0,
                                            "Invalid length of password hash.",
                                            0);
        errorMessages.AppendNewMessage(tmpMessageList);
        releaseSession(sessionID, errorMessages);
        releaseSessionInfo(session);
        return false;
    }
    
    char client_application[4];
    char client_version[6];
    int  client_sqlmode=IFR_INTERNAL;
    
    // -- check properties for packet construction
    const char *client_application_property=properties.getProperty(IFR_CONNECTPROPERTY_APPLICATION);
    strncpy(client_application, client_application_property, 3);
    client_application[3]='\0';
    
    const char *client_version_property=properties.getProperty(IFR_CONNECTPROPERTY_APPVERSION);
    strncpy(client_version, client_version_property, 5);
    client_version[5]='\0';
    
    const char *sqlmode_str=properties.getProperty(IFR_CONNECTPROPERTY_SQLMODE, "INTERNAL");
    if((STRICMP(sqlmode_str, "ADABAS")==0) 
       || (STRICMP(sqlmode_str, "SAPDB")==0)
       || (STRICMP(sqlmode_str, "INTERNAL")==0)) {
        client_sqlmode=IFR_INTERNAL;
    } else if(STRICMP(sqlmode_str, "ORACLE")==0) {
        client_sqlmode=IFR_ORACLE;
    } else if(STRICMP(sqlmode_str, "ANSI")==0) {
        client_sqlmode=IFR_ANSI;
    } else if(STRICMP(sqlmode_str, "DB2")==0) {
        client_sqlmode=IFR_DB2;
    }
    
    
    IFRUtil_RuntimeItem runtimeitem((SQLDBC_ClientRuntime&)*this, *allocator);

    // -- create the packet
    IFRPacket_RequestPacket *requestpacket = 
        new (*allocator) IFRPacket_RequestPacket((tsp1_packet*)packetList[0],
                                                packetSize,
                                                client_unicode,
                                                client_sqlmode,
                                                client_application,
                                                client_version,
                                                 runtimeitem,
                                                 memory_ok);
    if(IFR_MEM_NOT_OK || requestpacket==0) {
        OOM_ERROR(errorMessages);
        return false;
    }
    IFRPacket_RequestSegment connectCommandSegment=IFRPacket_RequestSegment(*requestpacket,
                                                                            IFRPacket_CommandMessageType::Dbs_C);
    IFRPacket_CommandPart connect_cmdpart;
    connectCommandSegment.addPart(connect_cmdpart);
    IFR_String connectCommandString(connectCommand, commandEncoding, *allocator, memory_ok);
    if(!memory_ok) {
        OOM_ERROR(errorMessages);
        return false;
    }
    IFR_ErrorHndl tmperror(*allocator);
    if (IFR_OK != connect_cmdpart.setText(connectCommandString, tmperror)) {
      SAPDBErr_MessageList tmpMessageList("SQLDBC",
                                          __FILE__,
                                          __LINE__,
                                          SAPDBErr_MessageList::Error,
                                          tmperror.getErrorCode(),
                                          tmperror.getErrorText(),
                                          0);
      errorMessages.AppendNewMessage(tmpMessageList);
      return false;      
    }
    IFRPacket_DataPart connect_datapart;    
    connectCommandSegment.addPart(connect_datapart);
    
    connect_datapart.AddArgument();
    unsigned char *dest=connect_datapart.GetWriteData();
    *dest=csp_defined_byte;
    ++dest;
    memcpy(dest, password, MYSQL_HASHED_PASSWORD_LENGTH);
    connect_datapart.ExtendLength(1 + MYSQL_HASHED_PASSWORD_LENGTH);
    dest += MYSQL_HASHED_PASSWORD_LENGTH;
    tsp00_TermId termid;
    sqltermid(termid);
    *dest=csp_defined_byte;
    ++dest;
    memcpy (dest, termid, sizeof(tsp00_TermId));
    connect_datapart.ExtendLength (1 + sizeof (tsp00_TermId));
    connectCommandSegment.closePart();
    connectCommandSegment.close();
    // -- send the packet 
    
    SAPDB_Bool success = request(sessionID,
                                 requestpacket->GetRawPacket(),
                                 requestpacket->Length(),
                                 errorMessages);
    if(success) {
        SAPDB_Int4 replyLength;
        success = receive(sessionID,
                          sessionInfoReply,
                          replyLength,
                          errorMessages);
    }

    if(!success) {
        releaseSessionInfo(session);
        releaseSession(sessionID, errorMessages);
        sessionID = -1;
    }

    IFRUtil_Delete(requestpacket, *allocator);
    
    // Free the slot.
    releaseSessionInfo(session);
    
    return success;
}

//----------------------------------------------------------------------
SAPDB_Bool 
SQLDBC_ProxyRuntime::releaseSession(SAPDB_Int8 sessionID,
                                    SAPDBErr_MessageList& errorMessages)
{
    return SQLDBC_ClientRuntime::releaseSession(sessionID, errorMessages);
}

//----------------------------------------------------------------------
SAPDB_Bool 
SQLDBC_ProxyRuntime::request(SAPDB_Int8  sessionID,
                             void       *requestData,
                             SAPDB_UInt4 requestDataLength,
                             SAPDBErr_MessageList& errorMessages)
{
    return SQLDBC_ClientRuntime::request(sessionID, 
                                         requestData,
                                         requestDataLength, 
                                         errorMessages);
}

//----------------------------------------------------------------------
SAPDB_Bool 
SQLDBC_ProxyRuntime::receive(SAPDB_Int8 sessionID,
                             void **replyData,
                             SAPDB_Int4& replyDataLength,
                             SAPDBErr_MessageList& errorMessages)
{
    return SQLDBC_ClientRuntime::receive(sessionID,
                                         replyData,
                                         replyDataLength,
                                         errorMessages);
}
//----------------------------------------------------------------------
SAPDB_Bool 
SQLDBC_ProxyRuntime::cancelCurrentCommand(SAPDB_Int8 sessionID,
                                          SAPDBErr_MessageList& errorMessages)
{
    return SQLDBC_ClientRuntime::cancelCurrentCommand(sessionID, errorMessages);
}

//----------------------------------------------------------------------
SAPDB_Bool 
SQLDBC_ProxyRuntime::createMutex(MutexHandle& mutexHandle, 
                                 SAPDBMem_IRawAllocator& allocator,
                                 SAPDBErr_MessageList& errorMessages)
{
    return SQLDBC_ClientRuntime::createMutex(mutexHandle, allocator, errorMessages);
}

//----------------------------------------------------------------------
SAPDB_Bool 
SQLDBC_ProxyRuntime::lockMutex(MutexHandle mutexHandle)
{
    return SQLDBC_ClientRuntime::lockMutex(mutexHandle);
}

//----------------------------------------------------------------------
SAPDB_Bool 
SQLDBC_ProxyRuntime::releaseMutex(MutexHandle mutexHandle)
{
    return SQLDBC_ClientRuntime::releaseMutex(mutexHandle);
}

//----------------------------------------------------------------------
SAPDB_Bool 
SQLDBC_ProxyRuntime::destroyMutex(MutexHandle& mutexHandle, 
                                  SAPDBMem_IRawAllocator& allocator,
                                  SAPDBErr_MessageList& errorMessages)
{
    return SQLDBC_ClientRuntime::destroyMutex(mutexHandle,
                                              allocator,
                                              errorMessages);
}

//----------------------------------------------------------------------
SAPDB_Bool  
SQLDBC_ProxyRuntime::createCounter(CounterHandle& counterHandle,
                                   SAPDBMem_IRawAllocator& allocator,
                                   SAPDBErr_MessageList& errorMessages)
{
    return SQLDBC_ClientRuntime::createCounter(counterHandle, allocator, errorMessages);
}

//----------------------------------------------------------------------
SAPDB_UInt4 
SQLDBC_ProxyRuntime::nextCounter(CounterHandle counterHandle)
{
    return SQLDBC_ClientRuntime::nextCounter(counterHandle);
}

//----------------------------------------------------------------------
SAPDB_Bool  
SQLDBC_ProxyRuntime::destroyCounter(CounterHandle counterHandle,
                                    SAPDBMem_IRawAllocator& allocator,
                                    SAPDBErr_MessageList& errorMessages)
{
    return SQLDBC_ClientRuntime::destroyCounter(counterHandle, 
                                                allocator,
                                                errorMessages);
}

//----------------------------------------------------------------------
SQLDBC_IRuntime::TaskID 
SQLDBC_ProxyRuntime::getCurrentTaskID()
{
    return SQLDBC_ClientRuntime::getCurrentTaskID();
}
//----------------------------------------------------------------------
SAPDB_Bool 
SQLDBC_ProxyRuntime::createSemaphore(SemaphoreHandle& semaphoreHandle,
                                     SAPDB_Int4 initialValue,
                                     SAPDBMem_IRawAllocator& allocator,
                                     SAPDBErr_MessageList errorMessages)
{
    return SQLDBC_ClientRuntime::createSemaphore(semaphoreHandle,
                                                 initialValue,
                                                 allocator,
                                                 errorMessages);
}

//----------------------------------------------------------------------
SAPDB_Bool 
SQLDBC_ProxyRuntime::waitSemaphore(SemaphoreHandle semaphoreHandle)
{
    return SQLDBC_ClientRuntime::waitSemaphore(semaphoreHandle);
}

//----------------------------------------------------------------------
SAPDB_Bool 
SQLDBC_ProxyRuntime::signalSemaphore(SemaphoreHandle semaphoreHandle)
{
    return SQLDBC_ClientRuntime::signalSemaphore(semaphoreHandle);
}

//----------------------------------------------------------------------
SAPDB_Bool 
SQLDBC_ProxyRuntime::destroySemaphore(SemaphoreHandle& semaphoreHandle,
                                      SAPDBMem_IRawAllocator& allocator,
                                      SAPDBErr_MessageList& errorMessages)
{
    return SQLDBC_ClientRuntime::destroySemaphore(semaphoreHandle, 
                                                  allocator, 
                                                  errorMessages);
}

//----------------------------------------------------------------------    
SQLDBC_IRuntime::TaskTraceContext* 
SQLDBC_ProxyRuntime::getTaskTraceContext()
{
    return SQLDBC_ClientRuntime::getTaskTraceContext();
}

//----------------------------------------------------------------------
void 
SQLDBC_ProxyRuntime::getFlags(char *flags)
{
  SQLDBC_ClientRuntime::getFlags(flags);
}

//----------------------------------------------------------------------
void 
SQLDBC_ProxyRuntime::addTraceFlags(unsigned int flags)
{
  SQLDBC_ClientRuntime::addTraceFlags(flags);
}

//----------------------------------------------------------------------
void 
SQLDBC_ProxyRuntime::removeTraceFlags(unsigned int flags)
{
  SQLDBC_ClientRuntime::removeTraceFlags(flags);
}

//----------------------------------------------------------------------
void 
SQLDBC_ProxyRuntime::write(const char *s, SAPDB_Int4 size)
{
    SQLDBC_ClientRuntime::write(s, size);
}

//----------------------------------------------------------------------
void 
SQLDBC_ProxyRuntime::writeln(const char *s, SAPDB_Int4 size)
{
    SQLDBC_ClientRuntime::writeln(s, size);
}

//----------------------------------------------------------------------
SAPDB_Bool 
SQLDBC_ProxyRuntime::getChallenge(const char    *servernode,
                                  const char    *serverdb,
                                  SAPDB_Int8&    sessionid,
                                  char          *challenge,
                                  SQLDBC_Length& challengeLength,
                                  char          *errortext,
                                  SQLDBC_Length  errortextLength)
{
    SAPDBErr_MessageList messages;
    char urlbuffer[256];
    if(!servernode || *servernode=='\0') {
        servernode="localhost";
    }
    sprintf(urlbuffer, "sapdbc://%s/%s", servernode, serverdb);
    SAPDB_Bool rc=getChallenge(urlbuffer,
                               sessionid,
                               challenge,
                               challengeLength,
                               messages,
                               0);
    if(!rc) {
        if(errortext) {
            strncpy(errortext, messages.Message(), errortextLength);
            errortext[errortextLength-1]='\0';
        }
    }
    return rc;
}




//----------------------------------------------------------------------
SAPDB_Bool 
SQLDBC_ProxyRuntime::getChallenge(const char    *connectURL,
                                  SAPDB_Int8&    sessionid,
                                  char          *challenge,
                                  SQLDBC_Length& challengeLength,
                                  SAPDBErr_MessageList& errorMessages,
                                  SAPDBMem_IRawAllocator* allocator)
{
    sessionid = -1;
    if(allocator == 0) {
        allocator = &getGlobalAllocator();
    }

    SessionInfo *session=acquireSessionInfo();
    IFR_Bool memory_ok=true;
    IFR_ConnectProperties properties(*allocator, memory_ok);
    if(!memory_ok) {
        return false;
    }
    if(!parseConnectURL(connectURL, 
                        properties,
                        errorMessages)) {
        releaseSessionInfo(session);
        return false;
    }
    
    // look whether a compname is present, and use it for the sqlinit
    const char *compname_property=properties.getProperty(IFR_CONNECTPROPERTY_COMPNAME, "R ifr80");
    tsp00_CompName compname;
    memset(compname, ' ', sizeof(tsp00_CompName));
    strncpy (compname, compname_property, sizeof(tsp00_CompName));
    compname[63]='\0';
    // make a runtime initialisation
    sqlinit (compname, NULL);
    tsp00_ErrTextc     errortext;
    tsp01_CommErr_Enum commerr;
    
    session->packetlist[0]=session->packetlist[1]=0;
    
    char *servernode = (char *)properties.getProperty(IFR_CONNECTPROPERTY_HOSTNAME, "");
    if(properties.getProperty(IFR_CONNECTPROPERTY_PORT)) {
        char *new_servernode = (char *)alloca(strlen(servernode) + 10);
        strcpy(new_servernode, servernode);
        strcat(new_servernode, ":");
        strcat(new_servernode, properties.getProperty(IFR_CONNECTPROPERTY_PORT));
        servernode = new_servernode;
    }
    const char *serverdb   = properties.getProperty(IFR_CONNECTPROPERTY_DATABASE, "");
    
    m_connectlock.lockConnect();

    SAPDB_Int4 sqlaconnect_sessionid=-1;
    commerr = SqlAConnect((char *)servernode,
                          REFCAST_MEO00 (tsp00_DbNamec) serverdb,                   
                          srvUser_esp01,
                          1,
                          &sqlaconnect_sessionid,
                          (tsp00_Int4 *)&session->packetsize,
                          session->packetlist,
                          errortext);
    m_connectlock.releaseConnect();
    session->sessionid = sqlaconnect_sessionid;
    sessionid = session->sessionid;
    
    if(commerr != commErrOk_esp01) {
        IFR_String errStr(errortext, 
                          sizeof(tsp00_ErrText), 
                          IFR_StringEncodingAscii, 
                          *allocator,
                          memory_ok);
        SAPDBErr_MessageList tmpMessageList("SQLDBC",
                                            __FILE__,
                                            __LINE__,
                                            SAPDBErr_MessageList::Error,
                                            0,
                                            errStr.getBuffer(),
                                            0);
        errorMessages.AppendNewMessage(tmpMessageList);
        sessionid = -1;
        releaseSessionInfo(session);
        return false;
    }
    
    IFRUtil_RuntimeItem runtimeitem((SQLDBC_ClientRuntime&)*this, *allocator);

    // -- create the packet
    IFRPacket_RequestPacket *requestpacket = 
        new (*allocator) IFRPacket_RequestPacket((tsp1_packet*)session->packetlist[0],
                                                 session->packetsize,
                                                 0,
                                                 IFR_INTERNAL,
                                                 "ODB",
                                                 "70404",
                                                 runtimeitem,
                                                 memory_ok);
    if(requestpacket == 0 || IFR_MEM_NOT_OK) {
        IFRUtil_Delete(requestpacket, *allocator);
        OOM_ERROR(errorMessages);
        return false;
    }
    IFRPacket_RequestSegment getChallengeSegment=IFRPacket_RequestSegment(*requestpacket,
                                                                          IFRPacket_CommandMessageType::GetChallenge_C);
    getChallengeSegment.close();
    SQLDBC_Bool success = request(session->sessionid,
                                  requestpacket->GetRawPacket(),
                                  requestpacket->Length(),
                                  errorMessages);
    SAPDB_Int4 replylength;
    void      *getchallengereply;
    if(success) {
        success = receive(session->sessionid,
                          &getchallengereply,
                          replylength,
                          errorMessages);
    }

    if(!success) {
        releaseSession(session->sessionid, errorMessages);
        sessionid =-1;
        releaseSessionInfo(session);
        return false;
    }
    
    IFRPacket_ReplyPacket replypacket;
    replypacket.setPacket((tsp1_packet*)getchallengereply, requestpacket->getLock());
    IFRPacket_ReplySegment firstsegment(replypacket);
    IFRPacket_DataPart datapart;
    IFR_Retcode part_rc=firstsegment.getPart(datapart);
    if(part_rc == IFR_NO_DATA_FOUND) {
        SAPDBErr_MessageList tmpMessageList("SQLDBC",
                                            __FILE__,
                                            __LINE__,
                                            SAPDBErr_MessageList::Error,
                                            0,
                                            "No challenge part found in database reply.",
                                            0);
        errorMessages.AppendNewMessage(tmpMessageList);
        releaseSession(session->sessionid, errorMessages);
        sessionid =-1;
        releaseSessionInfo(session);
        return false;
    }
    if(challengeLength < datapart.bufferLength()) {
        char text[80];
        sp77sprintf(text, 
                    80, 
                    "Length of buffer for challenge too small, need (%d) bytes.", 
                    datapart.bufferLength());
        SAPDBErr_MessageList tmpMessageList("SQLDBC",
                                            __FILE__,
                                            __LINE__,
                                            SAPDBErr_MessageList::Error,
                                            0,
                                            text,
                                            0);
        errorMessages.AppendNewMessage(tmpMessageList);
        releaseSession(session->sessionid, errorMessages);
        sessionid =-1;
        releaseSessionInfo(session);
        return false;
    }
    sessionid = session->sessionid;
    challengeLength = datapart.bufferLength();
    memcpy(challenge, datapart.GetReadData(0), challengeLength);
    replypacket.setPacket(0,0);
    IFRUtil_Delete(requestpacket, *allocator);
    return true;
}

//----------------------------------------------------------------------
SQLDBC_ProxyRuntime::SessionInfo*
SQLDBC_ProxyRuntime::acquireSessionInfo()
{
    lockMutex(pendingSessionsMutex);
    if(pendingSessionsCount == sizeof(pendingSessions)/sizeof(SessionInfo)) {
        releaseMutex(pendingSessionsMutex);
        return 0;
    }
    pendingSessionsCount++;
    for(SQLDBC_size_t i=0; i<sizeof(pendingSessions)/sizeof(SessionInfo); ++i) {
        if(pendingSessions[i].sessionid == -1) {
            pendingSessions[i].sessionid = -2;
            releaseMutex(pendingSessionsMutex);
            return pendingSessions + i;
        }
    }
    releaseMutex(pendingSessionsMutex);
    return 0;
}

//----------------------------------------------------------------------
SQLDBC_ProxyRuntime::SessionInfo*
SQLDBC_ProxyRuntime::getSessionInfo(SAPDB_Int8 sessionid)
{
    lockMutex(pendingSessionsMutex);
    for(SQLDBC_size_t i=0; i<sizeof(pendingSessions)/sizeof(SessionInfo); ++i) {
        if(pendingSessions[i].sessionid == sessionid) {
            releaseMutex(pendingSessionsMutex);
            return pendingSessions + i;
        }
    }
    releaseMutex(pendingSessionsMutex);
    return 0;
}

//----------------------------------------------------------------------
void
SQLDBC_ProxyRuntime::releaseSessionInfo(SessionInfo *sessioninfo)
{
    lockMutex(pendingSessionsMutex);
    memset(sessioninfo, 0, sizeof(SessionInfo));
    sessioninfo->sessionid = -1;
    pendingSessionsCount--;
    releaseMutex(pendingSessionsMutex);
}

//----------------------------------------------------------------------
static SQLDBC_ProxyRuntime *ProxyRuntime_Instance;

SAPDBMEM_STATIC_RAW_ALLOCATE(ProxyRuntime_InstanceData, 
                             sizeof(SQLDBC_ProxyRuntime));
static SQLDBC_Int4 ProxyRuntime_Initialized;

static void *initializeProxyRuntime(void *)
{
    teo07_Thread thread;
    tsp00_ErrTextc errortext;
    teo07_ThreadErr threaderror;
    sqlinit_multithreading(&thread,
                           errortext,
                           &threaderror);
    void *p = (void *)ProxyRuntime_InstanceData;
    ProxyRuntime_Instance = new (p) SQLDBC_ProxyRuntime();
    return 0;
}

//----------------------------------------------------------------------
static SQLDBC_ProxyRuntime*
SQLDBC_ProxyRuntimeInit()
{
    sqlonce(&ProxyRuntime_Initialized,
            (SQLONCE_InitFuncType)initializeProxyRuntime,
            0);
    
    return ProxyRuntime_Instance;
}

//----------------------------------------------------------------------
SQLDBC_IRuntime *
GetProxyRuntime(char *errorText, const SQLDBC_Int4 errorTextSize)
{
    return (SQLDBC_IProxyRuntime *)SQLDBC_ProxyRuntimeInit();
}

//----------------------------------------------------------------------
SQLDBC_Bool 
SQLDBC_Proxy_GetChallenge(SQLDBC_IRuntime *proxyRuntime,
                          const char *servernode,
                          const char *serverdb,
                          SQLDBC_Int8&   sessionid,
                          char          *challenge,
                          SQLDBC_Length& challengeLength,
                          char          *errortext,
                          SQLDBC_Length  errortextLength)
{
    SQLDBC_IProxyRuntime *rt=(SQLDBC_IProxyRuntime*)proxyRuntime;
    return rt->getChallenge(servernode, serverdb, (SAPDB_Int8&)sessionid, challenge, challengeLength, errortext, errortextLength);
}


