/*!
  @file           SQLDBC_ClientTrace.h
  @author         D039759
  @ingroup        SQLDBC
  @brief          Additional trace functionality only of use with Client Runtime.
  
  @see            

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2001-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
\endif
*/
#ifndef SQLDBC_CLIENTTRACE_H
#define SQLDBC_CLIENTTRACE_H

#include "Interfaces/Runtime/IFR_Types.h"
#include "Interfaces/Runtime/IFR_Trace.h"
#include "Interfaces/SQLDBC/SQLDBC_ClientRuntime.h"

#define SQLDBC_TraceStream IFR_TraceStream

enum SQLDBC_TraceLevel
{
    SQLDBC_TRACE_NONE          = 0,
    SQLDBC_TRACE_CALL          = 1,
    SQLDBC_TRACE_SHORT         = 1,
    SQLDBC_TRACE_LONG          = 3,
    SQLDBC_TRACE_DEBUG         = 2,
    SQLDBC_TRACE_PACKET        = 4,
    SQLDBC_TRACE_PROFILE       = 8,
    SQLDBC_TRACE_SQL           = 16,
    SQLDBC_TRACE_APPLICATION_LEVEL_1 = 32,
    SQLDBC_TRACE_APPLICATION_LEVEL_2 = 64,
    SQLDBC_TRACE_APPLICATION_LEVEL_3 = 128
};

/**
 * @internal 
 * @brief Interface to the trace for internal SQLDBC users.
 */ 
class SQLDBC_ClientTrace
    : public IFR_ITraceController
{
public:
    /**
     * Creates a new client trace object.
     * @param runtime The reference to the runtime.
     */
    SQLDBC_ClientTrace(SQLDBC_IRuntime* runtime);
    ~SQLDBC_ClientTrace();
    
    /**
     * Unconditional write. Writes always into the trace file.
     * @param data   The data that shall be traced, in UTF8 encoding.
     * @param length The length of data, in bytes.
     */
    virtual void write(const char *data, SQLDBC_Int4 length);

    /**
     * Unconditional write. Writes always into the trace file, and adds a newline.
     * @param data   The data that shall be traced, in UTF8 encoding.
     * @param length The length of data, in bytes.
     */
    virtual void writeln(const char *data, SQLDBC_Int4 length);
    
    void addTraceFlags(unsigned int flags);
    
    void removeTraceFlags(unsigned int flags);

private:
    virtual SQLDBC_IRuntime::TaskTraceContext *traceflags();
    virtual SAPDBMem_IRawAllocator& getAllocator();
    inline SQLDBC_TraceStream* tracestream() 
    {
        return &m_tracestream;
    }
    
    SQLDBC_ClientRuntime *m_clientruntime;
    SQLDBC_TraceStream    m_tracestream;

    friend SQLDBC_IRuntime::TaskTraceContext *IFR_GetTraceContext(SQLDBC_ClientTrace *trace);
    friend SQLDBC_TraceStream *IFR_GetTraceStream(SQLDBC_ClientTrace *trace);
    
};

//----------------------------------------------------------------------
inline SQLDBC_IRuntime::TaskTraceContext *
IFR_GetTraceContext(SQLDBC_ClientTrace *trace)
{
    if(trace) {
        return trace->traceflags();
    } else {
        return 0;
    }
}

inline SQLDBC_TraceStream *
IFR_GetTraceStream(SQLDBC_ClientTrace *trace)
{
    if(trace) {
        return trace->tracestream();
    } else {
        return 0;
    }
}

#define SQLDBC_TRACESTREAM(context, level) IFR_TRACESTREAM(context, level)


#endif
