/*!
  @file           IFR_FetchChunk.h
  @author         ThomasS
  @ingroup        IFR_Fetch
  @brief          Handles a piece of a resultset
  @see            

\if EMIT_LICENCE



    ========== licence begin  GPL
    Copyright (c) 2001-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end






\endif
*/
#ifndef IFR_FETCHCHUNK_H
#define IFR_FETCHCHUNK_H

#include "Interfaces/Runtime/IFR_Types.h"
#include "Interfaces/Runtime/Packet/IFRPacket_ReplySegment.h"

class IFR_FetchChunkDataPart;

/**
 * The outcome of a particular fetch operation.  A fetch operation
 * results in one (when the fetch size is 1) or more (when the fetch
 * size is >1) data rows returned from the database server. Depending on
 * the kind of the fetch, the positioning in the result at the database 
 * server and the start and end index computation does differ.
 */
class IFR_FetchChunk 
    : public IFRUtil_RuntimeItem
{
public:

  /**
   * Creates a new fetch chunk.
   * @param type the type of the fetch operation.
   * @param absoluteStartRow the start row of this chunk. If negative, this is calculated from 
   *   the end of the result set.
   * @param replyPacket the database server reply of the fetch
   * @param recordSize the size of one row.
   * @param maxRows the <code>m_MaxRows</code> property of the statement that created this result.
   * @param rowsInResultSet the number of rows in this result set, or -1 if not known.
   */
  IFR_FetchChunk (int type, 
		  int absoluteStartRow,
		  IFRPacket_ReplyPacket& replyPacket,
		  int recordSize,
		  int maxRows,
		  int rowsInResultSet);
  
    /**
     * Destructor. The copied part is not deleted, as the <code>IFR_ResultSet</code> manages it.
     */
    virtual ~IFR_FetchChunk(); 

  /**
   * Initialization of fetch chunk. Separate function, because constructor should not throw exceptions.
   *
   * @return <code>IFR_OK</code> if initialization ok; <code>IFR_NOT_OK</code> otherwise.
   */
  IFR_Retcode init ();

  /**
   * Gets the start index of the fetch chunk.
   *
   * @return The start index (smallest valid index).
   */
  int getStart ();

  /**
   * Gets the end index of the fetch chunk.
   *
   * @return The end index (largest valid index).
   */
  int getEnd ();

  /**
   * Sets the current record to the supplied absolute position.
   * @param row the absolute row. 
   * @return <code>true</code> if the row was set, <code>false</code> otherwise.
   */
  bool setRow (int row);

  /**
   * Updates the number of rows in the result set. 
   * @param rows the number of rows in the result set.
   */
  void setRowsInResultSet (int rows);

  /**
   * Get the current position within the result set.
   * @return the current position in the result set.
   */
  int getLogicalPos ();

  /**
   * Get the current offset within the result set.
   * @return the current offset in the result set.
   */
  int getCurrentOffset ();

  /**
   * Called because there is a result set where the last element
   * is now interesting. This is the fact in a <code>FETCH LAST</code>
   * operation.
   */
  void moveToUpperBound ();

  bool isForward ();

  /**
   * Returns whether this chunk is the first one.
   * <b>Take care, that this information may not be reliable.</b>
   * @return <code>true</code> if this is the first, and <code>false</code> if this
   *   is not first or the information is not known.
   */
  bool isFirst ();

  /**
   * Returns whether this chunk is the last one.
   * <b>Take care, that this information may not be reliable.</b>
   * @return <code>true</code> if this is the last, and <code>false</code> if this
   *   is not first or the information is not known.
   */
  bool isLast ();

  /**
   * Sets the <code>last</code> flag.
   * @param last the new value.
   */
  void setLast (bool last);

  /**
   * Gets the size of this chunk.
   * @return the number of rows in this chunk.
   */
  int size ();

  /**
   * Returns whether the given row is truly inside the chunk.
   * @param row the row to check. Rows <0 count from the end of the result.
   * @return <code>true</code> if the row is inside, <code>false</code> if it's not
   * or the condition could not be determined due to an unknown end of result set.
   */
  bool containsRow (int row);

  /**
   * Moves the position inside the chunk by a relative offset.
   * @param relativepos the relative moving offset.
   * @return <code>true</code> if it was moved, <code>false</code> otherwise.
   */
  bool move (int relativepos);

    /**
     * Returns the current data part positioned at the current position, 
     * hereby acquiring the data lock.
     * @param part Data part istance to which the data part is assigned.
     * @return <code>IFR_OK</code> if all went ok, <code>
     */
    IFR_Retcode getCurrentData (IFRPacket_DataPart& part);

  /**
   * The fetch operation type of a <tt>FETCH FIRST</tt>.
   */
  static int IFR_TYPE_FIRST;

  /**
   * The fetch operation type of a <tt>FETCH LAST</tt>.
   */
  static int IFR_TYPE_LAST;

  /**
   * The fetch operation type of a <tt>FETCH ABSOLUTE</tt> with an argument >1.
   */
  static int IFR_TYPE_ABSOLUTE_UP;

  /**
   * The fetch operation type of a <tt>FETCH ABSOLUTE</tt> with an argument <1.
   */
  static int IFR_TYPE_ABSOLUTE_DOWN;    

  /**
   * The fetch operation type of a <tt>FETCH RELATIVE</tt> with an argument >1.
   */
  static int IFR_TYPE_RELATIVE_UP;

  /**
   * The fetch operation type of a <tt>FETCH RELATIVE</tt> with an argument <1.
   */
  static int IFR_TYPE_RELATIVE_DOWN;
    

private:
    
  /**
   * Moves the position inside the chunk by a relative offset, but unchecked.
   * @param relativepos the relative moving offset.
   */
  void unsafeMove (int relativepos);

  /**
   * Determines whether this chunk is the first and/or last of
   * a result set. This is done by checking the index boundaries,
   * and also the LAST PART information of the reply packet.
   * A forward chunk is also the last if it contains the record at
   * the <code>maxRows</code> row, as the user decided to make
   * the limit here.
   * @param maxRows the <code>maxRows</code> limit of the statement
   */
  void determineFlags (int maxRows);

  /**
   * The type of the fetch operation (one of the <code>IFR_TYPE_XXX</code> constants).
   */
  int m_Type;    // type of fetch chunk

  /**
   * The number of bytes in a row.
   */
  int m_RecordSize;

  /**
   * The number of rows in the complete result set, or -1 if this is not known.
   */
  int m_RowsInResultSet;

  /**
   * The number of elements in this chunk.
   */
  int m_ChunkSize;

  /**
   * The index of the first row in this chunk. 
   */
  int m_StartIndex;

  /**
   * The index of the last row in this chunk.
   */
  int m_EndIndex;

  /**
   * The current index within this chunk, starting with 0.
   */
  int m_CurrentOffset;
  int m_AbsoluteStartRow;
  int m_MaxRows;

  /**
   * A flag indicating that this chunk is the last chunk of the result set.
   */
  bool m_Last;

  /**
   * A flag indicating that this chunk is the first chunk of the result set.
   */
  bool m_First;

    
    /**
     * The current record inside the data part (<code>m_ReplyData</code>).
     */
    IFR_Byte* m_CurrentRecord;

    IFRPacket_ReplySegment m_ReplySegment;  
    IFRPacket_ReplyPacket m_replypacket;  
    
    
    IFR_FetchChunkDataPart *m_copieddata;  //!< The copied data part from the original packet.
    friend class IFR_ResultSet;
};

#endif // IFR_FETCHCHUNK_H
