/*!
  @file   Data_PageAccessManager.hpp
  @author UweH
  @brief  The one and only page access manager in the kernel. (c++ interface to c and pascal coding)

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


\endif
*/
#ifndef DATA_PAGE_ACCESS_MANAGER_HPP
#define DATA_PAGE_ACCESS_MANAGER_HPP

#include "ggg00.h" // tgg00_TransContext
#include "gbd00.h" // tbd_current_tree
#include "DataAccess/Data_IPageAccessManager.hpp"
#include "DataAccess/Data_Page.hpp"
#include "DataAccess/Data_FileTypes.hpp"
/*!
    @class   Data_PageAccessManager
    @brief   encapsulates the page access manager (Data_Cache)
    @copydoc Data_IPageAccessManager
 */
class Data_PageAccessManager : public Data_IPageAccessManager
{
public:
    /*!
       @brief This initializes a context to access pages of a special type.
       @param TransContext [in/out] 
       @param FileType [in] 
       @param RootPageNo [in] 
     */
    Data_PageAccessManager (tgg00_TransContext  &TransContext,
                            Data_FileType           FileType,
                            Data_PageNo         RootPageNo);
    
    /*!
       @brief This initializes a context to access pages for a given file
       @param current [in/out] 
     */
    Data_PageAccessManager ( tbd_current_tree   &current )
    :m_Current( current )
    {}
    
    /*!
       @brief This initializes a context for and uses some internals from the given pam
       @param current [in/out] 
     */
    Data_PageAccessManager (Data_PageAccessManager  &Pam,
                            Data_FileType            FileType,
                            Data_PageNo              RootPageNo);
                            
    virtual bool GetPage (Data_Page                   &Page,
                          const Data_PageNo&           PageNo,
                          const Data_PageRecoveryMode &RecoveryMode,
                          Data_AccessMode              AccessMode);
    virtual bool GetPage (Data_Page                   &Page,
                          const Data_PageNo           &PageNo,
                          Data_AccessMode              AccessMode);
    virtual bool GetPage (Data_Page                   &Page,
                          const Data_PageNo           &PageNo);
    virtual void ReleasePage (Data_Page &Page);
    virtual bool CheckSpace (SAPDB_UInt pagecount ) const;
    virtual void NewPage (Data_Page&            Page,
                          Data_PageRecoveryMode Mode = Data_PageRecoveryMode());
    virtual void FreePage (Data_Page& Page);
    virtual void FreePageNo (const Data_PageNo&           PageNo,
                             const Data_PageRecoveryMode& RecoveryMode,
                             const SAPDB_Int4             PageConverterVersion);
    virtual void Invalidate();
    virtual bool IsValid() const;
    virtual void MarkPageIdAsUsed (const Data_PageId &pageid);
    virtual SAPDB_Int GetLastError () const;
    virtual void ResetLastError ();
    SAPDB_Int GetReferenceCount(){
        return m_Current.currCounter_bd00;
    }

private:
    /// The Accessmanagers can save themself here,so the page destructor can do a release.
    void RegisterToPage (Data_Page&                   Page,
                         tbd_node_ptrs&               Nptrs,
                         const Data_PageRecoveryMode& RecoveryMode,
                         Data_AccessMode              Mode);
	/// This is used by the vbd13 Interface.
    tbd_current_tree m_Current;
};
#endif // DATA_PAGE_ACCESS_MANAGER_HPP