/*!
  @file           Data_IPageAccessManager.hpp
  @author         UweH
  @brief          defines an interface to the Datacache

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
\endif
*/
#ifndef DATA_IPAGE_ACCESS_MANAGER_HPP
#define DATA_IPAGE_ACCESS_MANAGER_HPP

#include "SAPDBCommon/SAPDB_Types.hpp"
#include "DataAccess/Data_Types.hpp"

class Data_Page;

/*!
   @interface Data_IPageAccessManager
   @brief     This is an Interface to PageAccessManagers
 */
class Data_IPageAccessManager
{
public:
    /*!
        @brief  The page handle is set to a cached frame which is specified.
        @param  Page [out] 
        @param  PageNo [in] 
        @param  RecoveryMode [in] 
        @param  AccessMode [in] 
        @return true, if page could be found, else false

        The pointer to the frame in memory is given and a generic hint
     */
    virtual bool GetPage (Data_Page                   &Page,
                          const Data_PageNo&           PageNo,
                          const Data_PageRecoveryMode &RecoveryMode,
                          Data_AccessMode              AccessMode) = 0;
    /*!
        @brief  The page handle is set to a cached frame which is specified.
        @param  Page [out] 
        @param  PageNo [in] 
        @param  AccessMode [in] 
        @return true, if page could be found, else false
     */
    virtual bool GetPage (Data_Page                   &Page,
                          const Data_PageNo           &PageNo,
                          Data_AccessMode              AccessMode) = 0;
    /*!
        @brief  All parameters needed for getting the page are taken from Page.
        @param  Page [out] 
        @param  PageNo [in] 
        @return true, if page could be found, else false
     */
    virtual bool GetPage (Data_Page                   &Page,
                          const Data_PageNo           &PageNo) = 0;
    /// The specified page is released out of the cache.
    virtual void ReleasePage (Data_Page& Page) = 0;
    /*!
        @brief  This checks if the given amount of pages can be allocated.
        @param  pagecount [in] 
        @return true, if there are at least pagecount pages free.

        Before allocating new pages with NewPage this can be used to determine
        if there is space enough. pagecount is the amount of pages which should be allocated.
        If false is returned then the BasisError is set if there is not enough space.
     */
    virtual bool CheckSpace (SAPDB_UInt pagecount ) const = 0;
    /*!
        @brief This allocates a new page frame and assigns a new page id.
        @param Page [out] This is the handler for the new page frame.
        @param Mode [in] description: A new page with a unique PageId is given
     */
    virtual void NewPage (Data_Page&            Page,
                          Data_PageRecoveryMode Mode = Data_PageRecoveryMode()) = 0;
    /// A frame with a given Page handler is removed. The Page handler may be reused.
    virtual void FreePage (Data_Page& Page) = 0;
    /// A frame with a given page no is removed.
    virtual void FreePageNo (const Data_PageNo&           PageNo,
                             const Data_PageRecoveryMode& RecoveryMode,
                             const SAPDB_Int4             PageConverterVersion) = 0;
    /// This resets this handle
    virtual void Invalidate() = 0;
    /// If this handle is invalid false is returned.
    virtual bool IsValid() const = 0;
    /// This is used by cold verify
    virtual void MarkPageIdAsUsed (const Data_PageId &pageid) = 0;
    /// If any method fails this could offer some hints what was going wrong
    virtual SAPDB_Int GetLastError () const = 0;
    /// resets the last error to ok.
    virtual void ResetLastError () = 0;
};
#endif // DATA_IPAGE_ACCESS_MANAGER_HPP
