/*! 
  \file    DBMSrv_Reply.cpp
  \author  TiloH
  \ingroup common classes for the DBMServer
  \brief   definition of a class handling the reply buffer of DBMServer
           commands

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (C) 2002-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

\endif
*/


//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------

#include "SAPDB/DBM/Srv/DBMSrv_Reply.hpp"
#include "hcn36.h"


//-----------------------------------------------------------------------------
// file global variables
//-----------------------------------------------------------------------------

const size_t DummyBufferSize_DBMSrv_Reply=1204;
char         DummyBuffer_DBMSrv_Reply[DummyBufferSize_DBMSrv_Reply]; //<! a small dummy buffer, that is always available
int          DummyBufferLength;


//-----------------------------------------------------------------------------
// members of class DBMSrv_Reply
//-----------------------------------------------------------------------------

DBMSrv_Reply::DBMSrv_Reply()
  :replyData(DummyBuffer_DBMSrv_Reply),
   replyLength(&DummyBufferLength),
   maximalReplyLength(DummyBufferSize_DBMSrv_Reply)
{
}

DBMSrv_Reply::DBMSrv_Reply(char * ReplyData,
                           int  * ReplyLength,
                           int    ReplyLenMax)
  :replyData(ReplyData),
   replyLength(ReplyLength),
   maximalReplyLength(ReplyLenMax)
{
}

void DBMSrv_Reply::setTo(char * ReplyData,
                         int  * ReplyLength,
                         int    ReplyLenMax)
{
    replyData=ReplyData;
    replyLength=ReplyLength;
    maximalReplyLength=ReplyLenMax;
}

char * DBMSrv_Reply::giveData()
{
    return replyData;
}

const char * DBMSrv_Reply::giveData() const
{
    return replyData;
}

int * DBMSrv_Reply::giveLength()
{
    return replyLength;
}

void DBMSrv_Reply::adaptLength()
{
    *replyLength=(int)strlen(replyData);
}

int DBMSrv_Reply::giveMaximalLength() const
{
    return maximalReplyLength;
}

int DBMSrv_Reply::giveAvailableSpace() const
{
    return maximalReplyLength-1-(*replyLength);
}

void DBMSrv_Reply::startWithOK()
{
    startWithLine(ANSWER_OK_CN00);
}

void DBMSrv_Reply::startWithOKIfRCIsOk(tcn00_Error rc)
{
    if(OK_CN00==rc)
        startWithLine(ANSWER_OK_CN00);
}

tcn00_Error DBMSrv_Reply::startWithError(const tcn00_Error Error,
                                         int               ExtError,
                                         const char *      ExtErrorText,
                                         size_t            ExtErrorTextLength,
                                         int               ExtError2,
                                         const char *      ExtError2Text)
{
    char NumberBuf[50];
    int i=0;

    //search the error in the global error array
    struct
    {
        tcn00_Error   Value;
        const char  * Text;
    }errorArray[]=ERROR_TO_TEXT_CN00;

    while((ERR_UNKNOWN_CN00 != errorArray[i].Value) && (Error != errorArray[i].Value))
        i++;

    //write the error text to the reply buffer
    startWithLine((Error == OK_CN00) ? ANSWER_OK_CN00 : ANSWER_ERR_CN00);
    sprintf(NumberBuf, "%d", (int)Error); //50 chars should be sufficient
    appendStrings(NumberBuf, TOKEN_SEPSTRING_CN00);
    appendLine(errorArray[i].Text);

    if(0!=ExtErrorText && 0<ExtErrorTextLength)
    {
        sprintf(NumberBuf, "%d", (int)ExtError); //50 chars should be sufficient
        appendStrings(NumberBuf, TOKEN_SEPSTRING_CN00);
        appendLineN(ExtErrorText, ExtErrorTextLength);
    }

    if(0!=ExtError2Text)
    {
        sprintf(NumberBuf, "%d", (int)ExtError2); //50 chars should be sufficient
        appendStrings(NumberBuf, TOKEN_SEPSTRING_CN00);
        appendLine(ExtError2Text);
    }

    return Error;
}

tcn00_Error DBMSrv_Reply::startWithSQLError(    tcn00_Error  Error,
                                                const char*  SQLMessage,
                                                const int    SQLError) {
    
    return startWithError(Error, SQLError, SQLMessage, strlen(SQLMessage));
}


tcn00_Error DBMSrv_Reply::startWithRTEError(tcn00_Error  Error,
                                            const char * RTEMessage,
                                            int          RTEMessageLength,
                                            int          RTEError)
{
    size_t MessageLength=CStringLengthFromPascalString(RTEMessage, RTEMessageLength); //check if there is a terminating zero or trailing white spaces in RTEMessage

    return startWithError(Error, RTEError, RTEMessage, MessageLength);
}

tcn00_Error DBMSrv_Reply::startWithEventList(const teo200_EventList & EventList)
{
    const teo200_EventList * Event=&EventList;
    char NumberBuf[50];
  
    startWithLine(EventList.eo200_EventID()==cn00_1_ErrId(OK_CN00_1) ? ANSWER_OK_CN00 : ANSWER_ERR_CN00);

    while(0!=Event)
    {
        sprintf(NumberBuf, "%d", (int)Event->eo200_EventID()); //50 chars should be sufficient
        appendStrings(NumberBuf, TOKEN_SEPSTRING_CN00);    
        appendLine(Event->eo200_EventMsg());

        Event=Event->eo200_NextEvent();
    }

    return EventList.eo200_EventID();
}

tcn00_Error DBMSrv_Reply::startWithMessageList(const SAPDBErr_MessageList &MsgList)
{
    const SAPDBErr_MessageList* Msg=&MsgList;
    char NumberBuf[50];
  
    startWithLine(Msg->ID()==cn00_1_ErrId(OK_CN00_1) ? ANSWER_OK_CN00 : ANSWER_ERR_CN00);

    while(0!=Msg)
    {
        sprintf(NumberBuf, "%d", (int)Msg->ID()); //50 chars should be sufficient
        appendStrings(NumberBuf, TOKEN_SEPSTRING_CN00);    
        appendLine(Msg->Message());

        Msg=Msg->NextMessage();
    }

    return MsgList.ID();
}

void DBMSrv_Reply::startWithLine(const char * Line)
{
    *replyLength=0;
    appendLine(Line);
}

void DBMSrv_Reply::appendLine()
{
    appendString(LINE_SEPSTRING_CN00);
}

void DBMSrv_Reply::appendLine(const char * Line)
{
    appendString(Line);
    appendString(LINE_SEPSTRING_CN00);
}

void DBMSrv_Reply::appendLineN(const char * Line, size_t LineSize)
{
    appendStringN(Line, LineSize);
    appendString(LINE_SEPSTRING_CN00);
}

void DBMSrv_Reply::appendLines(const char * Line1, const char * Line2)
{
    appendLine(Line1);
    appendLine(Line2);
}

void DBMSrv_Reply::appendLines(const char * Line1, const char * Line2, const char * Line3)
{
    appendLine(Line1);
    appendLine(Line2);
    appendLine(Line3);
}

void DBMSrv_Reply::appendIndentedWrappedLineN(const char   * LineStart,
                                              const size_t   LineLength,
                                              const size_t   Indent,
                                              const size_t   WrapAt,
                                              size_t         PositionInOutputLine)
{
    appendIndentedWrappedLineN(LineStart, LineLength, Indent, WrapAt, 0, PositionInOutputLine);
}

void DBMSrv_Reply::appendIndentedWrappedLineN(const char   * LineStart,
                                              const size_t   LineLength,
                                              const size_t   Indent,
                                              const size_t   WrapAt,
                                              const char   * LineBreakString,
                                              size_t         PositionInOutputLine)
{
    const char * Position=cn36_FirstNonWhiteSpaceOf(LineStart, LineLength);

    if(0==LineLength)
        appendLine();

    while((size_t)(Position-LineStart) < LineLength) //as long as Line is not finished
    {
        if(PositionInOutputLine<Indent)                              //indent the lines by adding the right number of spaces
        {
            appendCharacterRepeatedly(' ', Indent-PositionInOutputLine);
            PositionInOutputLine=Indent;
        }

        size_t LengthOfNextWord=calculateLengthOfNextWord(Position, LineLength-(Position-LineStart), LineBreakString);

        appendStringN(Position, LengthOfNextWord); //write at least one word to each output line (that assures that the function is ending)
        PositionInOutputLine+=LengthOfNextWord;
        Position+=LengthOfNextWord;
        Position=cn36_FirstNonWhiteSpaceOf(Position, LineLength-(Position-LineStart));
       
        while(PositionInOutputLine<WrapAt &&             //there is still space in the output line -> add the next word
              (size_t)(Position-LineStart) < LineLength) //and there is another word left in Line
              
        {
            if(0==LineBreakString ||                                        //no special line breaks
               strlen(LineBreakString)>LineLength-(Position-LineStart) ||   //or they do not fit in the rest of the string
               0!=strncmp(LineBreakString, Position, strlen(LineBreakString))) //or they are not there
            {
                LengthOfNextWord=calculateLengthOfNextWord(Position, LineLength-(Position-LineStart), LineBreakString);

                if(PositionInOutputLine+1+LengthOfNextWord<WrapAt && //another space and word can be added before the LineWrap
                   (size_t)(Position-LineStart) < LineLength        )//and there is another word left in Line
                {
                    appendChar(' ');
                    appendStringN(Position, LengthOfNextWord);

                    PositionInOutputLine+=1+LengthOfNextWord;
                    Position+=LengthOfNextWord;
                    Position=cn36_FirstNonWhiteSpaceOf(Position, LineLength-(Position-LineStart));
                }
                else
                    PositionInOutputLine=WrapAt; //just end the loop
            }
            else
            {
                PositionInOutputLine=WrapAt; //just end the loop
                Position+=strlen(LineBreakString);
                Position=cn36_FirstNonWhiteSpaceOf(Position, LineLength-(Position-LineStart));
            }
        }

        appendLine();

        PositionInOutputLine=0; //we have a new output line after the line break
    }
}

void DBMSrv_Reply::appendString(const char *String)
{
    if( (maximalReplyLength-1) > (*replyLength) ) //is there space to add something and a terminating zero?
    {
        strncpy(replyData+(*replyLength), String, maximalReplyLength-1-(*replyLength)); //copy all of String that is still fitting into the reply buffer
        replyData[maximalReplyLength-1]='\0'; //make sure we have a terminating zero under all circumstances (strncpy does not guarantee that!)
   
        (*replyLength)+=(int)strlen(replyData+(*replyLength));
    }
}

void DBMSrv_Reply::appendStringN(const char *String, size_t StringSize)
{
    if( (maximalReplyLength-1) > (*replyLength) ) //is there space to add something and a terminating zero?
    {
        size_t SpaceInReply=maximalReplyLength-1-(*replyLength);
        
        if(StringSize>SpaceInReply) //take the minimum
            StringSize=SpaceInReply;

        strncpy(replyData+(*replyLength), String, StringSize); 
        replyData[maximalReplyLength-1]='\0';

        (*replyLength)+=(int)strlen(replyData+(*replyLength));
    }
}

void DBMSrv_Reply::appendStrings(const char *String1, const char *String2)
{
    appendString(String1);
    appendString(String2);
}

void DBMSrv_Reply::appendStrings(const char *String1, const char *String2, const char *String3)
{
    appendString(String1);
    appendString(String2);
    appendString(String3);
}

void DBMSrv_Reply::appendStringAndLine(const char *String1, const char *Line2)
{
    appendString(String1);
    appendLine(Line2);
}

void DBMSrv_Reply::appendStringAndLineN(const char *String1, const char *Line2, size_t Line2Length)
{
    appendString(String1);
    appendLineN(Line2, Line2Length);
}

void DBMSrv_Reply::appendStringsAndLine(const char *String1, const char *String2, const char *Line3)
{
    appendString(String1);
    appendString(String2);
    appendLine(Line3);
}

void DBMSrv_Reply::appendStringsStringNAndLine(const char   * String1,
                                               const char   * String2,
                                               const char   * String3,
                                               const char   * String4,
                                               const size_t   String4Length,
                                               const char   * Line)
{
    appendString(String1);
    appendString(String2);
    appendString(String3);
    appendStringN(String4, String4Length);
    appendLine(Line);
}

void DBMSrv_Reply::appendStringWithMinWidth(const char *String, size_t MinimalWidth)
{
    size_t LengthOfString=strlen(String);

    appendString(String);

    if(LengthOfString<MinimalWidth)
        appendCharacterRepeatedly(' ', MinimalWidth-LengthOfString);
}

void DBMSrv_Reply::appendStringNWithMinWidth(const char *String, size_t StringLength, size_t MinimalWidth)
{
    appendStringN(String, StringLength);

    if(StringLength<MinimalWidth)
        appendCharacterRepeatedly(' ', MinimalWidth-StringLength);
}

void DBMSrv_Reply::appendChar(const char Character)
{
    replyData[(*replyLength)++]=Character;
    replyData[*replyLength]='\0';
}

void DBMSrv_Reply::appendCharacterRepeatedly(const char Character, size_t NumberOfAppends)
{
    if(NumberOfAppends+(*replyLength) >= (size_t)maximalReplyLength)
       NumberOfAppends=maximalReplyLength-1-(*replyLength);

    memset(replyData+(*replyLength), Character, NumberOfAppends);

    (*replyLength)+=(int)NumberOfAppends;
    replyData[*replyLength]='\0';
}

char * DBMSrv_Reply::giveCurrentEnd()
{
    return replyData+(*replyLength);
}

void DBMSrv_Reply::skipAt(Position SkipPosition)
{
    size_t SkipLength=SkipPosition-replyData;

    replyData[SkipLength]='\0';

    (*replyLength)=(int)strlen(replyData);
}

void DBMSrv_Reply::insertString(Position InsertPosition, const char * String)
{
    size_t InsPos=InsertPosition-replyData;
    size_t StringLength=strlen(String);
    
    if( size_t(maximalReplyLength) > InsPos+StringLength && //is there enough space to insert the string and a terminating zero?
        size_t(*replyLength) > InsPos                      )//inserting in front of the terminating zero?
    {
        strncpy(replyData+InsPos, String, StringLength); //copy the string without the terminating zero

        if( InsPos+StringLength > size_t(*replyLength) ) //inserted string has overwritten the terminating zero
        {
            *(replyData+InsPos+StringLength)='\0';     //write a new terminating zero
            *replyLength=int(InsPos+StringLength);     //adjust the length member
        }
        //else nothing to do, old reply from InsertPosition was already longer than inserted string
    }
}
size_t DBMSrv_Reply::calculateLengthOfNextWord(const char *WordStart, size_t MaxWordLength, const char * LineBreakString)
{
    size_t rc=cn36_FirstWhiteSpaceOf(WordStart, MaxWordLength)-WordStart;

    if(0!=LineBreakString)
    {
        const char * NextArtificialLineBreak=strstr(WordStart, LineBreakString);

        if(0!=NextArtificialLineBreak && (size_t)(NextArtificialLineBreak-WordStart)<rc)
            rc=NextArtificialLineBreak-WordStart;  //words end at artificial line breaks
    }

    return rc;
}

size_t DBMSrv_Reply::CStringLengthFromPascalString(const char * SourcePascalString,
                                                   const int    PascalStringMaximalLength)
{
    size_t CStringLength=0;
    int  i;

    // at first looking for zero termination in source
    for(i=0; i<PascalStringMaximalLength-1 && SourcePascalString[i]!='\0'; i++);

    if('\0'==SourcePascalString[i])
        CStringLength=i;
    else
    {
        bool FoundLastNonSpace=false;

        CStringLength=PascalStringMaximalLength;
        
        while(CStringLength>0 && !FoundLastNonSpace)
        {
            if(' '==SourcePascalString[CStringLength-1])
                CStringLength--;
            else
                FoundLastNonSpace=true;
        }
    }

    return CStringLength;
}
