/*
 * Algebraic manipulator integration routines.
 *
 * Copyright (c) 1996 George Gesslein II.
 */

#include "am.h"
#include "externs.h"

/*
 * Return true if passed expression is a polynomial in
 * variable v.
 */
int
poly_in_v(equation, n, v, allow_divides)
token_type	*equation;
int		n;
long		v;
int		allow_divides;
{
	int	i, j, k;
	int	rv;

	rv = true;
	if (level1_plus(equation, n)) {
		for (i = 1, j = 0;; i += 2) {
			if (i >= n || equation[i].level == 1) {
				if (!poly_in_v_sub(&equation[j], i - j, v, allow_divides)) {
					rv = false;
					break;
				}
				j = i + 1;
			}
			if (i >= n)
				break;
		}
	} else {
		if (!poly_in_v_sub(equation, n, v, allow_divides))
			rv = false;
	}
	return rv;
}

/*
 * Return true if passed expression is strictly a single
 * polynomial term in v.
 */
int
poly_in_v_sub(equation, n, v, allow_divides)
token_type	*equation;
int		n;
long		v;
int		allow_divides;
{
	int	i, j, k;
	int	level;

	level = min_level(equation, n);
	for (i = 0; i < n; i++) {
		if (equation[i].kind == VARIABLE && equation[i].token.variable == v) {
			if (equation[i].level == level || equation[i].level == (level + 1)) {
				for (k = 1; k < n; k += 2) {
					if (equation[k].level == level) {
						switch (equation[k].token.operatr) {
						case DIVIDE:
							if (!allow_divides && k == (i - 1))
								return false;
						case TIMES:
							continue;
						case POWER:
							if (k == (i + 1))
								continue;
						default:
							return false;
						}
					}
				}
				if (equation[i].level == (level + 1)) {
					if ((i + 1) < n && equation[i+1].level == equation[i].level
					    && equation[i+1].token.operatr == POWER) {
						continue;
					}
				} else {
					continue;
				}
			}
			return false;
		}
	}
	return true;
}

/*
 * Make variable v always raised to a power.
 */
make_powers(equation, np, v)
token_type	*equation;
int		*np;
long		v;
{
	int	i;
	int	level;

	for (i = 0; i < *np; i += 2) {
		if (equation[i].kind == VARIABLE && equation[i].token.variable == v) {
			level = equation[i].level;
			if ((i + 1) >= *np || equation[i+1].token.operatr != POWER) {
				if (*np + 2 > n_tokens) {
					error_huge();
				}
				level++;
				equation[i].level = level;
				i++;
				blt(&equation[i+2], &equation[i], (*np - i) * sizeof(token_type));
				*np += 2;
				equation[i].level = level;
				equation[i].kind = OPERATOR;
				equation[i].token.operatr = POWER;
				i++;
				equation[i].level = level;
				equation[i].kind = CONSTANT;
				equation[i].token.constant = 1.0;
			}
		}
	}
}

/*
 * Integrate a polynomial expression with respect to v.
 *
 * Return true if successful.
 */
int
integrate(equation, np, v)
token_type	*equation;
int		*np;
long		v;
{
	int	i, j;

	make_powers(equation, np, v);
	if (level1_plus(equation, *np)) {
		for (j = 0, i = 1;; i += 2) {
			if (i >= *np || equation[i].level == 1) {
				if (!int_sub(equation, np, j, i, v))
					return false;
				for (i = j + 1;; i += 2) {
					if (i >= *np) {
						return true;
					}
					if (equation[i].level == 1) {
						j = i + 1;
						break;
					}
				}
			}
		}
	} else {
		if (!int_sub(equation, np, 0, *np, v))
			return false;
	}
	return true;
}

int
int_sub(equation, np, loc, eloc, v)
token_type	*equation;
int		*np;
int		loc, eloc;
long		v;
{
	int	i, j, k;
	int	len;
	int	level;
	int	mlevel;
	int	count;

	count = 0;
	for (i = loc; i < eloc; i += 2) {
		if (equation[i].kind == VARIABLE && equation[i].token.variable == v) {
			count++;
		}
	}
	if (count > 1)
		return false;
	mlevel = min_level(&equation[loc], eloc - loc) + 1;
	for (j = loc; j < eloc; j++)
		equation[j].level += 2;
	for (i = loc; i < eloc; i += 2) {
		if (equation[i].kind == VARIABLE && equation[i].token.variable == v) {
			i++;
			if (i >= eloc || equation[i].token.operatr != POWER)
				return false;
			level = equation[i].level;
			i++;
			for (j = i; j < eloc && equation[j].level >= level; j++)
				equation[j].level++;
			len = j - i;
			if (*np + len + 5 > n_tokens)
				error_huge();
			blt(&equation[j+2], &equation[j], (*np - j) * sizeof(token_type));
			*np += 2;
			eloc += 2;
			len += 2;
			level++;
			equation[j].level = level;
			equation[j].kind = OPERATOR;
			equation[j].token.operatr = PLUS;
			j++;
			equation[j].level = level;
			equation[j].kind = CONSTANT;
			equation[j].token.constant = 1.0;
			j++;
			blt(&equation[eloc+len+1], &equation[eloc], (*np - eloc) * sizeof(token_type));
			*np += len + 1;
			k = eloc;
			equation[k].level = mlevel;
			equation[k].kind = OPERATOR;
			equation[k].token.operatr = DIVIDE;
			k++;
			blt(&equation[k], &equation[i], len * sizeof(token_type));
			return true;
		}
	}
	if (*np + 2 > n_tokens) {
		error_huge();
	}
	blt(&equation[eloc+2], &equation[eloc], (*np - eloc) * sizeof(token_type));
	*np += 2;
	k = eloc;
	equation[k].level = mlevel;
	equation[k].kind = OPERATOR;
	equation[k].token.operatr = TIMES;
	k++;
	equation[k].level = mlevel;
	equation[k].kind = VARIABLE;
	equation[k].token.variable = v;
	return true;
}

/*
 * The integrate command.
 */
int
integrate_cmd(cp)
char	*cp;
{
	long	v;
	int	i, j;
	long	l;
	int	er;
	int	order;

	v = 0;
	order = 1;
	if (notdefined(cur_equation)) {
		return false;
	}
	if (*cp) {
		cp = parse_var2(&v, cp);
		if (cp == NULL) {
			return false;
		}
		if (isascii(*cp) && isdigit(*cp)) {
			order = atoi(cp);
			cp = skip_num(cp);
		}
	}
	if (extra_garbage(cp)) {
		return false;
	}
	if (order < 1) {
		printf(_("Order out of range.\n"));
		usage_flag = true;
		return false;
	}
	if (v == 0) {
		if (!prompt_var(&v)) {
			return false;
		}
	}
	i = next_espace();
	er = !solved_equation(cur_equation);
	er |= ((lhs[cur_equation][0].token.variable & VAR_MASK) <= SIGN);
	partial_flag = false;
	uf_simp(rhs[cur_equation], &n_rhs[cur_equation]);
	partial_flag = true;
	factorv(rhs[cur_equation], &n_rhs[cur_equation], v);
	blt(rhs[i], rhs[cur_equation], n_rhs[cur_equation] * sizeof(token_type));
	n_rhs[i] = n_rhs[cur_equation];
	for (j = 0; j < order; j++) {
		unsimp_power(rhs[i], &n_rhs[i], true);
		if (!poly_in_v(rhs[i], n_rhs[i], v, false)) {
			printf(_("integrate and laplace only work on polynomials in the specified variable.\n"));
			return false;
		}
		if (!integrate(rhs[i], &n_rhs[i], v)) {
			printf(_("Bug detected, please send this equation to the code maintainers.\n"));
			return false;
		}
		simp_loop(rhs[i], &n_rhs[i]);
	}
	blt(&lhs[i][0], &lhs[cur_equation][0], n_lhs[cur_equation] * sizeof(token_type));
	n_lhs[i] = n_lhs[cur_equation];
	if (!er) {
/* decrement the number of primes in the LHS variable */
		l = lhs[i][0].token.variable;
		for (j = 1; j <= order; j++) {
			l -= PRIME_INCREMENT;
			if (l < 0) {
				l += PRIME_INCREMENT;
				break;
			}
		}
		lhs[i][0].token.variable = l;
	}
	cur_equation = i;
	list_sub(cur_equation);
	return true;
}

/*
 * Compute the Laplace transform of a polynomial expression with respect to v.
 *
 * Return true if successful.
 */
int
laplace(equation, np, v)
token_type	*equation;
int		*np;
long		v;
{
	int	i, j;

	make_powers(equation, np, v);
	if (level1_plus(equation, *np)) {
		for (j = 0, i = 1;; i += 2) {
			if (i >= *np || equation[i].level == 1) {
				if (!laplace_sub(equation, np, j, i, v))
					return false;
				for (i = j + 1;; i += 2) {
					if (i >= *np) {
						return true;
					}
					if (equation[i].level == 1) {
						j = i + 1;
						break;
					}
				}
			}
		}
	} else {
		if (!laplace_sub(equation, np, 0, *np, v))
			return false;
	}
	return true;
}

int
laplace_sub(equation, np, loc, eloc, v)
token_type	*equation;
int		*np;
int		loc, eloc;
long		v;
{
	int	i, j, k;
	int	len;
	int	level;
	int	mlevel;
	int	count;

	count = 0;
	for (i = loc; i < eloc; i += 2) {
		if (equation[i].kind == VARIABLE && equation[i].token.variable == v) {
			count++;
		}
	}
	if (count > 1)
		return false;
	mlevel = min_level(&equation[loc], eloc - loc) + 1;
	for (j = loc; j < eloc; j++)
		equation[j].level += 2;
	for (i = loc; i < eloc; i += 2) {
		if (equation[i].kind == VARIABLE && equation[i].token.variable == v) {
			i++;
			if (i >= eloc || equation[i].token.operatr != POWER)
				return false;
			level = equation[i].level;
			i++;
			for (j = i; j < eloc && equation[j].level >= level; j++)
				equation[j].level++;
			len = j - i;
			if (*np + len + 7 > n_tokens)
				error_huge();
			blt(&equation[j+4], &equation[j], (*np - j) * sizeof(token_type));
			*np += 4;
			eloc += 4;
			level++;
			equation[j].level = level;
			equation[j].kind = OPERATOR;
			equation[j].token.operatr = PLUS;
			j++;
			equation[j].level = level;
			equation[j].kind = CONSTANT;
			equation[j].token.constant = 1.0;
			j++;
			for (k = i; k < j; k++)
				equation[k].level++;
			equation[j].level = level;
			equation[j].kind = OPERATOR;
			equation[j].token.operatr = TIMES;
			j++;
			equation[j].level = level;
			equation[j].kind = CONSTANT;
			equation[j].token.constant = -1.0;
			j++;
			blt(&equation[eloc+len+3], &equation[eloc], (*np - eloc) * sizeof(token_type));
			*np += len + 3;
			k = eloc;
			equation[k].level = mlevel;
			equation[k].kind = OPERATOR;
			equation[k].token.operatr = TIMES;
			k++;
			blt(&equation[k], &equation[i], len * sizeof(token_type));
			k += len;
			equation[k].level = mlevel + 1;
			equation[k].kind = OPERATOR;
			equation[k].token.operatr = FACTORIAL;
			k++;
			equation[k].level = mlevel + 1;
			equation[k].kind = CONSTANT;
			equation[k].token.constant = 0.0;
			k++;
			return true;
		}
	}
	if (*np + 2 > n_tokens) {
		error_huge();
	}
	blt(&equation[eloc+2], &equation[eloc], (*np - eloc) * sizeof(token_type));
	*np += 2;
	k = eloc;
	equation[k].level = mlevel;
	equation[k].kind = OPERATOR;
	equation[k].token.operatr = DIVIDE;
	k++;
	equation[k].level = mlevel;
	equation[k].kind = VARIABLE;
	equation[k].token.variable = v;
	return true;
}

/*
 * Compute the inverse Laplace transform of a polynomial expression with respect to v.
 *
 * Return true if successful.
 */
int
inv_laplace(equation, np, v)
token_type	*equation;
int		*np;
long		v;
{
	int	i, j;

	make_powers(equation, np, v);
	if (level1_plus(equation, *np)) {
		for (j = 0, i = 1;; i += 2) {
			if (i >= *np || equation[i].level == 1) {
				if (!inv_laplace_sub(equation, np, j, i, v))
					return false;
				for (i = j + 1;; i += 2) {
					if (i >= *np) {
						return true;
					}
					if (equation[i].level == 1) {
						j = i + 1;
						break;
					}
				}
			}
		}
	} else {
		if (!inv_laplace_sub(equation, np, 0, *np, v))
			return false;
	}
	return true;
}

int
inv_laplace_sub(equation, np, loc, eloc, v)
token_type	*equation;
int		*np;
int		loc, eloc;
long		v;
{
	int	i, j, k;
	int	len;
	int	level;
	int	mlevel;
	int	count;

	count = 0;
	for (i = loc; i < eloc; i += 2) {
		if (equation[i].kind == VARIABLE && equation[i].token.variable == v) {
			count++;
		}
	}
	if (count > 1)
		return false;
	mlevel = min_level(&equation[loc], eloc - loc) + 1;
	for (j = loc; j < eloc; j++)
		equation[j].level += 2;
	for (i = loc; i < eloc; i += 2) {
		if (equation[i].kind == VARIABLE && equation[i].token.variable == v) {
			i++;
			if (i >= eloc || equation[i].token.operatr != POWER)
				return false;
			if ((i - 2) <= loc || equation[i-2].token.operatr != DIVIDE)
				return false;
			level = equation[i].level;
			i++;
			for (j = i; j < eloc && equation[j].level >= level; j++)
				equation[j].level++;
			len = j - i;
			if (*np + len + 7 > n_tokens)
				error_huge();
			equation[i-3].token.operatr = TIMES;
			blt(&equation[j+2], &equation[j], (*np - j) * sizeof(token_type));
			*np += 2;
			eloc += 2;
			len += 2;
			level++;
			equation[j].level = level;
			equation[j].kind = OPERATOR;
			equation[j].token.operatr = MINUS;
			j++;
			equation[j].level = level;
			equation[j].kind = CONSTANT;
			equation[j].token.constant = 1.0;
			j++;
			blt(&equation[eloc+len+3], &equation[eloc], (*np - eloc) * sizeof(token_type));
			*np += len + 3;
			k = eloc;
			equation[k].level = mlevel;
			equation[k].kind = OPERATOR;
			equation[k].token.operatr = DIVIDE;
			k++;
			blt(&equation[k], &equation[i], len * sizeof(token_type));
			k += len;
			equation[k].level = mlevel + 1;
			equation[k].kind = OPERATOR;
			equation[k].token.operatr = FACTORIAL;
			k++;
			equation[k].level = mlevel + 1;
			equation[k].kind = CONSTANT;
			equation[k].token.constant = 0.0;
			k++;
			return true;
		}
	}
	return false;
}

/*
 * The laplace command.
 */
int
laplace_cmd(cp)
char	*cp;
{
	long	v;
	int	i, j;
	int	inverse_flag;

	v = 0;
	if (notdefined(cur_equation)) {
		return false;
	}
	inverse_flag = (strncasecmp(cp, "inverse", 7) == 0);
	if (inverse_flag) {
		cp = skip_param(cp);
	}
	if (*cp) {
		cp = parse_var2(&v, cp);
		if (cp == NULL) {
			return false;
		}
	}
	if (extra_garbage(cp)) {
		return false;
	}
	if (v == 0) {
		if (!prompt_var(&v)) {
			return false;
		}
	}
	i = next_espace();
	partial_flag = false;
	uf_simp(rhs[cur_equation], &n_rhs[cur_equation]);
	partial_flag = true;
	factorv(rhs[cur_equation], &n_rhs[cur_equation], v);
	blt(rhs[i], rhs[cur_equation], n_rhs[cur_equation] * sizeof(token_type));
	n_rhs[i] = n_rhs[cur_equation];
	if (inverse_flag) {
		if (!poly_in_v(rhs[i], n_rhs[i], v, true)) {
			printf(_("Inverse Laplace failed.\n"));
			return false;
		}
		if (!inv_laplace(rhs[i], &n_rhs[i], v)) {
			printf(_("Inverse Laplace failed.\n"));
			return false;
		}
	} else {
		if (!poly_in_v(rhs[i], n_rhs[i], v, false)) {
			printf(_("integrate and laplace only work on polynomials in the specified variable.\n"));
			return false;
		}
		if (!laplace(rhs[i], &n_rhs[i], v)) {
			printf(_("Bug detected, please send this equation to the code maintainers.\n"));
			return false;
		}
	}
	simp_loop(rhs[i], &n_rhs[i]);
	blt(&lhs[i][0], &lhs[cur_equation][0], n_lhs[cur_equation] * sizeof(token_type));
	n_lhs[i] = n_lhs[cur_equation];
	cur_equation = i;
	list_sub(cur_equation);
	return true;
}
