/* massXpert - the true massist's program.
   --------------------------------------
   Copyright(C) 2006,2007 Filippo Rusconi

   http://www.massxpert.org/massXpert

   This file is part of the massXpert project.

   The massxpert project is the successor to the "GNU polyxmass"
   project that is an official GNU project package(see
   www.gnu.org). The massXpert project is not endorsed by the GNU
   project, although it is released ---in its entirety--- under the
   GNU General Public License. A huge part of the code in massXpert
   is actually a C++ rewrite of code in GNU polyxmass. As such
   massXpert was started at the Centre National de la Recherche
   Scientifique(FRANCE), that granted me the formal authorization to
   publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License version 3, as published by the Free Software Foundation.
   

   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software; if not, write to the

   Free Software Foundation, Inc.,

   51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
*/


#include <QDebug>

#include "isotopicCurve.hpp"

#include <math.h>

namespace massXpert
{

  IsotopicCurve::IsotopicCurve()
    : m_mzRatio(0), m_intensity(0),
      m_fwhm(0),
      m_points(0), m_increment(0),
      m_normFactor(1), m_curveType(MXP_CURVE_TYPE_GAUSSIAN)
  {
  
  }


  IsotopicCurve::IsotopicCurve(double mzRatio, double intensity, 
                               double fwhm,
                               int points, float increment,
                               double normFactor, CurveType curveType)
    : m_mzRatio(mzRatio), m_intensity(intensity),
      m_fwhm(fwhm),
      m_points(points), m_increment(increment),
      m_normFactor(normFactor), m_curveType(curveType)
  {
  
  }


  IsotopicCurve::IsotopicCurve(const IsotopicCurve &other)
  {
    qFatal("Fatal error at %s@%d.Aborting.", __FILE__, __LINE__);
  }


  IsotopicCurve::~IsotopicCurve()
  {
    emptyPeakList();
  }

  const QList<IsotopicPeak *> &  
  IsotopicCurve::isotopicPeakList() const
  {
    return m_isotopicPeakList;
  }

  void
  IsotopicCurve::appendPeak(IsotopicPeak *peak)
  {
    if(!peak)
      qFatal("Fatal error at %s@%d.Aborting.", __FILE__, __LINE__);

    m_isotopicPeakList.append(peak);
  }
  

  void
  IsotopicCurve::emptyPeakList()
  {
    while(!m_isotopicPeakList.isEmpty())
      delete m_isotopicPeakList.takeFirst();
  }


  void
  IsotopicCurve::setMzRatio(double value)
  {
    m_mzRatio = value;
  }


  double  
  IsotopicCurve::mzRatio()
  {

    return m_mzRatio;
  }


  QString  
  IsotopicCurve::mzRatioString()
  {
    QString string;
    string.setNum(m_mzRatio);
  
    return string;
  }
    

  void
  IsotopicCurve::setIntensity(double value)
  {
    m_intensity = value;
  }


  double  
  IsotopicCurve::intensity()
  {
    return m_intensity;
  }


  QString  
  IsotopicCurve::intensityString()
  {
    QString string;
    string.setNum(m_intensity);
  
    return string;
  }


  void
  IsotopicCurve::setFwhm(double value)
  {
    m_fwhm = value;
  }


  double  
  IsotopicCurve::fwhm()
  {
    return m_fwhm;
  }


  QString  
  IsotopicCurve::fwhmString()
  {
    double fwhmValue = fwhm();
  
    QString string;
    string.setNum(fwhmValue);
  
    return string;
  }
  
  // For the lorentzian, that is half of the fwhm.
  double  
  IsotopicCurve::hwhm()
  {
    double fwhmValue = fwhm();

    return (fwhmValue / 2);
  }


  QString  
  IsotopicCurve::hwhmString()
  {
    QString string;
    string.setNum(m_fwhm/2);
  
    return string;
  }


  void
  IsotopicCurve::setPoints(int value)
  {
    m_points = value;
  }

  
  int
  IsotopicCurve::points()
  {
    return m_points;
  }

  
  QString
  IsotopicCurve::pointsString()
  {
    QString string;
    string.setNum(m_points);
    
    return string;
  }

  double  
  IsotopicCurve::a()
  {
    //  double pi = 3.1415926535897932384626433832795029;
  
    double a = (1 / (c()*sqrt(M_PI)));
   
    // qDebug() << __FILE__ << __LINE__
    //          << "a:" << a;
    
    return a;
  }


  QString  
  IsotopicCurve::aString()
  {
    double aValue = a();
  
    QString string;
    string.setNum(aValue);
  
    return string;
  }
  

  double  
  IsotopicCurve::c()
  {
    double c = m_fwhm / (2 * sqrt(2*log(2)));

    // qDebug() << __FILE__ << __LINE__
    //          << "c:" << c;
    
    return c;
  }

  QString  
  IsotopicCurve::cString()
  {

    double cValue = c();
  
    QString string;
    string.setNum(cValue);
  
    return string;
  }
  
  double  
  IsotopicCurve::gamma()
  {
    double gamma = fwhm() / 2;
  
    // qDebug() << __FILE__ << __LINE__
    //          << "gamma:" << gamma;

    return gamma;
  }

  QString  
  IsotopicCurve::gammaString()
  {
    double gammaValue = gamma();
  
    QString string;
    string.setNum(gammaValue);
  
    return string;
  }


  void
  IsotopicCurve::setIncrement(double value)
  {
    m_increment = value;
  }


  double
  IsotopicCurve::increment()
  {
    // But what is m_increment ? We want the curve to be able to go
    // down to baseline. Thus we want that the curve to have a "basis"
    // corresponding to twice the FWHM on the left of the centroid and
    // to twice the FWHM on the right (that makes in total
    // MXP_FWHM_PEAK_SPAN_FACTOR * FWHM).
    if(!m_increment)
      {
        m_increment = (MXP_FWHM_PEAK_SPAN_FACTOR * m_fwhm) / m_points;
      }
    
    return m_increment;
  }


  QString  
  IsotopicCurve::incrementString()
  {
    QString string;
    string.setNum(m_increment);
  
    return string;
  }


  QString  
  IsotopicCurve::config()
  {
    QString string;
    QString value;
    
    if(m_curveType == MXP_CURVE_TYPE_GAUSSIAN)
      {
        
        string += "Gauss:\n";
        
        value = QString().setNum(m_fwhm, 'f', 5);
        string += "FWHM:" ;
        string += value;
        string += "\n";
        
        double cValue = c();
        value = QString().setNum(cValue, 'f', 5);
        string += "c:" ;
        string += value;
        string += "\n";
              
        value = QString().setNum((cValue * cValue) , 'f', 5);
        string += "c^2:" ;
        string += value;
        string += "\n";
                
        value = QString().setNum(increment(), 'f', 5);
        string += "increment:" ;
        string += value;
        string += "\n\n";
      }
    
    if(m_curveType == MXP_CURVE_TYPE_LORENTZIAN)
      {
        
        string += "Gauss:\n";
        
        value = QString().setNum(m_fwhm, 'f', 5);
        string += "FWHM:" ;
        string += value;
        string += "\n";
        
        double gammaValue = gamma();
        value = QString().setNum(gammaValue, 'f', 5);
        string += "gamma:" ;
        string += value;
        string += "\n";
        
        value = QString().setNum((gammaValue * gammaValue) , 'f', 5);
        string += "gamma^2:" ;
        string += value;
        string += "\n";
        
        value = QString().setNum(increment(), 'f', 5);
        string += "increment:" ;
        string += value;
        string += "\n\n";
      }
    
    return string;
  }
  


  int  
  IsotopicCurve::calculateCurve()
  {
    if(m_curveType == MXP_CURVE_TYPE_GAUSSIAN)
      return calculateGaussianCurve();
    else
      return calculateLorentzianCurve();
  }


  int  
  IsotopicCurve::calculateGaussianCurve()
  {
    // qDebug() << __FILE__ << __LINE__
    //          << "calculating gaussian isotopicCurve.";
  
    double aValue = a();
    // We set a to 1
    aValue = 1;
  
    // The calls below will trigger the computation of m_fwhm, if it is
    // equal to -1 because it was not set manually in the parameters of
    // the command line.
    double cValue = c();
    double c2Value = cValue * cValue;
  
    // Were are the left and right points of the curve ? We have to
    // determine that using the m_points and m_increment values.

    // Compute the increment that will separate two consecutive points
    // of the curve.
    increment();
    
    // double leftPoint = m_mzRatio - ((m_points * m_increment) / 2);
    // double rightPoint = m_mzRatio + ((m_points * m_increment) / 2);

    double leftPoint = m_mzRatio - (2 * m_fwhm);
    double rightPoint = m_mzRatio + (2 * m_fwhm);
  
    // qDebug() << __FILE__ << __LINE__
    //          << "Calculting curve\n"
    //          << "m_mzRatio:" << m_mzRatio
    //          << "m_intensity:" << m_intensity
    //          << "m_fwhm:" << m_fwhm
    //          << "m_points:" << m_points
    //          << "m_increment:" << m_increment
    //          << "leftPoint:" << leftPoint
    //          << "rightPoint:" << rightPoint
    //          << "cValue:" << cValue
    //          << "c2Value:" << c2Value
    //          << "\n";

    int iterations = (rightPoint - leftPoint) / m_increment;
    double x = leftPoint;
    
    // QString debugString;
    for(int iter = 0 ; iter < iterations; ++iter)
      {
        // debugString += QString().setNum(x, 'f', 6);
        
        double y = m_intensity * aValue *
          exp(-1 * (pow((x - m_mzRatio), 2) / (2*c2Value)));
        
        // debugString += " ";
        // debugString += QString().setNum(y, 'f', 6);
        // debugString += "\n";
         
        IsotopicPeak *peak = new IsotopicPeak(x, y, 0, 0);
        
        m_isotopicPeakList.append(peak);

        x += m_increment;
      }
    
    // qDebug() << debugString;

    return m_isotopicPeakList.size();
  }


  int  
  IsotopicCurve::calculateLorentzianCurve()
  {
    // qDebug() << __FILE__ << __LINE__
    //          << "calculating lorentzian isotopicCurve.";
  
    double aValue = a();
    // We set a to 1
    aValue = 1;

    // The calls below will trigger the computation of m_fwhm, if it is
    // equal to -1 because it was not set manually in the parameters of
    // the command line.
    double gammaValue = gamma();
    double gamma2Value = gammaValue * gammaValue;
  
    // Were are the left and right points of the curve ? We have to
    // determine that using the m_points and m_increment values.

    // Compute the increment that will separate two consecutive points
    // of the curve.
    increment();
    
    // double leftPoint = m_mzRatio - ((m_points * m_increment) / 2);
    // double rightPoint = m_mzRatio + ((m_points * m_increment) / 2);

    double leftPoint = m_mzRatio - (2 * m_fwhm);
    double rightPoint = m_mzRatio + (2 * m_fwhm);
  
    // qDebug() << __FILE__ << __LINE__
    //          << "m_mzRatio:" << m_mzRatio
    //          << "m_points:" << m_points
    //          << "m_increment:" << m_increment
    //          << "gammaValue:" << gammaValue
    //          << "gamma2Value:" << gamma2Value
    //          << "leftPoint:" << leftPoint
    //          << "rightPoint:" << rightPoint;

    int iterations = (rightPoint - leftPoint) / m_increment;
    double x = leftPoint;

    // QString debugString;
    for(int iter = 0 ; iter < iterations; ++iter)
      {
        // debugString += QString().setNum(x, 'f', 6);

        double y = m_intensity * aValue *
          (gamma2Value / (pow((x - m_mzRatio), 2) + gamma2Value));
      
        // debugString += " ";
        // debugString += QString().setNum(y, 'f', 6);
        // debugString += "\n";

        IsotopicPeak *peak = new IsotopicPeak(x, y, 0, 0);
      
        m_isotopicPeakList.append(peak);

        x += m_increment;
      }
  
    // qDebug() << debugString;

    return m_isotopicPeakList.size();
  }


  double 
  IsotopicCurve::intensityAt(double mzRatio, double tolerance, bool *ok)
  {
    double minMzRatio = mzRatio - (tolerance/2);
    double maxMzRatio = mzRatio + (tolerance/2);

    for(int iter = 0, size = m_isotopicPeakList.size() ; 
        iter < size ; ++iter)
      {
        IsotopicPeak *peak = m_isotopicPeakList.at(iter);
      
        double mzRatio = peak->mass();
      
        if(mzRatio <= maxMzRatio && mzRatio >= minMzRatio)
          {
            *ok = true;
          
            return peak->relativeIntensity();
          }
      }
  
    // If we are here, that means that we have not found the wavenumber
    // in the curve, which is perfectly admissible. But let the caller
    // know that, because the 0 that is returned might well be a sincere
    // intensity...
  
    *ok = false;
  
    return 0;
  }

  QString 
  IsotopicCurve::dataAsString()
  {
    QString output;
    
    for(int iter = 0, size = m_isotopicPeakList.size();
        iter < size ; ++iter)
      {
        IsotopicPeak *peak = m_isotopicPeakList.at(iter);
        
        output += QString().setNum(peak->mass(), 'f', 10);
        output += " ";
        output += QString().setNum(peak->relativeIntensity(), 'f', 10);
        output += "\n";
      }
    
    return output;
  }

  
  bool
  IsotopicCurve::dataToFile(QString filePath)
  {
    QFile file(filePath);
    
    if(!file.open(QFile::WriteOnly | QFile::Truncate))
      {
        qDebug() << __FILE__ << __LINE__
                 << "Failed to open output file for writing.";
        
        return false;
      }
    
    QTextStream stream(&file);
    
    for(int iter = 0, size = m_isotopicPeakList.size();
        iter < size ; ++iter)
      {
        IsotopicPeak *peak = m_isotopicPeakList.at(iter);
        
        QString dataString = QString().setNum(peak->mass(), 'f', 10);
        dataString += " ";
        dataString += QString().setNum(peak->relativeIntensity(), 'f', 10);
        dataString += "\n";
        stream << dataString;
        // qDebug() << __FILE__ << __LINE__ << dataString;
      }
    
    stream.flush();
    file.close();

    return true;
  }
  
  
  
} // namespace massXpert
