// license:BSD-3-Clause
// copyright-holders:R. Belmont
/***************************************************************************

  Apple model 820-5037-C "Macintosh II Portrait Video Card"
  PCB is marked "Workstation/Portrait Card"
  640x870, 1, 2 or 4bpp grayscale

  Fs0900e0 = DAC control
  Fs0900e4 = DAC data
  Fs0A0000 = enable / ack VBL IRQ
  Fs0A0004 = disable VBL IRQ

***************************************************************************/

#include "emu.h"
#include "nubus_wsportrait.h"
#include "screen.h"

#define WSPORTRAIT_SCREEN_NAME  "wsport_screen"
#define WSPORTRAIT_ROM_REGION  "wsport_rom"

#define VRAM_SIZE   (0x80000)   // 512k max


ROM_START( wsportrait )
	ROM_REGION(0x1000, WSPORTRAIT_ROM_REGION, 0)
	ROM_LOAD( "341-0732.bin", 0x000000, 0x001000, CRC(ddc35b78) SHA1(ce2bf2374bb994c17962dba8f3d11bc1260e2644) )
ROM_END

//**************************************************************************
//  GLOBAL VARIABLES
//**************************************************************************

DEFINE_DEVICE_TYPE(NUBUS_WSPORTRAIT, nubus_wsportrait_device, "nb_wspt", "Macintosh II Portrait Video Card")


//-------------------------------------------------
//  device_add_mconfig - add device configuration
//-------------------------------------------------

MACHINE_CONFIG_START(nubus_wsportrait_device::device_add_mconfig)
	MCFG_SCREEN_ADD( WSPORTRAIT_SCREEN_NAME, RASTER)
	MCFG_SCREEN_UPDATE_DEVICE(DEVICE_SELF, nubus_wsportrait_device, screen_update)
	MCFG_SCREEN_SIZE(1024,960)
	MCFG_SCREEN_REFRESH_RATE(75.0)
	MCFG_SCREEN_VISIBLE_AREA(0, 640-1, 0, 870-1)
MACHINE_CONFIG_END

//-------------------------------------------------
//  rom_region - device-specific ROM region
//-------------------------------------------------

const tiny_rom_entry *nubus_wsportrait_device::device_rom_region() const
{
	return ROM_NAME( wsportrait );
}

//**************************************************************************
//  LIVE DEVICE
//**************************************************************************

//-------------------------------------------------
//  nubus_wsportrait_device - constructor
//-------------------------------------------------

nubus_wsportrait_device::nubus_wsportrait_device(const machine_config &mconfig, const char *tag, device_t *owner, uint32_t clock) :
	nubus_wsportrait_device(mconfig, NUBUS_WSPORTRAIT, tag, owner, clock)
{
}

nubus_wsportrait_device::nubus_wsportrait_device(const machine_config &mconfig, device_type type, const char *tag, device_t *owner, uint32_t clock) :
	device_t(mconfig, type, tag, owner, clock),
	device_video_interface(mconfig, *this),
	device_nubus_card_interface(mconfig, *this),
	m_vram32(nullptr), m_mode(0), m_vbl_disable(0), m_toggle(0), m_count(0), m_clutoffs(0), m_timer(nullptr)
{
	set_screen(*this, WSPORTRAIT_SCREEN_NAME);
}

//-------------------------------------------------
//  device_start - device-specific startup
//-------------------------------------------------

void nubus_wsportrait_device::device_start()
{
	uint32_t slotspace;

	install_declaration_rom(this, WSPORTRAIT_ROM_REGION, true);

	slotspace = get_slotspace();

	printf("[wsportrait %p] slotspace = %x\n", (void *)this, slotspace);

	m_vram.resize(VRAM_SIZE);
	m_vram32 = (uint32_t *)&m_vram[0];

	nubus().install_device(slotspace, slotspace+VRAM_SIZE-1, read32_delegate(FUNC(nubus_wsportrait_device::vram_r), this), write32_delegate(FUNC(nubus_wsportrait_device::vram_w), this));
	nubus().install_device(slotspace+0x900000, slotspace+0x900000+VRAM_SIZE-1, read32_delegate(FUNC(nubus_wsportrait_device::vram_r), this), write32_delegate(FUNC(nubus_wsportrait_device::vram_w), this));
	nubus().install_device(slotspace+0x80000, slotspace+0xeffff, read32_delegate(FUNC(nubus_wsportrait_device::wsportrait_r), this), write32_delegate(FUNC(nubus_wsportrait_device::wsportrait_w), this));

	m_timer = timer_alloc(0, nullptr);
	m_timer->adjust(screen().time_until_pos(869, 0), 0);
}

//-------------------------------------------------
//  device_reset - device-specific reset
//-------------------------------------------------

void nubus_wsportrait_device::device_reset()
{
	m_count = 0;
	m_clutoffs = 0;
	m_vbl_disable = 1;
	m_mode = 0;
	memset(&m_vram[0], 0, VRAM_SIZE);
	memset(m_palette, 0, sizeof(m_palette));
}


void nubus_wsportrait_device::device_timer(emu_timer &timer, device_timer_id tid, int param, void *ptr)
{
	if (!m_vbl_disable)
	{
		raise_slot_irq();
	}

	m_timer->adjust(screen().time_until_pos(869, 0), 0);
}

/***************************************************************************

  Workstation/Portrait emulation

***************************************************************************/

uint32_t nubus_wsportrait_device::screen_update(screen_device &screen, bitmap_rgb32 &bitmap, const rectangle &cliprect)
{
	uint32_t *scanline;
	int x, y;
	uint8_t pixels, *vram;

	// first time?  kick off the VBL timer
	vram = &m_vram[0x80];

	switch (m_mode)
	{
		case 0: // 1 bpp?
			for (y = 0; y < 870; y++)
			{
				scanline = &bitmap.pix32(y);
				for (x = 0; x < 640/8; x++)
				{
					pixels = vram[(y * 128) + (BYTE4_XOR_BE(x))];

					*scanline++ = m_palette[((pixels>>7)&0x1)];
					*scanline++ = m_palette[((pixels>>6)&0x1)];
					*scanline++ = m_palette[((pixels>>5)&0x1)];
					*scanline++ = m_palette[((pixels>>4)&0x1)];
					*scanline++ = m_palette[((pixels>>3)&0x1)];
					*scanline++ = m_palette[((pixels>>2)&0x1)];
					*scanline++ = m_palette[((pixels>>1)&0x1)];
					*scanline++ = m_palette[(pixels&1)];
				}
			}
			break;

		case 1: // 2 bpp
			for (y = 0; y < 480; y++)
			{
				scanline = &bitmap.pix32(y);
				for (x = 0; x < 640/4; x++)
				{
					pixels = vram[(y * 256) + (BYTE4_XOR_BE(x))];

					*scanline++ = m_palette[((pixels>>6)&3)];
					*scanline++ = m_palette[((pixels>>4)&3)];
					*scanline++ = m_palette[((pixels>>2)&3)];
					*scanline++ = m_palette[(pixels&3)];
				}
			}
			break;

		case 2: // 4 bpp
			for (y = 0; y < 480; y++)
			{
				scanline = &bitmap.pix32(y);

				for (x = 0; x < 640/2; x++)
				{
					pixels = vram[(y * 512) + (BYTE4_XOR_BE(x))];

					*scanline++ = m_palette[((pixels&0xf0)>>4)];
					*scanline++ = m_palette[(pixels&0xf)];
				}
			}
			break;

		default:
			fatalerror("wsportrait: unknown video mode %d\n", m_mode);
	}
	return 0;
}

WRITE32_MEMBER( nubus_wsportrait_device::wsportrait_w )
{
	data ^= 0xffffffff;
//  if (offset != 0x8000) printf("wsportrait: Write %08x @ %x, mask %08x\n", data, offset, mem_mask);

	switch (offset)
	{
		case 1:         // mode control
//          printf("%08x to mode 1\n", data);
			switch (data & 0xff000000)
			{
				case 0x20000000:
				case 0x24000000:
					m_mode = 0;
					break;

				case 0x40000000:
					m_mode = 1;
					break;

				case 0x50000000:
				case 0x80000000:
					m_mode = 2;
					break;
			}
			break;

		case 0x4038:    // DAC control
			m_clutoffs = (data>>24)&0xff;
			break;

		case 0x4039:    // DAC data - only 4 bits per component!
			m_colors[m_count] = (data>>24) & 0x0f;
			m_colors[m_count] |= (m_colors[m_count]<<4);
			m_count++;

			if (m_count == 3)
			{
//              logerror("RAMDAC: color %d = %02x %02x %02x %s\n", m_clutoffs, m_colors[0], m_colors[1], m_colors[2], machine().describe_context());
				m_palette[m_clutoffs] = rgb_t(m_colors[2], m_colors[2], m_colors[2]);
				m_clutoffs++;
				if (m_clutoffs > 255)
				{
					m_clutoffs = 0;
				}
				m_count = 0;
			}
			break;

		case 0x8000:
			lower_slot_irq();
			m_vbl_disable = false;
			break;

		case 0x8001:
			m_vbl_disable = true;
			break;
	}
}

READ32_MEMBER( nubus_wsportrait_device::wsportrait_r )
{
//  printf("wsportrait: Read @ %x, mask %08x\n", offset, mem_mask);

	/*
	  monitor types

	  0x0 = invalid
	  0x2 = invalid
	  0x4 = color: 640x870 1bpp, 640x480 2bpp and 4bpp
	  0x6 = 1bpp 640x384? and sets weird mode controls
	  0x8 = really odd (bitplaned?)
	  0xa = invalid
	  0xc = 640x480 grayscale
	  0xe = same as 0x6
	*/

	switch (offset)
	{
		case 0x4004:
			m_toggle ^= 0x00010000;
			return m_toggle | 0xfffc0000;   // bit 0 = vbl status, bits 1-3 = monitor type
	}
	return 0;
}

WRITE32_MEMBER( nubus_wsportrait_device::vram_w )
{
	data ^= 0xffffffff;
	COMBINE_DATA(&m_vram32[offset]);
}

READ32_MEMBER( nubus_wsportrait_device::vram_r )
{
	return m_vram32[offset] ^ 0xffffffff;
}
