/* * This file is part of Maliit framework *
 *
 * Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
 * All rights reserved.
 *
 * Contact: maliit-discuss@lists.maliit.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation
 * and appearing in the file LICENSE.LGPL included in the packaging
 * of this file.
 */

#include "mdbusglibinputcontextadaptor.h"
#include "mimserverconnection.h"
#include <maliit/settingdata.h>

#include <variantmarshalling.h>

#include <dbus/dbus.h>

G_DEFINE_TYPE(MDBusGlibInputContextAdaptor, m_dbus_glib_input_context_adaptor, G_TYPE_OBJECT)


static gboolean m_dbus_glib_input_context_adaptor_activation_lost_event(
    MDBusGlibInputContextAdaptor *obj, GError **/*error*/)
{
    Q_EMIT obj->imServerConnection->activationLostEvent();
    return TRUE;
}

static gboolean m_dbus_glib_input_context_adaptor_im_initiated_hide(
    MDBusGlibInputContextAdaptor *obj, GError **/*error*/)
{
    Q_EMIT obj->imServerConnection->imInitiatedHide();
    return TRUE;
}

static gboolean m_dbus_glib_input_context_adaptor_commit_string(
    MDBusGlibInputContextAdaptor *obj, const char *string, gint32 replaceStart,
    gint32 replaceLength, gint32 cursorPos, GError **/*error*/)
{
    Q_EMIT obj->imServerConnection->commitString(QString::fromUtf8(string), replaceStart,
                                    replaceLength, cursorPos);
    return TRUE;
}

QDataStream &operator>>(QDataStream &s, Maliit::PreeditTextFormat &t)
{
    int preeditFace;
    s >> t.start;
    s >> t.length;
    s >> preeditFace;
    t.preeditFace = static_cast<Maliit::PreeditFace>(preeditFace);
    return s;
}

static gboolean m_dbus_glib_input_context_adaptor_update_preedit(MDBusGlibInputContextAdaptor *obj,
                                                                 const char *string,
                                                                 GPtrArray *formatListData,
                                                                 gint32 replaceStart,
                                                                 gint32 replaceLength,
                                                                 gint32 cursorPos, GError **/*error*/)
{
    QList<Maliit::PreeditTextFormat> formatList;
    for (guint i = 0; i < formatListData->len; ++i) {
        GValueArray *itemData = ((GValueArray**)formatListData->pdata)[i];
        formatList.push_back(Maliit::PreeditTextFormat(
                                    g_value_get_int(g_value_array_get_nth(itemData, 0)),
                                    g_value_get_int(g_value_array_get_nth(itemData, 1)),
                                    Maliit::PreeditFace(
                                        g_value_get_int(g_value_array_get_nth(itemData, 2)))));
    }
    Q_EMIT obj->imServerConnection->updatePreedit(QString::fromUtf8(string),
                                     formatList,
                                     replaceStart,
                                     replaceLength,
                                     cursorPos);
    return TRUE;
}

static gboolean m_dbus_glib_input_context_adaptor_key_event(
    MDBusGlibInputContextAdaptor *obj, gint32 type, gint32 key, gint32 modifiers, const char *text,
    gboolean autoRepeat, gint32 count, guchar requestType, GError **/*error*/)
{
    Q_EMIT obj->imServerConnection->keyEvent(type, key, modifiers, QString::fromUtf8(text),
                                autoRepeat == TRUE, count,
                                static_cast<Maliit::EventRequestType>(requestType));
    return TRUE;
}

static gboolean m_dbus_glib_input_context_adaptor_update_input_method_area(
    MDBusGlibInputContextAdaptor *obj,
    gint32 left, gint32 top, gint32 width, gint32 height,
    GError **/*error*/)
{
    Q_EMIT obj->imServerConnection->updateInputMethodArea(QRect(left, top, width, height));
    return TRUE;
}

static gboolean m_dbus_glib_input_context_adaptor_set_global_correction_enabled(
    MDBusGlibInputContextAdaptor *obj, gboolean value, GError **/*error*/)
{
    Q_EMIT obj->imServerConnection->setGlobalCorrectionEnabled(value == TRUE);
    return TRUE;
}

static gboolean m_dbus_glib_input_context_adaptor_preedit_rectangle(
    MDBusGlibInputContextAdaptor *obj, gboolean *valid, gint *x, gint *y,
    gint *width, gint *height, GError **/*error*/)
{
    bool deserializedValidity;
    QRect rect;
    Q_EMIT obj->imServerConnection->getPreeditRectangle(rect, deserializedValidity);
    *valid = deserializedValidity ? TRUE : FALSE;
    *x = rect.x();
    *y = rect.y();
    *width = rect.width();
    *height = rect.height();
    return TRUE;
}

static gboolean m_dbus_glib_input_context_adaptor_set_redirect_keys(
    MDBusGlibInputContextAdaptor *obj, gboolean enabled, GError **/*error*/)
{
    Q_EMIT obj->imServerConnection->setRedirectKeys(enabled == TRUE);
    return TRUE;
}

static gboolean m_dbus_glib_input_context_adaptor_set_detectable_auto_repeat(
    MDBusGlibInputContextAdaptor *obj, gboolean enabled, GError **/*error*/)
{
    Q_EMIT obj->imServerConnection->setDetectableAutoRepeat(enabled == TRUE);
    return TRUE;
}

static gboolean m_dbus_glib_input_context_adaptor_set_selection(MDBusGlibInputContextAdaptor *obj,
                                                                gint32 start, gint32 length, GError **/*error*/)
{
    Q_EMIT obj->imServerConnection->setSelection(start,length);
    return TRUE;
}

static gboolean m_dbus_glib_input_context_adaptor_selection(
    MDBusGlibInputContextAdaptor *obj, gboolean *valid, gchar **gdata, GError **/*error*/)
{
    bool validity;
    QString selection;
    Q_EMIT obj->imServerConnection->getSelection(selection, validity);
    *valid = validity ? TRUE : FALSE;
    *gdata = NULL;

    if (validity && !selection.isEmpty()) {
        *gdata = g_strdup(selection.toUtf8().data());
    }
    return TRUE;
}

static gboolean m_dbus_glib_input_context_adaptor_set_language(
    MDBusGlibInputContextAdaptor *obj, const char *string,
    GError **/*error*/)
{
    Q_EMIT obj->imServerConnection->setLanguage(QString::fromUtf8(string));
    return TRUE;
}

static gboolean m_dbus_glib_input_context_adaptor_notify_extended_attribute_changed(
        MDBusGlibInputContextAdaptor *obj,
        gint32 id,
        const char *target,
        const char *targetItem,
        const char *attribute,
        GValue *valueData,
        GError **/*error*/)
{
    QVariant value;
    QString error_message;
    if (decodeVariant(&value, valueData, &error_message)) {
         Q_EMIT obj->imServerConnection->extendedAttributeChanged(static_cast<int>(id),
                                                                  QString::fromUtf8(target),
                                                                  QString::fromUtf8(targetItem),
                                                                  QString::fromUtf8(attribute),
                                                                  value);
    } else {
        qWarning() << "notify_extended_attribute_changed.arg[4]" + error_message;
    }
    return TRUE;
}

static gboolean m_dbus_glib_input_context_adaptor_plugin_settings_loaded(MDBusGlibInputContextAdaptor *obj,
                                                                         GPtrArray *settingsData, GError **/*error*/)
{
    QList<MImPluginSettingsInfo> settings;
    for (guint i = 0; i < settingsData->len; ++i) {
        GValueArray *plugin_data = ((GValueArray**)settingsData->pdata)[i];
        MImPluginSettingsInfo plugin;
        GPtrArray *entries_data = (GPtrArray *)g_value_get_boxed(g_value_array_get_nth(plugin_data, 4));

        plugin.description_language = QString::fromUtf8(g_value_get_string(g_value_array_get_nth(plugin_data, 0)));
        plugin.plugin_name = QString::fromUtf8(g_value_get_string(g_value_array_get_nth(plugin_data, 1)));
        plugin.plugin_description = QString::fromUtf8(g_value_get_string(g_value_array_get_nth(plugin_data, 2)));
        plugin.extension_id = g_value_get_int(g_value_array_get_nth(plugin_data, 3));

        for (guint i = 0; i < entries_data->len; ++i) {
            GValueArray *entry_data = ((GValueArray**)entries_data->pdata)[i];
            MImPluginSettingsEntry entry;

            QString error_message;
            if (!decodeVariantMap(&entry.attributes, (GHashTable *)g_value_get_boxed(g_value_array_get_nth(entry_data, 5)), &error_message)) {
                qWarning() << "m_dbus_glib_input_context_adaptor_plugin_settings_loaded entry attributes:" + error_message;

                return FALSE;
            }

            entry.description = QString::fromUtf8(g_value_get_string(g_value_array_get_nth(entry_data, 0)));
            entry.extension_key = QString::fromUtf8(g_value_get_string(g_value_array_get_nth(entry_data, 1)));
            entry.type = static_cast<Maliit::SettingEntryType>(g_value_get_int(g_value_array_get_nth(entry_data, 2)));

            bool valid = g_value_get_boolean(g_value_array_get_nth(entry_data, 3));

            if (!valid) {
                entry.value = QVariant();
            } else if (!decodeVariant(&entry.value, g_value_array_get_nth(entry_data, 4), &error_message)) {
                qWarning() << "m_dbus_glib_input_context_adaptor_plugin_settings_loaded entry value:" + error_message;
                return FALSE;
            }

            plugin.entries.append(entry);
        }

        settings.append(plugin);
    }
    Q_EMIT obj->imServerConnection->pluginSettingsReceived(settings);

    return TRUE;
}

#include "mdbusglibinputcontextadaptorglue.h"

static void
m_dbus_glib_input_context_adaptor_init(MDBusGlibInputContextAdaptor */*obj*/)
{
}

static void
m_dbus_glib_input_context_adaptor_class_init(MDBusGlibInputContextAdaptorClass */*klass*/)
{
    dbus_g_object_type_install_info(M_TYPE_DBUS_GLIB_INPUT_CONTEXT_ADAPTOR, &dbus_glib_m_dbus_glib_input_context_adaptor_object_info);
}
